# Plan: Backlink Reclaim UX Improvements

## Objective
Align the Backlink Reclaim tool's user experience with the Redirect Chains module for consistency, particularly around:
- Viewing successful vs failed redirects
- Tracking redirect progress
- Exporting results to CSV

---

## Current State Analysis

### What Backlink Reclaim Has
- Single and bulk redirect creation
- Basic "Redirects Applied" section with status filters (All/Fixed/Failed/Skipped)
- Two export functions (dead pages CSV + redirect results CSV)
- Scan results cached in database with `redirect_results` array

### What's Missing (vs Redirect Chains)
1. **Visual separation** between "Pending" and "Completed" redirects
2. **Collapsible sections** with toggle buttons and status badges
3. **Progress tracking** during bulk operations (percentage modal)
4. **Batch actions bar** at bottom with selected count
5. **Consistent filter button styling** with counts
6. **Section hints/descriptions** explaining each area

---

## Implementation Plan

### Phase 1: Reorganize UI Sections

#### 1.1 Split Results into Two Distinct Sections

**Current Structure:**
```
[Dead Pages with Backlinks] - Always visible table
[Redirects Applied] - Collapsible table at bottom
```

**New Structure:**
```
[Dead Pages Needing Redirects] - Collapsible section with toggle
  - Header: "X Dead Pages Needing Redirects" with collapse toggle
  - Subtitle hint: "Create 301 redirects to reclaim backlink value"
  - Table of pending dead pages
  - Batch actions bar at bottom

[Redirects Created] - Collapsible section (expanded by default after fixes)
  - Header: "X Redirects Created" with success icon
  - Subtitle hint: "These redirects are now active"
  - Status filter tabs: All | Success | Failed
  - Table of completed redirects
  - Export CSV button in header
```

#### 1.2 Update Section Headers

**Files to modify:**
- `modules/backlink-reclaim/views/tab-content.php`

**Changes:**
- Add collapsible wrapper div with `sf-results-section` class
- Add header with toggle button (chevron icon)
- Add section hint/subtitle text
- Add badge showing count (e.g., "26 pending")

**CSS classes to reuse from Redirect Chains:**
- `.sf-section-header` - Section header styling
- `.sf-section-toggle` - Toggle button
- `.sf-section-hint` - Subtitle hint text
- `.sf-section-badge` - Count badge

---

### Phase 2: Improve Batch Operations UX

#### 2.1 Add Batch Actions Bar

**Location:** Bottom of "Dead Pages Needing Redirects" section

**Components:**
```html
<div class="sf-batch-actions-bar">
  <span class="sf-selected-count">0 selected</span>
  <button class="sf-bulk-redirect-btn" disabled>
    Create Redirects for Selected
  </button>
</div>
```

**Behavior:**
- Bar appears/animates when items are selected
- Selected count updates in real-time
- Button disabled when nothing selected
- Sticky position when scrolling long lists

#### 2.2 Add Progress Modal for Bulk Operations

**Current behavior:** Button text changes to show progress inline

**New behavior:** Modal overlay showing:
```
+----------------------------------+
|     Creating Redirects...        |
|                                  |
|  [=====>            ] 35%        |
|                                  |
|  Processing: /old-page-url/      |
|  12 of 26 complete               |
|                                  |
|         [Cancel]                 |
+----------------------------------+
```

**Files to modify:**
- `modules/backlink-reclaim/assets/backlink-reclaim.js`
- `modules/backlink-reclaim/assets/backlink-reclaim.css`

---

### Phase 3: Enhanced Results Display

#### 3.1 Improve "Redirects Created" Section

**Current columns:**
- Dead Page URL
- New 301 Redirect URL
- Status
- Plugin Used
- Date Applied
- Backlinks

**Updated columns (match Redirect Chains pattern):**
| Column | Description |
|--------|-------------|
| Status | Icon + badge (✓ Fixed / ✗ Failed) |
| Was | Original dead page URL (strikethrough styling) |
| Now | New destination URL (green text) |
| Backlinks | Number reclaimed |
| When | Human-readable timestamp ("2 hours ago") |

#### 3.2 Status Filter Styling

**Current:** Basic buttons with counts

**Updated:** Match Redirect Chains filter button style:
- Active state: Teal background (#14b8a6)
- Inactive: Gray outline
- Disabled when count is 0
- Smooth transition on click

#### 3.3 Row Styling by Status

| Status | Row Background | Text Color |
|--------|---------------|------------|
| Success | Light green (#f0fdf4) | Green (#047857) |
| Failed | Light red (#fef2f2) | Red (#dc2626) |
| Skipped | Light gray (#f9fafb) | Gray (#6b7280) |

---

### Phase 4: Export Improvements

#### 4.1 Unified Export Button Location

**Current:** Export button in section header + separate export redirects button

**Updated:**
- "Export Dead Pages" button in pending section header
- "Download CSV" button in completed section header (exports redirect results)

#### 4.2 Enhanced CSV Export Content

**Dead Pages Export (existing):**
```csv
Dead Page,Backlinks,Referring Domains,Top Referrer,Status Code
```

**Redirect Results Export (enhanced):**
```csv
Dead Page URL (Was),Redirect URL (Now),Status,Status Message,Plugin Used,When Applied,Backlinks Reclaimed
/old-page/,https://example.com/new-page/,Success,Redirect created,Redirection,2024-01-26 10:35:00,15
```

---

### Phase 5: JavaScript Enhancements

#### 5.1 Update `createBulkRedirects()` Function

**Add:**
- Progress modal show/hide
- Percentage calculation
- Cancel button functionality
- Smooth animation on complete

#### 5.2 Update `addRedirectResultRow()` Function

**Add:**
- Strikethrough styling for "Was" column
- Green text for "Now" column
- Human-readable timestamp conversion
- Row animation on add

#### 5.3 Add Section Toggle Functions

**New functions:**
- `togglePendingSection()` - Collapse/expand pending redirects
- `toggleCompletedSection()` - Collapse/expand completed redirects
- `updateSectionCounts()` - Update badge counts after changes

#### 5.4 Update Selection Management

**Changes:**
- Move selected count to batch actions bar
- Add `updateSelectedCount()` function
- Show/hide batch bar based on selection state

---

### Phase 6: Data Persistence

#### 6.1 Separate Storage for Redirect Results

**Current:** Redirect results stored inside scan cache (lost if cache cleared)

**Updated:** Store redirect results in separate transient/option:
```php
// Transient key: sf_backlink_redirect_results_{domain_hash}
// Duration: 30 days (vs 7 days for scan cache)
```

This ensures:
- Redirect history persists longer than scan data
- Users can clear scan cache without losing redirect history
- Results available even after re-scanning

#### 6.2 Add Activity Logging

**Integration with `SF_Change_Logger`:**
```php
$logger->log_change([
    'fix_type' => 'backlink_redirect_created',
    'module' => 'backlink-reclaim',
    'original_value' => $source_url,
    'new_value' => $destination_url,
    'context' => json_encode([
        'backlink_count' => $count,
        'plugin' => $plugin_name,
        'status' => 'success'
    ])
]);
```

---

## File Changes Summary

| File | Changes |
|------|---------|
| `class-backlink-reclaim.php` | Add separate results storage, activity logging |
| `views/tab-content.php` | Reorganize sections, add collapsible headers, batch bar |
| `assets/backlink-reclaim.js` | Progress modal, section toggles, enhanced selection |
| `assets/backlink-reclaim.css` | New section styles, progress modal, status row colors |

---

## UI Mockup: New Section Structure

```
┌─────────────────────────────────────────────────────────────────┐
│ 🔙 Backlink Reclaim                                             │
│ Recover lost backlinks by redirecting dead pages                │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│ [Scan Section - Domain input, Scan button]                      │
│                                                                 │
│ ┌─ 📊 Stats Cards ─────────────────────────────────────────────┐│
│ │  26 Dead Pages  |  81 Backlinks  |  3 Redirects Created     ││
│ └──────────────────────────────────────────────────────────────┘│
│                                                                 │
│ ┌─ 🔗 Dead Pages Needing Redirects (23) ──────────── [▼] ─────┐│
│ │ Create 301 redirects to reclaim backlink value              ││
│ │                                                              ││
│ │ [Table of pending dead pages]                                ││
│ │ ☐ | Dead Page | Backlinks | Referrers | Redirect To | Action││
│ │ ───────────────────────────────────────────────────────────  ││
│ │ ☐ | /old-page | 7         | 5 domains | [input]    | Create ││
│ │ ☐ | /another  | 5         | 3 domains | [input]    | Create ││
│ │                                                              ││
│ │ ┌── Batch Actions ─────────────────────────────────────────┐││
│ │ │ 2 selected                    [Create Redirects (2)]     │││
│ │ └──────────────────────────────────────────────────────────┘││
│ └──────────────────────────────────────────────────────────────┘│
│                                                                 │
│ ┌─ ✓ Redirects Created (3) ─────────────────────── [▼] ───────┐│
│ │ These redirects are now active                   [Export CSV]││
│ │                                                              ││
│ │ [All (3)] [Success (2)] [Failed (1)]                         ││
│ │                                                              ││
│ │ [Table of completed redirects]                               ││
│ │ Status | Was (strikethrough) | Now (green) | Backlinks | When││
│ │ ───────────────────────────────────────────────────────────  ││
│ │ ✓ Fixed| /old-page          | /new-page   | 7          | 2h  ││
│ │ ✗ Fail | /another           | /page       | 5          | 2h  ││
│ └──────────────────────────────────────────────────────────────┘│
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

---

## Progress Modal Mockup

```
┌─────────────────────────────────────────┐
│                                         │
│        Creating Redirects...            │
│                                         │
│   ████████████░░░░░░░░░░░░░░  45%       │
│                                         │
│   Processing: /news/old-article/        │
│                                         │
│   12 of 26 redirects created            │
│   2 failed                              │
│                                         │
│              [Cancel]                   │
│                                         │
└─────────────────────────────────────────┘
```

---

## Success Criteria

1. ✅ Users can clearly see pending vs completed redirects in separate sections
2. ✅ Both sections are collapsible with visual indicators
3. ✅ Bulk operations show progress percentage in modal
4. ✅ Status filters work consistently with Redirect Chains styling
5. ✅ Export CSV available for redirect results
6. ✅ Redirect history persists beyond scan cache lifetime
7. ✅ Activity logged for dashboard tracking

---

## Estimated Scope

| Component | Complexity | Priority |
|-----------|------------|----------|
| Section reorganization | Medium | High |
| Collapsible sections | Low | High |
| Batch actions bar | Low | Medium |
| Progress modal | Medium | Medium |
| Status filter styling | Low | High |
| Row styling by status | Low | High |
| Separate results storage | Medium | Medium |
| Activity logging | Low | Low |

**Total estimated effort:** Medium complexity, mostly UI/UX changes with some data handling updates.
