# Plan: Broken Links - Fixed Tab Immediate Refresh

## Objective
Update the Broken Links tool so that when fixes are applied via "Apply All Fixes", the Fixed Links tab updates immediately (like the Page Title tool), and change the source title color to teal for brand consistency.

---

## Current Behavior vs Target Behavior

| Aspect | Current | Target |
|--------|---------|--------|
| Fixed rows in Fixable tab | Transformed in-place (stays in Fixable) | **Remove from Fixable tab** |
| Fixed Links tab | Only updates on page reload | **Updates immediately via AJAX** |
| After applying fixes | User must navigate away/back | **Auto-scroll to Fixed Links tab** |
| Source title color | Blue (`#2563eb`) | **Teal (`#14b8a6`)** |

---

## Changes Required

### 1. CSS Change - Source Title Color (Quick Fix)

**File:** `modules/broken-links/assets/broken-links.css`

**Line 3018:** Change from blue to teal:
```css
/* Before */
.sf-fixed-table .sf-source-info .sf-source-title {
    color: var(--sf-primary, #2563eb);  /* Blue */
}

/* After */
.sf-fixed-table .sf-source-info .sf-source-title {
    color: var(--sf-primary, #14b8a6);  /* Teal */
}
```

---

### 2. PHP - Add AJAX Endpoint for Fixed Section

**File:** `modules/broken-links/class-broken-links.php`

**Add new AJAX action registration** (around line 50):
```php
add_action('wp_ajax_sf_broken_links_get_fixed_section', [$this, 'ajax_get_fixed_section']);
```

**Add new method** `ajax_get_fixed_section()`:
- Check nonce and permissions
- Get results from storage
- Generate Fixed Links section HTML (matching tab-content.php structure)
- Return JSON with HTML and count

---

### 3. JavaScript - Update Apply Fixes Handler

**File:** `modules/broken-links/assets/broken-links.js`

**Modify `handleApplyAllFixes()` success handler** (lines 1072-1144):

Current flow:
1. Transform rows in place with `transformRowToFixed()`
2. Update fixed count
3. Show toast

New flow:
1. **Remove fixed rows** from Fixable Links section (fade out animation)
2. **Call new AJAX endpoint** to refresh Fixed Links section
3. **Auto-expand and scroll** to Fixed Links section
4. Update counts
5. Show toast

**Add new functions:**
- `refreshFixedSection()` - AJAX call to get updated HTML
- `updateFixableSectionCount()` - Update header count after row removal
- `bindFixedSectionEvents()` - Re-bind event handlers after refresh

---

## Detailed Implementation

### Phase 1: CSS Color Change

**File:** `modules/broken-links/assets/broken-links.css` (line 3018)

```css
.sf-fixed-table .sf-source-info .sf-source-title {
    display: block;
    font-weight: 500;
    color: var(--sf-primary, #14b8a6);  /* Changed from #2563eb */
    text-decoration: none;
    margin-bottom: 4px;
}
```

---

### Phase 2: PHP AJAX Endpoint

**File:** `modules/broken-links/class-broken-links.php`

Add to `init()` method:
```php
add_action('wp_ajax_sf_broken_links_get_fixed_section', [$this, 'ajax_get_fixed_section']);
```

Add new method:
```php
public function ajax_get_fixed_section() {
    check_ajax_referer('sf_broken_links_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $results = $this->get_results();
    if (empty($results)) {
        $results = $this->get_upload_data();
    }

    // Flatten fixed_links for display (one row per source page)
    $fixed_links = $this->flatten_fixed_links($results['fixed_links'] ?? []);
    $fixed_count = count($fixed_links);

    // Generate HTML using output buffering
    ob_start();
    // ... render Fixed Links section HTML matching tab-content.php structure
    $html = ob_get_clean();

    wp_send_json_success([
        'html' => $html,
        'fixed_count' => $fixed_count,
    ]);
}
```

---

### Phase 3: JavaScript Updates

**File:** `modules/broken-links/assets/broken-links.js`

#### 3a. Modify success handler in `handleApplyAllFixes()`:

```javascript
success: function(response) {
    hideProgressModal();

    if (response.success) {
        var successCount = 0;
        var fixedUrls = [];

        // Collect fixed URLs and remove rows from Fixable section
        $.each(fixesToApply, function(i, fix) {
            var $row = $('.sf-link-row[data-broken-url="' + escapeSelector(fix.broken_url) + '"]');
            var $panelRow = $('.sf-action-panel-row[data-for-url="' + escapeSelector(fix.broken_url) + '"]');

            if ($row.length) {
                successCount++;
                fixedUrls.push(fix.broken_url);

                // Close any open panels
                $panelRow.slideUp(200);
                $row.removeClass('sf-panel-open');

                // Fade out and remove the row
                $row.fadeOut(300, function() {
                    $(this).remove();
                    updateFixableSectionCount();
                });
                $panelRow.fadeOut(300, function() {
                    $(this).remove();
                });
            }

            // Remove from fixes tracking
            delete BrokenLinks.fixes[fix.broken_url];
        });

        clearFixesFromStorage();

        // Show toast
        showToast(response.data.message, 'success');

        // Refresh Fixed Links section via AJAX
        if (successCount > 0) {
            refreshFixedSection();
        }

        setTimeout(function() {
            updateBatchState();
        }, 350);
    } else {
        showToast(response.data.message || 'Fixes failed', 'error');
    }
}
```

#### 3b. Add new helper functions:

```javascript
/**
 * Update the Fixable Links section header count
 */
function updateFixableSectionCount() {
    var $section = $('.sf-section-fixable');
    var remainingCount = $section.find('.sf-link-row:not(.sf-row-fixed-complete)').length;

    // Update header text
    var $toggle = $section.find('.sf-section-toggle');
    var currentText = $toggle.text();
    var newText = currentText.replace(/\d+ Fixable Links?/, remainingCount + ' Fixable Link' + (remainingCount !== 1 ? 's' : ''));

    // Rebuild toggle content preserving structure
    // ... similar to Page Title implementation

    // Hide section if empty
    if (remainingCount === 0) {
        $section.slideUp(300);
    }
}

/**
 * Refresh the Fixed Links section via AJAX
 */
function refreshFixedSection() {
    var $fixedSection = $('.sf-section-fixed');

    $.ajax({
        url: sfBrokenLinksData.ajaxUrl,
        type: 'POST',
        data: {
            action: 'sf_broken_links_get_fixed_section',
            nonce: sfBrokenLinksData.nonce
        },
        success: function(response) {
            if (response.success && response.data.html) {
                if ($fixedSection.length) {
                    $fixedSection.html(response.data.html);
                } else {
                    // Create section if it doesn't exist
                    var $newSection = $('<div class="sf-results-section sf-section-fixed" id="sf-fixed-results"></div>');
                    $newSection.html(response.data.html);
                    // Insert in appropriate location
                    $('.sf-results-section').last().after($newSection);
                    $fixedSection = $newSection;
                }

                $fixedSection.show();

                // Ensure content is visible
                var $content = $fixedSection.find('.sf-fixed-content, .sf-table-wrapper');
                $content.show();

                // Scroll to Fixed section
                setTimeout(function() {
                    $('html, body').animate({
                        scrollTop: $fixedSection.offset().top - 50
                    }, 500);
                }, 400);

                // Re-bind events
                bindFixedSectionEvents();
            }
        }
    });
}

/**
 * Bind event handlers for Fixed section after AJAX refresh
 */
function bindFixedSectionEvents() {
    // Toggle handler
    $('.sf-section-fixed .sf-fixed-toggle').off('click').on('click', function() {
        var $btn = $(this);
        var $content = $btn.closest('.sf-section-fixed').find('.sf-table-wrapper');
        var isExpanded = $btn.attr('aria-expanded') === 'true';

        if (isExpanded) {
            $content.slideUp(300);
            $btn.attr('aria-expanded', 'false');
            $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
        } else {
            $content.slideDown(300);
            $btn.attr('aria-expanded', 'true');
            $btn.find('.sf-toggle-icon').addClass('sf-rotated');
        }
    });

    // Export CSV handler
    $('.sf-section-fixed .sf-export-fixed-btn').off('click').on('click', function() {
        exportFixedLinks();
    });

    // Re-initialize pagination if needed
    initFixedPagination();
}
```

---

## Files to Modify

| File | Changes |
|------|---------|
| `modules/broken-links/assets/broken-links.css` | Change source title color (line 3018) |
| `modules/broken-links/class-broken-links.php` | Add AJAX endpoint, register action |
| `modules/broken-links/assets/broken-links.js` | Update success handler, add helper functions |

---

## Testing Checklist

- [ ] Source page titles in Fixed Links tab display in teal color
- [ ] After applying fixes, rows fade out from Fixable Links section
- [ ] Fixable Links section header count updates correctly
- [ ] Fixed Links section refreshes with new items
- [ ] Fixed Links section auto-expands and page scrolls to it
- [ ] Pagination in Fixed Links section works after refresh
- [ ] Export CSV button works after refresh
- [ ] Toggle expand/collapse works after refresh
- [ ] If all Fixable rows are fixed, section hides
- [ ] If Fixed Links section didn't exist before, it's created correctly

---

## Consistency with Page Title Tool

This implementation mirrors the Page Title tool's approach:

| Feature | Page Title | Broken Links (Proposed) |
|---------|------------|------------------------|
| Row removal | Fade out + remove | Fade out + remove |
| Fixed section refresh | AJAX endpoint | AJAX endpoint |
| Auto-scroll | Yes | Yes |
| Event re-binding | `bindFixedSectionEvents()` | `bindFixedSectionEvents()` |
| Count updates | Header text updated | Header text updated |
