# Implementation Plan: Bulk CSV Upload for Image Alt Text Module

## Overview

Add bulk CSV upload capability to the Image Alt Text module, allowing users to add a `New_Alt_Text` column to their Screaming Frog export and apply alt text updates in bulk. Follows the same patterns established in the Page Title bulk upload feature.

---

## Files to Modify

| File | Changes |
|------|---------|
| `modules/image-alt-text/class-image-alt-text.php` | Add bulk detection, processing, and batch update methods |
| `modules/image-alt-text/views/tab-content.php` | Add bulk confirmation UI, progress modal, results display |
| `modules/image-alt-text/views/instructions.php` | Add "Prefer working in spreadsheets?" section |
| `modules/image-alt-text/assets/image-alt-text.css` | Add bulk-specific styles (match Page Title) |
| `modules/image-alt-text/assets/image-alt-text.js` | Add bulk processing handlers, progress tracking |

---

## Phase 1: PHP Backend (class-image-alt-text.php)

### 1.1 Add New AJAX Endpoints

Add to `init()` method after existing AJAX hooks (around line 45):

```php
add_action('wp_ajax_sf_image_alt_text_apply_bulk_updates', [$this, 'ajax_apply_bulk_updates']);
add_action('wp_ajax_sf_image_alt_text_download_bulk_preview', [$this, 'ajax_download_bulk_preview']);
add_action('wp_ajax_sf_image_alt_text_download_bulk_results', [$this, 'ajax_download_bulk_results']);
```

### 1.2 Add Column Detection Methods

Add after existing `standardize_image_columns()` method (around line 400):

```php
/**
 * Normalize column name for flexible matching
 * Removes spaces, dashes, underscores and lowercases
 */
private function normalize_column_name($name) {
    $normalized = strtolower(trim($name));
    $normalized = str_replace([' ', '-', '_'], '', $normalized);
    return $normalized;
}

/**
 * Check if CSV is a bulk update CSV (has new_alt_text column)
 * Returns array with column info or false
 */
public function is_bulk_update_csv($headers) {
    $new_alt_header = null;
    $url_header = null;
    $current_alt_header = null;
    $update_columns_found = [];

    foreach ($headers as $header) {
        $normalized = $this->normalize_column_name($header);
        $header_lower = strtolower(trim($header));

        // Check for new alt text column (required)
        // Must contain: "new" or "fix" or "replacement" AND contain "alt"
        $has_new_fix_replacement = (
            strpos($normalized, 'new') !== false ||
            strpos($normalized, 'fix') !== false ||
            strpos($normalized, 'replacement') !== false
        );
        $has_alt = strpos($normalized, 'alt') !== false;

        if ($has_new_fix_replacement && $has_alt) {
            $new_alt_header = $header_lower;
            $update_columns_found[] = 'new_alt_text';
        }

        // Check for other update columns (to detect mixed CSVs)
        if ($normalized === 'newtitle' || ($has_new_fix_replacement && strpos($normalized, 'title') !== false)) {
            $update_columns_found[] = 'new_title';
        }
        if ($normalized === 'newdescription' || $normalized === 'newmetadescription') {
            $update_columns_found[] = 'new_description';
        }

        // Check for image URL column (required)
        $url_patterns = ['address', 'source', 'url', 'imageurl', 'src'];
        if (in_array($normalized, $url_patterns)) {
            $url_header = $header_lower;
        }

        // Check for current alt text column (optional, for display)
        $alt_patterns = ['alttext', 'alt', 'currentalt'];
        // Don't match if it's the new alt column
        if (in_array($normalized, $alt_patterns) && !$has_new_fix_replacement) {
            $current_alt_header = $header_lower;
        }
    }

    // Error: multiple update column types
    if (count($update_columns_found) > 1) {
        return [
            'error' => 'multiple_update_columns',
            'columns' => $update_columns_found,
        ];
    }

    // Not a bulk update CSV (no new_alt column)
    if ($new_alt_header === null) {
        return false;
    }

    // Error: missing URL column
    if ($url_header === null) {
        return [
            'error' => 'no_url_column',
        ];
    }

    return [
        'is_bulk' => true,
        'new_alt_header' => $new_alt_header,
        'url_header' => $url_header,
        'current_alt_header' => $current_alt_header,
    ];
}
```

### 1.3 Modify ajax_process_csv() for Bulk Detection

Update existing `ajax_process_csv()` method (around line 905) to detect bulk mode:

```php
public function ajax_process_csv() {
    check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

    // Get uploaded file
    $uploads = get_option('sf_pending_uploads', []);
    if (!isset($uploads[$upload_id])) {
        wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
    }

    $file_path = $uploads[$upload_id]['path'];

    // Parse CSV headers first
    $parser = new SF_CSV_Parser();
    $parsed = $parser->parse($file_path);

    if (is_wp_error($parsed)) {
        wp_send_json_error(['message' => $parsed->get_error_message()]);
    }

    if (!empty($parsed['headers'])) {
        $bulk_check = $this->is_bulk_update_csv($parsed['headers']);

        // Check for bulk update errors
        if (is_array($bulk_check) && isset($bulk_check['error'])) {
            if ($bulk_check['error'] === 'multiple_update_columns') {
                wp_send_json_error([
                    'message' => __('Please upload a CSV with only one update column at a time. Found: ', 'screaming-fixes') . implode(', ', $bulk_check['columns']),
                ]);
                return;
            }
            if ($bulk_check['error'] === 'no_url_column') {
                wp_send_json_error([
                    'message' => __('Bulk update CSV detected but missing a URL column. Expected: Address, Source, URL, or Image_URL.', 'screaming-fixes'),
                ]);
                return;
            }
        }

        // If valid bulk update CSV, process for bulk mode
        if (is_array($bulk_check) && isset($bulk_check['is_bulk']) && $bulk_check['is_bulk'] === true) {
            $results = $this->process_bulk_csv($file_path, $bulk_check);

            if (is_wp_error($results)) {
                wp_send_json_error(['message' => $results->get_error_message()]);
            }

            wp_send_json_success([
                'message' => sprintf(__('%d images ready to update', 'screaming-fixes'), $results['ready_count']),
                'data' => $results,
                'is_bulk_update' => true,
            ]);
            return;
        }
    }

    // Standard analysis processing (existing behavior)
    $results = $this->process_csv($file_path);

    if (is_wp_error($results)) {
        wp_send_json_error(['message' => $results->get_error_message()]);
    }

    // ... rest of existing code ...
}
```

### 1.4 Add Bulk CSV Processing Method

Add new method for processing bulk update CSVs:

```php
/**
 * Process a bulk update CSV file
 * Returns categorized results: ready, not_matched, skipped_empty, duplicates
 */
public function process_bulk_csv($file_path, $column_info) {
    $parser = new SF_CSV_Parser();
    $parsed = $parser->parse($file_path);

    if (is_wp_error($parsed)) {
        return $parsed;
    }

    $new_alt_header = $column_info['new_alt_header'];
    $url_header = $column_info['url_header'];
    $current_alt_header = $column_info['current_alt_header'];

    // Build WordPress attachment URL map
    $wp_images = $this->build_wp_image_map();

    // Track results
    $ready_updates = [];
    $not_matched = [];
    $skipped_empty = [];
    $duplicates = [];
    $filename_matches = [];
    $multi_page_images = [];

    // Track seen URLs for duplicate detection
    $seen_urls = [];

    foreach ($parsed['rows'] as $row_index => $row) {
        $image_url = isset($row[$url_header]) ? trim($row[$url_header]) : '';
        $new_alt = isset($row[$new_alt_header]) ? trim($row[$new_alt_header]) : '';
        $current_alt = $current_alt_header && isset($row[$current_alt_header])
            ? trim($row[$current_alt_header])
            : '';

        if (empty($image_url)) {
            continue;
        }

        // Skip rows with empty new_alt (unless [DECORATIVE])
        if (empty($new_alt)) {
            $skipped_empty[] = [
                'image_url' => $image_url,
                'current_alt' => $current_alt,
                'new_alt' => '',
                'status' => 'Skipped - No new alt text',
            ];
            continue;
        }

        // Check for duplicates (use last occurrence)
        $normalized_url = $this->normalize_image_url($image_url);
        if (isset($seen_urls[$normalized_url])) {
            $duplicates[] = $image_url;
            // Remove previous entry from ready_updates
            $ready_updates = array_filter($ready_updates, function($item) use ($normalized_url) {
                return $this->normalize_image_url($item['image_url']) !== $normalized_url;
            });
            $ready_updates = array_values($ready_updates); // Re-index
        }
        $seen_urls[$normalized_url] = true;

        // Handle [DECORATIVE] keyword
        $is_decorative = strtoupper($new_alt) === '[DECORATIVE]';
        $final_alt = $is_decorative ? '' : $new_alt;

        // Try to match image
        $match_result = $this->match_image_to_wp($image_url, $wp_images);

        if ($match_result['matched']) {
            $update_item = [
                'image_url' => $image_url,
                'attachment_id' => $match_result['attachment_id'],
                'current_alt' => $match_result['current_alt'] ?: $current_alt,
                'new_alt' => $final_alt,
                'is_decorative' => $is_decorative,
                'pages_affected' => $match_result['pages_affected'],
                'pages_count' => count($match_result['pages_affected']),
                'match_type' => $match_result['match_type'], // 'exact' or 'filename'
                'status' => 'Ready',
            ];

            // Track filename-only matches
            if ($match_result['match_type'] === 'filename') {
                $filename_matches[] = $image_url;
            }

            // Track multi-page images
            if (count($match_result['pages_affected']) > 1) {
                $multi_page_images[] = $image_url;
            }

            $ready_updates[] = $update_item;
        } else {
            $not_matched[] = [
                'image_url' => $image_url,
                'current_alt' => $current_alt,
                'new_alt' => $final_alt,
                'is_decorative' => $is_decorative,
                'status' => $match_result['reason'] ?? 'Skipped - Image not found',
            ];
        }
    }

    $results = [
        'is_bulk_update' => true,
        'ready_updates' => $ready_updates,
        'not_matched' => $not_matched,
        'skipped_empty' => $skipped_empty,
        'ready_count' => count($ready_updates),
        'not_matched_count' => count($not_matched),
        'skipped_empty_count' => count($skipped_empty),
        'duplicates_count' => count(array_unique($duplicates)),
        'filename_match_count' => count($filename_matches),
        'multi_page_count' => count($multi_page_images),
        'total_rows' => count($parsed['rows']),
    ];

    // Store for batch processing
    $this->save_bulk_data($results);

    return $results;
}
```

### 1.5 Add Image Matching Methods

```php
/**
 * Normalize image URL for comparison
 */
private function normalize_image_url($url) {
    $url = trim($url);
    $url = rtrim($url, '/');
    // Remove protocol for comparison
    $url = preg_replace('/^https?:\/\//i', '', $url);
    // Remove www
    $url = preg_replace('/^www\./i', '', $url);
    // Lowercase
    $url = strtolower($url);
    return $url;
}

/**
 * Extract base filename (without size suffix)
 */
private function get_base_filename($url) {
    $filename = basename(parse_url($url, PHP_URL_PATH));
    // Remove extension
    $ext = pathinfo($filename, PATHINFO_EXTENSION);
    $name = pathinfo($filename, PATHINFO_FILENAME);
    // Remove size suffix like -300x200, -1024x768
    $name = preg_replace('/-\d+x\d+$/', '', $name);
    return $name . '.' . $ext;
}

/**
 * Build map of all WordPress images (attachments)
 */
private function build_wp_image_map() {
    global $wpdb;

    $images = [];

    // Get all image attachments
    $attachments = $wpdb->get_results("
        SELECT ID, guid, post_title
        FROM {$wpdb->posts}
        WHERE post_type = 'attachment'
        AND post_mime_type LIKE 'image/%'
    ");

    foreach ($attachments as $attachment) {
        $url = wp_get_attachment_url($attachment->ID);
        if (!$url) continue;

        $normalized = $this->normalize_image_url($url);
        $filename = $this->get_base_filename($url);
        $current_alt = get_post_meta($attachment->ID, '_wp_attachment_image_alt', true);

        // Find all posts/pages using this image
        $pages_affected = $this->find_posts_using_image($attachment->ID, $url);

        $images[$normalized] = [
            'attachment_id' => $attachment->ID,
            'url' => $url,
            'filename' => $filename,
            'current_alt' => $current_alt,
            'pages_affected' => $pages_affected,
        ];

        // Also index by filename for fallback matching
        if (!isset($images['filename:' . strtolower($filename)])) {
            $images['filename:' . strtolower($filename)] = [];
        }
        $images['filename:' . strtolower($filename)][] = $normalized;
    }

    return $images;
}

/**
 * Find all posts/pages that use an image
 */
private function find_posts_using_image($attachment_id, $image_url) {
    global $wpdb;

    $pages = [];
    $filename = basename($image_url);
    $base_filename = $this->get_base_filename($image_url);

    // Search for image in post content
    // Match both exact URL and filename (for different sizes)
    $results = $wpdb->get_results($wpdb->prepare("
        SELECT ID, post_title, post_type
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
        AND (
            post_content LIKE %s
            OR post_content LIKE %s
        )
    ",
        '%' . $wpdb->esc_like($filename) . '%',
        '%' . $wpdb->esc_like($base_filename) . '%'
    ));

    foreach ($results as $post) {
        $pages[] = [
            'post_id' => $post->ID,
            'title' => $post->post_title,
            'type' => $post->post_type,
            'edit_url' => get_edit_post_link($post->ID, 'raw'),
            'view_url' => get_permalink($post->ID),
        ];
    }

    return $pages;
}

/**
 * Match an image URL to WordPress attachment
 */
private function match_image_to_wp($image_url, $wp_images) {
    // Try exact URL match first
    $normalized = $this->normalize_image_url($image_url);

    if (isset($wp_images[$normalized])) {
        return [
            'matched' => true,
            'match_type' => 'exact',
            'attachment_id' => $wp_images[$normalized]['attachment_id'],
            'current_alt' => $wp_images[$normalized]['current_alt'],
            'pages_affected' => $wp_images[$normalized]['pages_affected'],
        ];
    }

    // Try with/without www
    $alt_normalized = strpos($normalized, 'www.') === 0
        ? substr($normalized, 4)
        : 'www.' . $normalized;

    if (isset($wp_images[$alt_normalized])) {
        return [
            'matched' => true,
            'match_type' => 'exact',
            'attachment_id' => $wp_images[$alt_normalized]['attachment_id'],
            'current_alt' => $wp_images[$alt_normalized]['current_alt'],
            'pages_affected' => $wp_images[$alt_normalized]['pages_affected'],
        ];
    }

    // Fall back to filename matching
    $filename = strtolower($this->get_base_filename($image_url));
    $filename_key = 'filename:' . $filename;

    if (isset($wp_images[$filename_key])) {
        $matching_urls = $wp_images[$filename_key];

        // If multiple images match filename, can't determine which one
        if (count($matching_urls) > 1) {
            return [
                'matched' => false,
                'reason' => "Multiple images match filename '" . basename($image_url) . "' - please use full URL",
            ];
        }

        $matched_url = $matching_urls[0];
        if (isset($wp_images[$matched_url])) {
            return [
                'matched' => true,
                'match_type' => 'filename',
                'attachment_id' => $wp_images[$matched_url]['attachment_id'],
                'current_alt' => $wp_images[$matched_url]['current_alt'],
                'pages_affected' => $wp_images[$matched_url]['pages_affected'],
            ];
        }
    }

    return [
        'matched' => false,
        'reason' => 'Image not found in WordPress media library',
    ];
}
```

### 1.6 Add Bulk Update Application Methods

```php
/**
 * AJAX handler for applying bulk updates
 */
public function ajax_apply_bulk_updates() {
    check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $offset = isset($_POST['offset']) ? (int) $_POST['offset'] : 0;
    $batch_size = isset($_POST['batch_size']) ? (int) $_POST['batch_size'] : 50;

    // Clear accumulated results at start
    if ($offset === 0) {
        delete_transient('sf_image_bulk_accumulated_' . get_current_user_id());
    }

    // Get stored bulk data
    $bulk_data = $this->get_bulk_data();

    if (empty($bulk_data) || empty($bulk_data['ready_updates'])) {
        wp_send_json_error(['message' => __('No bulk update data found.', 'screaming-fixes')]);
    }

    $updates = $bulk_data['ready_updates'];
    $results = $this->apply_bulk_image_updates($updates, $offset, $batch_size);

    // Accumulate results
    $accumulated = get_transient('sf_image_bulk_accumulated_' . get_current_user_id());
    if (!$accumulated) {
        $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => [], 'total_pages' => 0];
    }

    $accumulated['success'] += $results['success'];
    $accumulated['failed'] += $results['failed'];
    $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
    $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);
    $accumulated['total_pages'] += $results['pages_updated'] ?? 0;

    if ($results['complete']) {
        // Store final results
        $final_results = [
            'is_bulk_update' => true,
            'bulk_complete' => true,
            'fixed_images' => $accumulated['details'],
            'failed_updates' => $accumulated['errors'],
            'success_count' => $accumulated['success'],
            'failed_count' => $accumulated['failed'],
            'total_pages_updated' => $accumulated['total_pages'],
            'not_matched' => $bulk_data['not_matched'] ?? [],
            'skipped_empty' => $bulk_data['skipped_empty'] ?? [],
        ];

        $this->save_bulk_data($final_results);
        delete_transient('sf_image_bulk_accumulated_' . get_current_user_id());

        $results['total_success'] = $accumulated['success'];
        $results['total_failed'] = $accumulated['failed'];
        $results['total_pages_updated'] = $accumulated['total_pages'];
        $results['all_details'] = $accumulated['details'];
    } else {
        set_transient('sf_image_bulk_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
    }

    $results['total_count'] = count($updates);
    wp_send_json_success($results);
}

/**
 * Apply bulk image alt text updates
 */
private function apply_bulk_image_updates($updates, $offset = 0, $batch_size = 50) {
    $results = [
        'processed' => 0,
        'success' => 0,
        'failed' => 0,
        'pages_updated' => 0,
        'errors' => [],
        'details' => [],
        'complete' => false,
        'next_offset' => $offset + $batch_size,
    ];

    $batch = array_slice($updates, $offset, $batch_size);

    if (empty($batch)) {
        $results['complete'] = true;
        return $results;
    }

    // Start batch for undo capability
    if (class_exists('SF_Batch_Restore') && $offset === 0) {
        SF_Batch_Restore::start_batch('image_alt_text_bulk', 'Bulk Image Alt Text Update');
    }

    foreach ($batch as $update) {
        $results['processed']++;

        $attachment_id = isset($update['attachment_id']) ? (int) $update['attachment_id'] : 0;
        $new_alt = isset($update['new_alt']) ? $update['new_alt'] : '';
        $image_url = isset($update['image_url']) ? $update['image_url'] : '';
        $pages_affected = isset($update['pages_affected']) ? $update['pages_affected'] : [];

        if (!$attachment_id) {
            $results['failed']++;
            $results['errors'][] = [
                'image_url' => $image_url,
                'error' => __('Invalid attachment ID', 'screaming-fixes'),
            ];
            continue;
        }

        try {
            // Get original alt for undo
            $original_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);

            // Update attachment metadata
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $new_alt);

            // Update all post content where image appears
            $pages_updated = 0;
            foreach ($pages_affected as $page) {
                $post_id = $page['post_id'];
                $post = get_post($post_id);

                if ($post && $post->post_content) {
                    $updated_content = $this->update_image_alt_in_content(
                        $post->post_content,
                        $image_url,
                        $new_alt
                    );

                    if ($updated_content !== $post->post_content) {
                        // Track for undo
                        if (class_exists('SF_Batch_Restore')) {
                            SF_Batch_Restore::track_change($post_id, 'post_content', $post->post_content);
                        }

                        wp_update_post([
                            'ID' => $post_id,
                            'post_content' => $updated_content,
                        ]);
                        $pages_updated++;
                    }
                }
            }

            $results['success']++;
            $results['pages_updated'] += $pages_updated;
            $results['details'][] = [
                'image_url' => $image_url,
                'attachment_id' => $attachment_id,
                'original_alt' => $original_alt,
                'new_alt' => $new_alt,
                'pages_updated' => $pages_updated,
                'is_decorative' => empty($new_alt),
            ];

        } catch (Exception $e) {
            $results['failed']++;
            $results['errors'][] = [
                'image_url' => $image_url,
                'error' => $e->getMessage(),
            ];
        }
    }

    // Check if complete
    if ($offset + $batch_size >= count($updates)) {
        $results['complete'] = true;

        // Complete batch for undo
        if (class_exists('SF_Batch_Restore')) {
            SF_Batch_Restore::complete_batch();
        }
    }

    return $results;
}

/**
 * Update image alt text in HTML content
 */
private function update_image_alt_in_content($content, $image_url, $new_alt) {
    $filename = basename(parse_url($image_url, PHP_URL_PATH));
    $base_filename = $this->get_base_filename($image_url);

    // Escape for regex
    $filename_escaped = preg_quote($filename, '/');
    $base_escaped = preg_quote(pathinfo($base_filename, PATHINFO_FILENAME), '/');

    // Match img tags with this image (any size variant)
    // Pattern matches filename with optional size suffix
    $pattern = '/(<img[^>]*src=["\'][^"\']*' . $base_escaped . '(?:-\d+x\d+)?\.[a-z]+["\'][^>]*)(alt=["\'][^"\']*["\'])([^>]*>)/i';

    $replacement = function($matches) use ($new_alt) {
        // Replace existing alt
        return $matches[1] . 'alt="' . esc_attr($new_alt) . '"' . $matches[3];
    };

    $content = preg_replace_callback($pattern, $replacement, $content);

    // Also handle img tags without alt attribute
    $pattern_no_alt = '/(<img[^>]*src=["\'][^"\']*' . $base_escaped . '(?:-\d+x\d+)?\.[a-z]+["\'])([^>]*)(\/?>)/i';

    $content = preg_replace_callback($pattern_no_alt, function($matches) use ($new_alt) {
        // Check if alt already exists (shouldn't after previous replacement, but safety check)
        if (stripos($matches[0], 'alt=') !== false) {
            return $matches[0];
        }
        return $matches[1] . ' alt="' . esc_attr($new_alt) . '"' . $matches[2] . $matches[3];
    }, $content);

    return $content;
}
```

### 1.7 Add Bulk Data Storage Methods

```php
/**
 * Save bulk update data for later retrieval
 */
private function save_bulk_data($data) {
    $user_id = get_current_user_id();
    set_transient('sf_image_bulk_data_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
}

/**
 * Get stored bulk update data
 */
private function get_bulk_data() {
    $user_id = get_current_user_id();
    return get_transient('sf_image_bulk_data_' . $user_id);
}

/**
 * Clear bulk update data
 */
private function clear_bulk_data() {
    $user_id = get_current_user_id();
    delete_transient('sf_image_bulk_data_' . $user_id);
    delete_transient('sf_image_bulk_accumulated_' . $user_id);
}
```

### 1.8 Add CSV Download Methods

```php
/**
 * AJAX handler for downloading bulk preview CSV
 */
public function ajax_download_bulk_preview() {
    check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $bulk_data = $this->get_bulk_data();
    if (empty($bulk_data)) {
        wp_send_json_error(['message' => __('No bulk data found.', 'screaming-fixes')]);
    }

    $csv = $this->generate_bulk_preview_csv($bulk_data);

    wp_send_json_success([
        'csv' => $csv,
        'filename' => 'image-alt-text-preview-' . date('Y-m-d') . '.csv',
    ]);
}

/**
 * AJAX handler for downloading bulk results CSV
 */
public function ajax_download_bulk_results() {
    check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $bulk_data = $this->get_bulk_data();
    if (empty($bulk_data)) {
        wp_send_json_error(['message' => __('No bulk data found.', 'screaming-fixes')]);
    }

    $csv = $this->generate_bulk_results_csv($bulk_data);

    wp_send_json_success([
        'csv' => $csv,
        'filename' => 'image-alt-text-results-' . date('Y-m-d') . '.csv',
    ]);
}

/**
 * Generate preview CSV
 */
private function generate_bulk_preview_csv($bulk_data) {
    $lines = [];
    $lines[] = 'Image URL,Current Alt Text,New Alt Text,Pages Affected,Status';

    foreach ($bulk_data['ready_updates'] ?? [] as $row) {
        $lines[] = $this->csv_escape_row([
            $row['image_url'],
            $row['current_alt'] ?? '',
            $row['is_decorative'] ? '[DECORATIVE]' : $row['new_alt'],
            $row['pages_count'] ?? 0,
            $row['status'] ?? 'Ready',
        ]);
    }

    foreach ($bulk_data['not_matched'] ?? [] as $row) {
        $lines[] = $this->csv_escape_row([
            $row['image_url'],
            $row['current_alt'] ?? '',
            $row['is_decorative'] ? '[DECORATIVE]' : $row['new_alt'],
            0,
            $row['status'] ?? 'Skipped - Not found',
        ]);
    }

    foreach ($bulk_data['skipped_empty'] ?? [] as $row) {
        $lines[] = $this->csv_escape_row([
            $row['image_url'],
            $row['current_alt'] ?? '',
            '',
            0,
            'Skipped - No new alt text',
        ]);
    }

    return implode("\n", $lines);
}

/**
 * Generate results CSV
 */
private function generate_bulk_results_csv($bulk_data) {
    $lines = [];
    $lines[] = 'Image URL,Original Alt Text,New Alt Text,Pages Updated,Status';

    foreach ($bulk_data['fixed_images'] ?? [] as $row) {
        $lines[] = $this->csv_escape_row([
            $row['image_url'],
            $row['original_alt'] ?? '',
            $row['is_decorative'] ? '[DECORATIVE]' : $row['new_alt'],
            $row['pages_updated'] ?? 0,
            'Updated',
        ]);
    }

    foreach ($bulk_data['failed_updates'] ?? [] as $row) {
        $lines[] = $this->csv_escape_row([
            $row['image_url'],
            '',
            '',
            0,
            'Failed - ' . ($row['error'] ?? 'Unknown error'),
        ]);
    }

    foreach ($bulk_data['not_matched'] ?? [] as $row) {
        $lines[] = $this->csv_escape_row([
            $row['image_url'],
            '',
            $row['new_alt'] ?? '',
            0,
            'Skipped - Not found',
        ]);
    }

    return implode("\n", $lines);
}

/**
 * Escape a row for CSV output
 */
private function csv_escape_row($fields) {
    $escaped = [];
    foreach ($fields as $field) {
        $field = (string) $field;
        if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
            $field = '"' . str_replace('"', '""', $field) . '"';
        }
        $escaped[] = $field;
    }
    return implode(',', $escaped);
}
```

---

## Phase 2: Frontend JavaScript (image-alt-text.js)

### 2.1 Add Bulk Mode Detection and UI Switching

Add to `handleProcessSuccess` method to detect bulk mode:

```javascript
handleProcessSuccess: function(response) {
    var self = this;

    // Check if this is a bulk update response
    if (response.data && response.data.is_bulk_update) {
        self.showBulkConfirmation(response.data);
        return;
    }

    // Existing analysis display logic...
},
```

### 2.2 Add Bulk Confirmation Display Method

```javascript
showBulkConfirmation: function(data) {
    var self = this;

    // Hide analysis sections, show bulk confirmation
    $('.sf-fixable-section, .sf-manual-section, .sf-skipped-section').hide();

    // Build and show bulk confirmation UI
    var $confirmation = $('#sf-bulk-confirmation');

    // Update stats
    $confirmation.find('.sf-bulk-stat-ready .sf-bulk-stat-number').text(data.ready_count || 0);
    $confirmation.find('.sf-bulk-stat-notmatched .sf-bulk-stat-number').text(data.not_matched_count || 0);
    $confirmation.find('.sf-bulk-stat-skipped .sf-bulk-stat-number').text(data.skipped_empty_count || 0);

    // Show/hide warning sections
    if (data.duplicates_count > 0) {
        $confirmation.find('.sf-bulk-duplicates-warning')
            .show()
            .find('.sf-warning-count').text(data.duplicates_count);
    } else {
        $confirmation.find('.sf-bulk-duplicates-warning').hide();
    }

    if (data.filename_match_count > 0) {
        $confirmation.find('.sf-bulk-filename-warning')
            .show()
            .find('.sf-warning-count').text(data.filename_match_count);
    } else {
        $confirmation.find('.sf-bulk-filename-warning').hide();
    }

    if (data.multi_page_count > 0) {
        $confirmation.find('.sf-bulk-multipage-info')
            .show()
            .find('.sf-info-count').text(data.multi_page_count);
    } else {
        $confirmation.find('.sf-bulk-multipage-info').hide();
    }

    // Large file warning
    if (data.ready_count > 500) {
        $confirmation.find('.sf-bulk-large-warning').show();
    } else {
        $confirmation.find('.sf-bulk-large-warning').hide();
    }

    // Build preview table (first 10 rows)
    var $tbody = $confirmation.find('.sf-bulk-preview-table tbody');
    $tbody.empty();

    var preview = (data.ready_updates || []).slice(0, 10);
    preview.forEach(function(item) {
        var displayAlt = item.is_decorative ? '[DECORATIVE]' : item.new_alt;
        var $row = $('<tr class="sf-bulk-row">' +
            '<td class="sf-url-cell" title="' + self.escapeHtml(item.image_url) + '">' +
                self.truncate(item.image_url, 40) + '</td>' +
            '<td class="sf-alt-cell" title="' + self.escapeHtml(item.current_alt || '') + '">' +
                self.truncate(item.current_alt || '(none)', 60) + '</td>' +
            '<td class="sf-alt-cell" title="' + self.escapeHtml(displayAlt) + '">' +
                self.truncate(displayAlt, 60) + '</td>' +
            '<td>' + (item.pages_count || 0) + '</td>' +
            '<td><span class="sf-status-badge sf-status-ready">Ready</span></td>' +
        '</tr>');
        $tbody.append($row);
    });

    // Show count indicator
    if (data.ready_count > 10) {
        $confirmation.find('.sf-preview-count').text('(showing 10 of ' + data.ready_count + ')').show();
    } else {
        $confirmation.find('.sf-preview-count').hide();
    }

    $confirmation.show();

    // Store data for later use
    self.bulkData = data;
},
```

### 2.3 Add Bulk Processing Methods

```javascript
processBulkUpdate: function() {
    var self = this;

    if (self.bulkProcessing) {
        return;
    }

    self.bulkProcessing = true;

    var readyCount = self.bulkData ? self.bulkData.ready_count : 0;

    if (readyCount === 0) {
        ScreamingFixes.Toast.warning('No matching images found to update.');
        self.bulkProcessing = false;
        return;
    }

    // Show progress modal
    var $modal = $('#sf-bulk-progress-modal');
    $modal.find('.sf-bulk-progress-total').text(readyCount);
    $modal.find('.sf-bulk-progress-current').text(0);
    $modal.find('.sf-bulk-progress-percent').text('0');
    $modal.find('.sf-progress-fill').css('width', '0%');
    $modal.find('.sf-bulk-current-url').text('Starting...');
    $modal.show();

    $('.sf-bulk-confirm').prop('disabled', true);

    // Start processing in batches
    self.processBulkBatch(0, readyCount);
},

processBulkBatch: function(offset, total) {
    var self = this;
    var batchSize = 50;

    $.ajax({
        url: sfImageAltTextData.ajaxUrl,
        type: 'POST',
        data: {
            action: 'sf_image_alt_text_apply_bulk_updates',
            nonce: sfImageAltTextData.nonce,
            offset: offset,
            batch_size: batchSize
        },
        success: function(response) {
            if (response.success) {
                var data = response.data;
                var processed = data.processed || 0;
                var currentOffset = offset + processed;
                var percent = Math.min(100, Math.round((currentOffset / total) * 100));

                // Update progress UI
                var $modal = $('#sf-bulk-progress-modal');
                $modal.find('.sf-bulk-progress-current').text(currentOffset);
                $modal.find('.sf-bulk-progress-percent').text(percent);
                $modal.find('.sf-progress-fill').css('width', percent + '%');

                // Update current item indicator
                if (data.details && data.details.length > 0) {
                    var lastItem = data.details[data.details.length - 1];
                    $modal.find('.sf-bulk-current-url').text(
                        'Updated: ' + self.truncate(lastItem.image_url, 50)
                    );
                }

                if (data.complete) {
                    self.showBulkResults({
                        success: data.total_success || data.success || 0,
                        failed: data.total_failed || data.failed || 0,
                        total_pages: data.total_pages_updated || 0,
                        details: data.all_details || data.details || []
                    });
                } else {
                    // Continue with next batch
                    self.processBulkBatch(currentOffset, total);
                }
            } else {
                self.bulkProcessing = false;
                $('#sf-bulk-progress-modal').hide();
                $('.sf-bulk-confirm').prop('disabled', false);
                ScreamingFixes.Toast.error(response.data?.message || 'Bulk update failed.');
            }
        },
        error: function() {
            self.bulkProcessing = false;
            $('#sf-bulk-progress-modal').hide();
            $('.sf-bulk-confirm').prop('disabled', false);
            ScreamingFixes.Toast.error('Bulk update failed. Please try again.');
        }
    });
},
```

### 2.4 Add Results Display Method

```javascript
showBulkResults: function(results) {
    var self = this;

    self.bulkProcessing = false;
    $('#sf-bulk-progress-modal').hide();
    $('#sf-bulk-confirmation').hide();

    var $results = $('#sf-bulk-complete');

    // Update header message
    var message = results.success + ' images updated successfully.';
    if (results.failed > 0) {
        message += ' ' + results.failed + ' failed.';
    }
    $results.find('.sf-bulk-complete-message').text(message);

    // Update icon
    $results.find('.sf-bulk-complete-icon').text(results.failed > 0 ? '⚠️' : '✓');

    // Update stats
    $results.find('.sf-bulk-result-success .sf-bulk-result-value').text(results.success);
    $results.find('.sf-bulk-result-failed .sf-bulk-result-value').text(results.failed);

    // Show pages info if applicable
    if (results.total_pages > results.success) {
        $results.find('.sf-bulk-pages-info')
            .show()
            .find('.sf-pages-count').text(results.total_pages);
    } else {
        $results.find('.sf-bulk-pages-info').hide();
    }

    // Show/hide failed section
    if (results.failed > 0) {
        $results.find('.sf-bulk-result-failed').show();
    } else {
        $results.find('.sf-bulk-result-failed').hide();
    }

    // Build results preview table (first 10)
    var $tbody = $results.find('.sf-bulk-results-table tbody');
    $tbody.empty();

    var preview = (results.details || []).slice(0, 10);
    preview.forEach(function(item) {
        var displayAlt = item.is_decorative ? '[DECORATIVE]' : item.new_alt;
        var $row = $('<tr class="sf-bulk-row sf-bulk-row-success">' +
            '<td class="sf-url-cell" title="' + self.escapeHtml(item.image_url) + '">' +
                self.truncate(item.image_url, 40) + '</td>' +
            '<td class="sf-alt-cell" title="' + self.escapeHtml(item.original_alt || '') + '">' +
                self.truncate(item.original_alt || '(none)', 60) + '</td>' +
            '<td class="sf-alt-cell" title="' + self.escapeHtml(displayAlt) + '">' +
                self.truncate(displayAlt, 60) + '</td>' +
            '<td>' + (item.pages_updated || 0) + '</td>' +
            '<td><span class="sf-status-badge sf-status-success">✓ Updated</span></td>' +
        '</tr>');
        $tbody.append($row);
    });

    // Show count indicator
    if (results.details && results.details.length > 10) {
        $results.find('.sf-preview-count').text('(showing 10 of ' + results.details.length + ')').show();
    } else {
        $results.find('.sf-preview-count').hide();
    }

    $results.show();
},
```

### 2.5 Add Event Handlers

Add to `bindEvents` method:

```javascript
// Bulk confirmation buttons
$(document).on('click', '.sf-bulk-confirm', function() {
    self.processBulkUpdate();
});

$(document).on('click', '.sf-bulk-clear-btn', function() {
    self.clearBulkData();
});

$(document).on('click', '.sf-bulk-download-preview', function() {
    self.downloadBulkPreview();
});

$(document).on('click', '.sf-bulk-download-results', function() {
    self.downloadBulkResults();
});

$(document).on('click', '.sf-bulk-new-upload', function() {
    self.resetForNewUpload();
});
```

### 2.6 Add Helper Methods

```javascript
truncate: function(str, maxLen) {
    if (!str) return '';
    str = String(str);
    if (str.length <= maxLen) return str;
    return str.substring(0, maxLen) + '...';
},

escapeHtml: function(str) {
    if (!str) return '';
    var div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
},

downloadBulkPreview: function() {
    var self = this;

    $.ajax({
        url: sfImageAltTextData.ajaxUrl,
        type: 'POST',
        data: {
            action: 'sf_image_alt_text_download_bulk_preview',
            nonce: sfImageAltTextData.nonce
        },
        success: function(response) {
            if (response.success) {
                self.downloadCSV(response.data.csv, response.data.filename);
            } else {
                ScreamingFixes.Toast.error('Failed to generate preview CSV.');
            }
        }
    });
},

downloadBulkResults: function() {
    var self = this;

    $.ajax({
        url: sfImageAltTextData.ajaxUrl,
        type: 'POST',
        data: {
            action: 'sf_image_alt_text_download_bulk_results',
            nonce: sfImageAltTextData.nonce
        },
        success: function(response) {
            if (response.success) {
                self.downloadCSV(response.data.csv, response.data.filename);
            } else {
                ScreamingFixes.Toast.error('Failed to generate results CSV.');
            }
        }
    });
},

downloadCSV: function(csv, filename) {
    var blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    var link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = filename;
    link.click();
},

clearBulkData: function() {
    $('#sf-bulk-confirmation').hide();
    $('#sf-bulk-complete').hide();
    this.bulkData = null;
    ScreamingFixes.Toast.info('Bulk data cleared. Upload a new CSV to continue.');
},

resetForNewUpload: function() {
    this.clearBulkData();
    // Scroll to upload area or reset form
    $('.sf-upload-section').show();
    $('html, body').animate({
        scrollTop: $('.sf-upload-section').offset().top - 100
    }, 300);
},
```

---

## Phase 3: View Templates (tab-content.php)

### 3.1 Add Bulk Confirmation Section

Add after the stats section, before fixable images section:

```php
<?php
// Check if we have bulk update data
$bulk_data = get_transient('sf_image_bulk_data_' . get_current_user_id());
$is_bulk_mode = !empty($bulk_data) && !empty($bulk_data['is_bulk_update']);
$bulk_complete = !empty($bulk_data['bulk_complete']);
?>

<!-- Bulk Update Confirmation Section -->
<div class="sf-bulk-confirmation" id="sf-bulk-confirmation" style="display: none;">
    <div class="sf-bulk-header">
        <div class="sf-bulk-icon">📋</div>
        <h3><?php esc_html_e('Bulk Image Alt Text Update', 'screaming-fixes'); ?></h3>
    </div>

    <p class="sf-bulk-notice">
        <?php esc_html_e('Rows with empty values will be skipped. Use [DECORATIVE] to explicitly set empty alt text for decorative images.', 'screaming-fixes'); ?>
    </p>

    <div class="sf-bulk-stats">
        <div class="sf-bulk-stat sf-bulk-stat-ready">
            <div class="sf-bulk-stat-icon">✓</div>
            <div class="sf-bulk-stat-content">
                <span class="sf-bulk-stat-number">0</span>
                <span class="sf-bulk-stat-label"><?php esc_html_e('images ready to update', 'screaming-fixes'); ?></span>
            </div>
        </div>

        <div class="sf-bulk-stat sf-bulk-stat-notmatched" style="display: none;">
            <div class="sf-bulk-stat-icon">⚠️</div>
            <div class="sf-bulk-stat-content">
                <span class="sf-bulk-stat-number">0</span>
                <span class="sf-bulk-stat-label"><?php esc_html_e('images not matched', 'screaming-fixes'); ?></span>
            </div>
        </div>

        <div class="sf-bulk-stat sf-bulk-stat-skipped" style="display: none;">
            <div class="sf-bulk-stat-icon">—</div>
            <div class="sf-bulk-stat-content">
                <span class="sf-bulk-stat-number">0</span>
                <span class="sf-bulk-stat-label"><?php esc_html_e('rows skipped - no new alt text', 'screaming-fixes'); ?></span>
            </div>
        </div>
    </div>

    <!-- Warning Messages -->
    <div class="sf-bulk-duplicates-warning sf-bulk-warning" style="display: none;">
        <span class="dashicons dashicons-warning"></span>
        <span class="sf-warning-count">0</span> <?php esc_html_e('duplicate entries detected - using last occurrence for each. You may want to review your CSV.', 'screaming-fixes'); ?>
    </div>

    <div class="sf-bulk-filename-warning sf-bulk-warning" style="display: none;">
        <span class="dashicons dashicons-info"></span>
        <span class="sf-warning-count">0</span> <?php esc_html_e('images matched by filename only - verify these are the correct images.', 'screaming-fixes'); ?>
    </div>

    <div class="sf-bulk-multipage-info sf-bulk-info" style="display: none;">
        <span class="dashicons dashicons-info"></span>
        <span class="sf-info-count">0</span> <?php esc_html_e('images appear on multiple pages - alt text will be updated on all instances.', 'screaming-fixes'); ?>
    </div>

    <div class="sf-bulk-large-warning sf-bulk-warning" style="display: none;">
        <span class="dashicons dashicons-warning"></span>
        <?php esc_html_e('Large file detected. For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'); ?>
    </div>

    <!-- Preview Table -->
    <div class="sf-bulk-preview-section">
        <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?> <span class="sf-preview-count"></span></h4>
        <table class="sf-bulk-preview-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('Image URL', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('Current Alt Text', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('New Alt Text', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('Pages', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                </tr>
            </thead>
            <tbody>
                <!-- Populated by JavaScript -->
            </tbody>
        </table>
    </div>

    <!-- Action Buttons -->
    <div class="sf-bulk-actions">
        <button type="button" class="sf-button sf-button-secondary sf-bulk-download-preview">
            <span class="dashicons dashicons-download"></span>
            <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
        </button>
        <div class="sf-bulk-buttons">
            <button type="button" class="sf-button sf-button-secondary sf-bulk-clear-btn">
                <?php esc_html_e('Clear', 'screaming-fixes'); ?>
            </button>
            <button type="button" class="sf-button sf-button-primary sf-bulk-confirm">
                <?php esc_html_e('Confirm Update', 'screaming-fixes'); ?>
            </button>
        </div>
    </div>
</div>

<!-- Progress Modal -->
<div class="sf-bulk-progress-modal sf-modal" id="sf-bulk-progress-modal" style="display: none;">
    <div class="sf-bulk-progress-content">
        <div class="sf-bulk-progress-header">
            <span class="sf-bulk-progress-icon sf-spinning">⟳</span>
            <h3><?php esc_html_e('Updating Image Alt Text...', 'screaming-fixes'); ?></h3>
        </div>
        <div class="sf-bulk-progress-bar">
            <div class="sf-progress-fill" style="width: 0%;"></div>
        </div>
        <div class="sf-bulk-progress-stats">
            <span class="sf-bulk-progress-current">0</span> / <span class="sf-bulk-progress-total">0</span>
            (<span class="sf-bulk-progress-percent">0</span>%)
        </div>
        <div class="sf-bulk-current-url"></div>
    </div>
</div>

<!-- Bulk Complete Results Section -->
<div class="sf-bulk-complete" id="sf-bulk-complete" style="display: none;">
    <div class="sf-bulk-complete-header">
        <div class="sf-bulk-complete-icon">✓</div>
        <h3 class="sf-bulk-complete-message"></h3>
    </div>

    <div class="sf-bulk-pages-info" style="display: none;">
        <?php esc_html_e('Updates applied across', 'screaming-fixes'); ?> <span class="sf-pages-count">0</span> <?php esc_html_e('total page instances.', 'screaming-fixes'); ?>
    </div>

    <div class="sf-bulk-results-summary">
        <div class="sf-bulk-result-stat sf-bulk-result-success">
            <span class="sf-bulk-result-value">0</span>
            <span class="sf-bulk-result-label"><?php esc_html_e('Updated', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-bulk-result-stat sf-bulk-result-failed" style="display: none;">
            <span class="sf-bulk-result-value">0</span>
            <span class="sf-bulk-result-label"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
        </div>
    </div>

    <div class="sf-bulk-complete-actions">
        <button type="button" class="sf-button sf-button-secondary sf-bulk-download-results">
            <span class="dashicons dashicons-download"></span>
            <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
        </button>
        <button type="button" class="sf-button sf-button-primary sf-bulk-new-upload">
            <?php esc_html_e('Upload New CSV', 'screaming-fixes'); ?>
        </button>
    </div>

    <!-- Results Table -->
    <div class="sf-bulk-results-table-section">
        <h4><?php esc_html_e('Updated Images', 'screaming-fixes'); ?> <span class="sf-preview-count"></span></h4>
        <table class="sf-bulk-results-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('Original Alt Text', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('New Alt Text', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('Pages Updated', 'screaming-fixes'); ?></th>
                    <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                </tr>
            </thead>
            <tbody>
                <!-- Populated by JavaScript -->
            </tbody>
        </table>
    </div>
</div>
```

---

## Phase 4: Instructions Update (instructions.php)

### 4.1 Add Bulk Upload Section

Add after the existing steps, before the closing tags:

```php
<div class="sf-instructions-divider"></div>

<div class="sf-instructions-section sf-bulk-instructions">
    <h4><?php esc_html_e('Prefer working in spreadsheets?', 'screaming-fixes'); ?></h4>

    <p><?php esc_html_e('Add a New_Alt_Text column to your Screaming Frog export to specify replacement alt text directly in the spreadsheet. This lets you do all your work in Excel or Google Sheets before uploading.', 'screaming-fixes'); ?></p>

    <p><?php esc_html_e('Just drop the CSV into this tool and Screaming Fixes will automatically detect the format and apply your updates.', 'screaming-fixes'); ?></p>

    <div class="sf-example-table">
        <h5><?php esc_html_e('Example:', 'screaming-fixes'); ?></h5>
        <table class="sf-example-csv">
            <thead>
                <tr>
                    <th>Address</th>
                    <th>Alt Text</th>
                    <th>New_Alt_Text</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>/uploads/hero.jpg</td>
                    <td>old description</td>
                    <td>New descriptive alt text</td>
                </tr>
                <tr>
                    <td>/uploads/decorative-border.png</td>
                    <td>border</td>
                    <td>[DECORATIVE]</td>
                </tr>
                <tr>
                    <td>/uploads/product.jpg</td>
                    <td></td>
                    <td>Product name on white background</td>
                </tr>
            </tbody>
        </table>
    </div>

    <div class="sf-bulk-tips">
        <p><strong><?php esc_html_e('Tips:', 'screaming-fixes'); ?></strong></p>
        <ul>
            <li><?php esc_html_e('Rows without a New_Alt_Text value will be skipped.', 'screaming-fixes'); ?></li>
            <li><?php esc_html_e('Use [DECORATIVE] to explicitly set empty alt text for decorative images (per WCAG accessibility guidelines).', 'screaming-fixes'); ?></li>
            <li><?php esc_html_e('If an image appears on multiple pages, alt text will be updated on all instances.', 'screaming-fixes'); ?></li>
        </ul>
    </div>
</div>
```

---

## Phase 5: CSS Styling (image-alt-text.css)

### 5.1 Add Bulk-Specific Styles

Add at the end of the file:

```css
/* ============================================
   BULK UPDATE STYLES
   ============================================ */

/* Bulk Confirmation Section */
.sf-bulk-confirmation {
    background: white;
    border: 1px solid var(--sf-gray-200, #e5e7eb);
    border-radius: 12px;
    padding: 24px;
    margin-bottom: 24px;
}

.sf-bulk-header {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 16px;
}

.sf-bulk-icon {
    font-size: 28px;
}

.sf-bulk-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
    color: var(--sf-gray-900, #111827);
}

.sf-bulk-notice {
    background: var(--sf-info-bg, #eff6ff);
    border: 1px solid var(--sf-info-border, #bfdbfe);
    color: var(--sf-info-text, #1e40af);
    padding: 12px 16px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-size: 13px;
}

/* Bulk Stats */
.sf-bulk-stats {
    display: flex;
    gap: 16px;
    margin-bottom: 20px;
    flex-wrap: wrap;
}

.sf-bulk-stat {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 16px 20px;
    background: var(--sf-gray-50, #f9fafb);
    border: 1px solid var(--sf-gray-200, #e5e7eb);
    border-radius: 10px;
    min-width: 140px;
}

.sf-bulk-stat-icon {
    font-size: 20px;
    width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 8px;
    background: white;
}

.sf-bulk-stat-ready .sf-bulk-stat-icon {
    background: var(--sf-success-bg, #f0fdf4);
    color: var(--sf-success, #22c55e);
}

.sf-bulk-stat-notmatched .sf-bulk-stat-icon {
    background: var(--sf-warning-bg, #fffbeb);
    color: var(--sf-warning, #f59e0b);
}

.sf-bulk-stat-skipped .sf-bulk-stat-icon {
    background: var(--sf-gray-100, #f3f4f6);
    color: var(--sf-gray-500, #6b7280);
}

.sf-bulk-stat-content {
    display: flex;
    flex-direction: column;
}

.sf-bulk-stat-number {
    font-size: 24px;
    font-weight: 700;
    color: var(--sf-gray-900, #111827);
    line-height: 1;
}

.sf-bulk-stat-label {
    font-size: 12px;
    color: var(--sf-gray-600, #4b5563);
    margin-top: 4px;
}

/* Warnings and Info Messages */
.sf-bulk-warning,
.sf-bulk-info {
    display: flex;
    align-items: flex-start;
    gap: 10px;
    padding: 12px 16px;
    border-radius: 8px;
    margin-bottom: 12px;
    font-size: 13px;
}

.sf-bulk-warning {
    background: #fef3c7;
    border: 1px solid #f59e0b;
    color: #92400e;
}

.sf-bulk-info {
    background: #eff6ff;
    border: 1px solid #bfdbfe;
    color: #1e40af;
}

.sf-bulk-warning .dashicons,
.sf-bulk-info .dashicons {
    font-size: 16px;
    width: 16px;
    height: 16px;
    margin-top: 2px;
}

/* Preview Table */
.sf-bulk-preview-section {
    margin-top: 20px;
}

.sf-bulk-preview-section h4 {
    margin: 0 0 12px 0;
    font-size: 14px;
    font-weight: 600;
    color: var(--sf-gray-700, #374151);
}

.sf-preview-count {
    font-weight: 400;
    color: var(--sf-gray-500, #6b7280);
    font-size: 12px;
}

.sf-bulk-preview-table,
.sf-bulk-results-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 13px;
}

.sf-bulk-preview-table th,
.sf-bulk-results-table th {
    text-align: left;
    padding: 10px 12px;
    background: var(--sf-gray-50, #f9fafb);
    border: 1px solid var(--sf-gray-200, #e5e7eb);
    font-weight: 600;
    font-size: 11px;
    text-transform: uppercase;
    color: var(--sf-gray-600, #4b5563);
}

.sf-bulk-preview-table td,
.sf-bulk-results-table td {
    padding: 10px 12px;
    border: 1px solid var(--sf-gray-200, #e5e7eb);
    vertical-align: middle;
}

.sf-bulk-preview-table .sf-url-cell,
.sf-bulk-results-table .sf-url-cell {
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    font-family: monospace;
    font-size: 12px;
}

.sf-bulk-preview-table .sf-alt-cell,
.sf-bulk-results-table .sf-alt-cell {
    max-width: 180px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

/* Status Badges */
.sf-status-badge {
    display: inline-block;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 500;
}

.sf-status-ready {
    background: var(--sf-success-bg, #f0fdf4);
    color: var(--sf-success, #22c55e);
}

.sf-status-success {
    background: var(--sf-success-bg, #f0fdf4);
    color: var(--sf-success, #22c55e);
}

.sf-status-failed {
    background: var(--sf-error-bg, #fef2f2);
    color: var(--sf-error, #ef4444);
}

.sf-status-skipped {
    background: var(--sf-gray-100, #f3f4f6);
    color: var(--sf-gray-600, #4b5563);
}

/* Action Buttons */
.sf-bulk-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid var(--sf-gray-200, #e5e7eb);
}

.sf-bulk-buttons {
    display: flex;
    gap: 12px;
}

/* Progress Modal */
.sf-bulk-progress-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    z-index: 100000;
    display: flex;
    align-items: center;
    justify-content: center;
}

.sf-bulk-progress-content {
    background: white;
    border-radius: 12px;
    padding: 32px;
    width: 400px;
    max-width: 90%;
    text-align: center;
}

.sf-bulk-progress-header {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
    margin-bottom: 24px;
}

.sf-bulk-progress-icon {
    font-size: 24px;
}

.sf-bulk-progress-icon.sf-spinning {
    animation: sf-spin 1s linear infinite;
}

@keyframes sf-spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.sf-bulk-progress-header h3 {
    margin: 0;
    font-size: 16px;
    font-weight: 600;
}

.sf-bulk-progress-bar {
    height: 10px;
    background: var(--sf-gray-200, #e5e7eb);
    border-radius: 10px;
    overflow: hidden;
    margin-bottom: 12px;
}

.sf-bulk-progress-bar .sf-progress-fill {
    height: 100%;
    background: linear-gradient(90deg, var(--sf-primary, #14b8a6), var(--sf-primary-light, #2dd4bf));
    border-radius: 10px;
    transition: width 0.3s ease;
}

.sf-bulk-progress-stats {
    font-size: 14px;
    color: var(--sf-gray-700, #374151);
    margin-bottom: 8px;
}

.sf-bulk-current-url {
    font-size: 12px;
    color: var(--sf-gray-500, #6b7280);
    font-family: monospace;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Bulk Complete Section */
.sf-bulk-complete {
    background: white;
    border: 1px solid var(--sf-gray-200, #e5e7eb);
    border-radius: 12px;
    padding: 24px;
    margin-bottom: 24px;
}

.sf-bulk-complete-header {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 16px;
}

.sf-bulk-complete-icon {
    font-size: 32px;
    width: 48px;
    height: 48px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: var(--sf-success-bg, #f0fdf4);
    border-radius: 50%;
}

.sf-bulk-complete-message {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
    color: var(--sf-gray-900, #111827);
}

.sf-bulk-pages-info {
    background: var(--sf-info-bg, #eff6ff);
    padding: 12px 16px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-size: 13px;
    color: var(--sf-info-text, #1e40af);
}

.sf-bulk-results-summary {
    display: flex;
    gap: 16px;
    margin-bottom: 20px;
}

.sf-bulk-result-stat {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: 8px;
}

.sf-bulk-result-success {
    background: var(--sf-success-bg, #f0fdf4);
}

.sf-bulk-result-failed {
    background: var(--sf-error-bg, #fef2f2);
}

.sf-bulk-result-value {
    font-size: 24px;
    font-weight: 700;
}

.sf-bulk-result-success .sf-bulk-result-value {
    color: var(--sf-success, #22c55e);
}

.sf-bulk-result-failed .sf-bulk-result-value {
    color: var(--sf-error, #ef4444);
}

.sf-bulk-result-label {
    font-size: 13px;
    color: var(--sf-gray-600, #4b5563);
}

.sf-bulk-complete-actions {
    display: flex;
    gap: 12px;
    margin-bottom: 24px;
}

.sf-bulk-results-table-section {
    margin-top: 20px;
}

.sf-bulk-results-table-section h4 {
    margin: 0 0 12px 0;
    font-size: 14px;
    font-weight: 600;
}

.sf-bulk-row-success {
    background: var(--sf-success-bg, #f0fdf4);
}

.sf-bulk-row-failed {
    background: var(--sf-error-bg, #fef2f2);
}

/* Instructions Bulk Section */
.sf-instructions-divider {
    border-top: 1px solid var(--sf-gray-200, #e5e7eb);
    margin: 24px 0;
}

.sf-bulk-instructions h4 {
    margin: 0 0 12px 0;
    font-size: 15px;
    font-weight: 600;
}

.sf-example-table {
    margin: 16px 0;
}

.sf-example-table h5 {
    margin: 0 0 8px 0;
    font-size: 13px;
    font-weight: 600;
    color: var(--sf-gray-700, #374151);
}

.sf-example-csv {
    width: 100%;
    border-collapse: collapse;
    font-size: 12px;
    font-family: monospace;
}

.sf-example-csv th,
.sf-example-csv td {
    padding: 8px 12px;
    border: 1px solid var(--sf-gray-200, #e5e7eb);
    text-align: left;
}

.sf-example-csv th {
    background: var(--sf-gray-50, #f9fafb);
    font-weight: 600;
}

.sf-bulk-tips {
    margin-top: 16px;
}

.sf-bulk-tips p {
    margin: 0 0 8px 0;
    font-size: 13px;
}

.sf-bulk-tips ul {
    margin: 0;
    padding-left: 20px;
}

.sf-bulk-tips li {
    margin-bottom: 4px;
    font-size: 13px;
    color: var(--sf-gray-600, #4b5563);
}

/* Responsive */
@media (max-width: 768px) {
    .sf-bulk-stats {
        flex-direction: column;
    }

    .sf-bulk-stat {
        width: 100%;
    }

    .sf-bulk-actions {
        flex-direction: column;
        gap: 12px;
    }

    .sf-bulk-buttons {
        width: 100%;
        justify-content: flex-end;
    }

    .sf-bulk-preview-table,
    .sf-bulk-results-table {
        font-size: 11px;
    }

    .sf-bulk-preview-table .sf-url-cell,
    .sf-bulk-results-table .sf-url-cell {
        max-width: 120px;
    }
}
```

---

## Testing Checklist

### Unit Tests
- [ ] Column detection with various header formats
- [ ] URL normalization (http/https, trailing slashes, www)
- [ ] Filename extraction and size suffix stripping
- [ ] [DECORATIVE] keyword handling
- [ ] Duplicate row handling (last occurrence wins)
- [ ] Image matching (exact URL vs filename fallback)

### Integration Tests
- [ ] Full CSV upload → detection → processing flow
- [ ] Batch processing with 100+ images
- [ ] Progress tracking accuracy
- [ ] Results CSV generation
- [ ] Undo capability after bulk update

### UI Tests
- [ ] Stats display correctly
- [ ] Warning messages appear when applicable
- [ ] Preview table shows first 10 rows
- [ ] Progress modal updates smoothly
- [ ] Results display after completion
- [ ] Download buttons work
- [ ] Clear/reset functionality

### Edge Cases
- [ ] Empty CSV file
- [ ] CSV with only headers
- [ ] All rows have empty new_alt values
- [ ] All images not matched
- [ ] Mixed: some matched, some not
- [ ] Very long alt text values
- [ ] Special characters in alt text
- [ ] Images with multiple size variants
- [ ] Same image on 10+ pages

---

## Implementation Order

1. **Phase 1.1-1.3**: Add column detection and modify ajax_process_csv
2. **Phase 1.4-1.5**: Add bulk processing and image matching
3. **Phase 1.6-1.8**: Add batch updates and CSV downloads
4. **Phase 2**: JavaScript handlers and UI logic
5. **Phase 3**: View templates for confirmation/results
6. **Phase 4**: Instructions update
7. **Phase 5**: CSS styling
8. **Testing**: Full test suite

Estimated scope: ~800-1000 lines of new code across all files.
