# Image Alt Text - Batch Undo Implementation Plan

## Overview

This plan implements batch undo functionality for the Image Alt Text module, following the patterns established in Broken Links and Redirect Chains modules.

## UX Lessons Applied (from Broken Links & Redirect Chains)

1. **Show fixed items immediately** - After applying fixes, display them in a "Fixed Images" section
2. **Clear fixed items on undo** - When a batch is undone, remove those items from the "Fixed Images" display
3. **Store fixed URLs in batch metadata** - Track which image URLs were fixed for cleanup on undo
4. **Module-specific batch descriptions** - Already exists: "X alt texts updated"
5. **Consistent UI patterns** - Same collapsible section styling as other modules

## Current State Analysis

### Image Alt Text Module (`class-image-alt-text.php`)

**Key differences from other modules:**

| Aspect | Broken Links | Redirect Chains | Image Alt Text |
|--------|--------------|-----------------|----------------|
| URL field | `broken_url` | `address` | `image_url` |
| Fixed array | `fixed_links` | `fixed_redirects` | (to add) `fixed_images` |
| Fix action | `set_alt` / `ignore` | (none - all fixes) | `set_alt` / `ignore` |
| Extra data | `new_url` | `final_url` | `new_alt` |

**apply_fixes() method (lines 627-693):**
- Iterates through fixes
- Calls `update_image_alt_in_post()` for each
- Calls `update_results_after_fixes()` at end
- **Missing:** `SF_Batch_Restore::start_batch()` and `complete_batch()`

**update_results_after_fixes() method (lines 1221-1257):**
- Currently removes fixed images from results
- **Missing:** Adding to `fixed_images` array (like other modules)

### tab-content.php

**Current data check (line 153):**
```php
$has_data = !empty($results) && !empty($results['images']);
```
**Missing:** Check for `fixed_images` array

**Missing:** Fixed Images UI section (collapsible, like other modules)

### class-batch-restore.php

**start_batch() (lines 42-122):**
- Already handles `image_url` field (via fallback chain: `$fix['broken_url'] ?? $fix['url'] ?? $fix['address'] ?? ''`)
- Need to add explicit `image_url` support for clarity

**clear_fixed_links_for_batch() (lines 384-512):**
- Handles `broken-links` and `redirect-chains`
- **Missing:** `image-alt-text` case

**get_batch_description() (lines 596-632):**
- Already has `image-alt-text` case returning "X alt texts updated"

---

## Implementation Steps

### Step 1: Update `class-batch-restore.php`

#### 1a. Update `start_batch()` - Add explicit `image_url` support

In the URL collection loop (~line 101), add `image_url` to the fallback chain:

```php
// Store the URL that was fixed
// broken-links uses: broken_url or url
// redirect-chains uses: address
// image-alt-text uses: image_url
$url = $fix['broken_url'] ?? $fix['url'] ?? $fix['address'] ?? $fix['image_url'] ?? '';
```

#### 1b. Add `image-alt-text` case to `clear_fixed_links_for_batch()`

Add a new `elseif` block after the `redirect-chains` case (~line 512):

```php
} elseif ($module === 'image-alt-text') {
    $module_instance = SF_Module_Loader::get_module('image-alt-text');
    if (!$module_instance) {
        return;
    }

    // Get current results from transient
    $results = $module_instance->get_results();

    // Also check database if transient is empty
    if (empty($results)) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            'image-alt-text'
        ));

        if ($row) {
            $results = json_decode($row->data, true);
        }
    }

    if (empty($results) || empty($results['fixed_images'])) {
        return;
    }

    // Remove the undone URLs from fixed_images
    $urls_to_remove = array_flip($fixed_urls);
    $results['fixed_images'] = array_filter($results['fixed_images'], function($image) use ($urls_to_remove) {
        $image_url = $image['image_url'] ?? '';
        return !isset($urls_to_remove[$image_url]);
    });
    $results['fixed_images'] = array_values($results['fixed_images']); // Re-index
    $results['fixed_images_count'] = count($results['fixed_images']);

    // Save updated results to both transient and database
    set_transient('sf_image-alt-text_results', $results, HOUR_IN_SECONDS);

    // Update database
    global $wpdb;
    $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
    $session_id = 'user_' . get_current_user_id();

    $wpdb->update(
        $table_name,
        ['data' => wp_json_encode($results)],
        [
            'session_id' => $session_id,
            'module' => 'image-alt-text',
        ],
        ['%s'],
        ['%s', '%s']
    );
}
```

---

### Step 2: Update `class-image-alt-text.php`

#### 2a. Update `apply_fixes()` method

Add batch restore integration at the beginning and end:

```php
public function apply_fixes($fixes) {
    $results = [
        'total' => 0,
        'success' => 0,
        'failed' => 0,
        'skipped' => 0,
        'errors' => [],
        'details' => [],
    ];

    // Start batch tracking for undo capability
    SF_Batch_Restore::start_batch('image-alt-text', $fixes);

    // ... existing fix logic ...

    // Update stored results to reflect fixes applied
    $this->update_results_after_fixes($fixes);

    // Complete batch tracking
    SF_Batch_Restore::complete_batch($results);

    return $results;
}
```

#### 2b. Rewrite `update_results_after_fixes()` method

Change from removing fixed images to tracking them in `fixed_images` array:

```php
private function update_results_after_fixes($fixes) {
    $results = $this->get_results();

    if (empty($results)) {
        // Try database
        $results = $this->get_upload_data();
    }

    if (empty($results) || empty($results['images'])) {
        return;
    }

    // Initialize fixed_images array if not exists
    if (!isset($results['fixed_images'])) {
        $results['fixed_images'] = [];
    }

    // Create a map of fixed images by URL
    $fixed_map = [];
    foreach ($fixes as $fix) {
        if (!empty($fix['new_alt']) && ($fix['action'] ?? 'set_alt') !== 'ignore') {
            $url = $fix['image_url'] ?? '';
            if (!empty($url)) {
                $fixed_map[$url] = $fix['new_alt'];
            }
        }
    }

    // Move fixed images from images array to fixed_images array
    $remaining_images = [];
    $fixable_images = [];
    $manual_images = [];
    $skipped_images = [];

    foreach ($results['images'] as $image) {
        $image_url = $image['image_url'] ?? '';

        if (isset($fixed_map[$image_url])) {
            // This image was fixed - add to fixed_images with the new alt text
            $image['applied_alt'] = $fixed_map[$image_url];
            $image['fixed_at'] = current_time('mysql');
            $results['fixed_images'][] = $image;
        } else {
            // Not fixed - keep in appropriate category
            $remaining_images[] = $image;
            $category = $image['overall_category'] ?? 'fixable';
            switch ($category) {
                case 'fixable':
                    $fixable_images[] = $image;
                    break;
                case 'manual':
                    $manual_images[] = $image;
                    break;
                case 'skip':
                    $skipped_images[] = $image;
                    break;
            }
        }
    }

    // Update results arrays
    $results['images'] = $remaining_images;
    $results['fixable_images'] = $fixable_images;
    $results['manual_images'] = $manual_images;
    $results['skipped_images'] = $skipped_images;

    // Update counts
    $results['total_count'] = count($remaining_images);
    $results['fixable_count'] = count($fixable_images);
    $results['manual_count'] = count($manual_images);
    $results['skipped_count'] = count($skipped_images);
    $results['fixed_images_count'] = count($results['fixed_images']);

    // Recalculate total sources
    $total_sources = 0;
    foreach ($remaining_images as $image) {
        $total_sources += $image['source_count'] ?? count($image['sources'] ?? []);
    }
    $results['total_sources'] = $total_sources;

    // Save updated results to both transient and database
    $this->save_results($results);
    $this->save_upload_data($results);
}
```

---

### Step 3: Update `tab-content.php`

#### 3a. Update `$has_data` logic (around line 153)

```php
// Check if we have any data (images OR fixed_images - data persists after fixes)
$has_results = !empty($results);
$has_images = $has_results && !empty($results['images']);
$has_fixed_images = $has_results && !empty($results['fixed_images']);
$has_data = $has_images || $has_fixed_images;
```

#### 3b. Get fixed images data (after line 164)

```php
// Get fixed images
$fixed_images = $has_data ? ($results['fixed_images'] ?? []) : [];
$fixed_images_count = count($fixed_images);
```

#### 3c. Add Fixed Images UI section (after Skipped Images section, before Progress Modal ~line 497)

Add a collapsible "Fixed Images" section using the same CSS classes as other modules:

```php
<?php if (!empty($fixed_images)): ?>
<!-- Section 4: Fixed Images (Collapsed by default) -->
<div class="sf-results-section sf-section-fixed">
    <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="false">
        <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
        <?php printf(
            esc_html__('%d Fixed Images', 'screaming-fixes'),
            $fixed_images_count
        ); ?>
        <span class="sf-section-hint"><?php esc_html_e('Alt text has been applied to these images', 'screaming-fixes'); ?></span>
        <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
    </button>

    <div class="sf-fixed-content" style="display: none;">
        <div class="sf-fixed-note">
            <span class="dashicons dashicons-yes-alt"></span>
            <?php esc_html_e('These images have had alt text applied. You can undo recent changes from the Settings page.', 'screaming-fixes'); ?>
        </div>
        <div class="sf-table-wrapper">
            <table class="sf-results-table sf-fixed-table">
                <thead>
                    <tr>
                        <th class="sf-col-image"><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                        <th class="sf-col-alt-applied"><?php esc_html_e('Alt Text Applied', 'screaming-fixes'); ?></th>
                        <th class="sf-col-sources"><?php esc_html_e('Updated In', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($fixed_images as $image):
                        $image_url = $image['image_url'] ?? '';
                        $filename = $image['filename'] ?? basename($image_url);
                        $applied_alt = $image['applied_alt'] ?? '';
                        $sources = $image['fixable_sources'] ?? $image['sources'] ?? [];
                        $source_count = count($sources);
                    ?>
                        <tr class="sf-image-row sf-fixed-row">
                            <td class="sf-col-image">
                                <div class="sf-image-cell">
                                    <div class="sf-image-preview sf-image-preview-small">
                                        <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($applied_alt); ?>" loading="lazy" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                        <div class="sf-image-placeholder" style="display: none;">
                                            <span class="dashicons dashicons-format-image"></span>
                                        </div>
                                    </div>
                                    <div class="sf-image-info">
                                        <span class="sf-image-filename" title="<?php echo esc_attr($image_url); ?>">
                                            <?php echo esc_html($filename); ?>
                                        </span>
                                    </div>
                                </div>
                            </td>
                            <td class="sf-col-alt-applied">
                                <span class="sf-applied-alt-text"><?php echo esc_html($applied_alt); ?></span>
                            </td>
                            <td class="sf-col-sources">
                                <?php if ($source_count === 1): ?>
                                    <span class="sf-source-count"><?php esc_html_e('1 page', 'screaming-fixes'); ?></span>
                                <?php else: ?>
                                    <span class="sf-source-count"><?php printf(esc_html__('%d pages', 'screaming-fixes'), $source_count); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php endif; ?>
```

---

## Summary of Changes

| File | Changes |
|------|---------|
| `class-batch-restore.php` | 1. Add `image_url` to URL fallback chain<br>2. Add `image-alt-text` case to `clear_fixed_links_for_batch()` |
| `class-image-alt-text.php` | 1. Add `SF_Batch_Restore::start_batch()` call<br>2. Add `SF_Batch_Restore::complete_batch()` call<br>3. Rewrite `update_results_after_fixes()` to track `fixed_images` |
| `tab-content.php` | 1. Update `$has_data` logic to include `fixed_images`<br>2. Add Fixed Images section UI |

---

## Testing Checklist

- [ ] Upload image alt text CSV and verify results display
- [ ] Apply fixes to multiple images and verify:
  - [ ] Fixed images appear in "Fixed Images" section immediately
  - [ ] Images are removed from "Fixable Images" section
  - [ ] Batch appears in Settings → Restore Points
  - [ ] Batch description shows "X alt texts updated"
- [ ] Navigate away and return - verify Fixed Images persist
- [ ] Undo batch from Settings and verify:
  - [ ] Posts are restored to previous state
  - [ ] Fixed Images section is cleared/updated
  - [ ] Batch is removed from Restore Points
  - [ ] Success message shows correct post count
- [ ] Test with mixed actions (some set_alt, some ignore)
- [ ] Test with images that have no revisions (new posts)
