# Implementation Plan: Internal Link Builder Module

## Overview

Build an Internal Link Builder module that helps users add internal links to a priority page. Two modes: Discovery Mode (find opportunities by scanning content) and Bulk Upload Mode (CSV with pre-defined links). Follows existing Screaming Fixes patterns.

---

## Files to Create

```
modules/internal-link-builder/
├── class-internal-link-builder.php    # Main module class (~800 lines)
├── views/
│   ├── tab-content.php                # Main UI (both modes) (~600 lines)
│   └── instructions.php               # How-to guide (~150 lines)
└── assets/
    ├── internal-link-builder.css      # Styles (~500 lines)
    └── internal-link-builder.js       # JS handlers (~700 lines)
```

---

## Phase 1: Module Registration & Structure

### 1.1 Create Module Class Shell

**File:** `modules/internal-link-builder/class-internal-link-builder.php`

```php
<?php
/**
 * Internal Link Builder Module
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Internal_Link_Builder {

    /**
     * Module identifier
     */
    private $module_id = 'internal-link-builder';

    /**
     * Module name
     */
    private $name = 'Internal Link Builder';

    /**
     * Module slug for URLs
     */
    private $slug = 'internal-link-builder';

    /**
     * Constructor
     */
    public function __construct() {
        $this->init();
    }

    /**
     * Initialize module
     */
    public function init() {
        // Discovery Mode AJAX handlers
        add_action('wp_ajax_sf_ilb_extract_keywords', [$this, 'ajax_extract_keywords']);
        add_action('wp_ajax_sf_ilb_scan_opportunities', [$this, 'ajax_scan_opportunities']);
        add_action('wp_ajax_sf_ilb_add_links', [$this, 'ajax_add_links']);

        // Bulk Upload Mode AJAX handlers
        add_action('wp_ajax_sf_ilb_process_csv', [$this, 'ajax_process_csv']);
        add_action('wp_ajax_sf_ilb_apply_bulk_links', [$this, 'ajax_apply_bulk_links']);

        // Common AJAX handlers
        add_action('wp_ajax_sf_ilb_download_preview', [$this, 'ajax_download_preview']);
        add_action('wp_ajax_sf_ilb_download_results', [$this, 'ajax_download_results']);
        add_action('wp_ajax_sf_ilb_clear_data', [$this, 'ajax_clear_data']);

        // Asset enqueuing
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module assets
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'internal-link-builder') {
            return;
        }

        wp_enqueue_style(
            'sf-internal-link-builder',
            SF_PLUGIN_URL . 'modules/internal-link-builder/assets/internal-link-builder.css',
            ['screaming-fixes-admin'],
            SF_VERSION
        );

        wp_enqueue_script(
            'sf-internal-link-builder',
            SF_PLUGIN_URL . 'modules/internal-link-builder/assets/internal-link-builder.js',
            ['jquery', 'screaming-fixes-admin'],
            SF_VERSION,
            true
        );

        wp_localize_script('sf-internal-link-builder', 'sfILBData', [
            'nonce' => wp_create_nonce('sf_ilb_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'siteUrl' => home_url(),
            'i18n' => [
                'extracting' => __('Extracting keywords...', 'screaming-fixes'),
                'scanning' => __('Scanning for opportunities...', 'screaming-fixes'),
                'adding_links' => __('Adding internal links...', 'screaming-fixes'),
                'processing' => __('Processing CSV...', 'screaming-fixes'),
                'error' => __('An error occurred. Please try again.', 'screaming-fixes'),
                'no_keywords' => __('Please select at least one keyword.', 'screaming-fixes'),
                'max_keywords' => __('Maximum 10 keywords allowed.', 'screaming-fixes'),
                'confirm_add' => __('Add links to %d pages?', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Render module content
     */
    public function render() {
        include SF_PLUGIN_DIR . 'modules/internal-link-builder/views/tab-content.php';
    }

    /**
     * Render instructions
     */
    public function render_instructions() {
        include SF_PLUGIN_DIR . 'modules/internal-link-builder/views/instructions.php';
    }
}
```

### 1.2 Register Module in Main Plugin

Add to `screaming-fixes.php` or module loader:

```php
// In the modules array or loader
$modules['internal-link-builder'] = [
    'name' => __('Internal Link Builder', 'screaming-fixes'),
    'class' => 'SF_Internal_Link_Builder',
    'file' => 'modules/internal-link-builder/class-internal-link-builder.php',
];
```

---

## Phase 2: Discovery Mode - Backend (PHP)

### 2.1 Keyword Extraction

```php
/**
 * AJAX: Extract keywords from priority URL
 */
public function ajax_extract_keywords() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';

    if (empty($url)) {
        wp_send_json_error(['message' => __('Please enter a URL.', 'screaming-fixes')]);
    }

    // Convert URL to post ID
    $post_id = $this->url_to_post_id($url);

    if (!$post_id) {
        wp_send_json_error(['message' => __('URL not found in WordPress.', 'screaming-fixes')]);
    }

    $post = get_post($post_id);
    if (!$post) {
        wp_send_json_error(['message' => __('Post not found.', 'screaming-fixes')]);
    }

    $keywords = $this->extract_keywords_from_post($post_id, $post);

    wp_send_json_success([
        'post_id' => $post_id,
        'post_title' => $post->post_title,
        'url' => get_permalink($post_id),
        'keywords' => $keywords,
    ]);
}

/**
 * Extract keywords from post using multiple sources
 */
private function extract_keywords_from_post($post_id, $post) {
    $keywords = [];

    // 1. SEO Plugin Focus Keyword (highest priority)
    $focus_keyword = $this->get_focus_keyword($post_id);
    if ($focus_keyword) {
        // Handle comma-separated focus keywords (Rank Math supports multiple)
        $focus_parts = array_map('trim', explode(',', $focus_keyword));
        foreach ($focus_parts as $kw) {
            if (!empty($kw) && strlen($kw) >= 3) {
                $keywords[] = [
                    'keyword' => $kw,
                    'source' => 'focus_keyword',
                    'source_label' => __('SEO focus keyword', 'screaming-fixes'),
                    'selected' => true,
                ];
            }
        }
    }

    // 2. Page Title
    if (!empty($post->post_title)) {
        $title_words = $this->extract_meaningful_phrases($post->post_title);
        foreach ($title_words as $phrase) {
            if (!$this->keyword_exists($keywords, $phrase)) {
                $keywords[] = [
                    'keyword' => $phrase,
                    'source' => 'title',
                    'source_label' => __('from page title', 'screaming-fixes'),
                    'selected' => true,
                ];
            }
        }
    }

    // 3. H1 Tag (if different from title)
    $h1 = $this->extract_h1_from_content($post->post_content);
    if ($h1 && strtolower($h1) !== strtolower($post->post_title)) {
        $h1_phrases = $this->extract_meaningful_phrases($h1);
        foreach ($h1_phrases as $phrase) {
            if (!$this->keyword_exists($keywords, $phrase)) {
                $keywords[] = [
                    'keyword' => $phrase,
                    'source' => 'h1',
                    'source_label' => __('from H1', 'screaming-fixes'),
                    'selected' => true,
                ];
            }
        }
    }

    // 4. URL Slug
    $slug = $post->post_name;
    if ($slug) {
        $slug_phrase = str_replace('-', ' ', $slug);
        if (strlen($slug_phrase) >= 3 && !$this->keyword_exists($keywords, $slug_phrase)) {
            $keywords[] = [
                'keyword' => $slug_phrase,
                'source' => 'slug',
                'source_label' => __('from URL', 'screaming-fixes'),
                'selected' => count($keywords) < 4, // Auto-select first 4
            ];
        }
    }

    // Limit to reasonable number, mark extras as unselected
    $count = 0;
    foreach ($keywords as &$kw) {
        $count++;
        if ($count > 4) {
            $kw['selected'] = false;
        }
    }

    return array_slice($keywords, 0, 15); // Return max 15 suggestions
}

/**
 * Get focus keyword from SEO plugins
 */
private function get_focus_keyword($post_id) {
    // Rank Math
    $rm_keyword = get_post_meta($post_id, 'rank_math_focus_keyword', true);
    if ($rm_keyword) {
        return $rm_keyword;
    }

    // Yoast
    $yoast_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
    if ($yoast_keyword) {
        return $yoast_keyword;
    }

    // AIOSEO
    $aioseo_keyword = get_post_meta($post_id, '_aioseo_keywords', true);
    if ($aioseo_keyword) {
        return $aioseo_keyword;
    }

    return null;
}

/**
 * Extract H1 from post content
 */
private function extract_h1_from_content($content) {
    if (preg_match('/<h1[^>]*>(.*?)<\/h1>/is', $content, $matches)) {
        return wp_strip_all_tags($matches[1]);
    }
    return null;
}

/**
 * Extract meaningful phrases from text
 */
private function extract_meaningful_phrases($text) {
    $phrases = [];

    // Clean text
    $text = wp_strip_all_tags($text);
    $text = strtolower(trim($text));

    // Common stop words to filter
    $stop_words = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for',
                   'of', 'with', 'by', 'from', 'is', 'are', 'was', 'were', 'be', 'been',
                   'being', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would',
                   'could', 'should', 'may', 'might', 'must', 'shall', 'can', 'need',
                   'this', 'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it',
                   'we', 'they', 'what', 'which', 'who', 'whom', 'how', 'why', 'when',
                   'where', 'your', 'our', 'my', 'his', 'her', 'its', 'their'];

    // Add full phrase if meaningful
    $words = preg_split('/\s+/', $text);
    $meaningful_words = array_filter($words, function($w) use ($stop_words) {
        return strlen($w) >= 3 && !in_array($w, $stop_words);
    });

    // Full phrase (if 2-5 meaningful words)
    if (count($meaningful_words) >= 2 && count($meaningful_words) <= 5) {
        $phrases[] = $text;
    }

    // 2-3 word combinations
    $word_count = count($words);
    for ($i = 0; $i < $word_count - 1; $i++) {
        // 2-word phrase
        $two_word = $words[$i] . ' ' . $words[$i + 1];
        if ($this->is_meaningful_phrase($two_word, $stop_words)) {
            $phrases[] = $two_word;
        }

        // 3-word phrase
        if ($i < $word_count - 2) {
            $three_word = $words[$i] . ' ' . $words[$i + 1] . ' ' . $words[$i + 2];
            if ($this->is_meaningful_phrase($three_word, $stop_words)) {
                $phrases[] = $three_word;
            }
        }
    }

    return array_unique($phrases);
}

/**
 * Check if phrase is meaningful (not just stop words)
 */
private function is_meaningful_phrase($phrase, $stop_words) {
    $words = explode(' ', $phrase);
    $meaningful_count = 0;

    foreach ($words as $word) {
        if (strlen($word) >= 3 && !in_array($word, $stop_words)) {
            $meaningful_count++;
        }
    }

    return $meaningful_count >= 1 && strlen($phrase) >= 5;
}

/**
 * Check if keyword already exists in array
 */
private function keyword_exists($keywords, $new_keyword) {
    $new_lower = strtolower(trim($new_keyword));
    foreach ($keywords as $kw) {
        if (strtolower(trim($kw['keyword'])) === $new_lower) {
            return true;
        }
    }
    return false;
}
```

### 2.2 Opportunity Scanning

```php
/**
 * AJAX: Scan for link opportunities
 */
public function ajax_scan_opportunities() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $target_url = isset($_POST['target_url']) ? esc_url_raw($_POST['target_url']) : '';
    $target_post_id = isset($_POST['target_post_id']) ? absint($_POST['target_post_id']) : 0;
    $keywords = isset($_POST['keywords']) ? array_map('sanitize_text_field', $_POST['keywords']) : [];

    if (empty($target_url) || !$target_post_id) {
        wp_send_json_error(['message' => __('Invalid target URL.', 'screaming-fixes')]);
    }

    if (empty($keywords)) {
        wp_send_json_error(['message' => __('Please select at least one keyword.', 'screaming-fixes')]);
    }

    if (count($keywords) > 10) {
        wp_send_json_error(['message' => __('Maximum 10 keywords allowed.', 'screaming-fixes')]);
    }

    // Check site size for warning
    $post_count = wp_count_posts('post')->publish + wp_count_posts('page')->publish;
    $large_site = $post_count > 2000;

    // Set time limit for large sites
    if ($large_site) {
        set_time_limit(120);
    }

    $start_time = microtime(true);
    $opportunities = $this->find_link_opportunities($target_post_id, $target_url, $keywords);
    $scan_time = round(microtime(true) - $start_time, 2);

    // Check if scan timed out
    if ($scan_time > 30 && empty($opportunities['fixable'])) {
        wp_send_json_error([
            'message' => __('Scan is taking too long. Try reducing keywords or running a smaller batch.', 'screaming-fixes'),
            'timeout' => true,
        ]);
    }

    // Store results for later use
    $this->save_discovery_data([
        'target_url' => $target_url,
        'target_post_id' => $target_post_id,
        'keywords' => $keywords,
        'opportunities' => $opportunities,
        'scan_time' => $scan_time,
    ]);

    wp_send_json_success([
        'fixable' => $opportunities['fixable'],
        'manual' => $opportunities['manual'],
        'fixable_count' => count($opportunities['fixable']),
        'manual_count' => count($opportunities['manual']),
        'scan_time' => $scan_time,
        'large_site' => $large_site,
        'post_count' => $post_count,
    ]);
}

/**
 * Find link opportunities across all content
 */
private function find_link_opportunities($target_post_id, $target_url, $keywords) {
    global $wpdb;

    $opportunities = [
        'fixable' => [],
        'manual' => [],
    ];

    // Build keyword search pattern for SQL
    // Use LIKE with OR for each keyword
    $keyword_conditions = [];
    $keyword_values = [];

    foreach ($keywords as $keyword) {
        $keyword_conditions[] = "post_content LIKE %s";
        $keyword_values[] = '%' . $wpdb->esc_like($keyword) . '%';
    }

    $keyword_sql = implode(' OR ', $keyword_conditions);

    // Query for posts/pages containing any keyword
    // Exclude: target page, revisions, attachments
    $query = $wpdb->prepare(
        "SELECT ID, post_title, post_content, post_type, post_name
         FROM {$wpdb->posts}
         WHERE post_status = 'publish'
         AND post_type IN ('post', 'page')
         AND ID != %d
         AND ({$keyword_sql})
         ORDER BY post_date DESC
         LIMIT 500",
        array_merge([$target_post_id], $keyword_values)
    );

    $posts = $wpdb->get_results($query);

    if (empty($posts)) {
        return $opportunities;
    }

    // Get list of posts that already link to target
    $already_linked = $this->get_posts_linking_to($target_url);

    foreach ($posts as $post) {
        // Skip if already links to target
        if (in_array($post->ID, $already_linked)) {
            continue;
        }

        // Find keyword matches in content
        $matches = $this->find_keyword_matches($post->post_content, $keywords, $target_url);

        if (empty($matches)) {
            continue;
        }

        // Use first match as primary opportunity
        $primary_match = $matches[0];

        // Categorize the page
        $source_url = get_permalink($post->ID);
        $category = $this->categorize_source_page($post->ID, $source_url);

        $opportunity = [
            'source_post_id' => $post->ID,
            'source_url' => $source_url,
            'source_title' => $post->post_title,
            'edit_url' => get_edit_post_link($post->ID, 'raw'),
            'keyword' => $primary_match['keyword'],
            'context' => $primary_match['context'],
            'all_matches' => $matches,
            'match_count' => count($matches),
        ];

        if ($category['category'] === 'fixable') {
            $opportunities['fixable'][] = $opportunity;
        } else {
            $opportunity['note'] = $category['note'];
            $opportunities['manual'][] = $opportunity;
        }
    }

    return $opportunities;
}

/**
 * Find keyword matches in content with context
 */
private function find_keyword_matches($content, $keywords, $target_url) {
    $matches = [];

    // Strip HTML for text matching
    $text_content = wp_strip_all_tags($content);

    foreach ($keywords as $keyword) {
        $keyword_lower = strtolower($keyword);
        $content_lower = strtolower($text_content);

        // Find position of keyword
        $pos = strpos($content_lower, $keyword_lower);

        if ($pos === false) {
            continue;
        }

        // Check if keyword is already linked (in the HTML content)
        if ($this->is_keyword_already_linked($content, $keyword, $target_url)) {
            continue;
        }

        // Extract context (~50 chars around keyword)
        $context_start = max(0, $pos - 25);
        $context_end = min(strlen($text_content), $pos + strlen($keyword) + 25);
        $context = substr($text_content, $context_start, $context_end - $context_start);

        // Add ellipsis if truncated
        if ($context_start > 0) {
            $context = '...' . $context;
        }
        if ($context_end < strlen($text_content)) {
            $context = $context . '...';
        }

        $matches[] = [
            'keyword' => $keyword,
            'context' => $context,
            'position' => $pos,
        ];
    }

    return $matches;
}

/**
 * Check if keyword is already wrapped in a link
 */
private function is_keyword_already_linked($html_content, $keyword, $target_url = null) {
    // Pattern: keyword inside an <a> tag
    $escaped_keyword = preg_quote($keyword, '/');

    // Check if keyword appears within any anchor tag
    $pattern = '/<a\s[^>]*>.*?' . $escaped_keyword . '.*?<\/a>/is';

    if (preg_match($pattern, $html_content)) {
        return true;
    }

    return false;
}

/**
 * Get post IDs that already link to target URL
 */
private function get_posts_linking_to($target_url) {
    global $wpdb;

    $post_ids = [];

    // Normalize URL for searching
    $url_path = wp_parse_url($target_url, PHP_URL_PATH);
    $url_variations = [
        $target_url,
        rtrim($target_url, '/'),
        trailingslashit($target_url),
        $url_path,
        rtrim($url_path, '/'),
    ];

    foreach ($url_variations as $url) {
        $results = $wpdb->get_col($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ('post', 'page')
             AND post_content LIKE %s",
            '%href="' . $wpdb->esc_like($url) . '%'
        ));

        $post_ids = array_merge($post_ids, $results);
    }

    return array_unique($post_ids);
}

/**
 * Categorize source page as fixable or manual
 */
private function categorize_source_page($post_id, $source_url) {
    $path = wp_parse_url($source_url, PHP_URL_PATH) ?: '/';

    // Manual fix patterns (archives, pagination, etc.)
    $manual_patterns = [
        '/\/category\//' => __('Category archive - edit in theme or SEO plugin', 'screaming-fixes'),
        '/\/tag\//' => __('Tag archive - edit in theme or SEO plugin', 'screaming-fixes'),
        '/\/author\//' => __('Author archive - edit in theme or SEO plugin', 'screaming-fixes'),
        '/\/page\/\d+/' => __('Pagination page - controlled by source', 'screaming-fixes'),
        '/\/feed\/?/' => __('RSS feed - not editable', 'screaming-fixes'),
        '/\/\d{4}\/\d{2}\//' => __('Date archive - edit in theme settings', 'screaming-fixes'),
    ];

    foreach ($manual_patterns as $pattern => $note) {
        if (preg_match($pattern, $path)) {
            return ['category' => 'manual', 'note' => $note];
        }
    }

    // Check for homepage
    if ($path === '/' || $path === '' || $path === '/index.php') {
        $front_page_id = get_option('page_on_front');
        if ($front_page_id && $post_id == $front_page_id) {
            return ['category' => 'fixable', 'note' => ''];
        }
        return ['category' => 'manual', 'note' => __('Homepage - edit in theme or widgets', 'screaming-fixes')];
    }

    // Check for blog index
    $blog_page_id = get_option('page_for_posts');
    if ($blog_page_id && $post_id == $blog_page_id) {
        return ['category' => 'manual', 'note' => __('Blog index - edit in theme settings', 'screaming-fixes')];
    }

    // Valid post/page with content
    if ($post_id > 0) {
        $post = get_post($post_id);
        if ($post && in_array($post->post_type, ['post', 'page'])) {
            return ['category' => 'fixable', 'note' => ''];
        }
    }

    return ['category' => 'manual', 'note' => __('Page not found in WordPress', 'screaming-fixes')];
}
```

### 2.3 Link Addition

```php
/**
 * AJAX: Add internal links (Discovery Mode)
 */
public function ajax_add_links() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $links = isset($_POST['links']) ? $_POST['links'] : [];
    $target_url = isset($_POST['target_url']) ? esc_url_raw($_POST['target_url']) : '';
    $offset = isset($_POST['offset']) ? absint($_POST['offset']) : 0;
    $batch_size = isset($_POST['batch_size']) ? absint($_POST['batch_size']) : 50;

    if (empty($links) || empty($target_url)) {
        wp_send_json_error(['message' => __('Invalid request data.', 'screaming-fixes')]);
    }

    // Clear accumulated results at start
    if ($offset === 0) {
        delete_transient('sf_ilb_accumulated_' . get_current_user_id());
    }

    $results = $this->process_link_additions($links, $target_url, $offset, $batch_size);

    // Accumulate results
    $accumulated = get_transient('sf_ilb_accumulated_' . get_current_user_id());
    if (!$accumulated) {
        $accumulated = ['success' => 0, 'failed' => 0, 'skipped' => 0, 'details' => [], 'errors' => []];
    }

    $accumulated['success'] += $results['success'];
    $accumulated['failed'] += $results['failed'];
    $accumulated['skipped'] += $results['skipped'];
    $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
    $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

    if ($results['complete']) {
        // Store final results
        $this->save_results_data($accumulated);
        delete_transient('sf_ilb_accumulated_' . get_current_user_id());

        $results['total_success'] = $accumulated['success'];
        $results['total_failed'] = $accumulated['failed'];
        $results['total_skipped'] = $accumulated['skipped'];
        $results['all_details'] = $accumulated['details'];
    } else {
        set_transient('sf_ilb_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
    }

    $results['total_count'] = count($links);
    wp_send_json_success($results);
}

/**
 * Process link additions in batch
 */
private function process_link_additions($links, $target_url, $offset, $batch_size) {
    $results = [
        'processed' => 0,
        'success' => 0,
        'failed' => 0,
        'skipped' => 0,
        'details' => [],
        'errors' => [],
        'complete' => false,
        'next_offset' => $offset + $batch_size,
    ];

    $batch = array_slice($links, $offset, $batch_size);

    if (empty($batch)) {
        $results['complete'] = true;
        return $results;
    }

    // Group by source post to minimize DB operations
    $grouped = [];
    foreach ($batch as $link) {
        $post_id = isset($link['source_post_id']) ? absint($link['source_post_id']) : 0;
        if (!$post_id) continue;

        if (!isset($grouped[$post_id])) {
            $grouped[$post_id] = [];
        }
        $grouped[$post_id][] = $link;
    }

    // Process each source post
    foreach ($grouped as $post_id => $post_links) {
        $post = get_post($post_id);
        if (!$post) {
            foreach ($post_links as $link) {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $link['source_url'] ?? '',
                    'keyword' => $link['keyword'] ?? '',
                    'error' => __('Post not found', 'screaming-fixes'),
                ];
            }
            continue;
        }

        $content = $post->post_content;
        $original_content = $content;
        $links_added = 0;

        foreach ($post_links as $link) {
            $results['processed']++;
            $keyword = isset($link['keyword']) ? $link['keyword'] : '';
            $source_url = isset($link['source_url']) ? $link['source_url'] : get_permalink($post_id);

            if (empty($keyword)) {
                $results['skipped']++;
                continue;
            }

            // Add link to content
            $new_content = $this->add_link_to_content($content, $keyword, $target_url);

            if ($new_content !== $content) {
                $content = $new_content;
                $links_added++;
                $results['success']++;
                $results['details'][] = [
                    'source_url' => $source_url,
                    'source_title' => $post->post_title,
                    'keyword' => $keyword,
                    'target_url' => $target_url,
                    'status' => 'added',
                ];
            } else {
                $results['skipped']++;
                $results['errors'][] = [
                    'source_url' => $source_url,
                    'keyword' => $keyword,
                    'error' => __('Anchor text not found or already linked', 'screaming-fixes'),
                ];
            }
        }

        // Save if any changes were made
        if ($content !== $original_content) {
            // Track for undo
            if (class_exists('SF_Batch_Restore')) {
                SF_Batch_Restore::track_change($post_id, 'post_content', $original_content);
            }

            wp_update_post([
                'ID' => $post_id,
                'post_content' => $content,
            ]);
        }
    }

    // Check if complete
    if ($offset + $batch_size >= count($links)) {
        $results['complete'] = true;
    }

    return $results;
}

/**
 * Add link to content for first occurrence of keyword
 */
private function add_link_to_content($content, $keyword, $target_url) {
    // Escape for regex
    $escaped_keyword = preg_quote($keyword, '/');

    // Pattern: Match keyword NOT inside an existing <a> tag
    // Uses negative lookbehind and lookahead
    // This is complex because we need to ensure we're not inside a tag

    // First, check if keyword exists and is not already linked
    if (!preg_match('/(?<!["\'>])(' . $escaped_keyword . ')(?![^<]*<\/a>)/i', $content)) {
        return $content; // Keyword not found or already linked
    }

    // Replace first occurrence only
    // Pattern matches keyword that's not inside an anchor tag
    $pattern = '/(?<!["\'>])(' . $escaped_keyword . ')(?![^<]*<\/a>)/i';

    $replacement = '<a href="' . esc_url($target_url) . '">$1</a>';

    // Replace only first match
    $new_content = preg_replace($pattern, $replacement, $content, 1);

    return $new_content;
}
```

---

## Phase 3: Bulk Upload Mode - Backend (PHP)

### 3.1 CSV Processing

```php
/**
 * AJAX: Process bulk upload CSV
 */
public function ajax_process_csv() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

    $uploads = get_option('sf_pending_uploads', []);
    if (!isset($uploads[$upload_id])) {
        wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
    }

    $file_path = $uploads[$upload_id]['path'];

    // Parse CSV
    $parser = new SF_CSV_Parser();
    $parsed = $parser->parse($file_path);

    if (is_wp_error($parsed)) {
        wp_send_json_error(['message' => $parsed->get_error_message()]);
    }

    // Detect columns
    $columns = $this->detect_bulk_columns($parsed['headers']);

    if (!$columns['source'] || !$columns['anchor'] || !$columns['target']) {
        wp_send_json_error([
            'message' => __('CSV must contain Source_URL, Anchor_Text, and Target_URL columns.', 'screaming-fixes'),
        ]);
    }

    // Process rows
    $results = $this->process_bulk_csv_rows($parsed['rows'], $columns);

    // Store for later use
    $this->save_bulk_data($results);

    wp_send_json_success([
        'is_bulk_upload' => true,
        'ready_count' => $results['ready_count'],
        'manual_count' => $results['manual_count'],
        'skipped_count' => $results['skipped_count'],
        'duplicates_count' => $results['duplicates_count'],
        'total_rows' => $results['total_rows'],
        'ready_links' => $results['ready_links'],
        'manual_links' => $results['manual_links'],
        'skipped_links' => $results['skipped_links'],
    ]);
}

/**
 * Detect column mappings from headers
 */
private function detect_bulk_columns($headers) {
    $columns = [
        'source' => null,
        'anchor' => null,
        'target' => null,
    ];

    $source_patterns = ['source_url', 'sourceurl', 'source', 'page_url', 'pageurl', 'page', 'from_url', 'fromurl', 'from', 'address'];
    $anchor_patterns = ['anchor_text', 'anchortext', 'anchor', 'keyword', 'link_text', 'linktext', 'text'];
    $target_patterns = ['target_url', 'targeturl', 'target', 'destination', 'to_url', 'tourl', 'to', 'link_url', 'linkurl'];

    foreach ($headers as $header) {
        $normalized = $this->normalize_column_name($header);
        $header_lower = strtolower(trim($header));

        if (in_array($normalized, $source_patterns)) {
            $columns['source'] = $header_lower;
        } elseif (in_array($normalized, $anchor_patterns)) {
            $columns['anchor'] = $header_lower;
        } elseif (in_array($normalized, $target_patterns)) {
            $columns['target'] = $header_lower;
        }
    }

    return $columns;
}

/**
 * Normalize column name
 */
private function normalize_column_name($name) {
    $normalized = strtolower(trim($name));
    $normalized = str_replace([' ', '-', '_'], '', $normalized);
    return $normalized;
}

/**
 * Process bulk CSV rows
 */
private function process_bulk_csv_rows($rows, $columns) {
    $ready_links = [];
    $manual_links = [];
    $skipped_links = [];
    $seen_combinations = [];
    $duplicates = 0;

    // Build URL map for quick lookups
    $url_map = $this->build_wp_url_map();

    foreach ($rows as $row) {
        $source_url = isset($row[$columns['source']]) ? trim($row[$columns['source']]) : '';
        $anchor_text = isset($row[$columns['anchor']]) ? trim($row[$columns['anchor']]) : '';
        $target_url = isset($row[$columns['target']]) ? trim($row[$columns['target']]) : '';

        // Skip empty rows
        if (empty($source_url) || empty($anchor_text) || empty($target_url)) {
            continue;
        }

        // Check for duplicates (use last occurrence)
        $combo_key = md5($source_url . '|' . $anchor_text . '|' . $target_url);
        if (isset($seen_combinations[$combo_key])) {
            $duplicates++;
            // Remove previous entry
            $ready_links = array_filter($ready_links, function($item) use ($combo_key) {
                return $item['combo_key'] !== $combo_key;
            });
            $ready_links = array_values($ready_links);
        }
        $seen_combinations[$combo_key] = true;

        // Resolve source URL to post ID
        $normalized_source = $this->normalize_url($source_url);
        $source_post_id = 0;

        if (isset($url_map[$normalized_source])) {
            $source_post_id = $url_map[$normalized_source]['post_id'];
        } else {
            // Try direct lookup
            $source_post_id = $this->url_to_post_id($source_url);
        }

        // Categorize
        if (!$source_post_id) {
            $skipped_links[] = [
                'source_url' => $source_url,
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'status' => __('Skipped - Source URL not found', 'screaming-fixes'),
            ];
            continue;
        }

        // Check if source is archive/non-editable
        $category = $this->categorize_source_page($source_post_id, $source_url);

        if ($category['category'] === 'manual') {
            $manual_links[] = [
                'source_url' => $source_url,
                'source_post_id' => $source_post_id,
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'status' => __('Manual Fix', 'screaming-fixes'),
                'note' => $category['note'],
            ];
            continue;
        }

        // Check if anchor text exists in source content
        $post = get_post($source_post_id);
        if (!$post || stripos($post->post_content, $anchor_text) === false) {
            $skipped_links[] = [
                'source_url' => $source_url,
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'status' => sprintf(__('Skipped - Anchor text "%s" not found on page', 'screaming-fixes'), $anchor_text),
            ];
            continue;
        }

        // Check if link already exists
        if ($this->link_already_exists($post->post_content, $anchor_text, $target_url)) {
            $skipped_links[] = [
                'source_url' => $source_url,
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'status' => __('Skipped - Link already exists', 'screaming-fixes'),
            ];
            continue;
        }

        // Ready to add
        $ready_links[] = [
            'source_url' => $source_url,
            'source_post_id' => $source_post_id,
            'source_title' => $post->post_title,
            'anchor_text' => $anchor_text,
            'target_url' => $target_url,
            'status' => __('Ready', 'screaming-fixes'),
            'combo_key' => $combo_key,
        ];
    }

    return [
        'ready_links' => $ready_links,
        'manual_links' => $manual_links,
        'skipped_links' => $skipped_links,
        'ready_count' => count($ready_links),
        'manual_count' => count($manual_links),
        'skipped_count' => count($skipped_links),
        'duplicates_count' => $duplicates,
        'total_rows' => count($rows),
    ];
}

/**
 * Check if link already exists from anchor to target
 */
private function link_already_exists($content, $anchor_text, $target_url) {
    $escaped_anchor = preg_quote($anchor_text, '/');
    $escaped_url = preg_quote($target_url, '/');

    // Check for anchor text already linked to this target
    $pattern = '/<a\s[^>]*href=["\']' . $escaped_url . '["\'][^>]*>.*?' . $escaped_anchor . '.*?<\/a>/is';

    return (bool) preg_match($pattern, $content);
}

/**
 * Build WordPress URL map
 */
private function build_wp_url_map() {
    global $wpdb;
    $map = [];

    $posts = $wpdb->get_results(
        "SELECT ID, post_name, post_type FROM {$wpdb->posts}
         WHERE post_status = 'publish'
         AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')
         LIMIT 50000",
        ARRAY_A
    );

    foreach ($posts as $post) {
        $permalink = get_permalink($post['ID']);
        if ($permalink) {
            $normalized = $this->normalize_url($permalink);
            $map[$normalized] = [
                'post_id' => (int) $post['ID'],
                'post_type' => $post['post_type'],
            ];
        }
    }

    return $map;
}

/**
 * Normalize URL for comparison
 */
private function normalize_url($url) {
    $url = trim($url);
    $url = rtrim($url, '/');
    $url = preg_replace('/^https?:\/\//i', '', $url);
    $url = preg_replace('/^www\./i', '', $url);
    $url = strtolower($url);
    return $url;
}

/**
 * Convert URL to post ID
 */
private function url_to_post_id($url) {
    $post_id = url_to_postid($url);
    if ($post_id) return $post_id;

    $post_id = url_to_postid(trailingslashit($url));
    if ($post_id) return $post_id;

    $post_id = url_to_postid(untrailingslashit($url));
    if ($post_id) return $post_id;

    // Try slug extraction
    $path = wp_parse_url($url, PHP_URL_PATH);
    if ($path) {
        $slug = basename(rtrim($path, '/'));
        global $wpdb;
        $post_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts}
             WHERE post_name = %s
             AND post_status IN ('publish', 'draft', 'pending', 'private')
             AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')
             LIMIT 1",
            $slug
        ));
        if ($post_id) return (int) $post_id;
    }

    return 0;
}
```

### 3.2 Bulk Link Application

```php
/**
 * AJAX: Apply bulk links
 */
public function ajax_apply_bulk_links() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $offset = isset($_POST['offset']) ? absint($_POST['offset']) : 0;
    $batch_size = isset($_POST['batch_size']) ? absint($_POST['batch_size']) : 50;

    // Clear accumulated results at start
    if ($offset === 0) {
        delete_transient('sf_ilb_bulk_accumulated_' . get_current_user_id());
    }

    // Get stored bulk data
    $bulk_data = $this->get_bulk_data();

    if (empty($bulk_data) || empty($bulk_data['ready_links'])) {
        wp_send_json_error(['message' => __('No bulk data found.', 'screaming-fixes')]);
    }

    $links = $bulk_data['ready_links'];
    $results = $this->apply_bulk_links($links, $offset, $batch_size);

    // Accumulate results
    $accumulated = get_transient('sf_ilb_bulk_accumulated_' . get_current_user_id());
    if (!$accumulated) {
        $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
    }

    $accumulated['success'] += $results['success'];
    $accumulated['failed'] += $results['failed'];
    $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
    $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

    if ($results['complete']) {
        // Store final results
        $final_results = [
            'is_bulk_upload' => true,
            'bulk_complete' => true,
            'added_links' => $accumulated['details'],
            'failed_links' => $accumulated['errors'],
            'success_count' => $accumulated['success'],
            'failed_count' => $accumulated['failed'],
            'manual_links' => $bulk_data['manual_links'] ?? [],
            'skipped_links' => $bulk_data['skipped_links'] ?? [],
        ];

        $this->save_bulk_data($final_results);
        delete_transient('sf_ilb_bulk_accumulated_' . get_current_user_id());

        $results['total_success'] = $accumulated['success'];
        $results['total_failed'] = $accumulated['failed'];
        $results['all_details'] = $accumulated['details'];
    } else {
        set_transient('sf_ilb_bulk_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
    }

    $results['total_count'] = count($links);
    wp_send_json_success($results);
}

/**
 * Apply bulk links in batch
 */
private function apply_bulk_links($links, $offset, $batch_size) {
    $results = [
        'processed' => 0,
        'success' => 0,
        'failed' => 0,
        'details' => [],
        'errors' => [],
        'complete' => false,
        'next_offset' => $offset + $batch_size,
    ];

    $batch = array_slice($links, $offset, $batch_size);

    if (empty($batch)) {
        $results['complete'] = true;
        return $results;
    }

    // Start batch for undo
    if (class_exists('SF_Batch_Restore') && $offset === 0) {
        SF_Batch_Restore::start_batch('internal_link_builder', 'Bulk Internal Link Addition');
    }

    // Group by source post
    $grouped = [];
    foreach ($batch as $link) {
        $post_id = $link['source_post_id'];
        if (!isset($grouped[$post_id])) {
            $grouped[$post_id] = [];
        }
        $grouped[$post_id][] = $link;
    }

    foreach ($grouped as $post_id => $post_links) {
        $post = get_post($post_id);
        if (!$post) {
            foreach ($post_links as $link) {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $link['source_url'],
                    'anchor_text' => $link['anchor_text'],
                    'target_url' => $link['target_url'],
                    'error' => __('Post not found', 'screaming-fixes'),
                ];
            }
            continue;
        }

        $content = $post->post_content;
        $original_content = $content;

        foreach ($post_links as $link) {
            $results['processed']++;

            $new_content = $this->add_link_to_content(
                $content,
                $link['anchor_text'],
                $link['target_url']
            );

            if ($new_content !== $content) {
                $content = $new_content;
                $results['success']++;
                $results['details'][] = [
                    'source_url' => $link['source_url'],
                    'source_title' => $post->post_title,
                    'anchor_text' => $link['anchor_text'],
                    'target_url' => $link['target_url'],
                    'status' => 'added',
                ];
            } else {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $link['source_url'],
                    'anchor_text' => $link['anchor_text'],
                    'target_url' => $link['target_url'],
                    'error' => __('Failed to add link - anchor text may already be linked', 'screaming-fixes'),
                ];
            }
        }

        // Save if changes made
        if ($content !== $original_content) {
            if (class_exists('SF_Batch_Restore')) {
                SF_Batch_Restore::track_change($post_id, 'post_content', $original_content);
            }

            wp_update_post([
                'ID' => $post_id,
                'post_content' => $content,
            ]);
        }
    }

    if ($offset + $batch_size >= count($links)) {
        $results['complete'] = true;

        if (class_exists('SF_Batch_Restore')) {
            SF_Batch_Restore::complete_batch();
        }
    }

    return $results;
}
```

### 3.3 Data Storage & CSV Download

```php
/**
 * Save discovery mode data
 */
private function save_discovery_data($data) {
    $user_id = get_current_user_id();
    set_transient('sf_ilb_discovery_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
}

/**
 * Get discovery mode data
 */
private function get_discovery_data() {
    $user_id = get_current_user_id();
    return get_transient('sf_ilb_discovery_' . $user_id);
}

/**
 * Save bulk mode data
 */
private function save_bulk_data($data) {
    $user_id = get_current_user_id();
    set_transient('sf_ilb_bulk_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
}

/**
 * Get bulk mode data
 */
private function get_bulk_data() {
    $user_id = get_current_user_id();
    return get_transient('sf_ilb_bulk_' . $user_id);
}

/**
 * Save results data
 */
private function save_results_data($data) {
    $user_id = get_current_user_id();
    set_transient('sf_ilb_results_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
}

/**
 * AJAX: Download preview CSV
 */
public function ajax_download_preview() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    // Try bulk data first, then discovery data
    $bulk_data = $this->get_bulk_data();
    $discovery_data = $this->get_discovery_data();

    $lines = ['Source_URL,Anchor_Text,Target_URL,Status'];

    if (!empty($bulk_data['ready_links'])) {
        foreach ($bulk_data['ready_links'] as $link) {
            $lines[] = $this->csv_escape_row([
                $link['source_url'],
                $link['anchor_text'],
                $link['target_url'],
                'Ready',
            ]);
        }
        foreach ($bulk_data['manual_links'] ?? [] as $link) {
            $lines[] = $this->csv_escape_row([
                $link['source_url'],
                $link['anchor_text'],
                $link['target_url'],
                'Manual Fix - ' . ($link['note'] ?? ''),
            ]);
        }
        foreach ($bulk_data['skipped_links'] ?? [] as $link) {
            $lines[] = $this->csv_escape_row([
                $link['source_url'],
                $link['anchor_text'],
                $link['target_url'],
                $link['status'],
            ]);
        }
    } elseif (!empty($discovery_data['opportunities'])) {
        $target_url = $discovery_data['target_url'];
        foreach ($discovery_data['opportunities']['fixable'] ?? [] as $opp) {
            $lines[] = $this->csv_escape_row([
                $opp['source_url'],
                $opp['keyword'],
                $target_url,
                'Ready',
            ]);
        }
        foreach ($discovery_data['opportunities']['manual'] ?? [] as $opp) {
            $lines[] = $this->csv_escape_row([
                $opp['source_url'],
                $opp['keyword'],
                $target_url,
                'Manual Fix - ' . ($opp['note'] ?? ''),
            ]);
        }
    }

    wp_send_json_success([
        'csv' => implode("\n", $lines),
        'filename' => 'internal-links-preview-' . date('Y-m-d') . '.csv',
    ]);
}

/**
 * AJAX: Download results CSV
 */
public function ajax_download_results() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $results = $this->get_results_data();
    $bulk_data = $this->get_bulk_data();

    $lines = ['Source_URL,Anchor_Text,Target_URL,Status,Notes'];

    // Added links
    $details = $results['details'] ?? $bulk_data['added_links'] ?? [];
    foreach ($details as $link) {
        $lines[] = $this->csv_escape_row([
            $link['source_url'],
            $link['anchor_text'] ?? $link['keyword'] ?? '',
            $link['target_url'],
            'Added',
            '',
        ]);
    }

    // Failed/skipped
    $errors = $results['errors'] ?? $bulk_data['failed_links'] ?? [];
    foreach ($errors as $link) {
        $lines[] = $this->csv_escape_row([
            $link['source_url'],
            $link['anchor_text'] ?? $link['keyword'] ?? '',
            $link['target_url'] ?? '',
            'Failed',
            $link['error'] ?? '',
        ]);
    }

    wp_send_json_success([
        'csv' => implode("\n", $lines),
        'filename' => 'internal-links-results-' . date('Y-m-d') . '.csv',
    ]);
}

/**
 * Get results data
 */
private function get_results_data() {
    $user_id = get_current_user_id();
    return get_transient('sf_ilb_results_' . $user_id) ?: [];
}

/**
 * AJAX: Clear all data
 */
public function ajax_clear_data() {
    check_ajax_referer('sf_ilb_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
    }

    $user_id = get_current_user_id();
    delete_transient('sf_ilb_discovery_' . $user_id);
    delete_transient('sf_ilb_bulk_' . $user_id);
    delete_transient('sf_ilb_results_' . $user_id);
    delete_transient('sf_ilb_accumulated_' . $user_id);
    delete_transient('sf_ilb_bulk_accumulated_' . $user_id);

    wp_send_json_success(['message' => __('Data cleared.', 'screaming-fixes')]);
}

/**
 * CSV escape helper
 */
private function csv_escape_row($fields) {
    $escaped = [];
    foreach ($fields as $field) {
        $field = (string) $field;
        if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
            $field = '"' . str_replace('"', '""', $field) . '"';
        }
        $escaped[] = $field;
    }
    return implode(',', $escaped);
}
```

---

## Phase 4: Frontend Views

### 4.1 Main Tab Content (tab-content.php)

```php
<?php
/**
 * Internal Link Builder - Tab Content
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get stored data
$user_id = get_current_user_id();
$discovery_data = get_transient('sf_ilb_discovery_' . $user_id);
$bulk_data = get_transient('sf_ilb_bulk_' . $user_id);
$results_data = get_transient('sf_ilb_results_' . $user_id);

$has_discovery = !empty($discovery_data['opportunities']);
$has_bulk = !empty($bulk_data['ready_links']) || !empty($bulk_data['bulk_complete']);
$has_results = !empty($results_data['details']) || !empty($bulk_data['added_links']);

// Determine current state
$show_input = !$has_discovery && !$has_bulk && !$has_results;
$show_keywords = isset($_GET['step']) && $_GET['step'] === 'keywords';
$show_opportunities = $has_discovery && !$has_results;
$show_bulk_confirmation = $has_bulk && empty($bulk_data['bulk_complete']);
$show_results = $has_results || !empty($bulk_data['bulk_complete']);
?>

<div class="sf-internal-link-builder-module">

    <!-- Instructions Section (Collapsible) -->
    <div class="sf-instructions-section">
        <?php $this->render_instructions(); ?>
    </div>

    <!-- STEP 1: URL Input (Discovery Mode Start) -->
    <?php if ($show_input): ?>
    <div class="sf-input-section" id="sf-ilb-input">
        <div class="sf-section-header">
            <h3><?php esc_html_e('Internal Link Builder', 'screaming-fixes'); ?></h3>
            <p><?php esc_html_e('Enter the URL you want to rank better:', 'screaming-fixes'); ?></p>
        </div>

        <div class="sf-url-input-wrapper">
            <input type="url"
                   id="sf-ilb-priority-url"
                   class="sf-url-input"
                   placeholder="https://mysite.com/best-coffee-shops/"
                   autocomplete="off">
            <button type="button" class="sf-button sf-button-primary" id="sf-ilb-extract-btn">
                <?php esc_html_e('Extract Keywords', 'screaming-fixes'); ?>
            </button>
        </div>

        <div class="sf-input-divider">
            <span><?php esc_html_e('or', 'screaming-fixes'); ?></span>
        </div>

        <div class="sf-upload-section">
            <p><?php esc_html_e('Upload a CSV with pre-defined internal links:', 'screaming-fixes'); ?></p>
            <div class="sf-dropzone" id="sf-ilb-dropzone">
                <div class="sf-dropzone-content">
                    <span class="dashicons dashicons-upload"></span>
                    <p><?php esc_html_e('Drop CSV file here or click to upload', 'screaming-fixes'); ?></p>
                    <p class="sf-dropzone-hint"><?php esc_html_e('Required columns: Source_URL, Anchor_Text, Target_URL', 'screaming-fixes'); ?></p>
                </div>
                <input type="file" id="sf-ilb-file-input" accept=".csv" style="display: none;">
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- STEP 2: Keyword Preview -->
    <div class="sf-keywords-section" id="sf-ilb-keywords" style="display: none;">
        <div class="sf-section-header">
            <h3><?php esc_html_e('Keyword Preview', 'screaming-fixes'); ?></h3>
            <p class="sf-target-url-display"></p>
        </div>

        <div class="sf-keywords-list" id="sf-ilb-keywords-list">
            <!-- Populated by JavaScript -->
        </div>

        <div class="sf-add-keyword-row">
            <input type="text" id="sf-ilb-new-keyword" placeholder="<?php esc_attr_e('Add custom keyword...', 'screaming-fixes'); ?>">
            <button type="button" class="sf-button sf-button-secondary" id="sf-ilb-add-keyword-btn">
                <span class="dashicons dashicons-plus-alt2"></span>
                <?php esc_html_e('Add', 'screaming-fixes'); ?>
            </button>
        </div>

        <div class="sf-keywords-footer">
            <span class="sf-keyword-count">
                <span id="sf-ilb-selected-count">0</span> <?php esc_html_e('of 10 keywords selected', 'screaming-fixes'); ?>
            </span>
            <div class="sf-keywords-actions">
                <button type="button" class="sf-button sf-button-secondary" id="sf-ilb-back-btn">
                    <?php esc_html_e('Back', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-button sf-button-primary" id="sf-ilb-scan-btn">
                    <?php esc_html_e('Scan for Opportunities', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
    </div>

    <!-- STEP 3: Opportunities Display (Discovery Mode) -->
    <div class="sf-opportunities-section" id="sf-ilb-opportunities" style="<?php echo $show_opportunities ? '' : 'display: none;'; ?>">
        <?php if ($show_opportunities): ?>
        <div class="sf-section-header">
            <h3>
                <?php printf(
                    esc_html__('Found %d link opportunities for %s', 'screaming-fixes'),
                    count($discovery_data['opportunities']['fixable']) + count($discovery_data['opportunities']['manual']),
                    '<code>' . esc_html(wp_parse_url($discovery_data['target_url'], PHP_URL_PATH)) . '</code>'
                ); ?>
            </h3>
        </div>

        <!-- Stats -->
        <div class="sf-stats-breakdown">
            <div class="sf-stat-card sf-stat-fixable">
                <div class="sf-stat-icon">✓</div>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo count($discovery_data['opportunities']['fixable']); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Fixable (can auto-add links)', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php if (!empty($discovery_data['opportunities']['manual'])): ?>
            <div class="sf-stat-card sf-stat-manual">
                <div class="sf-stat-icon">⚠️</div>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo count($discovery_data['opportunities']['manual']); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Manual Fix (source can\'t be edited)', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <!-- Fixable Opportunities Table -->
        <?php if (!empty($discovery_data['opportunities']['fixable'])): ?>
        <div class="sf-table-section">
            <div class="sf-table-header">
                <label class="sf-select-all-label">
                    <input type="checkbox" id="sf-ilb-select-all" checked>
                    <?php esc_html_e('Select All', 'screaming-fixes'); ?>
                </label>
            </div>
            <table class="sf-opportunities-table">
                <thead>
                    <tr>
                        <th class="sf-col-checkbox"></th>
                        <th><?php esc_html_e('Source Page', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Keyword Found', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Context', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Action', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($discovery_data['opportunities']['fixable'] as $index => $opp): ?>
                    <tr class="sf-opportunity-row" data-index="<?php echo $index; ?>">
                        <td class="sf-col-checkbox">
                            <input type="checkbox" class="sf-opp-select" checked
                                   data-post-id="<?php echo esc_attr($opp['source_post_id']); ?>"
                                   data-keyword="<?php echo esc_attr($opp['keyword']); ?>"
                                   data-source-url="<?php echo esc_attr($opp['source_url']); ?>">
                        </td>
                        <td class="sf-url-cell">
                            <a href="<?php echo esc_url($opp['source_url']); ?>" target="_blank" title="<?php echo esc_attr($opp['source_url']); ?>">
                                <?php echo esc_html($this->truncate_url($opp['source_url'], 40)); ?>
                            </a>
                        </td>
                        <td class="sf-keyword-cell">
                            <span class="sf-keyword-badge"><?php echo esc_html($opp['keyword']); ?></span>
                        </td>
                        <td class="sf-context-cell">
                            <?php echo $this->highlight_keyword_in_context($opp['context'], $opp['keyword']); ?>
                        </td>
                        <td class="sf-action-cell">
                            <button type="button" class="sf-button sf-button-small sf-add-single-link"
                                    data-post-id="<?php echo esc_attr($opp['source_post_id']); ?>"
                                    data-keyword="<?php echo esc_attr($opp['keyword']); ?>"
                                    data-source-url="<?php echo esc_attr($opp['source_url']); ?>">
                                <?php esc_html_e('Add Link', 'screaming-fixes'); ?>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>

        <!-- Manual Fix Section (Collapsible) -->
        <?php if (!empty($discovery_data['opportunities']['manual'])): ?>
        <div class="sf-manual-section">
            <button type="button" class="sf-section-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-manual">⚠️</span>
                <?php printf(esc_html__('Manual Fix Required (%d)', 'screaming-fixes'), count($discovery_data['opportunities']['manual'])); ?>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>
            <div class="sf-manual-content" style="display: none;">
                <p class="sf-manual-explanation">
                    <?php esc_html_e('These pages can\'t be edited automatically because they don\'t have editable post content in WordPress (category archives, tag archives, author archives, etc.).', 'screaming-fixes'); ?>
                </p>
                <table class="sf-manual-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Keyword', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Reason', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($discovery_data['opportunities']['manual'] as $opp): ?>
                        <tr>
                            <td><?php echo esc_html($this->truncate_url($opp['source_url'], 40)); ?></td>
                            <td><?php echo esc_html($opp['keyword']); ?></td>
                            <td><?php echo esc_html($opp['note']); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Actions -->
        <div class="sf-opportunities-actions">
            <button type="button" class="sf-button sf-button-secondary" id="sf-ilb-download-preview">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Download All Opportunities CSV', 'screaming-fixes'); ?>
            </button>
            <div class="sf-action-buttons">
                <button type="button" class="sf-button sf-button-secondary" id="sf-ilb-clear">
                    <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-button sf-button-primary" id="sf-ilb-add-selected">
                    <?php esc_html_e('Add Selected Links', 'screaming-fixes'); ?>
                    (<span id="sf-ilb-selected-count-action">0</span>)
                </button>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Bulk Upload Confirmation -->
    <div class="sf-bulk-confirmation" id="sf-ilb-bulk-confirmation" style="<?php echo $show_bulk_confirmation ? '' : 'display: none;'; ?>">
        <?php if ($show_bulk_confirmation): ?>
        <div class="sf-section-header">
            <div class="sf-bulk-icon">📋</div>
            <h3><?php esc_html_e('Bulk Internal Link Upload', 'screaming-fixes'); ?></h3>
        </div>

        <div class="sf-bulk-stats">
            <div class="sf-bulk-stat sf-bulk-stat-ready">
                <div class="sf-bulk-stat-icon">✓</div>
                <div class="sf-bulk-stat-content">
                    <span class="sf-bulk-stat-number"><?php echo $bulk_data['ready_count']; ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('links ready to add', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php if ($bulk_data['manual_count'] > 0): ?>
            <div class="sf-bulk-stat sf-bulk-stat-manual">
                <div class="sf-bulk-stat-icon">⚠️</div>
                <div class="sf-bulk-stat-content">
                    <span class="sf-bulk-stat-number"><?php echo $bulk_data['manual_count']; ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('manual fix required', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php endif; ?>
            <?php if ($bulk_data['skipped_count'] > 0): ?>
            <div class="sf-bulk-stat sf-bulk-stat-skipped">
                <div class="sf-bulk-stat-icon">✗</div>
                <div class="sf-bulk-stat-content">
                    <span class="sf-bulk-stat-number"><?php echo $bulk_data['skipped_count']; ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('skipped (see details)', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <?php if ($bulk_data['duplicates_count'] > 0): ?>
        <div class="sf-bulk-warning">
            <span class="dashicons dashicons-warning"></span>
            <?php printf(esc_html__('%d duplicate entries detected - using last occurrence for each.', 'screaming-fixes'), $bulk_data['duplicates_count']); ?>
        </div>
        <?php endif; ?>

        <?php if ($bulk_data['ready_count'] > 500): ?>
        <div class="sf-bulk-warning">
            <span class="dashicons dashicons-warning"></span>
            <?php esc_html_e('Large file detected. Processing may take a few minutes.', 'screaming-fixes'); ?>
        </div>
        <?php endif; ?>

        <!-- Preview Table -->
        <div class="sf-bulk-preview-section">
            <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?>
                <span class="sf-preview-count">(<?php printf(esc_html__('showing %d of %d', 'screaming-fixes'), min(10, $bulk_data['ready_count']), $bulk_data['ready_count']); ?>)</span>
            </h4>
            <table class="sf-bulk-preview-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Anchor Text', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Target', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_slice($bulk_data['ready_links'], 0, 10) as $link): ?>
                    <tr>
                        <td title="<?php echo esc_attr($link['source_url']); ?>"><?php echo esc_html($this->truncate_url($link['source_url'], 35)); ?></td>
                        <td><?php echo esc_html($link['anchor_text']); ?></td>
                        <td title="<?php echo esc_attr($link['target_url']); ?>"><?php echo esc_html($this->truncate_url($link['target_url'], 35)); ?></td>
                        <td><span class="sf-status-badge sf-status-ready"><?php esc_html_e('Ready', 'screaming-fixes'); ?></span></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Actions -->
        <div class="sf-bulk-actions">
            <button type="button" class="sf-button sf-button-secondary sf-bulk-download-preview">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
            </button>
            <div class="sf-bulk-buttons">
                <button type="button" class="sf-button sf-button-secondary sf-bulk-clear">
                    <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-button sf-button-primary sf-bulk-confirm">
                    <?php esc_html_e('Confirm', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Progress Modal -->
    <div class="sf-progress-modal" id="sf-ilb-progress-modal" style="display: none;">
        <div class="sf-progress-content">
            <div class="sf-progress-header">
                <span class="sf-progress-icon sf-spinning">⟳</span>
                <h3><?php esc_html_e('Adding internal links...', 'screaming-fixes'); ?></h3>
            </div>
            <div class="sf-progress-bar">
                <div class="sf-progress-fill" style="width: 0%;"></div>
            </div>
            <div class="sf-progress-stats">
                <span class="sf-progress-current">0</span> / <span class="sf-progress-total">0</span>
                (<span class="sf-progress-percent">0</span>%)
            </div>
            <div class="sf-progress-current-item"></div>
        </div>
    </div>

    <!-- Results Section -->
    <div class="sf-results-section" id="sf-ilb-results" style="<?php echo $show_results ? '' : 'display: none;'; ?>">
        <?php
        $success_count = $results_data['success'] ?? $bulk_data['success_count'] ?? 0;
        $failed_count = ($results_data['failed'] ?? 0) + ($results_data['skipped'] ?? 0) + ($bulk_data['failed_count'] ?? 0);
        $details = $results_data['details'] ?? $bulk_data['added_links'] ?? [];
        ?>
        <div class="sf-results-header">
            <div class="sf-results-icon"><?php echo $failed_count > 0 ? '⚠️' : '✓'; ?></div>
            <h3>
                <?php
                if ($failed_count > 0) {
                    printf(esc_html__('%d internal links added successfully. %d skipped.', 'screaming-fixes'), $success_count, $failed_count);
                } else {
                    printf(esc_html__('%d internal links added successfully.', 'screaming-fixes'), $success_count);
                }
                ?>
            </h3>
        </div>

        <div class="sf-results-summary">
            <div class="sf-result-stat sf-result-success">
                <span class="sf-result-value"><?php echo $success_count; ?></span>
                <span class="sf-result-label"><?php esc_html_e('Added', 'screaming-fixes'); ?></span>
            </div>
            <?php if ($failed_count > 0): ?>
            <div class="sf-result-stat sf-result-skipped">
                <span class="sf-result-value"><?php echo $failed_count; ?></span>
                <span class="sf-result-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
            </div>
            <?php endif; ?>
        </div>

        <div class="sf-results-actions">
            <button type="button" class="sf-button sf-button-secondary sf-download-results">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
            </button>
            <button type="button" class="sf-button sf-button-primary sf-clear-results">
                <?php esc_html_e('Clear', 'screaming-fixes'); ?>
            </button>
        </div>

        <?php if (!empty($details)): ?>
        <div class="sf-results-table-section">
            <h4><?php esc_html_e('Links Added', 'screaming-fixes'); ?>
                <span class="sf-preview-count">(<?php printf(esc_html__('showing %d of %d', 'screaming-fixes'), min(10, count($details)), count($details)); ?>)</span>
            </h4>
            <table class="sf-results-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Anchor Text', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Target', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_slice($details, 0, 10) as $link): ?>
                    <tr class="sf-result-row sf-result-success">
                        <td title="<?php echo esc_attr($link['source_url']); ?>">
                            <?php echo esc_html($this->truncate_url($link['source_url'], 35)); ?>
                        </td>
                        <td><?php echo esc_html($link['anchor_text'] ?? $link['keyword'] ?? ''); ?></td>
                        <td title="<?php echo esc_attr($link['target_url']); ?>">
                            <?php echo esc_html($this->truncate_url($link['target_url'], 35)); ?>
                        </td>
                        <td><span class="sf-status-badge sf-status-success">✓ <?php esc_html_e('Added', 'screaming-fixes'); ?></span></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>

</div>
```

### 4.2 Helper Methods for Views

Add to the class:

```php
/**
 * Truncate URL for display
 */
private function truncate_url($url, $max_length = 40) {
    if (strlen($url) <= $max_length) {
        return $url;
    }
    return substr($url, 0, $max_length) . '...';
}

/**
 * Highlight keyword in context string
 */
private function highlight_keyword_in_context($context, $keyword) {
    $escaped_context = esc_html($context);
    $escaped_keyword = esc_html($keyword);

    // Case-insensitive highlight
    $pattern = '/(' . preg_quote($escaped_keyword, '/') . ')/i';
    return preg_replace($pattern, '<mark class="sf-keyword-highlight">$1</mark>', $escaped_context);
}
```

### 4.3 Instructions View (instructions.php)

```php
<?php
/**
 * Internal Link Builder - Instructions
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}
?>

<div class="sf-instructions-box sf-collapsible">
    <div class="sf-instructions-header">
        <h4>
            <span class="dashicons dashicons-info-outline"></span>
            <?php esc_html_e('How to Use Internal Link Builder', 'screaming-fixes'); ?>
        </h4>
        <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
    </div>

    <div class="sf-instructions-content" style="display: none;">
        <p><?php esc_html_e('Boost your priority page by adding internal links from your other content.', 'screaming-fixes'); ?></p>

        <div class="sf-instructions-columns">
            <div class="sf-instruction-column">
                <h5><?php esc_html_e('Option 1: Find Opportunities', 'screaming-fixes'); ?></h5>
                <ol>
                    <li><?php esc_html_e('Enter the URL you want to rank better', 'screaming-fixes'); ?></li>
                    <li><?php esc_html_e('Review and customize the keywords to search for (max 10)', 'screaming-fixes'); ?></li>
                    <li><?php esc_html_e('Scan your site for posts and pages mentioning those keywords', 'screaming-fixes'); ?></li>
                    <li><?php esc_html_e('Review opportunities and add links with one click', 'screaming-fixes'); ?></li>
                </ol>
            </div>

            <div class="sf-instruction-column">
                <h5><?php esc_html_e('Option 2: Bulk Upload', 'screaming-fixes'); ?></h5>
                <p><?php esc_html_e('Upload a CSV with your internal links pre-defined:', 'screaming-fixes'); ?></p>
                <table class="sf-example-table">
                    <thead>
                        <tr>
                            <th>Source_URL</th>
                            <th>Anchor_Text</th>
                            <th>Target_URL</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td>/blog/morning-routine/</td>
                            <td>coffee shops</td>
                            <td>/best-coffee-shops/</td>
                        </tr>
                        <tr>
                            <td>/about/</td>
                            <td>our espresso guide</td>
                            <td>/espresso-guide/</td>
                        </tr>
                    </tbody>
                </table>

                <p class="sf-instruction-note">
                    <strong><?php esc_html_e('Required columns:', 'screaming-fixes'); ?></strong><br>
                    • <code>Source_URL</code> - <?php esc_html_e('The page that will get the new link', 'screaming-fixes'); ?><br>
                    • <code>Anchor_Text</code> - <?php esc_html_e('The text to turn into a link (must exist on the source page)', 'screaming-fixes'); ?><br>
                    • <code>Target_URL</code> - <?php esc_html_e('The page being linked to', 'screaming-fixes'); ?>
                </p>
            </div>
        </div>

        <div class="sf-instruction-tip">
            <span class="dashicons dashicons-lightbulb"></span>
            <strong><?php esc_html_e('Tip:', 'screaming-fixes'); ?></strong>
            <?php esc_html_e('Use AI tools like ChatGPT or Claude to analyze your content and generate a CSV of internal link opportunities, then upload here to implement them.', 'screaming-fixes'); ?>
        </div>

        <div class="sf-instruction-note sf-note-warning">
            <strong><?php esc_html_e('Note:', 'screaming-fixes'); ?></strong>
            <?php esc_html_e('Archive pages (categories, tags, authors, dates) cannot be edited automatically as they don\'t have editable post content. Discovery Mode scans posts and pages only - use Bulk Upload for custom post types.', 'screaming-fixes'); ?>
        </div>
    </div>
</div>
```

---

## Phase 5: Frontend JavaScript (internal-link-builder.js)

```javascript
/**
 * Internal Link Builder Module JavaScript
 */
(function($) {
    'use strict';

    var ILB = {
        // State
        targetUrl: '',
        targetPostId: 0,
        keywords: [],
        opportunities: [],
        bulkData: null,
        processing: false,

        /**
         * Initialize module
         */
        init: function() {
            this.bindEvents();
            this.initDropzone();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            var self = this;

            // Step 1: Extract keywords
            $(document).on('click', '#sf-ilb-extract-btn', function() {
                self.extractKeywords();
            });

            $(document).on('keypress', '#sf-ilb-priority-url', function(e) {
                if (e.which === 13) {
                    self.extractKeywords();
                }
            });

            // Step 2: Keyword management
            $(document).on('change', '.sf-keyword-checkbox', function() {
                self.updateKeywordCount();
            });

            $(document).on('click', '#sf-ilb-add-keyword-btn', function() {
                self.addCustomKeyword();
            });

            $(document).on('keypress', '#sf-ilb-new-keyword', function(e) {
                if (e.which === 13) {
                    self.addCustomKeyword();
                }
            });

            $(document).on('click', '.sf-keyword-remove', function() {
                $(this).closest('.sf-keyword-item').remove();
                self.updateKeywordCount();
            });

            $(document).on('click', '#sf-ilb-back-btn', function() {
                self.showInputSection();
            });

            $(document).on('click', '#sf-ilb-scan-btn', function() {
                self.scanForOpportunities();
            });

            // Step 3: Opportunity actions
            $(document).on('change', '#sf-ilb-select-all', function() {
                var checked = $(this).prop('checked');
                $('.sf-opp-select').prop('checked', checked);
                self.updateSelectedCount();
            });

            $(document).on('change', '.sf-opp-select', function() {
                self.updateSelectedCount();
            });

            $(document).on('click', '.sf-add-single-link', function() {
                var $btn = $(this);
                self.addSingleLink($btn);
            });

            $(document).on('click', '#sf-ilb-add-selected', function() {
                self.addSelectedLinks();
            });

            // Bulk upload actions
            $(document).on('click', '.sf-bulk-confirm', function() {
                self.processBulkLinks();
            });

            // Common actions
            $(document).on('click', '#sf-ilb-clear, .sf-bulk-clear, .sf-clear-results', function() {
                self.clearData();
            });

            $(document).on('click', '#sf-ilb-download-preview, .sf-bulk-download-preview', function() {
                self.downloadPreview();
            });

            $(document).on('click', '.sf-download-results', function() {
                self.downloadResults();
            });

            // Collapsible sections
            $(document).on('click', '.sf-section-toggle', function() {
                var $toggle = $(this);
                var $content = $toggle.next('.sf-manual-content');
                var expanded = $toggle.attr('aria-expanded') === 'true';

                $toggle.attr('aria-expanded', !expanded);
                $content.slideToggle(200);
            });

            // Instructions toggle
            $(document).on('click', '.sf-instructions-header', function() {
                var $content = $(this).next('.sf-instructions-content');
                $content.slideToggle(200);
                $(this).find('.sf-toggle-icon').toggleClass('sf-rotated');
            });
        },

        /**
         * Initialize dropzone
         */
        initDropzone: function() {
            var self = this;
            var $dropzone = $('#sf-ilb-dropzone');
            var $fileInput = $('#sf-ilb-file-input');

            $dropzone.on('click', function() {
                $fileInput.click();
            });

            $dropzone.on('dragover dragenter', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $(this).addClass('sf-dragover');
            });

            $dropzone.on('dragleave drop', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $(this).removeClass('sf-dragover');
            });

            $dropzone.on('drop', function(e) {
                var files = e.originalEvent.dataTransfer.files;
                if (files.length) {
                    self.handleFileUpload(files[0]);
                }
            });

            $fileInput.on('change', function() {
                if (this.files.length) {
                    self.handleFileUpload(this.files[0]);
                }
            });
        },

        /**
         * Extract keywords from URL
         */
        extractKeywords: function() {
            var self = this;
            var url = $('#sf-ilb-priority-url').val().trim();

            if (!url) {
                ScreamingFixes.Toast.warning(sfILBData.i18n.error);
                return;
            }

            // Validate URL format
            if (!url.startsWith('http://') && !url.startsWith('https://')) {
                url = sfILBData.siteUrl + (url.startsWith('/') ? '' : '/') + url;
            }

            var $btn = $('#sf-ilb-extract-btn');
            $btn.prop('disabled', true).text(sfILBData.i18n.extracting);

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_extract_keywords',
                    nonce: sfILBData.nonce,
                    url: url
                },
                success: function(response) {
                    if (response.success) {
                        self.targetUrl = response.data.url;
                        self.targetPostId = response.data.post_id;
                        self.keywords = response.data.keywords;
                        self.showKeywordsSection();
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                },
                error: function() {
                    ScreamingFixes.Toast.error(sfILBData.i18n.error);
                },
                complete: function() {
                    $btn.prop('disabled', false).text('Extract Keywords');
                }
            });
        },

        /**
         * Show keywords section
         */
        showKeywordsSection: function() {
            var self = this;

            $('#sf-ilb-input').hide();
            $('#sf-ilb-keywords').show();

            // Display target URL
            $('.sf-target-url-display').html('<code>' + self.targetUrl + '</code>');

            // Build keyword list
            var $list = $('#sf-ilb-keywords-list');
            $list.empty();

            self.keywords.forEach(function(kw, index) {
                var $item = $(
                    '<div class="sf-keyword-item">' +
                        '<label class="sf-keyword-label">' +
                            '<input type="checkbox" class="sf-keyword-checkbox" ' +
                                (kw.selected ? 'checked' : '') +
                                ' data-keyword="' + self.escapeHtml(kw.keyword) + '">' +
                            '<span class="sf-keyword-text">' + self.escapeHtml(kw.keyword) + '</span>' +
                            '<span class="sf-keyword-source">(' + self.escapeHtml(kw.source_label) + ')</span>' +
                        '</label>' +
                        (kw.source === 'custom' ? '<button type="button" class="sf-keyword-remove" title="Remove">&times;</button>' : '') +
                    '</div>'
                );
                $list.append($item);
            });

            self.updateKeywordCount();
        },

        /**
         * Show input section
         */
        showInputSection: function() {
            $('#sf-ilb-keywords').hide();
            $('#sf-ilb-input').show();
        },

        /**
         * Add custom keyword
         */
        addCustomKeyword: function() {
            var self = this;
            var $input = $('#sf-ilb-new-keyword');
            var keyword = $input.val().trim();

            if (!keyword) return;

            // Check max
            if ($('.sf-keyword-checkbox:checked').length >= 10) {
                ScreamingFixes.Toast.warning(sfILBData.i18n.max_keywords);
                return;
            }

            // Check duplicate
            var exists = false;
            $('.sf-keyword-checkbox').each(function() {
                if ($(this).data('keyword').toLowerCase() === keyword.toLowerCase()) {
                    exists = true;
                    return false;
                }
            });

            if (exists) {
                ScreamingFixes.Toast.warning('Keyword already exists.');
                return;
            }

            // Add to list
            var $item = $(
                '<div class="sf-keyword-item">' +
                    '<label class="sf-keyword-label">' +
                        '<input type="checkbox" class="sf-keyword-checkbox" checked data-keyword="' + self.escapeHtml(keyword) + '">' +
                        '<span class="sf-keyword-text">' + self.escapeHtml(keyword) + '</span>' +
                        '<span class="sf-keyword-source">(custom)</span>' +
                    '</label>' +
                    '<button type="button" class="sf-keyword-remove" title="Remove">&times;</button>' +
                '</div>'
            );

            $('#sf-ilb-keywords-list').append($item);
            $input.val('');
            self.updateKeywordCount();
        },

        /**
         * Update keyword count display
         */
        updateKeywordCount: function() {
            var count = $('.sf-keyword-checkbox:checked').length;
            $('#sf-ilb-selected-count').text(count);

            // Disable scan if no keywords selected
            $('#sf-ilb-scan-btn').prop('disabled', count === 0);

            // Disable add if at max
            $('#sf-ilb-add-keyword-btn').prop('disabled', count >= 10);
        },

        /**
         * Scan for opportunities
         */
        scanForOpportunities: function() {
            var self = this;

            var keywords = [];
            $('.sf-keyword-checkbox:checked').each(function() {
                keywords.push($(this).data('keyword'));
            });

            if (keywords.length === 0) {
                ScreamingFixes.Toast.warning(sfILBData.i18n.no_keywords);
                return;
            }

            var $btn = $('#sf-ilb-scan-btn');
            $btn.prop('disabled', true).text(sfILBData.i18n.scanning);

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_scan_opportunities',
                    nonce: sfILBData.nonce,
                    target_url: self.targetUrl,
                    target_post_id: self.targetPostId,
                    keywords: keywords
                },
                success: function(response) {
                    if (response.success) {
                        self.opportunities = response.data;
                        // Reload page to show results
                        window.location.reload();
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                        $btn.prop('disabled', false).text('Scan for Opportunities');
                    }
                },
                error: function() {
                    ScreamingFixes.Toast.error(sfILBData.i18n.error);
                    $btn.prop('disabled', false).text('Scan for Opportunities');
                }
            });
        },

        /**
         * Update selected count for opportunities
         */
        updateSelectedCount: function() {
            var count = $('.sf-opp-select:checked').length;
            $('#sf-ilb-selected-count-action').text(count);
            $('#sf-ilb-add-selected').prop('disabled', count === 0);
        },

        /**
         * Add single link
         */
        addSingleLink: function($btn) {
            var self = this;
            var postId = $btn.data('post-id');
            var keyword = $btn.data('keyword');
            var sourceUrl = $btn.data('source-url');

            $btn.prop('disabled', true).text('Adding...');

            // Get target URL from stored data
            var targetUrl = '';
            var $targetDisplay = $('.sf-target-url-display code');
            if ($targetDisplay.length) {
                targetUrl = $targetDisplay.text();
            }

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_add_links',
                    nonce: sfILBData.nonce,
                    target_url: targetUrl,
                    links: [{
                        source_post_id: postId,
                        keyword: keyword,
                        source_url: sourceUrl
                    }],
                    offset: 0,
                    batch_size: 1
                },
                success: function(response) {
                    if (response.success && response.data.success > 0) {
                        $btn.closest('tr').addClass('sf-row-success');
                        $btn.text('✓ Added').removeClass('sf-button-primary').addClass('sf-button-success');
                        ScreamingFixes.Toast.success('Link added!');
                    } else {
                        $btn.prop('disabled', false).text('Add Link');
                        ScreamingFixes.Toast.error(response.data.errors?.[0]?.error || 'Failed to add link.');
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).text('Add Link');
                    ScreamingFixes.Toast.error(sfILBData.i18n.error);
                }
            });
        },

        /**
         * Add selected links
         */
        addSelectedLinks: function() {
            var self = this;
            var links = [];

            $('.sf-opp-select:checked').each(function() {
                links.push({
                    source_post_id: $(this).data('post-id'),
                    keyword: $(this).data('keyword'),
                    source_url: $(this).data('source-url')
                });
            });

            if (links.length === 0) {
                ScreamingFixes.Toast.warning('No links selected.');
                return;
            }

            // Get target URL
            var targetUrl = '';
            var $targetDisplay = $('.sf-target-url-display code');
            if ($targetDisplay.length) {
                targetUrl = $targetDisplay.text();
            }

            if (!targetUrl) {
                ScreamingFixes.Toast.error('Target URL not found.');
                return;
            }

            self.processLinksInBatches(links, targetUrl);
        },

        /**
         * Process links in batches
         */
        processLinksInBatches: function(links, targetUrl) {
            var self = this;
            self.processing = true;

            var $modal = $('#sf-ilb-progress-modal');
            $modal.find('.sf-progress-total').text(links.length);
            $modal.find('.sf-progress-current').text(0);
            $modal.find('.sf-progress-percent').text(0);
            $modal.find('.sf-progress-fill').css('width', '0%');
            $modal.show();

            self.processBatch(links, targetUrl, 0, links.length);
        },

        /**
         * Process single batch
         */
        processBatch: function(links, targetUrl, offset, total) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_add_links',
                    nonce: sfILBData.nonce,
                    target_url: targetUrl,
                    links: links,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = offset + (data.processed || 0);
                        var percent = Math.min(100, Math.round((processed / total) * 100));

                        // Update progress
                        var $modal = $('#sf-ilb-progress-modal');
                        $modal.find('.sf-progress-current').text(processed);
                        $modal.find('.sf-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        if (data.complete) {
                            self.processing = false;
                            $modal.hide();
                            ScreamingFixes.Toast.success('Links added successfully!');
                            window.location.reload();
                        } else {
                            // Continue with next batch
                            self.processBatch(links, targetUrl, processed, total);
                        }
                    } else {
                        self.processing = false;
                        $('#sf-ilb-progress-modal').hide();
                        ScreamingFixes.Toast.error(response.data.message || sfILBData.i18n.error);
                    }
                },
                error: function() {
                    self.processing = false;
                    $('#sf-ilb-progress-modal').hide();
                    ScreamingFixes.Toast.error(sfILBData.i18n.error);
                }
            });
        },

        /**
         * Handle file upload
         */
        handleFileUpload: function(file) {
            var self = this;

            if (!file.name.endsWith('.csv')) {
                ScreamingFixes.Toast.error('Please upload a CSV file.');
                return;
            }

            // Use shared uploader if available
            if (window.ScreamingFixes && ScreamingFixes.FileUploader) {
                ScreamingFixes.FileUploader.upload(file, function(uploadId) {
                    self.processUploadedCSV(uploadId);
                });
            } else {
                // Fallback: direct upload
                var formData = new FormData();
                formData.append('file', file);
                formData.append('action', 'sf_upload_csv');
                formData.append('nonce', sfILBData.nonce);

                $.ajax({
                    url: sfILBData.ajaxUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        if (response.success) {
                            self.processUploadedCSV(response.data.upload_id);
                        } else {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    },
                    error: function() {
                        ScreamingFixes.Toast.error(sfILBData.i18n.error);
                    }
                });
            }
        },

        /**
         * Process uploaded CSV
         */
        processUploadedCSV: function(uploadId) {
            var self = this;

            $('#sf-ilb-dropzone').addClass('sf-loading');

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_process_csv',
                    nonce: sfILBData.nonce,
                    upload_id: uploadId
                },
                success: function(response) {
                    if (response.success) {
                        self.bulkData = response.data;
                        window.location.reload();
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                },
                error: function() {
                    ScreamingFixes.Toast.error(sfILBData.i18n.error);
                },
                complete: function() {
                    $('#sf-ilb-dropzone').removeClass('sf-loading');
                }
            });
        },

        /**
         * Process bulk links
         */
        processBulkLinks: function() {
            var self = this;

            if (self.processing) return;
            self.processing = true;

            var $modal = $('#sf-ilb-progress-modal');
            var total = parseInt($('.sf-bulk-stat-ready .sf-bulk-stat-number').text()) || 0;

            $modal.find('.sf-progress-total').text(total);
            $modal.find('.sf-progress-current').text(0);
            $modal.find('.sf-progress-percent').text(0);
            $modal.find('.sf-progress-fill').css('width', '0%');
            $modal.show();

            self.processBulkBatch(0, total);
        },

        /**
         * Process bulk batch
         */
        processBulkBatch: function(offset, total) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_apply_bulk_links',
                    nonce: sfILBData.nonce,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = offset + (data.processed || 0);
                        var percent = Math.min(100, Math.round((processed / total) * 100));

                        var $modal = $('#sf-ilb-progress-modal');
                        $modal.find('.sf-progress-current').text(processed);
                        $modal.find('.sf-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        if (data.complete) {
                            self.processing = false;
                            $modal.hide();
                            ScreamingFixes.Toast.success('Bulk links added!');
                            window.location.reload();
                        } else {
                            self.processBulkBatch(processed, total);
                        }
                    } else {
                        self.processing = false;
                        $('#sf-ilb-progress-modal').hide();
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                },
                error: function() {
                    self.processing = false;
                    $('#sf-ilb-progress-modal').hide();
                    ScreamingFixes.Toast.error(sfILBData.i18n.error);
                }
            });
        },

        /**
         * Clear data
         */
        clearData: function() {
            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_clear_data',
                    nonce: sfILBData.nonce
                },
                success: function() {
                    window.location.reload();
                }
            });
        },

        /**
         * Download preview CSV
         */
        downloadPreview: function() {
            var self = this;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_download_preview',
                    nonce: sfILBData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.downloadCSV(response.data.csv, response.data.filename);
                    } else {
                        ScreamingFixes.Toast.error('Failed to generate CSV.');
                    }
                }
            });
        },

        /**
         * Download results CSV
         */
        downloadResults: function() {
            var self = this;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_download_results',
                    nonce: sfILBData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.downloadCSV(response.data.csv, response.data.filename);
                    } else {
                        ScreamingFixes.Toast.error('Failed to generate CSV.');
                    }
                }
            });
        },

        /**
         * Download CSV helper
         */
        downloadCSV: function(csv, filename) {
            var blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
            var link = document.createElement('a');
            link.href = URL.createObjectURL(blob);
            link.download = filename;
            link.click();
        },

        /**
         * Escape HTML
         */
        escapeHtml: function(str) {
            if (!str) return '';
            var div = document.createElement('div');
            div.textContent = str;
            return div.innerHTML;
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        if ($('.sf-internal-link-builder-module').length) {
            ILB.init();
        }
    });

    // Expose to global scope
    window.ScreamingFixes = window.ScreamingFixes || {};
    window.ScreamingFixes.InternalLinkBuilder = ILB;

})(jQuery);
```

---

## Phase 6: CSS Styling (internal-link-builder.css)

The CSS file should follow the same patterns as other modules. Key sections:

1. **Input Section** - URL input, dropzone styling
2. **Keywords Section** - Keyword list, checkboxes, add button
3. **Opportunities Table** - Same table patterns as other modules
4. **Bulk Confirmation** - Stats cards, preview table, warnings
5. **Progress Modal** - Same as other modules
6. **Results Section** - Same as other modules
7. **Responsive Breakpoints** - 768px, 992px, 1200px

See `modules/page-title/assets/page-title.css` for the complete styling patterns to replicate.

---

## Testing Checklist

### Discovery Mode
- [ ] URL input validation (WordPress URL required)
- [ ] Keyword extraction from SEO plugins, title, H1, slug
- [ ] Custom keyword addition (max 10)
- [ ] Keyword removal
- [ ] Opportunity scanning with progress
- [ ] Exclude pages already linking to target
- [ ] Context display with keyword highlighting
- [ ] Single link addition
- [ ] Bulk selected link addition
- [ ] Manual fix section (collapsed by default)

### Bulk Upload Mode
- [ ] CSV column detection (source, anchor, target)
- [ ] Validation: anchor text exists in source
- [ ] Validation: source URL found in WordPress
- [ ] Validation: link doesn't already exist
- [ ] Duplicate handling (last occurrence wins)
- [ ] Categorization: fixable vs manual vs skipped
- [ ] Batch processing with progress
- [ ] Custom post type support

### Common
- [ ] Preview CSV download
- [ ] Results CSV download
- [ ] Clear/reset functionality
- [ ] Undo capability (SF_Batch_Restore)
- [ ] Large site warning (>2000 posts)
- [ ] 30-second timeout handling

### Edge Cases
- [ ] Multiple keywords on same page
- [ ] Same anchor text appears multiple times (link first only)
- [ ] Anchor text already linked
- [ ] External target URL (allow with warning)
- [ ] Homepage as source
- [ ] Blog index as source

---

## Implementation Order

1. **Phase 1**: Create module structure, register with plugin
2. **Phase 2**: Discovery mode backend (keyword extraction, scanning)
3. **Phase 3**: Bulk upload backend (CSV processing, validation)
4. **Phase 4**: Frontend views (tab-content.php, instructions.php)
5. **Phase 5**: JavaScript handlers
6. **Phase 6**: CSS styling
7. **Testing**: Full test suite

**Estimated Total Lines:** ~2,750 lines across all files
