# Meta Description Module - Implementation Plan

## Overview

Build a module to audit and fix meta descriptions on WordPress posts/pages using Screaming Frog CSV exports. Updates are made via the user's installed SEO plugin (Rank Math, Yoast SEO, or All in One SEO).

## Scope (v1)

- **Input:** CSV upload only (no database scanning)
- **Post types:** Posts and pages only (no CPTs)
- **SEO plugins:** Rank Math, Yoast SEO, All in One SEO
- **Issue types:** Missing, Duplicate, Too Long (>155), Too Short (<70)
- **AI suggestions:** Yes, one at a time (like image-alt-text)
- **Undo:** Yes, restore original value via meta tracking

---

## Prerequisites

### Fix AIOSEO Detection Gap

**File:** `shared/class-plugin-detector.php`

The settings UI shows AIOSEO but the detector doesn't actually detect it. Need to add:

```php
/**
 * Check if All in One SEO is installed and active
 */
public static function is_aioseo_active() {
    return is_plugin_active('all-in-one-seo-pack/all_in_one_seo_pack.php') ||
           is_plugin_active('all-in-one-seo-pack-pro/all_in_one_seo_pack.php');
}

/**
 * Get AIOSEO data for a post
 */
public static function get_aioseo_data($post_id) {
    if (!self::is_aioseo_active()) {
        return null;
    }

    $data = [
        'description' => get_post_meta($post_id, '_aioseo_description', true),
        'title' => get_post_meta($post_id, '_aioseo_title', true),
    ];

    return $data;
}
```

**Update `get_active_seo_plugins()`:**
```php
if (self::is_aioseo_active()) {
    $plugins['aioseo'] = 'All in One SEO';
}
```

---

## CSV Structure

**Headers from Screaming Frog Meta Description export:**

| Column | Purpose |
|--------|---------|
| `Address` | Page URL - match to WordPress post via `url_to_postid()` |
| `Occurrences` | 0=missing, 1=unique, >1=duplicate |
| `Meta Description 1` | The description text (empty if missing) |
| `Meta Description 1 Length` | Character count |
| `Meta Description 1 Pixel Width` | SERP pixel width (store but not primary) |
| `Indexability` | `Indexable` or `Non-Indexable` |
| `Indexability Status` | `noindex`, `Canonicalised`, etc. |

**Filter Logic:**
- Skip rows where `Indexability` = `Non-Indexable`
- Skip rows where `Indexability Status` contains `noindex` or `Canonicalised`

---

## File Structure

```
modules/
└── meta-description/
    ├── class-meta-description.php    # Main module class
    ├── views/
    │   ├── tab-content.php           # Main UI
    │   └── instructions.php          # Screaming Frog export guide
    └── assets/
        ├── meta-description.css
        └── meta-description.js
```

---

## Data Model

### Processed Results Structure

```php
$results = [
    // All descriptions (for filtering)
    'descriptions' => [
        [
            'address' => 'https://example.com/page/',
            'post_id' => 123,
            'post_title' => 'Page Title',
            'edit_url' => 'https://example.com/wp-admin/post.php?post=123&action=edit',
            'current_description' => 'Current meta description text...',
            'description_length' => 45,
            'pixel_width' => 280,
            'issue_type' => 'too_short', // missing, duplicate, too_long, too_short, none
            'duplicate_group' => null, // or hash of description for grouping
            'category' => 'fixable', // fixable, manual, skip
            'category_note' => '', // reason if manual/skip
            'new_description' => '', // user input
        ],
        // ... more rows
    ],

    // Categorized arrays (like other modules)
    'fixable_descriptions' => [...],
    'manual_descriptions' => [...],
    'skipped_descriptions' => [...],
    'fixed_descriptions' => [...], // after applying fixes

    // Issue type arrays (for filtering)
    'missing' => [...],
    'duplicates' => [...], // grouped by description text
    'too_long' => [...],
    'too_short' => [...],

    // Counts
    'total_count' => 150,
    'fixable_count' => 120,
    'manual_count' => 10,
    'skipped_count' => 20,
    'missing_count' => 25,
    'duplicate_count' => 15,
    'too_long_count' => 8,
    'too_short_count' => 12,
    'fixed_count' => 0,

    // Meta
    'processed_at' => '2024-01-15 10:30:00',
    'seo_plugin' => 'rank-math', // which plugin will be used for updates
];
```

---

## Categorization Logic

### Category Assignment

```php
function categorize_description($address, $indexability, $indexability_status) {
    // 1. Check indexability first (from CSV)
    if ($indexability === 'Non-Indexable') {
        return ['category' => 'skip', 'note' => 'Non-indexable page'];
    }

    if (stripos($indexability_status, 'noindex') !== false) {
        return ['category' => 'skip', 'note' => 'Page has noindex'];
    }

    if (stripos($indexability_status, 'canonicalised') !== false) {
        return ['category' => 'skip', 'note' => 'Page is canonicalized'];
    }

    // 2. Try to resolve to post ID
    $post_id = url_to_postid($address);

    if (!$post_id) {
        // Check if it's homepage
        $home_url = trailingslashit(home_url());
        if (trailingslashit($address) === $home_url) {
            return ['category' => 'manual', 'note' => 'Homepage - update in SEO plugin settings', 'post_id' => 0];
        }

        return ['category' => 'skip', 'note' => 'Could not match to a post/page'];
    }

    // 3. Check post type (only posts/pages for v1)
    $post_type = get_post_type($post_id);
    if (!in_array($post_type, ['post', 'page'])) {
        return ['category' => 'skip', 'note' => "Unsupported post type: {$post_type}"];
    }

    // 4. Fixable!
    return ['category' => 'fixable', 'note' => '', 'post_id' => $post_id];
}
```

### Issue Type Assignment

```php
function determine_issue_type($description, $length, $occurrences) {
    if (empty($description) || $length === 0) {
        return 'missing';
    }

    if ($occurrences > 1) {
        return 'duplicate';
    }

    if ($length > 155) {
        return 'too_long';
    }

    if ($length < 70) {
        return 'too_short';
    }

    return 'none'; // No issues
}
```

### Duplicate Detection

After processing all rows, group by description text:

```php
function detect_duplicates($descriptions) {
    $groups = [];

    foreach ($descriptions as $index => $desc) {
        $text = $desc['current_description'];
        if (empty($text)) continue;

        $hash = md5($text);
        if (!isset($groups[$hash])) {
            $groups[$hash] = [];
        }
        $groups[$hash][] = $index;
    }

    // Mark duplicates
    foreach ($groups as $hash => $indices) {
        if (count($indices) > 1) {
            foreach ($indices as $index) {
                $descriptions[$index]['issue_type'] = 'duplicate';
                $descriptions[$index]['duplicate_group'] = $hash;
                $descriptions[$index]['duplicate_count'] = count($indices);
            }
        }
    }

    return $descriptions;
}
```

---

## UI Design

### Filter Dropdown

```html
<select id="sf-issue-filter">
    <option value="all">All Issues (150)</option>
    <option value="missing">Missing (25)</option>
    <option value="duplicate">Duplicate (15)</option>
    <option value="too_long">Over 155 chars (8)</option>
    <option value="too_short">Under 70 chars (12)</option>
</select>
```

### Stats Cards (like other modules)

```
[Total: 150] [Fixable: 120] [Manual: 10] [Skipped: 20]
```

### Issue Type Stats (secondary row)

```
[Missing: 25] [Duplicate: 15] [Too Long: 8] [Too Short: 12]
```

### Results Table

| Column | Content |
|--------|---------|
| Checkbox | Select for batch action |
| Page | Post title + edit link + URL path |
| Current | Current meta description (truncated) + length badge |
| Issue | Issue type badge (Missing/Duplicate/Too Long/Too Short) |
| New Description | Text input + AI suggest button + character counter |

### Character Counter

Show live character count as user types:
- Green: 70-155 characters (optimal)
- Yellow: 50-69 or 156-165 (warning)
- Red: <50 or >165 (bad)

---

## Apply Fixes Logic

### Update Meta Description

```php
function update_meta_description($post_id, $new_description) {
    $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();

    // Get original value for undo tracking
    $original = $this->get_current_description($post_id);

    // Determine which plugin to use (priority order)
    if (isset($seo_plugins['rank-math'])) {
        update_post_meta($post_id, 'rank_math_description', $new_description);
        $plugin_used = 'rank-math';
    } elseif (isset($seo_plugins['yoast'])) {
        update_post_meta($post_id, '_yoast_wpseo_metadesc', $new_description);
        $plugin_used = 'yoast';
    } elseif (isset($seo_plugins['aioseo'])) {
        update_post_meta($post_id, '_aioseo_description', $new_description);
        $plugin_used = 'aioseo';
    } else {
        return new WP_Error('no_seo_plugin', 'No supported SEO plugin detected.');
    }

    return [
        'success' => true,
        'original' => $original,
        'new' => $new_description,
        'plugin' => $plugin_used,
    ];
}
```

### Get Current Description

```php
function get_current_description($post_id) {
    $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();

    if (isset($seo_plugins['rank-math'])) {
        return get_post_meta($post_id, 'rank_math_description', true);
    } elseif (isset($seo_plugins['yoast'])) {
        return get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
    } elseif (isset($seo_plugins['aioseo'])) {
        return get_post_meta($post_id, '_aioseo_description', true);
    }

    return '';
}
```

---

## Undo Implementation

### Key Difference from Other Modules

Meta descriptions are stored in `postmeta`, not `post_content`. WordPress revisions don't track meta changes. We need to:

1. **Store original values** when applying fixes (in batch metadata)
2. **Restore meta values** directly when undoing (not via revisions)

### Batch Tracking Changes

**In `class-batch-restore.php`:**

For meta-description module, we can't use `wp_restore_post_revision()`. Instead:

```php
// In start_batch(), store original meta values
$meta_originals = [];
foreach ($fixes as $fix) {
    $post_id = $fix['post_id'];
    $meta_originals[$post_id] = $this->get_current_description($post_id);
}
$batch['meta_originals'] = $meta_originals;
```

**New undo method for meta-description:**

```php
// In undo_batch(), restore meta values directly
if ($module === 'meta-description') {
    $meta_originals = $batch['meta_originals'] ?? [];
    foreach ($meta_originals as $post_id => $original_value) {
        // Restore using same plugin detection logic
        $this->restore_meta_description($post_id, $original_value);
        $results['restored']++;
    }
}
```

### Update `get_batch_description()`

Add case for meta-description:

```php
case 'meta-description':
    $action_text = sprintf(
        _n('%d description updated', '%d descriptions updated', $item_count, 'screaming-fixes'),
        $item_count
    );
    break;
```

### Update `clear_fixed_links_for_batch()`

Add case for meta-description module to clear `fixed_descriptions` array.

---

## AI Suggestions

### Prompt Template

```php
$prompt = sprintf(
    "Write a compelling meta description for a web page.\n\n" .
    "Page URL: %s\n" .
    "Page Title: %s\n" .
    "Current Description: %s\n\n" .
    "Guidelines:\n" .
    "- Length: 120-155 characters (optimal for search results)\n" .
    "- Include a call to action when appropriate\n" .
    "- Be specific and descriptive\n" .
    "- Don't start with the page title\n" .
    "- Make it compelling for users to click\n\n" .
    "Respond with ONLY the meta description text, nothing else.",
    $address,
    $post_title,
    $current_description ?: '(none)'
);
```

---

## Implementation Steps

### Phase 1: Prerequisites

1. [ ] Add AIOSEO detection to `SF_Plugin_Detector`
2. [ ] Add `is_aioseo_active()` method
3. [ ] Add `get_aioseo_data()` method
4. [ ] Update `get_active_seo_plugins()` to include AIOSEO

### Phase 2: Module Core

5. [ ] Create module directory structure
6. [ ] Create `class-meta-description.php` with:
   - [ ] `can_handle_csv()` - detect meta description CSV headers
   - [ ] `process_csv()` - parse and categorize rows
   - [ ] `apply_fixes()` - update meta descriptions via SEO plugin
   - [ ] `get_current_description()` - read from appropriate meta key
   - [ ] `update_results_after_fixes()` - track fixed items
   - [ ] `get_ai_suggestion()` - Claude API integration

### Phase 3: Batch Restore Integration

7. [ ] Update `start_batch()` to store `meta_originals` for meta-description
8. [ ] Add meta-description case to `undo_batch()` for direct meta restore
9. [ ] Add `restore_meta_description()` helper method
10. [ ] Add meta-description case to `clear_fixed_links_for_batch()`
11. [ ] Add meta-description case to `get_batch_description()`

### Phase 4: UI

12. [ ] Create `views/tab-content.php` with:
    - [ ] Issue filter dropdown
    - [ ] Stats cards (category + issue type)
    - [ ] Results table with inline editing
    - [ ] Character counter
    - [ ] AI suggest button
    - [ ] Batch actions bar
    - [ ] Fixed Descriptions section

13. [ ] Create `views/instructions.php` with Screaming Frog export guide

14. [ ] Create `assets/meta-description.css`
15. [ ] Create `assets/meta-description.js` with:
    - [ ] Filter handling
    - [ ] Character counter
    - [ ] AI suggestion AJAX
    - [ ] Apply fixes AJAX
    - [ ] Checkbox selection

### Phase 5: Settings Update

16. [ ] Remove "Coming Soon" text from settings.php meta tags section

---

## Testing Checklist

- [ ] CSV upload and parsing
- [ ] Correct categorization (fixable/manual/skip)
- [ ] Issue detection (missing/duplicate/too_long/too_short)
- [ ] Filter dropdown works correctly
- [ ] Character counter updates in real-time
- [ ] AI suggestion generates appropriate description
- [ ] Apply fixes updates correct meta key based on active plugin
- [ ] Fixed descriptions appear in Fixed section
- [ ] Data persists when navigating away
- [ ] Undo restores original meta value
- [ ] Batch removed from restore points after undo
- [ ] Fixed descriptions cleared after undo
- [ ] Works with Rank Math
- [ ] Works with Yoast SEO
- [ ] Works with All in One SEO
- [ ] Handles case with no SEO plugin (shows error)

---

## SEO Plugin Meta Keys Reference

| Plugin | Meta Key |
|--------|----------|
| Rank Math | `rank_math_description` |
| Yoast SEO | `_yoast_wpseo_metadesc` |
| All in One SEO | `_aioseo_description` |

---

## Notes

1. **No revision-based undo** - Meta descriptions don't use WordPress revisions, so we track original values in batch metadata and restore directly.

2. **Plugin priority** - If multiple SEO plugins are active (unusual but possible), we use priority: Rank Math > Yoast > AIOSEO.

3. **Homepage special case** - Homepage meta description is typically set in SEO plugin global settings, not post meta. Marked as "manual" for now.

4. **Duplicate handling** - Duplicates are shown grouped together so user can see which pages share the same description and fix them consistently.
