# Plan: Redirect Chains Auto-Fixable UI Consistency

## Objective
Update the Auto-Fixable section in the Redirect Chain tool to match the look, feel, and behavior of the Fixable Descriptions tab in the Meta Description tool for cross-tool UI consistency.

---

## Summary of Changes

| Component | Current State | Target State |
|-----------|--------------|--------------|
| Section Header | Static, non-interactive | Collapsible toggle button (expanded by default) |
| Select All Checkbox | In action bar at top | **REMOVE** (redundant with table header checkbox) |
| "X Selected" Counter | In action bar at top | Move to bottom, next to Fix Selected button |
| "Fix Selected" Button | In action bar at top | Move to bottom inside collapsible section |
| Pagination | 50 per page, simple prev/next | 100 per page, numbered page buttons |
| Filter By | None | Add "Filter By: Type" (Internal/External) |

---

## Files to Modify

1. **`modules/redirect-chains/views/tab-content.php`** - HTML structure changes
2. **`modules/redirect-chains/assets/redirect-chains.css`** - Styling updates
3. **`modules/redirect-chains/assets/redirect-chains.js`** - Behavior/interaction logic

---

## Detailed Implementation Plan

### Phase 1: Remove Redundant Elements from Action Bar

**File:** `tab-content.php` (lines 257-288)

**Changes:**
- Remove the "Select All Fixable" checkbox and label from `.sf-action-left`
- Remove the "X Selected" counter (`#sf-selected-redirect-count`) from `.sf-action-left`
- Remove the "Fix Selected" button (`#sf-apply-redirect-fixes`) from `.sf-action-right`
- Keep: Clear Data button, Export button, View Fixed Redirects button

**Result:** Action bar becomes a simpler toolbar with utility actions only.

---

### Phase 2: Make Auto-Fixable Section Collapsible

**File:** `tab-content.php` (lines 291-300)

**Current:**
```php
<div class="sf-results-section sf-fixable-section">
    <h3 class="sf-section-header">
        <span class="sf-section-icon sf-icon-fixable">✓</span>
        Auto-Fixable (<?php echo $fixable_count; ?> redirects in <?php echo $total_fixable_pages_count; ?> pages)
    </h3>
```

**Target:**
```php
<div class="sf-results-section sf-fixable-section">
    <button type="button" class="sf-section-toggle sf-fixable-toggle" aria-expanded="true">
        <span class="sf-section-badge sf-badge-fixable">&#10003;</span>
        <?php printf(esc_html__('%d Auto-Fixable Redirects', 'screaming-fixes'), $fixable_count); ?>
        <span class="sf-section-hint"><?php esc_html_e('Redirect chains that can be automatically fixed', 'screaming-fixes'); ?></span>
        <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
    </button>
    <div class="sf-fixable-content">
        <!-- Content wrapped here -->
    </div>
</div>
```

**JS Handler:** Add toggle behavior matching Meta Description tool
- `slideToggle(200)` animation
- Toggle `aria-expanded` attribute
- Toggle `.sf-rotated` class on icon

---

### Phase 3: Add Filter By: Type Feature

**File:** `tab-content.php` - Add below the collapsible button, inside `.sf-fixable-content`

**New HTML:**
```php
<div class="sf-filter-row">
    <span class="sf-filter-label"><?php esc_html_e('Filter By:', 'screaming-fixes'); ?></span>
    <div class="sf-filter-buttons">
        <button type="button" class="sf-filter-btn" data-filter="internal">
            <span class="sf-filter-icon">🏠</span>
            <span class="sf-filter-text"><?php esc_html_e('Internal', 'screaming-fixes'); ?></span>
            <span class="sf-filter-count"><?php echo esc_html($internal_count); ?></span>
        </button>
        <button type="button" class="sf-filter-btn" data-filter="external">
            <span class="sf-filter-icon">🌐</span>
            <span class="sf-filter-text"><?php esc_html_e('External', 'screaming-fixes'); ?></span>
            <span class="sf-filter-count"><?php echo esc_html($external_count); ?></span>
        </button>
    </div>
</div>
```

**PHP:** Add count variables for internal/external redirects:
```php
$internal_count = 0;
$external_count = 0;
foreach ($fixable_redirects as $redirect) {
    if (($redirect['redirect_type']['class'] ?? '') === 'sf-type-internal') {
        $internal_count++;
    } else {
        $external_count++;
    }
}
```

**JS Handler:**
- Click filter button → add `.active` class
- Hide non-matching rows with `.sf-filtered-out` class
- Click active filter again → clear filter, show all
- Update pagination after filter change

**CSS:** Match Meta Description `.sf-issue-stat` button styling

---

### Phase 4: Update Pagination (100 per page, numbered buttons)

**File:** `tab-content.php`

**Current:** 50 per page, simple prev/next links
**Target:** 100 per page, numbered page buttons like Meta Description

**PHP Changes:**
```php
$per_page = 100; // Changed from 50
```

**Remove:** Top pagination info bar (lines 308-321)

**Update Bottom Pagination:**
```php
<div class="sf-pagination" data-section="fixable">
    <div class="sf-pagination-info">
        <?php printf(
            esc_html__('Showing %d-%d of %d', 'screaming-fixes'),
            $fixable_offset + 1,
            min($fixable_offset + $per_page, $total_fixable_items),
            $total_fixable_items
        ); ?>
    </div>
    <div class="sf-pagination-controls">
        <button class="sf-page-btn sf-page-prev" <?php echo $current_page_fixable <= 1 ? 'disabled' : ''; ?>>
            ← <?php esc_html_e('Previous', 'screaming-fixes'); ?>
        </button>
        <div class="sf-page-numbers">
            <?php for ($i = 1; $i <= min($total_fixable_pages, 7); $i++) : ?>
                <button class="sf-page-btn sf-page-num <?php echo $i === $current_page_fixable ? 'active' : ''; ?>"
                        data-page="<?php echo $i; ?>">
                    <?php echo $i; ?>
                </button>
            <?php endfor; ?>
        </div>
        <button class="sf-page-btn sf-page-next" <?php echo $current_page_fixable >= $total_fixable_pages ? 'disabled' : ''; ?>>
            <?php esc_html_e('Next', 'screaming-fixes'); ?> →
        </button>
    </div>
</div>
```

**JS:** Add client-side pagination handlers (similar to Meta Description)

**CSS:** Add styling for `.sf-page-num.active`, hover states, disabled states

---

### Phase 5: Add Bottom Batch Actions Bar

**File:** `tab-content.php` - Add inside `.sf-fixable-content`, after the table but before pagination

**New HTML:**
```php
<div class="sf-batch-actions">
    <div class="sf-selected-count">
        <span class="sf-count" id="sf-selected-redirect-count">0</span>
        <?php esc_html_e('selected', 'screaming-fixes'); ?>
    </div>
    <div class="sf-batch-buttons">
        <button type="button" class="sf-button sf-button-primary sf-apply-fixes" id="sf-apply-redirect-fixes" disabled>
            <span class="dashicons dashicons-yes"></span>
            <?php esc_html_e('Fix Selected', 'screaming-fixes'); ?>
        </button>
    </div>
</div>
```

**CSS:** Match Meta Description `.sf-batch-actions` styling:
```css
.sf-batch-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px;
    background: #f9fafb;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    margin-top: 16px;
}
```

---

### Phase 6: JavaScript Updates

**File:** `redirect-chains.js`

**Add/Update Functions:**

1. **Toggle Collapsible Section:**
```javascript
$('.sf-fixable-toggle').on('click', function() {
    var $content = $(this).next('.sf-fixable-content');
    var expanded = $(this).attr('aria-expanded') === 'true';

    $(this).attr('aria-expanded', !expanded);
    $(this).find('.sf-toggle-icon').toggleClass('sf-rotated');
    $content.slideToggle(200);
});
```

2. **Filter By Type:**
```javascript
filterByType: function(filterType) {
    var $rows = $('.sf-redirect-row');

    if (!filterType) {
        // Clear filter
        $rows.removeClass('sf-filtered-out');
        $('.sf-filter-btn').removeClass('active');
    } else {
        $rows.each(function() {
            var rowType = $(this).find('.sf-type-badge').hasClass('sf-type-internal') ? 'internal' : 'external';
            $(this).toggleClass('sf-filtered-out', rowType !== filterType);
            $(this).next('.sf-sources-row').toggleClass('sf-filtered-out', rowType !== filterType);
        });
        $('.sf-filter-btn').removeClass('active');
        $('.sf-filter-btn[data-filter="' + filterType + '"]').addClass('active');
    }

    this.updateSelectedCount();
    this.updatePagination();
}
```

3. **Update `updateSelectedCount()`** - Target new element location (unchanged ID)

4. **Add Client-Side Pagination** (if switching from server-side)

---

## Visual Mockup: Target Layout

```
┌─────────────────────────────────────────────────────────────────────────┐
│ [Clear Data]  [Download CSV]  [View X Fixed Redirects]                  │  ← Simplified Action Bar
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│ ▼ ✓ 45 Auto-Fixable Redirects                                           │  ← Collapsible Header
│     Redirect chains that can be automatically fixed                      │
├─────────────────────────────────────────────────────────────────────────┤
│ Filter By:  [🏠 Internal (32)]  [🌐 External (13)]                       │  ← Filter Buttons
├─────────────────────────────────────────────────────────────────────────┤
│ ☐ │ TYPE     │ REDIRECT URL    │ FINAL DEST  │ HOPS │ FOUND IN │ ACTIONS│
│───┼──────────┼─────────────────┼─────────────┼──────┼──────────┼────────│
│ ☐ │ Internal │ example.com/... │ example.com │  2   │ 3 pages ▼│ [Fix]  │
│ ☐ │ External │ othersit.com/...│ newsite.com │  3   │ 1 page  ▼│ [Fix]  │
│   │          │                 │             │      │          │        │
├─────────────────────────────────────────────────────────────────────────┤
│ 2 selected                                         [✓ Fix Selected]     │  ← Batch Actions Bar
├─────────────────────────────────────────────────────────────────────────┤
│ Showing 1-100 of 245     [←] [1] [2] [3] ... [7] [→]                    │  ← Pagination
└─────────────────────────────────────────────────────────────────────────┘
```

---

## CSS Classes to Add/Reuse

| Class | Purpose | Source |
|-------|---------|--------|
| `.sf-section-toggle` | Collapsible header button | Meta Description |
| `.sf-section-badge` | Badge icon in header | Meta Description |
| `.sf-section-hint` | Subtitle text in header | Meta Description |
| `.sf-toggle-icon.sf-rotated` | Arrow rotation state | Meta Description |
| `.sf-fixable-content` | Collapsible content wrapper | New |
| `.sf-filter-row` | Filter buttons container | New (similar to Meta's `.sf-issue-stats-row`) |
| `.sf-filter-btn` | Individual filter button | New (similar to Meta's `.sf-issue-stat`) |
| `.sf-filter-btn.active` | Active filter state | New |
| `.sf-filtered-out` | Hidden filtered rows | Meta Description |
| `.sf-batch-actions` | Bottom action bar | Meta Description |
| `.sf-selected-count` | Selected counter | Meta Description |
| `.sf-page-btn` | Pagination button | Meta Description |
| `.sf-page-num.active` | Active page number | Meta Description |

---

## Testing Checklist

- [ ] Collapsible section toggles open/closed with smooth animation
- [ ] Section is expanded by default on page load
- [ ] Filter buttons correctly filter rows by Internal/External type
- [ ] Clicking active filter clears the filter
- [ ] Filter counts are accurate
- [ ] "X selected" counter updates when checkboxes change
- [ ] "Fix Selected" button enables/disables based on selection
- [ ] Pagination shows 100 items per page
- [ ] Page numbers navigate correctly
- [ ] Previous/Next buttons work and disable at boundaries
- [ ] Filtering resets pagination to page 1
- [ ] All existing functionality preserved (Fix single, expand sources, etc.)
- [ ] Responsive layout works on smaller screens

---

## Estimated Scope

- **PHP Template:** ~150 lines modified
- **CSS:** ~100 lines added/modified
- **JavaScript:** ~80 lines added/modified
