/**
 * Screaming Fixes Admin JavaScript
 *
 * Core admin functionality: tabs, AJAX, toasts, settings
 */

(function($) {
    'use strict';

    // Namespace
    window.ScreamingFixes = window.ScreamingFixes || {};

    /**
     * Toast Notification System
     */
    ScreamingFixes.Toast = {
        container: null,

        init: function() {
            if (!this.container) {
                this.container = $('<div class="sf-toast-container"></div>');
                $('body').append(this.container);
            }
        },

        show: function(message, type, duration, options) {
            this.init();

            type = type || 'info';
            // Use typeof check so duration=0 means "no auto-dismiss"
            duration = (typeof duration === 'number') ? duration : 4000;
            options = options || {};

            var icons = {
                success: '&#10003;',
                error: '&#10007;',
                warning: '&#9888;',
                info: '&#8505;'
            };

            // Build message content
            var messageContent = message;
            if (options.action) {
                messageContent = '<span class="sf-toast-with-action">' +
                    '<span>' + message + '</span>' +
                    '<a href="#" class="sf-toast-action-link" data-action="' + (options.action.type || '') + '" ' +
                    'data-post-id="' + (options.action.postId || '') + '" ' +
                    'data-post-title="' + (options.action.postTitle || '').replace(/"/g, '&quot;') + '">' +
                    options.action.text +
                    '</a></span>';
            }

            var toast = $(
                '<div class="sf-toast sf-toast-' + type + '">' +
                    '<span class="sf-toast-icon">' + icons[type] + '</span>' +
                    '<span class="sf-toast-message">' + messageContent + '</span>' +
                    '<button type="button" class="sf-toast-close">&times;</button>' +
                '</div>'
            );

            this.container.append(toast);

            // Auto dismiss (longer duration if there's an action)
            // If duration is 0, don't auto-dismiss - user must click close
            var actualDuration = options.action ? Math.max(duration, 8000) : duration;
            var timeout = null;
            if (actualDuration > 0) {
                timeout = setTimeout(function() {
                    ScreamingFixes.Toast.dismiss(toast);
                }, actualDuration);
            }

            // Manual dismiss
            toast.find('.sf-toast-close').on('click', function() {
                if (timeout) {
                    clearTimeout(timeout);
                }
                ScreamingFixes.Toast.dismiss(toast);
            });

            // Handle action click
            toast.find('.sf-toast-action-link').on('click', function(e) {
                e.preventDefault();
                var actionType = $(this).data('action');

                if (actionType === 'viewRevisions') {
                    var postId = $(this).data('post-id');
                    var postTitle = $(this).data('post-title');
                    if (postId && ScreamingFixes.RevisionsModal) {
                        ScreamingFixes.RevisionsModal.open(postId, postTitle);
                    }
                }

                if (timeout) {
                    clearTimeout(timeout);
                }
                ScreamingFixes.Toast.dismiss(toast);
            });

            return toast;
        },

        dismiss: function(toast) {
            toast.addClass('sf-toast-out');
            setTimeout(function() {
                toast.remove();
            }, 300);
        },

        success: function(message, duration, options) {
            return this.show(message, 'success', duration, options);
        },

        error: function(message, duration, options) {
            return this.show(message, 'error', duration, options);
        },

        warning: function(message, duration, options) {
            return this.show(message, 'warning', duration, options);
        },

        info: function(message, duration, options) {
            return this.show(message, 'info', duration, options);
        },

        /**
         * Show success toast with View Revisions link
         *
         * @param {string} message - Success message
         * @param {number} postId - Post ID that was modified
         * @param {string} postTitle - Post title
         * @param {number} duration - Optional duration in ms
         */
        successWithRevisions: function(message, postId, postTitle, duration) {
            return this.show(message, 'success', duration, {
                action: {
                    type: 'viewRevisions',
                    text: 'View Revisions \u2192',
                    postId: postId,
                    postTitle: postTitle
                }
            });
        }
    };

    /**
     * AJAX Helper
     */
    ScreamingFixes.Ajax = {
        request: function(action, data, options) {
            options = options || {};

            var defaults = {
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                dataType: 'json',
                data: $.extend({
                    action: action,
                    nonce: screamingFixesData.nonce
                }, data)
            };

            var settings = $.extend({}, defaults, options);

            return $.ajax(settings);
        },

        post: function(action, data) {
            return this.request(action, data);
        }
    };

    /**
     * Settings Handler
     */
    ScreamingFixes.Settings = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            // API Key toggle visibility
            $('#sf-toggle-api-key').on('click', function() {
                var input = $('#sf-claude-api-key');
                var icon = $(this).find('.dashicons');

                if (input.attr('type') === 'password') {
                    input.attr('type', 'text');
                    icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
                } else {
                    input.attr('type', 'password');
                    icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
                }
            });

            // Test API Key
            $('#sf-test-api-key').on('click', function() {
                var btn = $(this);
                var apiKey = $('#sf-claude-api-key').val();
                var status = $('#sf-api-key-status');

                if (!apiKey) {
                    status.removeClass('sf-status-success').addClass('sf-status-error')
                        .text(screamingFixesData.i18n.enterApiKey || 'Please enter an API key.')
                        .show();
                    return;
                }

                btn.prop('disabled', true).text(screamingFixesData.i18n.testing || 'Testing...');

                ScreamingFixes.Ajax.post('sf_test_api_key', {
                    api_key: apiKey
                }).done(function(response) {
                    if (response.success) {
                        status.removeClass('sf-status-error').addClass('sf-status-success')
                            .text(response.data.message)
                            .show();
                    } else {
                        status.removeClass('sf-status-success').addClass('sf-status-error')
                            .text(response.data.message)
                            .show();
                    }
                }).fail(function() {
                    status.removeClass('sf-status-success').addClass('sf-status-error')
                        .text(screamingFixesData.i18n.requestFailed || 'Request failed. Please try again.')
                        .show();
                }).always(function() {
                    btn.prop('disabled', false).text(screamingFixesData.i18n.test || 'Test');
                });
            });

            // Save API Key
            $('#sf-save-api-key').on('click', function() {
                var btn = $(this);
                var apiKey = $('#sf-claude-api-key').val();

                btn.prop('disabled', true).text(screamingFixesData.i18n.saving || 'Saving...');

                ScreamingFixes.Ajax.post('sf_save_settings', {
                    settings: JSON.stringify({
                        claude_api_key: apiKey
                    })
                }).done(function(response) {
                    if (response.success) {
                        ScreamingFixes.Toast.success(response.data.message);
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                }).fail(function() {
                    ScreamingFixes.Toast.error(screamingFixesData.i18n.saveFailed || 'Failed to save.');
                }).always(function() {
                    btn.prop('disabled', false).text(screamingFixesData.i18n.save || 'Save');
                });
            });

            // Toggle DataForSEO password visibility
            $('#sf-toggle-dataforseo-password').on('click', function() {
                var input = $('#sf-dataforseo-password');
                var icon = $(this).find('.dashicons');

                if (input.attr('type') === 'password') {
                    input.attr('type', 'text');
                    icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
                } else {
                    input.attr('type', 'password');
                    icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
                }
            });

            // Test DataForSEO Connection
            $('#sf-test-dataforseo').on('click', function() {
                var btn = $(this);
                var status = $('#sf-dataforseo-status');
                var login = $('#sf-dataforseo-login').val();
                var password = $('#sf-dataforseo-password').val();

                if (!login || !password) {
                    status.removeClass('sf-status-success').addClass('sf-status-error')
                        .text('Please enter both login and password.')
                        .show();
                    return;
                }

                btn.prop('disabled', true).text(screamingFixesData.i18n.testing || 'Testing...');

                ScreamingFixes.Ajax.post('sf_test_dataforseo', {
                    login: login,
                    password: password
                }).done(function(response) {
                    if (response.success) {
                        var message = response.data.message;
                        if (response.data.balance !== undefined) {
                            message += ' Balance: $' + parseFloat(response.data.balance).toFixed(2);
                        }
                        status.removeClass('sf-status-error').addClass('sf-status-success')
                            .text(message)
                            .show();
                        ScreamingFixes.Toast.success(response.data.message);
                    } else {
                        status.removeClass('sf-status-success').addClass('sf-status-error')
                            .text(response.data.message)
                            .show();
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                }).fail(function() {
                    status.removeClass('sf-status-success').addClass('sf-status-error')
                        .text('Connection test failed.')
                        .show();
                }).always(function() {
                    btn.prop('disabled', false).text(screamingFixesData.i18n.test || 'Test Connection');
                });
            });

            // Save DataForSEO Credentials
            $('#sf-save-dataforseo').on('click', function() {
                var btn = $(this);
                var status = $('#sf-dataforseo-status');
                var login = $('#sf-dataforseo-login').val();
                var password = $('#sf-dataforseo-password').val();

                btn.prop('disabled', true).text(screamingFixesData.i18n.saving || 'Saving...');

                ScreamingFixes.Ajax.post('sf_save_settings', {
                    settings: JSON.stringify({
                        dataforseo_login: login,
                        dataforseo_password: password
                    })
                }).done(function(response) {
                    if (response.success) {
                        status.removeClass('sf-status-error').addClass('sf-status-success')
                            .text(response.data.message)
                            .show();
                        ScreamingFixes.Toast.success(response.data.message);
                    } else {
                        status.removeClass('sf-status-success').addClass('sf-status-error')
                            .text(response.data.message)
                            .show();
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                }).fail(function() {
                    ScreamingFixes.Toast.error(screamingFixesData.i18n.saveFailed || 'Failed to save.');
                }).always(function() {
                    btn.prop('disabled', false).text('Save Credentials');
                });
            });

            // Subscribe email - Footer
            $('#sf-footer-subscribe').on('click', function() {
                var btn = $(this);
                var input = $('#sf-footer-email');
                var email = input.val();
                var status = btn.closest('.sf-footer-cta').find('.sf-footer-status');

                ScreamingFixes.Settings.submitEmailToMailerLite(email, 'wp_plugin_footer', btn, status, input);
            });

            // Subscribe email - Settings page
            $('#sf-subscribe-btn').on('click', function() {
                var btn = $(this);
                var input = $('#sf-subscribe-email');
                var email = input.val();
                var status = btn.closest('.sf-settings-box').find('.sf-subscribe-status');

                ScreamingFixes.Settings.submitEmailToMailerLite(email, 'wp_plugin_settings', btn, status, input);
            });

            // Save redirect plugin preference (from integrations box)
            $('#sf-save-redirect-pref').on('click', function() {
                var btn = $(this);
                var originalText = btn.text();
                var settings = {
                    preferred_redirect_plugin: $('#sf-preferred-redirect').val()
                };

                btn.prop('disabled', true).text(screamingFixesData.i18n.saving || 'Saving...');

                ScreamingFixes.Ajax.post('sf_save_settings', {
                    settings: JSON.stringify(settings)
                }).done(function(response) {
                    if (response.success) {
                        ScreamingFixes.Toast.success(screamingFixesData.i18n.settingSaved || 'Preference saved.');
                        // Reload to update the "Using X for redirects" status
                        setTimeout(function() {
                            window.location.reload();
                        }, 1000);
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                }).fail(function() {
                    ScreamingFixes.Toast.error(screamingFixesData.i18n.saveFailed || 'Failed to save.');
                }).always(function() {
                    btn.prop('disabled', false).text(originalText);
                });
            });

            // Save advanced settings
            $('#sf-save-advanced').on('click', function() {
                var btn = $(this);
                var status = $('#sf-advanced-status');

                var settings = {
                    enable_logging: $('#sf-enable-logging').is(':checked'),
                    debug_mode: $('#sf-debug-mode').is(':checked')
                };

                btn.prop('disabled', true).text(screamingFixesData.i18n.saving || 'Saving...');

                ScreamingFixes.Ajax.post('sf_save_settings', {
                    settings: JSON.stringify(settings)
                }).done(function(response) {
                    if (response.success) {
                        status.text(screamingFixesData.i18n.saved || 'Saved!').show();
                        ScreamingFixes.Toast.success(response.data.message);
                        setTimeout(function() {
                            status.fadeOut();
                        }, 2000);
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                    }
                }).fail(function() {
                    ScreamingFixes.Toast.error(screamingFixesData.i18n.saveFailed || 'Failed to save.');
                }).always(function() {
                    btn.prop('disabled', false).text(screamingFixesData.i18n.saveSettings || 'Save Settings');
                });
            });

            // Individual checkbox changes (auto-save)
            $('.sf-setting-checkbox').on('change', function() {
                var checkbox = $(this);
                var setting = checkbox.data('setting');
                var value = checkbox.is(':checked');

                var settings = {};
                settings[setting] = value;

                ScreamingFixes.Ajax.post('sf_save_settings', {
                    settings: JSON.stringify(settings)
                }).done(function(response) {
                    if (response.success) {
                        ScreamingFixes.Toast.success(screamingFixesData.i18n.settingSaved || 'Setting saved.');
                    }
                });
            });
        },

        isValidEmail: function(email) {
            var regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return regex.test(email);
        },

        /**
         * Submit email to MailerLite via WP REST API
         *
         * @param {string} email - Email address to submit
         * @param {string} source - Source identifier (wp_plugin_footer or wp_plugin_settings)
         * @param {jQuery} btn - The submit button element
         * @param {jQuery} status - The status message element
         * @param {jQuery} input - The email input element
         */
        submitEmailToMailerLite: function(email, source, btn, status, input) {
            var self = this;

            // Validate email
            if (!email || !self.isValidEmail(email)) {
                if (status && status.length) {
                    status.removeClass('sf-status-success').addClass('sf-status-error')
                        .text(screamingFixesData.i18n.invalidEmail || 'Please enter a valid email.')
                        .show();
                }
                ScreamingFixes.Toast.error(screamingFixesData.i18n.invalidEmail || 'Please enter a valid email.');
                return;
            }

            // Disable button
            btn.prop('disabled', true);
            var originalText = btn.text();
            btn.text('Subscribing...');

            // Submit to MailerLite via WP REST API
            $.ajax({
                url: screamingFixesData.restUrl + 'screaming-fixes/v1/subscribe',
                type: 'POST',
                contentType: 'application/json',
                data: JSON.stringify({
                    email: email,
                    source: source
                }),
                success: function(response) {
                    if (response.success) {
                        // Success
                        if (status && status.length) {
                            status.removeClass('sf-status-error').addClass('sf-status-success')
                                .text('Thanks for subscribing!')
                                .show();
                        }
                        ScreamingFixes.Toast.success('Thanks for subscribing!');

                        // Clear input
                        if (input && input.length) {
                            input.val('');
                        }

                        // Store in WordPress to hide form on reload
                        ScreamingFixes.Ajax.post('sf_mark_subscribed', {});
                    } else {
                        throw new Error(response.error || 'Subscription failed');
                    }
                },
                error: function(xhr) {
                    var errorMsg = 'Subscription failed. Please try again.';

                    // Try to get error from response
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.error) {
                            errorMsg = response.error;
                        }
                    } catch (e) {
                        // Use default error message
                    }

                    if (status && status.length) {
                        status.removeClass('sf-status-success').addClass('sf-status-error')
                            .text(errorMsg)
                            .show();
                    }
                    ScreamingFixes.Toast.error(errorMsg);
                },
                complete: function() {
                    btn.prop('disabled', false).text(originalText);
                }
            });
        }
    };

    /**
     * Undo Handler
     */
    ScreamingFixes.Undo = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            $(document).on('click', '.sf-activity-undo', function(e) {
                e.preventDefault();

                var btn = $(this);
                var logId = btn.data('log-id');

                if (!confirm(screamingFixesData.i18n.confirmUndo || 'Are you sure you want to undo this change?')) {
                    return;
                }

                btn.prop('disabled', true).text(screamingFixesData.i18n.undoing || 'Undoing...');

                ScreamingFixes.Ajax.post('sf_undo_change', {
                    log_id: logId
                }).done(function(response) {
                    if (response.success) {
                        ScreamingFixes.Toast.success(response.data.message);
                        btn.closest('.sf-activity-item').fadeOut(function() {
                            $(this).remove();
                        });
                    } else {
                        ScreamingFixes.Toast.error(response.data.message);
                        btn.prop('disabled', false).text(screamingFixesData.i18n.undo || 'Undo');
                    }
                }).fail(function() {
                    ScreamingFixes.Toast.error(screamingFixesData.i18n.undoFailed || 'Undo failed.');
                    btn.prop('disabled', false).text(screamingFixesData.i18n.undo || 'Undo');
                });
            });
        }
    };

    /**
     * Notice Handler
     */
    ScreamingFixes.Notices = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            $(document).on('click', '.sf-notice-dismiss', function() {
                $(this).closest('.sf-notice').fadeOut(function() {
                    $(this).remove();
                });
            });
        }
    };

    /**
     * Tab Handling (for JS-based tabs if needed)
     */
    ScreamingFixes.Tabs = {
        init: function() {
            // Tabs are URL-based, but we can add smooth transitions
            this.highlightCurrent();
        },

        highlightCurrent: function() {
            var currentUrl = window.location.href;
            $('.sf-tab-link').each(function() {
                var link = $(this);
                if (currentUrl.indexOf(link.attr('href')) !== -1) {
                    link.addClass('sf-tab-active');
                }
            });
        }
    };

    /**
     * Revisions Modal Handler
     */
    ScreamingFixes.RevisionsModal = {
        modal: null,
        currentPostId: null,

        init: function() {
            this.modal = $('#sf-revisions-modal');
            this.bindEvents();
        },

        bindEvents: function() {
            var self = this;

            // Open modal when clicking "View Revisions" buttons
            $(document).on('click', '.sf-view-revisions-btn', function(e) {
                e.preventDefault();
                var postId = $(this).data('post-id');
                var postTitle = $(this).data('post-title');
                self.open(postId, postTitle);
            });

            // Close modal
            this.modal.on('click', '.sf-modal-close, .sf-modal-cancel, .sf-modal-overlay', function(e) {
                if (e.target === this) {
                    self.close();
                }
            });

            // ESC key to close
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && self.modal.is(':visible')) {
                    self.close();
                }
            });

            // Open editor button
            this.modal.on('click', '#sf-open-editor-btn', function(e) {
                // Allow default link behavior - it's a real link
                self.close();
            });
        },

        open: function(postId, postTitle) {
            this.currentPostId = postId;

            // Update modal content
            $('#sf-revisions-post-title').text('"' + postTitle + '"');

            // Set the editor link (adminUrl is admin.php, we need the base admin path)
            var adminBase = screamingFixesData.adminUrl.replace('admin.php', '');
            var editUrl = adminBase + 'post.php?post=' + postId + '&action=edit';
            $('#sf-open-editor-btn').attr('href', editUrl);

            // Show modal
            this.modal.fadeIn(200);
            $('body').addClass('sf-modal-open');
        },

        close: function() {
            this.modal.fadeOut(200);
            $('body').removeClass('sf-modal-open');
            this.currentPostId = null;
        }
    };

    /**
     * Restore Points (Batch Undo)
     */
    ScreamingFixes.RestorePoints = {
        container: null,
        batches: [],

        init: function() {
            this.container = $('#sf-restore-points-list');
            if (this.container.length) {
                this.loadBatches();
                this.bindEvents();
            }
        },

        bindEvents: function() {
            var self = this;

            // Undo batch button click
            $(document).on('click', '.sf-undo-batch-btn', function(e) {
                e.preventDefault();
                var batchId = $(this).data('batch-id');
                var batch = self.getBatchById(batchId);
                if (batch) {
                    self.showUndoConfirmation(batch);
                }
            });

            // Modal close/cancel
            $(document).on('click', '.sf-batch-undo-modal .sf-batch-undo-cancel', function() {
                self.closeModal();
            });

            $(document).on('click', '.sf-batch-undo-modal', function(e) {
                if ($(e.target).hasClass('sf-batch-undo-modal')) {
                    self.closeModal();
                }
            });

            // ESC key to close modal
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('.sf-batch-undo-modal').is(':visible')) {
                    self.closeModal();
                }
            });

            // Confirm undo
            $(document).on('click', '.sf-batch-undo-confirm', function() {
                var batchId = $(this).data('batch-id');
                var batch = self.getBatchById(batchId);
                self.executeBatchUndo(batchId, batch ? batch.post_count : 0);
            });
        },

        loadBatches: function() {
            var self = this;

            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_get_restore_batches',
                    nonce: screamingFixesData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.batches) {
                        self.batches = response.data.batches;
                        self.renderBatches();
                    } else {
                        self.renderEmpty();
                    }
                },
                error: function() {
                    self.renderError();
                }
            });
        },

        renderBatches: function() {
            if (!this.batches || this.batches.length === 0) {
                this.renderEmpty();
                return;
            }

            var html = '';
            for (var i = 0; i < this.batches.length; i++) {
                var batch = this.batches[i];
                html += this.renderBatchItem(batch);
            }

            this.container.html(html);
        },

        renderBatchItem: function(batch) {
            return '<div class="sf-restore-point-item" data-batch-id="' + batch.batch_id + '">' +
                '<div class="sf-restore-point-info">' +
                    '<div class="sf-restore-point-header">' +
                        '<span class="sf-restore-point-module">' + this.escapeHtml(batch.module_name) + '</span>' +
                        '<span class="sf-restore-point-time">' + this.escapeHtml(batch.timestamp_formatted) + '</span>' +
                    '</div>' +
                    '<p class="sf-restore-point-description">' + this.escapeHtml(batch.description) + '</p>' +
                '</div>' +
                '<div class="sf-restore-point-actions">' +
                    '<button type="button" class="sf-undo-batch-btn" data-batch-id="' + batch.batch_id + '">' +
                        '&#8617; Undo Batch' +
                    '</button>' +
                '</div>' +
            '</div>';
        },

        renderEmpty: function() {
            this.container.html(
                '<div class="sf-restore-points-empty">' +
                    '<span class="sf-restore-points-empty-icon">&#128260;</span>' +
                    'No restore points yet. Restore points are created when you click "Apply All Fixes".' +
                '</div>'
            );
        },

        renderError: function() {
            this.container.html(
                '<div class="sf-restore-points-empty">' +
                    'Failed to load restore points.' +
                '</div>'
            );
        },

        getBatchById: function(batchId) {
            for (var i = 0; i < this.batches.length; i++) {
                if (this.batches[i].batch_id === batchId) {
                    return this.batches[i];
                }
            }
            return null;
        },

        showUndoConfirmation: function(batch) {
            var self = this;

            // First check for modifications
            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_check_batch_modifications',
                    nonce: screamingFixesData.nonce,
                    batch_id: batch.batch_id
                },
                success: function(response) {
                    var warningHtml = '';
                    if (response.success && response.data.has_modifications) {
                        warningHtml = '<div class="sf-batch-undo-warning">' +
                            '<div class="sf-batch-undo-warning-title">' +
                                '<span>&#9888;</span> Warning' +
                            '</div>' +
                            response.data.count + ' post(s) have been edited since this batch. ' +
                            'Undoing will revert those changes too.' +
                        '</div>';
                    }

                    self.showModal(batch, warningHtml);
                },
                error: function() {
                    // Show modal anyway without warning
                    self.showModal(batch, '');
                }
            });
        },

        showModal: function(batch, warningHtml) {
            var html = '<div class="sf-batch-undo-modal">' +
                '<div class="sf-batch-undo-modal-content">' +
                    '<h3 class="sf-batch-undo-modal-title">Undo Batch Changes?</h3>' +
                    '<p class="sf-batch-undo-modal-message">' +
                        'This will restore <strong>' + batch.post_count + ' post(s)</strong> to their state before the ' +
                        '<strong>' + this.escapeHtml(batch.module_name) + '</strong> changes on ' +
                        this.escapeHtml(batch.timestamp_formatted) + '.' +
                    '</p>' +
                    warningHtml +
                    '<div class="sf-batch-undo-modal-actions">' +
                        '<button type="button" class="sf-batch-undo-cancel">Cancel</button>' +
                        '<button type="button" class="sf-batch-undo-confirm" data-batch-id="' + batch.batch_id + '">' +
                            'Undo ' + batch.post_count + ' Post(s)' +
                        '</button>' +
                    '</div>' +
                '</div>' +
            '</div>';

            $('body').append(html);
        },

        closeModal: function() {
            $('.sf-batch-undo-modal').remove();
        },

        executeBatchUndo: function(batchId, postCount) {
            var self = this;
            var $btn = $('.sf-batch-undo-confirm[data-batch-id="' + batchId + '"]');

            // Show progress message with post count
            var progressMsg = postCount > 0
                ? '<span class="sf-spinner"></span> Restoring ' + postCount + ' post' + (postCount !== 1 ? 's' : '') + '...'
                : '<span class="sf-spinner"></span> Restoring...';
            $btn.prop('disabled', true).html(progressMsg);

            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_undo_batch',
                    nonce: screamingFixesData.nonce,
                    batch_id: batchId
                },
                success: function(response) {
                    self.closeModal();

                    if (response.success) {
                        // Build detailed success message
                        var restored = response.data.restored || 0;
                        var failed = response.data.failed || 0;
                        var skipped = response.data.skipped || 0;

                        var successMsg = 'Successfully restored ' + restored + ' post' + (restored !== 1 ? 's' : '') + '.';
                        if (failed > 0 || skipped > 0) {
                            var extras = [];
                            if (failed > 0) extras.push(failed + ' failed');
                            if (skipped > 0) extras.push(skipped + ' skipped');
                            successMsg += ' (' + extras.join(', ') + ')';
                        }

                        ScreamingFixes.Toast.show(successMsg, 'success', 5000);

                        // Remove the batch from the list
                        self.batches = self.batches.filter(function(b) {
                            return b.batch_id !== batchId;
                        });
                        self.renderBatches();

                        // Show warnings if any
                        if (response.data.warnings && response.data.warnings.length > 0) {
                            setTimeout(function() {
                                var warningMsg = response.data.warnings.length === 1
                                    ? response.data.warnings[0]
                                    : response.data.warnings.length + ' warnings during restore. First: ' + response.data.warnings[0];
                                ScreamingFixes.Toast.show(warningMsg, 'warning', 8000);
                            }, 600);
                        }
                    } else {
                        ScreamingFixes.Toast.show(response.data.message || 'Undo failed', 'error');
                    }
                },
                error: function() {
                    self.closeModal();
                    ScreamingFixes.Toast.show('Failed to undo batch. Please try again.', 'error');
                }
            });
        },

        escapeHtml: function(str) {
            if (!str) return '';
            return String(str)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;');
        }
    };

    /**
     * Loading States
     */
    ScreamingFixes.Loading = {
        show: function(container, message) {
            message = message || 'Loading...';
            var loader = $(
                '<div class="sf-loading-overlay">' +
                    '<div class="sf-loading-content">' +
                        '<div class="sf-spinner"></div>' +
                        '<p>' + message + '</p>' +
                    '</div>' +
                '</div>'
            );
            $(container).css('position', 'relative').append(loader);
        },

        hide: function(container) {
            $(container).find('.sf-loading-overlay').remove();
        }
    };

    /**
     * Activity Log - Show More functionality
     */
    ScreamingFixes.ActivityLog = {
        init: function() {
            var self = this;

            // Show more button click
            $(document).on('click', '.sf-show-more-activity', function() {
                var $btn = $(this);
                var currentShowing = parseInt($btn.data('showing')) || 5;
                var total = parseInt($btn.data('total')) || 25;
                var newLimit = Math.min(currentShowing + 5, total, 25);

                self.loadMore(newLimit, $btn);
            });

            // Export change log CSV
            $(document).on('click', '#sf-export-change-log', function() {
                self.exportChangeLog($(this));
            });

            // Clear change log
            $(document).on('click', '#sf-clear-change-log', function() {
                self.clearChangeLog($(this));
            });
        },

        exportChangeLog: function($btn) {
            var originalText = $btn.html();
            $btn.prop('disabled', true).html('<span class="dashicons dashicons-download"></span> Exporting...');

            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_export_change_log',
                    nonce: screamingFixesData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.csv) {
                        var blob = new Blob([response.data.csv], { type: 'text/csv;charset=utf-8;' });
                        var url = URL.createObjectURL(blob);
                        var link = document.createElement('a');
                        link.href = url;
                        link.download = response.data.filename || 'screaming-fixes-change-log.csv';
                        document.body.appendChild(link);
                        link.click();
                        document.body.removeChild(link);
                        URL.revokeObjectURL(url);

                        ScreamingFixes.Toast.show('Change log exported successfully.', 'success');
                    } else {
                        ScreamingFixes.Toast.show(response.data.message || 'Export failed.', 'error');
                    }
                    $btn.prop('disabled', false).html(originalText);
                },
                error: function() {
                    ScreamingFixes.Toast.show('Failed to export change log.', 'error');
                    $btn.prop('disabled', false).html(originalText);
                }
            });
        },

        clearChangeLog: function($btn) {
            if (!confirm('Are you sure you want to clear the entire change log? This cannot be undone.')) {
                return;
            }

            var originalText = $btn.text();
            $btn.prop('disabled', true).text('Clearing...');

            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_clear_change_log',
                    nonce: screamingFixesData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        ScreamingFixes.Toast.show('Change log cleared.', 'success');
                        // Update the count display
                        $('.sf-log-entry-count').text('0');
                        // Hide export button
                        $('#sf-export-change-log').hide();
                    } else {
                        ScreamingFixes.Toast.show(response.data.message || 'Failed to clear log.', 'error');
                    }
                    $btn.prop('disabled', false).text(originalText);
                },
                error: function() {
                    ScreamingFixes.Toast.show('Failed to clear change log.', 'error');
                    $btn.prop('disabled', false).text(originalText);
                }
            });
        },

        loadMore: function(limit, $btn) {
            var self = this;
            var originalText = $btn.text();
            $btn.prop('disabled', true).text('Loading...');

            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_get_activity_log',
                    nonce: screamingFixesData.nonce,
                    limit: limit
                },
                success: function(response) {
                    if (response.success && response.data.entries) {
                        self.renderEntries(response.data.entries);

                        // Update button state
                        var total = response.data.total;
                        $btn.data('showing', limit);

                        if (limit >= total || limit >= 25) {
                            $btn.closest('.sf-activity-more').hide();
                        } else {
                            $btn.prop('disabled', false).text('Show more');
                        }
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).text(originalText);
                    ScreamingFixes.Toast.error('Failed to load activity log.');
                }
            });
        },

        renderEntries: function(entries) {
            var $list = $('.sf-activity-list');
            var html = '';

            for (var i = 0; i < entries.length; i++) {
                var entry = entries[i];
                html += '<li class="sf-activity-item">' +
                    '<span class="sf-activity-icon">' + entry.icon + '</span>' +
                    '<div class="sf-activity-content">' +
                        '<span class="sf-activity-module">' + this.escapeHtml(entry.module_label) + '</span>' +
                        '<span class="sf-activity-description">' + this.escapeHtml(entry.description) + '</span>' +
                    '</div>' +
                    '<span class="sf-activity-time" title="' + this.escapeHtml(entry.date_formatted) + '">' +
                        this.escapeHtml(entry.time_ago) +
                    '</span>' +
                '</li>';
            }

            $list.html(html);
        },

        escapeHtml: function(str) {
            if (!str) return '';
            return String(str)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;');
        }
    };

    /**
     * Welcome Message - Dismissible one-time message
     */
    ScreamingFixes.WelcomeMessage = {
        init: function() {
            var self = this;

            // Dismiss button click
            $('#sf-welcome-dismiss').on('click', function(e) {
                e.preventDefault();
                self.dismiss();
            });
        },

        dismiss: function() {
            var self = this;
            var $message = $('#sf-welcome-message');

            // Prevent double-clicks
            if ($message.hasClass('sf-welcome-dismissing')) {
                return;
            }

            // Add animation class
            $message.addClass('sf-welcome-dismissing');

            // Send AJAX request to save dismissal
            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_dismiss_welcome',
                    nonce: screamingFixesData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Remove element after animation
                        setTimeout(function() {
                            $message.remove();
                        }, 300);
                    } else {
                        console.error('SF Welcome dismiss failed:', response);
                        $message.removeClass('sf-welcome-dismissing');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('SF Welcome dismiss AJAX error:', status, error);
                    // Still remove on error - user experience > perfect tracking
                    setTimeout(function() {
                        $message.remove();
                    }, 300);
                }
            });
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        // Initialize all modules
        ScreamingFixes.Settings.init();
        ScreamingFixes.Undo.init();
        ScreamingFixes.Notices.init();
        ScreamingFixes.Tabs.init();
        ScreamingFixes.RevisionsModal.init();
        ScreamingFixes.RestorePoints.init();
        ScreamingFixes.ActivityLog.init();
        ScreamingFixes.WelcomeMessage.init();

        // Trigger custom event for other scripts
        $(document).trigger('sf:ready');
    });

})(jQuery);
