/**
 * Screaming Fixes CSV Upload Handler
 *
 * Drag & drop file handling, upload progress, auto-detection
 * Uses overlay approach to prevent drag flickering
 */

// CRITICAL: Prevent browser from opening dropped files (Safari compatible)
// This is a backup - the inline script in dashboard.php should run first
(function() {
    'use strict';

    // Skip if already loaded by inline script
    if (window._sfDragPreventionLoaded) {
        return;
    }
    window._sfDragPreventionLoaded = true;

    // Check if target is inside our upload zone
    var isUploadZone = function(el) {
        while (el) {
            if (el.classList && (el.classList.contains('sf-upload-box') || el.classList.contains('sf-upload-zone'))) {
                return true;
            }
            el = el.parentElement;
        }
        return false;
    };

    var opts = { capture: true, passive: false };

    // For dragover - always preventDefault to allow drop
    var handleDragOver = function(e) {
        e.preventDefault();
        if (e.dataTransfer) {
            e.dataTransfer.dropEffect = isUploadZone(e.target) ? 'copy' : 'none';
        }
    };

    // For drop - block if NOT in upload zone
    var handleDrop = function(e) {
        if (!isUploadZone(e.target)) {
            e.preventDefault();
            e.stopPropagation();
            return false;
        }
        e.preventDefault();
    };

    document.addEventListener('dragover', handleDragOver, opts);
    document.addEventListener('drop', handleDrop, opts);
    window.addEventListener('dragover', handleDragOver, opts);
    window.addEventListener('drop', handleDrop, opts);

    if (document.documentElement) {
        document.documentElement.addEventListener('dragover', handleDragOver, opts);
        document.documentElement.addEventListener('drop', handleDrop, opts);
    }
})();

(function($) {
    'use strict';

    // Namespace
    window.ScreamingFixes = window.ScreamingFixes || {};

    // Track global drag state
    var isDraggingFile = false;

    /**
     * CSV Upload Handler
     */
    ScreamingFixes.CSVUpload = {
        init: function() {
            this.setupGlobalDragHandlers();
            this.setupUploadBoxes();
            this.bindButtonEvents();
        },

        /**
         * Global drag handlers - detect when files enter/leave the window
         * Used for visual feedback (sf-drag-ready class) only
         */
        setupGlobalDragHandlers: function() {
            var self = this;
            var dragEnterCount = 0;

            // Detect when a file drag enters the window (for visual feedback)
            document.addEventListener('dragenter', function(e) {
                dragEnterCount++;
                if (self.hasFiles(e) && !isDraggingFile) {
                    isDraggingFile = true;
                    $('.sf-upload-box').addClass('sf-drag-ready');
                }
            });

            // Remove drag-ready when leaving window entirely
            document.addEventListener('dragleave', function(e) {
                dragEnterCount--;
                if (dragEnterCount <= 0) {
                    dragEnterCount = 0;
                    isDraggingFile = false;
                    $('.sf-upload-box').removeClass('sf-drag-ready sf-drag-over');
                }
            });

            // Cancel drag state on drop anywhere (cleanup after box handlers)
            document.addEventListener('drop', function(e) {
                e.preventDefault();
                dragEnterCount = 0;
                isDraggingFile = false;
                // Small delay to let box handler run first
                setTimeout(function() {
                    $('.sf-upload-box').removeClass('sf-drag-ready sf-drag-over');
                }, 10);
            });

            // Prevent default drag behavior on document
            document.addEventListener('dragover', function(e) {
                e.preventDefault();
            });
        },

        /**
         * Setup each upload box with direct drag handling
         */
        setupUploadBoxes: function() {
            var self = this;

            $('.sf-upload-box').each(function() {
                var box = $(this);
                var zone = box.find('.sf-upload-zone');
                var fileInput = box.find('.sf-file-input');

                // Click on zone to upload
                zone.on('click', function(e) {
                    if (!$(e.target).is('input')) {
                        fileInput.trigger('click');
                    }
                });

                // Keyboard accessibility
                zone.on('keydown', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        fileInput.trigger('click');
                    }
                });

                // File input change
                fileInput.on('change', function() {
                    if (this.files && this.files[0]) {
                        self.handleFile(box, this.files[0]);
                    }
                });

                // DIRECT BINDING: Drag events on the box itself
                // Use capture phase and passive: false for Safari compatibility
                var listenerOptions = { capture: true, passive: false };

                box[0].addEventListener('dragenter', function(e) {
                    // Only respond to file drags
                    if (!self.hasFiles(e)) return;

                    e.preventDefault();
                    e.stopPropagation();
                    box.addClass('sf-drag-over');
                }, listenerOptions);

                box[0].addEventListener('dragover', function(e) {
                    if (!self.hasFiles(e)) return;

                    e.preventDefault();
                    e.stopPropagation();
                    if (e.dataTransfer) {
                        e.dataTransfer.dropEffect = 'copy';
                    }
                    // Maintain the class while dragging over
                    box.addClass('sf-drag-over');
                }, listenerOptions);

                box[0].addEventListener('dragleave', function(e) {
                    e.preventDefault();
                    e.stopPropagation();

                    // Only remove if actually leaving the box bounds
                    var rect = box[0].getBoundingClientRect();
                    if (e.clientX <= rect.left || e.clientX >= rect.right ||
                        e.clientY <= rect.top || e.clientY >= rect.bottom) {
                        box.removeClass('sf-drag-over');
                    }
                }, listenerOptions);

                box[0].addEventListener('drop', function(e) {
                    e.preventDefault();
                    e.stopPropagation();

                    // Reset visual states
                    box.removeClass('sf-drag-over sf-drag-ready');
                    isDraggingFile = false;

                    // Process the dropped files
                    var files = e.dataTransfer && e.dataTransfer.files;
                    if (files && files.length > 0) {
                        self.handleFile(box, files[0]);
                    }
                }, listenerOptions);
            });
        },

        /**
         * Check if drag event contains files
         */
        hasFiles: function(e) {
            if (e.dataTransfer && e.dataTransfer.types) {
                for (var i = 0; i < e.dataTransfer.types.length; i++) {
                    if (e.dataTransfer.types[i] === 'Files') {
                        return true;
                    }
                }
            }
            return false;
        },

        /**
         * Bind button events (try again, remove file, view results)
         */
        bindButtonEvents: function() {
            // Try again button
            $(document).on('click', '.sf-try-again', function() {
                var box = $(this).closest('.sf-upload-box');
                ScreamingFixes.CSVUpload.reset(box);
            });

            // Remove file button
            $(document).on('click', '.sf-remove-file', function() {
                var infoBox = $(this).closest('.sf-file-info');
                var uploadId = infoBox.attr('id').replace('-info', '');
                var box = $('#' + uploadId);

                infoBox.hide();
                ScreamingFixes.CSVUpload.reset(box);
            });

            // View results button
            $(document).on('click', '.sf-view-results', function() {
                var box = $(this).closest('.sf-upload-box');
                var module = box.data('detected-module');
                var uploadId = box.data('upload-id');

                if (module) {
                    var url = screamingFixesData.adminUrl + '?page=screaming-fixes&tab=' + module;
                    if (uploadId) {
                        url += '&upload_id=' + uploadId;
                    }
                    window.location.href = url;
                }
            });
        },

        handleFile: function(box, file) {
            var self = this;

            // Validate file type
            if (!this.isValidCSV(file)) {
                this.showError(box, screamingFixesData.i18n.invalidFileType || 'Please upload a CSV file.');
                return;
            }

            // Validate file size (max 10MB - files larger than this typically exceed the 5,000 row limit)
            var maxSize = 10 * 1024 * 1024;
            if (file.size > maxSize) {
                this.showError(box, screamingFixesData.i18n.fileTooLarge || 'File is too large (max 10MB). Files this size typically exceed the 5,000 row limit. Please filter your export to create a smaller file.');
                return;
            }

            // Check if existing data exists and confirm overwrite
            var module = box.data('module');
            if (module && this.hasExistingData(module)) {
                var confirmMsg = screamingFixesData.i18n.confirmOverwrite ||
                    'You already have data loaded. Uploading a new file will replace your current results. Continue?';
                if (!confirm(confirmMsg)) {
                    return;
                }
            }

            // Show progress
            this.showProgress(box);

            // Create form data
            var formData = new FormData();
            formData.append('action', 'sf_upload_csv');
            formData.append('nonce', screamingFixesData.nonce);
            formData.append('csv_file', file);

            // Add module if specified
            var module = box.data('module');
            if (module) {
                formData.append('module', module);
            }

            // Upload
            $.ajax({
                url: screamingFixesData.ajaxUrl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                xhr: function() {
                    var xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener('progress', function(e) {
                        if (e.lengthComputable) {
                            var percent = Math.round((e.loaded / e.total) * 100);
                            self.updateProgress(box, percent);
                        }
                    }, false);
                    return xhr;
                },
                success: function(response) {
                    if (response.success) {
                        self.showProcessing(box);

                        // Short delay then show success
                        setTimeout(function() {
                            self.showSuccess(box, response.data, file.name);
                        }, 500);
                    } else {
                        self.showError(box, response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.log('SF Upload Error:', status, error);
                    console.log('Response:', xhr.responseText);
                    console.log('Status Code:', xhr.status);
                    var errorMsg = screamingFixesData.i18n.uploadFailed || 'Upload failed. Please try again.';

                    // Try to parse error response
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response && response.data && response.data.message) {
                            errorMsg = response.data.message;
                        }
                    } catch(e) {
                        // Check for common server errors
                        if (xhr.status === 413) {
                            errorMsg = 'File too large. Check your server\'s upload_max_filesize setting.';
                        } else if (xhr.status === 500) {
                            errorMsg = 'Server error. Check PHP error logs for details.';
                        } else if (xhr.status === 0) {
                            errorMsg = 'Request failed. File may be too large or connection timed out.';
                        }
                    }

                    self.showError(box, errorMsg);
                }
            });
        },

        isValidCSV: function(file) {
            var validTypes = ['text/csv', 'application/csv', 'text/comma-separated-values'];
            var extension = file.name.split('.').pop().toLowerCase();

            return validTypes.indexOf(file.type) !== -1 || extension === 'csv';
        },

        showProgress: function(box) {
            box.find('.sf-upload-zone').hide();
            box.find('.sf-upload-progress').show();
            this.updateProgress(box, 0);
        },

        updateProgress: function(box, percent) {
            box.find('.sf-progress-fill').css('width', percent + '%');
            box.find('.sf-progress-percent').text(percent + '%');

            if (percent < 100) {
                box.find('.sf-progress-text').text(screamingFixesData.i18n.uploading || 'Uploading...');
            } else {
                box.find('.sf-progress-text').text(screamingFixesData.i18n.processing || 'Processing...');
            }
        },

        showProcessing: function(box) {
            box.find('.sf-upload-progress').hide();
            box.find('.sf-upload-processing').show();
        },

        showSuccess: function(box, data, fileName) {
            box.find('.sf-upload-processing').hide();

            // Store detected module and upload ID
            box.data('detected-module', data.csv_type);
            box.data('session-id', data.session_id);
            box.data('upload-id', data.upload_id);

            // Build success message
            var typeLabels = {
                'broken-links': screamingFixesData.i18n.brokenLinks || 'Broken Links',
                'redirect-chains': screamingFixesData.i18n.redirectChains || 'Redirect Chains',
                'image-alt-text': screamingFixesData.i18n.imageAltText || 'Image Alt Text',
                'internal-link-builder': screamingFixesData.i18n.internalLinkBuilder || 'Internal Link Builder',
                'backlinks': screamingFixesData.i18n.backlinks || 'Backlinks'
            };

            var typeLabel = typeLabels[data.csv_type] || data.csv_type;
            var rowCount = data.row_count || 0;

            var detailText = '';
            if (data.csv_type !== 'unknown') {
                detailText = typeLabel + ' - ' + rowCount + ' ' + (screamingFixesData.i18n.rowsFound || 'rows found');
            } else {
                detailText = rowCount + ' ' + (screamingFixesData.i18n.rowsFound || 'rows found');
            }

            // Add truncation warning if file was too large
            if (data.truncated && data.total_in_file) {
                detailText += ' (limited from ' + data.total_in_file.toLocaleString() + ' total)';
            }

            box.find('.sf-success-detail').text(detailText);
            box.find('.sf-upload-success').show();

            // Show file info
            var infoBox = $('#' + box.attr('id') + '-info');
            if (infoBox.length) {
                infoBox.find('.sf-file-name').text(fileName);
                infoBox.find('.sf-file-type').text(typeLabel);
                var rowsText = rowCount + ' rows';
                if (data.truncated && data.total_in_file) {
                    rowsText = rowCount.toLocaleString() + ' of ' + data.total_in_file.toLocaleString() + ' rows';
                }
                infoBox.find('.sf-file-rows').text(rowsText);
                infoBox.show();
            }

            // Trigger event
            $(document).trigger('sf:csv-uploaded', [data, box]);

            // Show toast - include truncation warning if applicable
            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                var toastMsg = (screamingFixesData.i18n.uploadSuccess || 'CSV uploaded!') + ' ' + detailText;
                if (data.truncated) {
                    ScreamingFixes.Toast.warning(toastMsg);
                } else {
                    ScreamingFixes.Toast.success(toastMsg);
                }
            }
        },

        showError: function(box, message) {
            box.find('.sf-upload-progress, .sf-upload-processing').hide();
            box.find('.sf-error-detail').text(message);
            box.find('.sf-upload-error').show();

            // Trigger event
            $(document).trigger('sf:csv-upload-error', [message, box]);

            // Show toast
            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.error(message);
            }
        },

        reset: function(box) {
            // Hide all states
            box.find('.sf-upload-dragover, .sf-upload-progress, .sf-upload-processing, .sf-upload-success, .sf-upload-error').hide();

            // Show upload zone
            box.find('.sf-upload-zone').show();

            // Clear file input
            box.find('.sf-file-input').val('');

            // Clear stored data
            box.removeData('detected-module');
            box.removeData('session-id');

            // Reset progress
            box.find('.sf-progress-fill').css('width', '0%');
            box.find('.sf-progress-percent').text('0%');
        },

        getSessionId: function(box) {
            return box.data('session-id');
        },

        getDetectedModule: function(box) {
            return box.data('detected-module');
        },

        /**
         * Check if a module already has data loaded
         * @param {string} module - Module name (e.g., 'broken-links')
         * @return {boolean}
         */
        hasExistingData: function(module) {
            // Check for module-specific indicators that data is loaded
            switch (module) {
                case 'broken-links':
                    return $('.sf-broken-links-module .sf-results-section').length > 0 ||
                           $('.sf-broken-links-module .sf-section-fixable').length > 0;
                case 'image-alt-text':
                    return $('.sf-image-alt-module .sf-results-section').length > 0;
                case 'redirect-chains':
                    return $('.sf-redirect-chains-module .sf-results-section').length > 0;
                case 'backlinks':
                    return $('.sf-backlinks-module .sf-results-section').length > 0;
                default:
                    // Generic check - look for any results section
                    return $('.sf-results-section').length > 0;
            }
        }
    };

    /**
     * Quick Upload (Dashboard)
     * Auto-redirects to the appropriate module after upload
     */
    ScreamingFixes.QuickUpload = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            // Listen for successful uploads on dashboard
            $(document).on('sf:csv-uploaded', function(e, data, box) {
                // Only auto-redirect for dashboard uploads
                if (box.data('auto-detect') === 'true' && data.csv_type !== 'unknown') {
                    // Delay to show success message
                    setTimeout(function() {
                        window.location.href = screamingFixesData.adminUrl +
                            '?page=screaming-fixes&tab=' + data.csv_type +
                            '&upload_id=' + data.upload_id;
                    }, 1500);
                }
            });

            // Listen for module-specific uploads
            $(document).on('sf:csv-uploaded', function(e, data, box) {
                var targetModule = box.data('module');
                if (targetModule && data.csv_type === targetModule) {
                    // Trigger module processing
                    $(document).trigger('sf:csv:uploaded', {
                        module: targetModule,
                        upload_id: data.upload_id,
                        session_id: data.session_id,
                        data: data.data
                    });
                }
            });
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        ScreamingFixes.CSVUpload.init();
        ScreamingFixes.QuickUpload.init();
    });

})(jQuery);
