<?php
/**
 * Base module class for Screaming Fixes modules
 */

if (!defined('ABSPATH')) {
    exit;
}

abstract class SF_Module {
    /**
     * Module name
     */
    protected $name = '';

    /**
     * Module slug
     */
    protected $slug = '';

    /**
     * Module description
     */
    protected $description = '';

    /**
     * Module is enabled
     */
    protected $enabled = true;

    /**
     * Initialize the module
     */
    public function __construct() {
        $this->init();
    }

    /**
     * Initialize module - to be overridden by child classes
     */
    public function init() {
        // Override in child class
    }

    /**
     * Get module name
     */
    public function get_name() {
        return $this->name;
    }

    /**
     * Get module slug
     */
    public function get_slug() {
        return $this->slug;
    }

    /**
     * Get module description
     */
    public function get_description() {
        return $this->description;
    }

    /**
     * Check if module is enabled
     */
    public function is_enabled() {
        return $this->enabled && get_option('sf_enable_' . $this->slug, true);
    }

    /**
     * Enable the module
     */
    public function enable() {
        $this->enabled = true;
        update_option('sf_enable_' . $this->slug, true);
    }

    /**
     * Disable the module
     */
    public function disable() {
        $this->enabled = false;
        update_option('sf_enable_' . $this->slug, false);
    }

    /**
     * Run the module scan/check - to be overridden by child classes
     */
    abstract public function run_scan();

    /**
     * Get module settings - to be overridden by child classes
     */
    public function get_settings() {
        return [];
    }

    /**
     * Save module settings - to be overridden by child classes
     */
    public function save_settings($settings) {
        // Override in child class
    }

    /**
     * Get results for the module
     */
    public function get_results() {
        $results = get_transient('sf_' . $this->slug . '_results');
        return $results !== false ? $results : [];
    }

    /**
     * Save results for the module
     */
    protected function save_results($results) {
        set_transient('sf_' . $this->slug . '_results', $results, HOUR_IN_SECONDS);
    }

    /**
     * Clear results for the module
     */
    protected function clear_results() {
        delete_transient('sf_' . $this->slug . '_results');
    }
}
