<?php
/**
 * Admin functionality for Screaming Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Admin {

    /**
     * Initialize admin functionality
     */
    public function __construct() {
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_head', [$this, 'hide_third_party_notices'], 1);
        // CRITICAL: Add drag prevention script as early as possible
        // Using multiple hooks at lowest priority to ensure it runs first
        add_action('admin_head', [$this, 'print_drag_prevention_script'], -999);
        add_action('admin_print_scripts', [$this, 'print_drag_prevention_script'], -999);
    }

    /**
     * Hide third-party admin notices on Screaming Fixes pages
     *
     * Prevents notices from other plugins (like Redirection setup messages)
     * from cluttering our plugin's admin interface.
     */
    public function hide_third_party_notices() {
        $screen = get_current_screen();

        // Only apply to Screaming Fixes pages
        if (!$screen || strpos($screen->id, 'screaming-fixes') === false) {
            return;
        }

        // Remove all admin notices except our own
        remove_all_actions('admin_notices');
        remove_all_actions('all_admin_notices');

        // Re-add our own notice hook if we have any
        add_action('admin_notices', [$this, 'render_plugin_notices']);
    }

    /**
     * Render our own plugin notices (if any)
     */
    public function render_plugin_notices() {
        // Placeholder for any Screaming Fixes-specific notices
        // Can be extended later to show plugin-specific messages
    }

    /**
     * Register admin menu and pages
     *
     * Uses single page with tab navigation per PRD
     */
    public function register_admin_menu() {
        // Main menu page
        add_menu_page(
            __('Screaming Fixes', 'screaming-fixes'),
            __('Screaming Fixes', 'screaming-fixes'),
            'manage_options',
            'screaming-fixes',
            [$this, 'render_dashboard'],
            'dashicons-admin-tools',
            30
        );

        // Dashboard submenu (hidden duplicate of main)
        add_submenu_page(
            'screaming-fixes',
            __('Dashboard', 'screaming-fixes'),
            __('Dashboard', 'screaming-fixes'),
            'manage_options',
            'screaming-fixes'
        );

        // Note: Other menu items removed - using single page with tabs
        // as specified in the PRD navigation structure
    }

    /**
     * Render main dashboard
     * Routes to appropriate view based on ?tab= parameter
     */
    public function render_dashboard() {
        include SF_PLUGIN_DIR . 'admin/views/dashboard.php';
    }

    /**
     * Enqueue admin CSS and JavaScript
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our plugin pages
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        // Use file modification time for cache busting during development
        $css_version = $this->get_asset_version('admin/assets/css/admin.css');
        $admin_js_version = $this->get_asset_version('admin/assets/js/admin.js');
        $csv_js_version = $this->get_asset_version('admin/assets/js/csv-upload.js');

        // Main admin styles
        wp_enqueue_style(
            'screaming-fixes-admin',
            SF_PLUGIN_URL . 'admin/assets/css/admin.css',
            [],
            $css_version
        );

        // Dashicons for icons
        wp_enqueue_style('dashicons');

        // Main admin script
        wp_enqueue_script(
            'screaming-fixes-admin',
            SF_PLUGIN_URL . 'admin/assets/js/admin.js',
            ['jquery'],
            $admin_js_version,
            true
        );

        // CSV upload script
        wp_enqueue_script(
            'screaming-fixes-csv-upload',
            SF_PLUGIN_URL . 'admin/assets/js/csv-upload.js',
            ['jquery', 'screaming-fixes-admin'],
            $csv_js_version,
            true
        );

        // Localized data for JavaScript
        wp_localize_script(
            'screaming-fixes-admin',
            'screamingFixesData',
            $this->get_js_data()
        );
    }

    /**
     * Print inline script to prevent browser default drag/drop behavior
     * This runs in the <head> to ensure it's active before any user interaction
     */
    public function print_drag_prevention_script() {
        // Only output once
        static $printed = false;
        if ($printed) {
            return;
        }

        // Only output on Screaming Fixes pages
        // Check URL directly since get_current_screen() may not be ready at early priorities
        $page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
        if (strpos($page, 'screaming-fixes') === false) {
            return;
        }

        $printed = true;
        ?>
        <script>
        // SF Drag Prevention - prevents browser from opening dropped files (Safari compatible)
        if (!window._sfDragPreventionLoaded) {
            window._sfDragPreventionLoaded = true;
            (function(){
                // Check if target is inside our upload zone
                var isUploadZone = function(el) {
                    while (el) {
                        if (el.classList && (el.classList.contains('sf-upload-box') || el.classList.contains('sf-upload-zone'))) {
                            return true;
                        }
                        el = el.parentElement;
                    }
                    return false;
                };

                var opts = { capture: true, passive: false };

                // For dragover - always preventDefault to allow drop
                var handleDragOver = function(e) {
                    e.preventDefault();
                    if (e.dataTransfer) {
                        e.dataTransfer.dropEffect = isUploadZone(e.target) ? 'copy' : 'none';
                    }
                };

                // For drop - block if NOT in upload zone
                var handleDrop = function(e) {
                    if (!isUploadZone(e.target)) {
                        e.preventDefault();
                        e.stopPropagation();
                        return false;
                    }
                    e.preventDefault();
                };

                document.addEventListener('dragover', handleDragOver, opts);
                document.addEventListener('drop', handleDrop, opts);
                window.addEventListener('dragover', handleDragOver, opts);
                window.addEventListener('drop', handleDrop, opts);

                if (document.documentElement) {
                    document.documentElement.addEventListener('dragover', handleDragOver, opts);
                    document.documentElement.addEventListener('drop', handleDrop, opts);
                }
            })();
        }
        </script>
        <?php
    }

    /**
     * Get data to pass to JavaScript
     *
     * @return array
     */
    private function get_js_data() {
        return [
            'nonce' => wp_create_nonce('screaming_fixes_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'adminUrl' => admin_url('admin.php'),
            'restUrl' => rest_url(),
            'pluginUrl' => SF_PLUGIN_URL,
            'i18n' => [
                // General
                'loading' => __('Loading...', 'screaming-fixes'),
                'saving' => __('Saving...', 'screaming-fixes'),
                'saved' => __('Saved!', 'screaming-fixes'),
                'save' => __('Save', 'screaming-fixes'),
                'saveSettings' => __('Save Settings', 'screaming-fixes'),
                'saveFailed' => __('Failed to save.', 'screaming-fixes'),
                'test' => __('Test', 'screaming-fixes'),
                'testing' => __('Testing...', 'screaming-fixes'),
                'undo' => __('Undo', 'screaming-fixes'),
                'undoing' => __('Undoing...', 'screaming-fixes'),
                'undoFailed' => __('Undo failed.', 'screaming-fixes'),
                'confirmUndo' => __('Are you sure you want to undo this change?', 'screaming-fixes'),
                'requestFailed' => __('Request failed. Please try again.', 'screaming-fixes'),

                // Settings
                'enterApiKey' => __('Please enter an API key.', 'screaming-fixes'),
                'invalidEmail' => __('Please enter a valid email.', 'screaming-fixes'),
                'subscribeFailed' => __('Subscription failed.', 'screaming-fixes'),
                'settingSaved' => __('Setting saved.', 'screaming-fixes'),

                // Upload
                'uploading' => __('Uploading...', 'screaming-fixes'),
                'processing' => __('Processing...', 'screaming-fixes'),
                'uploadSuccess' => __('CSV uploaded!', 'screaming-fixes'),
                'uploadFailed' => __('Upload failed. Please try again.', 'screaming-fixes'),
                'invalidFileType' => __('Please upload a CSV file.', 'screaming-fixes'),
                'fileTooLarge' => __('File is too large. Maximum size is 10MB.', 'screaming-fixes'),
                'rowsFound' => __('rows found', 'screaming-fixes'),

                // Module names
                'brokenLinks' => __('Broken Links', 'screaming-fixes'),
                'redirectChains' => __('Redirect Chains', 'screaming-fixes'),
                'imageAltText' => __('Image Alt Text', 'screaming-fixes'),
                'backlinks' => __('Backlinks', 'screaming-fixes'),

                // Fixes
                'applyingFixes' => __('Applying fixes...', 'screaming-fixes'),
                'fixesApplied' => __('Fixes applied!', 'screaming-fixes'),
                'fixesFailed' => __('Some fixes failed.', 'screaming-fixes'),
                'noFixesSelected' => __('No fixes selected.', 'screaming-fixes'),
            ],
        ];
    }

    /**
     * Add admin body class for styling
     *
     * @param string $classes
     * @return string
     */
    public function admin_body_class($classes) {
        $screen = get_current_screen();

        if ($screen && strpos($screen->id, 'screaming-fixes') !== false) {
            $classes .= ' screaming-fixes-admin';
        }

        return $classes;
    }

    /**
     * Get asset version for cache busting
     * Uses file modification time in development, SF_VERSION in production
     *
     * @param string $relative_path Path relative to plugin directory
     * @return string Version string
     */
    private function get_asset_version($relative_path) {
        $file_path = SF_PLUGIN_DIR . $relative_path;

        // Use file modification time for automatic cache busting
        if (file_exists($file_path)) {
            return SF_VERSION . '.' . filemtime($file_path);
        }

        return SF_VERSION;
    }
}
