<?php
/**
 * Batch Restore System for Screaming Fixes
 *
 * Tracks batches of changes and enables one-click undo via WordPress revisions.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Batch_Restore {

    /**
     * Option name for storing batches
     */
    const OPTION_NAME = 'sf_restore_batches';

    /**
     * Maximum number of batches to keep
     */
    const MAX_BATCHES = 25;

    /**
     * Maximum age of batches in days
     */
    const MAX_AGE_DAYS = 90;

    /**
     * Current batch being built (before apply)
     * @var array|null
     */
    private static $current_batch = null;

    /**
     * Start a new batch - call this BEFORE applying fixes
     *
     * @param string $module Module name (e.g., 'broken-links')
     * @param array $fixes Array of fixes to be applied
     * @return string Batch ID
     */
    public static function start_batch($module, $fixes) {
        $batch_id = 'batch_' . time() . '_' . wp_generate_password(8, false);

        // Collect all post IDs that will be affected
        $post_ids = [];
        foreach ($fixes as $fix) {
            if (!empty($fix['post_ids'])) {
                $post_ids = array_merge($post_ids, (array) $fix['post_ids']);
            }
        }
        $post_ids = array_unique(array_filter($post_ids));

        // Get current revision IDs for each post BEFORE changes
        $pre_revisions = [];
        foreach ($post_ids as $post_id) {
            $post_id = absint($post_id);
            if (!$post_id) {
                continue;
            }

            // Get the most recent revision before our changes
            $revisions = wp_get_post_revisions($post_id, [
                'numberposts' => 1,
                'orderby' => 'ID',
                'order' => 'DESC',
            ]);

            if (!empty($revisions)) {
                $revision = array_shift($revisions);
                $pre_revisions[$post_id] = $revision->ID;
            } else {
                // No revision exists yet - store the current post state
                // We'll need to check post_modified later to detect if it was changed
                $post = get_post($post_id);
                if ($post) {
                    $pre_revisions[$post_id] = [
                        'type' => 'no_revision',
                        'post_modified' => $post->post_modified,
                    ];
                }
            }
        }

        // Count items being fixed (non-ignored) and collect fixed URLs
        $item_count = 0;
        $fixed_urls = [];
        foreach ($fixes as $fix) {
            // For broken-links: check action !== 'ignore'
            // For redirect-chains: all items are fixes (no action field)
            $is_fix = true;
            if (isset($fix['action']) && $fix['action'] === 'ignore') {
                $is_fix = false;
            }

            if ($is_fix) {
                $item_count++;
                // Store the URL that was fixed
                // broken-links uses: broken_url or url
                // redirect-chains uses: address
                // image-alt-text uses: image_url
                // meta-description uses: address
                $url = $fix['broken_url'] ?? $fix['url'] ?? $fix['address'] ?? $fix['image_url'] ?? '';

                // For meta-description, also store original values for direct meta restore
                if (!empty($fix['original_description'])) {
                    $fix['_meta_original'] = $fix['original_description'];
                }
                if (!empty($url)) {
                    $fixed_urls[] = $url;
                }
            }
        }

        // For meta-description module, store original meta values for direct restore
        $meta_originals = [];
        if ($module === 'meta-description') {
            foreach ($fixes as $fix) {
                $post_id = $fix['post_id'] ?? 0;
                if ($post_id && !empty($fix['original_description'])) {
                    $meta_originals[$post_id] = $fix['original_description'];
                }
            }
        }

        self::$current_batch = [
            'batch_id' => $batch_id,
            'module' => $module,
            'timestamp' => current_time('mysql'),
            'timestamp_unix' => time(),
            'user_id' => get_current_user_id(),
            'post_ids' => $post_ids,
            'pre_revisions' => $pre_revisions,
            'item_count' => $item_count,
            'post_count' => count($post_ids),
            'fixed_urls' => array_unique($fixed_urls),
            'meta_originals' => $meta_originals,
        ];

        return $batch_id;
    }

    /**
     * Complete the batch - call this AFTER applying fixes
     *
     * @param array $results Results from apply_fixes
     * @return bool Success
     */
    public static function complete_batch($results = []) {
        if (empty(self::$current_batch)) {
            return false;
        }

        $batch = self::$current_batch;

        // Update counts with actual results
        if (!empty($results)) {
            $batch['success_count'] = $results['success'] ?? 0;
            $batch['failed_count'] = $results['failed'] ?? 0;
        }

        // Get post revisions AFTER changes to verify changes were made
        $post_revisions = [];
        foreach ($batch['post_ids'] as $post_id) {
            $post_id = absint($post_id);
            if (!$post_id) {
                continue;
            }

            // Get the revision created by our changes
            $revisions = wp_get_post_revisions($post_id, [
                'numberposts' => 1,
                'orderby' => 'ID',
                'order' => 'DESC',
            ]);

            $pre_revision = $batch['pre_revisions'][$post_id] ?? null;

            if (!empty($revisions)) {
                $new_revision = array_shift($revisions);

                // Check if this is a new revision (created by our changes)
                if (is_array($pre_revision)) {
                    // Had no revision before - this is new
                    $post_revisions[$post_id] = [
                        'restore_to' => null, // Will need to use post content backup
                        'current_revision' => $new_revision->ID,
                    ];
                } elseif ($pre_revision && $new_revision->ID !== $pre_revision) {
                    // New revision was created - store the pre-change revision to restore to
                    $post_revisions[$post_id] = [
                        'restore_to' => $pre_revision,
                        'current_revision' => $new_revision->ID,
                    ];
                }
            }
        }

        $batch['post_revisions'] = $post_revisions;
        $batch['posts_changed'] = count($post_revisions);

        // Only save if actual changes were made
        if ($batch['posts_changed'] > 0) {
            self::save_batch($batch);
        }

        self::$current_batch = null;

        return true;
    }

    /**
     * Save a batch to storage
     *
     * @param array $batch Batch data
     */
    private static function save_batch($batch) {
        $batches = self::get_all_batches();

        // Add new batch at the beginning (newest first)
        array_unshift($batches, $batch);

        // Enforce limits
        $batches = self::enforce_limits($batches);

        update_option(self::OPTION_NAME, $batches, false);
    }

    /**
     * Get all stored batches
     *
     * @return array
     */
    public static function get_all_batches() {
        $batches = get_option(self::OPTION_NAME, []);

        if (!is_array($batches)) {
            return [];
        }

        // Clean up expired batches
        $batches = self::enforce_limits($batches);

        return $batches;
    }

    /**
     * Get a specific batch by ID
     *
     * @param string $batch_id
     * @return array|null
     */
    public static function get_batch($batch_id) {
        $batches = self::get_all_batches();

        foreach ($batches as $batch) {
            if ($batch['batch_id'] === $batch_id) {
                return $batch;
            }
        }

        return null;
    }

    /**
     * Enforce batch limits (max count and max age)
     *
     * @param array $batches
     * @return array Filtered batches
     */
    private static function enforce_limits($batches) {
        $cutoff_time = time() - (self::MAX_AGE_DAYS * DAY_IN_SECONDS);

        // Filter out expired batches
        $batches = array_filter($batches, function($batch) use ($cutoff_time) {
            return ($batch['timestamp_unix'] ?? 0) > $cutoff_time;
        });

        // Enforce max count
        if (count($batches) > self::MAX_BATCHES) {
            $batches = array_slice($batches, 0, self::MAX_BATCHES);
        }

        return array_values($batches);
    }

    /**
     * Undo a batch - restore all posts to their pre-change state
     *
     * @param string $batch_id
     * @return array Results with success/failure counts
     */
    public static function undo_batch($batch_id) {
        $batch = self::get_batch($batch_id);

        if (!$batch) {
            return [
                'success' => false,
                'message' => __('Batch not found.', 'screaming-fixes'),
            ];
        }

        $results = [
            'success' => true,
            'restored' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => [],
            'warnings' => [],
        ];

        $module = $batch['module'] ?? '';

        // Meta-description uses direct meta restore (not revisions)
        if ($module === 'meta-description') {
            $results = self::undo_meta_description_batch($batch, $results);

            // Remove the batch from storage after successful undo
            if ($results['restored'] > 0) {
                self::clear_fixed_links_for_batch($batch);
                self::remove_batch($batch_id);
            }

            $results['message'] = sprintf(
                __('Restored %d of %d meta descriptions.', 'screaming-fixes'),
                $results['restored'],
                $results['restored'] + $results['failed'] + $results['skipped']
            );

            return $results;
        }

        $post_revisions = $batch['post_revisions'] ?? [];

        foreach ($post_revisions as $post_id => $revision_data) {
            $post_id = absint($post_id);
            $restore_to = $revision_data['restore_to'] ?? null;

            // Check if post still exists
            $post = get_post($post_id);
            if (!$post) {
                $results['skipped']++;
                $results['warnings'][] = sprintf(
                    __('Post %d no longer exists.', 'screaming-fixes'),
                    $post_id
                );
                continue;
            }

            // Check if revision still exists
            if ($restore_to) {
                $revision = get_post($restore_to);
                if (!$revision || $revision->post_type !== 'revision') {
                    $results['failed']++;
                    $results['errors'][] = sprintf(
                        __('Revision for post "%s" no longer exists (may have been pruned).', 'screaming-fixes'),
                        $post->post_title
                    );
                    continue;
                }

                // Check if post was modified since our batch
                $current_revision_id = $revision_data['current_revision'] ?? 0;
                $latest_revisions = wp_get_post_revisions($post_id, [
                    'numberposts' => 1,
                    'orderby' => 'ID',
                    'order' => 'DESC',
                ]);

                if (!empty($latest_revisions)) {
                    $latest = array_shift($latest_revisions);
                    if ($latest->ID !== $current_revision_id) {
                        $results['warnings'][] = sprintf(
                            __('Post "%s" was modified after this batch. Restoring anyway.', 'screaming-fixes'),
                            $post->post_title
                        );
                    }
                }

                // Restore the revision
                $restored = wp_restore_post_revision($restore_to);

                if ($restored) {
                    $results['restored']++;
                } else {
                    $results['failed']++;
                    $results['errors'][] = sprintf(
                        __('Failed to restore post "%s".', 'screaming-fixes'),
                        $post->post_title
                    );
                }
            } else {
                // No revision to restore to - this shouldn't happen normally
                $results['skipped']++;
                $results['warnings'][] = sprintf(
                    __('No revision available for post "%s".', 'screaming-fixes'),
                    $post->post_title
                );
            }
        }

        // Remove the batch from storage after successful undo
        if ($results['restored'] > 0) {
            // Clear the fixed_links from the module's stored results
            self::clear_fixed_links_for_batch($batch);

            self::remove_batch($batch_id);
        }

        $results['message'] = sprintf(
            __('Restored %d of %d posts.', 'screaming-fixes'),
            $results['restored'],
            $results['restored'] + $results['failed'] + $results['skipped']
        );

        return $results;
    }

    /**
     * Undo meta description batch by restoring original meta values
     *
     * @param array $batch The batch data
     * @param array $results Current results array
     * @return array Updated results
     */
    private static function undo_meta_description_batch($batch, $results) {
        $meta_originals = $batch['meta_originals'] ?? [];

        if (empty($meta_originals)) {
            $results['message'] = __('No original values stored for this batch.', 'screaming-fixes');
            return $results;
        }

        // Determine which SEO plugin to use
        $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();
        $meta_key = null;

        if (isset($seo_plugins['rank-math'])) {
            $meta_key = 'rank_math_description';
        } elseif (isset($seo_plugins['yoast'])) {
            $meta_key = '_yoast_wpseo_metadesc';
        } elseif (isset($seo_plugins['aioseo'])) {
            $meta_key = '_aioseo_description';
        }

        if (!$meta_key) {
            $results['success'] = false;
            $results['errors'][] = __('No supported SEO plugin detected.', 'screaming-fixes');
            return $results;
        }

        foreach ($meta_originals as $post_id => $original_value) {
            $post_id = absint($post_id);

            // Check if post still exists
            $post = get_post($post_id);
            if (!$post) {
                $results['skipped']++;
                $results['warnings'][] = sprintf(
                    __('Post %d no longer exists.', 'screaming-fixes'),
                    $post_id
                );
                continue;
            }

            // Restore the original meta value
            $updated = update_post_meta($post_id, $meta_key, $original_value);

            // update_post_meta returns false if value is same, meta_id if new, true if updated
            // We consider it successful even if value was already the same
            $results['restored']++;
        }

        return $results;
    }

    /**
     * Clear fixed_links from module results after batch undo
     *
     * @param array $batch The batch that was undone
     */
    private static function clear_fixed_links_for_batch($batch) {
        $module = $batch['module'] ?? '';
        $fixed_urls = $batch['fixed_urls'] ?? [];

        if (empty($module) || empty($fixed_urls)) {
            return;
        }

        // Get the module's stored results
        if ($module === 'broken-links') {
            $module_instance = SF_Module_Loader::get_module('broken-links');
            if (!$module_instance) {
                return;
            }

            // Get current results from transient
            $results = $module_instance->get_results();

            // Also check database if transient is empty
            if (empty($results)) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
                $session_id = 'user_' . get_current_user_id();

                $row = $wpdb->get_row($wpdb->prepare(
                    "SELECT data FROM {$table_name}
                     WHERE session_id = %s AND module = %s AND expires_at > NOW()
                     ORDER BY created_at DESC LIMIT 1",
                    $session_id,
                    'broken-links'
                ));

                if ($row) {
                    $results = json_decode($row->data, true);
                }
            }

            if (empty($results) || empty($results['fixed_links'])) {
                return;
            }

            // Remove the undone URLs from fixed_links
            $urls_to_remove = array_flip($fixed_urls);
            $results['fixed_links'] = array_filter($results['fixed_links'], function($link) use ($urls_to_remove) {
                $broken_url = $link['broken_url'] ?? '';
                return !isset($urls_to_remove[$broken_url]);
            });
            $results['fixed_links'] = array_values($results['fixed_links']); // Re-index
            $results['fixed_count'] = count($results['fixed_links']);

            // Save updated results to both transient and database
            set_transient('sf_broken-links_results', $results, HOUR_IN_SECONDS);

            // Update database
            global $wpdb;
            $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
            $session_id = 'user_' . get_current_user_id();

            $wpdb->update(
                $table_name,
                ['data' => wp_json_encode($results)],
                [
                    'session_id' => $session_id,
                    'module' => 'broken-links',
                ],
                ['%s'],
                ['%s', '%s']
            );
        } elseif ($module === 'redirect-chains') {
            $module_instance = SF_Module_Loader::get_module('redirect-chains');
            if (!$module_instance) {
                return;
            }

            // Get current results from transient
            $results = $module_instance->get_results();

            // Also check database if transient is empty
            if (empty($results)) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
                $session_id = 'user_' . get_current_user_id();

                $row = $wpdb->get_row($wpdb->prepare(
                    "SELECT data FROM {$table_name}
                     WHERE session_id = %s AND module = %s AND expires_at > NOW()
                     ORDER BY created_at DESC LIMIT 1",
                    $session_id,
                    'redirect-chains'
                ));

                if ($row) {
                    $results = json_decode($row->data, true);
                }
            }

            if (empty($results) || empty($results['fixed_redirects'])) {
                return;
            }

            // Remove the undone URLs from fixed_redirects
            // For redirect-chains, the URL is stored in 'address' field
            $urls_to_remove = array_flip($fixed_urls);
            $results['fixed_redirects'] = array_filter($results['fixed_redirects'], function($redirect) use ($urls_to_remove) {
                $address = $redirect['address'] ?? '';
                return !isset($urls_to_remove[$address]);
            });
            $results['fixed_redirects'] = array_values($results['fixed_redirects']); // Re-index
            $results['fixed_count'] = count($results['fixed_redirects']);

            // Save updated results to both transient and database
            set_transient('sf_redirect-chains_results', $results, HOUR_IN_SECONDS);

            // Update database
            global $wpdb;
            $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
            $session_id = 'user_' . get_current_user_id();

            $wpdb->update(
                $table_name,
                ['data' => wp_json_encode($results)],
                [
                    'session_id' => $session_id,
                    'module' => 'redirect-chains',
                ],
                ['%s'],
                ['%s', '%s']
            );
        } elseif ($module === 'image-alt-text') {
            $module_instance = SF_Module_Loader::get_module('image-alt-text');
            if (!$module_instance) {
                return;
            }

            // Get current results from transient
            $results = $module_instance->get_results();

            // Also check database if transient is empty
            if (empty($results)) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
                $session_id = 'user_' . get_current_user_id();

                $row = $wpdb->get_row($wpdb->prepare(
                    "SELECT data FROM {$table_name}
                     WHERE session_id = %s AND module = %s AND expires_at > NOW()
                     ORDER BY created_at DESC LIMIT 1",
                    $session_id,
                    'image-alt-text'
                ));

                if ($row) {
                    $results = json_decode($row->data, true);
                }
            }

            if (empty($results) || empty($results['fixed_images'])) {
                return;
            }

            // Remove the undone URLs from fixed_images
            $urls_to_remove = array_flip($fixed_urls);
            $results['fixed_images'] = array_filter($results['fixed_images'], function($image) use ($urls_to_remove) {
                $image_url = $image['image_url'] ?? '';
                return !isset($urls_to_remove[$image_url]);
            });
            $results['fixed_images'] = array_values($results['fixed_images']); // Re-index
            $results['fixed_images_count'] = count($results['fixed_images']);

            // Save updated results to both transient and database
            set_transient('sf_image-alt-text_results', $results, HOUR_IN_SECONDS);

            // Update database
            global $wpdb;
            $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
            $session_id = 'user_' . get_current_user_id();

            $wpdb->update(
                $table_name,
                ['data' => wp_json_encode($results)],
                [
                    'session_id' => $session_id,
                    'module' => 'image-alt-text',
                ],
                ['%s'],
                ['%s', '%s']
            );
        } elseif ($module === 'meta-description') {
            $module_instance = SF_Module_Loader::get_module('meta-description');
            if (!$module_instance) {
                return;
            }

            // Get current results from transient
            $results = $module_instance->get_results();

            // Also check database if transient is empty
            if (empty($results)) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
                $session_id = 'user_' . get_current_user_id();

                $row = $wpdb->get_row($wpdb->prepare(
                    "SELECT data FROM {$table_name}
                     WHERE session_id = %s AND module = %s AND expires_at > NOW()
                     ORDER BY created_at DESC LIMIT 1",
                    $session_id,
                    'meta-description'
                ));

                if ($row) {
                    $results = json_decode($row->data, true);
                }
            }

            if (empty($results) || empty($results['fixed_descriptions'])) {
                return;
            }

            // Remove the undone URLs from fixed_descriptions
            $urls_to_remove = array_flip($fixed_urls);
            $results['fixed_descriptions'] = array_filter($results['fixed_descriptions'], function($desc) use ($urls_to_remove) {
                $address = $desc['address'] ?? '';
                return !isset($urls_to_remove[$address]);
            });
            $results['fixed_descriptions'] = array_values($results['fixed_descriptions']); // Re-index
            $results['fixed_count'] = count($results['fixed_descriptions']);

            // Save updated results to both transient and database
            set_transient('sf_meta-description_results', $results, HOUR_IN_SECONDS);

            // Update database
            global $wpdb;
            $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
            $session_id = 'user_' . get_current_user_id();

            $wpdb->update(
                $table_name,
                ['data' => wp_json_encode($results)],
                [
                    'session_id' => $session_id,
                    'module' => 'meta-description',
                ],
                ['%s'],
                ['%s', '%s']
            );
        }
    }

    /**
     * Remove a batch from storage
     *
     * @param string $batch_id
     * @return bool
     */
    public static function remove_batch($batch_id) {
        $batches = self::get_all_batches();

        $batches = array_filter($batches, function($batch) use ($batch_id) {
            return $batch['batch_id'] !== $batch_id;
        });

        update_option(self::OPTION_NAME, array_values($batches), false);

        return true;
    }

    /**
     * Check if any posts in a batch have been modified since the batch was created
     *
     * @param string $batch_id
     * @return array Posts that have been modified
     */
    public static function check_batch_modifications($batch_id) {
        $batch = self::get_batch($batch_id);

        if (!$batch) {
            return [];
        }

        $modified = [];
        $post_revisions = $batch['post_revisions'] ?? [];

        foreach ($post_revisions as $post_id => $revision_data) {
            $post_id = absint($post_id);
            $current_revision_id = $revision_data['current_revision'] ?? 0;

            $latest_revisions = wp_get_post_revisions($post_id, [
                'numberposts' => 1,
                'orderby' => 'ID',
                'order' => 'DESC',
            ]);

            if (!empty($latest_revisions)) {
                $latest = array_shift($latest_revisions);
                if ($latest->ID !== $current_revision_id) {
                    $post = get_post($post_id);
                    $modified[] = [
                        'post_id' => $post_id,
                        'post_title' => $post ? $post->post_title : 'Unknown',
                    ];
                }
            }
        }

        return $modified;
    }

    /**
     * Get module display name
     *
     * @param string $module Module slug
     * @return string Display name
     */
    public static function get_module_display_name($module) {
        $names = [
            'broken-links' => __('Broken Links', 'screaming-fixes'),
            'image-alt-text' => __('Image Alt Text', 'screaming-fixes'),
            'redirect-chains' => __('Redirect Chains', 'screaming-fixes'),
            'meta-description' => __('Meta Description', 'screaming-fixes'),
        ];

        return $names[$module] ?? ucwords(str_replace('-', ' ', $module));
    }

    /**
     * Get action description for display
     *
     * @param array $batch
     * @return string
     */
    public static function get_batch_description($batch) {
        $module = $batch['module'] ?? 'unknown';
        $item_count = $batch['item_count'] ?? 0;
        $post_count = $batch['posts_changed'] ?? $batch['post_count'] ?? 0;

        $action_text = '';
        switch ($module) {
            case 'broken-links':
                $action_text = sprintf(
                    _n('%d URL fixed', '%d URLs fixed', $item_count, 'screaming-fixes'),
                    $item_count
                );
                break;
            case 'image-alt-text':
                $action_text = sprintf(
                    _n('%d alt text updated', '%d alt texts updated', $item_count, 'screaming-fixes'),
                    $item_count
                );
                break;
            case 'redirect-chains':
                $action_text = sprintf(
                    _n('%d redirect fixed', '%d redirects fixed', $item_count, 'screaming-fixes'),
                    $item_count
                );
                break;
            case 'meta-description':
                $action_text = sprintf(
                    _n('%d description updated', '%d descriptions updated', $item_count, 'screaming-fixes'),
                    $item_count
                );
                break;
            default:
                $action_text = sprintf(
                    _n('%d item changed', '%d items changed', $item_count, 'screaming-fixes'),
                    $item_count
                );
        }

        return sprintf(
            _n('%d post', '%d posts', $post_count, 'screaming-fixes') . ' | ' . $action_text,
            $post_count
        );
    }
}
