<?php
/**
 * Changelog Class for Screaming Fixes
 *
 * Tracks changes made by the plugin for audit trail and undo guidance.
 * Leverages WordPress Revisions for actual undo functionality.
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * SF_Changelog Class
 *
 * Handles logging and retrieval of changes made by Screaming Fixes modules.
 */
class SF_Changelog {

    /**
     * Maximum number of changelog entries to keep
     */
    const MAX_ENTRIES = 25;

    /**
     * Log a change
     *
     * @param array $data Change data with keys:
     *   - change_type: Type of change (links_fixed, alt_text_added, redirect_created, etc.)
     *   - module: Module that made the change (broken-links, alt-text, redirect-chains, backlinks)
     *   - description: Human-readable description
     *   - post_id: (optional) Post ID if change was to a post
     *   - post_title: (optional) Post title
     *   - post_url: (optional) Post permalink
     *   - items_affected: (optional) Number of items affected, default 1
     * @return int|false Insert ID on success, false on failure
     */
    public static function log($data) {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        // Validate required fields
        if (empty($data['change_type']) || empty($data['module']) || empty($data['description'])) {
            return false;
        }

        $result = $wpdb->insert(
            $table,
            array(
                'change_type'    => sanitize_text_field($data['change_type']),
                'module'         => sanitize_text_field($data['module']),
                'description'    => sanitize_text_field($data['description']),
                'post_id'        => isset($data['post_id']) ? absint($data['post_id']) : null,
                'post_title'     => isset($data['post_title']) ? sanitize_text_field($data['post_title']) : null,
                'post_url'       => isset($data['post_url']) ? esc_url_raw($data['post_url']) : null,
                'items_affected' => isset($data['items_affected']) ? absint($data['items_affected']) : 1,
                'user_id'        => get_current_user_id(),
                'created_at'     => current_time('mysql'),
            ),
            array('%s', '%s', '%s', '%d', '%s', '%s', '%d', '%d', '%s')
        );

        if ($result === false) {
            return false;
        }

        $change_id = $wpdb->insert_id;

        // Prune old entries to keep only MAX_ENTRIES
        self::prune_old_entries();

        return $change_id;
    }

    /**
     * Get recent changes
     *
     * @param int $limit Number of entries to retrieve (max 25)
     * @return array Array of change records
     */
    public static function get_recent_changes($limit = 25) {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        // Enforce maximum limit
        $limit = min(absint($limit), self::MAX_ENTRIES);

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} ORDER BY created_at DESC LIMIT %d",
                $limit
            ),
            ARRAY_A
        );

        return $results ? $results : array();
    }

    /**
     * Get a single change by ID
     *
     * @param int $id Change ID
     * @return array|null Change record or null if not found
     */
    public static function get_change($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE id = %d",
                absint($id)
            ),
            ARRAY_A
        );

        return $result;
    }

    /**
     * Get changes by module
     *
     * @param string $module Module slug
     * @param int $limit Number of entries to retrieve
     * @return array Array of change records
     */
    public static function get_changes_by_module($module, $limit = 25) {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        $limit = min(absint($limit), self::MAX_ENTRIES);

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE module = %s ORDER BY created_at DESC LIMIT %d",
                sanitize_text_field($module),
                $limit
            ),
            ARRAY_A
        );

        return $results ? $results : array();
    }

    /**
     * Get the count of changes
     *
     * @return int Number of changelog entries
     */
    public static function get_count() {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        $count = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");

        return absint($count);
    }

    /**
     * Prune entries beyond MAX_ENTRIES
     *
     * Keeps only the most recent MAX_ENTRIES entries.
     */
    private static function prune_old_entries() {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        // Get the ID of the entry at position MAX_ENTRIES
        $cutoff_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM {$table} ORDER BY created_at DESC LIMIT 1 OFFSET %d",
                self::MAX_ENTRIES - 1
            )
        );

        if ($cutoff_id) {
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$table} WHERE id < %d",
                    absint($cutoff_id)
                )
            );
        }
    }

    /**
     * Clear all changelog entries
     *
     * Use with caution - primarily for testing or reset functionality.
     *
     * @return int|false Number of rows deleted or false on error
     */
    public static function clear_all() {
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_changelog';

        return $wpdb->query("TRUNCATE TABLE {$table}");
    }

    /**
     * Get module icon for display
     *
     * @param string $module Module slug
     * @return string Icon HTML entity
     */
    public static function get_module_icon($module) {
        $icons = array(
            'broken-links'    => '&#128279;', // 🔗
            'alt-text'        => '&#128444;', // 🖼
            'image-alt-text'  => '&#128444;', // 🖼
            'redirect-chains' => '&#128260;', // 🔄
            'backlinks'       => '&#128281;', // 🔙
            'backlink-reclaim' => '&#128281;', // 🔙
        );

        return isset($icons[$module]) ? $icons[$module] : '&#128736;'; // 🔧 default
    }

    /**
     * Get human-readable time ago string
     *
     * @param string $datetime MySQL datetime string
     * @return string Human-readable time difference
     */
    public static function get_time_ago($datetime) {
        $timestamp = strtotime($datetime);
        $diff = current_time('timestamp') - $timestamp;

        if ($diff < 60) {
            return __('Just now', 'screaming-fixes');
        } elseif ($diff < 3600) {
            $mins = floor($diff / 60);
            return sprintf(_n('%d minute ago', '%d minutes ago', $mins, 'screaming-fixes'), $mins);
        } elseif ($diff < 86400) {
            $hours = floor($diff / 3600);
            return sprintf(_n('%d hour ago', '%d hours ago', $hours, 'screaming-fixes'), $hours);
        } elseif ($diff < 604800) {
            $days = floor($diff / 86400);
            return sprintf(_n('%d day ago', '%d days ago', $days, 'screaming-fixes'), $days);
        } else {
            return date_i18n(get_option('date_format'), $timestamp);
        }
    }

    /**
     * Check if a post still exists
     *
     * @param int $post_id Post ID
     * @return bool True if post exists
     */
    public static function post_exists($post_id) {
        if (empty($post_id)) {
            return false;
        }
        return get_post_status($post_id) !== false;
    }

    /**
     * Get edit URL for a post
     *
     * @param int $post_id Post ID
     * @return string|null Edit URL or null if post doesn't exist
     */
    public static function get_edit_url($post_id) {
        if (!self::post_exists($post_id)) {
            return null;
        }
        return get_edit_post_link($post_id, 'raw');
    }

    /**
     * Get revisions URL for a post
     *
     * @param int $post_id Post ID
     * @return string|null Revisions URL or null if post doesn't exist or has no revisions
     */
    public static function get_revisions_url($post_id) {
        if (!self::post_exists($post_id)) {
            return null;
        }

        // Check if post has revisions
        $revisions = wp_get_post_revisions($post_id, array('numberposts' => 1));
        if (empty($revisions)) {
            // No revisions yet - return edit URL instead
            return get_edit_post_link($post_id, 'raw');
        }

        // Get the most recent revision and link to comparison
        $revision = array_shift($revisions);
        return admin_url('revision.php?revision=' . $revision->ID);
    }
}
