<?php
/**
 * Database functions for Screaming Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Database {

    /**
     * Database version
     * Increment this when adding/modifying tables
     * @var string
     */
    private static $db_version = '1.2.0';

    /**
     * Create necessary database tables
     */
    public static function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        // Change Log Table - for undo functionality
        $log_table = $wpdb->prefix . 'screaming_fixes_log';
        $sql_log = "CREATE TABLE {$log_table} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            post_id BIGINT UNSIGNED NOT NULL,
            fix_type VARCHAR(50) NOT NULL,
            module VARCHAR(50) NOT NULL,
            original_value LONGTEXT,
            new_value LONGTEXT,
            context TEXT,
            user_id BIGINT UNSIGNED,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            reverted_at DATETIME NULL,
            INDEX idx_post_id (post_id),
            INDEX idx_fix_type (fix_type),
            INDEX idx_created_at (created_at)
        ) {$charset_collate};";
        dbDelta($sql_log);

        // Backlink Scan Cache Table (for Backlink Reclaim)
        $backlink_cache_table = $wpdb->prefix . 'screaming_fixes_backlink_cache';
        $sql_backlink_cache = "CREATE TABLE {$backlink_cache_table} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            domain VARCHAR(255) NOT NULL,
            scan_data LONGTEXT NOT NULL,
            scanned_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            expires_at DATETIME NOT NULL,
            UNIQUE KEY idx_domain (domain),
            INDEX idx_expires (expires_at)
        ) {$charset_collate};";
        dbDelta($sql_backlink_cache);

        // Backlink Evaluator Cache Table (for Backlink Evaluator)
        $backlink_evaluator_table = $wpdb->prefix . 'screaming_fixes_backlink_evaluator_cache';
        $sql_backlink_evaluator = "CREATE TABLE {$backlink_evaluator_table} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            domain VARCHAR(255) NOT NULL,
            summary_data LONGTEXT,
            backlinks_data LONGTEXT,
            scanned_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            expires_at DATETIME NOT NULL,
            UNIQUE KEY idx_domain (domain),
            INDEX idx_expires (expires_at)
        ) {$charset_collate};";
        dbDelta($sql_backlink_evaluator);

        // Uploaded Data Table (Session Storage)
        $uploads_table = $wpdb->prefix . 'screaming_fixes_uploads';
        $sql_uploads = "CREATE TABLE {$uploads_table} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            session_id VARCHAR(64) NOT NULL,
            module VARCHAR(50) NOT NULL,
            data LONGTEXT NOT NULL,
            fixes LONGTEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            expires_at DATETIME NOT NULL,
            INDEX idx_session (session_id),
            INDEX idx_module (module),
            INDEX idx_expires (expires_at)
        ) {$charset_collate};";
        dbDelta($sql_uploads);

        // Changelog Table - for tracking changes and undo guidance
        $changelog_table = $wpdb->prefix . 'screaming_fixes_changelog';
        $sql_changelog = "CREATE TABLE {$changelog_table} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            change_type VARCHAR(50) NOT NULL,
            module VARCHAR(50) NOT NULL,
            description TEXT NOT NULL,
            post_id BIGINT UNSIGNED NULL,
            post_title VARCHAR(255) NULL,
            post_url TEXT NULL,
            items_affected INT DEFAULT 1,
            user_id BIGINT UNSIGNED NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_module (module),
            INDEX idx_created (created_at DESC)
        ) {$charset_collate};";
        dbDelta($sql_changelog);

        // Update database version
        update_option('sf_db_version', self::$db_version);
    }

    /**
     * Drop all plugin database tables
     */
    public static function drop_tables() {
        global $wpdb;

        $tables = [
            $wpdb->prefix . 'screaming_fixes_log',
            $wpdb->prefix . 'screaming_fixes_backlink_cache',
            $wpdb->prefix . 'screaming_fixes_backlink_evaluator_cache',
            $wpdb->prefix . 'screaming_fixes_uploads',
            $wpdb->prefix . 'screaming_fixes_changelog',
        ];

        foreach ($tables as $table) {
            $wpdb->query("DROP TABLE IF EXISTS {$table}");
        }

        delete_option('sf_db_version');
    }

    /**
     * Check if database needs upgrade
     *
     * @return bool
     */
    public static function needs_upgrade() {
        $current_version = get_option('sf_db_version', '0');
        return version_compare($current_version, self::$db_version, '<');
    }

    /**
     * Upgrade database if needed
     */
    public static function maybe_upgrade() {
        if (self::needs_upgrade()) {
            self::create_tables();
        }
    }

    /**
     * Clean up expired data
     */
    public static function cleanup_expired() {
        global $wpdb;

        // Clean up expired uploads
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}screaming_fixes_uploads WHERE expires_at < NOW()"
        );

        // Clean up expired backlink cache
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}screaming_fixes_backlink_cache WHERE expires_at < NOW()"
        );

        // Clean up expired backlink evaluator cache
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}screaming_fixes_backlink_evaluator_cache WHERE expires_at < NOW()"
        );
    }

    /**
     * Get table sizes for debugging
     *
     * @return array
     */
    public static function get_table_stats() {
        global $wpdb;

        $stats = [];
        $tables = [
            'screaming_fixes_log',
            'screaming_fixes_backlink_cache',
            'screaming_fixes_backlink_evaluator_cache',
            'screaming_fixes_uploads',
            'screaming_fixes_changelog',
        ];

        foreach ($tables as $table) {
            $full_table = $wpdb->prefix . $table;
            $count = $wpdb->get_var("SELECT COUNT(*) FROM {$full_table}");
            $stats[$table] = [
                'rows' => (int) $count,
            ];
        }

        return $stats;
    }
}
