<?php
/**
 * Core plugin orchestration for Screaming Fixes
 *
 * Handles plugin initialization, loading dependencies, and coordinating modules
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Plugin {

    /**
     * Singleton instance
     * @var SF_Plugin
     */
    private static $instance = null;

    /**
     * Plugin version
     * @var string
     */
    private $version;

    /**
     * Array of loaded modules
     * @var array
     */
    private $modules = [];

    /**
     * Get singleton instance
     *
     * @return SF_Plugin
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->version = SF_VERSION;
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required dependencies
     */
    private function load_dependencies() {
        // Core includes
        require_once SF_PLUGIN_DIR . 'includes/class-database.php';
        require_once SF_PLUGIN_DIR . 'includes/class-csv-detector.php';
        require_once SF_PLUGIN_DIR . 'includes/class-changelog.php';
        require_once SF_PLUGIN_DIR . 'includes/class-revisions-check.php';
        require_once SF_PLUGIN_DIR . 'includes/class-batch-restore.php';
        require_once SF_PLUGIN_DIR . 'includes/abstracts/class-module.php';

        // Shared utilities
        require_once SF_PLUGIN_DIR . 'shared/class-plugin-detector.php';
        require_once SF_PLUGIN_DIR . 'shared/class-csv-parser.php';
        require_once SF_PLUGIN_DIR . 'shared/class-link-scanner.php';
        require_once SF_PLUGIN_DIR . 'shared/class-link-fixer.php';
        require_once SF_PLUGIN_DIR . 'shared/class-change-logger.php';
        require_once SF_PLUGIN_DIR . 'shared/class-activity-log.php';
        require_once SF_PLUGIN_DIR . 'shared/class-redirect-manager.php';
        require_once SF_PLUGIN_DIR . 'shared/class-backlink-api.php';

        // Admin and AJAX - load modules immediately so AJAX handlers are registered
        // Note: is_admin() returns true for AJAX requests too
        if (is_admin()) {
            require_once SF_PLUGIN_DIR . 'includes/class-admin.php';
            require_once SF_PLUGIN_DIR . 'includes/class-module-loader.php';

            // Instantiate admin class now - admin_menu fires before admin_init
            new SF_Admin();

            // Load modules immediately so AJAX handlers are registered
            SF_Module_Loader::init();
            $this->modules = SF_Module_Loader::get_modules();

            // Initialize change logger AJAX handlers and cron
            SF_Change_Logger::init();
            SF_Change_Logger::schedule_cron();
        }
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action('init', [$this, 'init']);
        add_action('admin_init', [$this, 'admin_init']);

        // AJAX handlers
        add_action('wp_ajax_sf_upload_csv', [$this, 'ajax_upload_csv']);
        add_action('wp_ajax_sf_apply_fixes', [$this, 'ajax_apply_fixes']);
        add_action('wp_ajax_sf_undo_change', [$this, 'ajax_undo_change']);
        add_action('wp_ajax_sf_get_ai_suggestion', [$this, 'ajax_get_ai_suggestion']);
        add_action('wp_ajax_sf_subscribe_email', [$this, 'ajax_subscribe_email']);
        add_action('wp_ajax_sf_mark_subscribed', [$this, 'ajax_mark_subscribed']);
        add_action('wp_ajax_sf_test_api_key', [$this, 'ajax_test_api_key']);
        add_action('wp_ajax_sf_save_settings', [$this, 'ajax_save_settings']);
        add_action('wp_ajax_sf_test_dataforseo', [$this, 'ajax_test_dataforseo']);
        add_action('wp_ajax_sf_get_restore_batches', [$this, 'ajax_get_restore_batches']);
        add_action('wp_ajax_sf_undo_batch', [$this, 'ajax_undo_batch']);
        add_action('wp_ajax_sf_check_batch_modifications', [$this, 'ajax_check_batch_modifications']);
        add_action('wp_ajax_sf_get_activity_log', [$this, 'ajax_get_activity_log']);
        add_action('wp_ajax_sf_dismiss_welcome', [$this, 'ajax_dismiss_welcome']);
    }

    /**
     * Initialize plugin on 'init' hook
     */
    public function init() {
        // Load text domain for translations
        load_plugin_textdomain('screaming-fixes', false, dirname(plugin_basename(SF_PLUGIN_DIR)) . '/languages');
    }

    /**
     * Initialize admin functionality
     */
    public function admin_init() {
        // Modules already loaded in load_dependencies for AJAX support
    }

    /**
     * Get all loaded modules
     *
     * @return array
     */
    public function get_modules() {
        return $this->modules;
    }

    /**
     * Get a specific module by ID
     *
     * @param string $module_id Module identifier
     * @return SF_Module|null
     */
    public function get_module($module_id) {
        return $this->modules[$module_id] ?? null;
    }

    /**
     * Get plugin version
     *
     * @return string
     */
    public function get_version() {
        return $this->version;
    }

    /**
     * AJAX handler: Upload and process CSV
     */
    public function ajax_upload_csv() {
        // Increase memory limit for large CSV processing
        // This is a fallback - the parser also limits rows to prevent exhaustion
        @ini_set('memory_limit', '512M');

        // Enable error logging for debugging
        $debug = get_option('sf_debug_mode', false);
        if ($debug) {
            error_log('SF Upload: Starting upload handler');
            error_log('SF Upload: Memory limit set to ' . ini_get('memory_limit'));
        }

        // Check nonce
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'screaming_fixes_nonce')) {
            if ($debug) {
                error_log('SF Upload: Invalid nonce');
            }
            wp_send_json_error(['message' => __('Security check failed. Please refresh and try again.', 'screaming-fixes')]);
        }

        if (!current_user_can('manage_options')) {
            if ($debug) {
                error_log('SF Upload: Permission denied');
            }
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        if (empty($_FILES['csv_file'])) {
            if ($debug) {
                error_log('SF Upload: No file in request. $_FILES: ' . print_r($_FILES, true));
            }
            wp_send_json_error(['message' => __('No file uploaded.', 'screaming-fixes')]);
        }

        $file = $_FILES['csv_file'];

        if ($debug) {
            error_log('SF Upload: File received - ' . $file['name']);
            error_log('SF Upload: File size - ' . $file['size']);
            error_log('SF Upload: File error code - ' . $file['error']);
        }

        // Check for upload errors
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $upload_errors = [
                UPLOAD_ERR_INI_SIZE => __('File exceeds server upload_max_filesize limit.', 'screaming-fixes'),
                UPLOAD_ERR_FORM_SIZE => __('File exceeds form size limit.', 'screaming-fixes'),
                UPLOAD_ERR_PARTIAL => __('File was only partially uploaded.', 'screaming-fixes'),
                UPLOAD_ERR_NO_FILE => __('No file was uploaded.', 'screaming-fixes'),
                UPLOAD_ERR_NO_TMP_DIR => __('Missing temporary folder.', 'screaming-fixes'),
                UPLOAD_ERR_CANT_WRITE => __('Failed to write file to disk.', 'screaming-fixes'),
                UPLOAD_ERR_EXTENSION => __('Upload stopped by PHP extension.', 'screaming-fixes'),
            ];
            $error_msg = $upload_errors[$file['error']] ?? __('Unknown upload error.', 'screaming-fixes');
            if ($debug) {
                error_log('SF Upload: Upload error - ' . $error_msg);
            }
            wp_send_json_error(['message' => $error_msg]);
        }

        // Validate file type
        $file_type = wp_check_filetype($file['name']);
        if ($file_type['ext'] !== 'csv') {
            wp_send_json_error(['message' => __('Please upload a CSV file.', 'screaming-fixes')]);
        }

        // Move to temp location
        $upload_dir = wp_upload_dir();
        $temp_dir = $upload_dir['basedir'] . '/screaming-fixes-temp/';

        if (!file_exists($temp_dir)) {
            wp_mkdir_p($temp_dir);
        }

        $temp_file = $temp_dir . wp_unique_filename($temp_dir, $file['name']);

        if (!move_uploaded_file($file['tmp_name'], $temp_file)) {
            if ($debug) {
                error_log('SF Upload: Failed to move uploaded file');
            }
            wp_send_json_error(['message' => __('Failed to save uploaded file.', 'screaming-fixes')]);
        }

        if ($debug) {
            error_log('SF Upload: File saved to ' . $temp_file);
        }

        // Detect CSV type
        $detector = new SF_CSV_Detector();
        $csv_type = $detector->detect($temp_file);

        if ($debug) {
            error_log('SF Upload: Detected CSV type - ' . $csv_type);
        }

        if ($csv_type === 'unknown') {
            // Log headers for debugging
            if ($debug) {
                $headers = $detector->get_headers($temp_file);
                error_log('SF Upload: CSV headers - ' . print_r($headers, true));
            }
            unlink($temp_file);
            wp_send_json_error([
                'message' => __('Could not detect CSV type. Please ensure this is a Screaming Frog export.', 'screaming-fixes')
            ]);
        }

        // Parse CSV
        if ($debug) {
            error_log('SF Upload: Starting CSV parse');
        }

        $parser = new SF_CSV_Parser();
        $data = $parser->parse($temp_file);

        if (is_wp_error($data)) {
            if ($debug) {
                error_log('SF Upload: Parse error - ' . $data->get_error_message());
            }
            unlink($temp_file);
            wp_send_json_error(['message' => $data->get_error_message()]);
        }

        if ($debug) {
            error_log('SF Upload: Parsed ' . count($data['rows'] ?? []) . ' rows');
        }

        // Store pending upload for module to process later
        $upload_id = wp_generate_uuid4();
        $pending_uploads = get_option('sf_pending_uploads', []);
        $pending_uploads[$upload_id] = [
            'path' => $temp_file,
            'module' => $csv_type,
            'created_at' => current_time('mysql'),
        ];
        update_option('sf_pending_uploads', $pending_uploads);

        // Store in session table
        // Use user-based session ID so modules can retrieve data consistently
        global $wpdb;
        $session_id = 'user_' . get_current_user_id();

        // Delete any existing data for this session/module to avoid duplicates
        $wpdb->delete(
            $wpdb->prefix . 'screaming_fixes_uploads',
            [
                'session_id' => $session_id,
                'module' => $csv_type,
            ],
            ['%s', '%s']
        );

        $wpdb->insert(
            $wpdb->prefix . 'screaming_fixes_uploads',
            [
                'session_id' => $session_id,
                'module' => $csv_type,
                'data' => wp_json_encode($data),
                'created_at' => current_time('mysql'),
                'expires_at' => gmdate('Y-m-d H:i:s', strtotime('+24 hours')),
            ],
            ['%s', '%s', '%s', '%s', '%s']
        );

        // Don't include full data in response - it's stored in the database
        // Including 27k+ rows would cause memory/JSON encoding issues
        $response = [
            'session_id' => $session_id,
            'upload_id' => $upload_id,
            'csv_type' => $csv_type,
            'row_count' => count($data['rows'] ?? $data),
            'truncated' => $data['truncated'] ?? false,
        ];

        // Include truncation warning message if applicable
        if (!empty($data['truncated']) && !empty($data['truncated_message'])) {
            $response['truncated_message'] = $data['truncated_message'];
            $response['total_in_file'] = $data['total_in_file'] ?? 0;
        }

        wp_send_json_success($response);
    }

    /**
     * AJAX handler: Apply fixes
     */
    public function ajax_apply_fixes() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $fixes = isset($_POST['fixes']) ? json_decode(stripslashes($_POST['fixes']), true) : [];
        $module_id = isset($_POST['module']) ? sanitize_text_field($_POST['module']) : '';

        if (empty($fixes) || empty($module_id)) {
            wp_send_json_error(['message' => __('Invalid request data.', 'screaming-fixes')]);
        }

        $module = $this->get_module($module_id);

        if (!$module) {
            wp_send_json_error(['message' => __('Module not found.', 'screaming-fixes')]);
        }

        // Apply fixes via module
        $results = $module->apply_fixes($fixes);

        wp_send_json_success($results);
    }

    /**
     * AJAX handler: Undo a change
     */
    public function ajax_undo_change() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $log_id = isset($_POST['log_id']) ? absint($_POST['log_id']) : 0;

        if (!$log_id) {
            wp_send_json_error(['message' => __('Invalid log ID.', 'screaming-fixes')]);
        }

        $logger = new SF_Change_Logger();
        $result = $logger->undo_change($log_id);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success(['message' => __('Change reverted successfully.', 'screaming-fixes')]);
    }

    /**
     * AJAX handler: Get AI suggestion
     */
    public function ajax_get_ai_suggestion() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';
        $context = isset($_POST['context']) ? json_decode(stripslashes($_POST['context']), true) : [];

        if (empty($type) || empty($context)) {
            wp_send_json_error(['message' => __('Invalid request data.', 'screaming-fixes')]);
        }

        $suggestion = $this->get_ai_suggestion($type, $context);

        if (is_wp_error($suggestion)) {
            wp_send_json_error(['message' => $suggestion->get_error_message()]);
        }

        wp_send_json_success(['suggestion' => $suggestion]);
    }

    /**
     * AJAX handler: Subscribe email
     */
    public function ajax_subscribe_email() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';

        if (!is_email($email)) {
            wp_send_json_error(['message' => __('Invalid email address.', 'screaming-fixes')]);
        }

        // Store locally
        update_option('sf_email_subscribed', $email);

        // In future, send to email service
        // $this->send_to_email_service($email);

        wp_send_json_success(['message' => __('Subscribed successfully!', 'screaming-fixes')]);
    }

    /**
     * AJAX handler: Mark user as subscribed (called after Supabase submission)
     */
    public function ajax_mark_subscribed() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        // Mark as subscribed to hide the form
        update_option('sf_email_subscribed', true);

        wp_send_json_success();
    }

    /**
     * AJAX handler: Test API key
     */
    public function ajax_test_api_key() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';

        if (empty($api_key)) {
            wp_send_json_error(['message' => __('Please enter an API key.', 'screaming-fixes')]);
        }

        // Test the API key with a simple request
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode([
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 10,
                'messages' => [
                    ['role' => 'user', 'content' => 'Say "OK"'],
                ],
            ]),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => $response->get_error_message()]);
        }

        $code = wp_remote_retrieve_response_code($response);

        if ($code === 200) {
            wp_send_json_success(['message' => __('API key is valid!', 'screaming-fixes')]);
        } elseif ($code === 401) {
            wp_send_json_error(['message' => __('Invalid API key.', 'screaming-fixes')]);
        } else {
            wp_send_json_error(['message' => __('API test failed. Please try again.', 'screaming-fixes')]);
        }
    }

    /**
     * AJAX handler: Save settings
     */
    public function ajax_save_settings() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $settings = isset($_POST['settings']) ? json_decode(stripslashes($_POST['settings']), true) : [];

        if (isset($settings['claude_api_key'])) {
            update_option('sf_claude_api_key', sanitize_text_field($settings['claude_api_key']));
        }

        if (isset($settings['preferred_redirect_plugin'])) {
            update_option('sf_preferred_redirect_plugin', sanitize_text_field($settings['preferred_redirect_plugin']));
        }

        if (isset($settings['enable_logging'])) {
            update_option('sf_enable_logging', (bool) $settings['enable_logging']);
        }

        if (isset($settings['debug_mode'])) {
            update_option('sf_debug_mode', (bool) $settings['debug_mode']);
        }

        // DataForSEO credentials
        if (isset($settings['dataforseo_login'])) {
            update_option('sf_dataforseo_login', sanitize_text_field($settings['dataforseo_login']));
        }

        if (isset($settings['dataforseo_password'])) {
            update_option('sf_dataforseo_password', sanitize_text_field($settings['dataforseo_password']));
        }

        wp_send_json_success(['message' => __('Settings saved.', 'screaming-fixes')]);
    }

    /**
     * AJAX handler: Test DataForSEO connection
     */
    public function ajax_test_dataforseo() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $login = isset($_POST['login']) ? sanitize_text_field($_POST['login']) : '';
        $password = isset($_POST['password']) ? sanitize_text_field($_POST['password']) : '';

        if (empty($login) || empty($password)) {
            wp_send_json_error(['message' => __('Login and password are required.', 'screaming-fixes')]);
        }

        // Temporarily save credentials for testing
        update_option('sf_dataforseo_login', $login);
        update_option('sf_dataforseo_password', $password);

        // Test connection
        $api = new SF_Backlink_API();
        $result = $api->test_connection();

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get all restore batches
     */
    public function ajax_get_restore_batches() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $batches = SF_Batch_Restore::get_all_batches();

        // Format batches for display
        $formatted = [];
        foreach ($batches as $batch) {
            $formatted[] = [
                'batch_id' => $batch['batch_id'],
                'timestamp' => $batch['timestamp'],
                'timestamp_formatted' => date_i18n(
                    get_option('date_format') . ' ' . get_option('time_format'),
                    strtotime($batch['timestamp'])
                ),
                'module' => $batch['module'],
                'module_name' => SF_Batch_Restore::get_module_display_name($batch['module']),
                'description' => SF_Batch_Restore::get_batch_description($batch),
                'post_count' => $batch['posts_changed'] ?? $batch['post_count'] ?? 0,
                'item_count' => $batch['item_count'] ?? 0,
            ];
        }

        wp_send_json_success(['batches' => $formatted]);
    }

    /**
     * AJAX: Undo a batch of changes
     */
    public function ajax_undo_batch() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $batch_id = isset($_POST['batch_id']) ? sanitize_text_field($_POST['batch_id']) : '';

        if (empty($batch_id)) {
            wp_send_json_error(['message' => __('Batch ID is required.', 'screaming-fixes')]);
        }

        $result = SF_Batch_Restore::undo_batch($batch_id);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX: Check if posts in a batch have been modified since the batch
     */
    public function ajax_check_batch_modifications() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $batch_id = isset($_POST['batch_id']) ? sanitize_text_field($_POST['batch_id']) : '';

        if (empty($batch_id)) {
            wp_send_json_error(['message' => __('Batch ID is required.', 'screaming-fixes')]);
        }

        $modified = SF_Batch_Restore::check_batch_modifications($batch_id);

        wp_send_json_success([
            'has_modifications' => !empty($modified),
            'modified_posts' => $modified,
            'count' => count($modified),
        ]);
    }

    /**
     * Get AI suggestion for a fix
     *
     * @param string $type Type of fix ('broken_link', 'alt_text', 'redirect')
     * @param array $context Context data
     * @return string|WP_Error Suggestion or error
     */
    private function get_ai_suggestion($type, $context) {
        $api_key = get_option('sf_claude_api_key');

        if (!$api_key) {
            return new WP_Error('no_api_key', __('Claude API key not configured. Add it in Settings.', 'screaming-fixes'));
        }

        $prompt = $this->build_ai_prompt($type, $context);

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode([
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 500,
                'messages' => [
                    ['role' => 'user', 'content' => $prompt],
                ],
            ]),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['content'][0]['text'])) {
            return trim($body['content'][0]['text']);
        }

        return new WP_Error('api_error', __('Unexpected API response.', 'screaming-fixes'));
    }

    /**
     * Build AI prompt based on fix type
     *
     * @param string $type Fix type
     * @param array $context Context data
     * @return string Prompt
     */
    private function build_ai_prompt($type, $context) {
        $domain = wp_parse_url(home_url(), PHP_URL_HOST);

        switch ($type) {
            case 'broken_link':
                return sprintf(
                    "A website has a broken link: %s\n" .
                    "The link appears on the page: %s\n" .
                    "Suggest the best replacement URL on the domain %s.\n" .
                    "If no good replacement exists, respond with 'REMOVE'.\n" .
                    "Respond with just the URL path (e.g., /new-page/) or 'REMOVE'.",
                    $context['broken_url'] ?? '',
                    $context['page_title'] ?? 'Unknown',
                    $domain
                );

            case 'alt_text':
                return sprintf(
                    "Generate SEO-friendly alt text for an image.\n" .
                    "Image filename: %s\n" .
                    "Page context: %s\n" .
                    "Respond with just the alt text, 5-15 words, descriptive and natural.",
                    $context['filename'] ?? '',
                    $context['page_title'] ?? ''
                );

            case 'redirect':
                return sprintf(
                    "A webpage is returning 404: %s\n" .
                    "Domain: %s\n" .
                    "External sites are linking to this dead page.\n" .
                    "Suggest the best page on the site to redirect to.\n" .
                    "Respond with just the URL path (e.g., /best-matching-page/).",
                    $context['dead_url'] ?? '',
                    $domain
                );
        }

        return '';
    }

    /**
     * AJAX: Get activity log entries
     */
    public function ajax_get_activity_log() {
        check_ajax_referer('sf_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $limit = isset($_POST['limit']) ? absint($_POST['limit']) : 25;

        $entries = SF_Activity_Log::get_formatted_entries($limit);
        $total = SF_Activity_Log::get_total_count();
        $stats = SF_Activity_Log::get_stats();

        wp_send_json_success([
            'entries' => $entries,
            'total' => $total,
            'stats' => $stats,
        ]);
    }

    /**
     * AJAX: Dismiss welcome message permanently
     */
    public function ajax_dismiss_welcome() {
        check_ajax_referer('sf_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $user_id = get_current_user_id();

        // update_user_meta returns false if value is unchanged, so we check after update
        update_user_meta($user_id, 'sf_welcome_dismissed', '1');

        // Verify it was saved
        $saved = get_user_meta($user_id, 'sf_welcome_dismissed', true);

        if ($saved) {
            wp_send_json_success(['message' => __('Welcome message dismissed.', 'screaming-fixes')]);
        } else {
            wp_send_json_error(['message' => __('Failed to save preference.', 'screaming-fixes')]);
        }
    }

    /**
     * Plugin activation handler
     */
    public static function activate() {
        // Create database tables
        require_once SF_PLUGIN_DIR . 'includes/class-database.php';
        SF_Database::create_tables();

        // Set default options
        add_option('sf_enable_logging', true);
        add_option('sf_preferred_redirect_plugin', 'auto');
        add_option('sf_debug_mode', false);

        // Clear any cached data
        wp_cache_flush();
    }

    /**
     * Plugin deactivation handler
     */
    public static function deactivate() {
        // Unschedule change log trim cron
        SF_Change_Logger::unschedule_cron();

        // Clean up temp files
        $upload_dir = wp_upload_dir();
        $temp_dir = $upload_dir['basedir'] . '/screaming-fixes-temp/';

        if (is_dir($temp_dir)) {
            $files = glob($temp_dir . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
            rmdir($temp_dir);
        }

        // Clean up expired session data
        global $wpdb;
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}screaming_fixes_uploads WHERE expires_at < NOW()"
        );
    }
}
