<?php
/**
 * Revisions Check Class for Screaming Fixes
 *
 * Detects WordPress Revisions status and provides guidance for undo functionality.
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * SF_Revisions_Check Class
 *
 * Checks WordPress Revisions configuration and detects potential issues.
 */
class SF_Revisions_Check {

    /**
     * Get full revisions status
     *
     * @return array Status information with keys:
     *   - enabled: bool - Whether revisions are enabled
     *   - limit: string|int - 'unlimited' or number of revisions allowed
     *   - status_code: string - 'ok', 'limited', or 'disabled'
     *   - message: string - Human-readable status message
     *   - details: array - Additional details about configuration
     *   - plugins: array - Detected plugins that may affect revisions
     *   - hosting: array|null - Detected hosting provider info
     */
    public static function get_status() {
        $status = array(
            'enabled'     => true,
            'limit'       => 'unlimited',
            'status_code' => 'ok',
            'message'     => __('WordPress Revisions are enabled. You can undo changes via post revision history.', 'screaming-fixes'),
            'details'     => array(),
            'plugins'     => array(),
            'hosting'     => null,
        );

        // Check WP_POST_REVISIONS constant
        if (defined('WP_POST_REVISIONS')) {
            if (WP_POST_REVISIONS === false || WP_POST_REVISIONS === 0) {
                $status['enabled'] = false;
                $status['limit'] = 0;
                $status['status_code'] = 'disabled';
                $status['message'] = __('WordPress Revisions are disabled on this site. You won\'t be able to undo changes automatically.', 'screaming-fixes');
                $status['details'][] = __('WP_POST_REVISIONS is set to false in wp-config.php', 'screaming-fixes');
            } elseif (is_numeric(WP_POST_REVISIONS)) {
                $limit = intval(WP_POST_REVISIONS);
                $status['limit'] = $limit;
                $status['status_code'] = 'limited';
                $status['message'] = sprintf(
                    __('WordPress Revisions are limited to %d per post. Older revisions may be automatically deleted.', 'screaming-fixes'),
                    $limit
                );
                $status['details'][] = sprintf(
                    __('WP_POST_REVISIONS is set to %d in wp-config.php', 'screaming-fixes'),
                    $limit
                );
            }
        }

        // Detect plugins that may affect revisions
        $status['plugins'] = self::detect_revision_plugins();
        if (!empty($status['plugins'])) {
            $status['details'][] = sprintf(
                __('Detected plugins that may affect revisions: %s', 'screaming-fixes'),
                implode(', ', $status['plugins'])
            );
        }

        // Detect hosting provider
        $status['hosting'] = self::detect_hosting();
        if ($status['hosting']) {
            $status['details'][] = $status['hosting']['note'];
        }

        return $status;
    }

    /**
     * Check if revisions are enabled
     *
     * @return bool True if revisions are enabled
     */
    public static function is_enabled() {
        if (defined('WP_POST_REVISIONS')) {
            return WP_POST_REVISIONS !== false && WP_POST_REVISIONS !== 0;
        }
        return true; // Default is enabled
    }

    /**
     * Get the revision limit
     *
     * @return int|string Number of revisions allowed or 'unlimited'
     */
    public static function get_limit() {
        if (defined('WP_POST_REVISIONS')) {
            if (WP_POST_REVISIONS === false || WP_POST_REVISIONS === 0) {
                return 0;
            }
            if (is_numeric(WP_POST_REVISIONS)) {
                return intval(WP_POST_REVISIONS);
            }
        }
        return 'unlimited';
    }

    /**
     * Detect plugins known to affect revisions
     *
     * @return array Array of detected plugin names
     */
    private static function detect_revision_plugins() {
        $found = array();

        $plugins_to_check = array(
            'wp-optimize/wp-optimize.php'                       => 'WP-Optimize',
            'wp-rocket/wp-rocket.php'                           => 'WP Rocket',
            'perfmatters/perfmatters.php'                       => 'Perfmatters',
            'jeherve-disable-revisions/jeherve-disable-revisions.php' => 'Disable Revisions',
            'revision-control/revision-control.php'             => 'Revision Control',
            'wp-revisions-control/wp-revisions-control.php'     => 'WP Revisions Control',
            'better-delete-revision/better-delete-revision.php' => 'Better Delete Revision',
            'developer-developer/developer.php'                 => 'Developer',
            'simple-revisions-delete/simple-revisions-delete.php' => 'Simple Revisions Delete',
        );

        // Include plugin.php for is_plugin_active() if not already loaded
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        foreach ($plugins_to_check as $plugin_path => $plugin_name) {
            if (is_plugin_active($plugin_path)) {
                $found[] = $plugin_name;
            }
        }

        return $found;
    }

    /**
     * Detect hosting provider
     *
     * Some hosts have specific revision configurations.
     *
     * @return array|null Hosting info with 'name' and 'note', or null if not detected
     */
    public static function detect_hosting() {
        // WP Engine
        if (defined('WPE_APIKEY') || getenv('IS_WPE') || (defined('PWP_NAME') && PWP_NAME)) {
            return array(
                'name' => 'WP Engine',
                'note' => __('WP Engine detected. They typically limit revisions to 5 by default.', 'screaming-fixes'),
            );
        }

        // Kinsta
        if (defined('KINSTA_CACHE_ZONE') || getenv('KINSTA_CACHE_ZONE')) {
            return array(
                'name' => 'Kinsta',
                'note' => __('Kinsta detected. They allow unlimited revisions by default.', 'screaming-fixes'),
            );
        }

        // Flywheel
        if (defined('FLYWHEEL_CONFIG_DIR') || getenv('FLYWHEEL_CONFIG_DIR')) {
            return array(
                'name' => 'Flywheel',
                'note' => __('Flywheel detected. They support revisions by default.', 'screaming-fixes'),
            );
        }

        // SiteGround
        if (defined('STARTER_ASSETS_PATH') || class_exists('SiteGround_Optimizer\Supercacher\Supercacher')) {
            return array(
                'name' => 'SiteGround',
                'note' => __('SiteGround detected. Revisions should be available by default.', 'screaming-fixes'),
            );
        }

        // Pantheon
        if (defined('PANTHEON_ENVIRONMENT')) {
            return array(
                'name' => 'Pantheon',
                'note' => __('Pantheon detected. They support WordPress revisions.', 'screaming-fixes'),
            );
        }

        // Cloudways
        if (defined('STARTER_ASSETS_PATH') || strpos($_SERVER['SERVER_SOFTWARE'] ?? '', 'cloudways') !== false) {
            return array(
                'name' => 'Cloudways',
                'note' => __('Cloudways detected. Revisions depend on your server configuration.', 'screaming-fixes'),
            );
        }

        return null;
    }

    /**
     * Check if a specific post type supports revisions
     *
     * @param string $post_type Post type slug
     * @return bool True if post type supports revisions
     */
    public static function post_type_supports_revisions($post_type = 'post') {
        return post_type_supports($post_type, 'revisions');
    }

    /**
     * Get revision count for a post
     *
     * @param int $post_id Post ID
     * @return int Number of revisions
     */
    public static function get_revision_count($post_id) {
        $revisions = wp_get_post_revisions($post_id);
        return count($revisions);
    }

    /**
     * Check if a post has revisions available for undo
     *
     * @param int $post_id Post ID
     * @return bool True if post has at least one revision
     */
    public static function has_revisions($post_id) {
        return self::get_revision_count($post_id) > 0;
    }

    /**
     * Get the status badge HTML for display
     *
     * @return string HTML for status badge
     */
    public static function get_status_badge_html() {
        $status = self::get_status();

        switch ($status['status_code']) {
            case 'ok':
                return '<span class="sf-status-badge sf-status-ok">&#10003; ' . esc_html__('Enabled', 'screaming-fixes') . '</span>';
            case 'limited':
                return '<span class="sf-status-badge sf-status-limited">&#9888; ' . esc_html__('Limited', 'screaming-fixes') . '</span>';
            case 'disabled':
                return '<span class="sf-status-badge sf-status-disabled">&#10005; ' . esc_html__('Disabled', 'screaming-fixes') . '</span>';
            default:
                return '<span class="sf-status-badge sf-status-unknown">? ' . esc_html__('Unknown', 'screaming-fixes') . '</span>';
        }
    }

    /**
     * Get help text based on status
     *
     * @return string Help text HTML
     */
    public static function get_help_text() {
        $status = self::get_status();

        if ($status['status_code'] === 'disabled') {
            return sprintf(
                '<p class="sf-help-text sf-help-warning">%s</p><p class="sf-help-text">%s</p>',
                esc_html__('Revisions are disabled on this site. To enable them, add the following to your wp-config.php file:', 'screaming-fixes'),
                '<code>define(\'WP_POST_REVISIONS\', true);</code>'
            );
        }

        if ($status['status_code'] === 'limited') {
            return sprintf(
                '<p class="sf-help-text sf-help-info">%s</p>',
                sprintf(
                    esc_html__('Your site keeps %d revisions per post. If you need to undo older changes, consider increasing this limit in wp-config.php.', 'screaming-fixes'),
                    $status['limit']
                )
            );
        }

        return sprintf(
            '<p class="sf-help-text sf-help-success">%s</p>',
            esc_html__('Revisions are fully enabled. You can restore any post to a previous version at any time.', 'screaming-fixes')
        );
    }
}
