<?php
/**
 * MailerLite Email Subscription Endpoint
 *
 * Registers a REST API endpoint for email subscriptions via MailerLite.
 *
 * @package Screaming_Fixes
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Register the MailerLite subscription REST endpoint
 */
function sf_register_mailerlite_endpoint() {
    register_rest_route('screaming-fixes/v1', '/subscribe', array(
        'methods'             => 'POST',
        'callback'            => 'sf_handle_mailerlite_subscribe',
        'permission_callback' => '__return_true', // Public endpoint
    ));
}
add_action('rest_api_init', 'sf_register_mailerlite_endpoint');

/**
 * Handle the subscription request
 *
 * @param WP_REST_Request $request The request object.
 * @return WP_REST_Response
 */
function sf_handle_mailerlite_subscribe($request) {
    $params = $request->get_json_params();

    // Honeypot check - if website field is filled, it's a bot
    if (!empty($params['website'])) {
        // Return fake success to fool bots
        return new WP_REST_Response(array('success' => true), 200);
    }

    // Get and validate email
    $email = isset($params['email']) ? sanitize_email($params['email']) : '';

    if (empty($email) || !is_email($email)) {
        return new WP_REST_Response(
            array('success' => false, 'error' => 'Invalid email address'),
            400
        );
    }

    // Get API key from constant or option
    $api_key = defined('MAILERLITE_API_KEY') ? MAILERLITE_API_KEY : get_option('sf_mailerlite_api_key', '');

    if (empty($api_key)) {
        error_log('Screaming Fixes: MAILERLITE_API_KEY not configured');
        return new WP_REST_Response(
            array('success' => false, 'error' => 'Server configuration error'),
            500
        );
    }

    // MailerLite Group ID for First Index
    $group_id = '178597163422450957';

    // Make request to MailerLite API
    $response = wp_remote_post('https://connect.mailerlite.com/api/subscribers', array(
        'headers' => array(
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
            'Authorization' => 'Bearer ' . $api_key,
        ),
        'body'    => wp_json_encode(array(
            'email'  => $email,
            'groups' => array($group_id),
            'status' => 'active',
        )),
        'timeout' => 15,
    ));

    // Check for WP error
    if (is_wp_error($response)) {
        error_log('Screaming Fixes MailerLite Error: ' . $response->get_error_message());
        return new WP_REST_Response(
            array('success' => false, 'error' => 'Subscription failed'),
            500
        );
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body        = json_decode(wp_remote_retrieve_body($response), true);

    // Success responses: 200, 201 (created), 422/409 (already subscribed)
    if (in_array($status_code, array(200, 201), true)) {
        // Mark as subscribed in WordPress
        update_option('sf_email_subscribed', true);

        return new WP_REST_Response(array('success' => true), 200);
    }

    // Already subscribed is still a success from user perspective
    if (in_array($status_code, array(422, 409), true)) {
        update_option('sf_email_subscribed', true);

        return new WP_REST_Response(
            array('success' => true, 'message' => 'Already subscribed'),
            200
        );
    }

    // Log unexpected errors
    error_log('Screaming Fixes MailerLite Error: HTTP ' . $status_code . ' - ' . wp_json_encode($body));

    return new WP_REST_Response(
        array('success' => false, 'error' => 'Subscription failed'),
        500
    );
}
