/**
 * Backlink Evaluator JavaScript
 *
 * Handles all interactions for the Backlink Evaluator module including:
 * - Running backlink scans
 * - Loading screen with rotating messages
 * - Data table filtering, sorting, searching
 * - Pagination
 * - CSV export
 */

(function($) {
    'use strict';

    // Module state
    let state = {
        backlinks: [],
        filteredBacklinks: [],
        summary: {},
        currentFilter: 'all',
        currentSort: { column: 'rank', direction: 'desc' },
        currentPage: 1,
        perPage: 100,
        searchTerm: '',
        isScanning: false,
        loadingStep: 1,
        messageInterval: null,
    };

    // Initialize on document ready
    $(document).ready(function() {
        initBacklinkEvaluator();
    });

    /**
     * Initialize the module
     */
    function initBacklinkEvaluator() {
        // Load existing data from embedded JSON
        loadExistingData();

        // Bind event handlers
        bindEvents();

        // Initialize pagination if we have data
        if (state.backlinks.length > 0) {
            applyFiltersAndSort();
        }
    }

    /**
     * Load existing data from the page
     */
    function loadExistingData() {
        const dataEl = document.getElementById('sf-backlinks-data');
        if (dataEl) {
            try {
                const data = JSON.parse(dataEl.textContent);
                state.backlinks = data.backlinks || [];
                state.summary = data.summary || {};
                state.filteredBacklinks = [...state.backlinks];
            } catch (e) {
                console.error('Failed to parse backlinks data:', e);
            }
        }
    }

    /**
     * Bind all event handlers
     */
    function bindEvents() {
        // Run scan button
        $(document).on('click', '#sf-run-backlink-scan', handleRunScan);

        // Export CSV buttons
        $(document).on('click', '.sf-export-csv-btn', handleExportCSV);

        // Quick filter buttons
        $(document).on('click', '.sf-filter-btn', handleFilterClick);

        // Search input
        $(document).on('input', '#sf-backlink-search', debounce(handleSearch, 300));

        // Sortable columns
        $(document).on('click', '.sf-sortable', handleSortClick);

        // Pagination
        $(document).on('click', '.sf-page-prev', handlePrevPage);
        $(document).on('click', '.sf-page-next', handleNextPage);
        $(document).on('click', '.sf-page-number', handlePageClick);

        // Instructions toggle
        $(document).on('click', '.sf-instructions-header', handleInstructionsToggle);
    }

    /**
     * Handle run scan button click
     */
    function handleRunScan(e) {
        e.preventDefault();

        if (state.isScanning) return;

        state.isScanning = true;
        state.loadingStep = 1;

        // Show loading screen
        showLoadingScreen();

        // Start the scan
        runBacklinkScan();
    }

    /**
     * Show the loading screen
     */
    function showLoadingScreen() {
        const $loading = $('#sf-loading-screen');
        $loading.show();

        // Hide other content
        $('#sf-summary-cards').hide();
        $('.sf-filters-bar').hide();
        $('.sf-table-wrapper').hide();
        $('.sf-csv-upsell-bar').hide();
        $('.sf-empty-state').hide();

        // Start rotating messages
        updateLoadingMessage();
        state.messageInterval = setInterval(updateLoadingMessage, 3500);
    }

    /**
     * Hide the loading screen
     */
    function hideLoadingScreen() {
        const $loading = $('#sf-loading-screen');
        $loading.hide();

        // Clear message interval
        if (state.messageInterval) {
            clearInterval(state.messageInterval);
            state.messageInterval = null;
        }

        // Show content
        $('#sf-summary-cards').show();
        $('.sf-filters-bar').show();
        $('.sf-table-wrapper').show();
        $('.sf-csv-upsell-bar').show();

        state.isScanning = false;
    }

    /**
     * Update loading message based on current step
     */
    function updateLoadingMessage() {
        const i18n = window.sfBacklinkEvaluatorData?.i18n || {};
        let messages = [];

        switch (state.loadingStep) {
            case 1:
                messages = i18n.loadingStep1Messages || ['Fetching summary data...'];
                break;
            case 2:
                messages = i18n.loadingStep2Messages || ['Fetching backlink details...'];
                break;
            case 3:
                messages = i18n.loadingStep3Messages || ['Processing data...'];
                break;
        }

        const randomMessage = messages[Math.floor(Math.random() * messages.length)];
        $('#sf-loading-message').text(randomMessage);
        $('#sf-loading-step-current').text(state.loadingStep);

        // Update progress bar
        const progress = (state.loadingStep / 3) * 100;
        $('#sf-loading-bar-fill').css('width', progress + '%');
    }

    /**
     * Run the backlink scan via AJAX
     */
    function runBacklinkScan() {
        const data = window.sfBacklinkEvaluatorData || {};

        // Step 1: Fetching summary
        state.loadingStep = 1;
        updateLoadingMessage();

        $.ajax({
            url: data.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_backlink_evaluator_scan',
                nonce: data.nonce,
                domain: data.siteDomain,
            },
            timeout: 180000, // 3 minute timeout
            beforeSend: function() {
                // Simulate step progression
                setTimeout(() => {
                    state.loadingStep = 2;
                    updateLoadingMessage();
                }, 2000);

                setTimeout(() => {
                    state.loadingStep = 3;
                    updateLoadingMessage();
                }, 5000);
            },
            success: function(response) {
                if (response.success) {
                    // Update state with new data
                    state.backlinks = response.data.backlinks || [];
                    state.summary = response.data.summary || {};
                    state.filteredBacklinks = [...state.backlinks];
                    state.currentPage = 1;

                    // Update UI
                    updateSummaryCards();
                    updateFilterCounts();
                    applyFiltersAndSort();

                    // Hide loading and show success
                    hideLoadingScreen();
                    showToast(data.i18n?.scanComplete || 'Scan complete!', 'success');

                    // Reload page to show fresh data
                    window.location.reload();
                } else {
                    hideLoadingScreen();
                    showToast(response.data?.message || data.i18n?.scanFailed || 'Scan failed.', 'error');
                }
            },
            error: function(xhr, status, error) {
                hideLoadingScreen();
                let message = data.i18n?.scanFailed || 'Scan failed.';
                if (status === 'timeout') {
                    message = 'Request timed out. Please try again.';
                }
                showToast(message, 'error');
                console.error('Scan error:', error);
            }
        });
    }

    /**
     * Handle export CSV button click
     */
    function handleExportCSV(e) {
        e.preventDefault();

        const data = window.sfBacklinkEvaluatorData || {};
        const $btn = $(e.currentTarget);

        $btn.prop('disabled', true).addClass('sf-loading');

        $.ajax({
            url: data.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_backlink_evaluator_export',
                nonce: data.nonce,
                domain: data.siteDomain,
            },
            success: function(response) {
                if (response.success) {
                    // Decode base64 content and trigger download
                    const content = atob(response.data.content);
                    const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' });
                    const link = document.createElement('a');
                    link.href = URL.createObjectURL(blob);
                    link.download = response.data.filename;
                    link.click();
                    URL.revokeObjectURL(link.href);

                    showToast(data.i18n?.exportComplete || 'Download started!', 'success');
                } else {
                    showToast(response.data?.message || 'Export failed.', 'error');
                }
            },
            error: function() {
                showToast('Export failed.', 'error');
            },
            complete: function() {
                $btn.prop('disabled', false).removeClass('sf-loading');
            }
        });
    }

    /**
     * Handle filter button click
     */
    function handleFilterClick(e) {
        e.preventDefault();

        const $btn = $(e.currentTarget);
        const filter = $btn.data('filter');

        // Update active state
        $('.sf-filter-btn').removeClass('sf-filter-active');
        $btn.addClass('sf-filter-active');

        state.currentFilter = filter;
        state.currentPage = 1;

        applyFiltersAndSort();
    }

    /**
     * Handle search input
     */
    function handleSearch(e) {
        state.searchTerm = $(e.target).val().toLowerCase().trim();
        state.currentPage = 1;
        applyFiltersAndSort();
    }

    /**
     * Handle sortable column click
     */
    function handleSortClick(e) {
        const $th = $(e.currentTarget);
        const sortColumn = $th.data('sort');

        // Toggle direction if same column, otherwise default to desc for rank, asc for others
        if (state.currentSort.column === sortColumn) {
            state.currentSort.direction = state.currentSort.direction === 'asc' ? 'desc' : 'asc';
        } else {
            state.currentSort.column = sortColumn;
            state.currentSort.direction = sortColumn === 'rank' ? 'desc' : 'asc';
        }

        // Update UI
        $('.sf-sortable').removeClass('sf-sort-active sf-sort-asc sf-sort-desc');
        $th.addClass('sf-sort-active sf-sort-' + state.currentSort.direction);

        // Update sort icon
        $('.sf-sortable .sf-sort-icon').text('');
        $th.find('.sf-sort-icon').text(state.currentSort.direction === 'asc' ? '\u25B2' : '\u25BC');

        applyFiltersAndSort();
    }

    /**
     * Apply current filters and sorting to backlinks
     */
    function applyFiltersAndSort() {
        let filtered = [...state.backlinks];

        // Apply filter
        if (state.currentFilter !== 'all') {
            filtered = filtered.filter(link => {
                switch (state.currentFilter) {
                    case 'dofollow':
                        return link.dofollow === true;
                    case 'lost':
                        return link.is_lost === true;
                    case 'new':
                        return link.is_new === true;
                    default:
                        return true;
                }
            });
        }

        // Apply search
        if (state.searchTerm) {
            filtered = filtered.filter(link => {
                const domain = (link.domain_from || '').toLowerCase();
                const url = (link.url_from || '').toLowerCase();
                const anchor = (link.anchor || '').toLowerCase();
                const target = (link.url_to || '').toLowerCase();

                return domain.includes(state.searchTerm) ||
                       url.includes(state.searchTerm) ||
                       anchor.includes(state.searchTerm) ||
                       target.includes(state.searchTerm);
            });
        }

        // Apply sort
        filtered.sort((a, b) => {
            let aVal, bVal;

            switch (state.currentSort.column) {
                case 'domain':
                    aVal = (a.domain_from || '').toLowerCase();
                    bVal = (b.domain_from || '').toLowerCase();
                    break;
                case 'anchor':
                    aVal = (a.anchor || '').toLowerCase();
                    bVal = (b.anchor || '').toLowerCase();
                    break;
                case 'rank':
                    aVal = a.rank || 0;
                    bVal = b.rank || 0;
                    break;
                case 'date':
                    aVal = a.first_seen || '';
                    bVal = b.first_seen || '';
                    break;
                default:
                    return 0;
            }

            let result = 0;
            if (aVal < bVal) result = -1;
            if (aVal > bVal) result = 1;

            return state.currentSort.direction === 'desc' ? -result : result;
        });

        state.filteredBacklinks = filtered;

        // Render the table
        renderTable();
        updatePagination();
    }

    /**
     * Render the backlinks table
     */
    function renderTable() {
        const $tbody = $('#sf-backlinks-tbody');
        const start = (state.currentPage - 1) * state.perPage;
        const end = start + state.perPage;
        const pageData = state.filteredBacklinks.slice(start, end);

        if (pageData.length === 0) {
            $tbody.html('<tr><td colspan="8" class="sf-no-results">No backlinks match your filters.</td></tr>');
            return;
        }

        let html = '';
        pageData.forEach(link => {
            html += renderBacklinkRow(link);
        });

        $tbody.html(html);
    }

    /**
     * Render a single backlink row
     */
    function renderBacklinkRow(link) {
        const domainFrom = link.domain_from || '';
        const urlFrom = link.url_from || '';
        const urlTo = link.url_to || '';
        const anchor = link.anchor || '';
        const dofollow = link.dofollow || false;
        const rank = link.rank || 0;
        const firstSeen = link.first_seen || '';
        const isLost = link.is_lost || false;
        const isNew = link.is_new || false;
        const attributes = link.attributes || [];

        // Get display values
        const attrDisplay = getAttrDisplay(dofollow, attributes);
        const statusDisplay = getStatusDisplay(isLost, isNew);
        const rankClass = getRankClass(rank);

        // Format date
        let dateDisplay = '';
        let dateFull = '';
        if (firstSeen) {
            const date = new Date(firstSeen);
            dateDisplay = date.toISOString().split('T')[0];
            dateFull = date.toISOString().replace('T', ' ').substring(0, 19);
        }

        // Truncate values
        const urlFromTruncated = urlFrom.length > 60 ? urlFrom.substring(0, 57) + '...' : urlFrom;
        const anchorTruncated = anchor.length > 50 ? anchor.substring(0, 47) + '...' : anchor;

        let urlToPath = '/';
        try {
            urlToPath = new URL(urlTo).pathname || '/';
        } catch (e) {}
        const urlToTruncated = urlToPath.length > 50 ? urlToPath.substring(0, 47) + '...' : urlToPath;

        // Filter classes
        const filterClasses = [];
        if (dofollow) filterClasses.push('sf-filter-dofollow');
        if (isLost) filterClasses.push('sf-filter-lost');
        if (isNew) filterClasses.push('sf-filter-new');

        return `
            <tr class="sf-backlink-row ${filterClasses.join(' ')}"
                data-domain="${escapeHtml(domainFrom)}"
                data-anchor="${escapeHtml(anchor)}"
                data-rank="${rank}"
                data-date="${escapeHtml(firstSeen)}"
                data-dofollow="${dofollow ? '1' : '0'}"
                data-lost="${isLost ? '1' : '0'}"
                data-new="${isNew ? '1' : '0'}">
                <td class="sf-col-domain">
                    <span class="sf-domain-text">${escapeHtml(domainFrom)}</span>
                </td>
                <td class="sf-col-page">
                    <a href="${escapeHtml(urlFrom)}" target="_blank" rel="noopener" title="${escapeHtml(urlFrom)}">
                        ${escapeHtml(urlFromTruncated)}
                        <span class="sf-external-icon">&#8599;</span>
                    </a>
                </td>
                <td class="sf-col-anchor" title="${escapeHtml(anchor)}">
                    ${anchor ? escapeHtml(anchorTruncated) : '<span class="sf-empty-anchor">(no anchor)</span>'}
                </td>
                <td class="sf-col-attr">
                    <span class="sf-attr-badge ${attrDisplay.class}" title="${attrDisplay.title}">
                        ${attrDisplay.label}
                    </span>
                </td>
                <td class="sf-col-rank">
                    <span class="sf-rank-value ${rankClass}">${rank}</span>
                </td>
                <td class="sf-col-target" title="${escapeHtml(urlTo)}">
                    <a href="${escapeHtml(urlTo)}" target="_blank" rel="noopener">
                        ${escapeHtml(urlToTruncated)}
                    </a>
                </td>
                <td class="sf-col-date" title="${dateFull}">
                    ${dateDisplay}
                </td>
                <td class="sf-col-status">
                    <span class="sf-status-badge ${statusDisplay.class}" title="${statusDisplay.title}">
                        ${statusDisplay.label}
                    </span>
                </td>
            </tr>
        `;
    }

    /**
     * Get attribute display info
     */
    function getAttrDisplay(dofollow, attributes) {
        if (dofollow) {
            return { label: 'DF', class: 'sf-attr-dofollow', title: 'Dofollow - passes link equity' };
        }
        if (attributes.includes('sponsored')) {
            return { label: 'SP', class: 'sf-attr-sponsored', title: 'Sponsored - paid/advertisement link' };
        }
        if (attributes.includes('ugc')) {
            return { label: 'UGC', class: 'sf-attr-ugc', title: 'User Generated Content' };
        }
        return { label: 'NF', class: 'sf-attr-nofollow', title: 'Nofollow - does not pass link equity' };
    }

    /**
     * Get status display info
     */
    function getStatusDisplay(isLost, isNew) {
        if (isLost) {
            return { label: 'Lost', class: 'sf-status-lost', title: 'This link has been removed - reclamation opportunity' };
        }
        if (isNew) {
            return { label: 'New', class: 'sf-status-new', title: 'Recently discovered link' };
        }
        return { label: 'Live', class: 'sf-status-live', title: 'Active, healthy link' };
    }

    /**
     * Get rank CSS class
     */
    function getRankClass(rank) {
        if (rank >= 700) return 'sf-rank-high';
        if (rank >= 400) return 'sf-rank-medium';
        if (rank >= 100) return 'sf-rank-low';
        return 'sf-rank-very-low';
    }

    /**
     * Update pagination controls
     */
    function updatePagination() {
        const total = state.filteredBacklinks.length;
        const totalPages = Math.ceil(total / state.perPage);
        const start = (state.currentPage - 1) * state.perPage + 1;
        const end = Math.min(state.currentPage * state.perPage, total);

        $('.sf-page-start').text(total > 0 ? start : 0);
        $('.sf-page-end').text(end);
        $('.sf-page-total').text(total);

        // Enable/disable prev/next
        $('.sf-page-prev').prop('disabled', state.currentPage <= 1);
        $('.sf-page-next').prop('disabled', state.currentPage >= totalPages);

        // Generate page numbers
        const $pageNumbers = $('.sf-page-numbers');
        $pageNumbers.empty();

        if (totalPages <= 7) {
            for (let i = 1; i <= totalPages; i++) {
                $pageNumbers.append(createPageButton(i));
            }
        } else {
            // Show first, last, and pages around current
            const pages = [1];
            if (state.currentPage > 3) pages.push('...');
            for (let i = Math.max(2, state.currentPage - 1); i <= Math.min(totalPages - 1, state.currentPage + 1); i++) {
                pages.push(i);
            }
            if (state.currentPage < totalPages - 2) pages.push('...');
            if (totalPages > 1) pages.push(totalPages);

            pages.forEach(p => {
                if (p === '...') {
                    $pageNumbers.append('<span class="sf-page-ellipsis">...</span>');
                } else {
                    $pageNumbers.append(createPageButton(p));
                }
            });
        }
    }

    /**
     * Create a page number button
     */
    function createPageButton(pageNum) {
        const isActive = pageNum === state.currentPage;
        return `<button type="button" class="sf-page-number ${isActive ? 'sf-page-active' : ''}" data-page="${pageNum}">${pageNum}</button>`;
    }

    /**
     * Handle previous page click
     */
    function handlePrevPage() {
        if (state.currentPage > 1) {
            state.currentPage--;
            renderTable();
            updatePagination();
            scrollToTable();
        }
    }

    /**
     * Handle next page click
     */
    function handleNextPage() {
        const totalPages = Math.ceil(state.filteredBacklinks.length / state.perPage);
        if (state.currentPage < totalPages) {
            state.currentPage++;
            renderTable();
            updatePagination();
            scrollToTable();
        }
    }

    /**
     * Handle page number click
     */
    function handlePageClick(e) {
        const page = parseInt($(e.currentTarget).data('page'));
        if (page !== state.currentPage) {
            state.currentPage = page;
            renderTable();
            updatePagination();
            scrollToTable();
        }
    }

    /**
     * Scroll to table top
     */
    function scrollToTable() {
        const $table = $('.sf-table-wrapper');
        if ($table.length) {
            $('html, body').animate({
                scrollTop: $table.offset().top - 50
            }, 300);
        }
    }

    /**
     * Update summary cards with new data
     */
    function updateSummaryCards() {
        $('#sf-total-backlinks').text(numberFormat(state.summary.backlinks || 0));
        $('#sf-referring-domains').text(numberFormat(state.summary.referring_domains || 0));

        const dofollow = (state.summary.referring_domains || 0) - (state.summary.referring_domains_nofollow || 0);
        const percent = state.summary.referring_domains > 0
            ? ((dofollow / state.summary.referring_domains) * 100).toFixed(1)
            : 0;
        $('#sf-dofollow-percent').text(percent + '%');
    }

    /**
     * Update filter counts
     */
    function updateFilterCounts() {
        let dofollow = 0, lost = 0, newLinks = 0;

        state.backlinks.forEach(link => {
            if (link.dofollow) dofollow++;
            if (link.is_lost) lost++;
            if (link.is_new) newLinks++;
        });

        $('[data-filter="all"] .sf-filter-count').text(state.backlinks.length);
        $('[data-filter="dofollow"] .sf-filter-count').text(dofollow);
        $('[data-filter="lost"] .sf-filter-count').text(lost);
        $('[data-filter="new"] .sf-filter-count').text(newLinks);
    }

    /**
     * Handle instructions toggle
     */
    function handleInstructionsToggle(e) {
        const $header = $(e.currentTarget);
        const $box = $header.closest('.sf-instructions-box');
        const $content = $box.find('.sf-instructions-content');
        const isCollapsed = $box.attr('data-collapsed') === 'true';

        if (isCollapsed) {
            $content.slideDown(300);
            $box.attr('data-collapsed', 'false');
            $header.attr('aria-expanded', 'true');
            $header.find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
        } else {
            $content.slideUp(300);
            $box.attr('data-collapsed', 'true');
            $header.attr('aria-expanded', 'false');
            $header.find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
        }
    }

    /**
     * Show toast notification
     */
    function showToast(message, type) {
        // Use existing toast system if available
        if (typeof window.sfShowToast === 'function') {
            window.sfShowToast(message, type);
            return;
        }

        // Fallback toast
        const $toast = $(`<div class="sf-toast sf-toast-${type}">${escapeHtml(message)}</div>`);
        $('body').append($toast);

        setTimeout(() => $toast.addClass('sf-toast-visible'), 10);
        setTimeout(() => {
            $toast.removeClass('sf-toast-visible');
            setTimeout(() => $toast.remove(), 300);
        }, 3000);
    }

    /**
     * Format number with commas
     */
    function numberFormat(num) {
        return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
    }

    /**
     * Escape HTML special characters
     */
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    /**
     * Debounce function
     */
    function debounce(func, wait) {
        let timeout;
        return function(...args) {
            clearTimeout(timeout);
            timeout = setTimeout(() => func.apply(this, args), wait);
        };
    }

})(jQuery);
