<?php
/**
 * Backlink Evaluator Module for Screaming Fixes
 *
 * Provides live backlink data inside the WordPress admin dashboard.
 * Pulls data from DataForSEO Backlinks API and presents it in a
 * scannable, actionable UI table with CSV export.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Backlink_Evaluator {

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'backlink-evaluator';

    /**
     * Module name
     * @var string
     */
    protected $name;

    /**
     * Module slug
     * @var string
     */
    protected $slug = 'backlink-evaluator';

    /**
     * Cache duration in seconds (7 days)
     * @var int
     */
    const CACHE_DURATION = 604800; // 7 days

    /**
     * Disable caching entirely - set to true for development
     * @var bool
     */
    const DISABLE_CACHE = false;

    /**
     * Backlink Evaluator API instance
     * @var SF_Backlink_Evaluator_API
     */
    private $api;

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Backlink Evaluator', 'screaming-fixes');

        // Include the API class if not already loaded
        if (!class_exists('SF_Backlink_Evaluator_API')) {
            require_once SF_PLUGIN_DIR . 'shared/class-backlink-evaluator-api.php';
        }

        $this->api = new SF_Backlink_Evaluator_API();

        $this->init();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register AJAX handlers
        add_action('wp_ajax_sf_backlink_evaluator_scan', [$this, 'ajax_scan_domain']);
        add_action('wp_ajax_sf_backlink_evaluator_get_results', [$this, 'ajax_get_results']);
        add_action('wp_ajax_sf_backlink_evaluator_export', [$this, 'ajax_export_csv']);
        add_action('wp_ajax_sf_backlink_evaluator_clear_cache', [$this, 'ajax_clear_cache']);

        // Enqueue module assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        $current_subtab = isset($_GET['subtab']) ? sanitize_text_field($_GET['subtab']) : '';

        // Only load on backlinks tab with evaluator subtab
        if ($current_tab !== 'backlinks' || $current_subtab !== 'evaluator') {
            return;
        }

        // Use file modification time for cache busting
        $css_file = SF_PLUGIN_DIR . 'modules/backlink-evaluator/assets/backlink-evaluator.css';
        $js_file = SF_PLUGIN_DIR . 'modules/backlink-evaluator/assets/backlink-evaluator.js';
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        wp_enqueue_style(
            'sf-backlink-evaluator',
            SF_PLUGIN_URL . 'modules/backlink-evaluator/assets/backlink-evaluator.css',
            ['screaming-fixes-admin'],
            $css_version
        );

        wp_enqueue_script(
            'sf-backlink-evaluator',
            SF_PLUGIN_URL . 'modules/backlink-evaluator/assets/backlink-evaluator.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        // Get site URL for default domain
        $site_url = wp_parse_url(home_url(), PHP_URL_HOST);

        // Check API availability
        $user_has_api_key = SF_Backlink_Evaluator_API::has_user_credentials();

        // Get cached data info
        $cached = $this->get_cached_scan($site_url);
        $has_cached_data = !empty($cached);
        $last_scan_date = $has_cached_data ? $cached['scanned_at'] : null;
        $can_scan = $this->can_scan($site_url);
        $time_until_scan = $this->get_time_until_scan($site_url);

        wp_localize_script('sf-backlink-evaluator', 'sfBacklinkEvaluatorData', [
            'nonce' => wp_create_nonce('sf_backlink_evaluator_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'siteUrl' => home_url(),
            'siteDomain' => $site_url,
            'settingsUrl' => admin_url('admin.php?page=screaming-fixes&tab=settings'),
            'userHasApiKey' => $user_has_api_key,
            'hasCachedData' => $has_cached_data,
            'lastScanDate' => $last_scan_date,
            'canScan' => $can_scan,
            'timeUntilScan' => $time_until_scan,
            'i18n' => [
                'scanning' => __('Scanning domain...', 'screaming-fixes'),
                'fetchingSummary' => __('Fetching summary data...', 'screaming-fixes'),
                'fetchingBacklinks' => __('Fetching backlink details...', 'screaming-fixes'),
                'processing' => __('Processing and caching data...', 'screaming-fixes'),
                'scanComplete' => __('Scan complete!', 'screaming-fixes'),
                'scanFailed' => __('Scan failed.', 'screaming-fixes'),
                'exporting' => __('Generating CSV...', 'screaming-fixes'),
                'exportComplete' => __('Download started!', 'screaming-fixes'),
                'noData' => __('No backlink data available.', 'screaming-fixes'),
                'rateLimited' => __('Scan limit reached. Try again later or add your own DataForSEO API key.', 'screaming-fixes'),
                'cooldownMessage' => __('Next scan available in %s', 'screaming-fixes'),
                'unlimitedScansTitle' => __('Want unlimited scans?', 'screaming-fixes'),
                'unlimitedScansMessage' => __('Connect your DataForSEO API in Settings for unlimited scans with no cooldown.', 'screaming-fixes'),
                'connectApi' => __('Connect API', 'screaming-fixes'),
                // Loading screen messages - Step 1
                'loadingStep1Messages' => [
                    __('Heading out to find your links...', 'screaming-fixes'),
                    __("Knocking on DataForSEO's door...", 'screaming-fixes'),
                    __('Checking the entire interwebs...', 'screaming-fixes'),
                    __("Asking the internet who's talking about you...", 'screaming-fixes'),
                    __('Scheduling play date with DataForSEO API...', 'screaming-fixes'),
                ],
                // Loading screen messages - Step 2
                'loadingStep2Messages' => [
                    __('Pulling individual backlink records...', 'screaming-fixes'),
                    __('Rounding up 1,000 of your best links...', 'screaming-fixes'),
                    __('Sorting through the link neighborhood...', 'screaming-fixes'),
                    __('DataForSEO is being very generous today...', 'screaming-fixes'),
                    __('Mining the backlink goldmine...', 'screaming-fixes'),
                    __('Asking each link if it\'s still alive...', 'screaming-fixes'),
                ],
                // Loading screen messages - Step 3
                'loadingStep3Messages' => [
                    __('Organizing your link data...', 'screaming-fixes'),
                    __('Building your dashboard...', 'screaming-fixes'),
                    __('Almost there — just a few more seconds...', 'screaming-fixes'),
                    __('Tucking data into the WordPress cache...', 'screaming-fixes'),
                    __('Polishing the results for you...', 'screaming-fixes'),
                ],
            ],
        ]);
    }

    /**
     * Get the module's tab content view
     *
     * @return string Path to view file
     */
    public function get_view() {
        return SF_PLUGIN_DIR . 'modules/backlink-evaluator/views/tab-content.php';
    }

    /**
     * Check if domain can be scanned (rate limit check)
     *
     * @param string $domain
     * @return bool
     */
    public function can_scan($domain) {
        // Users with their own API key bypass cooldown
        if (SF_Backlink_Evaluator_API::has_user_credentials()) {
            return true;
        }

        // When cache is disabled, always allow scanning
        if (self::DISABLE_CACHE) {
            return true;
        }

        $cached = $this->get_cached_scan($domain);

        if (!$cached) {
            return true;
        }

        // Check if cache has expired
        $expires_at = strtotime($cached['expires_at']);
        return time() > $expires_at;
    }

    /**
     * Get time until next scan is available (in seconds)
     *
     * @param string $domain
     * @return int Seconds until next scan, 0 if available now
     */
    public function get_time_until_scan($domain) {
        // Users with their own API key have no cooldown
        if (SF_Backlink_Evaluator_API::has_user_credentials()) {
            return 0;
        }

        if (self::DISABLE_CACHE) {
            return 0;
        }

        $cached = $this->get_cached_scan($domain);

        if (!$cached) {
            return 0;
        }

        $expires_at = strtotime($cached['expires_at']);
        return max(0, $expires_at - time());
    }

    /**
     * Format time remaining as human-readable string
     *
     * @param int $seconds
     * @return string
     */
    public function format_time_remaining($seconds) {
        if ($seconds <= 0) {
            return __('now', 'screaming-fixes');
        }

        $days = floor($seconds / 86400);
        $hours = floor(($seconds % 86400) / 3600);
        $minutes = floor(($seconds % 3600) / 60);

        $parts = [];

        if ($days > 0) {
            $parts[] = sprintf(_n('%d day', '%d days', $days, 'screaming-fixes'), $days);
        }
        if ($hours > 0) {
            $parts[] = sprintf(_n('%d hour', '%d hours', $hours, 'screaming-fixes'), $hours);
        }
        if ($minutes > 0 && $days === 0) {
            $parts[] = sprintf(_n('%d minute', '%d minutes', $minutes, 'screaming-fixes'), $minutes);
        }

        if (empty($parts)) {
            return __('less than a minute', 'screaming-fixes');
        }

        return implode(', ', $parts);
    }

    /**
     * Get cached scan data for a domain
     *
     * @param string $domain
     * @return array|null
     */
    public function get_cached_scan($domain) {
        global $wpdb;

        $domain = $this->clean_domain($domain);
        $table = $wpdb->prefix . 'screaming_fixes_backlink_evaluator_cache';

        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table;
        if (!$table_exists) {
            return null;
        }

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE domain = %s",
            $domain
        ), ARRAY_A);

        if (!$row) {
            return null;
        }

        // Decode JSON data
        $row['summary_data'] = json_decode($row['summary_data'], true);
        $row['backlinks_data'] = json_decode($row['backlinks_data'], true);

        return $row;
    }

    /**
     * Cache scan results
     *
     * @param string $domain
     * @param array $data API response data
     * @return bool
     */
    public function cache_scan_results($domain, $data) {
        global $wpdb;

        $domain = $this->clean_domain($domain);
        $table = $wpdb->prefix . 'screaming_fixes_backlink_evaluator_cache';

        $expires_at = date('Y-m-d H:i:s', time() + self::CACHE_DURATION);

        $result = $wpdb->query($wpdb->prepare(
            "INSERT INTO {$table} (domain, summary_data, backlinks_data, scanned_at, expires_at)
             VALUES (%s, %s, %s, NOW(), %s)
             ON DUPLICATE KEY UPDATE
             summary_data = VALUES(summary_data),
             backlinks_data = VALUES(backlinks_data),
             scanned_at = NOW(),
             expires_at = VALUES(expires_at)",
            $domain,
            wp_json_encode($data['summary'] ?? []),
            wp_json_encode($data['backlinks'] ?? []),
            $expires_at
        ));

        return $result !== false;
    }

    /**
     * Clear cache for a domain
     *
     * @param string $domain Empty string clears all
     * @return bool
     */
    public function clear_cache($domain = '') {
        global $wpdb;

        $table = $wpdb->prefix . 'screaming_fixes_backlink_evaluator_cache';

        if (empty($domain)) {
            return $wpdb->query("TRUNCATE TABLE {$table}") !== false;
        }

        $domain = $this->clean_domain($domain);
        return $wpdb->delete($table, ['domain' => $domain]) !== false;
    }

    /**
     * Clean/normalize domain
     *
     * @param string $domain
     * @return string
     */
    private function clean_domain($domain) {
        // Remove protocol
        $domain = preg_replace('#^https?://#', '', $domain);
        // Remove www.
        $domain = preg_replace('/^www\./', '', $domain);
        // Remove trailing slash
        $domain = rtrim($domain, '/');
        // Remove any path
        $domain = explode('/', $domain)[0];

        return strtolower($domain);
    }

    /**
     * AJAX: Scan domain for backlinks
     */
    public function ajax_scan_domain() {
        check_ajax_referer('sf_backlink_evaluator_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($domain)) {
            // Default to site domain
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        }

        $domain = $this->clean_domain($domain);

        // Check if can scan (rate limit)
        if (!$this->can_scan($domain)) {
            $time_remaining = $this->get_time_until_scan($domain);
            wp_send_json_error([
                'message' => sprintf(
                    __('Scan limit reached. Next scan available in %s.', 'screaming-fixes'),
                    $this->format_time_remaining($time_remaining)
                ),
                'code' => 'rate_limited',
                'time_remaining' => $time_remaining,
            ]);
        }

        // Fetch backlink data
        $result = $this->api->fetch_all($domain);

        if (is_wp_error($result)) {
            wp_send_json_error([
                'message' => $result->get_error_message(),
                'code' => $result->get_error_code(),
            ]);
        }

        // Cache the results
        $this->cache_scan_results($domain, $result);

        // Log activity
        if (class_exists('SF_Activity_Log')) {
            $backlink_count = count($result['backlinks'] ?? []);
            SF_Activity_Log::log('backlink-evaluator', $backlink_count);
        }

        wp_send_json_success([
            'message' => __('Scan complete!', 'screaming-fixes'),
            'summary' => $result['summary'],
            'backlinks' => $result['backlinks'],
            'backlink_count' => count($result['backlinks'] ?? []),
            'domain' => $domain,
            'scanned_at' => current_time('mysql'),
            'api_used' => $this->api->get_last_api_used(),
        ]);
    }

    /**
     * AJAX: Get cached results
     */
    public function ajax_get_results() {
        check_ajax_referer('sf_backlink_evaluator_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($domain)) {
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        }

        $domain = $this->clean_domain($domain);

        $cached = $this->get_cached_scan($domain);

        if (!$cached) {
            wp_send_json_error([
                'message' => __('No cached data available.', 'screaming-fixes'),
                'code' => 'no_data',
            ]);
        }

        wp_send_json_success([
            'summary' => $cached['summary_data'],
            'backlinks' => $cached['backlinks_data'],
            'backlink_count' => count($cached['backlinks_data'] ?? []),
            'domain' => $domain,
            'scanned_at' => $cached['scanned_at'],
            'expires_at' => $cached['expires_at'],
            'can_scan' => $this->can_scan($domain),
            'time_until_scan' => $this->get_time_until_scan($domain),
        ]);
    }

    /**
     * AJAX: Export backlinks as CSV
     */
    public function ajax_export_csv() {
        check_ajax_referer('sf_backlink_evaluator_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($domain)) {
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        }

        $domain = $this->clean_domain($domain);

        $cached = $this->get_cached_scan($domain);

        if (!$cached || empty($cached['backlinks_data'])) {
            wp_send_json_error([
                'message' => __('No data to export.', 'screaming-fixes'),
            ]);
        }

        $backlinks = $cached['backlinks_data'];

        // Build CSV content
        $csv_lines = [];

        // Header row (24 columns)
        $headers = [
            'referring_page_url',
            'referring_domain',
            'target_page_url',
            'anchor_text',
            'is_dofollow',
            'link_rank',
            'first_seen',
            'is_lost',
            'is_new',
            'domain_from_rank',
            'page_from_rank',
            'backlink_spam_score',
            'link_type',
            'link_attributes',
            'referring_domain_tld',
            'url_from_https',
            'text_before_anchor',
            'text_after_anchor',
            'image_alt',
            'semantic_location',
            'links_count',
            'last_seen',
            'url_to_status_code',
            'url_from_status_code',
        ];

        $csv_lines[] = $this->csv_row($headers);

        // Data rows
        foreach ($backlinks as $link) {
            $row = [
                $link['url_from'] ?? '',
                $link['domain_from'] ?? '',
                $link['url_to'] ?? '',
                $link['anchor'] ?? '',
                ($link['dofollow'] ?? false) ? 'TRUE' : 'FALSE',
                $link['rank'] ?? 0,
                $link['first_seen'] ?? '',
                ($link['is_lost'] ?? false) ? 'TRUE' : 'FALSE',
                ($link['is_new'] ?? false) ? 'TRUE' : 'FALSE',
                $link['domain_from_rank'] ?? 0,
                $link['page_from_rank'] ?? 0,
                $link['backlink_spam_score'] ?? 0,
                $link['item_type'] ?? '',
                is_array($link['attributes'] ?? null) ? implode('|', $link['attributes']) : '',
                $link['domain_from_tld'] ?? '',
                ($link['url_from_https'] ?? false) ? 'TRUE' : 'FALSE',
                $link['text_pre'] ?? '',
                $link['text_post'] ?? '',
                $link['image_alt'] ?? '',
                $link['semantic_location'] ?? '',
                $link['links_count'] ?? 1,
                $link['last_seen'] ?? '',
                $link['url_to_status_code'] ?? '',
                $link['url_from_status_code'] ?? '',
            ];

            $csv_lines[] = $this->csv_row($row);
        }

        $csv_content = "\xEF\xBB\xBF" . implode("\n", $csv_lines); // UTF-8 BOM for Excel

        $filename = $domain . '-backlinks-overview-' . date('Y-m-d') . '.csv';

        wp_send_json_success([
            'filename' => $filename,
            'content' => base64_encode($csv_content),
            'count' => count($backlinks),
        ]);
    }

    /**
     * Create a CSV row with proper escaping
     *
     * @param array $fields
     * @return string
     */
    private function csv_row($fields) {
        $escaped = [];
        foreach ($fields as $field) {
            $field = str_replace('"', '""', $field);
            if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
                $field = '"' . $field . '"';
            }
            $escaped[] = $field;
        }
        return implode(',', $escaped);
    }

    /**
     * AJAX: Clear cache
     */
    public function ajax_clear_cache() {
        check_ajax_referer('sf_backlink_evaluator_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($domain)) {
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        }

        $domain = $this->clean_domain($domain);

        if ($this->clear_cache($domain)) {
            wp_send_json_success([
                'message' => __('Cache cleared.', 'screaming-fixes'),
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Failed to clear cache.', 'screaming-fixes'),
            ]);
        }
    }
}
