<?php
/**
 * Backlink Evaluator Tab Content
 *
 * Main view for the Backlink Evaluator module displaying:
 * - Summary cards (Total Backlinks, Referring Domains, Dofollow %)
 * - Data table with 8 columns
 * - Quick filters, search, and pagination
 * - CSV export functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include the API class if not already loaded
if (!class_exists('SF_Backlink_Evaluator_API')) {
    require_once SF_PLUGIN_DIR . 'shared/class-backlink-evaluator-api.php';
}

// Get site domain
$site_domain = wp_parse_url(home_url(), PHP_URL_HOST);
$site_domain = preg_replace('/^www\./', '', $site_domain);

// Check for cached data
global $wpdb;
$table = $wpdb->prefix . 'screaming_fixes_backlink_evaluator_cache';
$table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table;

$cached_data = null;
$has_data = false;
$summary = [];
$backlinks = [];
$scanned_at = null;
$can_scan = true;
$time_until_scan = 0;

if ($table_exists) {
    $cached_row = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$table} WHERE domain = %s",
        $site_domain
    ), ARRAY_A);

    if ($cached_row) {
        $cached_data = $cached_row;
        $summary = json_decode($cached_row['summary_data'], true) ?: [];
        $backlinks = json_decode($cached_row['backlinks_data'], true) ?: [];
        $scanned_at = $cached_row['scanned_at'];
        $has_data = !empty($backlinks);

        // Check cooldown
        $user_has_api_key = SF_Backlink_Evaluator_API::has_user_credentials();
        if (!$user_has_api_key) {
            $expires_at = strtotime($cached_row['expires_at']);
            $can_scan = time() > $expires_at;
            $time_until_scan = max(0, $expires_at - time());
        }
    }
}

// Calculate dofollow percentage
$dofollow_percent = 0;
if (!empty($summary['referring_domains']) && $summary['referring_domains'] > 0) {
    $dofollow = $summary['referring_domains'] - ($summary['referring_domains_nofollow'] ?? 0);
    $dofollow_percent = round(($dofollow / $summary['referring_domains']) * 100, 1);
}

// Count statuses for quick stats
$live_count = 0;
$lost_count = 0;
$new_count = 0;
$dofollow_link_count = 0;

if ($has_data) {
    foreach ($backlinks as $link) {
        if ($link['is_lost'] ?? false) {
            $lost_count++;
        } elseif ($link['is_new'] ?? false) {
            $new_count++;
        } else {
            $live_count++;
        }
        if ($link['dofollow'] ?? false) {
            $dofollow_link_count++;
        }
    }
}
?>

<div class="sf-backlink-evaluator-module">
    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#128202;</span>
            <h2><?php esc_html_e('Backlink Evaluator', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Analyze your backlink profile', 'screaming-fixes'); ?></span>
        </div>
        <?php if ($has_data && $scanned_at): ?>
            <div class="sf-module-stats">
                <span class="sf-stat sf-stat-info">
                    <?php
                    printf(
                        esc_html__('Last scan: %s', 'screaming-fixes'),
                        esc_html(date_i18n(get_option('date_format'), strtotime($scanned_at)))
                    );
                    ?>
                </span>
            </div>
        <?php endif; ?>
    </div>

    <!-- Instructions Section -->
    <?php include SF_PLUGIN_DIR . 'modules/backlink-evaluator/views/instructions.php'; ?>

    <!-- Action Bar -->
    <div class="sf-action-bar">
        <div class="sf-action-bar-left">
            <?php if ($can_scan): ?>
                <button type="button" class="sf-button sf-button-primary sf-run-scan-btn" id="sf-run-backlink-scan">
                    <span class="dashicons dashicons-update"></span>
                    <?php esc_html_e('Run Backlink Check', 'screaming-fixes'); ?>
                </button>
            <?php else: ?>
                <button type="button" class="sf-button sf-button-primary" disabled title="<?php echo esc_attr(sprintf(__('Next scan available in %s', 'screaming-fixes'), $this->format_time_remaining($time_until_scan) ?? '')); ?>">
                    <span class="dashicons dashicons-clock"></span>
                    <?php esc_html_e('On Cooldown', 'screaming-fixes'); ?>
                </button>
                <span class="sf-cooldown-info">
                    <?php
                    // Calculate human-readable time
                    $days = floor($time_until_scan / 86400);
                    $hours = floor(($time_until_scan % 86400) / 3600);
                    if ($days > 0) {
                        printf(esc_html(_n('Next scan in %d day', 'Next scan in %d days', $days, 'screaming-fixes')), $days);
                    } elseif ($hours > 0) {
                        printf(esc_html(_n('Next scan in %d hour', 'Next scan in %d hours', $hours, 'screaming-fixes')), $hours);
                    } else {
                        $minutes = ceil($time_until_scan / 60);
                        printf(esc_html(_n('Next scan in %d minute', 'Next scan in %d minutes', $minutes, 'screaming-fixes')), $minutes);
                    }
                    ?>
                </span>
            <?php endif; ?>
        </div>
        <div class="sf-action-bar-right">
            <?php if ($has_data): ?>
                <button type="button" class="sf-button sf-button-secondary sf-export-csv-btn" id="sf-export-backlinks-csv">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Download Full CSV (24 columns)', 'screaming-fixes'); ?>
                </button>
            <?php endif; ?>
        </div>
    </div>

    <!-- Loading Screen (hidden by default) -->
    <div class="sf-loading-screen" id="sf-loading-screen" style="display: none;">
        <div class="sf-loading-content">
            <div class="sf-loading-spinner">
                <span class="sf-spinner-icon">&#128269;</span>
            </div>
            <div class="sf-loading-progress">
                <div class="sf-loading-step-indicator">
                    <span class="sf-step-text"><?php esc_html_e('Step', 'screaming-fixes'); ?> <span id="sf-loading-step-current">1</span> <?php esc_html_e('of', 'screaming-fixes'); ?> 3</span>
                </div>
                <div class="sf-loading-bar">
                    <div class="sf-loading-bar-fill" id="sf-loading-bar-fill" style="width: 0%;"></div>
                </div>
                <div class="sf-loading-message" id="sf-loading-message">
                    <?php esc_html_e('Heading out to find your links...', 'screaming-fixes'); ?>
                </div>
            </div>
        </div>
    </div>

    <?php if ($has_data): ?>
        <!-- Summary Cards -->
        <div class="sf-summary-cards" id="sf-summary-cards">
            <div class="sf-summary-card sf-card-backlinks">
                <div class="sf-card-icon">&#128279;</div>
                <div class="sf-card-content">
                    <span class="sf-card-value" id="sf-total-backlinks"><?php echo esc_html(number_format($summary['backlinks'] ?? 0)); ?></span>
                    <span class="sf-card-label"><?php esc_html_e('Total Live Backlinks', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-summary-card sf-card-domains">
                <div class="sf-card-icon">&#127760;</div>
                <div class="sf-card-content">
                    <span class="sf-card-value" id="sf-referring-domains"><?php echo esc_html(number_format($summary['referring_domains'] ?? 0)); ?></span>
                    <span class="sf-card-label"><?php esc_html_e('Referring Domains', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-summary-card sf-card-dofollow">
                <div class="sf-card-icon">&#9989;</div>
                <div class="sf-card-content">
                    <span class="sf-card-value" id="sf-dofollow-percent"><?php echo esc_html($dofollow_percent); ?>%</span>
                    <span class="sf-card-label"><?php esc_html_e('Dofollow', 'screaming-fixes'); ?></span>
                </div>
            </div>
        </div>

        <!-- Quick Filters & Search -->
        <div class="sf-filters-bar">
            <div class="sf-quick-filters">
                <button type="button" class="sf-filter-btn sf-filter-active" data-filter="all">
                    <?php esc_html_e('All', 'screaming-fixes'); ?>
                    <span class="sf-filter-count"><?php echo esc_html(count($backlinks)); ?></span>
                </button>
                <button type="button" class="sf-filter-btn" data-filter="dofollow">
                    <?php esc_html_e('Dofollow Only', 'screaming-fixes'); ?>
                    <span class="sf-filter-count"><?php echo esc_html($dofollow_link_count); ?></span>
                </button>
                <button type="button" class="sf-filter-btn sf-filter-lost" data-filter="lost">
                    <?php esc_html_e('Lost Links', 'screaming-fixes'); ?>
                    <span class="sf-filter-count"><?php echo esc_html($lost_count); ?></span>
                </button>
                <button type="button" class="sf-filter-btn sf-filter-new" data-filter="new">
                    <?php esc_html_e('New Links', 'screaming-fixes'); ?>
                    <span class="sf-filter-count"><?php echo esc_html($new_count); ?></span>
                </button>
            </div>
            <div class="sf-search-box">
                <span class="dashicons dashicons-search"></span>
                <input type="text" class="sf-search-input" id="sf-backlink-search" placeholder="<?php esc_attr_e('Search domain, page, or anchor...', 'screaming-fixes'); ?>">
            </div>
        </div>

        <!-- Data Table -->
        <div class="sf-table-wrapper sf-paginated-table" data-section="backlinks" data-per-page="100" data-total="<?php echo esc_attr(count($backlinks)); ?>">
            <table class="sf-results-table sf-backlinks-table" id="sf-backlinks-table">
                <thead>
                    <tr>
                        <th class="sf-col-domain sf-sortable" data-sort="domain"><?php esc_html_e('Referring Domain', 'screaming-fixes'); ?> <span class="sf-sort-icon"></span></th>
                        <th class="sf-col-page"><?php esc_html_e('Referring Page', 'screaming-fixes'); ?></th>
                        <th class="sf-col-anchor sf-sortable" data-sort="anchor"><?php esc_html_e('Anchor Text', 'screaming-fixes'); ?> <span class="sf-sort-icon"></span></th>
                        <th class="sf-col-attr"><?php esc_html_e('Attr', 'screaming-fixes'); ?></th>
                        <th class="sf-col-rank sf-sortable sf-sort-active sf-sort-desc" data-sort="rank"><?php esc_html_e('Rank', 'screaming-fixes'); ?> <span class="sf-sort-icon">&#9660;</span></th>
                        <th class="sf-col-target"><?php esc_html_e('Target Page', 'screaming-fixes'); ?></th>
                        <th class="sf-col-date sf-sortable" data-sort="date"><?php esc_html_e('First Seen', 'screaming-fixes'); ?> <span class="sf-sort-icon"></span></th>
                        <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody id="sf-backlinks-tbody">
                    <?php
                    // Display first 100 rows (pagination handled by JS)
                    $display_backlinks = array_slice($backlinks, 0, 100);

                    foreach ($display_backlinks as $index => $link):
                        $domain_from = $link['domain_from'] ?? '';
                        $url_from = $link['url_from'] ?? '';
                        $url_to = $link['url_to'] ?? '';
                        $anchor = $link['anchor'] ?? '';
                        $dofollow = $link['dofollow'] ?? false;
                        $rank = $link['rank'] ?? 0;
                        $first_seen = $link['first_seen'] ?? '';
                        $is_lost = $link['is_lost'] ?? false;
                        $is_new = $link['is_new'] ?? false;
                        $attributes = $link['attributes'] ?? [];

                        // Get display info
                        $attr_display = SF_Backlink_Evaluator_API::get_link_attribute_display($dofollow, $attributes);
                        $status_display = SF_Backlink_Evaluator_API::get_status_display($is_lost, $is_new);
                        $rank_class = SF_Backlink_Evaluator_API::get_rank_class($rank);

                        // Format date
                        $date_display = '';
                        $date_full = '';
                        if (!empty($first_seen)) {
                            $timestamp = strtotime($first_seen);
                            $date_display = date('Y-m-d', $timestamp);
                            $date_full = date('Y-m-d H:i:s', $timestamp);
                        }

                        // Truncate long values
                        $url_from_truncated = strlen($url_from) > 60 ? substr($url_from, 0, 57) . '...' : $url_from;
                        $anchor_truncated = strlen($anchor) > 50 ? substr($anchor, 0, 47) . '...' : $anchor;
                        $url_to_path = wp_parse_url($url_to, PHP_URL_PATH) ?: '/';
                        $url_to_truncated = strlen($url_to_path) > 50 ? substr($url_to_path, 0, 47) . '...' : $url_to_path;

                        // Determine filter classes
                        $filter_classes = [];
                        if ($dofollow) $filter_classes[] = 'sf-filter-dofollow';
                        if ($is_lost) $filter_classes[] = 'sf-filter-lost';
                        if ($is_new) $filter_classes[] = 'sf-filter-new';
                    ?>
                    <tr class="sf-backlink-row <?php echo esc_attr(implode(' ', $filter_classes)); ?>"
                        data-domain="<?php echo esc_attr($domain_from); ?>"
                        data-anchor="<?php echo esc_attr($anchor); ?>"
                        data-rank="<?php echo esc_attr($rank); ?>"
                        data-date="<?php echo esc_attr($first_seen); ?>"
                        data-dofollow="<?php echo $dofollow ? '1' : '0'; ?>"
                        data-lost="<?php echo $is_lost ? '1' : '0'; ?>"
                        data-new="<?php echo $is_new ? '1' : '0'; ?>">
                        <td class="sf-col-domain">
                            <span class="sf-domain-text"><?php echo esc_html($domain_from); ?></span>
                        </td>
                        <td class="sf-col-page">
                            <a href="<?php echo esc_url($url_from); ?>" target="_blank" rel="noopener" title="<?php echo esc_attr($url_from); ?>">
                                <?php echo esc_html($url_from_truncated); ?>
                                <span class="sf-external-icon">&#8599;</span>
                            </a>
                        </td>
                        <td class="sf-col-anchor" title="<?php echo esc_attr($anchor); ?>">
                            <?php if (empty($anchor)): ?>
                                <span class="sf-empty-anchor"><?php esc_html_e('(no anchor)', 'screaming-fixes'); ?></span>
                            <?php else: ?>
                                <?php echo esc_html($anchor_truncated); ?>
                            <?php endif; ?>
                        </td>
                        <td class="sf-col-attr">
                            <span class="sf-attr-badge <?php echo esc_attr($attr_display['class']); ?>" title="<?php echo esc_attr($attr_display['title']); ?>">
                                <?php echo esc_html($attr_display['label']); ?>
                            </span>
                        </td>
                        <td class="sf-col-rank">
                            <span class="sf-rank-value <?php echo esc_attr($rank_class); ?>"><?php echo esc_html($rank); ?></span>
                        </td>
                        <td class="sf-col-target" title="<?php echo esc_attr($url_to); ?>">
                            <a href="<?php echo esc_url($url_to); ?>" target="_blank" rel="noopener">
                                <?php echo esc_html($url_to_truncated); ?>
                            </a>
                        </td>
                        <td class="sf-col-date" title="<?php echo esc_attr($date_full); ?>">
                            <?php echo esc_html($date_display); ?>
                        </td>
                        <td class="sf-col-status">
                            <span class="sf-status-badge <?php echo esc_attr($status_display['class']); ?>" title="<?php echo esc_attr($status_display['title']); ?>">
                                <?php echo esc_html($status_display['label']); ?>
                            </span>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <!-- Pagination -->
            <?php if (count($backlinks) > 100): ?>
            <div class="sf-pagination" data-section="backlinks">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                    <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, count($backlinks))); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?>
                    <span class="sf-page-total"><?php echo esc_html(count($backlinks)); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <!-- CSV Upsell Bar -->
        <div class="sf-csv-upsell-bar">
            <span class="sf-upsell-icon">&#128196;</span>
            <span class="sf-upsell-text">
                <?php esc_html_e('This table shows 8 key columns. Download the full CSV for 24 columns including domain authority scores, spam scores, semantic location, link context snippets, and HTTP status codes.', 'screaming-fixes'); ?>
            </span>
            <button type="button" class="sf-button sf-button-primary sf-button-small sf-export-csv-btn" id="sf-export-backlinks-csv-bottom">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Download Full CSV', 'screaming-fixes'); ?>
            </button>
        </div>

    <?php else: ?>
        <!-- Empty State -->
        <div class="sf-empty-state">
            <div class="sf-empty-icon">&#128202;</div>
            <h3><?php esc_html_e('No backlink data yet', 'screaming-fixes'); ?></h3>
            <p><?php esc_html_e('Click "Run Backlink Check" to analyze your site\'s backlink profile. Results will be cached for 7 days.', 'screaming-fixes'); ?></p>
            <?php if (!SF_Backlink_Evaluator_API::has_user_credentials()): ?>
                <p class="sf-empty-hint">
                    <?php
                    printf(
                        esc_html__('Want unlimited scans? %sConnect your DataForSEO API%s in Settings.', 'screaming-fixes'),
                        '<a href="' . esc_url(admin_url('admin.php?page=screaming-fixes&tab=settings')) . '">',
                        '</a>'
                    );
                    ?>
                </p>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <!-- Attribution Footer -->
    <div class="sf-attribution-footer">
        <?php esc_html_e('Backlink data powered by DataForSEO', 'screaming-fixes'); ?>
    </div>
</div>

<!-- Store backlinks data for JavaScript -->
<script type="application/json" id="sf-backlinks-data">
<?php echo wp_json_encode([
    'backlinks' => $backlinks,
    'summary' => $summary,
    'domain' => $site_domain,
    'scanned_at' => $scanned_at,
    'can_scan' => $can_scan,
    'time_until_scan' => $time_until_scan,
]); ?>
</script>
