/**
 * Backlink Reclaim Module JavaScript
 *
 * Handles domain scanning, redirect creation, and UI interactions
 */

(function($) {
    'use strict';

    var BacklinkReclaim = {

        // Pagination state
        pagination: {
            'dead-pages': { currentPage: 1, perPage: 100, totalItems: 0 },
            'redirects-applied': { currentPage: 1, perPage: 100, totalItems: 0 }
        },

        /**
         * Initialize the module
         */
        init: function() {
            this.bindEvents();
            this.updateSelectedCount();
            this.checkPostScanAlert();
            this.initSectionToggles();
            this.initPagination();
        },

        /**
         * Initialize collapsible section toggles
         */
        initSectionToggles: function() {
            var self = this;

            // Dead Pages section toggle
            $('.sf-dead-pages-toggle').on('click', function() {
                self.toggleSection('dead-pages');
            });

            // Redirects Applied section toggle
            $('.sf-redirects-toggle').on('click', function() {
                self.toggleSection('redirects-applied');
            });
        },

        /**
         * Toggle a collapsible section
         */
        toggleSection: function(section) {
            var $toggle, $content, $icon;

            if (section === 'dead-pages') {
                $toggle = $('.sf-dead-pages-toggle');
                $content = $('.sf-dead-pages-content');
                $icon = $toggle.find('.sf-toggle-icon');
            } else if (section === 'redirects-applied') {
                $toggle = $('.sf-redirects-toggle');
                $content = $('.sf-redirects-applied-content');
                $icon = $toggle.find('.sf-toggle-icon');
            } else {
                return;
            }

            var isExpanded = $toggle.attr('aria-expanded') === 'true';

            if (isExpanded) {
                // Collapse
                $content.slideUp(300);
                $toggle.attr('aria-expanded', 'false');
                $icon.removeClass('sf-rotated');
            } else {
                // Expand
                $content.slideDown(300);
                $toggle.attr('aria-expanded', 'true');
                $icon.addClass('sf-rotated');
            }
        },

        /**
         * Check if we should show post-scan alert for free users
         */
        checkPostScanAlert: function() {
            var self = this;
            var urlParams = new URLSearchParams(window.location.search);

            // Check if scan just completed and user doesn't have API credentials
            if (urlParams.get('scan_complete') === '1' && !sfBacklinkData.userHasApiKey) {
                // Remove the URL parameter to prevent showing again on refresh
                urlParams.delete('scan_complete');
                var newUrl = window.location.pathname + (urlParams.toString() ? '?' + urlParams.toString() : '');
                window.history.replaceState({}, '', newUrl);

                // Show the alert
                this.showPostScanAlert();
            }
        },

        /**
         * Show alert for free users after scan completion
         */
        showPostScanAlert: function() {
            var settingsUrl = sfBacklinkData.settingsUrl || (sfBacklinkData.ajaxUrl.replace('admin-ajax.php', 'admin.php?page=screaming-fixes&tab=settings'));

            var alertHtml = '<div class="sf-post-scan-alert">' +
                '<div class="sf-alert-icon">&#128161;</div>' +
                '<div class="sf-alert-content">' +
                    '<strong>' + (sfBacklinkData.i18n.unlimitedScansTitle || 'Want unlimited scans?') + '</strong>' +
                    '<p>' + (sfBacklinkData.i18n.unlimitedScansMessage || 'Connect your DataForSEO API in Settings for unlimited scans, or wait until tomorrow for another free scan.') + '</p>' +
                '</div>' +
                '<a href="' + settingsUrl + '" class="sf-button sf-button-primary sf-button-sm">' +
                    (sfBacklinkData.i18n.connectApi || 'Connect API') +
                '</a>' +
                '<button type="button" class="sf-alert-dismiss" title="Dismiss">&times;</button>' +
            '</div>';

            // Insert after the module header
            var $header = $('.sf-module-header');
            if ($header.length) {
                $(alertHtml).insertAfter($header);

                // Bind dismiss button
                $('.sf-post-scan-alert .sf-alert-dismiss').on('click', function() {
                    $(this).closest('.sf-post-scan-alert').fadeOut(200, function() {
                        $(this).remove();
                    });
                });
            }
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            var self = this;

            // Scan button
            $('#sf-scan-btn').on('click', function() {
                self.scanDomain();
            });

            // Domain input - allow Enter key
            $('#sf-domain-input').on('keypress', function(e) {
                if (e.which === 13) {
                    self.scanDomain();
                }
            });

            // Instructions toggle
            $(document).on('click', '.sf-instructions-header', function() {
                var $box = $(this).closest('.sf-instructions-box');
                var $content = $box.find('.sf-instructions-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Select all checkbox
            $('#sf-select-all').on('change', function() {
                var isChecked = $(this).prop('checked');
                $('.sf-row-checkbox').prop('checked', isChecked);
                self.updateSelectedCount();
            });

            // Individual row checkbox
            $(document).on('change', '.sf-row-checkbox', function() {
                self.updateSelectedCount();
            });

            // Redirect destination input
            $(document).on('input', '.sf-redirect-destination', function() {
                var $row = $(this).closest('.sf-backlink-row');
                var hasValue = $(this).val().trim() !== '';
                $row.find('.sf-create-redirect-btn').prop('disabled', !hasValue);
                $row.find('.sf-row-checkbox').prop('disabled', !hasValue);
                if (!hasValue) {
                    $row.find('.sf-row-checkbox').prop('checked', false);
                }
                self.updateSelectedCount();
            });

            // AI Suggest button (individual)
            $(document).on('click', '.sf-ai-suggest-btn', function() {
                var $row = $(this).closest('.sf-backlink-row');
                self.suggestDestination($row);
            });

            // AI Suggest All button
            $('#sf-ai-suggest-all').on('click', function() {
                self.suggestAllDestinations();
            });

            // Create redirect button (individual)
            $(document).on('click', '.sf-create-redirect-btn', function() {
                var $row = $(this).closest('.sf-backlink-row');
                self.createRedirect($row);
            });

            // Create all redirects button (legacy header button)
            $('#sf-create-all-redirects').on('click', function() {
                self.createBulkRedirects();
            });

            // Create selected redirects button (batch actions bar)
            $('#sf-create-selected-redirects').on('click', function() {
                self.createBulkRedirects();
            });

            // Export button
            $('#sf-export-results').on('click', function() {
                self.exportResults();
            });

            // Clear cache button
            $('#sf-clear-cache-btn').on('click', function() {
                self.clearCache();
            });

            // Export Redirects button
            $(document).on('click', '.sf-export-redirects-btn', function() {
                self.exportRedirects();
            });

            // Redirect status filter buttons
            $(document).on('click', '.sf-redirect-status-filters .sf-status-filter', function() {
                var status = $(this).data('status');
                self.filterRedirectResults(status);
            });

            // Pagination: Previous button
            $(document).on('click', '#sf-backlink-reclaim .sf-page-prev:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage - 1);
            });

            // Pagination: Next button
            $(document).on('click', '#sf-backlink-reclaim .sf-page-next:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage + 1);
            });

            // Pagination: Page number buttons
            $(document).on('click', '#sf-backlink-reclaim .sf-page-number:not(.active):not(.sf-page-ellipsis)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                var page = parseInt($(this).data('page'), 10);
                self.goToPage(section, page);
            });

            // Toggle referrers dropdown
            $(document).on('click', '.sf-expand-referrers', function(e) {
                e.stopPropagation();

                var $button = $(this);
                var $dropdown = $button.siblings('.sf-referrers-dropdown');
                var isActive = $button.hasClass('active');

                // Close all other dropdowns
                $('.sf-expand-referrers').not($button).removeClass('active');
                $('.sf-referrers-dropdown').not($dropdown).hide();

                // Toggle this one
                if (!isActive) {
                    $button.addClass('active');
                    $dropdown.show();
                } else {
                    $button.removeClass('active');
                    $dropdown.hide();
                }
            });

            // Close dropdown when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.sf-referrers-cell').length) {
                    $('.sf-expand-referrers').removeClass('active');
                    $('.sf-referrers-dropdown').hide();
                }
            });

            // Prevent dropdown from closing when clicking inside it
            $(document).on('click', '.sf-referrers-dropdown', function(e) {
                e.stopPropagation();
            });

            // AI Suggestion dropdown expand/collapse
            $(document).on('click', '.sf-ai-suggestion-header', function(e) {
                e.stopPropagation();
                var $header = $(this);
                var $dropdown = $header.siblings('.sf-ai-suggestion-dropdown');
                var isExpanded = $header.attr('data-expanded') === 'true';

                // Close other open AI dropdowns first
                $('.sf-ai-suggestion-dropdown').not($dropdown).slideUp(200);
                $('.sf-ai-suggestion-header').not($header).attr('data-expanded', 'false')
                    .find('.sf-ai-expand-toggle').text('▼');

                if (isExpanded) {
                    $dropdown.slideUp(200);
                    $header.attr('data-expanded', 'false');
                    $header.find('.sf-ai-expand-toggle').text('▼');
                } else {
                    $dropdown.slideDown(200);
                    $header.attr('data-expanded', 'true');
                    $header.find('.sf-ai-expand-toggle').text('▲');
                }
            });

            // Close AI dropdown when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.sf-ai-suggestion-wrapper').length) {
                    $('.sf-ai-suggestion-dropdown').slideUp(200);
                    $('.sf-ai-suggestion-header').attr('data-expanded', 'false')
                        .find('.sf-ai-expand-toggle').text('▼');
                }
            });

            // Prevent AI dropdown from closing when clicking inside it
            $(document).on('click', '.sf-ai-suggestion-dropdown', function(e) {
                e.stopPropagation();
            });
        },

        /**
         * Scan domain for broken backlinks
         */
        scanDomain: function() {
            var self = this;
            var domain = $('#sf-domain-input').val().trim();

            if (!domain) {
                this.showNotice('error', 'Please enter a domain.');
                return;
            }

            // Check prerequisites - user needs either their own API key or free scan available
            if (!sfBacklinkData.userHasApiKey && !sfBacklinkData.canUseFreeScan) {
                this.showNotice('error', sfBacklinkData.i18n.noCredentials);
                return;
            }

            var $btn = $('#sf-scan-btn');
            var originalHtml = $btn.html();

            // Show scanning state
            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> ' + sfBacklinkData.i18n.scanning);
            $('.sf-scan-section').addClass('sf-scanning');
            $('#sf-scanning-section').show();
            $('.sf-results-section, .sf-stats-summary, .sf-scan-info, .sf-empty-state').hide();

            // Show animated loading container with rotating messages
            var scanLoadingState = self.showScanLoadingContainer();

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_scan',
                    nonce: sfBacklinkData.nonce,
                    domain: domain
                },
                success: function(response) {
                    // Clean up loading state
                    self.hideScanLoadingContainer(scanLoadingState);
                    $btn.prop('disabled', false).html(originalHtml);
                    $('.sf-scan-section').removeClass('sf-scanning');
                    $('#sf-scanning-section').hide();

                    console.log('Scan response:', response);

                    if (response.success) {
                        // Render results directly without page reload
                        self.renderScanResults(response.data);

                        // Show post-scan alert for free users
                        if (!sfBacklinkData.userHasApiKey) {
                            self.showPostScanAlert();
                        }
                    } else {
                        var errorMsg = response.data.message || sfBacklinkData.i18n.scanFailed;

                        if (response.data.code === 'rate_limited') {
                            self.showNotice('warning', response.data.message);
                            if (response.data.data && response.data.data.remaining_seconds) {
                                $btn.prop('disabled', true);
                            }
                        } else {
                            self.showNotice('error', errorMsg);
                        }
                    }
                },
                error: function(xhr, status, error) {
                    // Clean up loading state
                    self.hideScanLoadingContainer(scanLoadingState);
                    $btn.prop('disabled', false).html(originalHtml);
                    $('.sf-scan-section').removeClass('sf-scanning');
                    $('#sf-scanning-section').hide();

                    console.error('AJAX error:', status, error);
                    self.showNotice('error', sfBacklinkData.i18n.scanFailed);
                }
            });
        },

        /**
         * Show animated loading container for domain scan
         */
        showScanLoadingContainer: function() {
            var self = this;

            // Remove any existing scan loading container
            $('.sf-scan-loading-container').remove();

            // Hide the static scanning content (the boring message)
            $('#sf-scanning-section .sf-scanning-content').hide();

            // Create loading container
            var $container = $('<div class="sf-scan-loading-container">' +
                '<div class="sf-scan-progress-bar"><div class="sf-scan-progress-fill"></div></div>' +
                '<div class="sf-scan-loading-message">' + self.scanLoadingMessages[0] + '</div>' +
                '</div>');

            // Insert into the scanning section (which is now visible)
            $('#sf-scanning-section').append($container);

            // Start message rotation (every 2.5 seconds)
            var messageIndex = 0;
            var interval = setInterval(function() {
                messageIndex = (messageIndex + 1) % self.scanLoadingMessages.length;
                var $message = $container.find('.sf-scan-loading-message');
                $message.addClass('fade-out');
                setTimeout(function() {
                    $message.text(self.scanLoadingMessages[messageIndex]).removeClass('fade-out');
                }, 300);
            }, 2500);

            return {
                container: $container,
                interval: interval
            };
        },

        /**
         * Hide scan loading container
         */
        hideScanLoadingContainer: function(loadingState) {
            if (loadingState && loadingState.interval) {
                clearInterval(loadingState.interval);
            }
            if (loadingState && loadingState.container) {
                loadingState.container.fadeOut(200, function() {
                    $(this).remove();
                });
            }
            // Restore the static content (in case scan section is shown again)
            $('#sf-scanning-section .sf-scanning-content').show();
        },

        /**
         * Get DR class for domain rank badge styling
         */
        getDrClass: function(dr) {
            if (dr >= 80) return 'high';
            if (dr >= 50) return 'medium';
            if (dr >= 20) return 'low';
            return 'very-low';
        },

        /**
         * Render scan results directly into the DOM without page reload
         */
        renderScanResults: function(data) {
            var self = this;
            var deadPages = data.dead_pages || [];
            var totalDeadPages = deadPages.length;
            var totalBacklinks = 0;

            for (var i = 0; i < deadPages.length; i++) {
                totalBacklinks += parseInt(deadPages[i].backlink_count, 10) || 0;
            }

            // Handle empty results
            if (totalDeadPages === 0) {
                $('.sf-empty-state').hide();
                // Show the "no dead backlinks" success state if it exists, otherwise create it
                var $noDeadBacklinks = $('.sf-no-dead-backlinks');
                if ($noDeadBacklinks.length) {
                    $noDeadBacklinks.show();
                } else {
                    var emptyHtml = '<div class="sf-no-dead-backlinks">' +
                        '<div class="sf-success-icon">&#127881;</div>' +
                        '<h3>No Dead Backlinks Found!</h3>' +
                        '<p>Great news! No external sites are linking to broken pages on your domain.</p>' +
                        '</div>';
                    $('.sf-scan-section').after(emptyHtml);
                }
                // Show scan info
                var $scanInfo = $('.sf-scan-info');
                if ($scanInfo.length) {
                    $scanInfo.find('strong').text('just now');
                    $scanInfo.show();
                } else {
                    $('.sf-scan-section').after(
                        '<div class="sf-scan-info"><span class="dashicons dashicons-calendar-alt"></span> Last scanned: <strong>just now</strong></div>'
                    );
                }
                return;
            }

            // Hide empty/no-data states
            $('.sf-empty-state, .sf-no-dead-backlinks').hide();

            // Update or create stats summary
            var $stats = $('.sf-stats-summary');
            if ($stats.length) {
                $stats.find('.sf-stat-card:first .sf-stat-number').text(totalDeadPages);
                $stats.find('.sf-stat-backlinks .sf-stat-number').text(totalBacklinks.toLocaleString());
                $stats.find('#sf-redirects-created').text('0');
                $stats.show();
            }

            // Update or create scan info
            var $scanInfo = $('.sf-scan-info');
            if ($scanInfo.length) {
                $scanInfo.find('strong').text('just now');
                $scanInfo.show();
            } else {
                $('.sf-stats-summary').after(
                    '<div class="sf-scan-info"><span class="dashicons dashicons-calendar-alt"></span> Last scanned: <strong>just now</strong></div>'
                );
            }

            // Update module header stats
            var $moduleStats = $('.sf-module-stats');
            var $statTotal = $moduleStats.find('.sf-stat-total');
            if ($statTotal.length) {
                $statTotal.html('<strong>' + totalDeadPages + '</strong> dead pages with backlinks');
            } else {
                $moduleStats.prepend('<span class="sf-stat sf-stat-total"><strong>' + totalDeadPages + '</strong> dead pages with backlinks</span>');
            }

            // Show Clear Cache button if not already visible
            if (!$('#sf-clear-cache-btn').length) {
                $('#sf-scan-btn').after(
                    '<button type="button" id="sf-clear-cache-btn" class="button sf-clear-cache-button" title="Clear cached data and rescan with live API">' +
                    '<span class="dashicons dashicons-trash"></span> Clear Cache</button>'
                );
            }

            // Build table rows
            var rowsHtml = '';
            for (var i = 0; i < deadPages.length; i++) {
                var page = deadPages[i];
                var previewDomains = (page.top_referrers || []).slice(0, 2);
                var domainNames = previewDomains.map(function(ref) { return ref.domain; });
                var moreCount = (page.top_referrers || []).length - 2;
                var previewText = domainNames.join(', ');
                if (moreCount > 0) {
                    previewText += ' <span class="sf-more">+' + moreCount + '</span>';
                }

                // Build referrer dropdown rows
                var referrerRows = '';
                var topRefs = (page.top_referrers || []).slice(0, 10);
                for (var j = 0; j < topRefs.length; j++) {
                    var ref = topRefs[j];
                    var dr = ref.domain_rank || 0;
                    var drClass = self.getDrClass(dr);
                    var refPath = '';
                    try {
                        var urlObj = new URL(ref.url);
                        refPath = urlObj.pathname || '/';
                    } catch(e) {
                        refPath = '/';
                    }
                    referrerRows += '<tr>' +
                        '<td class="sf-dr-cell"><span class="sf-dr-badge sf-dr-' + drClass + '">' + dr + '</span></td>' +
                        '<td class="sf-url-cell"><a href="' + self.escAttr(ref.url) + '" target="_blank" rel="noopener" title="' + self.escAttr(ref.url) + '">' +
                        '<span class="sf-referrer-domain">' + self.escHtml(ref.domain) + '</span>' +
                        '<span class="sf-referrer-path">' + self.escHtml(refPath) + '</span>' +
                        '<span class="sf-external-icon">&#8599;</span></a></td>' +
                        '<td class="sf-links-cell">' + (ref.backlinks || 1) + '</td></tr>';
                }

                var refDomainsText = page.referring_domains === 1 ? 'from 1 domains' : 'from ' + page.referring_domains + ' domains';

                rowsHtml += '<tr class="sf-backlink-row" data-index="' + i + '" data-source="' + self.escAttr(page.dead_page_path) + '" data-backlink-count="' + (page.backlink_count || 0) + '" data-status="pending">' +
                    '<td class="sf-col-checkbox"><input type="checkbox" class="sf-row-checkbox" /></td>' +
                    '<td class="sf-col-dead-page"><div class="sf-dead-page-cell">' +
                    '<span class="sf-status-badge sf-status-' + self.escAttr(page.status_code) + '">' + self.escHtml(page.status_code) + '</span>' +
                    '<div class="sf-page-info"><span class="sf-page-path" title="' + self.escAttr(page.dead_page) + '">' + self.escHtml(page.dead_page_path) + '</span>' +
                    '<a href="' + self.escAttr(page.dead_page) + '" target="_blank" rel="noopener" class="sf-external-link"><span class="dashicons dashicons-external"></span></a></div></div></td>' +
                    '<td class="sf-col-backlinks"><span class="sf-backlink-count">' + (page.backlink_count || 0).toLocaleString() + '</span>' +
                    '<span class="sf-domains-count">' + refDomainsText + '</span></td>' +
                    '<td class="sf-col-referrers sf-referrers-cell">' +
                    '<button type="button" class="sf-expand-referrers" data-dead-page="' + self.escAttr(page.dead_page_path) + '">' +
                    '<span class="sf-referrer-preview">' + previewText + '</span>' +
                    '<span class="sf-expand-icon">&#9660;</span></button>' +
                    '<div class="sf-referrers-dropdown" style="display: none;">' +
                    '<div class="sf-referrers-header"><strong>Top Referring Pages</strong>' +
                    '<span class="sf-referrers-count">' + topRefs.length + ' of ' + page.referring_domains + ' domains</span></div>' +
                    '<table class="sf-referrers-table"><thead><tr><th>DR</th><th>Referring Page</th><th>Links</th></tr></thead>' +
                    '<tbody>' + referrerRows + '</tbody></table></div></td>' +
                    '<td class="sf-col-redirect"><div class="sf-redirect-input-wrapper">' +
                    '<input type="text" class="sf-input sf-redirect-destination" placeholder="Enter destination URL" data-source="' + self.escAttr(page.dead_page_path) + '" />' +
                    '<button type="button" class="sf-ai-suggest-btn" title="AI: Find best replacement URL"><span class="sf-ai-icon">&#129302;</span></button></div></td>' +
                    '<td class="sf-col-actions"><button type="button" class="sf-button sf-button-primary sf-button-sm sf-create-redirect-btn" disabled>Create</button></td></tr>';
            }

            // Populate the table
            var $tbody = $('#sf-backlink-results');
            if ($tbody.length) {
                $tbody.html(rowsHtml);
            }

            // Update dead pages count in section header
            $('.sf-dead-pages-count').text(totalDeadPages);

            // Update table wrapper data attributes
            var $tableWrapper = $tbody.closest('.sf-paginated-table');
            if ($tableWrapper.length) {
                $tableWrapper.attr('data-total', totalDeadPages);
            }

            // Show results sections
            $('.sf-results-section.sf-section-dead-pages').show();
            $('.sf-dead-pages-content').show();

            // Make sure the section toggle shows expanded state
            var $toggle = $('.sf-dead-pages-toggle');
            $toggle.attr('aria-expanded', 'true');
            $toggle.find('.sf-toggle-icon').addClass('sf-rotated');

            // Reset select-all checkbox
            $('#sf-select-all').prop('checked', false);
        },

        /**
         * Escape HTML entities
         */
        escHtml: function(str) {
            if (!str) return '';
            var div = document.createElement('div');
            div.appendChild(document.createTextNode(String(str)));
            return div.innerHTML;
        },

        /**
         * Escape for use in HTML attributes
         */
        escAttr: function(str) {
            if (!str) return '';
            return String(str).replace(/&/g, '&amp;').replace(/"/g, '&quot;').replace(/'/g, '&#39;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
        },

        /**
         * Fun loading messages for domain scan
         */
        scanLoadingMessages: [
            "🔍 Hunting for backlinks...",
            "🕷️ Releasing the crawl spiders...",
            "📊 Evaluating link equity...",
            "🔗 Counting all the links pointing your way...",
            "🧮 Combobulating domain authority...",
            "🔬 Analyzing referring domains...",
            "🗺️ Mapping your backlink universe...",
            "🎯 Identifying the dead ones...",
            "💀 Finding the 404 graveyard...",
            "🧹 Sweeping through the link neighborhood...",
            "📡 Pinging external servers...",
            "🤔 Wondering why that site linked to you...",
            "🏆 Ranking your link juice...",
            "🔎 CSI: Backlink Investigation...",
            "☕ This is taking a while... grabbing coffee...",
            "🧠 Teaching AI about your domain...",
            "📈 Crunching the SEO numbers...",
            "🌐 Scanning the entire internet (just kidding, just your site)...",
            "🎰 Spinning the DataForSEO wheel...",
            "🏗️ Building your backlink report...",
            "⏳ Still working... SEO takes patience...",
            "🔥 Finding those hot link opportunities...",
            "🧩 Piecing together your link profile...",
            "🎪 Juggling API responses...",
            "✨ Almost there... probably..."
        ],

        /**
         * Fun loading messages for AI suggestion
         */
        loadingMessages: [
            "🔍 Scanning the interwebs...",
            "🧠 AI neurons firing...",
            "🕵️ Detective mode activated...",
            "📚 Reading every page on the site...",
            "🎯 Hunting for the perfect match...",
            "🔮 Consulting the SEO crystal ball...",
            "🚀 Searching at light speed...",
            "🤖 Beep boop... analyzing URLs...",
            "🎪 Juggling search results...",
            "🧩 Piecing together the puzzle...",
            "☕ AI is caffeinating...",
            "🎸 Speed-reading like a rockstar...",
            "🌐 Crawling through the matrix...",
            "🔬 Running URL experiments...",
            "🎯 Zeroing in on the best match..."
        ],

        /**
         * Suggest destination for a single row
         */
        suggestDestination: function($row) {
            var self = this;
            var $btn = $row.find('.sf-ai-suggest-btn');
            var $input = $row.find('.sf-redirect-destination');
            var deadPage = $row.data('source');
            var scannedDomain = $('#sf-domain-input').val().trim();

            // Get referrers from the dropdown data
            var referrers = [];
            $row.find('.sf-referrers-table tbody tr').each(function() {
                var $refRow = $(this);
                var domain = $refRow.find('.sf-referrer-domain').text().trim();
                var url = $refRow.find('.sf-url-cell a').attr('href');
                if (domain) {
                    referrers.push({ domain: domain, url: url || '' });
                }
            });

            // Show loading state with animated messages
            $btn.prop('disabled', true).addClass('sf-loading');
            $input.addClass('sf-ai-loading');

            // Create and show loading container
            var loadingState = self.showLoadingContainer($row);

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_suggest_destination',
                    nonce: sfBacklinkData.nonce,
                    dead_page: deadPage,
                    scanned_domain: scannedDomain,
                    referrers: JSON.stringify(referrers)
                },
                success: function(response) {
                    // Clean up loading state
                    self.hideLoadingContainer($row, loadingState);
                    $btn.prop('disabled', false).removeClass('sf-loading');
                    $input.removeClass('sf-ai-loading').attr('placeholder', sfBacklinkData.i18n.enterDestination || 'Enter destination URL');

                    if (response.success && response.data.destination) {
                        $input.val(response.data.destination).trigger('input');

                        // Add visual feedback based on source
                        var sourceClass = 'sf-ai-suggested';
                        if (response.data.source === 'ai') {
                            sourceClass = 'sf-ai-suggested sf-ai-source-ai';
                            // Show AI suggestion dropdown with details
                            self.showAiSuggestionDropdown($row, response.data);
                        } else if (response.data.source === 'wp_query') {
                            sourceClass = 'sf-ai-suggested sf-ai-source-wp';
                        } else if (response.data.source === 'ai_no_match') {
                            sourceClass = 'sf-ai-suggested sf-ai-source-nomatch';
                            self.showNotice('warning', response.data.explanation || 'AI could not find a suitable replacement.');
                        }

                        $input.addClass(sourceClass);
                        setTimeout(function() {
                            $input.removeClass('sf-ai-suggested sf-ai-source-ai sf-ai-source-wp sf-ai-source-nomatch');
                        }, 3000);
                    } else if (response.success && response.data.is_ai_no_match) {
                        // AI explicitly found no match - show homepage as fallback
                        var homepageUrl = 'https://' + scannedDomain.replace(/^(https?:\/\/)?(www\.)?/, '') + '/';
                        $input.val(homepageUrl).trigger('input');
                        $input.addClass('sf-ai-suggested sf-ai-source-nomatch');
                        self.showNotice('warning', response.data.explanation || 'No suitable replacement found. Defaulting to homepage.');
                        setTimeout(function() {
                            $input.removeClass('sf-ai-suggested sf-ai-source-nomatch');
                        }, 3000);
                    } else {
                        // Default to homepage
                        $input.val(sfBacklinkData.siteUrl + '/').trigger('input');
                    }
                },
                error: function() {
                    self.hideLoadingContainer($row, loadingState);
                    $btn.prop('disabled', false).removeClass('sf-loading');
                    $input.removeClass('sf-ai-loading').attr('placeholder', sfBacklinkData.i18n.enterDestination || 'Enter destination URL');
                    // Default to homepage
                    $input.val(sfBacklinkData.siteUrl + '/').trigger('input');
                }
            });
        },

        /**
         * Show animated loading container with rotating messages
         */
        showLoadingContainer: function($row) {
            var self = this;

            // Remove any existing loading container
            $row.find('.sf-ai-loading-container').remove();

            // Create loading container
            var $container = $('<div class="sf-ai-loading-container">' +
                '<div class="sf-ai-progress-bar"><div class="sf-ai-progress-fill"></div></div>' +
                '<div class="sf-ai-loading-message">' + self.loadingMessages[0] + '</div>' +
                '</div>');

            // Insert after the redirect input wrapper
            $row.find('.sf-redirect-input-wrapper').after($container);

            // Start message rotation
            var messageIndex = 0;
            var interval = setInterval(function() {
                messageIndex = (messageIndex + 1) % self.loadingMessages.length;
                var $message = $container.find('.sf-ai-loading-message');
                $message.addClass('fade-out');
                setTimeout(function() {
                    $message.text(self.loadingMessages[messageIndex]).removeClass('fade-out');
                }, 300);
            }, 2000);

            return {
                container: $container,
                interval: interval
            };
        },

        /**
         * Hide loading container
         */
        hideLoadingContainer: function($row, loadingState) {
            if (loadingState && loadingState.interval) {
                clearInterval(loadingState.interval);
            }
            if (loadingState && loadingState.container) {
                loadingState.container.fadeOut(200, function() {
                    $(this).remove();
                });
            }
        },

        /**
         * Show AI suggestion dropdown with details
         */
        showAiSuggestionDropdown: function($row, data) {
            var self = this;

            // Remove any existing dropdown
            $row.find('.sf-ai-suggestion-wrapper').remove();

            // Helper function to truncate URL for preview
            var truncateUrl = function(url, maxLength) {
                try {
                    var path = new URL(url).pathname;
                    if (path.length > maxLength) {
                        return path.substring(0, maxLength) + '...';
                    }
                    return path || '/';
                } catch(e) {
                    return url.substring(0, maxLength) + '...';
                }
            };

            // Helper function to get confidence display
            var getConfidenceDisplay = function(confidence) {
                switch(confidence) {
                    case 'high': return '🟢 High';
                    case 'medium': return '🟡 Medium';
                    case 'low': return '🟠 Low';
                    default: return '● Unknown';
                }
            };

            // Get status display
            var statusBadgeClass = 'sf-status-' + (data.verified || 'unknown');
            var statusText = data.verified === 'live' ? '✓ Live' : (data.verified === 'unverified' ? '● Unverified' : '● Unknown');
            var statusDetailText = data.verified === 'live' ? '✓ Verified Live' : (data.verified === 'unverified' ? '● Could not verify' : '● Unknown');

            // Build dropdown HTML
            var dropdownHtml = '<div class="sf-ai-suggestion-wrapper">' +
                '<div class="sf-ai-suggestion-header" data-expanded="false">' +
                    '<span class="sf-ai-url-preview">' + truncateUrl(data.destination, 35) + '</span>' +
                    '<span class="sf-ai-status-badge ' + statusBadgeClass + '">' + statusText + '</span>' +
                    '<span class="sf-ai-expand-toggle">▼</span>' +
                '</div>' +
                '<div class="sf-ai-suggestion-dropdown">' +
                    '<div class="sf-ai-dropdown-header">🤖 AI Suggestion Details</div>' +
                    '<div class="sf-ai-dropdown-content">' +
                        '<div class="sf-ai-detail-row">' +
                            '<span class="sf-ai-label">Suggested URL:</span>' +
                            '<a href="' + data.destination + '" target="_blank" class="sf-ai-url-full">' + data.destination + '</a>' +
                        '</div>' +
                        '<div class="sf-ai-detail-row">' +
                            '<span class="sf-ai-label">Status:</span>' +
                            '<span class="sf-ai-value ' + statusBadgeClass + '">' + statusDetailText + '</span>' +
                        '</div>' +
                        '<div class="sf-ai-detail-row">' +
                            '<span class="sf-ai-label">Confidence:</span>' +
                            '<span class="sf-ai-confidence-display sf-confidence-' + data.confidence + '">' + getConfidenceDisplay(data.confidence) + '</span>' +
                        '</div>' +
                        (data.explanation ? '<div class="sf-ai-detail-row sf-ai-explanation">' +
                            '<span class="sf-ai-label">Why this match:</span>' +
                            '<p class="sf-ai-value">"' + data.explanation + '"</p>' +
                        '</div>' : '') +
                    '</div>' +
                '</div>' +
            '</div>';

            // Insert after the redirect input wrapper
            var $cell = $row.find('.sf-col-redirect');
            $cell.find('.sf-redirect-input-wrapper').after(dropdownHtml);
        },

        /**
         * Suggest destinations for all rows
         */
        suggestAllDestinations: function() {
            var self = this;
            var $btn = $('#sf-ai-suggest-all');
            var $rows = $('.sf-backlink-row:not(.sf-row-completed)');
            var originalHtml = $btn.html();

            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> ' + sfBacklinkData.i18n.suggesting);

            var index = 0;
            var total = $rows.length;

            function processNext() {
                if (index >= total) {
                    $btn.prop('disabled', false).html(originalHtml);
                    self.showNotice('success', 'Suggestions complete for ' + total + ' pages.');
                    return;
                }

                var $row = $rows.eq(index);
                var $input = $row.find('.sf-redirect-destination');

                // Skip if already has a value
                if ($input.val().trim() !== '') {
                    index++;
                    processNext();
                    return;
                }

                self.suggestDestination($row);

                // Small delay between requests
                setTimeout(function() {
                    index++;
                    processNext();
                }, 200);
            }

            processNext();
        },

        /**
         * Create redirect for a single row
         */
        createRedirect: function($row) {
            var self = this;
            var $btn = $row.find('.sf-create-redirect-btn');
            var source = $row.data('source');
            var destination = $row.find('.sf-redirect-destination').val().trim();
            var domain = $('#sf-domain-input').val().trim();
            var backlinkCount = $row.data('backlink-count') || 0;

            if (!destination) {
                this.showNotice('error', 'Please enter a destination URL.');
                return;
            }

            if (!sfBacklinkData.hasRedirectPlugin) {
                this.showNotice('error', sfBacklinkData.i18n.noRedirectPlugin);
                return;
            }

            var originalText = $btn.text();
            $btn.prop('disabled', true).text('Creating...');

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_create_redirect',
                    nonce: sfBacklinkData.nonce,
                    source: source,
                    destination: destination,
                    domain: domain,
                    backlink_count: backlinkCount
                },
                success: function(response) {
                    if (response.success) {
                        $row.addClass('sf-row-completed');
                        $row.attr('data-status', 'success');

                        // Add to Redirects Applied section
                        if (response.data.redirect_result) {
                            self.addRedirectResultRow(response.data.redirect_result);
                        }

                        // Remove row from Dead Pages and update count
                        self.removeDeadPageRow($row);

                        self.updateRedirectCount();
                        self.updateSelectedCount();
                        self.showNotice('success', sfBacklinkData.i18n.redirectCreated);
                    } else {
                        $row.addClass('sf-row-failed');
                        $row.attr('data-status', 'failed');

                        // Add failed result to Redirects Applied section
                        if (response.data && response.data.redirect_result) {
                            self.addRedirectResultRow(response.data.redirect_result);
                        }

                        // Remove row from Dead Pages and update count (failed attempts too)
                        self.removeDeadPageRow($row);

                        self.showNotice('error', response.data.message || sfBacklinkData.i18n.redirectFailed);
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).text(originalText);
                    self.showNotice('error', sfBacklinkData.i18n.redirectFailed);
                }
            });
        },

        /**
         * Bulk redirect state
         */
        bulkState: {
            cancelled: false,
            inProgress: false
        },

        /**
         * Create redirects in bulk with progress modal
         */
        createBulkRedirects: function() {
            var self = this;
            var $checkedRows = $('.sf-row-checkbox:checked').closest('.sf-backlink-row:not(.sf-row-completed)');
            var domain = $('#sf-domain-input').val().trim();

            if ($checkedRows.length === 0) {
                this.showNotice('error', 'No redirects selected.');
                return;
            }

            if (!sfBacklinkData.hasRedirectPlugin) {
                this.showNotice('error', sfBacklinkData.i18n.noRedirectPlugin);
                return;
            }

            // Build redirects array
            var redirects = [];
            var rowMap = {};
            $checkedRows.each(function() {
                var $row = $(this);
                var source = $row.data('source');
                var destination = $row.find('.sf-redirect-destination').val().trim();
                var backlinkCount = $row.data('backlink-count') || 0;

                if (source && destination) {
                    redirects.push({
                        source: source,
                        destination: destination,
                        backlink_count: backlinkCount,
                        $row: $row
                    });
                    rowMap[source] = $row;
                }
            });

            if (redirects.length === 0) {
                this.showNotice('error', 'No valid redirects to create.');
                return;
            }

            // Show progress modal
            self.showProgressModal(redirects.length);
            self.bulkState.cancelled = false;
            self.bulkState.inProgress = true;

            // Process redirects one by one
            var processed = 0;
            var failed = 0;
            var succeeded = 0;

            function processNext() {
                if (self.bulkState.cancelled || processed >= redirects.length) {
                    // Done - close modal and show result
                    self.hideProgressModal();
                    self.bulkState.inProgress = false;
                    self.updateRedirectCount();
                    self.updateSelectedCount();

                    if (self.bulkState.cancelled) {
                        self.showNotice('warning', 'Cancelled. Created ' + succeeded + ' of ' + redirects.length + ' redirects.');
                    } else if (failed > 0) {
                        self.showNotice('warning', 'Created ' + succeeded + ' redirects, ' + failed + ' failed.');
                    } else {
                        self.showNotice('success', 'Successfully created ' + succeeded + ' redirects!');
                    }
                    return;
                }

                var redirect = redirects[processed];
                var $row = redirect.$row;

                // Update progress modal
                self.updateProgressModal(processed, redirects.length, redirect.source, failed);

                // Make AJAX call for this redirect
                $.ajax({
                    url: sfBacklinkData.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'sf_backlink_create_redirect',
                        nonce: sfBacklinkData.nonce,
                        source: redirect.source,
                        destination: redirect.destination,
                        domain: domain,
                        backlink_count: redirect.backlink_count
                    },
                    success: function(response) {
                        if (response.success) {
                            succeeded++;
                            $row.addClass('sf-row-completed');
                            $row.attr('data-status', 'success');

                            if (response.data.redirect_result) {
                                self.addRedirectResultRow(response.data.redirect_result);
                            }

                            // Remove from Dead Pages section
                            self.removeDeadPageRow($row);
                        } else {
                            failed++;
                            $row.addClass('sf-row-failed');
                            $row.attr('data-status', 'failed');

                            if (response.data && response.data.redirect_result) {
                                self.addRedirectResultRow(response.data.redirect_result);
                            }

                            // Remove from Dead Pages section (failed attempts too)
                            self.removeDeadPageRow($row);
                        }

                        processed++;
                        self.updateProgressModal(processed, redirects.length, '', failed);

                        // Small delay between requests
                        setTimeout(processNext, 100);
                    },
                    error: function() {
                        failed++;
                        $row.addClass('sf-row-failed');
                        $row.attr('data-status', 'failed');
                        $row.find('.sf-create-redirect-btn').text('Failed').addClass('sf-btn-failed');

                        processed++;
                        self.updateProgressModal(processed, redirects.length, '', failed);
                        setTimeout(processNext, 100);
                    }
                });
            }

            // Bind cancel button
            $('#sf-progress-cancel').off('click').on('click', function() {
                self.bulkState.cancelled = true;
            });

            // Start processing
            processNext();
        },

        /**
         * Show progress modal
         */
        showProgressModal: function(total) {
            var $modal = $('#sf-progress-modal');
            $modal.find('.sf-progress-bar-fill').css('width', '0%');
            $modal.find('.sf-progress-percent').text('0%');
            $modal.find('.sf-done-count').text('0');
            $modal.find('.sf-total-count').text(total);
            $modal.find('.sf-current-url').text('-');
            $modal.find('.sf-failed-count').text('0');
            $modal.find('.sf-progress-failed').hide();
            $modal.fadeIn(200);
        },

        /**
         * Update progress modal
         */
        updateProgressModal: function(done, total, currentUrl, failed) {
            var $modal = $('#sf-progress-modal');
            var percent = Math.round((done / total) * 100);

            $modal.find('.sf-progress-bar-fill').css('width', percent + '%');
            $modal.find('.sf-progress-percent').text(percent + '%');
            $modal.find('.sf-done-count').text(done);

            if (currentUrl) {
                // Truncate long URLs
                var displayUrl = currentUrl.length > 40 ? currentUrl.substring(0, 40) + '...' : currentUrl;
                $modal.find('.sf-current-url').text(displayUrl);
            }

            if (failed > 0) {
                $modal.find('.sf-failed-count').text(failed);
                $modal.find('.sf-progress-failed').show();
            }
        },

        /**
         * Hide progress modal
         */
        hideProgressModal: function() {
            $('#sf-progress-modal').fadeOut(200);
        },

        /**
         * Export results to CSV
         */
        exportResults: function() {
            var self = this;
            var $btn = $('#sf-export-results');
            var originalHtml = $btn.html();
            var domain = $('#sf-domain-input').val().trim();

            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> ' + sfBacklinkData.i18n.exporting);

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_export',
                    nonce: sfBacklinkData.nonce,
                    domain: domain
                },
                success: function(response) {
                    $btn.prop('disabled', false).html(originalHtml);

                    if (response.success) {
                        // Download the CSV
                        var blob = new Blob([response.data.csv], { type: 'text/csv' });
                        var url = window.URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = response.data.filename;
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                        window.URL.revokeObjectURL(url);
                    } else {
                        self.showNotice('error', response.data.message);
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).html(originalHtml);
                    self.showNotice('error', 'Export failed.');
                }
            });
        },

        /**
         * Update selected count and button state
         */
        updateSelectedCount: function() {
            var $checkedRows = $('.sf-row-checkbox:checked').closest('.sf-backlink-row:not(.sf-row-completed)');
            var count = $checkedRows.length;

            // Update legacy header button (if exists)
            var $btn = $('#sf-create-all-redirects');
            $btn.find('.sf-count').text('(' + count + ')');
            $btn.prop('disabled', count === 0);

            // Update batch actions bar
            var $batchBar = $('#sf-batch-actions-bar');
            var $batchCount = $batchBar.find('.sf-selected-count');
            var $batchBtn = $('#sf-create-selected-redirects');

            $batchCount.text(count);
            $batchBtn.prop('disabled', count === 0);

            // Show/hide batch actions bar with animation
            if (count > 0) {
                if (!$batchBar.is(':visible')) {
                    $batchBar.slideDown(200);
                }
            } else {
                if ($batchBar.is(':visible')) {
                    $batchBar.slideUp(200);
                }
            }
        },

        /**
         * Update redirect created count
         */
        updateRedirectCount: function() {
            var count = $('.sf-row-completed').length;
            $('#sf-redirects-created').text(count);
        },

        /**
         * Remove a row from the Dead Pages section and update the count
         */
        removeDeadPageRow: function($row) {
            var self = this;
            $row.fadeOut(300, function() {
                $(this).remove();
                // Update pagination after row removed
                self.pagination['dead-pages'].totalItems--;
                self.applyPagination('dead-pages');
                self.renderPageNumbers('dead-pages');
                self.updatePaginationInfo('dead-pages');
            });
            this.updateDeadPagesCount();
        },

        /**
         * Update the Dead Pages Needing Redirects header count
         */
        updateDeadPagesCount: function() {
            var visibleCount = $('.sf-backlink-row:visible').not('.sf-row-completed, .sf-row-failed').length;
            $('.sf-dead-pages-count').text(visibleCount);
        },

        /**
         * Add a row to the Redirects Applied section
         */
        addRedirectResultRow: function(result) {
            var self = this;
            var $section = $('#sf-redirects-applied-section');
            var $tbody = $section.find('.sf-redirects-table tbody');

            // Build status display
            var statusClass = 'sf-status-' + result.status;
            var statusText = result.status === 'success' ? 'Fixed' : (result.status === 'failed' ? 'Failed' : 'Skipped');
            if (result.message && result.status !== 'success') {
                statusText = result.message;
            }

            // Format date
            var dateDisplay = result.created_at || new Date().toLocaleString();

            // Build the row HTML
            var rowHtml = '<tr class="sf-redirect-result-row" data-status="' + result.status + '">' +
                '<td class="sf-col-dead-page">' +
                    '<a href="' + result.dead_page_url + '" target="_blank" class="sf-url-link">' + result.dead_page_url + '</a>' +
                '</td>' +
                '<td class="sf-col-redirect-url">' +
                    '<a href="' + result.redirect_url + '" target="_blank" class="sf-url-link">' + result.redirect_url + '</a>' +
                '</td>' +
                '<td class="sf-col-status">' +
                    '<span class="sf-status-badge ' + statusClass + '">' + statusText + '</span>' +
                '</td>' +
                '<td class="sf-col-plugin">' + (result.plugin || '-') + '</td>' +
                '<td class="sf-col-date">' + dateDisplay + '</td>' +
                '<td class="sf-col-backlinks">' + (result.backlink_count || 0) + '</td>' +
            '</tr>';

            // Prepend to table (newest first)
            $tbody.prepend(rowHtml);

            // Show section if hidden
            if (!$section.is(':visible')) {
                $section.show();
            }

            // Update status counts
            self.updateRedirectStatusCounts();

            // Update pagination
            self.pagination['redirects-applied'].totalItems++;
            self.resetPagination('redirects-applied');
        },

        /**
         * Update redirect status filter counts
         */
        updateRedirectStatusCounts: function() {
            var $rows = $('.sf-redirect-result-row');
            var allCount = $rows.length;
            var fixedCount = $rows.filter('[data-status="success"]').length;
            var failedCount = $rows.filter('[data-status="failed"]').length;
            var skippedCount = $rows.filter('[data-status="skipped"]').length;

            // Update section header count
            $('.sf-redirects-header-count').text(allCount);

            // Update filter button counts (new format without parentheses)
            $('.sf-redirect-status-count-all').text(allCount);
            $('.sf-redirect-status-count-success').text(fixedCount);
            $('.sf-redirect-status-count-failed').text(failedCount);
            $('.sf-redirect-status-count-skipped').text(skippedCount);

            // Update main stat card (shows successful redirects created)
            $('#sf-redirects-created').text(fixedCount);
        },

        /**
         * Filter redirect results by status
         */
        filterRedirectResults: function(status) {
            var $rows = $('.sf-redirect-result-row');

            // Use sf-filtered-out class for pagination compatibility
            if (status === 'all') {
                $rows.removeClass('sf-filtered-out');
            } else {
                $rows.addClass('sf-filtered-out');
                $rows.filter('[data-status="' + status + '"]').removeClass('sf-filtered-out');
            }

            // Update active filter button
            $('.sf-redirect-status-filters .sf-status-filter').removeClass('active');
            $('.sf-redirect-status-filters .sf-status-filter[data-status="' + status + '"]').addClass('active');

            // Reset pagination after filtering
            this.resetPagination('redirects-applied');
        },

        /**
         * Export redirect results to CSV
         */
        exportRedirects: function() {
            var self = this;
            var $btn = $('.sf-export-redirects-btn');
            var originalHtml = $btn.html();
            var domain = $('#sf-domain-input').val().trim();

            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> Exporting...');

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_export_redirects',
                    nonce: sfBacklinkData.nonce,
                    domain: domain
                },
                success: function(response) {
                    $btn.prop('disabled', false).html(originalHtml);

                    if (response.success) {
                        // Download the CSV
                        var blob = new Blob([response.data.csv], { type: 'text/csv' });
                        var url = window.URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = response.data.filename;
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                        window.URL.revokeObjectURL(url);

                        self.showNotice('success', 'Redirect results exported successfully!');
                    } else {
                        self.showNotice('error', response.data.message || 'Export failed.');
                    }
                },
                error: function() {
                    $btn.prop('disabled', false).html(originalHtml);
                    self.showNotice('error', 'Export failed.');
                }
            });
        },

        /**
         * Clear cached scan data
         */
        clearCache: function() {
            var self = this;
            var domain = $('#sf-domain-input').val().trim();

            console.log('Clear Cache: Sending domain:', domain);

            if (!confirm('Clear all cached scan data? You can then run a fresh scan.')) {
                return;
            }

            var $btn = $('#sf-clear-cache-btn');
            var originalHtml = $btn.html();

            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> Clearing...');

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_clear_cache',
                    nonce: sfBacklinkData.nonce,
                    domain: domain
                },
                success: function(response) {
                    console.log('Clear Cache Response:', response);
                    if (response.data && response.data.debug) {
                        console.log('Clear Cache Debug:', response.data.debug);
                    }
                    if (response.success) {
                        self.showNotice('success', 'Cache cleared! Reloading...');
                        // Pass the domain in URL so page knows which domain was cleared
                        var urlParams = new URLSearchParams(window.location.search);
                        urlParams.set('domain', domain);
                        urlParams.delete('scan_complete'); // Remove scan_complete flag
                        window.location.search = urlParams.toString();
                    } else {
                        $btn.prop('disabled', false).html(originalHtml);
                        self.showNotice('error', response.data.message || 'Failed to clear cache.');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Clear Cache Error:', status, error);
                    $btn.prop('disabled', false).html(originalHtml);
                    self.showNotice('error', 'Failed to clear cache.');
                }
            });
        },

        /**
         * Debug: Check credential state (call from console: BacklinkReclaim.debugCredentials())
         */
        debugCredentials: function() {
            var self = this;
            var domain = $('#sf-domain-input').val().trim();

            console.log('Checking credential and rate limit state...');

            $.ajax({
                url: sfBacklinkData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_backlink_debug_credentials',
                    nonce: sfBacklinkData.nonce,
                    domain: domain
                },
                success: function(response) {
                    if (response.success) {
                        var d = response.data;
                        console.log('=== Full Debug Info ===');
                        console.log('Domain:', d.domain);
                        console.log('--- Credentials ---');
                        console.log('has_credentials:', d.has_credentials);
                        console.log('user_has_api_key_method:', d.user_has_api_key_method);
                        console.log('has_user_credentials_method:', d.has_user_credentials_method);
                        console.log('Login:', d.login_preview, '(' + d.login_length + ' chars)');
                        console.log('Password exists:', d.password_exists, '(' + d.password_length + ' chars)');
                        console.log('Demo mode:', d.demo_mode);
                        console.log('--- Rate Limits ---');
                        console.log('can_scan_domain:', d.can_scan_domain);
                        console.log('time_until_scan:', d.time_until_scan, 'seconds');
                        console.log('can_use_free_scan:', d.can_use_free_scan);
                        console.log('time_since_last_free_scan:', d.time_since_last_free_scan, 'seconds');
                        console.log('--- Cache ---');
                        console.log('has_cached_scan:', d.has_cached_scan);
                        if (d.has_cached_scan) {
                            console.log('cached_scan_expires_at:', d.cached_scan_expires_at);
                            console.log('cached_scan_api_used:', d.cached_scan_api_used);
                        }
                        console.log('===========================');

                        // Show in UI
                        var msg = '=== CREDENTIALS ===\n' +
                                  'Has credentials: ' + (d.has_credentials ? 'YES' : 'NO') + '\n' +
                                  'Login: ' + d.login_preview + ' (' + d.login_length + ' chars)\n' +
                                  'Password: ' + (d.password_exists ? 'SET' : 'NOT SET') + '\n\n' +
                                  '=== RATE LIMITS ===\n' +
                                  'Domain: ' + d.domain + '\n' +
                                  'Can scan now: ' + (d.can_scan_domain ? 'YES' : 'NO') + '\n' +
                                  'Time until scan: ' + Math.round(d.time_until_scan / 3600) + ' hours\n' +
                                  'Can use free scan: ' + (d.can_use_free_scan ? 'YES' : 'NO') + '\n\n' +
                                  '=== CACHE ===\n' +
                                  'Has cached scan: ' + (d.has_cached_scan ? 'YES' : 'NO') + '\n' +
                                  (d.has_cached_scan ? 'Expires: ' + d.cached_scan_expires_at + '\nAPI used: ' + d.cached_scan_api_used : '');
                        alert(msg);
                    } else {
                        console.error('Debug failed:', response.data);
                        alert('Debug failed: ' + (response.data.message || 'Unknown error'));
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Debug AJAX error:', status, error);
                    alert('Debug AJAX error: ' + error);
                }
            });
        },

        /**
         * Show notice message
         */
        // =====================================
        // PAGINATION FUNCTIONS
        // =====================================

        initPagination: function() {
            var self = this;
            $('#sf-backlink-reclaim .sf-paginated-table').each(function() {
                var $wrapper = $(this);
                var section = $wrapper.data('section');
                var perPage = parseInt($wrapper.data('per-page'), 10) || 100;
                var total = parseInt($wrapper.data('total'), 10) || 0;

                if (self.pagination[section]) {
                    self.pagination[section].perPage = perPage;
                    self.pagination[section].totalItems = total;
                    self.pagination[section].currentPage = 1;

                    self.applyPagination(section);
                    self.renderPageNumbers(section);
                    self.updatePaginationInfo(section);
                }
            });
        },

        goToPage: function(section, page) {
            var pag = this.pagination[section];
            if (!pag) return;

            var totalPages = Math.ceil(this.getVisibleRowCount(section) / pag.perPage);
            page = Math.max(1, Math.min(page, totalPages));
            if (page === pag.currentPage) return;

            var scrollTop = $(window).scrollTop();
            pag.currentPage = page;
            this.applyPagination(section);
            this.renderPageNumbers(section);
            this.updatePaginationInfo(section);
            $(window).scrollTop(scrollTop);
        },

        applyPagination: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $wrapper = $('#sf-backlink-reclaim .sf-paginated-table[data-section="' + section + '"]');
            var $rows;

            if (section === 'dead-pages') {
                $rows = $wrapper.find('.sf-backlink-row');
            } else if (section === 'redirects-applied') {
                $rows = $wrapper.find('.sf-redirect-result-row');
            }

            if (!$rows || !$rows.length) return;

            var startIndex = (pag.currentPage - 1) * pag.perPage;
            var endIndex = startIndex + pag.perPage;

            var visibleIndex = 0;
            $rows.each(function() {
                var $row = $(this);
                var isFilteredOut = $row.hasClass('sf-filtered-out');

                if (isFilteredOut) {
                    $row.addClass('sf-page-hidden');
                } else {
                    if (visibleIndex >= startIndex && visibleIndex < endIndex) {
                        $row.removeClass('sf-page-hidden');
                    } else {
                        $row.addClass('sf-page-hidden');
                    }
                    visibleIndex++;
                }
            });

            var $pagination = $('#sf-backlink-reclaim .sf-pagination[data-section="' + section + '"]');
            var totalPages = Math.ceil(this.getVisibleRowCount(section) / pag.perPage);
            $pagination.find('.sf-page-prev').prop('disabled', pag.currentPage <= 1);
            $pagination.find('.sf-page-next').prop('disabled', pag.currentPage >= totalPages);
        },

        getVisibleRowCount: function(section) {
            var $wrapper = $('#sf-backlink-reclaim .sf-paginated-table[data-section="' + section + '"]');
            var $rows;

            if (section === 'dead-pages') {
                $rows = $wrapper.find('.sf-backlink-row:not(.sf-filtered-out)');
            } else if (section === 'redirects-applied') {
                $rows = $wrapper.find('.sf-redirect-result-row:not(.sf-filtered-out)');
            }

            return $rows ? $rows.length : 0;
        },

        renderPageNumbers: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('#sf-backlink-reclaim .sf-pagination[data-section="' + section + '"]');
            var $container = $pagination.find('.sf-page-numbers');

            var visibleCount = this.getVisibleRowCount(section);
            var totalPages = Math.ceil(visibleCount / pag.perPage);
            var currentPage = pag.currentPage;

            if (totalPages <= 1) {
                $container.empty();
                $pagination.find('.sf-page-prev, .sf-page-next').prop('disabled', true);
                return;
            }

            var html = '';
            var maxVisible = 7;

            if (totalPages <= maxVisible) {
                for (var i = 1; i <= totalPages; i++) {
                    html += this.renderPageButton(i, currentPage);
                }
            } else {
                html += this.renderPageButton(1, currentPage);
                if (currentPage > 3) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }
                var startPage = Math.max(2, currentPage - 1);
                var endPage = Math.min(totalPages - 1, currentPage + 1);
                if (currentPage <= 3) endPage = Math.min(4, totalPages - 1);
                if (currentPage >= totalPages - 2) startPage = Math.max(2, totalPages - 3);
                for (var j = startPage; j <= endPage; j++) {
                    html += this.renderPageButton(j, currentPage);
                }
                if (currentPage < totalPages - 2) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }
                html += this.renderPageButton(totalPages, currentPage);
            }

            $container.html(html);
        },

        renderPageButton: function(pageNum, currentPage) {
            var isActive = pageNum === currentPage;
            return '<button type="button" class="sf-page-number' + (isActive ? ' active' : '') + '" data-page="' + pageNum + '">' + pageNum + '</button>';
        },

        updatePaginationInfo: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('#sf-backlink-reclaim .sf-pagination[data-section="' + section + '"]');
            var visibleCount = this.getVisibleRowCount(section);

            var start = Math.min((pag.currentPage - 1) * pag.perPage + 1, visibleCount);
            var end = Math.min(pag.currentPage * pag.perPage, visibleCount);
            if (visibleCount === 0) { start = 0; end = 0; }

            $pagination.find('.sf-page-start').text(start);
            $pagination.find('.sf-page-end').text(end);
            $pagination.find('.sf-page-total').text(visibleCount);
        },

        resetPagination: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            pag.currentPage = 1;
            this.applyPagination(section);
            this.renderPageNumbers(section);
            this.updatePaginationInfo(section);
        },

        showNotice: function(type, message) {
            // Use the global ScreamingFixes toast if available
            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                if (type === 'success') {
                    ScreamingFixes.Toast.success(message);
                } else if (type === 'error') {
                    ScreamingFixes.Toast.error(message);
                } else if (type === 'warning') {
                    ScreamingFixes.Toast.warning(message);
                } else {
                    ScreamingFixes.Toast.info(message);
                }
                return;
            }

            // Fallback to alert
            alert(message);
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        if ($('#sf-backlink-reclaim').length) {
            BacklinkReclaim.init();
        }
    });

    // Expose BacklinkReclaim globally for debugging
    window.BacklinkReclaim = BacklinkReclaim;

})(jQuery);
