<?php
/**
 * Backlink Reclaim Module for Screaming Fixes
 *
 * Finds external backlinks pointing to 404 pages and creates redirects
 * to reclaim that link equity.
 *
 * Unlike other modules, this uses DataForSEO API instead of CSV upload.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Backlink_Reclaim {

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'backlink-reclaim';

    /**
     * Module name
     * @var string
     */
    protected $name;

    /**
     * Module slug
     * @var string
     */
    protected $slug = 'backlink-reclaim';

    /**
     * Cache duration in seconds (7 days)
     * Users can clear cache manually if they need fresh data sooner
     * @var int
     */
    const CACHE_DURATION = 604800; // 7 days

    /**
     * Disable caching entirely - set to true to troubleshoot API issues
     * When true, every scan hits the API and results are stored only for the session
     * @var bool
     */
    const DISABLE_CACHE = false;

    /**
     * Backlink API instance
     * @var SF_Backlink_API
     */
    private $api;

    /**
     * Redirect Manager instance
     * @var SF_Redirect_Manager
     */
    private $redirect_manager;

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Backlink Reclaim', 'screaming-fixes');
        $this->api = new SF_Backlink_API();
        $this->redirect_manager = new SF_Redirect_Manager();

        $this->init();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register AJAX handlers
        add_action('wp_ajax_sf_backlink_scan', [$this, 'ajax_scan_domain']);
        add_action('wp_ajax_sf_backlink_get_results', [$this, 'ajax_get_results']);
        add_action('wp_ajax_sf_backlink_create_redirect', [$this, 'ajax_create_redirect']);
        add_action('wp_ajax_sf_backlink_create_redirects_bulk', [$this, 'ajax_create_redirects_bulk']);
        add_action('wp_ajax_sf_backlink_suggest_destination', [$this, 'ajax_suggest_destination']);
        add_action('wp_ajax_sf_backlink_export', [$this, 'ajax_export']);
        add_action('wp_ajax_sf_backlink_export_redirects', [$this, 'ajax_export_redirects']);
        add_action('wp_ajax_sf_backlink_test_api', [$this, 'ajax_test_api']);
        add_action('wp_ajax_sf_backlink_clear_cache', [$this, 'ajax_clear_cache']);
        add_action('wp_ajax_sf_backlink_debug_credentials', [$this, 'ajax_debug_credentials']);

        // Enqueue module assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'backlinks') {
            return;
        }

        // Use file modification time for cache busting
        $css_file = SF_PLUGIN_DIR . 'modules/backlink-reclaim/assets/backlink-reclaim.css';
        $js_file = SF_PLUGIN_DIR . 'modules/backlink-reclaim/assets/backlink-reclaim.js';
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        wp_enqueue_style(
            'sf-backlink-reclaim',
            SF_PLUGIN_URL . 'modules/backlink-reclaim/assets/backlink-reclaim.css',
            ['screaming-fixes-admin'],
            $css_version
        );

        wp_enqueue_script(
            'sf-backlink-reclaim',
            SF_PLUGIN_URL . 'modules/backlink-reclaim/assets/backlink-reclaim.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        // Get site URL for default domain
        $site_url = wp_parse_url(home_url(), PHP_URL_HOST);

        // Check free scan availability
        $user_has_api_key = SF_Backlink_API::user_has_api_key();
        $last_free_scan = get_option('sf_last_free_scan', 0);
        $can_use_free_scan = (time() - $last_free_scan) > 86400;

        wp_localize_script('sf-backlink-reclaim', 'sfBacklinkData', [
            'nonce' => wp_create_nonce('sf_backlink_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'siteUrl' => home_url(),
            'siteDomain' => $site_url,
            'settingsUrl' => admin_url('admin.php?page=screaming-fixes&tab=settings'),
            'userHasApiKey' => $user_has_api_key,
            'canUseFreeScan' => $can_use_free_scan,
            'hasRedirectPlugin' => $this->redirect_manager->has_capability(),
            'activeRedirectPlugin' => $this->redirect_manager->get_capability_status()['active_plugin_name'] ?? '',
            'i18n' => [
                'scanning' => __('Scanning domain...', 'screaming-fixes'),
                'scanComplete' => __('Scan complete!', 'screaming-fixes'),
                'scanFailed' => __('Scan failed.', 'screaming-fixes'),
                'creatingRedirect' => __('Creating redirect...', 'screaming-fixes'),
                'redirectCreated' => __('Redirect created!', 'screaming-fixes'),
                'redirectFailed' => __('Failed to create redirect.', 'screaming-fixes'),
                'noDeadPages' => __('No dead pages with backlinks found.', 'screaming-fixes'),
                'rateLimited' => __('Scan limit reached. Try again later.', 'screaming-fixes'),
                'noCredentials' => __('No scan available. Add your DataForSEO API key in Settings.', 'screaming-fixes'),
                'noRedirectPlugin' => __('No redirect plugin detected.', 'screaming-fixes'),
                'confirmBulk' => __('Create redirects for %d pages?', 'screaming-fixes'),
                'suggesting' => __('AI searching for best match...', 'screaming-fixes'),
                'exporting' => __('Exporting...', 'screaming-fixes'),
                'enterDestination' => __('Enter destination URL', 'screaming-fixes'),
                'aiNoMatch' => __('AI could not find a suitable replacement page.', 'screaming-fixes'),
                'unlimitedScansTitle' => __('Want unlimited scans?', 'screaming-fixes'),
                'unlimitedScansMessage' => __('Connect your DataForSEO API in Settings for unlimited scans, or wait until tomorrow for another free scan.', 'screaming-fixes'),
                'connectApi' => __('Connect API', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Get the module's tab content view
     *
     * @return string Path to view file
     */
    public function get_view() {
        return SF_PLUGIN_DIR . 'modules/backlink-reclaim/views/tab-content.php';
    }

    /**
     * Check if domain can be scanned (rate limit check)
     *
     * @param string $domain
     * @return bool
     */
    public function can_scan($domain) {
        // When cache is disabled, always allow scanning
        if (self::DISABLE_CACHE) {
            return true;
        }

        $cached = $this->get_cached_scan($domain);

        if (!$cached) {
            return true;
        }

        // Check if cache has expired
        $expires_at = strtotime($cached['expires_at']);
        return time() > $expires_at;
    }

    /**
     * Get time until next scan is available
     *
     * @param string $domain
     * @return int Seconds until next scan, 0 if available now
     */
    public function get_time_until_scan($domain) {
        // When cache is disabled, always return 0 (can scan immediately)
        if (self::DISABLE_CACHE) {
            return 0;
        }

        $cached = $this->get_cached_scan($domain);

        if (!$cached) {
            return 0;
        }

        $expires_at = strtotime($cached['expires_at']);
        $remaining = $expires_at - time();

        return max(0, $remaining);
    }

    /**
     * Get cached scan results
     *
     * @param string $domain
     * @return array|null
     */
    public function get_cached_scan($domain) {
        global $wpdb;

        $original_domain = $domain;
        $domain = $this->clean_domain($domain);

        // When cache is disabled, use short-lived transient instead of database
        if (self::DISABLE_CACHE) {
            $transient_key = 'sf_backlink_scan_' . md5($domain);
            $cached = get_transient($transient_key);

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF Backlink Cache: CACHE DISABLED - Using transient for domain: ' . $domain);
                error_log('SF Backlink Cache: Transient found: ' . ($cached ? 'YES' : 'NO'));
            }

            if ($cached) {
                return $cached;
            }
            return null;
        }

        $table = $wpdb->prefix . 'screaming_fixes_backlink_cache';

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF Backlink Cache: Looking for domain: ' . $domain . ' (original: ' . $original_domain . ')');
        }

        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE domain = %s",
                $domain
            ),
            ARRAY_A
        );

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            if ($row) {
                error_log('SF Backlink Cache: Found cached scan, scanned_at: ' . ($row['scanned_at'] ?? 'unknown'));
            } else {
                error_log('SF Backlink Cache: No cache found for domain: ' . $domain);
                // Check what domains ARE in the cache
                $all_domains = $wpdb->get_col("SELECT domain FROM {$table}");
                error_log('SF Backlink Cache: Domains in cache: ' . implode(', ', $all_domains));
            }
        }

        if (!$row) {
            return null;
        }

        $row['scan_data'] = json_decode($row['scan_data'], true);

        return $row;
    }

    /**
     * Cache scan results
     *
     * @param string $domain
     * @param array $data
     * @return bool
     */
    public function cache_scan_results($domain, $data) {
        global $wpdb;

        $domain = $this->clean_domain($domain);

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF Backlink Cache: Saving scan for domain: ' . $domain);
            error_log('SF Backlink Cache: Data has ' . count($data['dead_pages'] ?? []) . ' dead pages');
            error_log('SF Backlink Cache: DISABLE_CACHE is ' . (self::DISABLE_CACHE ? 'TRUE' : 'FALSE'));
        }

        // When cache is disabled, use short-lived transient (5 minutes) instead of database
        // This allows users to see results in the same session without hitting API repeatedly
        if (self::DISABLE_CACHE) {
            $transient_key = 'sf_backlink_scan_' . md5($domain);
            $transient_data = [
                'domain' => $domain,
                'scan_data' => $data,
                'scanned_at' => current_time('mysql'),
                'expires_at' => date('Y-m-d H:i:s', time() + 300), // 5 minutes
            ];

            $result = set_transient($transient_key, $transient_data, 300); // 5 minute expiration

            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF Backlink Cache: Stored in transient (5 min expiry): ' . ($result ? 'SUCCESS' : 'FAILED'));
            }

            return $result;
        }

        // Normal database caching
        $table = $wpdb->prefix . 'screaming_fixes_backlink_cache';
        $expires_at = date('Y-m-d H:i:s', time() + self::CACHE_DURATION);

        // Use INSERT ... ON DUPLICATE KEY UPDATE
        $result = $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO {$table} (domain, scan_data, scanned_at, expires_at)
                 VALUES (%s, %s, NOW(), %s)
                 ON DUPLICATE KEY UPDATE
                 scan_data = VALUES(scan_data),
                 scanned_at = NOW(),
                 expires_at = VALUES(expires_at)",
                $domain,
                wp_json_encode($data),
                $expires_at
            )
        );

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            if ($result === false) {
                error_log('SF Backlink Cache: Database error: ' . $wpdb->last_error);
            } else {
                error_log('SF Backlink Cache: Save successful, affected rows: ' . $result);
            }
        }

        return $result !== false;
    }

    /**
     * Scan domain for broken backlinks
     *
     * @param string $domain
     * @param bool $use_free_tier Whether to use free tier API
     * @return array|WP_Error
     */
    public function scan_domain($domain, $use_free_tier = false) {
        $domain = $this->clean_domain($domain);

        // Check if user has their own API credentials (unlimited scans)
        $has_own_credentials = SF_Backlink_API::has_user_credentials();

        // Only check rate limit if user doesn't have their own API key
        if (!$has_own_credentials && !$this->can_scan($domain)) {
            $remaining = $this->get_time_until_scan($domain);
            return new WP_Error(
                'rate_limited',
                sprintf(
                    __('Scan limit reached. You can scan this domain again in %s.', 'screaming-fixes'),
                    $this->format_time_remaining($remaining)
                ),
                ['remaining_seconds' => $remaining]
            );
        }

        // Use appropriate API instance
        $api = new SF_Backlink_API($use_free_tier);

        // Check if we have credentials
        if (!$api->has_credentials()) {
            return new WP_Error(
                'no_credentials',
                __('No API credentials available. Please add your DataForSEO API key in Settings.', 'screaming-fixes')
            );
        }

        // Perform the scan
        $results = $api->get_broken_backlinks($domain);

        // Get which API was used for debugging
        $api_used = $api->get_last_api_used();

        if (is_wp_error($results)) {
            return $results;
        }

        // Get debug info from the scan
        $scan_debug = $api->get_last_scan_debug();

        // Prepare data for caching
        $scan_data = [
            'domain' => $domain,
            'dead_pages' => $results,
            'total_dead_pages' => count($results),
            'total_backlinks' => array_sum(array_column($results, 'backlink_count')),
            'scanned_at' => current_time('mysql'),
            'api_used' => $api_used, // Track which API was used
            'debug' => $scan_debug, // Include debug info for troubleshooting
        ];

        // Cache the results
        $this->cache_scan_results($domain, $scan_data);

        return $scan_data;
    }

    /**
     * Create a redirect
     *
     * @param string $source Source URL path
     * @param string $destination Destination URL
     * @param int $status_code HTTP status code (default 301)
     * @param array $metadata Optional metadata (backlink_count, etc.)
     * @return array|WP_Error
     */
    public function create_redirect($source, $destination, $status_code = 301, $metadata = []) {
        if (!$this->redirect_manager->has_capability()) {
            return new WP_Error(
                'no_redirect_plugin',
                __('No redirect plugin available. Install Rank Math, Yoast Premium, or Redirection.', 'screaming-fixes')
            );
        }

        // Clean the source path
        $source_path = wp_parse_url($source, PHP_URL_PATH);
        if (!$source_path) {
            $source_path = '/' . ltrim($source, '/');
        }

        // Validate destination
        if (empty($destination)) {
            return new WP_Error('invalid_destination', __('Destination URL is required.', 'screaming-fixes'));
        }

        // Create the redirect
        $result = $this->redirect_manager->create_redirect($source_path, $destination, $status_code);

        if (is_wp_error($result)) {
            return $result;
        }

        // Get the active redirect plugin name
        $plugin_status = $this->redirect_manager->get_capability_status();
        $plugin_name = $plugin_status['active_plugin_name'] ?? __('Unknown', 'screaming-fixes');

        // Log the change
        $logger = new SF_Change_Logger();
        $logger->log_change(0, 'redirect', $source_path, $destination, [
            'module' => 'backlink-reclaim',
            'source' => $source_path,
            'status_code' => $status_code,
            'plugin' => $plugin_name,
        ]);

        return [
            'success' => true,
            'source' => $source_path,
            'destination' => $destination,
            'status_code' => $status_code,
            'plugin' => $plugin_name,
            'created_at' => current_time('mysql'),
            'backlink_count' => isset($metadata['backlink_count']) ? (int) $metadata['backlink_count'] : 0,
        ];
    }

    /**
     * Store redirect result in cached scan data
     *
     * @param string $domain Domain the scan is for
     * @param array $redirect_result Redirect result data
     */
    public function store_redirect_result($domain, $redirect_result) {
        $cached = $this->get_cached_scan($domain);

        if (!$cached || empty($cached['scan_data'])) {
            return false;
        }

        $scan_data = $cached['scan_data'];

        // Initialize redirect_results array if not exists
        if (!isset($scan_data['redirect_results'])) {
            $scan_data['redirect_results'] = [];
        }

        // Add the new redirect result
        $scan_data['redirect_results'][] = $redirect_result;

        // Save back to cache
        return $this->cache_scan_results($domain, $scan_data);
    }

    /**
     * Get redirect results from cached scan data
     *
     * @param string $domain Domain to get results for
     * @return array Redirect results
     */
    public function get_redirect_results($domain) {
        $cached = $this->get_cached_scan($domain);

        if (!$cached || empty($cached['scan_data'])) {
            return [];
        }

        return $cached['scan_data']['redirect_results'] ?? [];
    }

    /**
     * AJAX: Scan domain
     */
    public function ajax_scan_domain() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($domain)) {
            wp_send_json_error(['message' => __('Domain is required.', 'screaming-fixes')]);
        }

        // Determine if we should use free tier
        $user_has_api_key = SF_Backlink_API::user_has_api_key();
        $use_free_tier = false;

        // User has their own API key - skip rate limit check entirely
        if ($user_has_api_key) {
            // Direct to scan with no rate limit
            $result = $this->scan_domain($domain, false);

            if (is_wp_error($result)) {
                wp_send_json_error([
                    'message' => $result->get_error_message(),
                    'code' => $result->get_error_code(),
                    'data' => $result->get_error_data(),
                ]);
            }

            wp_send_json_success($result);
            return; // Important: exit after sending response
        }

        // No API key - check free scan availability
        $last_free_scan = get_option('sf_last_free_scan', 0);
        $can_use_free_scan = (time() - $last_free_scan) > 86400;

        if (!$can_use_free_scan) {
            $hours_remaining = ceil((86400 - (time() - $last_free_scan)) / 3600);
            wp_send_json_error([
                'message' => sprintf(
                    __('Free scan limit reached. Next scan available in %d hours. Add your own DataForSEO API key for unlimited scans.', 'screaming-fixes'),
                    $hours_remaining
                ),
                'code' => 'rate_limited',
            ]);
        }

        $use_free_tier = true;
        $result = $this->scan_domain($domain, $use_free_tier);

        if (is_wp_error($result)) {
            wp_send_json_error([
                'message' => $result->get_error_message(),
                'code' => $result->get_error_code(),
                'data' => $result->get_error_data(),
            ]);
        }

        // Track free scan usage
        if ($use_free_tier) {
            update_option('sf_last_free_scan', time());
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get cached results
     */
    public function ajax_get_results() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($domain)) {
            wp_send_json_error(['message' => __('Domain is required.', 'screaming-fixes')]);
        }

        $cached = $this->get_cached_scan($domain);

        if (!$cached) {
            wp_send_json_success([
                'has_data' => false,
                'can_scan' => true,
            ]);
        }

        $can_scan = $this->can_scan($domain);
        $time_until_scan = $this->get_time_until_scan($domain);

        wp_send_json_success([
            'has_data' => true,
            'can_scan' => $can_scan,
            'time_until_scan' => $time_until_scan,
            'time_until_scan_formatted' => $this->format_time_remaining($time_until_scan),
            'next_scan_at' => date('Y-m-d H:i:s', time() + $time_until_scan),
            'scan_data' => $cached['scan_data'],
            'scanned_at' => $cached['scanned_at'],
        ]);
    }

    /**
     * AJAX: Create single redirect
     */
    public function ajax_create_redirect() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : '';
        $destination = isset($_POST['destination']) ? esc_url_raw($_POST['destination']) : '';
        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';
        $backlink_count = isset($_POST['backlink_count']) ? (int) $_POST['backlink_count'] : 0;

        if (empty($source) || empty($destination)) {
            wp_send_json_error(['message' => __('Source and destination are required.', 'screaming-fixes')]);
        }

        $metadata = [
            'backlink_count' => $backlink_count,
        ];

        $result = $this->create_redirect($source, $destination, 301, $metadata);

        if (is_wp_error($result)) {
            // Store failed redirect result
            $redirect_result = [
                'dead_page_url' => $source,
                'redirect_url' => $destination,
                'status' => 'failed',
                'status_message' => $result->get_error_message(),
                'plugin' => '',
                'created_at' => current_time('mysql'),
                'backlink_count' => $backlink_count,
            ];

            if (!empty($domain)) {
                $this->store_redirect_result($domain, $redirect_result);
            }

            wp_send_json_error([
                'message' => $result->get_error_message(),
                'redirect_result' => $redirect_result,
            ]);
        }

        // Store successful redirect result
        $redirect_result = [
            'dead_page_url' => $source,
            'redirect_url' => $result['destination'],
            'status' => 'success',
            'status_message' => __('Redirect created', 'screaming-fixes'),
            'plugin' => $result['plugin'],
            'created_at' => $result['created_at'],
            'backlink_count' => $result['backlink_count'],
        ];

        if (!empty($domain)) {
            $this->store_redirect_result($domain, $redirect_result);
        }

        // Log to activity log for dashboard
        SF_Activity_Log::log('backlink-reclaim', 1);

        wp_send_json_success([
            'success' => true,
            'source' => $result['source'],
            'destination' => $result['destination'],
            'status_code' => $result['status_code'],
            'plugin' => $result['plugin'],
            'created_at' => $result['created_at'],
            'redirect_result' => $redirect_result,
        ]);
    }

    /**
     * AJAX: Create redirects in bulk
     */
    public function ajax_create_redirects_bulk() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $redirects = isset($_POST['redirects']) ? $_POST['redirects'] : [];
        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        if (empty($redirects) || !is_array($redirects)) {
            wp_send_json_error(['message' => __('No redirects provided.', 'screaming-fixes')]);
        }

        $success = 0;
        $failed = 0;
        $errors = [];
        $redirect_results = [];

        foreach ($redirects as $redirect) {
            $source = sanitize_text_field($redirect['source'] ?? '');
            $destination = esc_url_raw($redirect['destination'] ?? '');
            $backlink_count = isset($redirect['backlink_count']) ? (int) $redirect['backlink_count'] : 0;

            if (empty($source) || empty($destination)) {
                $failed++;
                $redirect_result = [
                    'dead_page_url' => $source,
                    'redirect_url' => $destination,
                    'status' => 'skipped',
                    'status_message' => __('Missing source or destination URL', 'screaming-fixes'),
                    'plugin' => '',
                    'created_at' => current_time('mysql'),
                    'backlink_count' => $backlink_count,
                ];
                $redirect_results[] = $redirect_result;
                if (!empty($domain)) {
                    $this->store_redirect_result($domain, $redirect_result);
                }
                continue;
            }

            $metadata = [
                'backlink_count' => $backlink_count,
            ];

            $result = $this->create_redirect($source, $destination, 301, $metadata);

            if (is_wp_error($result)) {
                $failed++;
                $errors[] = [
                    'source' => $source,
                    'message' => $result->get_error_message(),
                ];
                $redirect_result = [
                    'dead_page_url' => $source,
                    'redirect_url' => $destination,
                    'status' => 'failed',
                    'status_message' => $result->get_error_message(),
                    'plugin' => '',
                    'created_at' => current_time('mysql'),
                    'backlink_count' => $backlink_count,
                ];
            } else {
                $success++;
                $redirect_result = [
                    'dead_page_url' => $source,
                    'redirect_url' => $result['destination'],
                    'status' => 'success',
                    'status_message' => __('Redirect created', 'screaming-fixes'),
                    'plugin' => $result['plugin'],
                    'created_at' => $result['created_at'],
                    'backlink_count' => $result['backlink_count'],
                ];
            }

            $redirect_results[] = $redirect_result;
            if (!empty($domain)) {
                $this->store_redirect_result($domain, $redirect_result);
            }
        }

        // Log to activity log for dashboard
        if ($success > 0) {
            SF_Activity_Log::log('backlink-reclaim', $success);
        }

        wp_send_json_success([
            'success' => $success,
            'failed' => $failed,
            'errors' => $errors,
            'redirect_results' => $redirect_results,
            'message' => sprintf(
                __('Created %d redirects. %d failed.', 'screaming-fixes'),
                $success,
                $failed
            ),
        ]);
    }

    /**
     * AJAX: Get AI suggested destination
     */
    public function ajax_suggest_destination() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $dead_page = isset($_POST['dead_page']) ? sanitize_text_field($_POST['dead_page']) : '';
        $scanned_domain = isset($_POST['scanned_domain']) ? sanitize_text_field($_POST['scanned_domain']) : '';
        $referrers_json = isset($_POST['referrers']) ? wp_unslash($_POST['referrers']) : '[]';

        if (empty($dead_page)) {
            wp_send_json_error(['message' => __('Dead page URL is required.', 'screaming-fixes')]);
        }

        // If no scanned domain provided, extract from dead_page URL
        if (empty($scanned_domain)) {
            $scanned_domain = wp_parse_url($dead_page, PHP_URL_HOST);
            if (empty($scanned_domain)) {
                $scanned_domain = wp_parse_url(home_url(), PHP_URL_HOST);
            }
        }

        // Parse referrers JSON
        $referrers = json_decode($referrers_json, true);
        if (!is_array($referrers)) {
            $referrers = array();
        }

        // Sanitize referrers
        $sanitized_referrers = array();
        foreach ($referrers as $ref) {
            if (isset($ref['domain'])) {
                $sanitized_referrers[] = array(
                    'domain' => sanitize_text_field($ref['domain']),
                    'url' => isset($ref['url']) ? esc_url_raw($ref['url']) : '',
                );
            }
        }

        $result = $this->api->get_suggested_destination($dead_page, $scanned_domain, $sanitized_referrers);

        // Determine if this is a homepage fallback
        $site_url = 'https://' . preg_replace('#^https?://#', '', $scanned_domain);
        $is_homepage = isset($result['destination']) && $result['destination'] === trailingslashit($site_url);

        wp_send_json_success([
            'destination' => isset($result['destination']) ? $result['destination'] : trailingslashit($site_url),
            'source' => isset($result['source']) ? $result['source'] : 'unknown',
            'confidence' => isset($result['confidence']) ? $result['confidence'] : null,
            'explanation' => isset($result['explanation']) ? $result['explanation'] : '',
            'verified' => isset($result['verified']) ? $result['verified'] : null,
            'is_homepage' => $is_homepage,
            'is_ai' => isset($result['source']) && $result['source'] === 'ai',
            'is_ai_no_match' => isset($result['source']) && $result['source'] === 'ai_no_match',
        ]);
    }

    /**
     * AJAX: Export results
     */
    public function ajax_export() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        $cached = $this->get_cached_scan($domain);

        if (!$cached || empty($cached['scan_data']['dead_pages'])) {
            wp_send_json_error(['message' => __('No data to export.', 'screaming-fixes')]);
        }

        $dead_pages = $cached['scan_data']['dead_pages'];

        // Build flat CSV with one row per referring URL per dead page
        $csv = "Dead Page URL,Status Code,Total Backlinks,Referring Domains,Referring URL,Referring Domain,Domain Rank\n";

        $row_count = 0;
        $max_rows = 1000;

        foreach ($dead_pages as $page) {
            if ($row_count >= $max_rows) {
                break;
            }

            $dead_page_url = isset($page['dead_page']) ? $page['dead_page'] : '';
            $status_code = isset($page['status_code']) ? $page['status_code'] : 404;
            $backlink_count = isset($page['backlink_count']) ? $page['backlink_count'] : 0;
            $referring_domains = isset($page['referring_domains']) ? $page['referring_domains'] : 0;

            // Use all_referrers if available (from direct API), fall back to top_referrers
            $referrers = !empty($page['all_referrers']) ? $page['all_referrers'] : (!empty($page['top_referrers']) ? $page['top_referrers'] : array());

            if (empty($referrers)) {
                // Still output the dead page even with no referrer details
                $csv .= sprintf(
                    '"%s",%d,%d,%d,"","",0' . "\n",
                    str_replace('"', '""', $dead_page_url),
                    $status_code,
                    $backlink_count,
                    $referring_domains
                );
                $row_count++;
                continue;
            }

            foreach ($referrers as $referrer) {
                if ($row_count >= $max_rows) {
                    break;
                }

                $ref_url = isset($referrer['url']) ? $referrer['url'] : '';
                $ref_domain = isset($referrer['domain']) ? $referrer['domain'] : '';
                $ref_rank = isset($referrer['domain_rank']) ? (int) $referrer['domain_rank'] : 0;

                $csv .= sprintf(
                    '"%s",%d,%d,%d,"%s","%s",%d' . "\n",
                    str_replace('"', '""', $dead_page_url),
                    $status_code,
                    $backlink_count,
                    $referring_domains,
                    str_replace('"', '""', $ref_url),
                    str_replace('"', '""', $ref_domain),
                    $ref_rank
                );
                $row_count++;
            }
        }

        wp_send_json_success([
            'csv' => $csv,
            'filename' => 'backlink-reclaim-' . $domain . '-' . gmdate('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Export redirect results as CSV
     */
    public function ajax_export_redirects() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';

        $redirect_results = $this->get_redirect_results($domain);

        if (empty($redirect_results)) {
            wp_send_json_error(['message' => __('No redirect results to export.', 'screaming-fixes')]);
        }

        // Build CSV with columns: Dead Page URL, New 301 Redirect URL, Status, Redirect Plugin Used, Date Applied, Number of Backlinks
        $csv = "Dead Page URL,New 301 Redirect URL,Status,Redirect Plugin Used,Date Applied,Number of Backlinks\n";

        foreach ($redirect_results as $result) {
            $dead_page = isset($result['dead_page_url']) ? $result['dead_page_url'] : '';
            $redirect_url = isset($result['redirect_url']) ? $result['redirect_url'] : '';
            $status = isset($result['status']) ? ucfirst($result['status']) : '';
            $status_message = isset($result['status_message']) ? $result['status_message'] : '';
            $plugin = isset($result['plugin']) ? $result['plugin'] : '';
            $created_at = isset($result['created_at']) ? $result['created_at'] : '';
            $backlink_count = isset($result['backlink_count']) ? (int) $result['backlink_count'] : 0;

            // Combine status with message for more context
            $status_display = $status;
            if ($status === 'Failed' && !empty($status_message)) {
                $status_display = 'Failed - ' . $status_message;
            } elseif ($status === 'Success') {
                $status_display = 'Fixed';
            } elseif ($status === 'Skipped' && !empty($status_message)) {
                $status_display = 'Skipped - ' . $status_message;
            }

            $csv .= sprintf(
                '"%s","%s","%s","%s","%s",%d' . "\n",
                str_replace('"', '""', $dead_page),
                str_replace('"', '""', $redirect_url),
                str_replace('"', '""', $status_display),
                str_replace('"', '""', $plugin),
                str_replace('"', '""', $created_at),
                $backlink_count
            );
        }

        wp_send_json_success([
            'csv' => $csv,
            'filename' => 'redirect-results-' . $domain . '-' . date('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Test API connection
     */
    public function ajax_test_api() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $result = $this->api->test_connection();

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Clear cached scan data
     */
    public function ajax_clear_cache() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Get info about what's in cache before clearing
        global $wpdb;
        $table = $wpdb->prefix . 'screaming_fixes_backlink_cache';
        $cached_before = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");

        // Clear ALL cached scans to ensure a fresh start
        $this->clear_cache('');

        // Check if cache was actually cleared
        $cached_after = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");

        wp_send_json_success([
            'message' => __('All cached scan data cleared.', 'screaming-fixes'),
            'debug' => [
                'cached_before' => (int) $cached_before,
                'cached_after' => (int) $cached_after,
            ],
        ]);
    }

    /**
     * AJAX: Debug credentials state
     */
    public function ajax_debug_credentials() {
        check_ajax_referer('sf_backlink_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $domain = isset($_POST['domain']) ? sanitize_text_field($_POST['domain']) : '';
        if (empty($domain)) {
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        }
        $domain = $this->clean_domain($domain);

        // Get all debug info
        $debug_info = SF_Backlink_API::get_credential_debug_info();

        // Add rate limit info
        $debug_info['domain'] = $domain;
        $debug_info['user_has_api_key_method'] = SF_Backlink_API::user_has_api_key();
        $debug_info['has_user_credentials_method'] = SF_Backlink_API::has_user_credentials();
        $debug_info['can_scan_domain'] = $this->can_scan($domain);
        $debug_info['time_until_scan'] = $this->get_time_until_scan($domain);
        $debug_info['last_free_scan'] = get_option('sf_last_free_scan', 0);
        $debug_info['time_since_last_free_scan'] = time() - get_option('sf_last_free_scan', 0);
        $debug_info['can_use_free_scan'] = (time() - get_option('sf_last_free_scan', 0)) > 86400;

        // Check cached scan
        $cached = $this->get_cached_scan($domain);
        $debug_info['has_cached_scan'] = !empty($cached);
        if ($cached) {
            $debug_info['cached_scan_expires_at'] = $cached['expires_at'];
            $debug_info['cached_scan_api_used'] = isset($cached['scan_data']['api_used']) ? $cached['scan_data']['api_used'] : 'unknown';
        }

        wp_send_json_success($debug_info);
    }

    /**
     * Clear cached scan data
     *
     * @param string $domain Optional domain to clear, or empty to clear all
     */
    public function clear_cache($domain = '') {
        global $wpdb;

        $table = $wpdb->prefix . 'screaming_fixes_backlink_cache';

        if (!empty($domain)) {
            $domain = $this->clean_domain($domain);
            $wpdb->delete($table, ['domain' => $domain]);
            // Also clear transient if using disabled cache mode
            delete_transient('sf_backlink_scan_' . md5($domain));
        } else {
            $wpdb->query("TRUNCATE TABLE {$table}");
            // Clear all backlink scan transients - query for transient keys
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_sf_backlink_scan_%' OR option_name LIKE '_transient_timeout_sf_backlink_scan_%'");
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF Backlink Cache: Cache cleared for domain: ' . ($domain ?: 'ALL'));
        }
    }

    /**
     * Clean domain input
     *
     * @param string $domain
     * @return string
     */
    private function clean_domain($domain) {
        $domain = strtolower(trim($domain));
        $domain = preg_replace('#^https?://#', '', $domain);
        $domain = preg_replace('#^www\.#', '', $domain);
        $domain = rtrim($domain, '/');
        return $domain;
    }

    /**
     * Format remaining time in human readable format
     *
     * @param int $seconds
     * @return string
     */
    private function format_time_remaining($seconds) {
        if ($seconds <= 0) {
            return __('now', 'screaming-fixes');
        }

        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);

        if ($hours > 0) {
            return sprintf(
                _n('%d hour', '%d hours', $hours, 'screaming-fixes'),
                $hours
            );
        }

        return sprintf(
            _n('%d minute', '%d minutes', $minutes, 'screaming-fixes'),
            $minutes
        );
    }
}
