<?php
/**
 * Backlink Reclaim Tab Content
 *
 * Main view for the Backlink Reclaim module with multiple states:
 * - Free scan available
 * - User has own API key (unlimited scans)
 * - Rate limited (free scan used)
 * - Results found
 */

if (!defined('ABSPATH')) {
    exit;
}

// Initialize module and get status
$backlink_api = new SF_Backlink_API();
$redirect_manager = new SF_Redirect_Manager();

$redirect_status = $redirect_manager->get_capability_status();
$has_redirect_plugin = $redirect_manager->has_capability();

// Get site domain (pre-fill input)
$site_url = home_url();
$site_domain = wp_parse_url($site_url, PHP_URL_HOST);
// Remove www. prefix for cleaner default
$site_domain = preg_replace('/^www\./', '', $site_domain);

// Check if user has their own API key
$user_has_api_key = !empty(get_option('sf_dataforseo_login')) && !empty(get_option('sf_dataforseo_password'));

// Check if demo mode is enabled
$is_demo_mode = SF_Backlink_API::is_demo_mode();

// Check free scan availability (1 per day per site)
$last_free_scan = get_option('sf_last_free_scan', 0);
$can_use_free_scan = (time() - $last_free_scan) > 86400; // 24 hours
$hours_remaining = $can_use_free_scan ? 0 : ceil((86400 - (time() - $last_free_scan)) / 3600);

// Determine which domain to use
// Priority: URL parameter > site domain
$backlink_module = new SF_Backlink_Reclaim();
$current_domain = $site_domain;
$display_domain = $site_domain; // Domain shown in input field

// Check if a specific domain was passed in the URL (e.g., after scan redirect)
if (isset($_GET['domain']) && !empty($_GET['domain'])) {
    $current_domain = sanitize_text_field($_GET['domain']);
    $current_domain = preg_replace('#^https?://#', '', $current_domain);
    $current_domain = preg_replace('#^www\.#', '', $current_domain);
    $current_domain = rtrim($current_domain, '/');
    $display_domain = $current_domain;
}

// Try to get cached scan for the current domain only
// No fallback to other domains - this prevents confusion
$cached_scan = $backlink_module->get_cached_scan($current_domain);


// Check if we have any scan data (even if empty results)
$has_scan_data = !empty($cached_scan) && isset($cached_scan['scan_data']);
$has_data = $has_scan_data && !empty($cached_scan['scan_data']['dead_pages']);
$has_empty_results = $has_scan_data && empty($cached_scan['scan_data']['dead_pages']);
$can_scan = $user_has_api_key || $can_use_free_scan;
$time_until_scan = $backlink_module->get_time_until_scan($current_domain);

// Get scan data if available
$scan_data = $has_data ? $cached_scan['scan_data'] : ($has_empty_results ? $cached_scan['scan_data'] : null);
$dead_pages = $has_data ? $scan_data['dead_pages'] : [];
$total_dead_pages = $has_data ? count($dead_pages) : 0;
$total_backlinks = $has_data ? array_sum(array_column($dead_pages, 'backlink_count')) : 0;
$scanned_at = $has_scan_data ? $cached_scan['scanned_at'] : null;
$scan_debug = $has_scan_data && isset($cached_scan['scan_data']['debug']) ? $cached_scan['scan_data']['debug'] : null;

// Get redirect results if available
$redirect_results = $has_scan_data && isset($cached_scan['scan_data']['redirect_results']) ? $cached_scan['scan_data']['redirect_results'] : [];
$redirect_results_count = count($redirect_results);
$successful_redirects = array_filter($redirect_results, function($r) { return $r['status'] === 'success'; });
$failed_redirects = array_filter($redirect_results, function($r) { return $r['status'] === 'failed'; });
$skipped_redirects = array_filter($redirect_results, function($r) { return $r['status'] === 'skipped'; });
?>

<!-- Progress Modal for Bulk Operations -->
<div class="sf-progress-modal-overlay" id="sf-progress-modal" style="display: none;">
    <div class="sf-progress-modal">
        <div class="sf-progress-modal-header">
            <span class="sf-progress-icon">&#128279;</span>
            <h3 class="sf-progress-title"><?php esc_html_e('Creating Redirects...', 'screaming-fixes'); ?></h3>
        </div>
        <div class="sf-progress-modal-content">
            <div class="sf-progress-bar-wrapper">
                <div class="sf-progress-bar">
                    <div class="sf-progress-bar-fill" style="width: 0%;"></div>
                </div>
                <span class="sf-progress-percent">0%</span>
            </div>
            <div class="sf-progress-details">
                <p class="sf-progress-current"><?php esc_html_e('Processing:', 'screaming-fixes'); ?> <span class="sf-current-url">-</span></p>
                <p class="sf-progress-count"><span class="sf-done-count">0</span> <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-total-count">0</span> <?php esc_html_e('redirects created', 'screaming-fixes'); ?></p>
                <p class="sf-progress-failed" style="display: none;"><span class="sf-failed-count">0</span> <?php esc_html_e('failed', 'screaming-fixes'); ?></p>
            </div>
        </div>
        <div class="sf-progress-modal-footer">
            <button type="button" class="sf-button sf-button-secondary" id="sf-progress-cancel">
                <?php esc_html_e('Cancel', 'screaming-fixes'); ?>
            </button>
        </div>
    </div>
</div>

<div class="sf-backlink-reclaim-module" id="sf-backlink-reclaim">

    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#128281;</span>
            <h2><?php esc_html_e('Backlink Reclaim', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Recover lost backlinks by redirecting dead pages', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-module-stats">
            <?php if ($has_data): ?>
                <span class="sf-stat sf-stat-total">
                    <strong><?php echo esc_html($total_dead_pages); ?></strong>
                    <?php esc_html_e('dead pages with backlinks', 'screaming-fixes'); ?>
                </span>
            <?php endif; ?>
            <?php if ($has_redirect_plugin): ?>
                <span class="sf-seo-plugin-badge">
                    <?php printf(esc_html__('Using %s', 'screaming-fixes'), esc_html($redirect_status['active_plugin_name'])); ?>
                </span>
            <?php else: ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=screaming-fixes&tab=settings')); ?>" class="sf-seo-plugin-badge sf-plugin-badge-warning">
                    <?php esc_html_e('No Redirect Plugin', 'screaming-fixes'); ?>
                </a>
            <?php endif; ?>
        </div>
    </div>

    <?php if ($is_demo_mode): ?>
    <!-- Demo Mode Banner -->
    <div class="sf-demo-banner">
        <span class="sf-demo-icon">&#9888;</span>
        <div class="sf-demo-content">
            <strong><?php esc_html_e('Demo Mode Active', 'screaming-fixes'); ?></strong>
            <span><?php esc_html_e('Displaying sample data for testing. Set DEMO_MODE to false in class-backlink-api.php for production use.', 'screaming-fixes'); ?></span>
        </div>
    </div>
    <?php endif; ?>

    <?php if (SF_Backlink_Reclaim::DISABLE_CACHE): ?>
    <!-- Cache Disabled Banner -->
    <div class="sf-demo-banner" style="background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%); border-color: #f59e0b;">
        <span class="sf-demo-icon">&#128270;</span>
        <div class="sf-demo-content">
            <strong><?php esc_html_e('Debug Mode: Cache Disabled', 'screaming-fixes'); ?></strong>
            <span><?php esc_html_e('Each scan hits the API directly. Results stored for 5 minutes only. Check your PHP error log for detailed API responses. Set DISABLE_CACHE to false in class-backlink-reclaim.php for production.', 'screaming-fixes'); ?></span>
        </div>
    </div>
    <?php endif; ?>

    <!-- Instructions Section (Collapsible) -->
    <?php include SF_PLUGIN_DIR . 'modules/backlink-reclaim/views/instructions.php'; ?>

    <!-- Domain Scan Section -->
    <div class="sf-scan-section">
        <h3>
            <span class="sf-section-icon">&#127760;</span>
            <?php esc_html_e('Scan Your Domain', 'screaming-fixes'); ?>
        </h3>

        <div class="sf-domain-input-row">
            <input
                type="text"
                id="sf-domain-input"
                class="sf-domain-input"
                value="<?php echo esc_attr($display_domain); ?>"
                placeholder="example.com"
            />
            <button
                type="button"
                id="sf-scan-btn"
                class="button button-primary sf-scan-button"
                <?php echo (!$can_scan && !$has_data) ? 'disabled' : ''; ?>
            >
                <span class="dashicons dashicons-search"></span>
                <?php esc_html_e('Scan for Dead Backlinks', 'screaming-fixes'); ?>
            </button>
            <?php if ($has_scan_data): ?>
            <button
                type="button"
                id="sf-clear-cache-btn"
                class="button sf-clear-cache-button"
                title="<?php esc_attr_e('Clear cached data and rescan with live API', 'screaming-fixes'); ?>"
            >
                <span class="dashicons dashicons-trash"></span>
                <?php esc_html_e('Clear Cache', 'screaming-fixes'); ?>
            </button>
            <?php endif; ?>
        </div>

        <?php if ($user_has_api_key): ?>
            <p class="sf-scan-note sf-scan-note-unlimited">
                <span class="sf-note-icon">&#10003;</span>
                <?php esc_html_e('Using your DataForSEO API key. Unlimited scans available.', 'screaming-fixes'); ?>
            </p>
        <?php endif; ?>
    </div>

    <!-- Scanning Progress (Hidden by default) -->
    <div class="sf-scanning-section" id="sf-scanning-section" style="display: none;">
        <div class="sf-scanning-content">
            <div class="sf-spinner-large"></div>
            <h3 id="sf-scanning-status"><?php esc_html_e('Scanning domain...', 'screaming-fixes'); ?></h3>
            <p class="sf-scanning-note"><?php esc_html_e('Finding backlinks to 404 pages. This may take a minute.', 'screaming-fixes'); ?></p>
        </div>
    </div>

    <?php if ($has_data): ?>

    <!-- Stats Summary -->
    <div class="sf-stats-summary">
        <div class="sf-stat-card">
            <span class="sf-stat-number"><?php echo esc_html($total_dead_pages); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Dead Pages', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-stat-card sf-stat-backlinks">
            <span class="sf-stat-number"><?php echo esc_html(number_format($total_backlinks)); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Total Backlinks', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-stat-card sf-stat-reclaimed">
            <span class="sf-stat-number" id="sf-redirects-created"><?php echo esc_html(count($successful_redirects)); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Redirects Created', 'screaming-fixes'); ?></span>
        </div>
    </div>

    <!-- Scan Info -->
    <div class="sf-scan-info">
        <span class="dashicons dashicons-calendar-alt"></span>
        <?php
        printf(
            esc_html__('Last scanned: %s', 'screaming-fixes'),
            '<strong>' . esc_html(human_time_diff(strtotime($scanned_at), current_time('timestamp'))) . ' ' . esc_html__('ago', 'screaming-fixes') . '</strong>'
        );
        ?>
    </div>

    <!-- Dead Pages Needing Redirects Section (Collapsible) -->
    <div class="sf-results-section sf-section-dead-pages" id="sf-dead-pages-section">
        <div class="sf-results-header sf-results-header-collapsible sf-dead-pages-header">
            <button type="button" class="sf-section-toggle sf-dead-pages-toggle" aria-expanded="true">
                <span class="sf-section-badge sf-badge-warning">&#128279;</span>
                <span class="sf-dead-pages-title">
                    <?php esc_html_e('Dead Pages Needing Redirects', 'screaming-fixes'); ?> (<span class="sf-dead-pages-count"><?php echo esc_html($total_dead_pages); ?></span>)
                </span>
                <span class="sf-section-hint"><?php esc_html_e('Create 301 redirects to reclaim backlink value from these pages', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
            </button>
            <div class="sf-results-actions">
                <button type="button" id="sf-export-results" class="sf-button sf-button-secondary">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <div class="sf-dead-pages-content">

        <div class="sf-table-wrapper sf-paginated-table" data-section="dead-pages" data-per-page="100" data-total="<?php echo esc_attr($total_dead_pages); ?>">
            <table class="sf-table sf-backlink-table">
                <thead>
                    <tr>
                        <th class="sf-col-checkbox">
                            <input type="checkbox" id="sf-select-all" />
                        </th>
                        <th class="sf-col-dead-page"><?php esc_html_e('Dead Page', 'screaming-fixes'); ?></th>
                        <th class="sf-col-backlinks"><?php esc_html_e('Backlinks', 'screaming-fixes'); ?></th>
                        <th class="sf-col-referrers"><?php esc_html_e('Top Referrers', 'screaming-fixes'); ?></th>
                        <th class="sf-col-redirect"><?php esc_html_e('Redirect To', 'screaming-fixes'); ?></th>
                        <th class="sf-col-actions"><?php esc_html_e('Actions', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody id="sf-backlink-results">
                    <?php foreach ($dead_pages as $index => $page): ?>
                    <tr class="sf-backlink-row" data-index="<?php echo esc_attr($index); ?>" data-source="<?php echo esc_attr($page['dead_page_path']); ?>" data-backlink-count="<?php echo esc_attr($page['backlink_count']); ?>" data-status="pending">
                        <td class="sf-col-checkbox">
                            <input type="checkbox" class="sf-row-checkbox" />
                        </td>
                        <td class="sf-col-dead-page">
                            <div class="sf-dead-page-cell">
                                <span class="sf-status-badge sf-status-<?php echo esc_attr($page['status_code']); ?>">
                                    <?php echo esc_html($page['status_code']); ?>
                                </span>
                                <div class="sf-page-info">
                                    <span class="sf-page-path" title="<?php echo esc_attr($page['dead_page']); ?>">
                                        <?php echo esc_html($page['dead_page_path']); ?>
                                    </span>
                                    <a href="<?php echo esc_url($page['dead_page']); ?>" target="_blank" rel="noopener" class="sf-external-link">
                                        <span class="dashicons dashicons-external"></span>
                                    </a>
                                </div>
                            </div>
                        </td>
                        <td class="sf-col-backlinks">
                            <span class="sf-backlink-count"><?php echo esc_html(number_format($page['backlink_count'])); ?></span>
                            <span class="sf-domains-count">
                                <?php
                                printf(
                                    esc_html__('from %d domains', 'screaming-fixes'),
                                    $page['referring_domains']
                                );
                                ?>
                            </span>
                        </td>
                        <td class="sf-col-referrers sf-referrers-cell">
                            <button type="button" class="sf-expand-referrers" data-dead-page="<?php echo esc_attr($page['dead_page_path']); ?>">
                                <span class="sf-referrer-preview">
                                    <?php
                                    $preview_domains = array_slice($page['top_referrers'], 0, 2);
                                    $domain_names = array_map(function($ref) { return $ref['domain']; }, $preview_domains);
                                    echo esc_html(implode(', ', $domain_names));
                                    if (count($page['top_referrers']) > 2) {
                                        echo ' <span class="sf-more">+' . (count($page['top_referrers']) - 2) . '</span>';
                                    }
                                    ?>
                                </span>
                                <span class="sf-expand-icon">&#9660;</span>
                            </button>

                            <div class="sf-referrers-dropdown" style="display: none;">
                                <div class="sf-referrers-header">
                                    <strong><?php esc_html_e('Top Referring Pages', 'screaming-fixes'); ?></strong>
                                    <span class="sf-referrers-count">
                                        <?php
                                        printf(
                                            esc_html__('%d of %d domains', 'screaming-fixes'),
                                            count($page['top_referrers']),
                                            $page['referring_domains']
                                        );
                                        ?>
                                    </span>
                                </div>
                                <table class="sf-referrers-table">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('DR', 'screaming-fixes'); ?></th>
                                            <th><?php esc_html_e('Referring Page', 'screaming-fixes'); ?></th>
                                            <th><?php esc_html_e('Links', 'screaming-fixes'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach (array_slice($page['top_referrers'], 0, 10) as $referrer): ?>
                                        <tr>
                                            <td class="sf-dr-cell">
                                                <span class="sf-dr-badge sf-dr-<?php echo esc_attr(SF_Backlink_API::get_dr_class($referrer['domain_rank'] ?? 0)); ?>">
                                                    <?php echo esc_html($referrer['domain_rank'] ?? 0); ?>
                                                </span>
                                            </td>
                                            <td class="sf-url-cell">
                                                <a href="<?php echo esc_url($referrer['url']); ?>" target="_blank" rel="noopener" title="<?php echo esc_attr($referrer['url']); ?>">
                                                    <span class="sf-referrer-domain"><?php echo esc_html($referrer['domain']); ?></span>
                                                    <span class="sf-referrer-path"><?php echo esc_html(wp_parse_url($referrer['url'], PHP_URL_PATH) ?: '/'); ?></span>
                                                    <span class="sf-external-icon">&#8599;</span>
                                                </a>
                                            </td>
                                            <td class="sf-links-cell"><?php echo esc_html($referrer['backlinks'] ?? 1); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </td>
                        <td class="sf-col-redirect">
                            <div class="sf-redirect-input-wrapper">
                                <input
                                    type="text"
                                    class="sf-input sf-redirect-destination"
                                    placeholder="<?php esc_attr_e('Enter destination URL', 'screaming-fixes'); ?>"
                                    data-source="<?php echo esc_attr($page['dead_page_path']); ?>"
                                />
                                <button type="button" class="sf-ai-suggest-btn" title="<?php esc_attr_e('AI: Find best replacement URL', 'screaming-fixes'); ?>">
                                    <span class="sf-ai-icon">&#129302;</span>
                                </button>
                            </div>
                        </td>
                        <td class="sf-col-actions">
                            <button type="button" class="sf-button sf-button-primary sf-button-sm sf-create-redirect-btn" disabled>
                                <?php esc_html_e('Create', 'screaming-fixes'); ?>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <?php if ($total_dead_pages > 100): ?>
        <div class="sf-pagination" data-section="dead-pages">
            <div class="sf-pagination-info">
                <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $total_dead_pages); ?></span>
                <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($total_dead_pages); ?></span>
            </div>
            <div class="sf-pagination-controls">
                <button type="button" class="sf-page-btn sf-page-prev" disabled>
                    <span class="dashicons dashicons-arrow-left-alt2"></span>
                </button>
                <div class="sf-page-numbers"></div>
                <button type="button" class="sf-page-btn sf-page-next">
                    <span class="dashicons dashicons-arrow-right-alt2"></span>
                </button>
            </div>
        </div>
        <?php endif; ?>

        </div><!-- .sf-dead-pages-content -->

        <!-- Batch Actions Bar - Outside collapsible content so it stays visible -->
        <div class="sf-batch-actions-bar" id="sf-batch-actions-bar" style="display: none;">
            <div class="sf-batch-info">
                <span class="sf-selected-count">0</span> <?php esc_html_e('selected', 'screaming-fixes'); ?>
            </div>
            <div class="sf-batch-buttons">
                <button type="button" id="sf-create-selected-redirects" class="sf-button sf-button-primary">
                    <span class="dashicons dashicons-admin-links"></span>
                    <?php esc_html_e('Create Redirects', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
    </div><!-- .sf-section-dead-pages -->

    <!-- Redirects Applied Section (Collapsible) -->
    <div class="sf-results-section sf-section-redirects-applied" id="sf-redirects-applied-section" <?php echo $redirect_results_count === 0 ? 'style="display: none;"' : ''; ?>>
        <div class="sf-results-header sf-results-header-collapsible sf-redirects-applied-header">
            <button type="button" class="sf-section-toggle sf-redirects-toggle" aria-expanded="<?php echo $redirect_results_count > 0 ? 'true' : 'false'; ?>">
                <span class="sf-section-badge sf-badge-success">&#10004;</span>
                <span class="sf-redirects-title">
                    <?php esc_html_e('Redirects Applied', 'screaming-fixes'); ?> (<span class="sf-redirects-header-count"><?php echo esc_html($redirect_results_count); ?></span>)
                </span>
                <span class="sf-section-hint"><?php esc_html_e('View redirect results and export to CSV', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon <?php echo $redirect_results_count > 0 ? 'sf-rotated' : ''; ?>"></span>
            </button>
            <div class="sf-results-actions">
                <button type="button" class="sf-button sf-button-secondary sf-export-redirects-btn">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <div class="sf-redirects-applied-content" <?php echo $redirect_results_count === 0 ? 'style="display: none;"' : ''; ?>>
            <!-- Status Filter Buttons -->
            <div class="sf-status-filter-row sf-redirect-status-filters">
                <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                <div class="sf-status-filters">
                    <button type="button" class="sf-status-filter active" data-status="all">
                        <span class="sf-status-icon">&#128203;</span>
                        <span class="sf-status-count sf-redirect-status-count-all"><?php echo esc_html($redirect_results_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                        <span class="sf-status-icon">&#9989;</span>
                        <span class="sf-status-count sf-redirect-status-count-success"><?php echo esc_html(count($successful_redirects)); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                        <span class="sf-status-icon">&#10060;</span>
                        <span class="sf-status-count sf-redirect-status-count-failed"><?php echo esc_html(count($failed_redirects)); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                        <span class="sf-status-icon">&#9203;</span>
                        <span class="sf-status-count sf-redirect-status-count-skipped"><?php echo esc_html(count($skipped_redirects)); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                    </button>
                </div>
            </div>

            <!-- Redirects Applied Table -->
            <div class="sf-table-wrapper sf-paginated-table" data-section="redirects-applied" data-per-page="100" data-total="<?php echo esc_attr($redirect_results_count); ?>">
                <table class="sf-table sf-redirects-table">
                    <thead>
                        <tr>
                            <th class="sf-col-dead-page"><?php esc_html_e('Dead Page URL', 'screaming-fixes'); ?></th>
                            <th class="sf-col-redirect-url"><?php esc_html_e('New 301 Redirect', 'screaming-fixes'); ?></th>
                            <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            <th class="sf-col-plugin"><?php esc_html_e('Plugin Used', 'screaming-fixes'); ?></th>
                            <th class="sf-col-date"><?php esc_html_e('Date Applied', 'screaming-fixes'); ?></th>
                            <th class="sf-col-backlinks"><?php esc_html_e('Backlinks', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="sf-redirects-applied-tbody">
                        <?php foreach ($redirect_results as $result):
                            $status_class = '';
                            $status_icon = '';
                            $status_label = '';

                            if ($result['status'] === 'success') {
                                $status_class = 'sf-redirect-row-success';
                                $status_icon = '&#9989;';
                                $status_label = __('Fixed', 'screaming-fixes');
                            } elseif ($result['status'] === 'failed') {
                                $status_class = 'sf-redirect-row-failed';
                                $status_icon = '&#10060;';
                                $status_label = __('Failed', 'screaming-fixes');
                            } else {
                                $status_class = 'sf-redirect-row-skipped';
                                $status_icon = '&#9203;';
                                $status_label = __('Skipped', 'screaming-fixes');
                            }
                        ?>
                            <tr class="sf-redirect-result-row <?php echo esc_attr($status_class); ?>" data-status="<?php echo esc_attr($result['status']); ?>">
                                <td class="sf-col-dead-page">
                                    <span class="sf-dead-page-url" title="<?php echo esc_attr($result['dead_page_url']); ?>">
                                        <?php echo esc_html($result['dead_page_url']); ?>
                                    </span>
                                </td>
                                <td class="sf-col-redirect-url">
                                    <?php if (!empty($result['redirect_url'])): ?>
                                        <a href="<?php echo esc_url($result['redirect_url']); ?>" target="_blank" rel="noopener" class="sf-redirect-url-link">
                                            <?php echo esc_html($result['redirect_url']); ?>
                                            <span class="dashicons dashicons-external"></span>
                                        </a>
                                    <?php else: ?>
                                        <span class="sf-no-redirect">&mdash;</span>
                                    <?php endif; ?>
                                </td>
                                <td class="sf-col-status">
                                    <span class="sf-redirect-status-badge sf-status-<?php echo esc_attr($result['status']); ?>">
                                        <span class="sf-status-icon"><?php echo $status_icon; ?></span>
                                        <?php echo esc_html($status_label); ?>
                                    </span>
                                    <?php if (!empty($result['status_message']) && $result['status'] !== 'success'): ?>
                                        <span class="sf-status-message"><?php echo esc_html($result['status_message']); ?></span>
                                    <?php endif; ?>
                                </td>
                                <td class="sf-col-plugin">
                                    <?php echo !empty($result['plugin']) ? esc_html($result['plugin']) : '&mdash;'; ?>
                                </td>
                                <td class="sf-col-date">
                                    <?php
                                    if (!empty($result['created_at'])) {
                                        echo esc_html(human_time_diff(strtotime($result['created_at']), current_time('timestamp')) . ' ' . __('ago', 'screaming-fixes'));
                                    } else {
                                        echo '&mdash;';
                                    }
                                    ?>
                                </td>
                                <td class="sf-col-backlinks">
                                    <?php echo esc_html(number_format($result['backlink_count'] ?? 0)); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($redirect_results_count > 100): ?>
            <div class="sf-pagination" data-section="redirects-applied">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $redirect_results_count); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($redirect_results_count); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" disabled>
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-btn sf-page-next">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
            </div>
            <?php endif; ?>

            <div class="sf-info-notice sf-redirects-info-notice">
                <span class="dashicons dashicons-info-outline"></span>
                <p><?php esc_html_e('After applying redirects, verify they are working by checking your redirect plugin settings. If redirects don\'t appear to be active on your site, try clearing your browser cache and any website caching plugin (e.g., WP Super Cache, W3 Total Cache, LiteSpeed Cache) to see the changes live.', 'screaming-fixes'); ?></p>
            </div>
        </div>
    </div>

    <?php elseif ($has_empty_results): ?>

    <!-- Success State - No Dead Backlinks Found -->
    <div class="sf-no-dead-backlinks">
        <div class="sf-success-icon">&#127881;</div>
        <h3><?php esc_html_e('No Dead Backlinks Found!', 'screaming-fixes'); ?></h3>
        <p class="sf-success-message">
            <?php esc_html_e("Your site is in great shape! We couldn't find any external sites linking to broken pages.", 'screaming-fixes'); ?>
        </p>
        <p class="sf-success-subtext">
            <?php esc_html_e("Either your internal linking game is chef's kiss, or the internet just really respects your URL structure.", 'screaming-fixes'); ?> &#128104;&#8205;&#127859;
        </p>
        <div class="sf-success-tips">
            <p><strong><?php esc_html_e('Keep it that way:', 'screaming-fixes'); ?></strong></p>
            <ul>
                <li>&#9989; <?php esc_html_e('Set up 301 redirects when you delete or move pages', 'screaming-fixes'); ?></li>
                <li>&#9989; <?php esc_html_e('Run this scan monthly to catch new issues early', 'screaming-fixes'); ?></li>
                <li>&#9989; <?php esc_html_e('Use consistent URL structures to avoid accidental 404s', 'screaming-fixes'); ?></li>
            </ul>
        </div>
        <p class="sf-scan-again-note">
            <?php
            printf(
                esc_html__('Last scanned: %s ago', 'screaming-fixes'),
                '<strong>' . esc_html(human_time_diff(strtotime($scanned_at), current_time('timestamp'))) . '</strong>'
            );
            ?>
        </p>

        <?php if ($scan_debug): ?>
        <div class="sf-scan-debug" style="margin-top: 20px; padding: 15px; background: #f8f9fa; border: 1px solid #e5e7eb; border-radius: 8px; font-size: 13px;">
            <p style="margin: 0 0 10px; font-weight: 600; color: #6b7280;">&#128269; Scan Details (for debugging):</p>
            <ul style="margin: 0; padding-left: 20px; color: #6b7280;">
                <li><strong>Backlinks found in API:</strong> <?php echo esc_html($scan_debug['total_backlinks_from_api'] ?? 'N/A'); ?></li>
                <li><strong>Unique target URLs:</strong> <?php echo esc_html($scan_debug['unique_target_urls'] ?? 'N/A'); ?></li>
                <li><strong>URLs checked for 404:</strong> <?php echo esc_html($scan_debug['urls_checked'] ?? 'N/A'); ?></li>
                <li><strong>Broken URLs found:</strong> <?php echo esc_html($scan_debug['broken_urls_found'] ?? 'N/A'); ?></li>
            </ul>
            <?php if (isset($scan_debug['total_backlinks_from_api']) && $scan_debug['total_backlinks_from_api'] === 0): ?>
            <p style="margin: 10px 0 0; color: #dc2626;">
                &#9888; <strong>Note:</strong> DataForSEO returned 0 backlinks for this domain. This could mean:
                <br>&bull; The domain is new or hasn't been indexed by DataForSEO yet
                <br>&bull; The domain name may need to be entered differently (try with/without www)
                <br>&bull; There genuinely are no external backlinks to this domain
            </p>
            <?php elseif (isset($scan_debug['total_backlinks_from_api']) && $scan_debug['total_backlinks_from_api'] > 0 && ($scan_debug['broken_urls_found'] ?? 0) === 0): ?>
            <p style="margin: 10px 0 0; color: #22c55e;">
                &#9989; <strong>Good news:</strong> DataForSEO found <?php echo esc_html($scan_debug['total_backlinks_from_api']); ?> backlinks, but all target pages on your site returned 200 OK - no broken pages!
            </p>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>

    <?php else: ?>

    <!-- Empty State - No Scan Yet -->
    <div class="sf-empty-state">
        <div class="sf-empty-icon">&#128279;</div>
        <h3><?php esc_html_e('No scan data yet', 'screaming-fixes'); ?></h3>
        <p><?php esc_html_e('Enter your domain above and click "Scan for Dead Backlinks" to find external links pointing to broken pages.', 'screaming-fixes'); ?></p>
    </div>

    <?php endif; ?>

</div>
