/**
 * Broken Links Module JavaScript
 *
 * Handles UI interactions for the broken links fixer
 */

(function($) {
    'use strict';

    // Module state
    var BrokenLinks = {
        fixes: {},
        selectedRows: [],
        isProcessing: false,
        currentFilter: 'all',
        pagination: {
            fixable: { currentPage: 1, perPage: 100, totalItems: 0 },
            manual: { currentPage: 1, perPage: 100, totalItems: 0 },
            skipped: { currentPage: 1, perPage: 100, totalItems: 0 },
            fixed: { currentPage: 1, perPage: 100, totalItems: 0 }
        }
    };

    // localStorage key for persisting fixes across pages
    var STORAGE_KEY = 'sf_broken_links_fixes';

    // Fun loading messages for CSV processing
    var csvLoadingMessages = [
        "📄 Parsing CSV file...",
        "🔗 Scanning for broken links...",
        "🔍 Inspecting URLs...",
        "🩹 Getting ready to fix things...",
        "☕ Getting coffee...",
        "🚽 Quick bathroom break...",
        "🔧 Grabbing the toolbox...",
        "🕵️ Investigating 404s...",
        "🧊 Chilling with the 404s...",
        "💃 Getting hyperlinky with it...",
        "🏥 Preparing the link hospital...",
        "🔨 Unbreaking the internet...",
        "🎯 Targeting broken URLs...",
        "🗺️ Mapping redirect destinations...",
        "🧹 Sweeping up dead links...",
        "🚀 Almost there...",
        "🤔 Thinking really hard...",
        "🔬 Analyzing response codes...",
        "✨ Polishing the final details..."
    ];

    /**
     * Initialize the module
     */
    function init() {
        bindEvents();
        loadExistingData();
        loadFixesFromStorage();
        checkForPendingUpload();
        initPagination();
        initFilters();
    }

    /**
     * Check if we arrived with an upload_id to process
     */
    function checkForPendingUpload() {
        var urlParams = new URLSearchParams(window.location.search);
        var uploadId = urlParams.get('upload_id');

        if (uploadId) {
            processPendingUpload(uploadId);
            // Clean URL
            var cleanUrl = window.location.href.replace(/[?&]upload_id=[^&]+/, '').replace(/&$/, '');
            window.history.replaceState({}, document.title, cleanUrl);
        }
    }

    /**
     * Process a pending upload
     */
    function processPendingUpload(uploadId) {
        // Show loading container with rotating messages
        var loadingState = showCSVLoadingContainer();

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_process_csv',
                nonce: sfBrokenLinksData.nonce,
                upload_id: uploadId
            },
            success: function(response) {
                // Clean up loading state
                hideCSVLoadingContainer(loadingState);

                if (response.success) {
                    showToast(response.data.message, 'success');
                    // Reload to show results
                    window.location.reload();
                } else {
                    showToast(response.data.message || sfBrokenLinksData.i18n.fixesFailed, 'error');
                }
            },
            error: function() {
                // Clean up loading state
                hideCSVLoadingContainer(loadingState);

                showToast(sfBrokenLinksData.i18n.fixesFailed, 'error');
            }
        });
    }

    /**
     * Bind all event handlers
     */
    function bindEvents() {
        // Instructions toggle
        $(document).on('click', '.sf-instructions-toggle', toggleInstructions);

        // Sources list toggle
        $(document).on('click', '.sf-sources-toggle', toggleSourcesList);

        // Section toggles (Fixable, Manual, Skipped)
        $(document).on('click', '.sf-section-toggle', toggleSection);

        // Fixed links section toggle
        $(document).on('click', '.sf-fixed-links-toggle', toggleFixedLinksSection);

        // Show all sources button (opens modal)
        $(document).on('click', '.sf-show-all-sources', handleShowAllSources);

        // Modal close button
        $(document).on('click', '.sf-modal-close', closeSourcesModal);

        // Close modal on background click
        $(document).on('click', '.sf-sources-modal', function(e) {
            if ($(e.target).is('.sf-sources-modal')) {
                closeSourcesModal();
            }
        });

        // Copy sources button
        $(document).on('click', '.sf-copy-sources', handleCopySources);

        // Export sources button
        $(document).on('click', '.sf-export-sources', handleExportSources);

        // Close modal on Escape key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && $('.sf-sources-modal').is(':visible')) {
                closeSourcesModal();
            }
            // Close dropdowns on Escape
            if (e.key === 'Escape' && $('.sf-fix-dropdown-menu:visible').length) {
                closeAllDropdowns();
            }
        });

        // NEW: Fix dropdown button click
        $(document).on('click', '.sf-fix-dropdown-btn', handleDropdownToggle);

        // NEW: Fix option selection
        $(document).on('click', '.sf-fix-option', handleFixOptionSelect);

        // NEW: Close dropdowns when clicking outside
        $(document).on('click', function(e) {
            if (!$(e.target).closest('.sf-fix-dropdown-wrapper').length) {
                closeAllDropdowns();
            }
        });

        // NEW: Manual URL input change
        $(document).on('input', '.sf-manual-url-input .sf-new-url', handleManualUrlInput);

        // NEW: Verify URL button
        $(document).on('click', '.sf-verify-url-btn', handleVerifyUrl);

        // NEW: Change fix button (to re-open dropdown)
        $(document).on('click', '.sf-fix-change-btn', handleChangeFix);

        // NEW: Individual Fix button click
        $(document).on('click', '.sf-fix-btn', handleSingleFix);

        // Legacy: Fix action dropdown change (for old markup compatibility)
        $(document).on('change', '.sf-fix-action', handleFixActionChange);

        // Legacy: New URL input change
        $(document).on('input', '.sf-new-url-input .sf-new-url', handleNewUrlChange);

        // AI suggestion button (single) - legacy
        $(document).on('click', '.sf-ai-suggest-btn', handleAiSuggest);

        // Row selection (legacy)
        $(document).on('change', '.sf-row-select', handleRowSelect);
        $(document).on('change', '#sf-select-all', handleSelectAll);

        // Apply fixes button (legacy)
        $(document).on('click', '.sf-apply-fixes', handleApplyFixes);

        // NEW: Apply All Fixes button
        $(document).on('click', '.sf-apply-all-fixes', handleApplyAllFixes);

        // Expandable Panel: Edit button click
        $(document).on('click', '.sf-edit-btn', handleEditButtonClick);
        $(document).on('click', '.sf-fix-status-failed', handleEditButtonClick);
        $(document).on('click', '.sf-section-fixed .sf-status-badge.sf-status-failed', handleFixedFailedBadgeClick);

        // Expandable Panel: Close button click
        $(document).on('click', '.sf-panel-close', handlePanelClose);

        // Expandable Panel: Action buttons (Remove, Ignore, Get AI)
        $(document).on('click', '.sf-action-panel .sf-action-btn', handlePanelAction);

        // Expandable Panel: Save button click
        $(document).on('click', '.sf-action-panel .sf-save-btn', handlePanelSave);

        // Expandable Panel: Use AI suggestion button
        $(document).on('click', '.sf-action-panel .sf-use-ai-btn', handleUseAiSuggestion);

        // Expandable Panel: URL input change
        $(document).on('input', '.sf-action-panel .sf-replacement-url', handlePanelUrlInput);

        // Export button
        $(document).on('click', '.sf-export-btn', handleExport);

        // Export fixed links button
        $(document).on('click', '.sf-export-fixed-btn', handleExportFixedLinks);

        // Clear results button
        $(document).on('click', '.sf-clear-results-btn', handleClearResults);

        // CSV upload success (from parent csv-upload.js)
        $(document).on('sf:csv:uploaded', handleCsvUploaded);

        // Pagination: Previous button
        $(document).on('click', '.sf-broken-links-module .sf-page-prev:not(:disabled)', function(e) {
            e.preventDefault();
            var section = $(this).closest('.sf-pagination').data('section');
            goToPage(section, BrokenLinks.pagination[section].currentPage - 1);
        });

        // Pagination: Next button
        $(document).on('click', '.sf-broken-links-module .sf-page-next:not(:disabled)', function(e) {
            e.preventDefault();
            var section = $(this).closest('.sf-pagination').data('section');
            goToPage(section, BrokenLinks.pagination[section].currentPage + 1);
        });

        // Pagination: Page number buttons
        $(document).on('click', '.sf-broken-links-module .sf-page-number:not(.active):not(.sf-page-ellipsis)', function(e) {
            e.preventDefault();
            var section = $(this).closest('.sf-pagination').data('section');
            var page = parseInt($(this).data('page'), 10);
            goToPage(section, page);
        });

        // Filter buttons click
        $(document).on('click', '.sf-filter-stat', function() {
            var $btn = $(this);
            var filter = $btn.data('filter');

            // Check if filter is disabled
            if ($btn.hasClass('sf-filter-disabled')) {
                if (filter === 'fixed') {
                    showToast(sfBrokenLinksData.i18n.noFixesApplied || 'No fixes have been applied yet. Apply link fixes first to view fixed links.', 'info');
                } else if (filter === 'images') {
                    showToast('No broken images found in the uploaded data.', 'info');
                }
                return;
            }

            filterByType(filter);
        });

        // Bulk upload event handlers
        $(document).on('click', '#sf-bulk-confirm', handleBulkConfirm);
        $(document).on('click', '#sf-bulk-clear, #sf-bulk-new-upload', handleBulkClear);
        $(document).on('click', '#sf-bulk-download-preview', handleBulkDownloadPreview);
        $(document).on('click', '#sf-bulk-download-results', handleBulkDownloadResults);
    }

    /**
     * Load existing data if available
     */
    function loadExistingData() {
        // Data is already rendered server-side, just initialize state
        $('.sf-link-row').each(function() {
            var $row = $(this);
            var brokenUrl = $row.data('broken-url');
            var postIds = ($row.data('post-ids') || '').toString().split(',').filter(Boolean);

            BrokenLinks.fixes[brokenUrl] = {
                action: '',
                new_url: '',
                post_ids: postIds
            };
        });
    }

    // =====================================
    // LOCALSTORAGE PERSISTENCE
    // =====================================

    /**
     * Save fixes to localStorage for persistence across pages
     */
    function saveFixesToStorage() {
        try {
            // Only save fixes that have an action set (not empty state)
            var fixesToSave = {};
            $.each(BrokenLinks.fixes, function(brokenUrl, fix) {
                if (fix.action && fix.action !== '') {
                    fixesToSave[brokenUrl] = {
                        action: fix.action,
                        new_url: fix.new_url || '',
                        post_ids: fix.post_ids || []
                    };
                }
            });
            localStorage.setItem(STORAGE_KEY, JSON.stringify(fixesToSave));
        } catch (e) {
            console.warn('Failed to save fixes to localStorage:', e);
        }
    }

    /**
     * Load fixes from localStorage
     */
    function loadFixesFromStorage() {
        try {
            var stored = localStorage.getItem(STORAGE_KEY);
            if (!stored) {
                return;
            }

            var storedFixes = JSON.parse(stored);
            var restoredCount = 0;

            // Merge stored fixes with current page state
            $.each(storedFixes, function(brokenUrl, fix) {
                // Check if this URL exists on the current page
                var $row = $('.sf-link-row[data-broken-url="' + escapeSelector(brokenUrl) + '"]');

                if ($row.length > 0) {
                    // Row is on current page - restore the fix state
                    if (!BrokenLinks.fixes[brokenUrl]) {
                        BrokenLinks.fixes[brokenUrl] = {
                            post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
                        };
                    }
                    BrokenLinks.fixes[brokenUrl].action = fix.action;
                    BrokenLinks.fixes[brokenUrl].new_url = fix.new_url || '';

                    // Update the row UI to show staged state
                    restoreRowState($row, fix);
                    restoredCount++;
                } else {
                    // Row is on a different page - keep in memory for Apply All
                    BrokenLinks.fixes[brokenUrl] = fix;
                }
            });

            // Update the batch state
            updateBatchState();

            if (restoredCount > 0) {
                console.log('Restored ' + restoredCount + ' fixes from previous session');
            }
        } catch (e) {
            console.warn('Failed to load fixes from localStorage:', e);
        }
    }

    /**
     * Clear fixes from localStorage
     */
    function clearFixesFromStorage() {
        try {
            localStorage.removeItem(STORAGE_KEY);
        } catch (e) {
            console.warn('Failed to clear fixes from localStorage:', e);
        }
    }

    /**
     * Restore row state from stored fix
     */
    function restoreRowState($row, fix) {
        var brokenUrl = $row.data('broken-url');
        var action = fix.action;
        var newUrl = fix.new_url;

        if (action === 'ignore') {
            $row.addClass('sf-row-ignored');
            updateRowStatusDisplay($row, 'Ignored', 'sf-status-ignored');
        } else if (action === 'remove_link') {
            $row.addClass('sf-row-staged');
            updateRowStatusDisplay($row, 'Remove Hyperlink', 'sf-status-staged', false, null);
        } else if (action === 'replace' && newUrl) {
            $row.addClass('sf-row-staged');
            updateRowStatusDisplay($row, 'Replace', 'sf-status-staged', false, newUrl);
        }
    }

    /**
     * Toggle instructions visibility
     */
    function toggleInstructions() {
        var $toggle = $(this);
        var $content = $toggle.next('.sf-instructions-content');
        var isExpanded = $toggle.attr('aria-expanded') === 'true';

        $toggle.attr('aria-expanded', !isExpanded);
        $content.slideToggle(200);
        $toggle.find('.sf-toggle-icon').toggleClass('sf-rotated', !isExpanded);
    }

    /**
     * Toggle sources list visibility
     */
    function toggleSourcesList() {
        var $toggle = $(this);
        var $list = $toggle.next('.sf-sources-list');

        $list.slideToggle(200);
        $toggle.toggleClass('sf-expanded');
        $toggle.find('.dashicons').toggleClass('dashicons-arrow-down-alt2 dashicons-arrow-up-alt2');
    }

    /**
     * Toggle section visibility (Fixable, Manual, Skipped, Fixed)
     */
    function toggleSection() {
        var $toggle = $(this);
        var isExpanded = $toggle.attr('aria-expanded') === 'true';
        var $content = null;

        // For fixable section, toggle is inside .sf-results-header wrapper
        if ($toggle.hasClass('sf-fixable-toggle')) {
            $content = $toggle.closest('.sf-results-header').siblings('.sf-fixable-content');
        }
        // For fixed section, toggle is inside .sf-results-header wrapper
        else if ($toggle.hasClass('sf-fixed-toggle')) {
            $content = $toggle.closest('.sf-results-header').siblings('.sf-fixed-content');
        }
        // For manual section, content is next sibling
        else if ($toggle.hasClass('sf-manual-toggle')) {
            $content = $toggle.next('.sf-manual-content');
        }
        // For skipped section, content is next sibling
        else {
            $content = $toggle.next('.sf-skipped-content');
        }

        if (!$content || $content.length === 0) {
            return;
        }

        $toggle.attr('aria-expanded', !isExpanded);
        $content.slideToggle(200);
        $toggle.find('.sf-toggle-icon').toggleClass('sf-rotated', !isExpanded);
    }

    /**
     * Toggle fixed links section visibility
     */
    function toggleFixedLinksSection() {
        var $toggle = $(this);
        var $content = $toggle.next('.sf-fixed-links-content');
        var isExpanded = $toggle.attr('aria-expanded') === 'true';

        $toggle.attr('aria-expanded', !isExpanded);
        $content.slideToggle(200);
        $toggle.find('.sf-toggle-icon').toggleClass('sf-rotated', !isExpanded);
    }

    /**
     * Handle "Show all X sources" button click - opens modal
     */
    function handleShowAllSources() {
        var $btn = $(this);
        var brokenUrl = $btn.data('broken-url');
        var total = $btn.data('total');

        // Find the toggle button which has the sources data
        var $toggle = $btn.closest('.sf-sources-wrapper').find('.sf-sources-toggle');
        var sourcesData = $toggle.data('sources');

        if (!sourcesData || !brokenUrl) {
            return;
        }

        openSourcesModal(brokenUrl, sourcesData, total);
    }

    /**
     * Open the sources modal with all sources listed
     */
    function openSourcesModal(brokenUrl, sources, total) {
        var $modal = $('.sf-sources-modal');

        // Store data for copy/export
        $modal.data('broken-url', brokenUrl);
        $modal.data('sources', sources);

        // Update modal content
        $modal.find('.sf-sources-count').text(total || sources.length);
        $modal.find('.sf-broken-url-display').text(brokenUrl);

        // Build sources list
        var $list = $modal.find('.sf-sources-full-ul');
        $list.empty();

        sources.forEach(function(source) {
            var $li = $('<li></li>');

            var $link = $('<a></a>')
                .attr('href', source.url)
                .attr('target', '_blank')
                .attr('rel', 'noopener')
                .text(source.path || source.url);

            $li.append($link);

            // Add edit link if post_id exists
            if (source.post_id && source.edit_url) {
                var $editLink = $('<a></a>')
                    .addClass('sf-edit-link')
                    .attr('href', source.edit_url)
                    .attr('target', '_blank')
                    .attr('rel', 'noopener')
                    .attr('title', sfBrokenLinksData.i18n.editPost || 'Edit post')
                    .html('<span class="dashicons dashicons-edit"></span>');
                $li.append($editLink);
            }

            $list.append($li);
        });

        // Show modal
        $modal.fadeIn(200);

        // Trap focus in modal
        $modal.find('.sf-modal-close').focus();
    }

    /**
     * Close the sources modal
     */
    function closeSourcesModal() {
        $('.sf-sources-modal').fadeOut(200);
    }

    /**
     * Handle copy all sources button
     */
    function handleCopySources() {
        var $btn = $(this);
        var $modal = $('.sf-sources-modal');
        var sources = $modal.data('sources');

        if (!sources || !sources.length) {
            return;
        }

        // Build text with all URLs
        var urls = sources.map(function(source) {
            return source.url;
        }).join('\n');

        // Copy to clipboard
        copyToClipboard(urls).then(function() {
            // Show success state
            $btn.addClass('sf-copy-success');
            var originalText = $btn.html();
            $btn.html('<span class="dashicons dashicons-yes"></span> ' + (sfBrokenLinksData.i18n.copied || 'Copied!'));

            setTimeout(function() {
                $btn.removeClass('sf-copy-success');
                $btn.html(originalText);
            }, 2000);
        }).catch(function() {
            showToast(sfBrokenLinksData.i18n.copyFailed || 'Failed to copy', 'error');
        });
    }

    /**
     * Handle export sources as CSV button
     */
    function handleExportSources() {
        var $modal = $('.sf-sources-modal');
        var brokenUrl = $modal.data('broken-url');
        var sources = $modal.data('sources');

        if (!sources || !sources.length) {
            return;
        }

        // Build CSV content
        var csvContent = 'Source URL,Post Title,Broken URL\n';
        sources.forEach(function(source) {
            var postTitle = (source.post_title || '').replace(/"/g, '""');
            var sourceUrl = (source.url || '').replace(/"/g, '""');
            var brokenUrlEscaped = brokenUrl.replace(/"/g, '""');
            csvContent += '"' + sourceUrl + '","' + postTitle + '","' + brokenUrlEscaped + '"\n';
        });

        // Create download
        var blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        var link = document.createElement('a');
        var url = URL.createObjectURL(blob);

        // Generate filename from broken URL
        var filename = 'sources-' + brokenUrl.replace(/[^a-z0-9]/gi, '-').substring(0, 50) + '.csv';

        link.setAttribute('href', url);
        link.setAttribute('download', filename);
        link.style.visibility = 'hidden';

        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        showToast(sfBrokenLinksData.i18n.exportComplete || 'CSV exported', 'success');
    }

    /**
     * Copy text to clipboard (modern API with fallback)
     */
    function copyToClipboard(text) {
        if (navigator.clipboard && navigator.clipboard.writeText) {
            return navigator.clipboard.writeText(text);
        }

        // Fallback for older browsers
        return new Promise(function(resolve, reject) {
            var textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.opacity = '0';
            document.body.appendChild(textarea);
            textarea.select();

            try {
                var success = document.execCommand('copy');
                document.body.removeChild(textarea);
                if (success) {
                    resolve();
                } else {
                    reject();
                }
            } catch (err) {
                document.body.removeChild(textarea);
                reject(err);
            }
        });
    }

    // =====================================
    // NEW DROPDOWN HANDLERS
    // =====================================

    /**
     * Close all open dropdowns
     */
    function closeAllDropdowns() {
        $('.sf-fix-dropdown-btn').removeClass('sf-active');
        $('.sf-fix-dropdown-menu').hide();
    }

    /**
     * Handle dropdown button toggle
     */
    function handleDropdownToggle(e) {
        e.stopPropagation();
        var $btn = $(this);
        var $menu = $btn.next('.sf-fix-dropdown-menu');
        var isOpen = $menu.is(':visible');

        // Close all other dropdowns first
        closeAllDropdowns();

        if (!isOpen) {
            $btn.addClass('sf-active');
            $menu.show();
        }
    }

    /**
     * Handle fix option selection from dropdown
     */
    function handleFixOptionSelect(e) {
        var $option = $(this);
        var $row = $option.closest('.sf-link-row');
        var brokenUrl = $row.data('broken-url');
        var action = $option.data('action');
        var $selector = $row.find('.sf-fix-selector');
        var sourceCount = $row.find('.sf-fix-btn').data('source-count') || 1;

        // Close the dropdown
        closeAllDropdowns();

        // Initialize fix data if needed
        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        // Hide all states first
        $selector.find('.sf-fix-dropdown-wrapper').hide();
        $selector.find('.sf-manual-url-input').hide();
        $selector.find('.sf-ai-loading').hide();
        $selector.find('.sf-ai-result').hide();
        $selector.find('.sf-fix-selected-display').hide();

        // Handle based on action
        switch (action) {
            case 'manual':
                $selector.find('.sf-manual-url-input').show();
                $selector.find('.sf-manual-url-input .sf-new-url').focus();
                BrokenLinks.fixes[brokenUrl].action = 'replace';
                BrokenLinks.fixes[brokenUrl].new_url = '';
                break;

            case 'ai_suggest':
                $selector.find('.sf-ai-loading').show();
                BrokenLinks.fixes[brokenUrl].action = 'replace';
                // Trigger AI suggestion
                fetchAiSuggestion($row, brokenUrl, sourceCount);
                break;

            case 'remove_link':
            case 'remove_all':
                BrokenLinks.fixes[brokenUrl].action = action;
                BrokenLinks.fixes[brokenUrl].new_url = '';
                showFixSelectedState($row, action);
                break;

            case 'ignore':
                BrokenLinks.fixes[brokenUrl].action = 'ignore';
                BrokenLinks.fixes[brokenUrl].new_url = '';
                showFixSelectedState($row, action);
                $row.addClass('sf-fix-ignored');
                break;
        }

        updateFixButton($row);
        updateBatchState();
    }

    /**
     * Fetch AI suggestion for a URL
     */
    function fetchAiSuggestion($row, brokenUrl, sourceCount) {
        var $selector = $row.find('.sf-fix-selector');
        var $loading = $selector.find('.sf-ai-loading');

        // Fun loading messages
        var loadingMessages = [
            'Consulting the URL oracle...',
            'Searching the archives...',
            'Finding your perfect match...',
            'Analyzing the void...',
            'Hunting for alternatives...',
            'Mining the internet...',
            'Decoding the matrix...'
        ];

        // Rotate loading messages
        var messageIndex = 0;
        var messageInterval = setInterval(function() {
            messageIndex = (messageIndex + 1) % loadingMessages.length;
            $loading.find('.sf-ai-loading-text').text(loadingMessages[messageIndex]);
        }, 2000);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_get_ai_suggestion',
                nonce: sfBrokenLinksData.nonce,
                url: brokenUrl,
                post_count: sourceCount
            },
            success: function(response) {
                clearInterval(messageInterval);

                if (response.success && response.data.suggestion) {
                    var suggestion = response.data.suggestion;

                    if (suggestion.action === 'replace' && suggestion.new_url) {
                        BrokenLinks.fixes[brokenUrl].action = 'replace';
                        BrokenLinks.fixes[brokenUrl].new_url = suggestion.new_url;
                        showAiResult($row, suggestion.new_url);
                    } else if (suggestion.action === 'remove_link') {
                        BrokenLinks.fixes[brokenUrl].action = 'remove_link';
                        showFixSelectedState($row, 'remove_link');
                    } else {
                        // No good suggestion found, show manual input
                        $selector.find('.sf-ai-loading').hide();
                        $selector.find('.sf-manual-url-input').show();
                        showToast(sfBrokenLinksData.i18n.noSuggestion || 'No replacement found. Please enter manually.', 'warning');
                    }
                } else {
                    // Error - show manual input
                    $selector.find('.sf-ai-loading').hide();
                    $selector.find('.sf-fix-dropdown-wrapper').show();
                    showToast(response.data.message || sfBrokenLinksData.i18n.aiFailed, 'error');
                }

                updateFixButton($row);
                updateBatchState();
            },
            error: function() {
                clearInterval(messageInterval);
                $selector.find('.sf-ai-loading').hide();
                $selector.find('.sf-fix-dropdown-wrapper').show();
                showToast(sfBrokenLinksData.i18n.aiFailed, 'error');
            }
        });
    }

    /**
     * Show AI result in the UI
     */
    function showAiResult($row, url) {
        var $selector = $row.find('.sf-fix-selector');

        $selector.find('.sf-ai-loading').hide();
        $selector.find('.sf-ai-result').show();
        $selector.find('.sf-ai-url-text').text(url);
        $selector.find('.sf-ai-url-link').attr('href', url);

        $row.addClass('sf-fix-ready');
    }

    /**
     * Show fix selected state (for remove/ignore actions)
     */
    function showFixSelectedState($row, action) {
        var $selector = $row.find('.sf-fix-selector');
        var $display = $selector.find('.sf-fix-selected-display');
        var $label = $display.find('.sf-fix-action-label');

        var labels = {
            'remove_link': '🔗 Remove link (keep text)',
            'remove_all': '🗑️ Remove link & text',
            'ignore': '🚫 Ignored'
        };

        $label.text(labels[action] || action);
        $display.show();

        if (action !== 'ignore') {
            $row.addClass('sf-fix-ready');
        }
    }

    /**
     * Handle manual URL input
     */
    function handleManualUrlInput() {
        var $input = $(this);
        var $row = $input.closest('.sf-link-row');
        var brokenUrl = $input.data('broken-url');
        var newUrl = $input.val().trim();

        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        BrokenLinks.fixes[brokenUrl].action = 'replace';
        BrokenLinks.fixes[brokenUrl].new_url = newUrl;

        // Update row state
        if (newUrl && isValidUrl(newUrl)) {
            $row.addClass('sf-fix-ready');
            $input.removeClass('sf-url-invalid').addClass('sf-url-valid');
        } else {
            $row.removeClass('sf-fix-ready');
            $input.removeClass('sf-url-valid');
            if (newUrl) {
                $input.addClass('sf-url-invalid');
            } else {
                $input.removeClass('sf-url-invalid');
            }
        }

        updateFixButton($row);
        updateBatchState();
    }

    /**
     * Handle verify URL button click
     */
    function handleVerifyUrl() {
        var $btn = $(this);
        var $row = $btn.closest('.sf-link-row');
        var $input = $row.find('.sf-manual-url-input .sf-new-url');
        var url = $input.val().trim();

        if (!url || !isValidUrl(url)) {
            showToast(sfBrokenLinksData.i18n.invalidUrl || 'Please enter a valid URL', 'warning');
            return;
        }

        $btn.addClass('sf-loading');
        $btn.find('.dashicons').removeClass('dashicons-yes-alt').addClass('dashicons-update sf-spinning');

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_verify_url',
                nonce: sfBrokenLinksData.nonce,
                url: url
            },
            success: function(response) {
                if (response.success && response.data.valid) {
                    $input.removeClass('sf-url-invalid').addClass('sf-url-valid');
                    showToast(sfBrokenLinksData.i18n.urlValid || 'URL is valid!', 'success');
                } else {
                    $input.removeClass('sf-url-valid').addClass('sf-url-invalid');
                    showToast(response.data.message || sfBrokenLinksData.i18n.urlInvalid || 'URL returned ' + (response.data.status || 'error'), 'warning');
                }
            },
            error: function() {
                showToast(sfBrokenLinksData.i18n.verifyFailed || 'Could not verify URL', 'error');
            },
            complete: function() {
                $btn.removeClass('sf-loading');
                $btn.find('.dashicons').removeClass('dashicons-update sf-spinning').addClass('dashicons-yes-alt');
            }
        });
    }

    /**
     * Handle change fix button (re-open dropdown)
     */
    function handleChangeFix() {
        var $btn = $(this);
        var $row = $btn.closest('.sf-link-row');
        var $selector = $row.find('.sf-fix-selector');
        var brokenUrl = $row.data('broken-url');

        // Reset fix state
        if (BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl].action = '';
            BrokenLinks.fixes[brokenUrl].new_url = '';
        }

        // Hide all states and show dropdown
        $selector.find('.sf-manual-url-input').hide();
        $selector.find('.sf-ai-loading').hide();
        $selector.find('.sf-ai-result').hide();
        $selector.find('.sf-fix-selected-display').hide();
        $selector.find('.sf-fix-dropdown-wrapper').show();

        // Reset row state
        $row.removeClass('sf-fix-ready sf-fix-ignored');

        updateFixButton($row);
        updateBatchState();
    }

    /**
     * Handle single fix button click
     */
    function handleSingleFix() {
        var $btn = $(this);
        var $row = $btn.closest('.sf-link-row');
        var brokenUrl = $row.data('broken-url');
        var fix = BrokenLinks.fixes[brokenUrl];

        if (!fix || !fix.action || fix.action === 'ignore') {
            return;
        }

        if (fix.action === 'replace' && !fix.new_url) {
            showToast(sfBrokenLinksData.i18n.enterUrl || 'Please enter a replacement URL', 'warning');
            return;
        }

        $btn.addClass('sf-fixing');
        $btn.prop('disabled', true);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_apply_fixes',
                nonce: sfBrokenLinksData.nonce,
                fixes: [{
                    broken_url: brokenUrl,
                    action: fix.action,
                    new_url: fix.new_url || '',
                    post_ids: fix.post_ids || []
                }]
            },
            success: function(response) {
                if (response.success) {
                    $btn.removeClass('sf-fixing').addClass('sf-fixed');
                    $btn.text('Fixed!');
                    showToast(response.data.message || sfBrokenLinksData.i18n.fixApplied || 'Fix applied!', 'success');

                    // Fade out and remove row after delay
                    setTimeout(function() {
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            delete BrokenLinks.fixes[brokenUrl];
                            updateBatchState();
                            updateTableCounts();
                        });
                    }, 1000);
                } else {
                    $btn.removeClass('sf-fixing');
                    $btn.prop('disabled', false);
                    showToast(response.data.message || sfBrokenLinksData.i18n.fixesFailed, 'error');
                }
            },
            error: function() {
                $btn.removeClass('sf-fixing');
                $btn.prop('disabled', false);
                showToast(sfBrokenLinksData.i18n.fixesFailed, 'error');
            }
        });
    }

    /**
     * Handle Apply All Fixes button
     */
    function handleApplyAllFixes() {
        if (BrokenLinks.isProcessing) {
            return;
        }

        // Collect all valid fixes
        var fixesToApply = [];
        $.each(BrokenLinks.fixes, function(brokenUrl, fix) {
            if (fix.action && fix.action !== '' && fix.action !== 'ignore') {
                if (fix.action === 'replace' && !fix.new_url) {
                    return; // Skip invalid replace fixes
                }
                fixesToApply.push({
                    broken_url: brokenUrl,
                    action: fix.action,
                    new_url: fix.new_url || '',
                    post_ids: fix.post_ids || []
                });
            }
        });

        if (fixesToApply.length === 0) {
            showToast(sfBrokenLinksData.i18n.noFixesSelected || 'No fixes to apply', 'warning');
            return;
        }

        // Confirm
        var confirmMsg = (sfBrokenLinksData.i18n.confirmApply || 'Apply %d fixes?').replace('%d', fixesToApply.length);
        if (!confirm(confirmMsg)) {
            return;
        }

        BrokenLinks.isProcessing = true;
        showProgressModal(fixesToApply.length);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_apply_fixes',
                nonce: sfBrokenLinksData.nonce,
                fixes: fixesToApply
            },
            success: function(response) {
                hideProgressModal();

                if (response.success) {
                    var successCount = 0;

                    // Collect unique post IDs for potential "View Revisions" link
                    var uniquePostIds = [];
                    var postTitleMap = {};
                    $.each(fixesToApply, function(i, fix) {
                        var $row = $('.sf-link-row[data-broken-url="' + escapeSelector(fix.broken_url) + '"]');
                        if ($row.length) {
                            var postIds = ($row.data('post-ids') || '').toString().split(',');
                            $.each(postIds, function(j, pid) {
                                pid = parseInt(pid, 10);
                                if (pid && uniquePostIds.indexOf(pid) === -1) {
                                    uniquePostIds.push(pid);
                                    // Try to get post title from row data
                                    var title = $row.data('post-title') || $row.find('.sf-row-posts').text().trim();
                                    if (title) {
                                        postTitleMap[pid] = title;
                                    }
                                }
                            });
                        }
                    });

                    // Show toast with View Revisions link if single post, otherwise show Settings link
                    if (uniquePostIds.length === 1 && window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        var postId = uniquePostIds[0];
                        var postTitle = postTitleMap[postId] || 'Post';
                        window.ScreamingFixes.Toast.successWithRevisions(
                            response.data.message,
                            postId,
                            postTitle
                        );
                    } else {
                        showToast(response.data.message + ' <a href="' + sfBrokenLinksData.settingsUrl + '" style="color: inherit; text-decoration: underline;">View Change History &rarr;</a>', 'success');
                    }

                    // Build map of failed URLs with error details
                    var results = response.data.results || {};
                    var failedUrlErrors = {};
                    if (results.errors && results.errors.length) {
                        $.each(results.errors, function(i, err) {
                            if (err.broken_url) {
                                failedUrlErrors[err.broken_url] = {
                                    reason: err.error || '',
                                    suggestion: err.suggestion || ''
                                };
                            }
                        });
                    }

                    // Remove fixed rows from Fixable section (with animation), mark failed ones
                    $.each(fixesToApply, function(i, fix) {
                        var $row = $('.sf-link-row[data-broken-url="' + escapeSelector(fix.broken_url) + '"]');
                        var $panelRow = $('.sf-action-panel-row[data-for-url="' + escapeSelector(fix.broken_url) + '"]');

                        if ($row.length) {
                            if (failedUrlErrors[fix.broken_url]) {
                                // Failed — remove from Fixable (moves to Fixes Applied)
                                $panelRow.slideUp(200);
                                $row.removeClass('sf-panel-open');
                                $row.fadeOut(300, function() {
                                    $(this).remove();
                                    updateFixableSectionCount();
                                });
                                $panelRow.fadeOut(300, function() {
                                    $(this).remove();
                                });
                                delete BrokenLinks.fixes[fix.broken_url];
                            } else {
                                successCount++;

                                // Close any open panels
                                $panelRow.slideUp(200);
                                $row.removeClass('sf-panel-open');

                                // Fade out and remove the row
                                $row.fadeOut(300, function() {
                                    $(this).remove();
                                    updateFixableSectionCount();
                                });
                                $panelRow.fadeOut(300, function() {
                                    $(this).remove();
                                });

                                // Remove from fixes tracking
                                delete BrokenLinks.fixes[fix.broken_url];
                            }
                        }
                    });

                    // Update localStorage — clear if no remaining fixes, otherwise save remaining
                    if (Object.keys(BrokenLinks.fixes).length === 0) {
                        clearFixesFromStorage();
                    } else {
                        saveFixesToStorage();
                    }

                    // Refresh the Fixes Applied section via AJAX (updateFixedCount runs after AJAX completes)
                    refreshFixedSection();

                    // Update failed count for Can't Fix filter
                    updateFailedCount();

                    setTimeout(function() {
                        updateBatchState();
                    }, 350);
                } else {
                    showToast(response.data.message || sfBrokenLinksData.i18n.fixesFailed, 'error');
                }
            },
            error: function() {
                hideProgressModal();
                showToast(sfBrokenLinksData.i18n.fixesFailed, 'error');
            },
            complete: function() {
                BrokenLinks.isProcessing = false;
            }
        });
    }

    /**
     * Update the Fix button state for a row
     */
    function updateFixButton($row) {
        var brokenUrl = $row.data('broken-url');
        var fix = BrokenLinks.fixes[brokenUrl];
        var $btn = $row.find('.sf-fix-btn');

        var isValid = fix && fix.action && fix.action !== 'ignore';
        if (fix && fix.action === 'replace') {
            isValid = isValid && fix.new_url && isValidUrl(fix.new_url);
        }

        $btn.prop('disabled', !isValid);
    }

    /**
     * Update batch action state
     */
    function updateBatchState() {
        var fixCount = 0;

        $.each(BrokenLinks.fixes, function(brokenUrl, fix) {
            if (fix.action && fix.action !== '' && fix.action !== 'ignore') {
                if (fix.action === 'replace' && !fix.new_url) {
                    return;
                }
                fixCount++;
            }
        });

        // Update fixes count
        $('.sf-fixes-count').text(fixCount);

        // Update Apply All button
        var $applyBtn = $('.sf-apply-all-fixes');
        $applyBtn.prop('disabled', fixCount === 0);
        if (fixCount > 0) {
            $applyBtn.html('<span class="dashicons dashicons-yes"></span> Apply ' + fixCount + ' ' + (fixCount === 1 ? 'Fix' : 'Fixes'));
        } else {
            $applyBtn.html('<span class="dashicons dashicons-yes"></span> Apply All Fixes');
        }

        // Also update legacy button
        updateApplyButton();
    }

    /**
     * Validate URL format
     */
    function isValidUrl(string) {
        try {
            new URL(string);
            return true;
        } catch (_) {
            return false;
        }
    }

    // =====================================
    // END NEW DROPDOWN HANDLERS
    // =====================================

    // =====================================
    // EXPANDABLE PANEL HANDLERS
    // =====================================

    /**
     * Handle Edit button click - expand panel
     */
    function handleEditButtonClick(e) {
        e.preventDefault();
        var $btn = $(this);
        var $row = $btn.closest('.sf-link-row');
        var brokenUrl = $btn.data('broken-url');

        // Find the panel row for this broken URL
        var $panelRow = $row.next('.sf-action-panel-row');

        if (!$panelRow.length) {
            return;
        }

        // Close any other open panels first
        $('.sf-action-panel-row:visible').not($panelRow).each(function() {
            var $otherPanel = $(this);
            $otherPanel.slideUp(200);
            $otherPanel.prev('.sf-link-row').removeClass('sf-panel-open');
        });

        // Toggle this panel
        if ($panelRow.is(':visible')) {
            $panelRow.slideUp(200);
            $row.removeClass('sf-panel-open');
        } else {
            // Populate error info box if this row has failed
            var $errorInfo = $panelRow.find('.sf-panel-error-info');
            var errorReason = $row.data('error-reason');
            var errorSuggestion = $row.data('error-suggestion');

            if (errorReason || errorSuggestion) {
                if (errorReason) {
                    $errorInfo.find('.sf-error-info-reason').text(errorReason).show();
                } else {
                    $errorInfo.find('.sf-error-info-reason').hide();
                }
                if (errorSuggestion) {
                    $errorInfo.find('.sf-error-info-suggestion').text(errorSuggestion).show();
                } else {
                    $errorInfo.find('.sf-error-info-suggestion').hide();
                }
                $errorInfo.show();
            } else {
                $errorInfo.hide();
            }

            $panelRow.slideDown(200);
            $row.addClass('sf-panel-open');

            // Focus the URL input if empty
            var $urlInput = $panelRow.find('.sf-replacement-url');
            if (!$urlInput.val()) {
                setTimeout(function() {
                    $urlInput.focus();
                }, 250);
            }
        }
    }

    /**
     * Handle panel close button
     */
    function handlePanelClose(e) {
        e.preventDefault();
        var $btn = $(this);
        var $panelRow = $btn.closest('.sf-action-panel-row');
        var $row = $panelRow.prev('.sf-link-row');

        $panelRow.slideUp(200);
        $row.removeClass('sf-panel-open');
    }

    /**
     * Handle click on Failed badge in the Fixes Applied section - toggle inline error detail
     */
    function handleFixedFailedBadgeClick(e) {
        e.preventDefault();
        var $badge = $(this);
        var $row = $badge.closest('.sf-fixed-row');
        var $detailRow = $row.next('.sf-fixed-error-detail-row');

        // If detail row already exists, toggle it
        if ($detailRow.length) {
            $detailRow.slideToggle(200);
            $badge.toggleClass('sf-badge-expanded');
            return;
        }

        // Build error detail row from the title attribute (status_message)
        var statusMessage = $badge.attr('title') || '';
        var reason = '';
        var suggestion = '';

        if (statusMessage) {
            reason = statusMessage;
        } else {
            reason = 'This fix could not be applied.';
        }

        // Common troubleshooting suggestions based on error content
        if (!suggestion) {
            var lowerReason = reason.toLowerCase();
            if (lowerReason.indexOf('permission') !== -1) {
                suggestion = 'Your user role may not have permission to edit this content. Contact a site administrator.';
            } else if (lowerReason.indexOf('url not found in content') !== -1 || lowerReason.indexOf('not found in') !== -1) {
                suggestion = 'The broken URL may have already been changed, or it exists in a format the plugin couldn\'t match (e.g., encoded characters, relative URL, or inside a shortcode).';
            } else if (lowerReason.indexOf('could not update') !== -1 || lowerReason.indexOf('database') !== -1) {
                suggestion = 'There may be a database issue. Try again, or check your WordPress error log for details.';
            } else {
                suggestion = 'Try fixing this link manually by editing the post directly. If the issue persists, the link may be in a location the plugin cannot modify (e.g., a page builder, shortcode, or serialized data).';
            }
        }

        var detailHtml = '<tr class="sf-fixed-error-detail-row">' +
            '<td colspan="4">' +
            '<div class="sf-fixed-error-detail">' +
            '<div class="sf-error-info-icon">&#9888;&#65039;</div>' +
            '<div class="sf-error-info-content">' +
            '<strong class="sf-error-info-reason">' + $('<span>').text(reason).html() + '</strong>' +
            '<span class="sf-error-info-suggestion">' + $('<span>').text(suggestion).html() + '</span>' +
            '</div>' +
            '</div>' +
            '</td>' +
            '</tr>';

        $row.after(detailHtml);
        $row.next('.sf-fixed-error-detail-row').hide().slideDown(200);
        $badge.addClass('sf-badge-expanded');
    }

    /**
     * Handle panel action buttons (Remove, Ignore, Get AI)
     */
    function handlePanelAction(e) {
        e.preventDefault();
        var $btn = $(this);
        var action = $btn.data('action');
        var $panelRow = $btn.closest('.sf-action-panel-row');
        var $row = $panelRow.prev('.sf-link-row');
        var brokenUrl = $row.data('broken-url');
        var $panel = $panelRow.find('.sf-action-panel');

        if (!brokenUrl || !action) {
            return;
        }

        // Initialize fix data if needed
        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        switch (action) {
            case 'remove_link':
                // Safety guard: prevent remove for image URLs
                var isImageUrl = $row.data('is-image') === true || $row.data('is-image') === 'true';
                if (isImageUrl) {
                    showToast('Remove broken images manually in the post editor for safety.', 'info');
                    return;
                }
                BrokenLinks.fixes[brokenUrl].action = 'remove_link';
                BrokenLinks.fixes[brokenUrl].new_url = '';
                // Stage the fix (don't apply yet)
                stagePanelFix($row, $panelRow, brokenUrl, 'Remove Hyperlink');
                break;

            case 'ignore':
                BrokenLinks.fixes[brokenUrl].action = 'ignore';
                BrokenLinks.fixes[brokenUrl].new_url = '';
                // Mark as ignored and close panel
                $row.addClass('sf-row-ignored');
                updateRowStatusDisplay($row, 'Ignored', 'sf-status-ignored');
                $panelRow.slideUp(200);
                $row.removeClass('sf-panel-open');
                updateBatchState();
                saveFixesToStorage();
                break;

            case 'ai_suggest':
                // Show AI loading state
                $panel.find('.sf-panel-ai-loading').show();
                $panel.find('.sf-panel-ai-result').hide();
                fetchPanelAiSuggestion($row, $panelRow, brokenUrl);
                break;
        }
    }

    /**
     * Fetch AI suggestion for panel
     */
    function fetchPanelAiSuggestion($row, $panelRow, brokenUrl) {
        var $panel = $panelRow.find('.sf-action-panel');
        var $loading = $panel.find('.sf-panel-ai-loading');
        var sourceCount = $row.data('source-count') || 1;

        // Fun loading messages
        var loadingMessages = [
            'Consulting the URL oracle...',
            'Searching the archives...',
            'Finding your perfect match...',
            'Analyzing the void...',
            'Hunting for alternatives...',
            'Mining the internet...',
            'Decoding the matrix...',
            'Channeling digital wisdom...',
            'Scanning the web archives...'
        ];

        // Rotate loading messages
        var messageIndex = 0;
        $loading.find('.sf-ai-loading-text').text(loadingMessages[messageIndex]);
        var messageInterval = setInterval(function() {
            messageIndex = (messageIndex + 1) % loadingMessages.length;
            $loading.find('.sf-ai-loading-text').text(loadingMessages[messageIndex]);
        }, 2000);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_get_ai_suggestion',
                nonce: sfBrokenLinksData.nonce,
                url: brokenUrl,
                post_count: sourceCount
            },
            success: function(response) {
                clearInterval(messageInterval);
                $loading.hide();

                if (response.success && response.data.suggestion) {
                    var suggestion = response.data.suggestion;

                    if (suggestion.action === 'replace' && suggestion.new_url) {
                        // Show AI result with clickable URL
                        var $result = $panel.find('.sf-panel-ai-result');
                        var $urlLink = $result.find('.sf-ai-url');
                        $urlLink.text(suggestion.new_url);
                        $urlLink.attr('href', suggestion.new_url);
                        $result.data('suggested-url', suggestion.new_url);
                        $result.show();

                        // Also populate the manual input
                        $panel.find('.sf-replacement-url').val(suggestion.new_url);
                    } else if (suggestion.action === 'remove_link') {
                        // Stage the remove_link fix with AI badge
                        if (!BrokenLinks.fixes[brokenUrl]) {
                            BrokenLinks.fixes[brokenUrl] = {
                                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
                            };
                        }
                        BrokenLinks.fixes[brokenUrl].action = 'remove_link';
                        BrokenLinks.fixes[brokenUrl].new_url = '';

                        // Stage the fix with AI badge
                        stagePanelFix($row, $panelRow, brokenUrl, 'Remove Hyperlink', true, null);
                        showToast('AI suggests removing this link. Click "Apply All Fixes" to execute.', 'success');
                    } else {
                        showToast(sfBrokenLinksData.i18n.noSuggestion || 'No replacement found. Please enter manually.', 'warning');
                    }
                } else {
                    showToast(response.data.message || sfBrokenLinksData.i18n.aiFailed, 'error');
                }
            },
            error: function() {
                clearInterval(messageInterval);
                $loading.hide();
                showToast(sfBrokenLinksData.i18n.aiFailed, 'error');
            }
        });
    }

    /**
     * Handle "Use" button for AI suggestion
     */
    function handleUseAiSuggestion(e) {
        e.preventDefault();
        var $btn = $(this);
        var $result = $btn.closest('.sf-panel-ai-result');
        var $panelRow = $btn.closest('.sf-action-panel-row');
        var $row = $panelRow.prev('.sf-link-row');
        var brokenUrl = $row.data('broken-url');
        var suggestedUrl = $result.data('suggested-url');

        if (!suggestedUrl) {
            return;
        }

        // Set the fix
        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        BrokenLinks.fixes[brokenUrl].action = 'replace';
        BrokenLinks.fixes[brokenUrl].new_url = suggestedUrl;

        // Stage the fix (with AI badge since it came from AI suggestion)
        stagePanelFix($row, $panelRow, brokenUrl, 'Replace', true, suggestedUrl);
    }

    /**
     * Handle URL input change in panel
     */
    function handlePanelUrlInput() {
        var $input = $(this);
        var url = $input.val().trim();

        // Simple validation feedback
        if (url && isValidUrl(url)) {
            $input.css('border-color', 'var(--sf-success, #22c55e)');
        } else if (url) {
            $input.css('border-color', 'var(--sf-error, #ef4444)');
        } else {
            $input.css('border-color', '');
        }
    }

    /**
     * Handle Save button click in panel
     */
    function handlePanelSave(e) {
        e.preventDefault();
        var $btn = $(this);
        var $panelRow = $btn.closest('.sf-action-panel-row');
        var $row = $panelRow.prev('.sf-link-row');
        var $panel = $panelRow.find('.sf-action-panel');
        var brokenUrl = $row.data('broken-url');
        var newUrl = $panel.find('.sf-replacement-url').val().trim();

        if (!newUrl) {
            showToast(sfBrokenLinksData.i18n.enterUrl || 'Please enter a replacement URL', 'warning');
            $panel.find('.sf-replacement-url').focus();
            return;
        }

        if (!isValidUrl(newUrl)) {
            showToast(sfBrokenLinksData.i18n.invalidUrl || 'Please enter a valid URL', 'warning');
            $panel.find('.sf-replacement-url').focus();
            return;
        }

        // Initialize fix data if needed
        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        BrokenLinks.fixes[brokenUrl].action = 'replace';
        BrokenLinks.fixes[brokenUrl].new_url = newUrl;

        // Stage the fix (don't apply yet)
        stagePanelFix($row, $panelRow, brokenUrl, 'Replace', false, newUrl);
    }

    /**
     * Stage a fix (save selection without applying)
     * Updates UI to show fix is ready, closes panel
     *
     * @param {jQuery} $row - The table row
     * @param {jQuery} $panelRow - The panel row
     * @param {string} brokenUrl - The broken URL
     * @param {string} statusText - Status text to display (e.g., "Replace", "Remove")
     * @param {boolean} isAiSuggested - Whether this fix was suggested by AI
     * @param {string} fullUrl - Optional full URL for clickable link display
     */
    function stagePanelFix($row, $panelRow, brokenUrl, statusText, isAiSuggested, fullUrl) {
        // Update row to show it's staged
        $row.addClass('sf-row-staged');

        // Mark if AI suggested
        if (isAiSuggested) {
            $row.addClass('sf-ai-suggested');
        }

        updateRowStatusDisplay($row, statusText, 'sf-status-staged', isAiSuggested, fullUrl);

        // Close panel
        $panelRow.slideUp(200);
        $row.removeClass('sf-panel-open');

        // Update batch state to reflect new staged fix
        updateBatchState();

        // Save to localStorage for persistence across pages
        saveFixesToStorage();

        // Show confirmation toast (only if not batch processing)
        if (!BrokenLinks.isProcessing) {
            showToast('Fix staged. Click "Apply All Fixes" to execute.', 'success');
        }
    }

    /**
     * Apply fix from panel and update UI
     */
    function applyPanelFix($row, $panelRow, brokenUrl) {
        var fix = BrokenLinks.fixes[brokenUrl];

        if (!fix || !fix.action) {
            return;
        }

        // Show loading state
        $panelRow.find('.sf-save-btn').prop('disabled', true).text('Fixing...');

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_apply_fixes',
                nonce: sfBrokenLinksData.nonce,
                fixes: [{
                    broken_url: brokenUrl,
                    action: fix.action,
                    new_url: fix.new_url || '',
                    post_ids: fix.post_ids || []
                }]
            },
            success: function(response) {
                if (response.success) {
                    var results = response.data.results || {};
                    var successCount = results.success || 0;

                    if (successCount > 0) {
                        showToast(response.data.message || sfBrokenLinksData.i18n.fixApplied || 'Fix applied!', 'success');

                        // Update row status
                        $row.addClass('sf-row-fixed');
                        updateRowStatusDisplay($row, 'Fixed', 'sf-status-fixed');

                        // Close panel
                        $panelRow.slideUp(200);
                        $row.removeClass('sf-panel-open');

                        // Refresh fixed section (updateFixedCount runs after AJAX completes)
                        refreshFixedSection();

                        // Fade out row after a delay
                        setTimeout(function() {
                            $row.fadeOut(300, function() {
                                $panelRow.remove();
                                $(this).remove();
                                delete BrokenLinks.fixes[brokenUrl];
                                updateBatchState();
                                updateTableCounts();
                            });
                        }, 1000);
                    } else {
                        // Fix failed — remove from Fixable, it moves to Fixes Applied
                        showToast(response.data.message || sfBrokenLinksData.i18n.fixesFailed, 'error');
                        $panelRow.slideUp(200);
                        $row.removeClass('sf-panel-open');
                        refreshFixedSection();
                        setTimeout(function() {
                            $row.fadeOut(300, function() {
                                $panelRow.remove();
                                $(this).remove();
                                delete BrokenLinks.fixes[brokenUrl];
                                updateBatchState();
                                updateTableCounts();
                            });
                        }, 500);
                    }
                } else {
                    // Server error — remove from Fixable, it moves to Fixes Applied
                    showToast(response.data.message || sfBrokenLinksData.i18n.fixesFailed, 'error');
                    $panelRow.slideUp(200);
                    $row.removeClass('sf-panel-open');
                    refreshFixedSection();
                    setTimeout(function() {
                        $row.fadeOut(300, function() {
                            $panelRow.remove();
                            $(this).remove();
                            delete BrokenLinks.fixes[brokenUrl];
                            updateBatchState();
                            updateTableCounts();
                        });
                    }, 500);
                }
            },
            error: function() {
                // Network error — keep row in place since nothing was saved server-side
                $panelRow.find('.sf-save-btn').prop('disabled', false).html('&#10003; Save');
                showToast(sfBrokenLinksData.i18n.fixesFailed, 'error');
            }
        });
    }

    /**
     * Update row status display
     *
     * @param {jQuery} $row - The table row
     * @param {string} statusText - Status text to display (e.g., "Replace", "Remove")
     * @param {string} statusClass - CSS class for status styling
     * @param {boolean} isAiSuggested - Whether this was an AI suggestion
     * @param {string} fullUrl - Optional full URL to display as clickable link
     */
    function updateRowStatusDisplay($row, statusText, statusClass, isAiSuggested, fullUrl) {
        var $status = $row.find('.sf-row-status');
        $status.removeClass('sf-status-pending sf-status-fixed sf-status-ignored sf-status-staged')
               .addClass(statusClass);

        // Remove any existing AI badge and URL wrapper
        $status.find('.sf-ai-badge').remove();
        $status.find('.sf-status-url-wrapper').remove();

        // Update status text
        var $statusText = $status.find('.sf-status-text');
        $statusText.text(statusText);

        // Add AI badge if this was an AI suggestion
        if (isAiSuggested) {
            var aiBadge = '<span class="sf-ai-badge"><span class="sf-ai-badge-icon">&#129302;</span>AI</span>';
            $statusText.after(aiBadge);
        }

        // Add clickable URL link if provided
        if (fullUrl) {
            var shortUrl = fullUrl.length > 30 ? fullUrl.substring(0, 27) + '...' : fullUrl;
            var urlHtml = '<div class="sf-status-url-wrapper">' +
                '<a href="' + escapeHtml(fullUrl) + '" target="_blank" rel="noopener" class="sf-status-url-link" title="' + escapeHtml(fullUrl) + '">' +
                escapeHtml(shortUrl) + '</a></div>';
            $status.append(urlHtml);
        }
    }

    // =====================================
    // END EXPANDABLE PANEL HANDLERS
    // =====================================

    /**
     * Handle fix action dropdown change
     */
    function handleFixActionChange() {
        var $select = $(this);
        var $row = $select.closest('.sf-link-row');
        var brokenUrl = $select.data('broken-url');
        var action = $select.val();

        // Store the fix
        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }
        BrokenLinks.fixes[brokenUrl].action = action;

        // Show/hide new URL input for replace action
        var $newUrlInput = $row.find('.sf-new-url-input');
        if (action === 'replace') {
            $newUrlInput.slideDown(150);
            $row.find('.sf-new-url').focus();
        } else {
            $newUrlInput.slideUp(150);
            BrokenLinks.fixes[brokenUrl].new_url = '';
        }

        // Update row status
        updateRowStatus($row, action);
        updateApplyButton();
    }

    /**
     * Handle new URL input change
     */
    function handleNewUrlChange() {
        var $input = $(this);
        var brokenUrl = $input.data('broken-url');
        var newUrl = $input.val().trim();

        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {};
        }
        BrokenLinks.fixes[brokenUrl].new_url = newUrl;

        updateApplyButton();
    }

    /**
     * Handle single AI suggestion request
     */
    function handleAiSuggest() {
        var $btn = $(this);
        var $row = $btn.closest('.sf-link-row');
        var brokenUrl = $btn.data('broken-url');
        var sourceCount = $btn.data('source-count');

        if (BrokenLinks.isProcessing) {
            return;
        }

        // Show loading state
        $btn.addClass('sf-loading');
        $btn.find('.dashicons').removeClass('dashicons-welcome-learn-more').addClass('dashicons-update sf-spinning');

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_get_ai_suggestion',
                nonce: sfBrokenLinksData.nonce,
                url: brokenUrl,
                post_count: sourceCount
            },
            success: function(response) {
                if (response.success && response.data.suggestion) {
                    applyAiSuggestion($row, response.data.suggestion);
                    showToast(sfBrokenLinksData.i18n.aiComplete, 'success');
                } else {
                    showToast(response.data.message || sfBrokenLinksData.i18n.aiFailed, 'error');
                }
            },
            error: function() {
                showToast(sfBrokenLinksData.i18n.aiFailed, 'error');
            },
            complete: function() {
                $btn.removeClass('sf-loading');
                $btn.find('.dashicons').removeClass('dashicons-update sf-spinning').addClass('dashicons-welcome-learn-more');
            }
        });
    }

    /**
     * Apply AI suggestion to a row (new UI)
     */
    function applyAiSuggestionNewUI($row, suggestion) {
        var brokenUrl = $row.data('broken-url');
        var $selector = $row.find('.sf-fix-selector');

        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        // Hide dropdown
        $selector.find('.sf-fix-dropdown-wrapper').hide();
        $selector.find('.sf-manual-url-input').hide();
        $selector.find('.sf-ai-loading').hide();

        if (suggestion.action === 'replace' && suggestion.new_url) {
            BrokenLinks.fixes[brokenUrl].action = 'replace';
            BrokenLinks.fixes[brokenUrl].new_url = suggestion.new_url;
            showAiResult($row, suggestion.new_url);
        } else if (suggestion.action === 'remove_link') {
            BrokenLinks.fixes[brokenUrl].action = 'remove_link';
            BrokenLinks.fixes[brokenUrl].new_url = '';
            showFixSelectedState($row, 'remove_link');
        } else {
            // Unknown - show dropdown again
            $selector.find('.sf-fix-dropdown-wrapper').show();
            $row.addClass('sf-needs-review');
        }

        updateFixButton($row);
    }

    /**
     * Apply AI suggestion to a row
     */
    function applyAiSuggestion($row, suggestion) {
        var brokenUrl = $row.data('broken-url');
        var $select = $row.find('.sf-fix-action');
        var $newUrlInput = $row.find('.sf-new-url');
        var $newUrlWrapper = $row.find('.sf-new-url-input');

        if (!BrokenLinks.fixes[brokenUrl]) {
            BrokenLinks.fixes[brokenUrl] = {
                post_ids: ($row.data('post-ids') || '').toString().split(',').filter(Boolean)
            };
        }

        if (suggestion.action === 'replace' && suggestion.new_url) {
            $select.val('replace');
            $newUrlWrapper.show();
            $newUrlInput.val(suggestion.new_url);
            BrokenLinks.fixes[brokenUrl].action = 'replace';
            BrokenLinks.fixes[brokenUrl].new_url = suggestion.new_url;
        } else if (suggestion.action === 'remove_link') {
            $select.val('remove_link');
            $newUrlWrapper.hide();
            BrokenLinks.fixes[brokenUrl].action = 'remove_link';
            BrokenLinks.fixes[brokenUrl].new_url = '';
        } else if (suggestion.action === 'unknown') {
            // Mark as needing manual review
            $row.addClass('sf-needs-review');
        }

        updateRowStatus($row, suggestion.action);
        updateApplyButton();
    }

    /**
     * Handle row selection
     */
    function handleRowSelect() {
        var $checkbox = $(this);
        var index = $checkbox.val();
        var isChecked = $checkbox.is(':checked');

        if (isChecked) {
            if (BrokenLinks.selectedRows.indexOf(index) === -1) {
                BrokenLinks.selectedRows.push(index);
            }
        } else {
            BrokenLinks.selectedRows = BrokenLinks.selectedRows.filter(function(i) {
                return i !== index;
            });
        }

        updateSelectedCount();
        updateSelectAllState();
    }

    /**
     * Handle select all checkbox
     */
    function handleSelectAll() {
        var isChecked = $(this).is(':checked');

        $('.sf-row-select').prop('checked', isChecked);

        if (isChecked) {
            BrokenLinks.selectedRows = [];
            $('.sf-row-select').each(function() {
                BrokenLinks.selectedRows.push($(this).val());
            });
        } else {
            BrokenLinks.selectedRows = [];
        }

        updateSelectedCount();
    }

    /**
     * Update select all checkbox state
     */
    function updateSelectAllState() {
        var total = $('.sf-row-select').length;
        var checked = $('.sf-row-select:checked').length;

        $('#sf-select-all').prop('checked', total > 0 && checked === total);
        $('#sf-select-all').prop('indeterminate', checked > 0 && checked < total);
    }

    /**
     * Update selected count display
     */
    function updateSelectedCount() {
        $('.sf-selected-count .sf-count').text(BrokenLinks.selectedRows.length);
    }

    /**
     * Update row visual status
     */
    function updateRowStatus($row, action) {
        var $status = $row.find('.sf-fix-status');
        var $icon = $status.find('.sf-status-icon');

        $row.removeClass('sf-fix-set sf-fix-ignore sf-fix-remove');

        if (action === 'replace') {
            $row.addClass('sf-fix-set');
            $icon.html('<span class="dashicons dashicons-yes-alt" style="color: var(--sf-success);"></span>');
        } else if (action === 'remove_link' || action === 'remove_all') {
            $row.addClass('sf-fix-remove');
            $icon.html('<span class="dashicons dashicons-minus" style="color: var(--sf-warning);"></span>');
        } else if (action === 'ignore') {
            $row.addClass('sf-fix-ignore');
            $icon.html('<span class="dashicons dashicons-dismiss" style="color: var(--sf-gray-400);"></span>');
        } else {
            $icon.html('');
        }
    }

    /**
     * Update Apply Fixes button state
     */
    function updateApplyButton() {
        var fixCount = 0;

        $.each(BrokenLinks.fixes, function(brokenUrl, fix) {
            if (fix.action && fix.action !== '' && fix.action !== 'ignore') {
                // For replace action, need a new URL
                if (fix.action === 'replace' && !fix.new_url) {
                    return;
                }
                fixCount++;
            }
        });

        var $btn = $('.sf-apply-fixes');
        $btn.prop('disabled', fixCount === 0);

        if (fixCount > 0) {
            $btn.html('<span class="dashicons dashicons-yes"></span> Apply ' + fixCount + ' ' + (fixCount === 1 ? 'Fix' : 'Fixes'));
        } else {
            $btn.html('<span class="dashicons dashicons-yes"></span> Apply Fixes');
        }
    }

    /**
     * Handle Apply Fixes button click
     */
    function handleApplyFixes() {
        if (BrokenLinks.isProcessing) {
            return;
        }

        // Collect fixes to apply
        var fixesToApply = [];
        $.each(BrokenLinks.fixes, function(brokenUrl, fix) {
            if (fix.action && fix.action !== '' && fix.action !== 'ignore') {
                if (fix.action === 'replace' && !fix.new_url) {
                    return;
                }
                fixesToApply.push({
                    broken_url: brokenUrl,
                    action: fix.action,
                    new_url: fix.new_url || '',
                    post_ids: fix.post_ids || []
                });
            }
        });

        if (fixesToApply.length === 0) {
            showToast(sfBrokenLinksData.i18n.noFixesSelected, 'warning');
            return;
        }

        // Confirm
        var confirmMsg = sfBrokenLinksData.i18n.confirmApply.replace('%d', fixesToApply.length);
        if (!confirm(confirmMsg)) {
            return;
        }

        BrokenLinks.isProcessing = true;
        showProgressModal(fixesToApply.length);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_apply_fixes',
                nonce: sfBrokenLinksData.nonce,
                fixes: fixesToApply
            },
            success: function(response) {
                hideProgressModal();

                if (response.success) {
                    showToast(response.data.message, 'success');

                    // Remove fixed rows from table
                    $.each(fixesToApply, function(i, fix) {
                        var $row = $('.sf-link-row[data-broken-url="' + escapeSelector(fix.broken_url) + '"]');
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            updateTableCounts();
                        });
                    });

                    // Clear applied fixes from state
                    $.each(fixesToApply, function(i, fix) {
                        delete BrokenLinks.fixes[fix.broken_url];
                    });

                    updateApplyButton();
                } else {
                    showToast(response.data.message || sfBrokenLinksData.i18n.fixesFailed, 'error');
                }
            },
            error: function() {
                hideProgressModal();
                showToast(sfBrokenLinksData.i18n.fixesFailed, 'error');
            },
            complete: function() {
                BrokenLinks.isProcessing = false;
            }
        });
    }

    /**
     * Handle Export button click
     */
    function handleExport() {
        var $btn = $(this);

        if (BrokenLinks.isProcessing) {
            return;
        }

        $btn.addClass('sf-loading');
        $btn.prop('disabled', true);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_export',
                nonce: sfBrokenLinksData.nonce
            },
            success: function(response) {
                if (response.success && response.data.csv) {
                    // Create download
                    var blob = new Blob([response.data.csv], { type: 'text/csv;charset=utf-8;' });
                    var link = document.createElement('a');
                    var url = URL.createObjectURL(blob);

                    link.setAttribute('href', url);
                    link.setAttribute('download', response.data.filename);
                    link.style.visibility = 'hidden';

                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);

                    showToast(sfBrokenLinksData.i18n.exportComplete, 'success');
                } else {
                    showToast(response.data.message || 'Export failed', 'error');
                }
            },
            error: function() {
                showToast('Export failed', 'error');
            },
            complete: function() {
                $btn.removeClass('sf-loading');
                $btn.prop('disabled', false);
            }
        });
    }

    /**
     * Handle Export Fixed Links button click
     */
    function handleExportFixedLinks() {
        var $btn = $(this);

        if (BrokenLinks.isProcessing) {
            return;
        }

        $btn.addClass('sf-loading');
        $btn.prop('disabled', true);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_export_fixed',
                nonce: sfBrokenLinksData.nonce
            },
            success: function(response) {
                if (response.success && response.data.csv) {
                    // Create download
                    var blob = new Blob([response.data.csv], { type: 'text/csv;charset=utf-8;' });
                    var link = document.createElement('a');
                    var url = URL.createObjectURL(blob);

                    link.setAttribute('href', url);
                    link.setAttribute('download', response.data.filename);
                    link.style.visibility = 'hidden';

                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);

                    showToast('Fixed links exported successfully', 'success');
                } else {
                    showToast(response.data.message || 'Export failed', 'error');
                }
            },
            error: function() {
                showToast('Export failed', 'error');
            },
            complete: function() {
                $btn.removeClass('sf-loading');
                $btn.prop('disabled', false);
            }
        });
    }

    /**
     * Handle Clear Results button click
     */
    function handleClearResults() {
        var confirmMsg = sfBrokenLinksData.i18n.confirmClear ||
            'Are you sure you want to clear all results? This cannot be undone.';

        if (!confirm(confirmMsg)) {
            return;
        }

        var $btn = $(this);
        $btn.prop('disabled', true);

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_clear',
                nonce: sfBrokenLinksData.nonce
            },
            success: function(response) {
                if (response.success) {
                    // Clear localStorage fixes as well
                    clearFixesFromStorage();
                    BrokenLinks.fixes = {};

                    showToast(sfBrokenLinksData.i18n.resultsCleared || 'Results cleared.', 'success');
                    // Reload page to show empty state
                    setTimeout(function() {
                        window.location.reload();
                    }, 500);
                } else {
                    showToast(response.data.message || 'Failed to clear results.', 'error');
                    $btn.prop('disabled', false);
                }
            },
            error: function() {
                showToast('Failed to clear results.', 'error');
                $btn.prop('disabled', false);
            }
        });
    }

    /**
     * Handle CSV upload completion
     */
    function handleCsvUploaded(event, data) {
        if (data.module === 'broken-links') {
            // Redirect with upload_id to trigger CSV processing
            if (data.upload_id) {
                var url = window.location.href.split('?')[0] +
                    '?page=screaming-fixes&tab=broken-links&upload_id=' + data.upload_id;
                window.location.href = url;
            } else {
                // Fallback to reload
                window.location.reload();
            }
        }
    }

    /**
     * Update table row counts after fixes
     */
    function updateTableCounts() {
        var remaining = $('.sf-link-row:visible').length;

        // Update header stats
        $('.sf-module-stats .sf-stat:first-child strong').text(remaining);

        // Update results header
        $('.sf-results-header h3').html(
            '<span class="dashicons dashicons-warning"></span> ' +
            remaining + ' Broken Links Found'
        );

        // Show empty state if no links remain
        if (remaining === 0) {
            $('.sf-results-section').fadeOut(300, function() {
                $('.sf-empty-state').fadeIn(300);
            });
        }
    }

    /**
     * Transform a row to show fixed state
     *
     * @param {jQuery} $row - The table row
     * @param {Object} fix - The fix that was applied
     */
    function transformRowToFixed($row, fix) {
        var brokenUrl = fix.broken_url;
        var action = fix.action;
        var newUrl = fix.new_url || '';

        // Add fixed class for styling
        $row.addClass('sf-row-fixed-complete');
        $row.removeClass('sf-row-staged sf-row-ignored');

        // Update the URL cell to show transformation
        var $urlCell = $row.find('.sf-col-broken-url .sf-url-cell');
        var displayBrokenUrl = brokenUrl.length > 35 ? brokenUrl.substring(0, 32) + '...' : brokenUrl;

        var transformationHtml = '';
        if (action === 'replace' && newUrl) {
            var displayNewUrl = newUrl.length > 35 ? newUrl.substring(0, 32) + '...' : newUrl;
            transformationHtml =
                '<div class="sf-url-transformation">' +
                    '<span class="sf-url-old" title="' + escapeHtml(brokenUrl) + '">' + escapeHtml(displayBrokenUrl) + '</span>' +
                    '<span class="sf-url-arrow">&rarr;</span>' +
                    '<a href="' + escapeHtml(newUrl) + '" target="_blank" rel="noopener" class="sf-url-new" title="' + escapeHtml(newUrl) + '">' + escapeHtml(displayNewUrl) + '</a>' +
                '</div>';
        } else if (action === 'remove_link') {
            transformationHtml =
                '<div class="sf-url-transformation">' +
                    '<span class="sf-url-old" title="' + escapeHtml(brokenUrl) + '">' + escapeHtml(displayBrokenUrl) + '</span>' +
                    '<span class="sf-url-arrow">&rarr;</span>' +
                    '<span class="sf-url-removed">[link removed]</span>' +
                '</div>';
        } else if (action === 'remove_all') {
            transformationHtml =
                '<div class="sf-url-transformation">' +
                    '<span class="sf-url-old" title="' + escapeHtml(brokenUrl) + '">' + escapeHtml(displayBrokenUrl) + '</span>' +
                    '<span class="sf-url-arrow">&rarr;</span>' +
                    '<span class="sf-url-removed">[removed]</span>' +
                '</div>';
        }

        $urlCell.html(transformationHtml);

        // Update the status column
        var statusText = '';
        var statusIcon = '&#10003;'; // checkmark
        var statusLink = '';
        switch (action) {
            case 'replace':
                statusText = 'Replaced';
                // Add clickable link to the new URL in status column
                if (newUrl) {
                    var shortNewUrl = newUrl.length > 25 ? newUrl.substring(0, 22) + '...' : newUrl;
                    statusLink = '<a href="' + escapeHtml(newUrl) + '" target="_blank" rel="noopener" class="sf-status-url-link" title="' + escapeHtml(newUrl) + '">' + escapeHtml(shortNewUrl) + '</a>';
                }
                break;
            case 'remove_link':
                statusText = 'Unlinked';
                break;
            case 'remove_all':
                statusText = 'Removed';
                break;
            default:
                statusText = 'Fixed';
        }

        var $status = $row.find('.sf-row-status');
        $status.removeClass('sf-status-pending sf-status-staged sf-status-ignored')
               .addClass('sf-status-fixed-complete');

        var statusHtml = '<span class="sf-status-icon-fixed">' + statusIcon + '</span>' +
            '<span class="sf-status-text">' + statusText + '</span>';

        // Add the clickable URL link below the status text
        if (statusLink) {
            statusHtml += '<div class="sf-status-url-wrapper">' + statusLink + '</div>';
        }

        $status.html(statusHtml);

        // Update the action column - replace Edit button with page count
        var sourceCount = $row.data('source-count') || 1;
        var $actionCell = $row.find('.sf-col-fix-action');
        $actionCell.html(
            '<span class="sf-fixed-pages">' +
                '<span class="sf-fixed-icon">&#10003;</span> ' +
                sourceCount + ' ' + (sourceCount === 1 ? 'page' : 'pages') +
            '</span>'
        );

        // Update the Found In column text
        var $foundIn = $row.find('.sf-col-found-in');
        var $sourceLabel = $foundIn.find('.sf-source-count-label, .sf-sources-toggle');
        if ($sourceLabel.length) {
            var newLabel = 'Fixed on ' + sourceCount + ' ' + (sourceCount === 1 ? 'page' : 'pages');
            if ($sourceLabel.hasClass('sf-sources-toggle')) {
                $sourceLabel.html(newLabel + ' <span class="dashicons dashicons-arrow-down-alt2"></span>');
            } else {
                $sourceLabel.text(newLabel);
            }
        }

        // Animate the transformation
        $row.css('opacity', '0.5');
        setTimeout(function() {
            $row.css('opacity', '1');
        }, 100);
    }

    /**
     * Escape HTML special characters
     */
    function escapeHtml(str) {
        if (!str) return '';
        return str
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    /**
     * Update the fixed count display
     */
    function updateFixedCount() {
        var fixedCount = $('.sf-link-row.sf-row-fixed-complete').length;
        var pendingCount = $('.sf-link-row:not(.sf-row-fixed-complete)').length;
        var totalCount = fixedCount + pendingCount;

        // Update the section header to show progress
        var $sectionHeader = $('.sf-section-fixable .sf-results-header h3');
        if (fixedCount > 0) {
            $sectionHeader.html(
                '<span class="sf-section-badge sf-badge-fixable">&#10003;</span>' +
                pendingCount + ' Fixable Links' +
                '<span class="sf-fixed-counter">' +
                    '<span class="sf-fixed-count-icon">&#10003;</span> ' +
                    fixedCount + ' fixed' +
                '</span>' +
                '<span class="sf-section-hint">Links in post/page content - can be auto-fixed</span>'
            );
        }

        // Update the stat cards
        var $fixableStat = $('.sf-stat-card.sf-stat-fixable .sf-stat-number');
        if ($fixableStat.length) {
            $fixableStat.text(pendingCount);
        }

        // Update the Fixed filter button count and enable/disable state
        var $fixedFilter = $('.sf-filter-stat.sf-filter-fixed');
        if ($fixedFilter.length) {
            // Also count items in the Fixed Links section if it exists
            var fixedSectionCount = $('.sf-section-fixed .sf-fixed-row').length;
            var totalFixed = fixedCount + fixedSectionCount;

            // Update the count
            $fixedFilter.find('.sf-fixed-filter-count').text(totalFixed);

            // Enable/disable based on whether there are any fixed items
            if (totalFixed > 0) {
                $fixedFilter.removeClass('sf-filter-disabled');
            } else {
                $fixedFilter.addClass('sf-filter-disabled');
            }
        }
    }

    /**
     * Update the Can't Fix filter button count
     */
    function updateFailedCount() {
        var failedCount = $('.sf-link-row.sf-row-failed').length;
        var $failedFilter = $('.sf-filter-stat.sf-filter-failed');
        if ($failedFilter.length) {
            $failedFilter.find('.sf-failed-filter-count').text(failedCount);
            if (failedCount > 0) {
                $failedFilter.removeClass('sf-filter-disabled');
            } else {
                $failedFilter.addClass('sf-filter-disabled');
            }
        }
    }

    /**
     * Show inline error reason and suggestion on a failed row
     */
    function showInlineError($row, reason, suggestion) {
        var $status = $row.find('.sf-fix-status');
        $status.show();
        $status.find('.sf-fix-status-success').hide();
        var $failed = $status.find('.sf-fix-status-failed');
        $failed.show();

        // Store error details on the row for display in the edit panel
        if (reason) {
            $row.data('error-reason', reason);
        }
        if (suggestion) {
            $row.data('error-suggestion', suggestion);
        }
    }

    /**
     * Update the Fixable Links section header count after rows are removed
     */
    function updateFixableSectionCount() {
        var $section = $('.sf-section-fixable');
        var $toggle = $section.find('.sf-section-toggle');
        var remainingCount = $section.find('.sf-link-row:not(.sf-row-fixed-complete)').length;

        // Update the toggle button text - find and update the count
        var $badge = $toggle.find('.sf-section-badge');
        var $hint = $toggle.find('.sf-section-hint');
        var $icon = $toggle.find('.sf-toggle-icon');

        // Rebuild the toggle content with updated count
        var badgeHtml = $badge.length ? $badge.prop('outerHTML') : '<span class="sf-section-badge sf-badge-fixable">&#10003;</span>';
        var hintHtml = $hint.length ? $hint.prop('outerHTML') : '<span class="sf-section-hint">Links in post/page content - can be auto-fixed</span>';
        var iconHtml = $icon.length ? $icon.prop('outerHTML') : '<span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>';

        var labelText = remainingCount + ' Fixable Link' + (remainingCount !== 1 ? 's' : '');
        $toggle.html(badgeHtml + ' ' + labelText + ' ' + hintHtml + ' ' + iconHtml);

        // Update stat cards
        var $fixableStat = $('.sf-stat-card.sf-stat-fixable .sf-stat-number');
        if ($fixableStat.length) {
            $fixableStat.text(remainingCount);
        }

        // Hide the section if no rows remain
        if (remainingCount === 0) {
            $section.slideUp(300);
        }
    }

    /**
     * Refresh the Fixed Links section via AJAX
     * Called after successfully applying fixes
     */
    function refreshFixedSection() {
        var $fixedSection = $('.sf-section-fixed');

        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_get_fixed_section',
                nonce: sfBrokenLinksData.nonce
            },
            success: function(response) {
                if (response.success && response.data.html) {
                    // Check if Fixed section exists
                    if ($fixedSection.length) {
                        // Replace the content
                        $fixedSection.html(response.data.html);
                    } else {
                        // Create the Fixed section if it doesn't exist
                        var $newSection = $('<div class="sf-results-section sf-section-fixed" id="sf-fixed-results"></div>');
                        $newSection.html(response.data.html);

                        // Insert after the last results section or before the progress modal
                        var $progressModal = $('.sf-progress-modal');
                        if ($progressModal.length) {
                            $newSection.insertBefore($progressModal);
                        } else {
                            var $lastSection = $('.sf-results-section').last();
                            if ($lastSection.length) {
                                $newSection.insertAfter($lastSection);
                            } else {
                                $('.sf-broken-links-module').append($newSection);
                            }
                        }
                        $fixedSection = $newSection;
                    }

                    // Ensure the section is visible
                    $fixedSection.show();

                    // Ensure content is visible (section should be expanded)
                    var $content = $fixedSection.find('.sf-fixed-content');
                    $content.show();

                    // Scroll to the Fixed section with a small delay for animation
                    setTimeout(function() {
                        $('html, body').animate({
                            scrollTop: $fixedSection.offset().top - 50
                        }, 500);
                    }, 400);

                    // Re-bind event handlers for the new content
                    bindFixedSectionEvents();

                    // Initialize pagination if needed
                    initFixedPagination();

                    // Update fixed filter count now that the section HTML is loaded
                    updateFixedCount();
                    updateFixStatusCounts();
                }
            }
        });
    }

    /**
     * Bind event handlers for Fixed section elements after AJAX refresh
     */
    function bindFixedSectionEvents() {
        // Fixed section toggle
        $('.sf-section-fixed .sf-fixed-toggle').off('click').on('click', function() {
            var $btn = $(this);
            var $content = $btn.closest('.sf-section-fixed').find('.sf-fixed-content');
            var isExpanded = $btn.attr('aria-expanded') === 'true';

            if (isExpanded) {
                $content.slideUp(300);
                $btn.attr('aria-expanded', 'false');
                $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
            } else {
                $content.slideDown(300);
                $btn.attr('aria-expanded', 'true');
                $btn.find('.sf-toggle-icon').addClass('sf-rotated');
            }
        });

        // Export Fixed CSV button - uses the existing document-level handler
        // (handleExportFixedLinks is already bound via $(document).on('click', '.sf-export-fixed-btn', ...))

        // Status filter buttons in Fixes Applied section
        $('.sf-section-fixed .sf-bl-status-filters .sf-status-filter').off('click').on('click', function() {
            var status = $(this).data('status');
            $(this).addClass('active').siblings().removeClass('active');
            filterFixResults(status);
        });
    }

    /**
     * Filter rows in the Fixes Applied section by status
     */
    function filterFixResults(status) {
        var $section = $('.sf-section-fixed');
        var $rows = $section.find('.sf-fixed-table tbody .sf-fixed-row');

        if (status === 'all') {
            $rows.removeClass('sf-filtered-out');
        } else {
            $rows.addClass('sf-filtered-out');
            $rows.filter('[data-status="' + status + '"]').removeClass('sf-filtered-out');
        }

        initFixedPagination();
    }

    /**
     * Update status filter counts in the Fixes Applied section
     */
    function updateFixStatusCounts() {
        var $section = $('.sf-section-fixed');
        var $rows = $section.find('.sf-fixed-table tbody .sf-fixed-row');

        var allCount = $rows.length;
        var successCount = $rows.filter('[data-status="success"]').length;
        var failedCount = $rows.filter('[data-status="failed"]').length;
        var skippedCount = $rows.filter('[data-status="skipped"]').length;

        $section.find('.sf-bl-status-count-all').text(allCount);
        $section.find('.sf-bl-status-count-success').text(successCount);
        $section.find('.sf-bl-status-count-failed').text(failedCount);
        $section.find('.sf-bl-status-count-skipped').text(skippedCount);
    }

    /**
     * Initialize pagination for Fixed section after AJAX refresh
     */
    function initFixedPagination() {
        var $table = $('.sf-section-fixed .sf-paginated-table');
        if ($table.length === 0) return;

        var perPage = parseInt($table.data('per-page'), 10) || 100;
        var $rows = $table.find('tbody tr');
        var totalRows = $rows.length;

        if (totalRows <= perPage) return;

        // Hide rows beyond first page
        $rows.each(function(index) {
            $(this).toggle(index < perPage);
        });

        // Setup pagination controls
        var $pagination = $('.sf-section-fixed .sf-pagination');
        if ($pagination.length) {
            var totalPages = Math.ceil(totalRows / perPage);
            var $pageNumbers = $pagination.find('.sf-page-numbers');
            $pageNumbers.empty();

            for (var i = 1; i <= Math.min(totalPages, 7); i++) {
                var $btn = $('<button type="button" class="sf-page-btn sf-page-num' + (i === 1 ? ' active' : '') + '" data-page="' + i + '">' + i + '</button>');
                $pageNumbers.append($btn);
            }

            // Bind pagination click handlers
            $pagination.find('.sf-page-num').off('click').on('click', function() {
                var page = parseInt($(this).data('page'), 10);
                goToFixedPage(page, perPage, $rows, $pagination);
            });

            $pagination.find('.sf-page-prev').off('click').on('click', function() {
                var currentPage = parseInt($pagination.find('.sf-page-num.active').data('page'), 10) || 1;
                if (currentPage > 1) {
                    goToFixedPage(currentPage - 1, perPage, $rows, $pagination);
                }
            });

            $pagination.find('.sf-page-next').off('click').on('click', function() {
                var currentPage = parseInt($pagination.find('.sf-page-num.active').data('page'), 10) || 1;
                var totalPages = Math.ceil(totalRows / perPage);
                if (currentPage < totalPages) {
                    goToFixedPage(currentPage + 1, perPage, $rows, $pagination);
                }
            });
        }
    }

    /**
     * Navigate to a specific page in Fixed section pagination
     */
    function goToFixedPage(page, perPage, $rows, $pagination) {
        var start = (page - 1) * perPage;
        var end = start + perPage;

        $rows.each(function(index) {
            $(this).toggle(index >= start && index < end);
        });

        // Update active state
        $pagination.find('.sf-page-num').removeClass('active');
        $pagination.find('.sf-page-num[data-page="' + page + '"]').addClass('active');

        // Update prev/next buttons
        var totalPages = Math.ceil($rows.length / perPage);
        $pagination.find('.sf-page-prev').prop('disabled', page <= 1);
        $pagination.find('.sf-page-next').prop('disabled', page >= totalPages);

        // Update info text
        $pagination.find('.sf-page-start').text(start + 1);
        $pagination.find('.sf-page-end').text(Math.min(end, $rows.length));
    }

    /**
     * Show progress modal
     */
    function showProgressModal(total) {
        var $modal = $('.sf-progress-modal');
        $modal.find('.sf-progress-total').text(total);
        $modal.find('.sf-progress-current').text('0');
        $modal.find('.sf-progress-fill').css('width', '0%');
        $modal.fadeIn(200);
    }

    /**
     * Hide progress modal
     */
    function hideProgressModal() {
        $('.sf-progress-modal').fadeOut(200);
    }

    // =====================================
    // BULK UPLOAD HANDLERS
    // =====================================

    /**
     * Handle bulk confirm button click
     */
    function handleBulkConfirm() {
        var $btn = $(this);
        var totalCount = parseInt($btn.data('count'), 10) || 0;

        if (totalCount === 0) {
            showToast(sfBrokenLinksData.i18n.noFixesSelected || 'No fixes to apply', 'warning');
            return;
        }

        // Show large file warning if needed
        if (totalCount > 500) {
            var continueAnyway = confirm(
                sfBrokenLinksData.i18n.bulkLargeFileWarning.replace('%d', totalCount) +
                '\n\n' +
                (sfBrokenLinksData.i18n.bulkContinueAnyway || 'Continue Anyway?')
            );
            if (!continueAnyway) {
                return;
            }
        }

        // Show progress modal
        showBulkProgressModal(totalCount);

        // Start batch processing
        processBulkFixBatch(0, 50, totalCount);
    }

    /**
     * Show the bulk progress modal
     */
    function showBulkProgressModal(total) {
        var $modal = $('#sf-bulk-progress-modal');
        $('#sf-bulk-progress-fill').css('width', '0%');
        $('#sf-bulk-progress-current').text('0');
        $('#sf-bulk-progress-total').text(total);
        $('#sf-bulk-progress-url').text(sfBrokenLinksData.i18n.bulkProcessing || 'Processing...');
        $modal.show();
    }

    /**
     * Update the bulk progress modal
     */
    function updateBulkProgress(current, total) {
        var percent = Math.round((current / total) * 100);
        $('#sf-bulk-progress-fill').css('width', percent + '%');
        $('#sf-bulk-progress-current').text(current);
    }

    /**
     * Hide the bulk progress modal
     */
    function hideBulkProgressModal() {
        $('#sf-bulk-progress-modal').hide();
    }

    /**
     * Process a batch of bulk fixes
     */
    function processBulkFixBatch(offset, batchSize, totalCount) {
        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_apply_bulk_fixes',
                nonce: sfBrokenLinksData.nonce,
                offset: offset,
                batch_size: batchSize
            },
            success: function(response) {
                if (response.success) {
                    var data = response.data;
                    var processed = offset + data.processed;

                    // Update progress
                    updateBulkProgress(processed, totalCount);

                    if (data.complete) {
                        // All done! Hide modal and reload
                        hideBulkProgressModal();

                        var successCount = data.total_success || data.success || 0;
                        var failedCount = data.total_failed || data.failed || 0;

                        if (failedCount > 0) {
                            showToast(
                                sfBrokenLinksData.i18n.bulkPartialComplete
                                    .replace('%d', successCount)
                                    .replace('%d', failedCount),
                                'warning'
                            );
                        } else {
                            showToast(
                                sfBrokenLinksData.i18n.bulkComplete.replace('%d', successCount),
                                'success'
                            );
                        }

                        // Reload to show results
                        window.location.reload();
                    } else {
                        // Process next batch
                        processBulkFixBatch(data.next_offset, batchSize, totalCount);
                    }
                } else {
                    hideBulkProgressModal();
                    showToast(response.data.message || 'Bulk fix failed', 'error');
                }
            },
            error: function() {
                hideBulkProgressModal();
                showToast(sfBrokenLinksData.i18n.fixesFailed || 'Bulk fix failed', 'error');
            }
        });
    }

    /**
     * Handle bulk clear button click
     */
    function handleBulkClear() {
        $.ajax({
            url: sfBrokenLinksData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sf_broken_links_cancel_bulk',
                nonce: sfBrokenLinksData.nonce
            },
            success: function(response) {
                if (response.success) {
                    showToast(response.data.message || 'Cleared', 'success');
                    window.location.reload();
                } else {
                    showToast(response.data.message || 'Failed to clear', 'error');
                }
            },
            error: function() {
                showToast('Failed to clear', 'error');
            }
        });
    }

    /**
     * Handle bulk download preview CSV
     */
    function handleBulkDownloadPreview() {
        // Create a form and submit it to trigger the download
        var $form = $('<form>', {
            method: 'POST',
            action: sfBrokenLinksData.ajaxUrl
        });

        $form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_broken_links_download_preview' }));
        $form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfBrokenLinksData.nonce }));

        $form.appendTo('body').submit().remove();
    }

    /**
     * Handle bulk download results CSV
     */
    function handleBulkDownloadResults() {
        var $form = $('<form>', {
            method: 'POST',
            action: sfBrokenLinksData.ajaxUrl
        });

        $form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_broken_links_download_results' }));
        $form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfBrokenLinksData.nonce }));

        $form.appendTo('body').submit().remove();
    }

    // =====================================
    // PAGINATION FUNCTIONS
    // =====================================

    /**
     * Initialize pagination for all sections
     */
    function initPagination() {
        $('.sf-paginated-table').each(function() {
            var $wrapper = $(this);
            var section = $wrapper.data('section');
            var perPage = parseInt($wrapper.data('per-page'), 10) || 100;
            var total = parseInt($wrapper.data('total'), 10) || 0;

            if (BrokenLinks.pagination[section]) {
                BrokenLinks.pagination[section].perPage = perPage;
                BrokenLinks.pagination[section].totalItems = total;
                BrokenLinks.pagination[section].currentPage = 1;

                applyPagination(section);
                renderPageNumbers(section);
            }
        });
    }

    /**
     * Go to a specific page
     */
    function goToPage(section, page) {
        var pag = BrokenLinks.pagination[section];
        if (!pag) return;

        var totalPages = Math.ceil(getVisibleRowCount(section) / pag.perPage);
        page = Math.max(1, Math.min(page, totalPages));

        if (page === pag.currentPage) return;

        // Preserve scroll position before making changes
        var scrollTop = $(window).scrollTop();

        pag.currentPage = page;
        applyPagination(section);
        renderPageNumbers(section);
        updatePaginationInfo(section);

        // Restore scroll position after DOM changes
        $(window).scrollTop(scrollTop);
    }

    /**
     * Apply pagination to a section - show/hide rows based on current page
     */
    function applyPagination(section) {
        var pag = BrokenLinks.pagination[section];
        if (!pag) return;

        var $wrapper = $('.sf-paginated-table[data-section="' + section + '"]');
        var $rows;

        // Different row selectors for different sections
        if (section === 'fixable') {
            $rows = $wrapper.find('tbody .sf-link-row');
        } else if (section === 'manual') {
            $rows = $wrapper.find('tbody .sf-manual-row');
        } else if (section === 'skipped') {
            $rows = $wrapper.find('.sf-skipped-row');
        } else if (section === 'fixed') {
            $rows = $wrapper.find('.sf-fixed-link-item');
        }

        if (!$rows || !$rows.length) return;

        var startIndex = (pag.currentPage - 1) * pag.perPage;
        var endIndex = startIndex + pag.perPage;

        var visibleIndex = 0;
        $rows.each(function() {
            var $row = $(this);
            var isFilteredOut = $row.hasClass('sf-filtered-out');

            if (isFilteredOut) {
                $row.addClass('sf-page-hidden');
            } else {
                if (visibleIndex >= startIndex && visibleIndex < endIndex) {
                    $row.removeClass('sf-page-hidden');
                } else {
                    $row.addClass('sf-page-hidden');
                }
                visibleIndex++;
            }
        });

        // Also hide action panels for hidden rows (fixable section)
        if (section === 'fixable') {
            $wrapper.find('.sf-action-panel-row').each(function() {
                var $panel = $(this);
                var forUrl = $panel.data('for-url');
                var $row = $wrapper.find('.sf-link-row[data-broken-url="' + escapeSelector(forUrl) + '"]');
                if ($row.hasClass('sf-page-hidden') || $row.hasClass('sf-filtered-out')) {
                    $panel.hide();
                }
            });
        }

        // Update button states
        var $pagination = $('.sf-pagination[data-section="' + section + '"]');
        var totalPages = Math.ceil(getVisibleRowCount(section) / pag.perPage);

        $pagination.find('.sf-page-prev').prop('disabled', pag.currentPage <= 1);
        $pagination.find('.sf-page-next').prop('disabled', pag.currentPage >= totalPages);
    }

    /**
     * Render page number buttons
     */
    function renderPageNumbers(section) {
        var pag = BrokenLinks.pagination[section];
        if (!pag) return;

        var $pagination = $('.sf-pagination[data-section="' + section + '"]');
        var $container = $pagination.find('.sf-page-numbers');

        var visibleCount = getVisibleRowCount(section);
        var totalPages = Math.ceil(visibleCount / pag.perPage);
        var currentPage = pag.currentPage;

        if (totalPages <= 1) {
            $container.empty();
            $pagination.find('.sf-page-prev, .sf-page-next').prop('disabled', true);
            return;
        }

        var html = '';
        var maxVisible = 7;

        if (totalPages <= maxVisible) {
            // Show all page numbers
            for (var i = 1; i <= totalPages; i++) {
                html += renderPageButton(i, currentPage);
            }
        } else {
            // Show with ellipsis
            html += renderPageButton(1, currentPage);

            if (currentPage > 3) {
                html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
            }

            var startPage = Math.max(2, currentPage - 1);
            var endPage = Math.min(totalPages - 1, currentPage + 1);

            if (currentPage <= 3) {
                endPage = Math.min(4, totalPages - 1);
            }
            if (currentPage >= totalPages - 2) {
                startPage = Math.max(2, totalPages - 3);
            }

            for (var j = startPage; j <= endPage; j++) {
                html += renderPageButton(j, currentPage);
            }

            if (currentPage < totalPages - 2) {
                html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
            }

            html += renderPageButton(totalPages, currentPage);
        }

        $container.html(html);
    }

    /**
     * Render a single page button
     */
    function renderPageButton(pageNum, currentPage) {
        var isActive = pageNum === currentPage;
        return '<button type="button" class="sf-page-number' + (isActive ? ' active' : '') + '" data-page="' + pageNum + '">' + pageNum + '</button>';
    }

    /**
     * Update pagination info text
     */
    function updatePaginationInfo(section) {
        var pag = BrokenLinks.pagination[section];
        if (!pag) return;

        var $pagination = $('.sf-pagination[data-section="' + section + '"]');
        var visibleCount = getVisibleRowCount(section);

        var start = Math.min((pag.currentPage - 1) * pag.perPage + 1, visibleCount);
        var end = Math.min(pag.currentPage * pag.perPage, visibleCount);

        if (visibleCount === 0) {
            start = 0;
            end = 0;
        }

        $pagination.find('.sf-page-start').text(start);
        $pagination.find('.sf-page-end').text(end);
        $pagination.find('.sf-page-total').text(visibleCount);
    }

    /**
     * Get count of visible (non-filtered) rows in a section
     */
    function getVisibleRowCount(section) {
        var $wrapper = $('.sf-paginated-table[data-section="' + section + '"]');
        var $rows;

        if (section === 'fixable') {
            $rows = $wrapper.find('tbody .sf-link-row:not(.sf-filtered-out)');
        } else if (section === 'manual') {
            $rows = $wrapper.find('tbody .sf-manual-row:not(.sf-filtered-out)');
        } else if (section === 'skipped') {
            $rows = $wrapper.find('.sf-skipped-row:not(.sf-filtered-out)');
        } else if (section === 'fixed') {
            $rows = $wrapper.find('.sf-fixed-link-item:not(.sf-filtered-out)');
        }

        return $rows ? $rows.length : 0;
    }

    /**
     * Reset pagination to page 1
     */
    function resetPagination(section) {
        var pag = BrokenLinks.pagination[section];
        if (!pag) return;

        pag.currentPage = 1;
        applyPagination(section);
        renderPageNumbers(section);
        updatePaginationInfo(section);
    }

    // =====================================
    // FILTER FUNCTIONS
    // =====================================

    /**
     * Initialize filter buttons
     */
    function initFilters() {
        // Set initial active state
        $('.sf-filter-stat[data-filter="all"]').addClass('active');
    }

    /**
     * Filter fixable links by type (all, internal, external, fixed)
     */
    function filterByType(filterType) {
        BrokenLinks.currentFilter = filterType;

        // Handle "fixed" filter - show fixed rows and expand Fixed section
        if (filterType === 'fixed') {
            var $wrapper = $('.sf-paginated-table[data-section="fixable"]');
            var $rows = $wrapper.find('tbody .sf-link-row');

            // Show only rows with fixed status
            $rows.each(function() {
                var $row = $(this);
                var status = $row.attr('data-status') || 'pending';
                if ($row.hasClass('sf-row-fixed-complete') || status === 'success' || status === 'fixed') {
                    $row.removeClass('sf-filtered-out');
                } else {
                    $row.addClass('sf-filtered-out');
                }
            });

            // Reset pagination to page 1 when filter changes
            resetPagination('fixable');

            // Update filter button active state
            $('.sf-filter-stat').removeClass('active');
            $('.sf-filter-stat[data-filter="fixed"]').addClass('active');

            // Auto-expand Fixed Links section
            var $fixedSection = $('#sf-fixed-results');
            if ($fixedSection.length) {
                var $fixedToggle = $fixedSection.find('.sf-fixed-toggle');
                if ($fixedToggle.length && $fixedToggle.attr('aria-expanded') === 'false') {
                    $fixedToggle.trigger('click');
                }
            }

            return;
        }

        // Handle "failed" filter - show only failed rows
        if (filterType === 'failed') {
            var $wrapper = $('.sf-paginated-table[data-section="fixable"]');
            var $rows = $wrapper.find('tbody .sf-link-row');

            $rows.each(function() {
                var $row = $(this);
                var status = $row.attr('data-status') || 'pending';
                if (status === 'failed' || $row.hasClass('sf-row-failed')) {
                    $row.removeClass('sf-filtered-out');
                } else {
                    $row.addClass('sf-filtered-out');
                }
            });

            // Reset pagination to page 1 when filter changes
            resetPagination('fixable');

            // Update filter button active state
            $('.sf-filter-stat').removeClass('active');
            $('.sf-filter-stat[data-filter="failed"]').addClass('active');

            return;
        }

        // Standard filtering for all/internal/external
        var $wrapper = $('.sf-paginated-table[data-section="fixable"]');
        var $rows = $wrapper.find('tbody .sf-link-row');

        if (filterType === 'all') {
            $rows.removeClass('sf-filtered-out');
        } else if (filterType === 'images') {
            $rows.each(function() {
                var $row = $(this);
                var isImage = $row.data('is-image') === true || $row.data('is-image') === 'true';
                if (isImage) {
                    $row.removeClass('sf-filtered-out');
                } else {
                    $row.addClass('sf-filtered-out');
                }
            });
        } else {
            $rows.each(function() {
                var $row = $(this);
                var isInternal = $row.data('is-internal') === true || $row.data('is-internal') === 'true';

                if (filterType === 'internal' && isInternal) {
                    $row.removeClass('sf-filtered-out');
                } else if (filterType === 'external' && !isInternal) {
                    $row.removeClass('sf-filtered-out');
                } else {
                    $row.addClass('sf-filtered-out');
                }
            });
        }

        // Reset pagination to page 1 when filter changes
        resetPagination('fixable');

        // Update filter button active state
        $('.sf-filter-stat').removeClass('active');
        $('.sf-filter-stat[data-filter="' + filterType + '"]').addClass('active');
    }

    /**
     * Show toast notification (uses parent admin.js function)
     */
    function showToast(message, type) {
        if (typeof window.sfShowToast === 'function') {
            window.sfShowToast(message, type);
        } else if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
            window.ScreamingFixes.Toast[type] ? window.ScreamingFixes.Toast[type](message) : window.ScreamingFixes.Toast.info(message);
        } else {
            // Fallback
            console.log('[' + type + '] ' + message);
        }
    }

    /**
     * Escape selector special characters
     */
    function escapeSelector(str) {
        return str.replace(/[!"#$%&'()*+,.\/:;<=>?@[\\\]^`{|}~]/g, '\\$&');
    }

    /**
     * Show animated loading container for CSV processing
     */
    function showCSVLoadingContainer() {
        // Remove any existing loading container
        $('.sf-csv-loading-overlay').remove();

        // Create loading overlay
        var $overlay = $('<div class="sf-csv-loading-overlay">' +
            '<div class="sf-csv-loading-container">' +
                '<div class="sf-csv-loading-spinner"></div>' +
                '<div class="sf-csv-loading-progress">' +
                    '<div class="sf-csv-progress-bar"><div class="sf-csv-progress-fill"></div></div>' +
                '</div>' +
                '<div class="sf-csv-loading-message">' + csvLoadingMessages[0] + '</div>' +
                '<div class="sf-csv-loading-subtext">Please wait while we analyze your data...</div>' +
            '</div>' +
        '</div>');

        // Insert into the module or body
        var $module = $('.sf-broken-links-module');
        if ($module.length) {
            $module.append($overlay);
        } else {
            $('body').append($overlay);
        }

        // Animate progress bar
        var $progressFill = $overlay.find('.sf-csv-progress-fill');
        $progressFill.css('width', '0%');

        // Simulate progress (0-90% over ~15 seconds, slower as it goes)
        var progress = 0;
        var progressInterval = setInterval(function() {
            if (progress < 90) {
                // Progress slows down as it gets higher
                var increment = Math.max(0.5, (90 - progress) / 30);
                progress = Math.min(90, progress + increment);
                $progressFill.css('width', progress + '%');
            }
        }, 200);

        // Start message rotation (every 2.5 seconds)
        var messageIndex = 0;
        var messageInterval = setInterval(function() {
            messageIndex = (messageIndex + 1) % csvLoadingMessages.length;
            var $message = $overlay.find('.sf-csv-loading-message');
            $message.addClass('sf-fade-out');
            setTimeout(function() {
                $message.text(csvLoadingMessages[messageIndex]).removeClass('sf-fade-out');
            }, 300);
        }, 2500);

        return {
            overlay: $overlay,
            messageInterval: messageInterval,
            progressInterval: progressInterval
        };
    }

    /**
     * Hide CSV loading container
     */
    function hideCSVLoadingContainer(loadingState) {
        if (loadingState) {
            if (loadingState.messageInterval) {
                clearInterval(loadingState.messageInterval);
            }
            if (loadingState.progressInterval) {
                clearInterval(loadingState.progressInterval);
            }
            if (loadingState.overlay) {
                // Complete the progress bar before hiding
                loadingState.overlay.find('.sf-csv-progress-fill').css('width', '100%');
                loadingState.overlay.find('.sf-csv-loading-message').text('✅ Processing complete!');

                setTimeout(function() {
                    loadingState.overlay.fadeOut(300, function() {
                        $(this).remove();
                    });
                }, 500);
            }
        }
    }

    // =====================================
    // SHOW ALL BUTTON HANDLERS (LEGACY - replaced by pagination)
    // =====================================

    /**
     * Handle Show All Manual Links button (LEGACY - no longer used)
     * Kept for backwards compatibility if needed
     */
    function handleShowAllManual() {
        // No longer used - replaced by pagination
        console.log('handleShowAllManual: deprecated, using pagination instead');
    }

    /**
     * Handle Show All Skipped Links button (LEGACY - no longer used)
     * Kept for backwards compatibility if needed
     */
    function handleShowAllSkipped() {
        // No longer used - replaced by pagination
        console.log('handleShowAllSkipped: deprecated, using pagination instead');
    }

    /**
     * Render sources column in JS (simplified version for dynamic rows)
     */
    function renderSourcesColumnJS(sources, brokenUrl) {
        var sourceCount = sources.length;

        if (sourceCount === 0) {
            return '<span class="sf-source-count-label">No sources</span>';
        }

        if (sourceCount === 1) {
            var source = sources[0];
            var path = source.source_url ? (new URL(source.source_url, window.location.origin).pathname || source.source_url) : '';
            var html = '<div class="sf-single-source">' +
                '<div class="sf-source-count-label">Found on 1 page</div>';

            if (source.post_id) {
                html += '<a href="' + escapeHtml(source.edit_url || '') + '" target="_blank" rel="noopener" class="sf-source-title">' +
                    escapeHtml(source.post_title || 'Edit') + '</a>';
            }

            html += '<div class="sf-source-url">' +
                '<a href="' + escapeHtml(source.source_url || '') + '" target="_blank" rel="noopener" class="sf-source-url-link">' +
                escapeHtml(path) + '</a></div>';

            if (source.anchor) {
                html += '<div class="sf-anchor-text"><small>Anchor:</small> ' + escapeHtml(source.anchor) + '</div>';
            }

            html += '</div>';
            return html;
        }

        // Multiple sources - show count with toggle
        var sourcesJson = JSON.stringify(sources.map(function(s) {
            return {
                url: s.source_url || '',
                path: s.source_url ? (new URL(s.source_url, window.location.origin).pathname || s.source_url) : '',
                post_id: s.post_id || 0,
                post_title: s.post_title || '',
                edit_url: s.edit_url || '',
                anchor: s.anchor || ''
            };
        }));

        return '<div class="sf-sources-wrapper">' +
            '<button type="button" class="sf-sources-toggle" data-broken-url="' + escapeHtml(brokenUrl) + '" ' +
            'data-sources=\'' + sourcesJson.replace(/'/g, '&#39;') + '\' data-total="' + sourceCount + '">' +
            'Found on ' + sourceCount + ' pages <span class="dashicons dashicons-arrow-down-alt2"></span></button>' +
            '<div class="sf-sources-list" style="display: none;"><ul class="sf-sources-ul"></ul></div></div>';
    }

    // Expandable bulk preview rows
    $(document).on('click', '.sf-bulk-preview-table .sf-desc-expandable', function(e) {
        if ($(e.target).closest('a').length) return;
        var $row = $(this);
        var rowIndex = $row.data('row-index');
        var $detailRow = $('tr.sf-bulk-detail-row[data-for-row="' + rowIndex + '"]');
        if ($row.hasClass('sf-expanded')) {
            $row.removeClass('sf-expanded');
            $detailRow.slideUp(200);
        } else {
            $row.addClass('sf-expanded');
            $detailRow.slideDown(200);
        }
    });

    // Initialize when document is ready
    $(document).ready(init);

})(jQuery);
