<?php
/**
 * Broken Links Module for Screaming Fixes
 *
 * Finds and fixes broken internal links from Screaming Frog exports
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Broken_Links extends SF_Module {

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'broken-links';

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Broken Links', 'screaming-fixes');
        $this->slug = 'broken-links';
        $this->description = __('Find and fix broken internal links automatically.', 'screaming-fixes');

        parent::__construct();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register AJAX handlers
        add_action('wp_ajax_sf_broken_links_process_csv', [$this, 'ajax_process_csv']);
        add_action('wp_ajax_sf_broken_links_apply_fixes', [$this, 'ajax_apply_fixes']);
        add_action('wp_ajax_sf_broken_links_get_ai_suggestion', [$this, 'ajax_get_ai_suggestion']);
        add_action('wp_ajax_sf_broken_links_ai_suggest_all', [$this, 'ajax_ai_suggest_all']);
        add_action('wp_ajax_sf_broken_links_export', [$this, 'ajax_export']);
        add_action('wp_ajax_sf_broken_links_export_fixed', [$this, 'ajax_export_fixed']);
        add_action('wp_ajax_sf_broken_links_get_data', [$this, 'ajax_get_data']);
        add_action('wp_ajax_sf_broken_links_clear', [$this, 'ajax_clear_results']);
        add_action('wp_ajax_sf_broken_links_get_fixed_section', [$this, 'ajax_get_fixed_section']);

        // Bulk upload AJAX handlers
        add_action('wp_ajax_sf_broken_links_apply_bulk_fixes', [$this, 'ajax_apply_bulk_fixes']);
        add_action('wp_ajax_sf_broken_links_cancel_bulk', [$this, 'ajax_cancel_bulk']);
        add_action('wp_ajax_sf_broken_links_download_preview', [$this, 'ajax_download_preview']);
        add_action('wp_ajax_sf_broken_links_download_results', [$this, 'ajax_download_results']);

        // Enqueue module assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        // Check if we're on the broken links tab
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'broken-links') {
            return;
        }

        // Use file modification time for cache busting
        $css_file = SF_PLUGIN_DIR . 'modules/broken-links/assets/broken-links.css';
        $js_file = SF_PLUGIN_DIR . 'modules/broken-links/assets/broken-links.js';
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        wp_enqueue_style(
            'sf-broken-links',
            SF_PLUGIN_URL . 'modules/broken-links/assets/broken-links.css',
            ['screaming-fixes-admin'],
            $css_version
        );

        wp_enqueue_script(
            'sf-broken-links',
            SF_PLUGIN_URL . 'modules/broken-links/assets/broken-links.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        // Check if API key is configured
        $api_key = get_option('sf_claude_api_key');
        $has_api_key = !empty($api_key);

        wp_localize_script('sf-broken-links', 'sfBrokenLinksData', [
            'nonce' => wp_create_nonce('sf_broken_links_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'hasApiKey' => $has_api_key,
            'settingsUrl' => admin_url('admin.php?page=screaming-fixes&tab=settings'),
            'i18n' => [
                'processing' => __('Processing CSV...', 'screaming-fixes'),
                'scanningPosts' => __('Scanning posts...', 'screaming-fixes'),
                'applyingFixes' => __('Applying fixes...', 'screaming-fixes'),
                'fixesApplied' => __('Fixes applied successfully!', 'screaming-fixes'),
                'fixesFailed' => __('Some fixes failed. Check the results.', 'screaming-fixes'),
                'noFixesSelected' => __('No fixes selected.', 'screaming-fixes'),
                'aiSuggesting' => __('Getting AI suggestions...', 'screaming-fixes'),
                'aiComplete' => __('AI suggestions complete.', 'screaming-fixes'),
                'aiFailed' => __('Failed to get AI suggestions.', 'screaming-fixes'),
                'selectFix' => __('Select fix type', 'screaming-fixes'),
                'replace' => __('Replace URL', 'screaming-fixes'),
                'removeLink' => __('Remove link (keep text)', 'screaming-fixes'),
                'removeAll' => __('Remove link and text', 'screaming-fixes'),
                'ignore' => __('Ignore', 'screaming-fixes'),
                'confirmApply' => __('Apply fixes to %d broken links?', 'screaming-fixes'),
                'exporting' => __('Exporting...', 'screaming-fixes'),
                'exportComplete' => __('Export complete.', 'screaming-fixes'),
                'confirmClear' => __('Are you sure you want to clear all results? This cannot be undone.', 'screaming-fixes'),
                'resultsCleared' => __('Results cleared.', 'screaming-fixes'),
                // Bulk upload i18n
                'bulkProcessing' => __('Processing bulk fix CSV...', 'screaming-fixes'),
                'bulkReady' => __('%d broken link fixes ready to apply', 'screaming-fixes'),
                'bulkNotMatched' => __('%d URLs not matched', 'screaming-fixes'),
                'bulkSkippedEmpty' => __('%d rows skipped - no fix URL provided', 'screaming-fixes'),
                'bulkUpdating' => __('Applying broken link fixes...', 'screaming-fixes'),
                'bulkComplete' => __('%d broken links fixed successfully.', 'screaming-fixes'),
                'bulkPartialComplete' => __('%d links fixed successfully. %d failed - see details below.', 'screaming-fixes'),
                'bulkLargeFileWarning' => __('Large file detected (%d fixes). For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'),
                'bulkContinueAnyway' => __('Continue Anyway', 'screaming-fixes'),
                'bulkNoFixColumn' => __('CSV must contain a fix column (Broken_Link_Fix, New_Destination, etc.) for bulk updates.', 'screaming-fixes'),
                'bulkNoSourceColumn' => __('CSV must contain a source URL column (Source, Address, etc.).', 'screaming-fixes'),
                'bulkNoDestColumn' => __('CSV must contain a destination column (Destination, Broken_Link, etc.) for matching broken links.', 'screaming-fixes'),
                'noFixesApplied' => __('No fixes have been applied yet. Apply link fixes first to view fixed links.', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Run the module scan
     *
     * @return array Scan results
     */
    public function run_scan() {
        // This module doesn't do active scanning - it processes uploaded CSVs
        return $this->get_results();
    }

    /**
     * Check if this module can handle a CSV based on headers
     *
     * @param array $headers CSV column headers (lowercase)
     * @return bool
     */
    public function can_handle_csv($headers) {
        // Normalize headers to lowercase
        $headers = array_map('strtolower', $headers);

        // Screaming Frog "All Inlinks" or "Client Error (4xx)" export
        // Must have destination/url and status code
        $has_destination = in_array('destination', $headers) ||
                           in_array('url', $headers) ||
                           in_array('address', $headers);

        $has_status = in_array('status code', $headers) ||
                      in_array('status', $headers);

        $has_source = in_array('source', $headers) ||
                      in_array('from', $headers);

        // Must have destination AND (status OR source)
        return $has_destination && ($has_status || $has_source);
    }

    /**
     * Process uploaded CSV file
     *
     * Uses the Source URL and Link Position from CSV to identify which post
     * contains the broken link and categorize it as fixable, manual, or skip.
     *
     * @param string $file_path Path to uploaded CSV
     * @return array|WP_Error Processed results or error
     */
    /**
     * Maximum number of rows allowed in CSV upload
     */
    const MAX_CSV_ROWS = 5000;

    public function process_csv($file_path) {
        $parser = new SF_CSV_Parser();

        // Parse the CSV
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Standardize column names
        $parsed = $parser->standardize_columns($parsed, 'broken-links');

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    /* translators: 1: number of rows in uploaded file, 2: maximum allowed rows */
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately. Tip: In Screaming Frog or Excel, you can filter by specific status codes or sort by source URL to create logical batches.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Process each row - use Source URL and Link Position to categorize
        $all_links = [];
        foreach ($parsed['rows'] as $row) {
            $status_code = isset($row['status_code']) ? intval($row['status_code']) : 0;

            // Include if no status code (assume broken) or if 4xx error
            if ($status_code === 0 || ($status_code >= 400 && $status_code < 500)) {
                $broken_url = $row['url'] ?? $row['destination'] ?? '';
                $source_url = $row['source'] ?? '';
                $anchor = $row['anchor'] ?? '';
                $link_position = $row['link_position'] ?? $row['position'] ?? '';

                if (empty($broken_url)) {
                    continue;
                }

                // Categorize this link based on source URL and link position
                $categorization = $this->categorize_link($source_url, $link_position);

                $post_id = $categorization['post_id'];
                $post_title = $post_id ? get_the_title($post_id) : '';
                $edit_url = $post_id ? get_edit_post_link($post_id, 'raw') : '';

                $all_links[] = [
                    'broken_url' => $broken_url,
                    'status_code' => $status_code,
                    'anchor' => $anchor,
                    'source_url' => $source_url,
                    'link_position' => $link_position,
                    'post_id' => $post_id,
                    'post_title' => $post_title,
                    'edit_url' => $edit_url,
                    'fix_category' => $categorization['category'],
                    'fix_note' => $categorization['note'],
                    'location' => $categorization['location'],
                ];
            }
        }

        // Group by broken URL so same broken link on multiple pages shows together
        $grouped = [];
        foreach ($all_links as $link) {
            $broken_url = $link['broken_url'];

            if (!isset($grouped[$broken_url])) {
                $grouped[$broken_url] = [
                    'broken_url' => $broken_url,
                    'status_code' => $link['status_code'],
                    'sources' => [],
                    'fix' => null,
                    'new_url' => '',
                ];
            }

            // Add this source to the list (avoid duplicates by source_url + link_position)
            $source_key = $link['source_url'] . '_' . $link['link_position'];
            if (!isset($grouped[$broken_url]['sources'][$source_key])) {
                $grouped[$broken_url]['sources'][$source_key] = [
                    'post_id' => $link['post_id'],
                    'post_title' => $link['post_title'],
                    'source_url' => $link['source_url'],
                    'edit_url' => $link['edit_url'],
                    'anchor' => $link['anchor'],
                    'link_position' => $link['link_position'],
                    'fix_category' => $link['fix_category'],
                    'fix_note' => $link['fix_note'],
                    'location' => $link['location'],
                ];
            }
        }

        // Convert sources from associative to indexed array and categorize each broken link
        $fixable_links = [];
        $manual_links = [];
        $skipped_links = [];

        foreach ($grouped as $broken_url => $data) {
            $data['sources'] = array_values($data['sources']);
            $data['source_count'] = count($data['sources']);

            // Count sources by category for this broken URL
            $fixable_sources = [];
            $manual_sources = [];
            $skip_sources = [];

            foreach ($data['sources'] as $source) {
                switch ($source['fix_category']) {
                    case 'fixable':
                        $fixable_sources[] = $source;
                        break;
                    case 'manual':
                        $manual_sources[] = $source;
                        break;
                    case 'skip':
                        $skip_sources[] = $source;
                        break;
                }
            }

            // Store categorized sources
            $data['fixable_sources'] = $fixable_sources;
            $data['manual_sources'] = $manual_sources;
            $data['skip_sources'] = $skip_sources;
            $data['fixable_count'] = count($fixable_sources);
            $data['manual_count'] = count($manual_sources);
            $data['skip_count'] = count($skip_sources);

            // Determine overall category for this broken URL
            // If ANY source is fixable, it goes in fixable
            // If no fixable but some manual, goes in manual
            // If only skipped, goes in skipped
            if (!empty($fixable_sources)) {
                $data['overall_category'] = 'fixable';
                $fixable_links[] = $data;
            } elseif (!empty($manual_sources)) {
                $data['overall_category'] = 'manual';
                $manual_links[] = $data;
            } else {
                $data['overall_category'] = 'skip';
                $skipped_links[] = $data;
            }
        }

        // Calculate totals
        $total_fixable = 0;
        $total_manual = 0;
        $total_skipped = 0;

        foreach ($fixable_links as $link) {
            $total_fixable += $link['fixable_count'];
        }
        foreach ($manual_links as $link) {
            $total_manual += $link['manual_count'];
        }
        foreach ($skipped_links as $link) {
            $total_skipped += $link['skip_count'];
        }

        $results = [
            'broken_links' => array_merge($fixable_links, $manual_links, $skipped_links),
            'fixable_links' => $fixable_links,
            'manual_links' => $manual_links,
            'skipped_links' => $skipped_links,
            'total_count' => count($fixable_links) + count($manual_links) + count($skipped_links),
            'total_sources' => $total_fixable + $total_manual + $total_skipped,
            'fixable_count' => count($fixable_links),
            'manual_count' => count($manual_links),
            'skipped_count' => count($skipped_links),
            'fixable_sources' => $total_fixable,
            'manual_sources' => $total_manual,
            'skipped_sources' => $total_skipped,
            'processed_at' => current_time('mysql'),
        ];

        // Save results
        $this->save_results($results);

        // Also save to uploads table for persistence
        $this->save_upload_data($results);

        return $results;
    }

    /**
     * Categorize a link based on its source URL and link position
     *
     * Uses Screaming Frog's Link Position column for accurate categorization,
     * with URL pattern fallbacks.
     *
     * @param string $source_url The source URL where the broken link was found
     * @param string $link_position The Link Position from Screaming Frog CSV
     * @return array Category info with 'category', 'post_id', and 'note'
     */
    private function categorize_link($source_url, $link_position = '') {
        $position = strtolower(trim($link_position));
        $path = wp_parse_url($source_url, PHP_URL_PATH);
        $path = $path ?: '/';

        // First check Link Position from Screaming Frog - this is most reliable
        if (!empty($position)) {
            // Navigation links - manual fix
            if (in_array($position, ['navigation', 'nav', 'menu'])) {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Navigation menu - fix in Appearance → Menus', 'screaming-fixes'),
                    'location' => 'navigation',
                ];
            }

            // Footer links - manual fix
            if ($position === 'footer') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Footer - fix in Appearance → Widgets or theme settings', 'screaming-fixes'),
                    'location' => 'footer',
                ];
            }

            // Sidebar links - manual fix
            if ($position === 'sidebar') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Sidebar - fix in Appearance → Widgets', 'screaming-fixes'),
                    'location' => 'sidebar',
                ];
            }

            // Aside links (widgets, related posts, etc.) - manual fix
            if ($position === 'aside') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Aside/Widget area - fix in Appearance → Widgets or related posts plugin', 'screaming-fixes'),
                    'location' => 'aside',
                ];
            }

            // Header/Logo - manual fix
            if (in_array($position, ['header', 'logo', 'site logo'])) {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Header/Logo - fix in Appearance → Customize', 'screaming-fixes'),
                    'location' => 'header',
                ];
            }

            // Breadcrumb - manual fix
            if ($position === 'breadcrumb') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Breadcrumb - fix in SEO plugin or theme settings', 'screaming-fixes'),
                    'location' => 'breadcrumb',
                ];
            }

            // Content links - try to find the post
            if ($position === 'content') {
                $post_id = $this->url_to_post_id($source_url);
                if ($post_id > 0) {
                    return [
                        'category' => 'fixable',
                        'post_id' => $post_id,
                        'note' => '',
                        'location' => 'content',
                    ];
                }
                // Content link but no post ID - might be dynamic page
            }
        }

        // Check URL patterns for dynamic/unfixable pages

        // Homepage - flag for manual fix
        if ($path === '/' || $path === '' || $path === '/index.php') {
            return [
                'category' => 'manual',
                'post_id' => 0,
                'note' => __('Homepage - fix manually in theme or widgets', 'screaming-fixes'),
                'location' => 'homepage',
            ];
        }

        // Pagination - skip (will be fixed when source post is fixed)
        if (preg_match('/\/page\/\d+\/?$/', $path)) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Pagination page - will be fixed when source content is fixed', 'screaming-fixes'),
                'location' => 'pagination',
            ];
        }

        // Author pages - skip
        if (strpos($path, '/author/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Author archive - dynamic page', 'screaming-fixes'),
                'location' => 'author',
            ];
        }

        // Category pages - skip
        if (strpos($path, '/category/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Category archive - dynamic page', 'screaming-fixes'),
                'location' => 'category',
            ];
        }

        // Tag pages - skip
        if (strpos($path, '/tag/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Tag archive - dynamic page', 'screaming-fixes'),
                'location' => 'tag',
            ];
        }

        // Feed URLs - skip
        if (strpos($path, '/feed') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Feed URL - dynamic page', 'screaming-fixes'),
                'location' => 'feed',
            ];
        }

        // Search pages - skip
        if (strpos($path, '/search/') !== false || strpos($source_url, '?s=') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Search results - dynamic page', 'screaming-fixes'),
                'location' => 'search',
            ];
        }

        // Date archives - skip
        if (preg_match('/\/\d{4}\/\d{2}\/?$/', $path) || preg_match('/\/\d{4}\/?$/', $path)) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Date archive - dynamic page', 'screaming-fixes'),
                'location' => 'date_archive',
            ];
        }

        // Try to get post ID for remaining URLs
        $post_id = $this->url_to_post_id($source_url);

        if ($post_id > 0) {
            // Check if it's a nav menu item
            $post_type = get_post_type($post_id);
            if ($post_type === 'nav_menu_item') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Navigation menu item - fix in Appearance → Menus', 'screaming-fixes'),
                    'location' => 'navigation',
                ];
            }

            return [
                'category' => 'fixable',
                'post_id' => $post_id,
                'note' => '',
                'location' => 'content',
            ];
        }

        // No post ID found - might be widget, theme template, or custom
        return [
            'category' => 'manual',
            'post_id' => 0,
            'note' => __('Source not found in database - may be in theme, widget, or plugin', 'screaming-fixes'),
            'location' => 'unknown',
        ];
    }

    /**
     * Convert a URL to a WordPress post ID
     *
     * @param string $url The URL to convert
     * @return int Post ID or 0 if not found
     */
    private function url_to_post_id($url) {
        if (empty($url)) {
            return 0;
        }

        // Try WordPress built-in function first
        $post_id = url_to_postid($url);

        if ($post_id) {
            return $post_id;
        }

        // Try with trailing slash variations
        $url_with_slash = trailingslashit($url);
        $url_without_slash = untrailingslashit($url);

        $post_id = url_to_postid($url_with_slash);
        if ($post_id) {
            return $post_id;
        }

        $post_id = url_to_postid($url_without_slash);
        if ($post_id) {
            return $post_id;
        }

        // Try extracting path and searching by slug
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $path = trim($path, '/');
            $slug = basename($path);

            if ($slug) {
                global $wpdb;

                // Search by post_name (slug)
                $post_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts}
                     WHERE post_name = %s
                     AND post_status IN ('publish', 'draft', 'pending', 'private')
                     AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')
                     LIMIT 1",
                    $slug
                ));

                if ($post_id) {
                    return (int) $post_id;
                }
            }
        }

        return 0;
    }

    /**
     * Get issue count for dashboard display
     *
     * @return int Count of broken links
     */
    public function get_issue_count() {
        $results = $this->get_results();

        if (empty($results) || !isset($results['total_count'])) {
            // Check uploads table
            $upload_data = $this->get_upload_data();
            if ($upload_data && isset($upload_data['total_count'])) {
                return (int) $upload_data['total_count'];
            }
            return 0;
        }

        return (int) $results['total_count'];
    }

    /**
     * Apply approved fixes
     *
     * @param array $fixes Array of fixes to apply
     * @return array Results with success/failure counts
     */
    public function apply_fixes($fixes) {
        $fixer = new SF_Link_Fixer();

        // Start batch tracking for undo capability
        SF_Batch_Restore::start_batch('broken-links', $fixes);

        $results = [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => [],
            'details' => [],
        ];

        foreach ($fixes as $fix) {
            // Skip if no action or ignored
            if (empty($fix['action']) || $fix['action'] === 'ignore') {
                $results['skipped']++;
                continue;
            }

            $broken_url = $fix['broken_url'] ?? $fix['url'] ?? '';
            $new_url = $fix['new_url'] ?? '';
            $action = $fix['action'];
            $post_ids = $fix['post_ids'] ?? [];

            if (empty($broken_url)) {
                $results['skipped']++;
                continue;
            }

            $results['total']++;

            // If specific post IDs provided, fix only those posts
            if (!empty($post_ids)) {
                foreach ($post_ids as $post_id) {
                    if (!$post_id) {
                        continue;
                    }

                    $fix_result = $fixer->replace_url_in_post(
                        $post_id,
                        $broken_url,
                        $new_url,
                        $action,
                        'broken-links'
                    );

                    if (is_wp_error($fix_result)) {
                        $results['failed']++;
                        $error_code = $fix_result->get_error_code();
                        $results['errors'][] = [
                            'post_id' => $post_id,
                            'broken_url' => $broken_url,
                            'error' => $fix_result->get_error_message(),
                            'error_code' => $error_code,
                            'suggestion' => $this->get_error_suggestion($error_code),
                        ];
                    } else {
                        $results['success']++;
                    }
                }
            } else {
                // Fall back to bulk method if no specific post IDs
                $fix_result = $fixer->bulk_replace_url($broken_url, $new_url, $action, 'broken-links');

                $results['success'] += $fix_result['success'];
                $results['failed'] += $fix_result['failed'];
                $results['errors'] = array_merge($results['errors'], $fix_result['errors']);
            }

            $results['details'][] = [
                'broken_url' => $broken_url,
                'action' => $action,
                'new_url' => $new_url,
            ];
        }

        // Update stored results to reflect fixes applied
        // Pass results so we only mark actually successful fixes
        $this->update_results_after_fixes($fixes, $results);

        // Complete batch tracking
        SF_Batch_Restore::complete_batch($results);

        return $results;
    }

    /**
     * Get a user-friendly suggestion for a given error code
     */
    private function get_error_suggestion($error_code) {
        $suggestions = [
            'post_not_found' => __('The post may have been deleted or moved to trash. Re-upload your CSV to refresh the data.', 'screaming-fixes'),
            'no_change' => __('The broken URL was not found in the post content. It may have already been fixed or the content was modified.', 'screaming-fixes'),
            'invalid_action' => __('The fix action type is not supported. Try using Replace or Remove instead.', 'screaming-fixes'),
            'write_failed' => __('The post content could not be saved. Check file permissions or try deactivating caching plugins temporarily.', 'screaming-fixes'),
        ];

        return isset($suggestions[$error_code]) ? $suggestions[$error_code] : __('An unexpected error occurred. Please try again.', 'screaming-fixes');
    }

    /**
     * Get AI suggestion for a broken link fix
     *
     * @param string $broken_url The broken URL
     * @param array $context Additional context
     * @return string|WP_Error Suggested URL or error
     */
    public function get_ai_suggestion($broken_url, $context = []) {
        $api_key = get_option('sf_claude_api_key');

        if (empty($api_key)) {
            return new WP_Error('no_api_key', __('Claude API key not configured. Add it in Settings.', 'screaming-fixes'));
        }

        $site_url = home_url();
        $site_domain = wp_parse_url($site_url, PHP_URL_HOST);

        // Determine if this is an external link
        $broken_url_domain = wp_parse_url($broken_url, PHP_URL_HOST);
        $is_external = $broken_url_domain && $broken_url_domain !== $site_domain;

        // Get the base URL for the broken link (scheme + host)
        $broken_url_scheme = wp_parse_url($broken_url, PHP_URL_SCHEME) ?: 'https';
        $broken_url_base = $broken_url_scheme . '://' . $broken_url_domain;

        // Build prompt based on whether link is internal or external
        if ($is_external) {
            // Extract the path from the broken URL for context
            $broken_url_path = wp_parse_url($broken_url, PHP_URL_PATH) ?: '';

            $prompt = sprintf(
                "A CONFIRMED BROKEN external link (returning 404 error) was found on a WordPress site.\n\n" .
                "Broken URL: %s\n" .
                "External domain: %s\n" .
                "URL path: %s\n" .
                "Appears in: %d post(s)\n" .
                "%s\n\n" .
                "This URL has been verified as broken (404). Based on the URL structure and anchor text, suggest ONE action:\n\n" .
                "1. REPLACEMENT URL: If you can infer a likely correct URL based on:\n" .
                "   - Common typos in the path (e.g., 'recpies' -> 'recipes')\n" .
                "   - URL restructuring patterns (e.g., '/podcast/financing-info/' might be '/financing/' or '/resources/financing/')\n" .
                "   - The anchor text suggesting what content was intended\n" .
                "   Respond with the FULL corrected URL on the same domain (e.g., %s/new-path/)\n\n" .
                "2. REMOVE: If you cannot confidently suggest a replacement, respond with: REMOVE\n" .
                "   This is the SAFE DEFAULT - it's better to remove a broken link than leave it broken.\n\n" .
                "IMPORTANT: Since this link is CONFIRMED broken, do NOT respond with UNKNOWN.\n" .
                "Either suggest a replacement URL or recommend REMOVE.\n\n" .
                "Respond with ONLY: the full replacement URL or REMOVE - nothing else.",
                $broken_url,
                $broken_url_domain,
                $broken_url_path,
                $context['post_count'] ?? 0,
                !empty($context['anchor']) ? "Anchor text: \"" . $context['anchor'] . "\"" : "No anchor text available",
                $broken_url_base
            );
        } else {
            // Internal link - extract path for context
            $broken_url_path = wp_parse_url($broken_url, PHP_URL_PATH) ?: '';

            $prompt = sprintf(
                "A CONFIRMED BROKEN internal link (returning 404 error) was found on a WordPress site.\n\n" .
                "Broken URL: %s\n" .
                "Site domain: %s\n" .
                "URL path: %s\n" .
                "Appears in: %d post(s)\n" .
                "%s\n\n" .
                "This internal link has been verified as broken (404). Suggest ONE action:\n\n" .
                "1. REPLACEMENT PATH: If you can infer a likely correct path based on:\n" .
                "   - Common typos (e.g., '/abuot/' -> '/about/')\n" .
                "   - URL restructuring (e.g., '/blog/post-title/' might be '/post-title/')\n" .
                "   - The anchor text suggesting what content was intended\n" .
                "   Respond with just the path (e.g., /new-page/)\n\n" .
                "2. REMOVE: If you cannot confidently suggest a replacement.\n" .
                "   This is the safe default for clearly invalid paths.\n\n" .
                "3. UNKNOWN: Only if the path looks legitimate but you can't determine where it moved.\n\n" .
                "Respond with ONLY: the replacement path, REMOVE, or UNKNOWN - nothing else.",
                $broken_url,
                $site_domain,
                $broken_url_path,
                $context['post_count'] ?? 0,
                !empty($context['anchor']) ? "Anchor text: \"" . $context['anchor'] . "\"" : "No anchor text available"
            );
        }

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode([
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 200,
                'messages' => [
                    ['role' => 'user', 'content' => $prompt],
                ],
            ]),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return new WP_Error('api_error', sprintf(__('API returned status %d', 'screaming-fixes'), $status_code));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['content'][0]['text'])) {
            $suggestion = trim($body['content'][0]['text']);

            // Parse the response
            if ($suggestion === 'REMOVE') {
                return [
                    'action' => 'remove_link',
                    'new_url' => '',
                    'suggestion' => $suggestion,
                ];
            } elseif ($suggestion === 'UNKNOWN') {
                // For external links, treat UNKNOWN as remove_link (safer default)
                // For internal links, keep as unknown so user can decide
                if ($is_external) {
                    return [
                        'action' => 'remove_link',
                        'new_url' => '',
                        'suggestion' => $suggestion,
                    ];
                }
                return [
                    'action' => 'unknown',
                    'new_url' => '',
                    'suggestion' => $suggestion,
                ];
            } else {
                // It's a URL or path
                $new_url = $suggestion;

                if ($is_external) {
                    // For external links, if AI returned a full URL, use it directly
                    if (strpos($new_url, 'http') === 0) {
                        // Verify it's on the same external domain (safety check)
                        $suggested_domain = wp_parse_url($new_url, PHP_URL_HOST);
                        if ($suggested_domain !== $broken_url_domain) {
                            // AI suggested a different domain - that's suspicious, treat as unknown
                            return [
                                'action' => 'unknown',
                                'new_url' => '',
                                'suggestion' => $suggestion,
                            ];
                        }
                    } else {
                        // AI returned just a path, prepend the external domain's base URL
                        if (strpos($new_url, '/') !== 0) {
                            $new_url = '/' . $new_url;
                        }
                        $new_url = $broken_url_base . $new_url;
                    }
                } else {
                    // For internal links, build URL using site's home_url
                    if (strpos($new_url, '/') !== 0 && strpos($new_url, 'http') !== 0) {
                        $new_url = '/' . $new_url;
                    }
                    if (strpos($new_url, 'http') !== 0) {
                        $new_url = home_url($new_url);
                    }
                }

                return [
                    'action' => 'replace',
                    'new_url' => $new_url,
                    'suggestion' => $suggestion,
                ];
            }
        }

        return new WP_Error('api_error', __('Unexpected API response', 'screaming-fixes'));
    }

    /**
     * AJAX: Process uploaded CSV
     */
    public function ajax_process_csv() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

        if (empty($upload_id)) {
            wp_send_json_error(['message' => __('No upload ID provided.', 'screaming-fixes')]);
        }

        // Get the uploaded file path from the uploads option
        $uploads = get_option('sf_pending_uploads', []);

        if (!isset($uploads[$upload_id])) {
            wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
        }

        $file_path = $uploads[$upload_id]['path'];

        if (!file_exists($file_path)) {
            wp_send_json_error(['message' => __('File not found.', 'screaming-fixes')]);
        }

        // Check if this is a bulk fix CSV (has fix column)
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (!is_wp_error($parsed) && !empty($parsed['headers'])) {
            $bulk_check = $this->is_bulk_fix_csv($parsed['headers']);

            // Check for bulk errors first
            if (is_array($bulk_check) && isset($bulk_check['error'])) {
                if ($bulk_check['error'] === 'no_source_column') {
                    wp_send_json_error(['message' => __('Bulk fix CSV detected but missing a source URL column. Please include a column named "Source" or "Address".', 'screaming-fixes')]);
                    return;
                }
                if ($bulk_check['error'] === 'no_destination_column') {
                    wp_send_json_error(['message' => __('Bulk fix CSV detected but missing a destination column. Please include a column named "Destination" or "Broken_Link".', 'screaming-fixes')]);
                    return;
                }
            }

            // If it's a valid bulk fix CSV, process it differently
            if (is_array($bulk_check) && isset($bulk_check['is_bulk']) && $bulk_check['is_bulk'] === true) {
                $results = $this->process_bulk_csv($file_path);

                if (is_wp_error($results)) {
                    wp_send_json_error(['message' => $results->get_error_message()]);
                }

                // Clean up the temp file reference
                unset($uploads[$upload_id]);
                update_option('sf_pending_uploads', $uploads);

                wp_send_json_success([
                    'message' => sprintf(
                        __('%d broken link fixes ready to apply', 'screaming-fixes'),
                        $results['ready_count']
                    ),
                    'data' => $results,
                    'is_bulk_update' => true,
                ]);
                return;
            }
        }

        // Standard processing for analysis CSVs
        $results = $this->process_csv($file_path);

        if (is_wp_error($results)) {
            wp_send_json_error(['message' => $results->get_error_message()]);
        }

        // Clean up the temp file reference
        unset($uploads[$upload_id]);
        update_option('sf_pending_uploads', $uploads);

        wp_send_json_success([
            'message' => sprintf(
                __('Found %d broken links: %d fixable, %d manual, %d skipped.', 'screaming-fixes'),
                $results['total_count'],
                $results['fixable_count'],
                $results['manual_count'],
                $results['skipped_count']
            ),
            'data' => $results,
        ]);
    }

    /**
     * AJAX: Get stored data
     */
    public function ajax_get_data() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            wp_send_json_success(['data' => null]);
        }

        wp_send_json_success(['data' => $results]);
    }

    /**
     * AJAX: Apply fixes
     */
    public function ajax_apply_fixes() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $fixes = isset($_POST['fixes']) ? $_POST['fixes'] : [];

        if (empty($fixes)) {
            wp_send_json_error(['message' => __('No fixes provided.', 'screaming-fixes')]);
        }

        // Sanitize fixes array
        $sanitized_fixes = [];
        foreach ($fixes as $fix) {
            // Support both old 'url' and new 'broken_url' keys
            $broken_url = isset($fix['broken_url']) ? esc_url_raw($fix['broken_url']) : '';
            if (empty($broken_url) && isset($fix['url'])) {
                $broken_url = esc_url_raw($fix['url']);
            }

            // Sanitize post_ids array
            $post_ids = [];
            if (isset($fix['post_ids']) && is_array($fix['post_ids'])) {
                $post_ids = array_map('absint', $fix['post_ids']);
                $post_ids = array_filter($post_ids); // Remove zeros
            }

            $sanitized_fixes[] = [
                'broken_url' => $broken_url,
                'action' => isset($fix['action']) ? sanitize_text_field($fix['action']) : '',
                'new_url' => isset($fix['new_url']) ? esc_url_raw($fix['new_url']) : '',
                'post_ids' => $post_ids,
            ];
        }

        $results = $this->apply_fixes($sanitized_fixes);

        // Log to changelog for each affected post
        if ($results['success'] > 0) {
            $this->log_fixes_to_changelog($sanitized_fixes, $results);

            // Log to activity log for dashboard
            SF_Activity_Log::log('broken-links', $results['success']);
        }

        wp_send_json_success([
            'message' => sprintf(
                __('Fixed %d posts. %d failed.', 'screaming-fixes'),
                $results['success'],
                $results['failed']
            ),
            'results' => $results,
        ]);
    }

    /**
     * Log fixes to changelog for undo guidance
     *
     * @param array $fixes Array of fixes that were applied
     * @param array $results Results from apply_fixes
     */
    private function log_fixes_to_changelog($fixes, $results) {
        // Group fixes by post ID for consolidated logging
        $posts_affected = [];

        foreach ($fixes as $fix) {
            if (empty($fix['action']) || $fix['action'] === 'ignore') {
                continue;
            }

            $post_ids = $fix['post_ids'] ?? [];
            foreach ($post_ids as $post_id) {
                if (!$post_id) {
                    continue;
                }

                if (!isset($posts_affected[$post_id])) {
                    $posts_affected[$post_id] = [
                        'count' => 0,
                        'actions' => [],
                    ];
                }

                $posts_affected[$post_id]['count']++;
                $posts_affected[$post_id]['actions'][] = $fix['action'];
            }
        }

        // Create a changelog entry for each unique post
        foreach ($posts_affected as $post_id => $data) {
            $post = get_post($post_id);
            if (!$post) {
                continue;
            }

            // Determine the description based on actions
            $action_counts = array_count_values($data['actions']);
            $descriptions = [];

            if (isset($action_counts['replace'])) {
                $descriptions[] = sprintf(
                    _n('Replaced %d link', 'Replaced %d links', $action_counts['replace'], 'screaming-fixes'),
                    $action_counts['replace']
                );
            }
            if (isset($action_counts['remove_link'])) {
                $descriptions[] = sprintf(
                    _n('Removed %d hyperlink', 'Removed %d hyperlinks', $action_counts['remove_link'], 'screaming-fixes'),
                    $action_counts['remove_link']
                );
            }
            if (isset($action_counts['remove_all'])) {
                $descriptions[] = sprintf(
                    _n('Removed %d link with text', 'Removed %d links with text', $action_counts['remove_all'], 'screaming-fixes'),
                    $action_counts['remove_all']
                );
            }

            $description = !empty($descriptions)
                ? implode(', ', $descriptions)
                : sprintf(__('Fixed %d broken links', 'screaming-fixes'), $data['count']);

            SF_Changelog::log([
                'change_type'    => 'links_fixed',
                'module'         => 'broken-links',
                'description'    => $description,
                'post_id'        => $post_id,
                'post_title'     => $post->post_title,
                'post_url'       => get_permalink($post_id),
                'items_affected' => $data['count'],
            ]);
        }
    }

    /**
     * AJAX: Get AI suggestion for a single URL
     */
    public function ajax_get_ai_suggestion() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $post_count = isset($_POST['post_count']) ? absint($_POST['post_count']) : 0;
        $anchor = isset($_POST['anchor']) ? sanitize_text_field($_POST['anchor']) : '';

        if (empty($url)) {
            wp_send_json_error(['message' => __('No URL provided.', 'screaming-fixes')]);
        }

        $suggestion = $this->get_ai_suggestion($url, [
            'post_count' => $post_count,
            'anchor' => $anchor,
        ]);

        if (is_wp_error($suggestion)) {
            wp_send_json_error(['message' => $suggestion->get_error_message()]);
        }

        wp_send_json_success([
            'suggestion' => $suggestion,
            'url' => $url,
        ]);
    }

    /**
     * AJAX: Get AI suggestions for all broken links
     */
    public function ajax_ai_suggest_all() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results) || empty($results['broken_links'])) {
            wp_send_json_error(['message' => __('No broken links found.', 'screaming-fixes')]);
        }

        $suggestions = [];
        $success_count = 0;
        $failed_count = 0;

        foreach ($results['broken_links'] as $index => $link) {
            // Skip if already has a fix set
            if (!empty($link['fix'])) {
                continue;
            }

            $suggestion = $this->get_ai_suggestion($link['url'], [
                'post_count' => $link['post_count'],
                'anchor' => $link['anchor'] ?? '',
            ]);

            if (!is_wp_error($suggestion)) {
                $suggestions[$link['url']] = $suggestion;
                $success_count++;
            } else {
                $failed_count++;
            }

            // Small delay to avoid rate limiting
            usleep(100000); // 100ms
        }

        wp_send_json_success([
            'message' => sprintf(
                __('Generated %d suggestions. %d failed.', 'screaming-fixes'),
                $success_count,
                $failed_count
            ),
            'suggestions' => $suggestions,
        ]);
    }

    /**
     * AJAX: Export results as CSV
     */
    public function ajax_export() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        // Also check upload data if transient is empty
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        // Check for fixable_links (new structure) or broken_links (legacy)
        $fixable_links = $results['fixable_links'] ?? [];
        $manual_links = $results['manual_links'] ?? [];
        $skipped_links = $results['skipped_links'] ?? [];

        if (empty($fixable_links) && empty($manual_links) && empty($skipped_links)) {
            wp_send_json_error(['message' => __('No data to export.', 'screaming-fixes')]);
        }

        $export_data = [];

        // Export fixable links
        foreach ($fixable_links as $link) {
            $sources = $link['fixable_sources'] ?? $link['sources'] ?? [];
            $post_titles = array_column($sources, 'post_title');
            $source_count = count($sources);

            $export_data[] = [
                'Broken URL' => $link['broken_url'] ?? '',
                'Status Code' => $link['status_code'] ?? '',
                'Category' => 'Fixable',
                'Found In' => $source_count . ' ' . _n('page', 'pages', $source_count, 'screaming-fixes'),
                'Post Titles' => implode('; ', array_filter($post_titles)),
            ];
        }

        // Export manual links
        foreach ($manual_links as $link) {
            $sources = $link['manual_sources'] ?? $link['sources'] ?? [];
            $source_count = count($sources);
            $locations = array_unique(array_filter(array_column($sources, 'location')));

            $export_data[] = [
                'Broken URL' => $link['broken_url'] ?? '',
                'Status Code' => $link['status_code'] ?? '',
                'Category' => 'Manual Fix Required',
                'Found In' => $source_count . ' ' . _n('location', 'locations', $source_count, 'screaming-fixes'),
                'Post Titles' => implode('; ', $locations),
            ];
        }

        // Export skipped links
        foreach ($skipped_links as $link) {
            $sources = $link['skip_sources'] ?? $link['sources'] ?? [];
            $source_count = count($sources);

            $export_data[] = [
                'Broken URL' => $link['broken_url'] ?? '',
                'Status Code' => $link['status_code'] ?? '',
                'Category' => 'Skipped (Dynamic Page)',
                'Found In' => $source_count . ' ' . _n('page', 'pages', $source_count, 'screaming-fixes'),
                'Post Titles' => '',
            ];
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'broken-links-export-' . gmdate('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Export fixed links to CSV
     */
    public function ajax_export_fixed() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        // Also check upload data if transient is empty
        if (empty($results) || empty($results['fixed_links'])) {
            $results = $this->get_upload_data();
        }

        if (empty($results) || empty($results['fixed_links'])) {
            wp_send_json_error(['message' => __('No fixed links to export.', 'screaming-fixes')]);
        }

        $export_data = [];
        foreach ($results['fixed_links'] as $link) {
            $sources = $link['fixable_sources'] ?? $link['sources'] ?? [];
            $broken_url = $link['broken_url'] ?? '';
            $new_url = $link['new_url'] ?? '';
            $fix_action = $link['fix_action'] ?? 'replace';
            $is_removed = ($fix_action === 'remove' || $fix_action === 'remove_link' || $fix_action === 'remove_all');
            $status = $link['status'] ?? 'success';
            $status_message = $link['status_message'] ?? '';

            // Get status label
            if ($status === 'failed') {
                $status_label = 'Failed';
            } elseif ($status === 'skipped') {
                $status_label = 'Skipped';
            } else {
                $status_label = $is_removed ? 'Removed' : 'Replaced';
            }

            // Flatten: each source page becomes its own row
            foreach ($sources as $source) {
                $source_url = $source['source_url'] ?? '';
                $source_title = $source['post_title'] ?? '';

                // Format "Found In" as title + URL for clarity
                $found_in = $source_title ? $source_title . ' (' . $source_url . ')' : $source_url;

                $export_data[] = [
                    'Found In' => $found_in,
                    'Broken URL' => $broken_url,
                    'Fixed URL' => $is_removed ? 'Link removed' : $new_url,
                    'Status' => $status_label,
                    'Status Message' => $status_message,
                ];
            }
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'fixed-links-' . gmdate('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Clear all results
     */
    public function ajax_clear_results() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Clear transient data
        $this->clear_results();

        // Clear from uploads table
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ], ['%s', '%s']);

        // Also clear any pending uploads for this module
        $pending_uploads = get_option('sf_pending_uploads', []);
        foreach ($pending_uploads as $id => $upload) {
            if (isset($upload['module']) && $upload['module'] === 'broken-links') {
                // Delete temp file if exists
                if (!empty($upload['path']) && file_exists($upload['path'])) {
                    @unlink($upload['path']);
                }
                unset($pending_uploads[$id]);
            }
        }
        update_option('sf_pending_uploads', $pending_uploads);

        wp_send_json_success([
            'message' => __('Results cleared successfully.', 'screaming-fixes'),
        ]);
    }

    /**
     * AJAX: Get the Fixed Links section HTML
     * Used to refresh the Fixed section after applying fixes without page reload
     */
    public function ajax_get_fixed_section() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Get stored data
        $results = $this->get_results();
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        $all_fixed_links = $results['fixed_links'] ?? [];

        // Flatten fixed links so each source page becomes its own row
        $flattened_fixed_links = [];
        foreach ($all_fixed_links as $link) {
            $sources = $link['fixable_sources'] ?? $link['sources'] ?? [];
            $broken_url = $link['broken_url'] ?? '';
            $new_url = $link['new_url'] ?? '';
            $fix_action = $link['fix_action'] ?? 'replace';
            $fixed_at = $link['fixed_at'] ?? '';

            $status = $link['status'] ?? 'success';
            $status_message = $link['status_message'] ?? '';

            foreach ($sources as $source) {
                $flattened_fixed_links[] = [
                    'source_url' => $source['source_url'] ?? '',
                    'source_title' => $source['post_title'] ?? '',
                    'edit_url' => $source['edit_url'] ?? '',
                    'post_id' => $source['post_id'] ?? 0,
                    'anchor' => $source['anchor'] ?? '',
                    'broken_url' => $broken_url,
                    'new_url' => $new_url,
                    'fix_action' => $fix_action,
                    'fixed_at' => $fixed_at,
                    'status' => $status,
                    'status_message' => $status_message,
                ];
            }
        }

        $fixed_links = $flattened_fixed_links;
        $fixed_count = count($fixed_links);

        // Count by status
        $success_count = 0;
        $failed_count = 0;
        $skipped_count = 0;
        foreach ($fixed_links as $fl) {
            $s = $fl['status'] ?? 'success';
            if ($s === 'failed') { $failed_count++; }
            elseif ($s === 'skipped') { $skipped_count++; }
            else { $success_count++; }
        }

        // Generate the HTML
        ob_start();
        ?>
        <div class="sf-results-header sf-results-header-collapsible sf-results-header-fixed">
            <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="true">
                <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
                <span class="sf-fixes-applied-title">
                    <?php printf(
                        esc_html__('Fixes Applied (%d)', 'screaming-fixes'),
                        $fixed_count
                    ); ?>
                </span>
                <span class="sf-section-hint"><?php esc_html_e('View fix results and export to CSV', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
            </button>
            <div class="sf-results-actions">
                <button type="button" class="sf-button sf-button-secondary sf-export-fixed-btn">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <div class="sf-fixed-content">
            <div class="sf-status-filter-row sf-bl-status-filters">
                <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                <div class="sf-status-filters">
                    <button type="button" class="sf-status-filter active" data-status="all">
                        <span class="sf-status-icon">&#128203;</span>
                        <span class="sf-status-count sf-bl-status-count-all"><?php echo esc_html($fixed_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                        <span class="sf-status-icon">&#9989;</span>
                        <span class="sf-status-count sf-bl-status-count-success"><?php echo esc_html($success_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                        <span class="sf-status-icon">&#10060;</span>
                        <span class="sf-status-count sf-bl-status-count-failed"><?php echo esc_html($failed_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                        <span class="sf-status-icon">&#9203;</span>
                        <span class="sf-status-count sf-bl-status-count-skipped"><?php echo esc_html($skipped_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                    </button>
                </div>
            </div>
            <div class="sf-table-wrapper sf-paginated-table" data-section="fixed" data-per-page="100" data-total="<?php echo esc_attr($fixed_count); ?>">
                <table class="sf-results-table sf-fixed-table">
                    <thead>
                        <tr>
                            <th class="sf-col-source"><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                            <th class="sf-col-was"><?php esc_html_e('Was', 'screaming-fixes'); ?></th>
                            <th class="sf-col-now"><?php esc_html_e('Now', 'screaming-fixes'); ?></th>
                            <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fixed_links as $index => $link):
                            $source_url = $link['source_url'] ?? '';
                            $source_title = $link['source_title'] ?? '';
                            $edit_url = $link['edit_url'] ?? '';
                            $anchor = $link['anchor'] ?? '';
                            $broken_url = $link['broken_url'] ?? '';
                            $new_url = $link['new_url'] ?? '';
                            $fix_action = $link['fix_action'] ?? 'replace';
                            $is_removed = ($fix_action === 'remove' || $fix_action === 'remove_link' || $fix_action === 'remove_all');
                            $status = $link['status'] ?? 'success';
                            $status_message = $link['status_message'] ?? '';

                            if ($status === 'failed') {
                                $row_class = 'sf-fixed-row-failed';
                                $status_icon = '&#10060;';
                                $status_label = __('Failed', 'screaming-fixes');
                            } elseif ($status === 'skipped') {
                                $row_class = 'sf-fixed-row-skipped';
                                $status_icon = '&#9203;';
                                $status_label = __('Skipped', 'screaming-fixes');
                            } else {
                                $row_class = 'sf-fixed-row-success';
                                $status_icon = '&#9989;';
                                $status_label = $is_removed ? __('Removed', 'screaming-fixes') : __('Replaced', 'screaming-fixes');
                            }
                        ?>
                            <tr class="sf-fixed-row <?php echo esc_attr($row_class); ?>" data-status="<?php echo esc_attr($status); ?>" data-index="<?php echo esc_attr($index); ?>">
                                <td class="sf-col-source">
                                    <div class="sf-source-info">
                                        <?php if (!empty($edit_url)): ?>
                                            <a href="<?php echo esc_url($edit_url); ?>" target="_blank" rel="noopener" class="sf-source-title">
                                                <?php echo esc_html($source_title ?: __('(Untitled)', 'screaming-fixes')); ?>
                                            </a>
                                        <?php else: ?>
                                            <span class="sf-source-title"><?php echo esc_html($source_title ?: __('(Untitled)', 'screaming-fixes')); ?></span>
                                        <?php endif; ?>
                                        <div class="sf-source-url">
                                            <a href="<?php echo esc_url($source_url); ?>" target="_blank" rel="noopener">
                                                <?php echo esc_html(wp_parse_url($source_url, PHP_URL_PATH) ?: $source_url); ?>
                                            </a>
                                        </div>
                                        <?php if (!empty($anchor)): ?>
                                            <div class="sf-anchor-text">
                                                <span class="sf-anchor-label"><?php esc_html_e('Anchor:', 'screaming-fixes'); ?></span>
                                                <?php echo esc_html($anchor); ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="sf-col-was">
                                    <a href="<?php echo esc_url($broken_url); ?>" target="_blank" rel="noopener" class="sf-broken-url-link" title="<?php echo esc_attr($broken_url); ?>">
                                        <?php echo esc_html($broken_url); ?>
                                    </a>
                                </td>
                                <td class="sf-col-now">
                                    <?php if ($is_removed): ?>
                                        <span class="sf-removed-text"><?php esc_html_e('Link removed', 'screaming-fixes'); ?></span>
                                    <?php else: ?>
                                        <a href="<?php echo esc_url($new_url); ?>" target="_blank" rel="noopener" class="sf-new-url-link" title="<?php echo esc_attr($new_url); ?>">
                                            <?php echo esc_html($new_url); ?>
                                        </a>
                                    <?php endif; ?>
                                </td>
                                <td class="sf-col-status">
                                    <span class="sf-status-badge sf-status-<?php echo esc_attr($status); ?>"<?php if (!empty($status_message) && $status !== 'success') echo ' title="' . esc_attr($status_message) . '"'; ?>>
                                        <span class="sf-status-icon"><?php echo $status_icon; ?></span>
                                        <?php echo esc_html($status_label); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($fixed_count > 100): ?>
            <div class="sf-pagination" data-section="fixed">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                    <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $fixed_count)); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?>
                    <span class="sf-page-total"><?php echo esc_html($fixed_count); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" disabled title="<?php esc_attr_e('Previous page', 'screaming-fixes'); ?>">
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-btn sf-page-next" title="<?php esc_attr_e('Next page', 'screaming-fixes'); ?>">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
        $html = ob_get_clean();

        wp_send_json_success([
            'html' => $html,
            'fixed_count' => $fixed_count,
        ]);
    }

    /**
     * Save upload data to database
     *
     * @param array $data Data to save
     */
    private function save_upload_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        // Delete any existing data for this session/module
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Insert new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get upload data from database
     *
     * @return array|null Data or null if not found
     */
    private function get_upload_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            return json_decode($row->data, true);
        }

        return null;
    }

    /**
     * Get or create session ID
     *
     * @return string Session ID
     */
    private function get_session_id() {
        if (!session_id()) {
            // Use user ID as session for logged-in users
            return 'user_' . get_current_user_id();
        }
        return session_id();
    }

    /**
     * Update results after fixes are applied
     *
     * Moves fixed URLs to a separate fixed_links array and removes them from
     * category arrays. This ensures users can see their completed fixes.
     *
     * @param array $fixes Applied fixes
     * @param array $apply_results Results from apply_fixes() with success/failure info
     */
    private function update_results_after_fixes($fixes, $apply_results = []) {
        $results = $this->get_results();

        // Also check upload data if transient is empty
        if (empty($results) || empty($results['broken_links'])) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            return;
        }

        // Initialize fixed_links array if not exists
        if (!isset($results['fixed_links'])) {
            $results['fixed_links'] = [];
        }

        // Build a map of failed URLs with their error messages
        $failed_url_messages = [];
        if (!empty($apply_results['errors'])) {
            foreach ($apply_results['errors'] as $error) {
                if (!empty($error['broken_url'])) {
                    $error_msg = $error['error'] ?? __('Unknown error', 'screaming-fixes');
                    if (strpos($error_msg, 'not found') !== false || strpos($error_msg, 'No occurrences') !== false) {
                        $error_msg = __('URL not found in this content. It may have already been fixed or the URL is on an external site.', 'screaming-fixes');
                    } elseif (strpos($error_msg, 'permission') !== false) {
                        $error_msg = __('Permission denied. You may not have access to edit this content.', 'screaming-fixes');
                    } elseif (strpos($error_msg, 'post does not exist') !== false || strpos($error_msg, 'Invalid post') !== false) {
                        $error_msg = __('This page no longer exists in the database.', 'screaming-fixes');
                    }
                    $failed_url_messages[$error['broken_url']] = $error_msg;
                }
            }
        }

        // Create a map of ALL attempted fixes (success + failed) with status
        $fixed_urls = [];
        foreach ($fixes as $fix) {
            if (!empty($fix['action']) && $fix['action'] !== 'ignore') {
                $url = $fix['broken_url'] ?? $fix['url'] ?? '';
                if (!empty($url)) {
                    if (isset($failed_url_messages[$url])) {
                        $fixed_urls[$url] = [
                            'action' => $fix['action'],
                            'new_url' => $fix['new_url'] ?? '',
                            'fixed_at' => current_time('mysql'),
                            'status' => 'failed',
                            'status_message' => $failed_url_messages[$url],
                        ];
                    } else {
                        $fixed_urls[$url] = [
                            'action' => $fix['action'],
                            'new_url' => $fix['new_url'] ?? '',
                            'fixed_at' => current_time('mysql'),
                            'status' => 'success',
                            'status_message' => '',
                        ];
                    }
                }
            }
        }

        if (empty($fixed_urls)) {
            return;
        }

        // Helper function to find and extract fixed URLs from an array
        $extract_fixed = function ($links, $source_type) use ($fixed_urls, &$results) {
            $remaining = [];
            foreach ($links as $link) {
                $broken_url = $link['broken_url'] ?? '';
                if (isset($fixed_urls[$broken_url])) {
                    // Add to fixed_links with fix details and status
                    $fixed_link = $link;
                    $fixed_link['fix_action'] = $fixed_urls[$broken_url]['action'];
                    $fixed_link['new_url'] = $fixed_urls[$broken_url]['new_url'];
                    $fixed_link['fixed_at'] = $fixed_urls[$broken_url]['fixed_at'];
                    $fixed_link['status'] = $fixed_urls[$broken_url]['status'];
                    $fixed_link['status_message'] = $fixed_urls[$broken_url]['status_message'];
                    $fixed_link['original_category'] = $source_type;
                    $results['fixed_links'][] = $fixed_link;
                } else {
                    $remaining[] = $link;
                }
            }
            return $remaining;
        };

        // Extract fixed URLs from specific category arrays only
        // NOTE: Skip $results['broken_links'] here because it is a merged superset
        // of fixable_links + manual_links + skipped_links. Extracting from it AND
        // the individual arrays would create duplicate entries in fixed_links.
        if (!empty($results['fixable_links'])) {
            $results['fixable_links'] = $extract_fixed($results['fixable_links'], 'fixable');
        }

        if (!empty($results['manual_links'])) {
            $results['manual_links'] = $extract_fixed($results['manual_links'], 'manual');
        }

        if (!empty($results['skipped_links'])) {
            $results['skipped_links'] = $extract_fixed($results['skipped_links'], 'skipped');
        }

        // Rebuild broken_links as the merged superset (minus the fixed ones)
        $results['broken_links'] = array_merge(
            $results['fixable_links'] ?? [],
            $results['manual_links'] ?? [],
            $results['skipped_links'] ?? []
        );

        // Recalculate all counts
        $results['total_count'] = count($results['broken_links'] ?? []);
        $results['fixable_count'] = count($results['fixable_links'] ?? []);
        $results['manual_count'] = count($results['manual_links'] ?? []);
        $results['skipped_count'] = count($results['skipped_links'] ?? []);
        $results['fixed_count'] = count($results['fixed_links'] ?? []);

        // Recalculate source counts
        $total_sources = 0;
        $fixable_sources = 0;
        $manual_sources = 0;
        $skipped_sources = 0;

        foreach ($results['fixable_links'] ?? [] as $link) {
            $count = $link['fixable_count'] ?? count($link['fixable_sources'] ?? $link['sources'] ?? []);
            $fixable_sources += $count;
            $total_sources += $count;
        }

        foreach ($results['manual_links'] ?? [] as $link) {
            $count = $link['manual_count'] ?? count($link['manual_sources'] ?? $link['sources'] ?? []);
            $manual_sources += $count;
            $total_sources += $count;
        }

        foreach ($results['skipped_links'] ?? [] as $link) {
            $count = $link['skip_count'] ?? count($link['skip_sources'] ?? $link['sources'] ?? []);
            $skipped_sources += $count;
            $total_sources += $count;
        }

        $results['total_sources'] = $total_sources;
        $results['fixable_sources'] = $fixable_sources;
        $results['manual_sources'] = $manual_sources;
        $results['skipped_sources'] = $skipped_sources;

        // Update the processed_at timestamp to indicate data was modified
        $results['last_updated'] = current_time('mysql');

        // Save updated results to both transient and database
        $this->save_results($results);
        $this->save_upload_data($results);
    }

    /**
     * Get module ID
     *
     * @return string Module ID
     */
    public function get_module_id() {
        return $this->module_id;
    }

    /**
     * ============================================
     * BULK UPLOAD METHODS
     * ============================================
     */

    /**
     * Normalize a column name for flexible matching
     * Lowercases, removes spaces, dashes, and underscores
     *
     * @param string $name Column name
     * @return string Normalized name
     */
    private function normalize_column_name($name) {
        $normalized = strtolower(trim($name));
        $normalized = str_replace([' ', '-', '_'], '', $normalized);
        return $normalized;
    }

    /**
     * Normalize a URL for matching
     * Strips trailing slashes and normalizes protocol
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private function normalize_url($url) {
        $url = trim($url);
        // Remove trailing slash
        $url = rtrim($url, '/');
        // Normalize protocol - convert http to https for comparison
        $url = preg_replace('/^http:\/\//i', 'https://', $url);
        // Convert to lowercase for comparison
        $url = strtolower($url);
        return $url;
    }

    /**
     * Check if CSV is a bulk fix CSV (has a fix column)
     *
     * @param array $headers CSV headers
     * @return array|false Array with column header info or false if not a bulk CSV
     */
    public function is_bulk_fix_csv($headers) {
        $fix_header = null;
        $source_header = null;
        $destination_header = null;

        // Fix column patterns (the new URL to replace the broken one)
        $fix_patterns = ['brokenlinkfix', 'newdestination', 'newlink', 'fixedurl', 'replacementurl', 'fixurl', 'newurl'];

        // Source column patterns (where the broken link was found)
        $source_patterns = ['source', 'address', 'url', 'pageurl', 'sourceurl'];

        // Destination column patterns (the broken link itself)
        $destination_patterns = ['destination', 'brokenlink', 'oldurl', 'brokenurl', 'linkurl'];

        foreach ($headers as $header) {
            $normalized = $this->normalize_column_name($header);
            $header_lower = strtolower(trim($header));

            // Check for fix column
            if (in_array($normalized, $fix_patterns)) {
                $fix_header = $header_lower;
            }

            // Check for source column
            if (in_array($normalized, $source_patterns)) {
                $source_header = $header_lower;
            }

            // Check for destination column
            if (in_array($normalized, $destination_patterns)) {
                $destination_header = $header_lower;
            }
        }

        // Not a bulk fix CSV if no fix column
        if ($fix_header === null) {
            return false;
        }

        // Must have source column for matching to WordPress
        if ($source_header === null) {
            return ['error' => 'no_source_column'];
        }

        // Must have destination column for finding the broken link
        if ($destination_header === null) {
            return ['error' => 'no_destination_column'];
        }

        return [
            'is_bulk' => true,
            'fix_header' => $fix_header,
            'source_header' => $source_header,
            'destination_header' => $destination_header,
        ];
    }

    /**
     * Process a bulk fix CSV file
     *
     * @param string $file_path Path to CSV file
     * @return array|WP_Error Processing results
     */
    public function process_bulk_csv($file_path) {
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Check if this is a bulk fix CSV
        $bulk_check = $this->is_bulk_fix_csv($parsed['headers']);

        if ($bulk_check === false) {
            return new WP_Error('not_bulk_csv', __('CSV must contain a fix column (Broken_Link_Fix, New_Destination, etc.) for bulk updates.', 'screaming-fixes'));
        }

        if (isset($bulk_check['error'])) {
            if ($bulk_check['error'] === 'no_source_column') {
                return new WP_Error('no_source_column', __('CSV must contain a source URL column (Source, Address, etc.).', 'screaming-fixes'));
            }
            if ($bulk_check['error'] === 'no_destination_column') {
                return new WP_Error('no_destination_column', __('CSV must contain a destination column (Destination, Broken_Link, etc.) for matching broken links.', 'screaming-fixes'));
            }
        }

        // Get the header names
        $fix_header = $bulk_check['fix_header'];
        $source_header = $bulk_check['source_header'];
        $destination_header = $bulk_check['destination_header'];

        // Build WordPress URL lookup map
        $wp_urls = $this->build_wp_url_map();

        // Process rows - track by source+destination for duplicate handling
        $seen_keys = [];
        $ready_fixes = [];
        $not_matched = [];
        $skipped_empty = [];
        $duplicates_overwritten = 0;
        $url_warnings = [];

        foreach ($parsed['rows'] as $row_index => $row) {
            $source_url = isset($row[$source_header]) ? trim($row[$source_header]) : '';
            $destination_url = isset($row[$destination_header]) ? trim($row[$destination_header]) : '';
            $fix_url = isset($row[$fix_header]) ? trim($row[$fix_header]) : '';

            if (empty($source_url) || empty($destination_url)) {
                continue; // Skip rows without required URLs
            }

            // Skip rows with empty fix URL
            if (empty($fix_url)) {
                $skipped_empty[] = [
                    'source_url' => $source_url,
                    'destination_url' => $destination_url,
                    'fix_url' => '',
                    'status' => 'Skipped - No fix URL provided',
                ];
                continue;
            }

            // Validate fix URL and add warnings for suspicious patterns
            $url_warning = $this->validate_fix_url($fix_url);
            if ($url_warning) {
                $url_warnings[] = [
                    'source_url' => $source_url,
                    'fix_url' => $fix_url,
                    'warning' => $url_warning,
                ];
            }

            // Track duplicates - use last occurrence (overwrite previous)
            $key = $this->normalize_url($source_url) . '|' . $this->normalize_url($destination_url);
            if (isset($seen_keys[$key])) {
                $duplicates_overwritten++;
                // Remove the previous entry from ready_fixes
                foreach ($ready_fixes as $idx => $existing) {
                    $existing_key = $this->normalize_url($existing['source_url']) . '|' . $this->normalize_url($existing['broken_url']);
                    if ($existing_key === $key) {
                        unset($ready_fixes[$idx]);
                        break;
                    }
                }
                $ready_fixes = array_values($ready_fixes); // Re-index
            }
            $seen_keys[$key] = true;

            // Try to match source URL to WordPress
            $normalized_source = $this->normalize_url($source_url);
            $post_id = null;
            $post_title = '';

            if (isset($wp_urls[$normalized_source])) {
                $post_id = $wp_urls[$normalized_source]['post_id'];
                $post_title = get_the_title($post_id);
            }

            if ($post_id) {
                $ready_fixes[] = [
                    'source_url' => $source_url,
                    'broken_url' => $destination_url,
                    'fix_url' => $fix_url,
                    'post_id' => $post_id,
                    'post_title' => $post_title,
                    'status' => 'Ready',
                ];
            } else {
                $not_matched[] = [
                    'source_url' => $source_url,
                    'broken_url' => $destination_url,
                    'fix_url' => $fix_url,
                    'status' => 'Skipped - Source URL not found in WordPress',
                ];
            }
        }

        $results = [
            'is_bulk_update' => true,
            'ready_fixes' => $ready_fixes,
            'not_matched' => $not_matched,
            'skipped_empty' => $skipped_empty,
            'url_warnings' => $url_warnings,
            'ready_count' => count($ready_fixes),
            'not_matched_count' => count($not_matched),
            'skipped_empty_count' => count($skipped_empty),
            'duplicates_overwritten' => $duplicates_overwritten,
            'total_rows' => count($parsed['rows']),
        ];

        // Store bulk data for later use
        $this->save_bulk_data($results);

        return $results;
    }

    /**
     * Validate a fix URL and return warning message if suspicious
     *
     * @param string $url URL to validate
     * @return string|null Warning message or null if valid
     */
    private function validate_fix_url($url) {
        // Check for relative URL without leading slash
        if (!preg_match('/^(https?:\/\/|\/)/i', $url)) {
            return __('Relative URL without leading slash - may not work as expected', 'screaming-fixes');
        }

        // Check for path navigation
        if (strpos($url, '../') !== false || strpos($url, './') !== false) {
            return __('URL contains path navigation (./ or ../) - may not work as expected', 'screaming-fixes');
        }

        // Check for non-standard protocols
        if (preg_match('/^[a-z]+:/i', $url) && !preg_match('/^https?:/i', $url)) {
            return __('Non-standard URL protocol detected', 'screaming-fixes');
        }

        return null;
    }

    /**
     * Build a map of WordPress URLs to post IDs
     *
     * @return array URL map
     */
    private function build_wp_url_map() {
        global $wpdb;

        $map = [];

        // Get all published posts and pages
        $posts = $wpdb->get_results(
            "SELECT ID, post_name, post_type FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ('post', 'page')
             LIMIT 50000",
            ARRAY_A
        );

        foreach ($posts as $post) {
            $permalink = get_permalink($post['ID']);
            if ($permalink) {
                $normalized = $this->normalize_url($permalink);
                $map[$normalized] = [
                    'post_id' => (int) $post['ID'],
                    'post_type' => $post['post_type'],
                ];
            }
        }

        return $map;
    }

    /**
     * Apply bulk fixes in a batch
     *
     * @param array $fixes Array of fixes to apply
     * @param int $offset Starting offset
     * @param int $batch_size Batch size
     * @return array Results
     */
    public function apply_bulk_fixes($fixes, $offset = 0, $batch_size = 50) {
        $fixer = new SF_Link_Fixer();

        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => [],
            'details' => [],
            'complete' => false,
            'next_offset' => $offset + $batch_size,
        ];

        $batch = array_slice($fixes, $offset, $batch_size);

        if (empty($batch)) {
            $results['complete'] = true;
            return $results;
        }

        foreach ($batch as $fix) {
            $results['processed']++;

            $post_id = isset($fix['post_id']) ? (int) $fix['post_id'] : 0;
            $broken_url = isset($fix['broken_url']) ? trim($fix['broken_url']) : '';
            $fix_url = isset($fix['fix_url']) ? trim($fix['fix_url']) : '';
            $source_url = isset($fix['source_url']) ? $fix['source_url'] : '';

            if (!$post_id || empty($broken_url) || empty($fix_url)) {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $source_url,
                    'broken_url' => $broken_url,
                    'error' => __('Invalid post ID or missing URLs', 'screaming-fixes'),
                ];
                continue;
            }

            // Apply the fix using SF_Link_Fixer
            $fix_result = $fixer->replace_url_in_post(
                $post_id,
                $broken_url,
                $fix_url,
                'replace',
                'broken-links-bulk'
            );

            if (is_wp_error($fix_result)) {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $source_url,
                    'broken_url' => $broken_url,
                    'post_id' => $post_id,
                    'error' => $fix_result->get_error_message(),
                ];
            } else {
                $results['success']++;
                $results['details'][] = [
                    'source_url' => $source_url,
                    'broken_url' => $broken_url,
                    'fix_url' => $fix_url,
                    'post_id' => $post_id,
                    'post_title' => get_the_title($post_id),
                ];
            }
        }

        // Check if we've processed all fixes
        if ($offset + $batch_size >= count($fixes)) {
            $results['complete'] = true;
        }

        return $results;
    }

    /**
     * Save bulk data
     *
     * @param array $data Bulk data
     */
    private function save_bulk_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        // Clear any existing data
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Save new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get stored bulk data
     *
     * @return array|null Bulk data or null
     */
    private function get_bulk_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            $data = json_decode($row->data, true);
            if (isset($data['is_bulk_update']) && $data['is_bulk_update']) {
                return $data;
            }
        }

        return null;
    }

    /**
     * AJAX: Apply bulk fixes (handles batching)
     */
    public function ajax_apply_bulk_fixes() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $offset = isset($_POST['offset']) ? (int) $_POST['offset'] : 0;
        $batch_size = isset($_POST['batch_size']) ? (int) $_POST['batch_size'] : 50;

        // Clear accumulated transient at the start of a new batch run
        if ($offset === 0) {
            delete_transient('sf_bulk_links_accumulated_' . get_current_user_id());
        }

        // Get stored bulk data
        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data) || empty($bulk_data['ready_fixes'])) {
            wp_send_json_error(['message' => __('No bulk fix data found.', 'screaming-fixes')]);
        }

        $fixes = $bulk_data['ready_fixes'];
        $results = $this->apply_bulk_fixes($fixes, $offset, $batch_size);

        // If complete, store the results for display
        if ($results['complete']) {
            // Get accumulated results from all batches
            $accumulated = get_transient('sf_bulk_links_accumulated_' . get_current_user_id());
            if (!$accumulated) {
                $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
            }

            $accumulated['success'] += $results['success'];
            $accumulated['failed'] += $results['failed'];
            $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
            $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

            // Store final results
            $final_results = [
                'is_bulk_update' => true,
                'bulk_complete' => true,
                'fixed_links' => $accumulated['details'],
                'failed_fixes' => $accumulated['errors'],
                'success_count' => $accumulated['success'],
                'failed_count' => $accumulated['failed'],
                'not_matched' => $bulk_data['not_matched'] ?? [],
                'skipped_empty' => $bulk_data['skipped_empty'] ?? [],
            ];

            $this->save_bulk_data($final_results);

            // Clean up accumulated transient
            delete_transient('sf_bulk_links_accumulated_' . get_current_user_id());

            // Log to activity log
            if ($accumulated['success'] > 0) {
                SF_Activity_Log::log('broken-links', $accumulated['success']);
            }

            $results['total_success'] = $accumulated['success'];
            $results['total_failed'] = $accumulated['failed'];
            $results['all_details'] = $accumulated['details'];
            $results['all_errors'] = $accumulated['errors'];
        } else {
            // Accumulate results for multi-batch operations
            $accumulated = get_transient('sf_bulk_links_accumulated_' . get_current_user_id());
            if (!$accumulated) {
                $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
            }

            $accumulated['success'] += $results['success'];
            $accumulated['failed'] += $results['failed'];
            $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
            $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

            set_transient('sf_bulk_links_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
        }

        $results['total_count'] = count($fixes);

        wp_send_json_success($results);
    }

    /**
     * AJAX: Cancel bulk fix (clear data)
     */
    public function ajax_cancel_bulk() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Clear bulk data
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Clear accumulated transient
        delete_transient('sf_bulk_links_accumulated_' . get_current_user_id());

        wp_send_json_success(['message' => __('Bulk fix cancelled.', 'screaming-fixes')]);
    }

    /**
     * AJAX: Download preview CSV
     */
    public function ajax_download_preview() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data)) {
            wp_die(__('No bulk data found.', 'screaming-fixes'));
        }

        $csv_content = $this->generate_preview_csv($bulk_data);
        $filename = 'broken-links-preview-' . date('Y-m-d-His') . '.csv';

        // Send CSV headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output BOM for Excel compatibility
        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * AJAX: Download results CSV
     */
    public function ajax_download_results() {
        check_ajax_referer('sf_broken_links_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data)) {
            wp_die(__('No results data found.', 'screaming-fixes'));
        }

        $csv_content = $this->generate_results_csv($bulk_data);
        $filename = 'broken-links-results-' . date('Y-m-d-His') . '.csv';

        // Send CSV headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output BOM for Excel compatibility
        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * Generate preview CSV data for download
     *
     * @param array $bulk_data Bulk data
     * @return string CSV content
     */
    public function generate_preview_csv($bulk_data) {
        $lines = [];

        // Header
        $lines[] = 'Source URL,Broken URL,Fix URL,Post Title,Status';

        // Ready fixes
        foreach ($bulk_data['ready_fixes'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['source_url'],
                $row['broken_url'],
                $row['fix_url'],
                $row['post_title'] ?? '',
                'Ready',
            ]);
        }

        // Not matched
        foreach ($bulk_data['not_matched'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['source_url'],
                $row['broken_url'],
                $row['fix_url'],
                '',
                'Skipped - Source URL not found',
            ]);
        }

        // Skipped empty
        foreach ($bulk_data['skipped_empty'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['source_url'],
                $row['destination_url'] ?? '',
                '',
                '',
                'Skipped - No fix URL provided',
            ]);
        }

        return implode("\n", $lines);
    }

    /**
     * Generate results CSV data for download
     *
     * @param array $bulk_data Bulk results data
     * @return string CSV content
     */
    public function generate_results_csv($bulk_data) {
        $lines = [];

        // Header
        $lines[] = 'Source URL,Broken URL,Fix URL,Post Title,Status';

        // Successful fixes
        foreach ($bulk_data['fixed_links'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['source_url'],
                $row['broken_url'],
                $row['fix_url'],
                $row['post_title'] ?? '',
                'Fixed',
            ]);
        }

        // Failed fixes
        foreach ($bulk_data['failed_fixes'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['source_url'],
                $row['broken_url'] ?? '',
                '',
                '',
                'Failed - ' . ($row['error'] ?? 'Unknown error'),
            ]);
        }

        // Not matched (from original upload)
        foreach ($bulk_data['not_matched'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['source_url'],
                $row['broken_url'],
                $row['fix_url'],
                '',
                'Skipped - Source URL not found',
            ]);
        }

        return implode("\n", $lines);
    }

    /**
     * Escape a row for CSV output
     *
     * @param array $fields Fields to escape
     * @return string Escaped CSV row
     */
    private function csv_escape_row($fields) {
        $escaped = [];
        foreach ($fields as $field) {
            // Escape quotes and wrap in quotes if contains comma, quote, or newline
            if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
                $field = '"' . str_replace('"', '""', $field) . '"';
            }
            $escaped[] = $field;
        }
        return implode(',', $escaped);
    }
}
