<?php
/**
 * Broken Links Tab Content
 *
 * Main view for the Broken Links module with categorized sections:
 * - Fixable: Links in post/page content that can be auto-fixed
 * - Manual: Links in navigation, footer, widgets that need manual fixing
 * - Skipped: Dynamic pages (pagination, archives) that will be fixed when source is fixed
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Helper function to render the sources column
 * Defined at top to ensure it's available when called
 *
 * For single source: shows the source directly
 * For multiple sources: shows expandable list limited to 10, with "Show all X" for larger lists
 *
 * @param array  $sources    Array of source data
 * @param string $broken_url The broken URL
 * @param array  $link_data  Optional full link data with fixable_sources, manual_sources, etc.
 */
if (!function_exists('sf_render_sources_column')):
function sf_render_sources_column($sources, $broken_url, $link_data = null) {
    $source_count = count($sources);
    $display_limit = 10;

    // Check if this link also has manual sources (for fixable section display)
    $manual_sources = [];
    $manual_count = 0;
    if ($link_data) {
        $manual_sources = $link_data['manual_sources'] ?? [];
        $manual_count = count($manual_sources);
    }

    ob_start();

    if ($source_count === 1 && $manual_count === 0):
        // Single source, no manual sources - show directly without toggle
        $source = $sources[0];
        ?>
        <div class="sf-single-source">
            <div class="sf-source-count-label"><?php esc_html_e('Found on 1 page', 'screaming-fixes'); ?></div>
            <?php if (!empty($source['post_id'])): ?>
                <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" rel="noopener" class="sf-source-title">
                    <?php echo esc_html($source['post_title']); ?>
                </a>
            <?php else: ?>
                <span class="sf-source-not-found">
                    <span class="dashicons dashicons-warning"></span>
                    <?php esc_html_e('Post not found', 'screaming-fixes'); ?>
                </span>
            <?php endif; ?>
            <div class="sf-source-url">
                <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" rel="noopener" class="sf-source-url-link">
                    <?php echo esc_html(wp_parse_url($source['source_url'], PHP_URL_PATH) ?: $source['source_url']); ?>
                </a>
            </div>
            <?php if (!empty($source['anchor'])): ?>
                <div class="sf-anchor-text">
                    <small><?php esc_html_e('Anchor:', 'screaming-fixes'); ?></small>
                    <?php echo esc_html($source['anchor']); ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    else:
        // Multiple sources or has manual sources - show toggle button with expandable list
        $sources_json = wp_json_encode(array_map(function($s) {
            return [
                'url' => $s['source_url'] ?? '',
                'path' => wp_parse_url($s['source_url'] ?? '', PHP_URL_PATH) ?: ($s['source_url'] ?? ''),
                'post_id' => $s['post_id'] ?? 0,
                'post_title' => $s['post_title'] ?? '',
                'edit_url' => $s['edit_url'] ?? '',
                'anchor' => $s['anchor'] ?? '',
                'fix_category' => $s['fix_category'] ?? 'fixable',
                'fix_note' => $s['fix_note'] ?? '',
                'location' => $s['location'] ?? '',
            ];
        }, $sources));

        $total_sources = $source_count + $manual_count;
        ?>
        <div class="sf-sources-wrapper">
            <button type="button" class="sf-sources-toggle"
                    data-broken-url="<?php echo esc_attr($broken_url); ?>"
                    data-sources="<?php echo esc_attr($sources_json); ?>"
                    data-total="<?php echo esc_attr($total_sources); ?>">
                <?php printf(
                    esc_html__('Found on %d pages', 'screaming-fixes'),
                    $total_sources
                ); ?>
                <span class="dashicons dashicons-arrow-down-alt2"></span>
            </button>
            <div class="sf-sources-list" style="display: none;">
                <?php if ($source_count > 0): ?>
                <div class="sf-sources-category sf-sources-fixable">
                    <div class="sf-sources-category-header">
                        <span class="sf-category-badge sf-badge-fixable">&#10003;</span>
                        <?php printf(
                            esc_html(_n('%d Fixable (in content)', '%d Fixable (in content)', $source_count, 'screaming-fixes')),
                            $source_count
                        ); ?>
                    </div>
                    <ul class="sf-sources-ul">
                        <?php
                        $displayed = 0;
                        foreach ($sources as $source):
                            if ($displayed >= $display_limit) break;
                            $displayed++;
                            $path = wp_parse_url($source['source_url'], PHP_URL_PATH) ?: $source['source_url'];
                        ?>
                            <li class="sf-source-li">
                                <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" rel="noopener" class="sf-source-path-link">
                                    <?php echo esc_html($path); ?>
                                </a>
                                <?php if (!empty($source['post_id'])): ?>
                                    <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" rel="noopener" class="sf-edit-link" title="<?php esc_attr_e('Edit post', 'screaming-fixes'); ?>">
                                        <span class="dashicons dashicons-edit"></span>
                                    </a>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                    <?php if ($source_count > $display_limit): ?>
                        <div class="sf-sources-more">
                            <?php printf(
                                esc_html__('+ %d more fixable sources', 'screaming-fixes'),
                                $source_count - $display_limit
                            ); ?>
                        </div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>

                <?php if ($manual_count > 0): ?>
                <div class="sf-sources-category sf-sources-manual">
                    <div class="sf-sources-category-header">
                        <span class="sf-category-badge sf-badge-manual">&#9888;</span>
                        <?php printf(
                            esc_html(_n('%d Manual Fix Required', '%d Manual Fixes Required', $manual_count, 'screaming-fixes')),
                            $manual_count
                        ); ?>
                    </div>
                    <ul class="sf-sources-ul sf-manual-sources-ul">
                        <?php
                        $displayed = 0;
                        foreach ($manual_sources as $source):
                            if ($displayed >= 5) break; // Show fewer manual sources
                            $displayed++;
                            $path = wp_parse_url($source['source_url'], PHP_URL_PATH) ?: $source['source_url'];
                            $location = $source['location'] ?? '';
                            $note = $source['fix_note'] ?? '';
                        ?>
                            <li class="sf-source-li sf-source-manual">
                                <div class="sf-source-manual-info">
                                    <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" rel="noopener" class="sf-source-path-link">
                                        <?php echo esc_html($path); ?>
                                    </a>
                                    <?php if ($location): ?>
                                        <span class="sf-source-location-badge"><?php echo esc_html(ucfirst($location)); ?></span>
                                    <?php endif; ?>
                                </div>
                                <?php if ($note): ?>
                                    <div class="sf-source-manual-note">
                                        <small><?php echo esc_html($note); ?></small>
                                    </div>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                    <?php if ($manual_count > 5): ?>
                        <div class="sf-sources-more">
                            <?php printf(
                                esc_html__('+ %d more manual sources', 'screaming-fixes'),
                                $manual_count - 5
                            ); ?>
                        </div>
                    <?php endif; ?>
                    <div class="sf-manual-note-tip">
                        <small><span class="dashicons dashicons-info"></span> <?php esc_html_e('Manual sources require fixing in theme settings, widgets, or menus.', 'screaming-fixes'); ?></small>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($total_sources > $display_limit): ?>
                    <button type="button" class="sf-show-all-sources"
                            data-broken-url="<?php echo esc_attr($broken_url); ?>"
                            data-total="<?php echo esc_attr($total_sources); ?>">
                        <?php printf(
                            esc_html__('Show all %d sources', 'screaming-fixes'),
                            $total_sources
                        ); ?>
                    </button>
                <?php endif; ?>
            </div>
        </div>
        <?php
    endif;

    return ob_get_clean();
}
endif;

// Get module instance and data
$module = SF_Module_Loader::get_module('broken-links');
$results = $module ? $module->get_results() : null;

// Check for upload data if no transient results
if (empty($results)) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
    $session_id = 'user_' . get_current_user_id();

    $row = $wpdb->get_row($wpdb->prepare(
        "SELECT data FROM {$table_name}
         WHERE session_id = %s AND module = %s AND expires_at > NOW()
         ORDER BY created_at DESC LIMIT 1",
        $session_id,
        'broken-links'
    ));

    if ($row) {
        $results = json_decode($row->data, true);
    }
}

// Check if this is a bulk update result
$is_bulk_update = !empty($results['is_bulk_update']);
$bulk_complete = !empty($results['bulk_complete']);

// Bulk update specific data
$bulk_ready_fixes = $is_bulk_update ? ($results['ready_fixes'] ?? []) : [];
$bulk_not_matched = $is_bulk_update ? ($results['not_matched'] ?? []) : [];
$bulk_skipped_empty = $is_bulk_update ? ($results['skipped_empty'] ?? []) : [];
$bulk_url_warnings = $is_bulk_update ? ($results['url_warnings'] ?? []) : [];
$bulk_ready_count = $is_bulk_update ? ($results['ready_count'] ?? count($bulk_ready_fixes)) : 0;
$bulk_not_matched_count = $is_bulk_update ? ($results['not_matched_count'] ?? count($bulk_not_matched)) : 0;
$bulk_skipped_empty_count = $is_bulk_update ? ($results['skipped_empty_count'] ?? count($bulk_skipped_empty)) : 0;
$bulk_duplicates_overwritten = $is_bulk_update ? ($results['duplicates_overwritten'] ?? 0) : 0;

// Bulk complete results
$bulk_fixed_links = $bulk_complete ? ($results['fixed_links'] ?? []) : [];
$bulk_failed_fixes = $bulk_complete ? ($results['failed_fixes'] ?? []) : [];
$bulk_success_count = $bulk_complete ? ($results['success_count'] ?? 0) : 0;
$bulk_failed_count = $bulk_complete ? ($results['failed_count'] ?? 0) : 0;

// Check if we have valid bulk data to display
$has_valid_bulk_data = false;
if ($is_bulk_update) {
    if (!$bulk_complete && !empty($results['ready_fixes']) && count($results['ready_fixes']) > 0) {
        $has_valid_bulk_data = true;
    }
    // For bulk complete, require actual results
    $fixed_count_check = isset($results['fixed_links']) ? count($results['fixed_links']) : 0;
    $failed_count_check = isset($results['failed_fixes']) ? count($results['failed_fixes']) : 0;
    if ($bulk_complete && ($fixed_count_check > 0 || $failed_count_check > 0)) {
        $has_valid_bulk_data = true;
    }

    // If bulk update mode but no valid data, clear everything and reset
    if (!$has_valid_bulk_data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();
        $wpdb->delete($table_name, ['session_id' => $session_id, 'module' => 'broken-links']);
        delete_transient('sf_broken-links_results');
        $results = null;
        $is_bulk_update = false;
        $bulk_complete = false;
    }
}

// Check if we have any data (broken links OR fixed links - data persists after fixes)
$has_results = !empty($results) && is_array($results);
$has_broken_links = $has_results && !empty($results['broken_links']) && !$is_bulk_update;
$has_fixed_links = $has_results && !empty($results['fixed_links']) && !$is_bulk_update;
$has_data = $has_broken_links || $has_fixed_links || $has_valid_bulk_data;

// Get categorized links (full dataset)
$all_fixable_links = $has_results ? ($results['fixable_links'] ?? []) : [];
$all_manual_links = $has_results ? ($results['manual_links'] ?? []) : [];
$all_skipped_links = $has_results ? ($results['skipped_links'] ?? []) : [];
$all_fixed_links = $has_results ? ($results['fixed_links'] ?? []) : [];

// Flatten fixed links so each source page becomes its own row
$flattened_fixed_links = [];
foreach ($all_fixed_links as $link) {
    $sources = $link['fixable_sources'] ?? $link['sources'] ?? [];
    $broken_url = $link['broken_url'] ?? '';
    $new_url = $link['new_url'] ?? '';
    $fix_action = $link['fix_action'] ?? 'replace';
    $fixed_at = $link['fixed_at'] ?? '';

    $status = $link['status'] ?? 'success';
    $status_message = $link['status_message'] ?? '';

    foreach ($sources as $source) {
        $flattened_fixed_links[] = [
            'source_url' => $source['source_url'] ?? '',
            'source_title' => $source['post_title'] ?? '',
            'edit_url' => $source['edit_url'] ?? '',
            'post_id' => $source['post_id'] ?? 0,
            'broken_url' => $broken_url,
            'new_url' => $new_url,
            'fix_action' => $fix_action,
            'fixed_at' => $fixed_at,
            'status' => $status,
            'status_message' => $status_message,
        ];
    }
}

// Count fixed links by status
$fixed_success_count = 0;
$fixed_failed_count = 0;
$fixed_skipped_count = 0;
foreach ($flattened_fixed_links as $fl) {
    $s = $fl['status'] ?? 'success';
    if ($s === 'failed') { $fixed_failed_count++; }
    elseif ($s === 'skipped') { $fixed_skipped_count++; }
    else { $fixed_success_count++; }
}

// Get counts (totals)
$total_count = $has_results ? ($results['total_count'] ?? 0) : 0;
$fixable_count = $has_results ? ($results['fixable_count'] ?? count($all_fixable_links)) : 0;
$manual_count = $has_results ? ($results['manual_count'] ?? count($all_manual_links)) : 0;
$skipped_count = $has_results ? ($results['skipped_count'] ?? count($all_skipped_links)) : 0;
$fixed_count = count($flattened_fixed_links); // Count is now based on flattened (per-source) data

// Client-side pagination settings (100 items per page, handled by JavaScript)
$per_page = 100;

// Use full arrays for all sections - JavaScript handles pagination
$fixable_links = $all_fixable_links;
$manual_links = $all_manual_links;
$skipped_links = $all_skipped_links;
$fixed_links = $flattened_fixed_links; // Use flattened data for display

// Count internal vs external links and images for filter stats
$site_host = wp_parse_url(home_url(), PHP_URL_HOST);
$internal_count = 0;
$external_count = 0;
$image_count = 0;
$image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'ico', 'bmp', 'avif', 'tiff'];
foreach ($all_fixable_links as $link) {
    $broken_url_check = $link['broken_url'] ?? '';
    $link_host = wp_parse_url($broken_url_check, PHP_URL_HOST);
    if ($site_host === $link_host) {
        $internal_count++;
    } else {
        $external_count++;
    }
    // Check if URL is an image
    $path = wp_parse_url($broken_url_check, PHP_URL_PATH);
    $ext = strtolower(pathinfo($path ?? '', PATHINFO_EXTENSION));
    if (in_array($ext, $image_extensions, true)) {
        $image_count++;
    }
}

// Legacy: Keep helper function for compatibility but it's no longer used for main pagination
/**
 * Helper function to build pagination URL (legacy, kept for compatibility)
 */
function sf_build_pagination_url($base_url, $page, $per_page, $status, $type) {
    $args = [];
    if ($page > 1) {
        $args['paged'] = $page;
    }
    if ($per_page !== 100) {
        $args['per_page'] = $per_page;
    }
    if (!empty($status)) {
        $args['status'] = $status;
    }
    if (!empty($type)) {
        $args['type'] = $type;
    }
    return add_query_arg($args, $base_url);
}
?>

<div class="sf-broken-links-module">
    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#128279;</span>
            <h2><?php esc_html_e('Broken Links', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Fix 404s & update dead links from CSV uploads', 'screaming-fixes'); ?></span>
        </div>
        <?php if ($has_data): ?>
            <div class="sf-module-stats">
                <span class="sf-stat sf-stat-total">
                    <strong><?php echo esc_html($total_count); ?></strong>
                    <?php esc_html_e('broken links', 'screaming-fixes'); ?>
                </span>
            </div>
        <?php endif; ?>
    </div>

    <!-- Instructions Section (Collapsible) - Closed by default for consistency with other tools -->
    <div class="sf-instructions-section">
        <button type="button" class="sf-instructions-toggle" aria-expanded="false">
            <span class="dashicons dashicons-info"></span>
            <?php esc_html_e('How to get this report from Screaming Frog', 'screaming-fixes'); ?>
            <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
        </button>
        <div class="sf-instructions-content" style="display: none;">
            <?php include SF_PLUGIN_DIR . 'modules/broken-links/views/instructions.php'; ?>
        </div>
    </div>

    <?php if ($has_data): ?>
        <!-- Stats Breakdown -->
        <div class="sf-stats-breakdown">
            <div class="sf-stat-card sf-stat-fixable">
                <span class="sf-stat-icon">&#10003;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($fixable_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Fixable', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-stat-card sf-stat-manual">
                <span class="sf-stat-icon">&#9888;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($manual_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Manual Fix', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-stat-card sf-stat-skipped">
                <span class="sf-stat-icon">&#8856;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($skipped_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                </div>
            </div>
        </div>

    <?php endif; ?>

    <?php if ($is_bulk_update && !$bulk_complete && $has_valid_bulk_data): ?>
        <!-- ==========================================
             BULK FIX CONFIRMATION UI
             ========================================== -->
        <div class="sf-bulk-confirmation" id="sf-bulk-confirmation">
            <div class="sf-bulk-summary">
                <div class="sf-bulk-icon">&#128279;</div>
                <h3><?php esc_html_e('Bulk Broken Link Fixes', 'screaming-fixes'); ?></h3>

                <div class="sf-bulk-stats">
                    <div class="sf-bulk-stat sf-bulk-stat-ready">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_ready_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('broken link fixes ready to apply', 'screaming-fixes'); ?></span>
                    </div>

                    <?php if ($bulk_not_matched_count > 0): ?>
                    <div class="sf-bulk-stat sf-bulk-stat-warning">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_not_matched_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('source URLs not matched', 'screaming-fixes'); ?></span>
                    </div>
                    <?php endif; ?>

                    <?php if ($bulk_skipped_empty_count > 0): ?>
                    <div class="sf-bulk-stat sf-bulk-stat-skipped">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_skipped_empty_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('rows skipped - no fix URL provided', 'screaming-fixes'); ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <?php if ($bulk_duplicates_overwritten > 0): ?>
                <div class="sf-bulk-notice">
                    <span class="dashicons dashicons-info"></span>
                    <?php printf(
                        esc_html__('%d duplicate rows found - using last occurrence for each source+destination pair.', 'screaming-fixes'),
                        $bulk_duplicates_overwritten
                    ); ?>
                </div>
                <?php endif; ?>

                <?php if (!empty($bulk_url_warnings)): ?>
                <div class="sf-bulk-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <?php printf(
                        esc_html__('%d fix URLs have potential issues (shown in preview). Review before applying.', 'screaming-fixes'),
                        count($bulk_url_warnings)
                    ); ?>
                </div>
                <?php endif; ?>

                <?php if ($bulk_ready_count > 500): ?>
                <div class="sf-bulk-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <?php printf(
                        esc_html__('Large file detected (%d fixes). For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'),
                        $bulk_ready_count
                    ); ?>
                </div>
                <?php endif; ?>

                <div class="sf-bulk-preview-tip" style="margin-bottom: 14px; padding: 10px 14px; background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; font-size: 13px; line-height: 1.5; color: #334155;">
                    <span class="dashicons dashicons-info-outline" style="color: #22c55e; font-size: 16px; width: 16px; height: 16px; margin-right: 4px; vertical-align: text-bottom;"></span>
                    <?php printf(
                        esc_html__('Use %1$sDownload Preview CSV%2$s to review the full list of Ready (%3$d) and Can\'t Update (%4$d) broken links before confirming.', 'screaming-fixes'),
                        '<strong>', '</strong>',
                        $bulk_ready_count,
                        $bulk_not_matched_count
                    ); ?>
                </div>

                <div class="sf-bulk-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-download-preview" id="sf-bulk-download-preview">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
                    </button>
                    <div class="sf-bulk-buttons">
                        <button type="button" class="sf-button sf-button-secondary sf-bulk-clear-btn" id="sf-bulk-clear">
                            <span class="dashicons dashicons-trash"></span>
                            <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                        </button>
                        <button type="button" class="sf-button sf-button-primary sf-bulk-confirm" id="sf-bulk-confirm" data-count="<?php echo esc_attr($bulk_ready_count); ?>">
                            <?php esc_html_e('Confirm & Apply Fixes', 'screaming-fixes'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Preview Table -->
            <div class="sf-bulk-preview">
                <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, $bulk_ready_count)); ?> of <?php echo esc_html($bulk_ready_count); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Source Page', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Broken URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Fix URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            // Show first 10 ready fixes
                            $preview_items = array_slice($bulk_ready_fixes, 0, 10);
                            foreach ($preview_items as $index => $item):
                                $source_path = wp_parse_url($item['source_url'], PHP_URL_PATH) ?: $item['source_url'];
                                $broken_display = strlen($item['broken_url']) > 75 ? substr($item['broken_url'], 0, 72) . '...' : $item['broken_url'];
                                $fix_display = strlen($item['fix_url']) > 75 ? substr($item['fix_url'], 0, 72) . '...' : $item['fix_url'];
                                $has_long_content = strlen($item['source_url']) > 75 || strlen($item['broken_url']) > 75 || strlen($item['fix_url']) > 75;
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-ready<?php echo $has_long_content ? ' sf-desc-expandable' : ''; ?>" data-row-index="<?php echo esc_attr($index); ?>">
                                <td class="sf-col-source">
                                    <div class="sf-source-cell">
                                        <button type="button" class="sf-desc-toggle">
                                            <span class="sf-desc-text"><?php echo esc_html($source_path); ?></span>
                                            <?php if ($has_long_content): ?>
                                                <span class="sf-desc-expand-icon dashicons dashicons-arrow-down-alt2"></span>
                                            <?php endif; ?>
                                        </button>
                                        <?php if (!empty($item['post_title'])): ?>
                                        <span class="sf-source-title"><?php echo esc_html($item['post_title']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="sf-col-broken">
                                    <span class="sf-url-broken" title="<?php echo esc_attr($item['broken_url']); ?>"><?php echo esc_html($broken_display); ?></span>
                                </td>
                                <td class="sf-col-fix">
                                    <span class="sf-url-fix" title="<?php echo esc_attr($item['fix_url']); ?>"><?php echo esc_html($fix_display); ?></span>
                                </td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-ready"><?php esc_html_e('Ready', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php if ($has_long_content): ?>
                            <tr class="sf-bulk-detail-row" data-for-row="<?php echo esc_attr($index); ?>" style="display: none;">
                                <td colspan="4">
                                    <div class="sf-desc-detail-panel">
                                        <div class="sf-desc-detail-columns">
                                            <div class="sf-desc-detail-col sf-desc-detail-current">
                                                <div class="sf-desc-detail-label"><?php esc_html_e('Source Page', 'screaming-fixes'); ?></div>
                                                <div class="sf-desc-detail-value"><?php echo esc_html($item['source_url']); ?></div>
                                            </div>
                                            <div class="sf-desc-detail-col sf-desc-detail-current">
                                                <div class="sf-desc-detail-label"><?php esc_html_e('Broken URL', 'screaming-fixes'); ?></div>
                                                <div class="sf-desc-detail-value" style="text-decoration: line-through; color: #6b7280;"><?php echo esc_html($item['broken_url']); ?></div>
                                            </div>
                                            <div class="sf-desc-detail-arrow">&rarr;</div>
                                            <div class="sf-desc-detail-col sf-desc-detail-new">
                                                <div class="sf-desc-detail-label"><?php esc_html_e('Fix URL', 'screaming-fixes'); ?></div>
                                                <div class="sf-desc-detail-value"><?php echo esc_html($item['fix_url']); ?></div>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <?php endif; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <?php if ($bulk_not_matched_count > 0): ?>
                <h4 class="sf-mt-20"><?php esc_html_e('Can\'t Update / Skipped Broken Links', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html($bulk_not_matched_count); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Source URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Broken URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $unmatched_preview = array_slice($bulk_not_matched, 0, 10);
                            foreach ($unmatched_preview as $item):
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-skipped">
                                <td class="sf-col-source"><?php echo esc_html($item['source_url']); ?></td>
                                <td class="sf-col-broken"><?php echo esc_html($item['broken_url']); ?></td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-skipped"><?php esc_html_e('Can\'t update', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if ($bulk_not_matched_count > 10): ?>
                            <tr class="sf-bulk-row-more">
                                <td colspan="3"><?php printf(esc_html__('...and %d more skipped URLs', 'screaming-fixes'), $bulk_not_matched_count - 10); ?></td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="sf-bulk-unmatched-info" style="margin-top: 12px; padding: 12px 16px; background: #f0fdfa; border: 1px solid #99f6e4; border-radius: 6px; font-size: 13px; line-height: 1.5; color: #334155;">
                    <strong><?php esc_html_e('Why can\'t these broken links be updated?', 'screaming-fixes'); ?></strong>
                    <ul style="margin: 6px 0 0 18px; padding: 0;">
                        <li><?php esc_html_e('The broken link is located in a widget, site-wide navigation, header, footer, or other theme areas outside of post/page content.', 'screaming-fixes'); ?></li>
                        <li><?php esc_html_e('The source URL belongs to a category, tag, archive, author, or homepage — these are generated dynamically by WordPress.', 'screaming-fixes'); ?></li>
                        <li><?php esc_html_e('The source URL was not found on this WordPress site (e.g., it belongs to a different domain or the post has been deleted).', 'screaming-fixes'); ?></li>
                    </ul>
                    <p style="margin: 8px 0 0;"><?php esc_html_e('You will need to manually update these broken links by editing the widget, menu, or theme settings where they appear.', 'screaming-fixes'); ?></p>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Bulk Progress Modal (hidden by default) -->
        <div class="sf-bulk-progress-modal sf-modal" id="sf-bulk-progress-modal" style="display: none;">
            <div class="sf-modal-content sf-bulk-progress-content">
                <div class="sf-bulk-progress-header">
                    <span class="sf-bulk-progress-icon sf-spinning">&#8634;</span>
                    <h3><?php esc_html_e('Applying Broken Link Fixes...', 'screaming-fixes'); ?></h3>
                </div>
                <div class="sf-bulk-progress-bar">
                    <div class="sf-bulk-progress-fill" id="sf-bulk-progress-fill" style="width: 0%;"></div>
                </div>
                <div class="sf-bulk-progress-stats">
                    <span class="sf-bulk-progress-current" id="sf-bulk-progress-current">0</span>
                    <span class="sf-bulk-progress-separator">/</span>
                    <span class="sf-bulk-progress-total" id="sf-bulk-progress-total">0</span>
                    <span class="sf-bulk-progress-label"><?php esc_html_e('complete', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-bulk-progress-url" id="sf-bulk-progress-url">
                    <?php esc_html_e('Preparing...', 'screaming-fixes'); ?>
                </div>
            </div>
        </div>

    <?php elseif ($bulk_complete && $has_valid_bulk_data): ?>
        <!-- ==========================================
             BULK FIX COMPLETE RESULTS
             ========================================== -->
        <div class="sf-bulk-complete" id="sf-bulk-complete">
            <div class="sf-bulk-complete-summary <?php echo $bulk_failed_count > 0 ? 'sf-has-failures' : ''; ?>">
                <div class="sf-bulk-complete-icon">
                    <?php echo $bulk_failed_count > 0 ? '&#9888;' : '&#10004;'; ?>
                </div>
                <h3>
                    <?php
                    if ($bulk_failed_count > 0) {
                        printf(
                            esc_html__('%d broken links fixed successfully. %d failed - see details below.', 'screaming-fixes'),
                            $bulk_success_count,
                            $bulk_failed_count
                        );
                    } else {
                        printf(
                            esc_html__('%d broken links fixed successfully.', 'screaming-fixes'),
                            $bulk_success_count
                        );
                    }
                    ?>
                    <?php if ($bulk_not_matched_count > 0 || $bulk_skipped_empty_count > 0): ?>
                        <span class="sf-bulk-skipped-note">
                            <?php printf(esc_html__('%d rows skipped.', 'screaming-fixes'), $bulk_not_matched_count + $bulk_skipped_empty_count); ?>
                        </span>
                    <?php endif; ?>
                </h3>

                <div class="sf-bulk-complete-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-download-results" id="sf-bulk-download-results">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-primary sf-bulk-clear-btn" id="sf-bulk-new-upload">
                        <?php esc_html_e('Clear & Start New', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <!-- Results Table (first 10) -->
            <?php if (!empty($bulk_fixed_links)): ?>
            <div class="sf-bulk-results-table">
                <h4><?php esc_html_e('Fixed Links', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, count($bulk_fixed_links))); ?> of <?php echo esc_html(count($bulk_fixed_links)); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Source Page', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Broken URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Fix URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $results_preview = array_slice($bulk_fixed_links, 0, 10);
                            foreach ($results_preview as $item):
                                $source_path = wp_parse_url($item['source_url'], PHP_URL_PATH) ?: $item['source_url'];
                                $broken_display = strlen($item['broken_url']) > 40 ? substr($item['broken_url'], 0, 37) . '...' : $item['broken_url'];
                                $fix_display = strlen($item['fix_url']) > 40 ? substr($item['fix_url'], 0, 37) . '...' : $item['fix_url'];
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-success">
                                <td class="sf-col-source">
                                    <div class="sf-source-cell">
                                        <a href="<?php echo esc_url($item['source_url']); ?>" target="_blank">
                                            <?php echo esc_html($source_path); ?>
                                        </a>
                                        <?php if (!empty($item['post_title'])): ?>
                                        <span class="sf-source-title"><?php echo esc_html($item['post_title']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="sf-col-broken" title="<?php echo esc_attr($item['broken_url']); ?>"><?php echo esc_html($broken_display); ?></td>
                                <td class="sf-col-fix" title="<?php echo esc_attr($item['fix_url']); ?>"><?php echo esc_html($fix_display); ?></td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-success"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Failed Updates -->
            <?php if (!empty($bulk_failed_fixes)): ?>
            <div class="sf-bulk-failed-table">
                <h4 class="sf-mt-20"><?php esc_html_e('Failed Fixes', 'screaming-fixes'); ?></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Source URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Error', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($bulk_failed_fixes as $item): ?>
                            <tr class="sf-bulk-row sf-bulk-row-failed">
                                <td class="sf-col-source"><?php echo esc_html($item['source_url']); ?></td>
                                <td class="sf-col-error"><?php echo esc_html($item['error']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
        </div>

    <?php elseif (!$has_data): ?>
    <!-- Upload Section - Prominent when no data -->
    <div class="sf-upload-section">
        <h3>
            <span class="dashicons dashicons-upload"></span>
            <?php esc_html_e('Upload Broken Links CSV', 'screaming-fixes'); ?>
        </h3>
        <?php
        $args = [
            'module' => 'broken-links',
            'title' => __('Drop your Screaming Frog CSV file', 'screaming-fixes'),
            'description' => __('Drag & drop CSV here or click to browse', 'screaming-fixes'),
            'accepted_types' => '.csv',
            'max_rows' => 5000,
        ];
        include SF_PLUGIN_DIR . 'admin/views/partials/upload-box.php';
        ?>
    </div>
    <?php endif; ?>

    <?php if ($has_data): ?>

        <?php if (!empty($all_fixable_links)): ?>
        <!-- Filter Buttons Row -->
        <div class="sf-filter-stats-row">
            <span class="sf-filter-label"><?php esc_html_e('Filter By:', 'screaming-fixes'); ?></span>
            <div class="sf-filter-stats">
                <div class="sf-filter-stat sf-filter-all active" data-filter="all">
                    <span class="sf-filter-icon">&#128203;</span>
                    <span class="sf-filter-count"><?php echo esc_html($fixable_count); ?></span>
                    <span class="sf-filter-name"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-stat sf-filter-internal" data-filter="internal">
                    <span class="sf-filter-icon">&#127968;</span>
                    <span class="sf-filter-count"><?php echo esc_html($internal_count); ?></span>
                    <span class="sf-filter-name"><?php esc_html_e('Internal', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-stat sf-filter-external" data-filter="external">
                    <span class="sf-filter-icon">&#127760;</span>
                    <span class="sf-filter-count"><?php echo esc_html($external_count); ?></span>
                    <span class="sf-filter-name"><?php esc_html_e('External', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-stat sf-filter-images<?php echo $image_count === 0 ? ' sf-filter-disabled' : ''; ?>" data-filter="images">
                    <span class="sf-filter-icon">&#128247;</span>
                    <span class="sf-filter-count"><?php echo esc_html($image_count); ?></span>
                    <span class="sf-filter-name"><?php esc_html_e('Images', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-stat sf-filter-fixed<?php echo $fixed_count === 0 ? ' sf-filter-disabled' : ''; ?>" data-filter="fixed">
                    <span class="sf-filter-icon">&#9989;</span>
                    <span class="sf-filter-count sf-fixed-filter-count"><?php echo esc_html($fixed_count); ?></span>
                    <span class="sf-filter-name"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-stat sf-filter-failed sf-filter-disabled" data-filter="failed">
                    <span class="sf-filter-icon">&#10060;</span>
                    <span class="sf-filter-count sf-failed-filter-count">0</span>
                    <span class="sf-filter-name"><?php esc_html_e("Can't Fix", 'screaming-fixes'); ?></span>
                </div>
            </div>
            <button type="button" class="sf-button sf-button-link sf-clear-results-btn" title="<?php esc_attr_e('Clear results and start over', 'screaming-fixes'); ?>">
                <span class="dashicons dashicons-trash"></span>
                <?php esc_html_e('Clear Data', 'screaming-fixes'); ?>
            </button>
        </div>

        <!-- Section 1: Fixable Links -->
        <div class="sf-results-section sf-section-fixable">
            <div class="sf-results-header sf-results-header-collapsible">
                <button type="button" class="sf-section-toggle sf-fixable-toggle" aria-expanded="true">
                    <span class="sf-section-badge sf-badge-fixable">&#10003;</span>
                    <?php printf(
                        esc_html__('%d Fixable Links', 'screaming-fixes'),
                        $fixable_count
                    ); ?>
                    <span class="sf-section-hint"><?php esc_html_e('Links in post/page content - can be auto-fixed', 'screaming-fixes'); ?></span>
                    <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
                </button>
                <div class="sf-results-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-export-btn">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <div class="sf-fixable-content">
            <!-- Fixable Results Table with Pagination -->
            <div class="sf-table-wrapper sf-paginated-table" data-section="fixable" data-per-page="100" data-total="<?php echo esc_attr($fixable_count); ?>">
                <table class="sf-results-table sf-fixable-table" id="sf-broken-links-table">
                    <thead>
                        <tr>
                            <th class="sf-col-found-in"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                            <th class="sf-col-broken-url"><?php esc_html_e('Broken URL', 'screaming-fixes'); ?></th>
                            <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            <th class="sf-col-fix-action"><?php esc_html_e('Fix', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fixable_links as $index => $link):
                            $broken_url = $link['broken_url'] ?? '';
                            $status_code = $link['status_code'] ?? 0;
                            $sources = $link['fixable_sources'] ?? $link['sources'] ?? [];
                            $source_count = count($sources);

                            // Determine if internal or external
                            $site_host = wp_parse_url(home_url(), PHP_URL_HOST);
                            $link_host = wp_parse_url($broken_url, PHP_URL_HOST);
                            $is_internal = ($site_host === $link_host);

                            // Collect post IDs for fixable sources
                            $post_ids = array_filter(array_column($sources, 'post_id'));

                            // Detect if this is an image URL
                            $url_path = wp_parse_url($broken_url, PHP_URL_PATH);
                            $url_ext = strtolower(pathinfo($url_path ?? '', PATHINFO_EXTENSION));
                            $is_image = in_array($url_ext, $image_extensions, true);
                        ?>
                            <tr class="sf-link-row<?php echo $is_image ? ' sf-row-image' : ''; ?>"
                                data-broken-url="<?php echo esc_attr($broken_url); ?>"
                                data-index="<?php echo esc_attr($index); ?>"
                                data-category="fixable"
                                data-status="pending"
                                data-status-code="<?php echo esc_attr($status_code); ?>"
                                data-source-count="<?php echo esc_attr($source_count); ?>"
                                data-is-internal="<?php echo $is_internal ? 'true' : 'false'; ?>"
                                data-is-image="<?php echo $is_image ? 'true' : 'false'; ?>"
                                data-post-ids="<?php echo esc_attr(implode(',', $post_ids)); ?>">
                                <td class="sf-col-found-in">
                                    <?php echo sf_render_sources_column($sources, $broken_url, $link); ?>
                                </td>
                                <td class="sf-col-broken-url">
                                    <div class="sf-url-cell">
                                        <?php if ($is_image): ?>
                                            <span class="sf-image-badge" title="<?php esc_attr_e('Broken image', 'screaming-fixes'); ?>">&#128247;</span>
                                        <?php endif; ?>
                                        <a href="<?php echo esc_url($broken_url); ?>" target="_blank" rel="noopener" class="sf-broken-url" title="<?php echo esc_attr($broken_url); ?>">
                                            <?php
                                            // Truncate URL for display
                                            $display_url = strlen($broken_url) > 50 ? substr($broken_url, 0, 47) . '...' : $broken_url;
                                            echo esc_html($display_url);
                                            ?>
                                        </a>
                                    </div>
                                </td>
                                <td class="sf-col-status">
                                    <span class="sf-row-status sf-status-pending">
                                        <?php if (!empty($status_code)): ?>
                                            <span class="sf-status-code-badge sf-code-<?php echo esc_attr($status_code); ?>"><?php echo esc_html($status_code); ?></span>
                                        <?php endif; ?>
                                        <span class="sf-status-text"><?php esc_html_e('Pending', 'screaming-fixes'); ?></span>
                                    </span>
                                </td>
                                <td class="sf-col-fix-action">
                                    <button type="button" class="sf-edit-btn" data-broken-url="<?php echo esc_attr($broken_url); ?>">
                                        <span class="dashicons dashicons-edit"></span>
                                        <?php esc_html_e('Edit', 'screaming-fixes'); ?>
                                    </button>
                                    <div class="sf-fix-status" style="display: none;">
                                        <div class="sf-fix-status-success" style="display: none;">
                                            <span class="sf-fix-status-icon">&#9989;</span>
                                            <span class="sf-fix-status-text"><?php esc_html_e('Fixed successfully', 'screaming-fixes'); ?></span>
                                        </div>
                                        <div class="sf-fix-status-failed" style="display: none;">
                                            <span class="sf-fix-status-icon">&#10060;</span>
                                            <span class="sf-fix-status-text"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <!-- Expandable Action Panel (hidden by default) -->
                            <tr class="sf-action-panel-row" style="display: none;" data-for-url="<?php echo esc_attr($broken_url); ?>">
                                <td colspan="4">
                                    <div class="sf-action-panel">
                                        <!-- Error info box (shown when fix has failed) -->
                                        <div class="sf-panel-error-info" style="display: none;">
                                            <div class="sf-error-info-icon">&#9888;&#65039;</div>
                                            <div class="sf-error-info-content">
                                                <strong class="sf-error-info-reason"></strong>
                                                <span class="sf-error-info-suggestion"></span>
                                            </div>
                                        </div>

                                        <div class="sf-panel-info">
                                            &#128202; <?php esc_html_e('Affects', 'screaming-fixes'); ?> <strong><?php echo esc_html($source_count); ?></strong> <?php echo esc_html(_n('page', 'pages', $source_count, 'screaming-fixes')); ?>
                                            &middot; <?php esc_html_e('Status:', 'screaming-fixes'); ?> <span class="sf-status-badge sf-status-<?php echo esc_attr($status_code); ?>"><?php echo esc_html($status_code ?: '?'); ?></span>
                                            &middot; <span class="sf-type-badge"><?php echo $is_internal ? esc_html__('Internal', 'screaming-fixes') : esc_html__('External', 'screaming-fixes'); ?></span>
                                        </div>

                                        <div class="sf-panel-actions">
                                            <?php if ($is_image): ?>
                                            <button type="button" class="sf-action-btn sf-remove-btn sf-remove-disabled" data-action="remove_link" disabled title="<?php esc_attr_e('Remove broken images manually in the post editor for safety', 'screaming-fixes'); ?>">
                                                &#128465; <?php esc_html_e('Remove Image', 'screaming-fixes'); ?>
                                            </button>
                                            <?php else: ?>
                                            <button type="button" class="sf-action-btn sf-remove-btn" data-action="remove_link">
                                                &#128465; <?php esc_html_e('Remove Hyperlink', 'screaming-fixes'); ?>
                                            </button>
                                            <?php endif; ?>
                                            <button type="button" class="sf-action-btn sf-ignore-btn" data-action="ignore">
                                                &#9193; <?php esc_html_e('Ignore', 'screaming-fixes'); ?>
                                            </button>
                                            <button type="button" class="sf-action-btn sf-ai-btn" data-action="ai_suggest">
                                                &#129302; <?php esc_html_e('AI Suggest', 'screaming-fixes'); ?>
                                            </button>
                                            <button type="button" class="sf-close-btn sf-panel-close">
                                                &#10005; <?php esc_html_e('Close', 'screaming-fixes'); ?>
                                            </button>
                                        </div>

                                        <!-- AI Loading State -->
                                        <div class="sf-panel-ai-loading" style="display: none;">
                                            <div class="sf-ai-spinner"></div>
                                            <span class="sf-ai-loading-text"><?php esc_html_e('Finding best URL...', 'screaming-fixes'); ?></span>
                                        </div>

                                        <!-- AI Result (shown after AI suggestion) -->
                                        <div class="sf-panel-ai-result" style="display: none;">
                                            <span class="sf-ai-badge"><span class="sf-ai-badge-icon">&#129302;</span>AI</span>
                                            <a href="#" class="sf-ai-url" target="_blank" rel="noopener"></a>
                                            <button type="button" class="sf-use-ai-btn"><?php esc_html_e('Use This URL', 'screaming-fixes'); ?></button>
                                        </div>

                                        <div class="sf-panel-manual">
                                            <label><?php esc_html_e('Or enter replacement URL:', 'screaming-fixes'); ?></label>
                                            <input type="url" class="sf-url-input sf-replacement-url" placeholder="https://example.com/page" data-broken-url="<?php echo esc_attr($broken_url); ?>" />
                                            <button type="button" class="sf-save-btn" data-action="replace">
                                                &#10003; <?php esc_html_e('Save', 'screaming-fixes'); ?>
                                            </button>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination Controls (Bottom) -->
            <?php if ($fixable_count > 100): ?>
            <div class="sf-pagination" data-section="fixable">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                    <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $fixable_count)); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?>
                    <span class="sf-page-total"><?php echo esc_html($fixable_count); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" disabled title="<?php esc_attr_e('Previous page', 'screaming-fixes'); ?>">
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-btn sf-page-next" title="<?php esc_attr_e('Next page', 'screaming-fixes'); ?>">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
            </div>
            <?php endif; ?>
            </div><!-- /.sf-fixable-content -->

            <!-- Batch Actions - Outside collapsible content so it stays visible -->
            <div class="sf-batch-actions">
                <div class="sf-batch-info">
                    <span class="sf-fixes-ready">
                        <span class="sf-fixes-count">0</span> <?php esc_html_e('fixes ready', 'screaming-fixes'); ?>
                    </span>
                </div>
                <div class="sf-batch-buttons">
                    <button type="button" class="sf-button sf-button-primary sf-apply-all-fixes" disabled>
                        <span class="dashicons dashicons-yes"></span>
                        <?php esc_html_e('Apply All Fixes', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($all_manual_links)):
            // Calculate total manual sources for display
            $total_manual_sources = $has_data ? ($results['manual_sources'] ?? 0) : 0;
            if ($total_manual_sources === 0) {
                foreach ($all_manual_links as $link) {
                    $total_manual_sources += count($link['manual_sources'] ?? $link['sources'] ?? []);
                }
            }
        ?>
        <!-- Section 2: Manual Fix Required (Collapsed by default) -->
        <div class="sf-results-section sf-section-manual">
            <button type="button" class="sf-section-toggle sf-manual-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-manual">&#9888;</span>
                <?php printf(
                    esc_html__('%d Manual Fix Required', 'screaming-fixes'),
                    $manual_count
                ); ?>
                <?php if ($total_manual_sources > $manual_count): ?>
                    <span class="sf-section-hint"><?php printf(
                        esc_html__('Found on %d pages - fix in navigation, footer, widgets', 'screaming-fixes'),
                        $total_manual_sources
                    ); ?></span>
                <?php else: ?>
                    <span class="sf-section-hint"><?php esc_html_e('Links in navigation, footer, widgets - fix manually', 'screaming-fixes'); ?></span>
                <?php endif; ?>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-manual-content" style="display: none;">
            <!-- Manual Results Table with Pagination -->
            <div class="sf-table-wrapper sf-paginated-table" data-section="manual" data-per-page="100" data-total="<?php echo esc_attr($manual_count); ?>">
                <table class="sf-results-table sf-manual-table">
                    <thead>
                        <tr>
                            <th class="sf-col-url"><?php esc_html_e('Broken URL', 'screaming-fixes'); ?></th>
                            <th class="sf-col-sources"><?php esc_html_e('Found On', 'screaming-fixes'); ?></th>
                            <th class="sf-col-howto"><?php esc_html_e('How to Fix', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($manual_links as $index => $link):
                            $broken_url = $link['broken_url'] ?? '';
                            $status_code = $link['status_code'] ?? 0;
                            $sources = $link['manual_sources'] ?? $link['sources'] ?? [];

                            // Get the primary fix note (from first source)
                            $primary_note = '';
                            $primary_location = 'unknown';
                            if (!empty($sources[0])) {
                                $primary_note = $sources[0]['fix_note'] ?? '';
                                $primary_location = $sources[0]['location'] ?? 'unknown';
                            }

                            $location_icons = [
                                'navigation' => '&#9776;',
                                'footer' => '&#9604;',
                                'sidebar' => '&#9612;',
                                'aside' => '&#9612;',
                                'header' => '&#9600;',
                                'breadcrumb' => '&#10095;',
                                'homepage' => '&#8962;',
                                'unknown' => '&#63;',
                            ];
                            $icon = $location_icons[$primary_location] ?? '&#63;';
                        ?>
                            <tr class="sf-link-row sf-manual-row"
                                data-broken-url="<?php echo esc_attr($broken_url); ?>"
                                data-index="<?php echo esc_attr($index); ?>"
                                data-category="manual">
                                <td class="sf-col-url">
                                    <div class="sf-url-cell">
                                        <a href="<?php echo esc_url($broken_url); ?>" target="_blank" rel="noopener" class="sf-broken-url">
                                            <?php echo esc_html($broken_url); ?>
                                        </a>
                                        <?php if (!empty($status_code)): ?>
                                            <span class="sf-status-code sf-status-<?php echo esc_attr($status_code); ?>">
                                                <?php echo esc_html($status_code); ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="sf-col-sources">
                                    <?php echo sf_render_sources_column($sources, $broken_url); ?>
                                </td>
                                <td class="sf-col-howto">
                                    <div class="sf-fix-howto">
                                        <span class="sf-location-icon"><?php echo $icon; ?></span>
                                        <span class="sf-fix-note"><?php echo esc_html($primary_note); ?></span>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Pagination Controls for Manual -->
                <?php if ($manual_count > 100): ?>
                <div class="sf-pagination" data-section="manual">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                        <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $manual_count)); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?>
                        <span class="sf-page-total"><?php echo esc_html($manual_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-btn sf-page-prev" disabled title="<?php esc_attr_e('Previous page', 'screaming-fixes'); ?>">
                            <span class="dashicons dashicons-arrow-left-alt2"></span>
                        </button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-btn sf-page-next" title="<?php esc_attr_e('Next page', 'screaming-fixes'); ?>">
                            <span class="dashicons dashicons-arrow-right-alt2"></span>
                        </button>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            </div><!-- /.sf-manual-content -->
        </div>
        <?php endif; ?>

        <?php if (!empty($all_skipped_links)): ?>
        <!-- Section 3: Skipped (Collapsed by default) -->
        <div class="sf-results-section sf-section-skipped">
            <button type="button" class="sf-section-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-skipped">&#8856;</span>
                <?php printf(
                    esc_html__('%d Skipped (Dynamic Pages)', 'screaming-fixes'),
                    $skipped_count
                ); ?>
                <span class="sf-section-hint"><?php esc_html_e('Will be fixed when source content is fixed', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-skipped-content" style="display: none;">
                <div class="sf-skipped-note">
                    <span class="dashicons dashicons-info"></span>
                    <?php esc_html_e('These links appear on dynamic pages (pagination, archives, etc.) that pull content from posts. They will be automatically fixed when you fix the broken links in the source posts. In some cases, this could be pages we didn\'t find in your database or 410 status pages we can\'t update.', 'screaming-fixes'); ?>
                </div>
                <div class="sf-skipped-links-list sf-paginated-table" data-section="skipped" data-per-page="100" data-total="<?php echo esc_attr($skipped_count); ?>">
                    <?php foreach ($skipped_links as $index => $link):
                        $broken_url = $link['broken_url'] ?? '';
                        $sources = $link['skip_sources'] ?? $link['sources'] ?? [];
                    ?>
                        <div class="sf-skipped-link-item sf-skipped-row" data-index="<?php echo esc_attr($index); ?>">
                            <div class="sf-skipped-link-url">
                                <a href="<?php echo esc_url($broken_url); ?>" target="_blank" rel="noopener" class="sf-broken-url">
                                    <?php echo esc_html($broken_url); ?>
                                </a>
                            </div>
                            <div class="sf-skipped-sources">
                                <?php foreach ($sources as $source): ?>
                                    <span class="sf-skipped-source">
                                        <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" rel="noopener">
                                            <?php echo esc_html($source['fix_note'] ?? $source['source_url']); ?>
                                        </a>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>

                    <!-- Pagination Controls for Skipped -->
                    <?php if ($skipped_count > 100): ?>
                    <div class="sf-pagination" data-section="skipped">
                        <div class="sf-pagination-info">
                            <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                            <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $skipped_count)); ?></span>
                            <?php esc_html_e('of', 'screaming-fixes'); ?>
                            <span class="sf-page-total"><?php echo esc_html($skipped_count); ?></span>
                        </div>
                        <div class="sf-pagination-controls">
                            <button type="button" class="sf-page-btn sf-page-prev" disabled title="<?php esc_attr_e('Previous page', 'screaming-fixes'); ?>">
                                <span class="dashicons dashicons-arrow-left-alt2"></span>
                            </button>
                            <div class="sf-page-numbers"></div>
                            <button type="button" class="sf-page-btn sf-page-next" title="<?php esc_attr_e('Next page', 'screaming-fixes'); ?>">
                                <span class="dashicons dashicons-arrow-right-alt2"></span>
                            </button>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if ($fixed_count > 0): ?>
        <!-- Section 4: Fixes Applied -->
        <div class="sf-results-section sf-section-fixed" id="sf-fixed-results">
            <div class="sf-results-header sf-results-header-collapsible sf-results-header-fixed">
                <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="true">
                    <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
                    <span class="sf-fixes-applied-title">
                        <?php printf(
                            esc_html__('Fixes Applied (%d)', 'screaming-fixes'),
                            $fixed_count
                        ); ?>
                    </span>
                    <span class="sf-section-hint"><?php esc_html_e('View fix results and export to CSV', 'screaming-fixes'); ?></span>
                    <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
                </button>
                <div class="sf-results-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-export-fixed-btn">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <div class="sf-fixed-content">
                <div class="sf-status-filter-row sf-bl-status-filters">
                    <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                    <div class="sf-status-filters">
                        <button type="button" class="sf-status-filter active" data-status="all">
                            <span class="sf-status-icon">&#128203;</span>
                            <span class="sf-status-count sf-bl-status-count-all"><?php echo esc_html($fixed_count); ?></span>
                            <span class="sf-status-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                        </button>
                        <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                            <span class="sf-status-icon">&#9989;</span>
                            <span class="sf-status-count sf-bl-status-count-success"><?php echo esc_html($fixed_success_count); ?></span>
                            <span class="sf-status-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                        </button>
                        <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                            <span class="sf-status-icon">&#10060;</span>
                            <span class="sf-status-count sf-bl-status-count-failed"><?php echo esc_html($fixed_failed_count); ?></span>
                            <span class="sf-status-label"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                        </button>
                        <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                            <span class="sf-status-icon">&#9203;</span>
                            <span class="sf-status-count sf-bl-status-count-skipped"><?php echo esc_html($fixed_skipped_count); ?></span>
                            <span class="sf-status-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                        </button>
                    </div>
                </div>
                <div class="sf-table-wrapper sf-paginated-table" data-section="fixed" data-per-page="100" data-total="<?php echo esc_attr($fixed_count); ?>">
                    <table class="sf-results-table sf-fixed-table">
                        <thead>
                            <tr>
                                <th class="sf-col-source"><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                                <th class="sf-col-was"><?php esc_html_e('Was', 'screaming-fixes'); ?></th>
                                <th class="sf-col-now"><?php esc_html_e('Now', 'screaming-fixes'); ?></th>
                                <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($fixed_links as $index => $link):
                                $source_url = $link['source_url'] ?? '';
                                $source_title = $link['source_title'] ?? '';
                                $edit_url = $link['edit_url'] ?? '';
                                $broken_url = $link['broken_url'] ?? '';
                                $new_url = $link['new_url'] ?? '';
                                $fix_action = $link['fix_action'] ?? 'replace';
                                $is_removed = ($fix_action === 'remove' || $fix_action === 'remove_link' || $fix_action === 'remove_all');
                                $status = $link['status'] ?? 'success';
                                $status_message = $link['status_message'] ?? '';

                                if ($status === 'failed') {
                                    $row_class = 'sf-fixed-row-failed';
                                    $status_icon = '&#10060;';
                                    $status_label = __('Failed', 'screaming-fixes');
                                } elseif ($status === 'skipped') {
                                    $row_class = 'sf-fixed-row-skipped';
                                    $status_icon = '&#9203;';
                                    $status_label = __('Skipped', 'screaming-fixes');
                                } else {
                                    $row_class = 'sf-fixed-row-success';
                                    $status_icon = '&#9989;';
                                    $status_label = $is_removed ? __('Removed', 'screaming-fixes') : __('Replaced', 'screaming-fixes');
                                }
                            ?>
                                <tr class="sf-fixed-row <?php echo esc_attr($row_class); ?>" data-status="<?php echo esc_attr($status); ?>" data-index="<?php echo esc_attr($index); ?>">
                                    <td class="sf-col-source">
                                        <div class="sf-source-info">
                                            <?php if (!empty($edit_url)): ?>
                                                <a href="<?php echo esc_url($edit_url); ?>" target="_blank" rel="noopener" class="sf-source-title">
                                                    <?php echo esc_html($source_title ?: __('(Untitled)', 'screaming-fixes')); ?>
                                                </a>
                                            <?php else: ?>
                                                <span class="sf-source-title"><?php echo esc_html($source_title ?: __('(Untitled)', 'screaming-fixes')); ?></span>
                                            <?php endif; ?>
                                            <div class="sf-source-url">
                                                <a href="<?php echo esc_url($source_url); ?>" target="_blank" rel="noopener">
                                                    <?php echo esc_html(wp_parse_url($source_url, PHP_URL_PATH) ?: $source_url); ?>
                                                </a>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="sf-col-was">
                                        <a href="<?php echo esc_url($broken_url); ?>" target="_blank" rel="noopener" class="sf-broken-url-link" title="<?php echo esc_attr($broken_url); ?>">
                                            <?php echo esc_html($broken_url); ?>
                                        </a>
                                    </td>
                                    <td class="sf-col-now">
                                        <?php if ($is_removed): ?>
                                            <span class="sf-removed-text"><?php esc_html_e('Link removed', 'screaming-fixes'); ?></span>
                                        <?php else: ?>
                                            <a href="<?php echo esc_url($new_url); ?>" target="_blank" rel="noopener" class="sf-new-url-link" title="<?php echo esc_attr($new_url); ?>">
                                                <?php echo esc_html($new_url); ?>
                                            </a>
                                        <?php endif; ?>
                                    </td>
                                    <td class="sf-col-status">
                                        <span class="sf-status-badge sf-status-<?php echo esc_attr($status); ?>"<?php if (!empty($status_message) && $status !== 'success') echo ' title="' . esc_attr($status_message) . '"'; ?>>
                                            <span class="sf-status-icon"><?php echo $status_icon; ?></span>
                                            <?php echo esc_html($status_label); ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination Controls for Fixed -->
                <?php if ($fixed_count > 100): ?>
                <div class="sf-pagination" data-section="fixed">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                        <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $fixed_count)); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?>
                        <span class="sf-page-total"><?php echo esc_html($fixed_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-btn sf-page-prev" disabled title="<?php esc_attr_e('Previous page', 'screaming-fixes'); ?>">
                            <span class="dashicons dashicons-arrow-left-alt2"></span>
                        </button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-btn sf-page-next" title="<?php esc_attr_e('Next page', 'screaming-fixes'); ?>">
                            <span class="dashicons dashicons-arrow-right-alt2"></span>
                        </button>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Progress Modal -->
        <div class="sf-modal sf-progress-modal" style="display: none;">
            <div class="sf-modal-content">
                <div class="sf-progress-header">
                    <span class="sf-progress-icon dashicons dashicons-update sf-spinning"></span>
                    <h3 class="sf-progress-title"><?php esc_html_e('Applying Fixes...', 'screaming-fixes'); ?></h3>
                </div>
                <div class="sf-progress-bar">
                    <div class="sf-progress-fill" style="width: 0%;"></div>
                </div>
                <div class="sf-progress-status">
                    <span class="sf-progress-current">0</span> / <span class="sf-progress-total">0</span>
                    <?php esc_html_e('fixes applied', 'screaming-fixes'); ?>
                </div>
            </div>
        </div>

        <!-- Sources Modal (for "Show all X sources") -->
        <div class="sf-modal sf-sources-modal" style="display: none;">
            <div class="sf-modal-content sf-sources-modal-content">
                <div class="sf-modal-header">
                    <h3 class="sf-modal-title">
                        <span class="sf-sources-count">0</span>
                        <?php esc_html_e('Pages Containing This Broken Link', 'screaming-fixes'); ?>
                    </h3>
                    <button type="button" class="sf-modal-close" aria-label="<?php esc_attr_e('Close', 'screaming-fixes'); ?>">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                <div class="sf-modal-body">
                    <p class="sf-broken-url-display"></p>
                    <div class="sf-sources-full-list">
                        <ul class="sf-sources-full-ul"></ul>
                    </div>
                </div>
                <div class="sf-modal-footer">
                    <button type="button" class="sf-button sf-button-secondary sf-copy-sources">
                        <span class="dashicons dashicons-clipboard"></span>
                        <?php esc_html_e('Copy All URLs', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-secondary sf-export-sources">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>
        </div>

    <?php else: ?>
        <!-- Empty State -->
        <div class="sf-empty-state">
            <div class="sf-empty-icon">&#128269;</div>
            <h3><?php esc_html_e('No broken links data yet', 'screaming-fixes'); ?></h3>
            <p><?php esc_html_e('Upload a Screaming Frog CSV to find and fix broken links on your site.', 'screaming-fixes'); ?></p>
        </div>
    <?php endif; ?>
</div>
