/**
 * Screaming Fixes - Image Alt Text Module
 *
 * Handles UI interactions, AI suggestions, and applying alt text fixes
 */

(function($) {
    'use strict';

    // Namespace
    window.ScreamingFixes = window.ScreamingFixes || {};

    /**
     * Image Alt Text Handler
     */
    ScreamingFixes.ImageAltText = {
        // Store current data
        data: null,
        selectedRows: [],
        bulkProcessing: false,

        // Pagination state
        pagination: {
            fixable: { currentPage: 1, perPage: 100, totalItems: 0 },
            manual: { currentPage: 1, perPage: 100, totalItems: 0 },
            skipped: { currentPage: 1, perPage: 100, totalItems: 0 },
            fixed: { currentPage: 1, perPage: 100, totalItems: 0 }
        },

        /**
         * Initialize the module
         */
        init: function() {
            this.bindEvents();
            this.loadExistingData();
            this.initPagination();
        },

        /**
         * Bind all event handlers
         */
        bindEvents: function() {
            var self = this;

            // Select all checkbox
            $(document).on('change', '#sf-select-all', function() {
                var checked = $(this).prop('checked');
                $('.sf-row-select').prop('checked', checked);
                self.updateSelectedCount();
            });

            // Individual row selection
            $(document).on('change', '.sf-row-select', function() {
                self.updateSelectedCount();
            });

            // Alt text input change - enable save button when text entered
            $(document).on('input', '.sf-alt-text-input', function() {
                var row = $(this).closest('.sf-image-row');
                var saveBtn = row.find('.sf-save-alt-btn');
                var altText = $(this).val().trim();

                // Enable/disable the row's save button based on input
                saveBtn.prop('disabled', !altText);

                self.updateApplyButtonState();
            });

            // Ignore button toggle
            $(document).on('click', '.sf-ignore-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-image-row');
                var input = row.find('.sf-alt-text-input');
                var saveBtn = row.find('.sf-save-alt-btn');
                var aiBtn = row.find('.sf-ai-btn');

                // Toggle the active state
                btn.toggleClass('sf-active');
                var isIgnored = btn.hasClass('sf-active');

                if (isIgnored) {
                    input.prop('disabled', true).addClass('sf-disabled');
                    saveBtn.prop('disabled', true);
                    aiBtn.prop('disabled', true).addClass('sf-disabled');
                    row.addClass('sf-row-ignored');
                    // Update button text
                    btn.find('.dashicons').removeClass('dashicons-hidden').addClass('dashicons-visibility');
                    btn.contents().filter(function() {
                        return this.nodeType === 3;
                    }).last()[0].textContent = ' Ignored';
                } else {
                    input.prop('disabled', false).removeClass('sf-disabled');
                    aiBtn.prop('disabled', false).removeClass('sf-disabled');
                    row.removeClass('sf-row-ignored');
                    // Re-enable save if there's text
                    var altText = input.val().trim();
                    saveBtn.prop('disabled', !altText);
                    // Update button text back
                    btn.find('.dashicons').removeClass('dashicons-visibility').addClass('dashicons-hidden');
                    btn.contents().filter(function() {
                        return this.nodeType === 3;
                    }).last()[0].textContent = ' Ignore';
                }

                self.updateApplyButtonState();
            });

            // Individual row Save button
            $(document).on('click', '.sf-save-alt-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-image-row');
                self.saveRowAltText(row, btn);
            });

            // AI suggest single image
            $(document).on('click', '.sf-ai-suggest-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-image-row');
                self.getAISuggestion(row, btn);
            });

            // AI suggest all
            $(document).on('click', '.sf-ai-suggest-all', function(e) {
                e.preventDefault();
                self.getAISuggestionsForAll($(this));
            });

            // Apply fixes
            $(document).on('click', '.sf-apply-fixes', function(e) {
                e.preventDefault();
                self.applyFixes();
            });

            // Export CSV
            $(document).on('click', '.sf-export-btn', function(e) {
                e.preventDefault();
                self.exportCSV();
            });

            // Toggle sources list
            $(document).on('click', '.sf-sources-toggle', function(e) {
                e.preventDefault();
                var wrapper = $(this).closest('.sf-sources-wrapper');
                var list = wrapper.find('.sf-sources-list');
                var icon = $(this).find('.dashicons');

                list.slideToggle(200);
                icon.toggleClass('dashicons-arrow-down-alt2 dashicons-arrow-up-alt2');
            });

            // Close modal
            $(document).on('click', '.sf-modal-close, .sf-modal', function(e) {
                if (e.target === this) {
                    $('.sf-modal').hide();
                }
            });

            // Copy sources
            $(document).on('click', '.sf-copy-sources', function(e) {
                e.preventDefault();
                self.copySources();
            });

            // Instructions toggle (collapsible box style)
            $(document).on('click', '.sf-instructions-header', function() {
                var $box = $(this).closest('.sf-instructions-box');
                var $content = $box.find('.sf-instructions-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Section toggle handlers (Skipped, Manual, Fixable, Fixed)
            $(document).on('click', '.sf-section-toggle', function() {
                var $toggle = $(this);
                var $section = $toggle.closest('.sf-results-section');
                var icon = $toggle.find('.sf-toggle-icon');
                var expanded = $toggle.attr('aria-expanded') === 'true';

                // Find the content div based on section type
                var content;
                if ($section.hasClass('sf-section-skipped')) {
                    content = $section.find('.sf-skipped-content');
                } else if ($section.hasClass('sf-section-manual')) {
                    content = $section.find('.sf-manual-content');
                } else if ($section.hasClass('sf-section-fixable')) {
                    content = $section.find('.sf-fixable-content');
                } else if ($section.hasClass('sf-section-fixed')) {
                    content = $section.find('.sf-fixed-content');
                }

                if (content && content.length) {
                    content.slideToggle(200);
                    $toggle.attr('aria-expanded', !expanded);
                    icon.toggleClass('sf-rotated');
                }
            });

            // Listen for CSV upload events
            $(document).on('sf:csv-uploaded', function(e, data, box) {
                if (box.data('module') === 'image-alt-text') {
                    self.processUploadedCSV(data);
                }
            });

            // Clear data button
            $(document).on('click', '.sf-image-alt-text-module .sf-clear-data-btn', function(e) {
                e.preventDefault();
                self.clearData();
            });

            // Bulk update: Confirm
            $(document).on('click', '.sf-image-alt-text-module .sf-bulk-confirm', function(e) {
                e.preventDefault();
                self.processBulkUpdate();
            });

            // Bulk update: Clear
            $(document).on('click', '.sf-image-alt-text-module .sf-bulk-clear-btn', function(e) {
                e.preventDefault();
                self.clearBulkData();
            });

            // Bulk update: Download preview
            $(document).on('click', '.sf-image-alt-text-module .sf-bulk-download-preview', function(e) {
                e.preventDefault();
                self.downloadBulkPreview();
            });

            // Bulk update: Download results
            $(document).on('click', '.sf-image-alt-text-module .sf-bulk-download-results', function(e) {
                e.preventDefault();
                self.downloadBulkResults();
            });

            // Bulk update: New upload
            $(document).on('click', '.sf-image-alt-text-module .sf-bulk-new-upload', function(e) {
                e.preventDefault();
                self.startNewUpload();
            });

            // Bulk update: Clear complete
            $(document).on('click', '.sf-image-alt-text-module .sf-bulk-complete-clear', function(e) {
                e.preventDefault();
                self.clearBulkCompleteData();
            });

            // Pagination: Previous button
            $(document).on('click', '.sf-image-alt-text-module .sf-page-prev:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage - 1);
            });

            // Pagination: Next button
            $(document).on('click', '.sf-image-alt-text-module .sf-page-next:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage + 1);
            });

            // Pagination: Page number buttons
            $(document).on('click', '.sf-image-alt-text-module .sf-page-number:not(.active):not(.sf-page-ellipsis)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                var page = parseInt($(this).data('page'), 10);
                self.goToPage(section, page);
            });

            // Filter button clicks
            $(document).on('click', '.sf-image-alt-text-module .sf-issue-stat:not(.sf-filter-disabled)', function() {
                var filter = $(this).data('filter');
                self.filterRows(filter);

                // Update active state
                $('.sf-image-alt-text-module .sf-issue-stat').removeClass('active');
                $(this).addClass('active');

                // Show/hide Export Results button based on filter
                var $exportResultsBtn = $('.sf-image-alt-text-module .sf-export-results-btn');
                if (filter === 'fixed' || filter === 'failed') {
                    $exportResultsBtn.show();
                } else {
                    $exportResultsBtn.hide();
                }
            });

            // Export fixed images CSV
            $(document).on('click', '.sf-image-alt-text-module .sf-export-fixed-btn', function(e) {
                e.preventDefault();
                self.exportFixedCSV();
            });

            // Export results (fixed + can't fix) CSV
            $(document).on('click', '.sf-image-alt-text-module .sf-export-results-btn', function(e) {
                e.preventDefault();
                self.exportResultsCSV();
            });
        },

        /**
         * Load existing data if available
         */
        loadExistingData: function() {
            var self = this;

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_get_data',
                    nonce: sfImageAltTextData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.data) {
                        self.data = response.data.data;
                    }
                }
            });
        },

        /**
         * Process uploaded CSV
         */
        processUploadedCSV: function(uploadData) {
            var self = this;

            // Show processing state
            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.info(sfImageAltTextData.i18n.processing);
            }

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_process_csv',
                    nonce: sfImageAltTextData.nonce,
                    upload_id: uploadData.upload_id
                },
                success: function(response) {
                    if (response.success) {
                        self.data = response.data.data;

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message);
                        }

                        // Reload the page to show results
                        setTimeout(function() {
                            window.location.reload();
                        }, 1000);
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfImageAltTextData.i18n.processingFailed || 'Processing failed.');
                    }
                }
            });
        },

        /**
         * Update selected count display
         */
        updateSelectedCount: function() {
            var selected = $('.sf-row-select:checked').length;
            $('.sf-selected-count .sf-count').text(selected);
            this.updateApplyButtonState();
        },

        /**
         * Update apply button state
         */
        updateApplyButtonState: function() {
            var hasValidFixes = false;

            $('.sf-image-row[data-category="fixable"]').each(function() {
                var row = $(this);
                var altText = row.find('.sf-alt-text-input').val().trim();
                var ignored = row.find('.sf-ignore-btn').hasClass('sf-active');

                if (altText && !ignored) {
                    hasValidFixes = true;
                    return false; // break
                }
            });

            $('.sf-apply-fixes').prop('disabled', !hasValidFixes);
        },

        /**
         * Get AI suggestion for a single image
         */
        getAISuggestion: function(row, btn) {
            var self = this;
            var imageUrl = row.data('image-url');
            var filename = row.data('filename');
            var sourceTitles = (row.data('source-titles') || '').split('|').filter(Boolean);

            btn.addClass('sf-loading');
            btn.find('.dashicons').removeClass('dashicons-welcome-learn-more').addClass('dashicons-update sf-spinning');

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_get_ai_suggestion',
                    nonce: sfImageAltTextData.nonce,
                    image_url: imageUrl,
                    filename: filename,
                    source_titles: sourceTitles
                },
                success: function(response) {
                    btn.removeClass('sf-loading');
                    btn.find('.dashicons').removeClass('dashicons-update sf-spinning').addClass('dashicons-welcome-learn-more');

                    if (response.success) {
                        var input = row.find('.sf-alt-text-input');
                        input.val(response.data.suggestion);
                        input.addClass('sf-ai-suggested');

                        // Flash effect
                        input.addClass('sf-highlight');
                        setTimeout(function() {
                            input.removeClass('sf-highlight');
                        }, 1000);

                        self.updateApplyButtonState();

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success('AI suggestion added');
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    }
                },
                error: function() {
                    btn.removeClass('sf-loading');
                    btn.find('.dashicons').removeClass('dashicons-update sf-spinning').addClass('dashicons-welcome-learn-more');

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfImageAltTextData.i18n.aiFailed);
                    }
                }
            });
        },

        /**
         * Get AI suggestions for all images
         */
        getAISuggestionsForAll: function(btn) {
            var self = this;
            var rows = $('.sf-image-row[data-category="fixable"]').filter(function() {
                return !$(this).find('.sf-alt-text-input').val().trim();
            });

            if (rows.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.info('All images already have alt text entered.');
                }
                return;
            }

            btn.addClass('sf-loading');
            btn.prop('disabled', true);
            btn.find('.dashicons').removeClass('dashicons-welcome-learn-more').addClass('dashicons-update sf-spinning');

            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.info(sfImageAltTextData.i18n.aiSuggesting);
            }

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_ai_suggest_all',
                    nonce: sfImageAltTextData.nonce
                },
                success: function(response) {
                    btn.removeClass('sf-loading');
                    btn.prop('disabled', false);
                    btn.find('.dashicons').removeClass('dashicons-update sf-spinning').addClass('dashicons-welcome-learn-more');

                    if (response.success) {
                        var suggestions = response.data.suggestions;

                        // Apply suggestions to rows
                        $.each(suggestions, function(imageUrl, suggestion) {
                            var row = $('.sf-image-row[data-image-url="' + imageUrl + '"]');
                            if (row.length) {
                                var input = row.find('.sf-alt-text-input');
                                if (!input.val().trim()) {
                                    input.val(suggestion);
                                    input.addClass('sf-ai-suggested');
                                }
                            }
                        });

                        self.updateApplyButtonState();

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message);
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    }
                },
                error: function() {
                    btn.removeClass('sf-loading');
                    btn.prop('disabled', false);
                    btn.find('.dashicons').removeClass('dashicons-update sf-spinning').addClass('dashicons-welcome-learn-more');

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfImageAltTextData.i18n.aiFailed);
                    }
                }
            });
        },

        /**
         * Apply alt text fixes
         */
        applyFixes: function() {
            var self = this;
            var fixes = [];

            $('.sf-image-row[data-category="fixable"]').each(function() {
                var row = $(this);
                var imageUrl = row.data('image-url');
                var altText = row.find('.sf-alt-text-input').val().trim();
                var ignored = row.find('.sf-ignore-btn').hasClass('sf-active');
                var postIds = (row.data('post-ids') || '').toString().split(',').filter(Boolean);

                if (ignored) {
                    fixes.push({
                        image_url: imageUrl,
                        action: 'ignore',
                        new_alt: '',
                        post_ids: postIds
                    });
                } else if (altText) {
                    fixes.push({
                        image_url: imageUrl,
                        action: 'set_alt',
                        new_alt: altText,
                        post_ids: postIds
                    });
                }
            });

            if (fixes.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfImageAltTextData.i18n.noFixesSelected);
                }
                return;
            }

            // Show progress modal
            var modal = $('.sf-progress-modal');
            modal.find('.sf-progress-title').text(sfImageAltTextData.i18n.applyingFixes);
            modal.find('.sf-progress-total').text(fixes.length);
            modal.find('.sf-progress-current').text('0');
            modal.find('.sf-progress-fill').css('width', '0%');
            modal.show();

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_apply_fixes',
                    nonce: sfImageAltTextData.nonce,
                    fixes: fixes
                },
                success: function(response) {
                    modal.find('.sf-progress-fill').css('width', '100%');

                    setTimeout(function() {
                        modal.hide();

                        if (response.success) {
                            // Get counts from the results object
                            var resultsData = response.data.results || {};
                            var successCount = resultsData.success || 0;
                            var failedCount = resultsData.failed || 0;

                            // Get per-image results for marking rows
                            var imageResults = response.data.image_results || [];

                            // Mark rows based on per-image results
                            imageResults.forEach(function(result) {
                                var row = $('.sf-image-row[data-image-url="' + result.image_url + '"]');
                                var statusIcon = row.find('.sf-fix-status .sf-status-icon');

                                if (result.success) {
                                    // Mark as fixed
                                    row.addClass('sf-row-fixed').attr('data-status', 'fixed');
                                    row.find('.sf-alt-text-input').prop('disabled', true);
                                    row.find('.sf-ai-btn').prop('disabled', true).addClass('sf-disabled');
                                    row.find('.sf-ignore-btn').prop('disabled', true).addClass('sf-disabled');
                                    row.find('.sf-save-alt-btn').addClass('sf-saved').prop('disabled', true).html('<span class="dashicons dashicons-yes"></span> Saved');
                                    statusIcon.html('&#9989;').addClass('sf-status-fixed');
                                } else {
                                    // Mark as failed - keep alt text visible
                                    row.addClass('sf-row-failed').attr('data-status', 'failed');
                                    statusIcon.html('&#10060;').addClass('sf-status-failed');

                                    // Show error message
                                    var errorMsg = result.error || "Can't update - may be in theme/widget";
                                    var existingError = row.find('.sf-error-message');
                                    if (existingError.length) {
                                        existingError.text(errorMsg);
                                    } else {
                                        row.find('.sf-alt-text-controls').append(
                                            '<span class="sf-error-message">' + errorMsg + '</span>'
                                        );
                                    }
                                }
                            });

                            // Update filter counts
                            self.updateFilterCounts();

                            // Show toast
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                if (failedCount > 0) {
                                    ScreamingFixes.Toast.warning(successCount + ' fixed, ' + failedCount + " couldn't be fixed");
                                } else {
                                    ScreamingFixes.Toast.success(response.data.message);
                                }
                            }
                        } else {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error(response.data.message);
                            }
                        }
                    }, 500);
                },
                error: function() {
                    modal.hide();
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfImageAltTextData.i18n.fixesFailed);
                    }
                }
            });
        },

        /**
         * Save alt text for a single row
         */
        saveRowAltText: function(row, btn) {
            var self = this;
            var imageUrl = row.data('image-url');
            var altText = row.find('.sf-alt-text-input').val().trim();
            var postIds = (row.data('post-ids') || '').toString().split(',').filter(Boolean);

            if (!altText) {
                return;
            }

            // Show saving state
            btn.addClass('sf-saving').prop('disabled', true);

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_apply_fixes',
                    nonce: sfImageAltTextData.nonce,
                    fixes: [{
                        image_url: imageUrl,
                        action: 'set_alt',
                        new_alt: altText,
                        post_ids: postIds
                    }]
                },
                success: function(response) {
                    btn.removeClass('sf-saving');

                    if (response.success) {
                        var imageResults = response.data.image_results || [];
                        var result = imageResults[0] || {};

                        if (result.success) {
                            // Mark as fixed
                            row.addClass('sf-row-fixed').attr('data-status', 'fixed');
                            row.find('.sf-alt-text-input').prop('disabled', true);
                            row.find('.sf-ai-btn').prop('disabled', true).addClass('sf-disabled');
                            row.find('.sf-ignore-btn').prop('disabled', true).addClass('sf-disabled');
                            btn.addClass('sf-saved').html('<span class="dashicons dashicons-yes"></span> Saved');

                            var statusIcon = row.find('.sf-fix-status .sf-status-icon');
                            statusIcon.html('&#9989;').addClass('sf-status-fixed');

                            self.updateFilterCounts();

                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.success('Alt text saved successfully');
                            }
                        } else {
                            // Mark as failed
                            row.addClass('sf-row-failed').attr('data-status', 'failed');
                            btn.prop('disabled', false);

                            var statusIcon = row.find('.sf-fix-status .sf-status-icon');
                            statusIcon.html('&#10060;').addClass('sf-status-failed');

                            var errorMsg = result.error || "Can't update - may be in theme/widget";
                            var existingError = row.find('.sf-error-message');
                            if (existingError.length) {
                                existingError.text(errorMsg);
                            } else {
                                row.find('.sf-alt-text-controls').append(
                                    '<span class="sf-error-message">' + errorMsg + '</span>'
                                );
                            }

                            self.updateFilterCounts();

                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error(errorMsg);
                            }
                        }
                    } else {
                        btn.prop('disabled', false);
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Save failed');
                        }
                    }
                },
                error: function() {
                    btn.removeClass('sf-saving').prop('disabled', false);
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Save failed. Please try again.');
                    }
                }
            });
        },

        /**
         * Update counts after fixes applied
         */
        updateCounts: function() {
            var fixableCount = $('.sf-image-row[data-category="fixable"]:not(.sf-row-fixed):not(.sf-row-failed)').length;
            var totalCount = $('.sf-image-row[data-category="fixable"]').length;

            // Update stats
            $('.sf-stat-fixable .sf-stat-number').text(fixableCount);
            $('.sf-stat-total strong').text(totalCount);

            // Update filter counts
            this.updateFilterCounts();
        },

        /**
         * Update filter button counts
         */
        updateFilterCounts: function() {
            var allCount = $('.sf-section-fixable .sf-image-row[data-category="fixable"]').length;
            var pendingCount = $('.sf-section-fixable .sf-image-row[data-category="fixable"]:not(.sf-row-fixed):not(.sf-row-failed)').length;
            var fixedCount = $('.sf-section-fixable .sf-image-row.sf-row-fixed').length;
            var failedCount = $('.sf-section-fixable .sf-image-row.sf-row-failed').length;

            // Update counts in filter buttons
            $('.sf-image-alt-text-module .sf-count-all').text(allCount);
            $('.sf-image-alt-text-module .sf-count-pending').text(pendingCount);
            $('.sf-image-alt-text-module .sf-count-fixed').text(fixedCount);
            $('.sf-image-alt-text-module .sf-count-failed').text(failedCount);

            // Enable/disable filter buttons based on counts
            var $fixedBtn = $('.sf-image-alt-text-module .sf-issue-fixed');
            var $failedBtn = $('.sf-image-alt-text-module .sf-issue-failed');

            if (fixedCount > 0) {
                $fixedBtn.removeClass('sf-filter-disabled');
            } else {
                $fixedBtn.addClass('sf-filter-disabled');
            }

            if (failedCount > 0) {
                $failedBtn.removeClass('sf-filter-disabled');
            } else {
                $failedBtn.addClass('sf-filter-disabled');
            }

            // Update section header count
            $('.sf-section-fixable .sf-fixable-toggle').contents().filter(function() {
                return this.nodeType === 3 && this.textContent.match(/\d+ Fixable/);
            }).each(function() {
                this.textContent = this.textContent.replace(/\d+ Fixable/, allCount + ' Fixable');
            });
        },

        /**
         * Filter rows based on status
         */
        filterRows: function(filter) {
            var self = this;
            var $rows = $('.sf-section-fixable .sf-image-row[data-category="fixable"]');

            // Use sf-filtered-out class for pagination compatibility
            switch(filter) {
                case 'all':
                    $rows.removeClass('sf-filtered-out');
                    break;
                case 'pending':
                    $rows.addClass('sf-filtered-out');
                    $rows.filter(':not(.sf-row-fixed):not(.sf-row-failed)').removeClass('sf-filtered-out');
                    break;
                case 'fixed':
                    $rows.addClass('sf-filtered-out');
                    $rows.filter('.sf-row-fixed').removeClass('sf-filtered-out');
                    break;
                case 'failed':
                    $rows.addClass('sf-filtered-out');
                    $rows.filter('.sf-row-failed').removeClass('sf-filtered-out');
                    break;
            }

            // Also hide action panel rows for filtered-out rows
            $rows.each(function() {
                var $row = $(this);
                var imageUrl = $row.data('image-url');
                var $actionRow = $('.sf-action-panel-row[data-for-url="' + imageUrl + '"]');
                if ($row.hasClass('sf-filtered-out')) {
                    $actionRow.hide();
                }
            });

            // Reset pagination after filtering
            self.resetPagination('fixable');
        },

        /**
         * Export fixed images as CSV (from the Fixed Images section)
         */
        exportFixedCSV: function() {
            var self = this;
            var fixedImages = [];

            // Get fixed images from the Fixed Images section table
            $('.sf-section-fixed .sf-fixed-table .sf-fixed-row').each(function() {
                var $row = $(this);
                var imageUrl = $row.data('image-url');
                var appliedAlt = $row.data('applied-alt') || $row.find('.sf-applied-alt-text').text().trim();
                var filename = $row.find('.sf-image-filename').text().trim();

                // Get sources/found in info
                var foundIn = [];
                $row.find('.sf-source-title, .sf-source-url-link, .sf-source-path-link').each(function() {
                    var text = $(this).text().trim();
                    if (text && foundIn.indexOf(text) === -1) {
                        foundIn.push(text);
                    }
                });

                fixedImages.push({
                    image_url: imageUrl,
                    filename: filename,
                    alt_text: appliedAlt,
                    found_in: foundIn.join('; ')
                });
            });

            if (fixedImages.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning('No fixed images to export');
                }
                return;
            }

            // Build CSV content with columns: Image URL, Filename, Found In, New Alt Text
            var csv = 'Image URL,Filename,Found In,New Alt Text\n';
            fixedImages.forEach(function(img) {
                csv += '"' + (img.image_url || '').replace(/"/g, '""') + '",';
                csv += '"' + (img.filename || '').replace(/"/g, '""') + '",';
                csv += '"' + (img.found_in || '').replace(/"/g, '""') + '",';
                csv += '"' + (img.alt_text || '').replace(/"/g, '""') + '"\n';
            });

            // Download
            var blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
            var link = document.createElement('a');
            var url = URL.createObjectURL(blob);
            link.setAttribute('href', url);
            link.setAttribute('download', 'fixed-images-' + new Date().toISOString().slice(0, 10) + '.csv');
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);

            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.success('Exported ' + fixedImages.length + ' fixed images');
            }
        },

        /**
         * Export combined results (fixed + can't fix) via AJAX
         */
        exportResultsCSV: function() {
            var self = this;

            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.info('Exporting results...');
            }

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_export_results',
                    nonce: sfImageAltTextData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Download the CSV
                        var blob = new Blob([response.data.csv], { type: 'text/csv;charset=utf-8;' });
                        var link = document.createElement('a');
                        var url = URL.createObjectURL(blob);
                        link.setAttribute('href', url);
                        link.setAttribute('download', response.data.filename);
                        link.style.visibility = 'hidden';
                        document.body.appendChild(link);
                        link.click();
                        document.body.removeChild(link);

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success('Results exported successfully');
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Export failed');
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Export failed. Please try again.');
                    }
                }
            });
        },

        /**
         * Export results as CSV
         */
        exportCSV: function() {
            var self = this;

            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.info(sfImageAltTextData.i18n.exporting);
            }

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_export_results',
                    nonce: sfImageAltTextData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Create and trigger download
                        var blob = new Blob([response.data.csv], { type: 'text/csv;charset=utf-8;' });
                        var link = document.createElement('a');
                        var url = URL.createObjectURL(blob);
                        link.setAttribute('href', url);
                        link.setAttribute('download', response.data.filename);
                        link.style.visibility = 'hidden';
                        document.body.appendChild(link);
                        link.click();
                        document.body.removeChild(link);

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(sfImageAltTextData.i18n.exportComplete);
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Export failed.');
                    }
                }
            });
        },

        /**
         * Copy all source URLs to clipboard
         */
        copySources: function() {
            var modal = $('.sf-sources-modal');
            var sources = modal.data('sources') || [];
            var urls = sources.map(function(s) { return s.url; }).join('\n');

            navigator.clipboard.writeText(urls).then(function() {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.success('URLs copied to clipboard');
                }
            }).catch(function() {
                // Fallback
                var textarea = document.createElement('textarea');
                textarea.value = urls;
                document.body.appendChild(textarea);
                textarea.select();
                document.execCommand('copy');
                document.body.removeChild(textarea);

                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.success('URLs copied to clipboard');
                }
            });
        },

        // =====================================
        // PAGINATION FUNCTIONS
        // =====================================

        /**
         * Initialize pagination for all sections
         */
        initPagination: function() {
            var self = this;
            $('.sf-image-alt-text-module .sf-paginated-table').each(function() {
                var $wrapper = $(this);
                var section = $wrapper.data('section');
                var perPage = parseInt($wrapper.data('per-page'), 10) || 100;
                var total = parseInt($wrapper.data('total'), 10) || 0;

                if (self.pagination[section]) {
                    self.pagination[section].perPage = perPage;
                    self.pagination[section].totalItems = total;
                    self.pagination[section].currentPage = 1;

                    self.applyPagination(section);
                    self.renderPageNumbers(section);
                    self.updatePaginationInfo(section);
                }
            });
        },

        /**
         * Go to a specific page
         */
        goToPage: function(section, page) {
            var pag = this.pagination[section];
            if (!pag) return;

            var totalPages = Math.ceil(this.getVisibleRowCount(section) / pag.perPage);
            page = Math.max(1, Math.min(page, totalPages));

            if (page === pag.currentPage) return;

            var scrollTop = $(window).scrollTop();

            pag.currentPage = page;
            this.applyPagination(section);
            this.renderPageNumbers(section);
            this.updatePaginationInfo(section);

            $(window).scrollTop(scrollTop);
        },

        /**
         * Apply pagination to a section - show/hide rows based on current page
         */
        applyPagination: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $wrapper = $('.sf-image-alt-text-module .sf-paginated-table[data-section="' + section + '"]');
            var $rows;

            if (section === 'fixable') {
                $rows = $wrapper.find('tbody .sf-image-row[data-category="fixable"]');
            } else if (section === 'manual') {
                $rows = $wrapper.find('tbody .sf-manual-row');
            } else if (section === 'skipped') {
                $rows = $wrapper.find('.sf-skipped-link-item');
            } else if (section === 'fixed') {
                $rows = $wrapper.find('tbody .sf-fixed-row');
            }

            if (!$rows || !$rows.length) return;

            var startIndex = (pag.currentPage - 1) * pag.perPage;
            var endIndex = startIndex + pag.perPage;

            var visibleIndex = 0;
            $rows.each(function() {
                var $row = $(this);
                var isFilteredOut = $row.hasClass('sf-filtered-out');

                if (isFilteredOut) {
                    $row.addClass('sf-page-hidden');
                } else {
                    if (visibleIndex >= startIndex && visibleIndex < endIndex) {
                        $row.removeClass('sf-page-hidden');
                    } else {
                        $row.addClass('sf-page-hidden');
                    }
                    visibleIndex++;
                }
            });

            // Update button states
            var $pagination = $('.sf-image-alt-text-module .sf-pagination[data-section="' + section + '"]');
            var totalPages = Math.ceil(this.getVisibleRowCount(section) / pag.perPage);

            $pagination.find('.sf-page-prev').prop('disabled', pag.currentPage <= 1);
            $pagination.find('.sf-page-next').prop('disabled', pag.currentPage >= totalPages);
        },

        /**
         * Get count of visible (non-filtered) rows in a section
         */
        getVisibleRowCount: function(section) {
            var $wrapper = $('.sf-image-alt-text-module .sf-paginated-table[data-section="' + section + '"]');
            var $rows;

            if (section === 'fixable') {
                $rows = $wrapper.find('tbody .sf-image-row[data-category="fixable"]:not(.sf-filtered-out)');
            } else if (section === 'manual') {
                $rows = $wrapper.find('tbody .sf-manual-row:not(.sf-filtered-out)');
            } else if (section === 'skipped') {
                $rows = $wrapper.find('.sf-skipped-link-item:not(.sf-filtered-out)');
            } else if (section === 'fixed') {
                $rows = $wrapper.find('tbody .sf-fixed-row:not(.sf-filtered-out)');
            }

            return $rows ? $rows.length : 0;
        },

        /**
         * Render page number buttons
         */
        renderPageNumbers: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('.sf-image-alt-text-module .sf-pagination[data-section="' + section + '"]');
            var $container = $pagination.find('.sf-page-numbers');

            var visibleCount = this.getVisibleRowCount(section);
            var totalPages = Math.ceil(visibleCount / pag.perPage);
            var currentPage = pag.currentPage;

            if (totalPages <= 1) {
                $container.empty();
                $pagination.find('.sf-page-prev, .sf-page-next').prop('disabled', true);
                return;
            }

            var html = '';
            var maxVisible = 7;

            if (totalPages <= maxVisible) {
                for (var i = 1; i <= totalPages; i++) {
                    html += this.renderPageButton(i, currentPage);
                }
            } else {
                html += this.renderPageButton(1, currentPage);

                if (currentPage > 3) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }

                var startPage = Math.max(2, currentPage - 1);
                var endPage = Math.min(totalPages - 1, currentPage + 1);

                if (currentPage <= 3) {
                    endPage = Math.min(4, totalPages - 1);
                }
                if (currentPage >= totalPages - 2) {
                    startPage = Math.max(2, totalPages - 3);
                }

                for (var j = startPage; j <= endPage; j++) {
                    html += this.renderPageButton(j, currentPage);
                }

                if (currentPage < totalPages - 2) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }

                html += this.renderPageButton(totalPages, currentPage);
            }

            $container.html(html);
        },

        /**
         * Render a single page button
         */
        renderPageButton: function(pageNum, currentPage) {
            var isActive = pageNum === currentPage;
            return '<button type="button" class="sf-page-number' + (isActive ? ' active' : '') + '" data-page="' + pageNum + '">' + pageNum + '</button>';
        },

        /**
         * Update pagination info text
         */
        updatePaginationInfo: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('.sf-image-alt-text-module .sf-pagination[data-section="' + section + '"]');
            var visibleCount = this.getVisibleRowCount(section);

            var start = Math.min((pag.currentPage - 1) * pag.perPage + 1, visibleCount);
            var end = Math.min(pag.currentPage * pag.perPage, visibleCount);

            if (visibleCount === 0) {
                start = 0;
                end = 0;
            }

            $pagination.find('.sf-page-start').text(start);
            $pagination.find('.sf-page-end').text(end);
            $pagination.find('.sf-page-total').text(visibleCount);
        },

        /**
         * Reset pagination to page 1
         */
        resetPagination: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            pag.currentPage = 1;
            this.applyPagination(section);
            this.renderPageNumbers(section);
            this.updatePaginationInfo(section);
        },

        /**
         * Process bulk update
         */
        processBulkUpdate: function() {
            var self = this;

            if (self.bulkProcessing) {
                return;
            }

            self.bulkProcessing = true;

            var $section = $('.sf-image-alt-text-module .sf-bulk-confirmation');
            var matchedImages = parseInt($section.find('.sf-bulk-stat-ready .sf-bulk-stat-number').text(), 10) || 0;
            if (matchedImages === 0) {
                matchedImages = parseInt($('.sf-image-alt-text-module .sf-bulk-confirm').data('count'), 10) || 0;
            }

            if (matchedImages === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning('No matching images found to update.');
                }
                self.bulkProcessing = false;
                return;
            }

            var $modal = $('.sf-image-alt-text-module .sf-bulk-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-bulk-progress-current');
            var $total = $modal.find('.sf-bulk-progress-total');
            var $percent = $modal.find('.sf-bulk-progress-percent');
            var $currentUrl = $modal.find('.sf-bulk-current-url');

            $total.text(matchedImages);
            $current.text(0);
            $percent.text('0');
            $fill.css('width', '0%');
            $currentUrl.text('Starting...');
            $modal.show();

            $('.sf-image-alt-text-module .sf-bulk-confirm').prop('disabled', true);

            self.processBulkBatch(0, matchedImages);
        },

        /**
         * Process a batch of bulk updates
         */
        processBulkBatch: function(offset, total) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_apply_bulk_updates',
                    nonce: sfImageAltTextData.nonce,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = data.processed || 0;
                        var currentOffset = offset + processed;
                        var percent = Math.min(100, Math.round((currentOffset / total) * 100));

                        var $modal = $('.sf-image-alt-text-module .sf-bulk-progress-modal');
                        $modal.find('.sf-bulk-progress-current').text(currentOffset);
                        $modal.find('.sf-bulk-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        if (data.complete) {
                            self.bulkProcessing = false;
                            $modal.hide();
                            location.reload();
                        } else {
                            self.processBulkBatch(currentOffset, total);
                        }
                    } else {
                        self.bulkProcessing = false;
                        $('.sf-image-alt-text-module .sf-bulk-progress-modal').hide();
                        $('.sf-image-alt-text-module .sf-bulk-confirm').prop('disabled', false);

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Bulk update failed.');
                        }
                    }
                },
                error: function() {
                    self.bulkProcessing = false;
                    $('.sf-image-alt-text-module .sf-bulk-progress-modal').hide();
                    $('.sf-image-alt-text-module .sf-bulk-confirm').prop('disabled', false);

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Bulk update failed.');
                    }
                }
            });
        },

        /**
         * Clear bulk data and return to upload state
         */
        clearBulkData: function() {
            if (!confirm(sfImageAltTextData.i18n.confirmClear || 'Are you sure you want to clear all image alt text data? This will allow you to upload a new CSV file.')) {
                return;
            }

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_clear_data',
                    nonce: sfImageAltTextData.nonce
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Clear bulk complete results
         */
        clearBulkCompleteData: function() {
            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_clear_data',
                    nonce: sfImageAltTextData.nonce
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Download bulk preview CSV
         */
        downloadBulkPreview: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfImageAltTextData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_image_alt_text_download_preview' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfImageAltTextData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Download bulk results CSV
         */
        downloadBulkResults: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfImageAltTextData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_image_alt_text_download_results' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfImageAltTextData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Start new upload after bulk complete
         */
        startNewUpload: function() {
            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_clear_data',
                    nonce: sfImageAltTextData.nonce
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Clear all image alt text data
         */
        clearData: function() {
            // Confirm before clearing
            if (!confirm(sfImageAltTextData.i18n.confirmClear || 'Are you sure you want to clear all image alt text data? This will allow you to upload a new CSV file.')) {
                return;
            }

            var $btn = $('.sf-clear-data-btn');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfImageAltTextData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_image_alt_text_clear_data',
                    nonce: sfImageAltTextData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message || 'Data cleared successfully.');
                        }
                        // Reload page to show upload form
                        location.reload();
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Failed to clear data.');
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to clear data.');
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        // Only init if we're on the image-alt-text tab
        if ($('.sf-image-alt-text-module').length) {
            ScreamingFixes.ImageAltText.init();
        }
    });

})(jQuery);
