<?php
/**
 * Image Alt Text Module for Screaming Fixes
 *
 * Finds and fixes images with missing alt text from Screaming Frog exports
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Image_Alt_Text extends SF_Module {

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'image-alt-text';

    /**
     * Maximum number of CSV rows allowed
     * @var int
     */
    const MAX_CSV_ROWS = 5000;

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Image Alt Text', 'screaming-fixes');
        $this->slug = 'image-alt-text';
        $this->description = __('Find and fix images missing alt text automatically.', 'screaming-fixes');

        parent::__construct();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register AJAX handlers
        add_action('wp_ajax_sf_image_alt_text_process_csv', [$this, 'ajax_process_csv']);
        add_action('wp_ajax_sf_image_alt_text_apply_fixes', [$this, 'ajax_apply_fixes']);
        add_action('wp_ajax_sf_image_alt_text_get_ai_suggestion', [$this, 'ajax_get_ai_suggestion']);
        add_action('wp_ajax_sf_image_alt_text_ai_suggest_all', [$this, 'ajax_ai_suggest_all']);
        add_action('wp_ajax_sf_image_alt_text_export', [$this, 'ajax_export']);
        add_action('wp_ajax_sf_image_alt_text_export_results', [$this, 'ajax_export_results']);
        add_action('wp_ajax_sf_image_alt_text_get_data', [$this, 'ajax_get_data']);
        add_action('wp_ajax_sf_image_alt_text_clear_data', [$this, 'ajax_clear_data']);
        add_action('wp_ajax_sf_image_alt_text_apply_bulk_updates', [$this, 'ajax_apply_bulk_updates']);
        add_action('wp_ajax_sf_image_alt_text_download_preview', [$this, 'ajax_download_bulk_preview']);
        add_action('wp_ajax_sf_image_alt_text_download_results', [$this, 'ajax_download_bulk_results']);

        // Enqueue module assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        // Check if we're on the image alt text tab
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'image-alt-text') {
            return;
        }

        // Use file modification time for cache busting
        $css_file = SF_PLUGIN_DIR . 'modules/image-alt-text/assets/image-alt-text.css';
        $js_file = SF_PLUGIN_DIR . 'modules/image-alt-text/assets/image-alt-text.js';
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        wp_enqueue_style(
            'sf-image-alt-text',
            SF_PLUGIN_URL . 'modules/image-alt-text/assets/image-alt-text.css',
            ['screaming-fixes-admin'],
            $css_version
        );

        wp_enqueue_script(
            'sf-image-alt-text',
            SF_PLUGIN_URL . 'modules/image-alt-text/assets/image-alt-text.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        wp_localize_script('sf-image-alt-text', 'sfImageAltTextData', [
            'nonce' => wp_create_nonce('sf_image_alt_text_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'i18n' => [
                'processing' => __('Processing CSV...', 'screaming-fixes'),
                'scanningImages' => __('Scanning images...', 'screaming-fixes'),
                'applyingFixes' => __('Applying alt text fixes...', 'screaming-fixes'),
                'fixesApplied' => __('Alt text fixes applied successfully!', 'screaming-fixes'),
                'fixesFailed' => __('Some fixes failed. Check the results.', 'screaming-fixes'),
                'noFixesSelected' => __('No fixes selected.', 'screaming-fixes'),
                'aiSuggesting' => __('Getting AI suggestions for alt text...', 'screaming-fixes'),
                'aiComplete' => __('AI suggestions complete.', 'screaming-fixes'),
                'aiFailed' => __('Failed to get AI suggestions.', 'screaming-fixes'),
                'setAltText' => __('Set alt text', 'screaming-fixes'),
                'ignore' => __('Ignore', 'screaming-fixes'),
                'confirmApply' => __('Apply alt text to %d images?', 'screaming-fixes'),
                'exporting' => __('Exporting...', 'screaming-fixes'),
                'exportComplete' => __('Export complete.', 'screaming-fixes'),
                'confirmClear' => __('Are you sure you want to clear all image alt text data? This will allow you to upload a new CSV file.', 'screaming-fixes'),
                'dataCleared' => __('Data cleared successfully.', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Run the module scan
     *
     * @return array Scan results
     */
    public function run_scan() {
        // This module doesn't do active scanning - it processes uploaded CSVs
        return $this->get_results();
    }

    /**
     * Check if this module can handle a CSV based on headers
     *
     * @param array $headers CSV column headers (lowercase)
     * @return bool
     */
    public function can_handle_csv($headers) {
        // Normalize headers to lowercase
        $headers = array_map('strtolower', $headers);

        // Screaming Frog "Images" export
        // Must have: Destination (image URL), Alt Text column
        // Optional: Type, Source, Link Position
        $has_destination = in_array('destination', $headers) ||
                           in_array('image url', $headers) ||
                           in_array('url', $headers);

        $has_alt_text = in_array('alt text', $headers) ||
                        in_array('alt', $headers) ||
                        in_array('image alt text', $headers);

        // Also check for Type column with "Image" value indicator
        $has_type = in_array('type', $headers);

        // Must have destination AND alt text columns
        // Having Type column is a strong indicator this is an Images export
        return $has_destination && $has_alt_text;
    }

    /**
     * Process uploaded CSV file
     *
     * Uses the Source URL and Link Position from CSV to identify which post
     * contains the image and categorize it as fixable, manual, or skip.
     *
     * @param string $file_path Path to uploaded CSV
     * @return array|WP_Error Processed results or error
     */
    public function process_csv($file_path) {
        $parser = new SF_CSV_Parser();

        // Parse the CSV
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Standardize column names for image-alt-text
        $parsed = $this->standardize_image_columns($parsed);

        // Process each row - filter to only images with missing alt text
        $all_images = [];
        foreach ($parsed['rows'] as $row) {
            $alt_text = isset($row['alt_text']) ? trim($row['alt_text']) : '';
            $type = isset($row['type']) ? strtolower(trim($row['type'])) : '';

            // Only include rows where alt text is empty
            if (!empty($alt_text)) {
                continue;
            }

            // Optionally filter by Type = "Image" if present
            if (!empty($type) && $type !== 'image') {
                continue;
            }

            $image_url = $row['destination'] ?? $row['url'] ?? '';
            $source_url = $row['source'] ?? '';
            $anchor = $row['anchor'] ?? '';
            $link_position = $row['link_position'] ?? $row['position'] ?? '';

            if (empty($image_url)) {
                continue;
            }

            // Categorize this image based on source URL and link position
            $categorization = $this->categorize_image($source_url, $link_position);

            $post_id = $categorization['post_id'];
            $post_title = $post_id ? get_the_title($post_id) : '';
            $edit_url = $post_id ? get_edit_post_link($post_id, 'raw') : '';

            $all_images[] = [
                'image_url' => $image_url,
                'alt_text' => $alt_text,
                'anchor' => $anchor,
                'source_url' => $source_url,
                'link_position' => $link_position,
                'post_id' => $post_id,
                'post_title' => $post_title,
                'edit_url' => $edit_url,
                'fix_category' => $categorization['category'],
                'fix_note' => $categorization['note'],
                'location' => $categorization['location'],
            ];
        }

        // Group by image URL so same image on multiple pages shows together
        $grouped = [];
        foreach ($all_images as $image) {
            $image_url = $image['image_url'];

            if (!isset($grouped[$image_url])) {
                $grouped[$image_url] = [
                    'image_url' => $image_url,
                    'filename' => basename(wp_parse_url($image_url, PHP_URL_PATH)),
                    'sources' => [],
                    'suggested_alt' => '',
                    'new_alt' => '',
                ];
            }

            // Add this source to the list (avoid duplicates by source_url + link_position)
            $source_key = $image['source_url'] . '_' . $image['link_position'];
            if (!isset($grouped[$image_url]['sources'][$source_key])) {
                $grouped[$image_url]['sources'][$source_key] = [
                    'post_id' => $image['post_id'],
                    'post_title' => $image['post_title'],
                    'source_url' => $image['source_url'],
                    'edit_url' => $image['edit_url'],
                    'anchor' => $image['anchor'],
                    'link_position' => $image['link_position'],
                    'fix_category' => $image['fix_category'],
                    'fix_note' => $image['fix_note'],
                    'location' => $image['location'],
                ];
            }
        }

        // Convert sources from associative to indexed array and categorize each image
        $fixable_images = [];
        $manual_images = [];
        $skipped_images = [];

        foreach ($grouped as $image_url => $data) {
            $data['sources'] = array_values($data['sources']);
            $data['source_count'] = count($data['sources']);

            // Count sources by category for this image
            $fixable_sources = [];
            $manual_sources = [];
            $skip_sources = [];

            foreach ($data['sources'] as $source) {
                switch ($source['fix_category']) {
                    case 'fixable':
                        $fixable_sources[] = $source;
                        break;
                    case 'manual':
                        $manual_sources[] = $source;
                        break;
                    case 'skip':
                        $skip_sources[] = $source;
                        break;
                }
            }

            // Store categorized sources
            $data['fixable_sources'] = $fixable_sources;
            $data['manual_sources'] = $manual_sources;
            $data['skip_sources'] = $skip_sources;
            $data['fixable_count'] = count($fixable_sources);
            $data['manual_count'] = count($manual_sources);
            $data['skip_count'] = count($skip_sources);

            // Determine overall category for this image
            // If ANY source is fixable, it goes in fixable
            // If no fixable but some manual, goes in manual
            // If only skipped, goes in skipped
            if (!empty($fixable_sources)) {
                $data['overall_category'] = 'fixable';
                $fixable_images[] = $data;
            } elseif (!empty($manual_sources)) {
                $data['overall_category'] = 'manual';
                $manual_images[] = $data;
            } else {
                $data['overall_category'] = 'skip';
                $skipped_images[] = $data;
            }
        }

        // Calculate totals
        $total_fixable = 0;
        $total_manual = 0;
        $total_skipped = 0;

        foreach ($fixable_images as $image) {
            $total_fixable += $image['fixable_count'];
        }
        foreach ($manual_images as $image) {
            $total_manual += $image['manual_count'];
        }
        foreach ($skipped_images as $image) {
            $total_skipped += $image['skip_count'];
        }

        $results = [
            'images' => array_merge($fixable_images, $manual_images, $skipped_images),
            'fixable_images' => $fixable_images,
            'manual_images' => $manual_images,
            'skipped_images' => $skipped_images,
            'total_count' => count($fixable_images) + count($manual_images) + count($skipped_images),
            'total_sources' => $total_fixable + $total_manual + $total_skipped,
            'fixable_count' => count($fixable_images),
            'manual_count' => count($manual_images),
            'skipped_count' => count($skipped_images),
            'fixable_sources' => $total_fixable,
            'manual_sources' => $total_manual,
            'skipped_sources' => $total_skipped,
            'processed_at' => current_time('mysql'),
        ];

        // Save results
        $this->save_results($results);

        // Also save to uploads table for persistence
        $this->save_upload_data($results);

        return $results;
    }

    /**
     * Standardize column names for image alt text CSV
     *
     * @param array $parsed Parsed CSV data
     * @return array Standardized data
     */
    private function standardize_image_columns($parsed) {
        $column_map = [
            'destination' => ['destination', 'image url', 'url', 'image_url', 'img url'],
            'alt_text' => ['alt text', 'alt', 'image alt text', 'alt_text', 'alttext'],
            'source' => ['source', 'from', 'page url', 'source url', 'page'],
            'type' => ['type', 'content type'],
            'link_position' => ['link position', 'position', 'location'],
            'anchor' => ['anchor', 'anchor text', 'link text'],
            'status_code' => ['status code', 'status', 'http status'],
            'size' => ['size (bytes)', 'size', 'file size', 'bytes'],
        ];

        // Create a header map
        $header_map = [];
        foreach ($parsed['headers'] as $index => $header) {
            $normalized = strtolower(trim($header));
            foreach ($column_map as $standard => $variants) {
                if (in_array($normalized, $variants)) {
                    $header_map[$index] = $standard;
                    break;
                }
            }
            if (!isset($header_map[$index])) {
                $header_map[$index] = $normalized;
            }
        }

        // Remap rows
        $remapped_rows = [];
        foreach ($parsed['rows'] as $row) {
            $new_row = [];
            foreach ($row as $key => $value) {
                if (is_numeric($key) && isset($header_map[$key])) {
                    $new_row[$header_map[$key]] = $value;
                } else {
                    // Try to find the standard name
                    $normalized_key = strtolower(trim($key));
                    $found = false;
                    foreach ($column_map as $standard => $variants) {
                        if (in_array($normalized_key, $variants) || $normalized_key === $standard) {
                            $new_row[$standard] = $value;
                            $found = true;
                            break;
                        }
                    }
                    if (!$found) {
                        $new_row[$key] = $value;
                    }
                }
            }
            $remapped_rows[] = $new_row;
        }

        $parsed['rows'] = $remapped_rows;
        return $parsed;
    }

    /**
     * Categorize an image based on its source URL and link position
     *
     * Uses Screaming Frog's Link Position column for accurate categorization.
     * Content = fixable (in post/page content)
     * Navigation/Footer/Sidebar = manual (theme areas)
     *
     * @param string $source_url The source URL where the image was found
     * @param string $link_position The Link Position from Screaming Frog CSV
     * @return array Category info with 'category', 'post_id', and 'note'
     */
    private function categorize_image($source_url, $link_position = '') {
        $position = strtolower(trim($link_position));
        $path = wp_parse_url($source_url, PHP_URL_PATH);
        $path = $path ?: '/';

        // First check Link Position from Screaming Frog - this is most reliable
        if (!empty($position)) {
            // Navigation images - manual fix
            if (in_array($position, ['navigation', 'nav', 'menu'])) {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Navigation - fix in Appearance → Menus or theme settings', 'screaming-fixes'),
                    'location' => 'navigation',
                ];
            }

            // Footer images - manual fix
            if ($position === 'footer') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Footer - fix in Appearance → Widgets or theme settings', 'screaming-fixes'),
                    'location' => 'footer',
                ];
            }

            // Sidebar images - manual fix
            if ($position === 'sidebar') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Sidebar - fix in Appearance → Widgets', 'screaming-fixes'),
                    'location' => 'sidebar',
                ];
            }

            // Header/Logo - manual fix
            if (in_array($position, ['header', 'logo', 'site logo'])) {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Header/Logo - fix in Appearance → Customize', 'screaming-fixes'),
                    'location' => 'header',
                ];
            }

            // Content images - try to find the post
            if ($position === 'content') {
                $post_id = $this->url_to_post_id($source_url);
                if ($post_id > 0) {
                    return [
                        'category' => 'fixable',
                        'post_id' => $post_id,
                        'note' => '',
                        'location' => 'content',
                    ];
                }
                // Content image but no post ID - might be dynamic page
            }
        }

        // Check URL patterns for dynamic/unfixable pages

        // Homepage - flag for manual fix
        if ($path === '/' || $path === '' || $path === '/index.php') {
            return [
                'category' => 'manual',
                'post_id' => 0,
                'note' => __('Homepage - fix manually in theme or widgets', 'screaming-fixes'),
                'location' => 'homepage',
            ];
        }

        // Pagination - skip
        if (preg_match('/\/page\/\d+\/?$/', $path)) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Pagination page - will be fixed when source content is fixed', 'screaming-fixes'),
                'location' => 'pagination',
            ];
        }

        // Author pages - skip
        if (strpos($path, '/author/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Author archive - dynamic page', 'screaming-fixes'),
                'location' => 'author',
            ];
        }

        // Category pages - skip
        if (strpos($path, '/category/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Category archive - dynamic page', 'screaming-fixes'),
                'location' => 'category',
            ];
        }

        // Tag pages - skip
        if (strpos($path, '/tag/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Tag archive - dynamic page', 'screaming-fixes'),
                'location' => 'tag',
            ];
        }

        // Try to get post ID for remaining URLs
        $post_id = $this->url_to_post_id($source_url);

        if ($post_id > 0) {
            return [
                'category' => 'fixable',
                'post_id' => $post_id,
                'note' => '',
                'location' => 'content',
            ];
        }

        // No post ID found - might be widget, theme template, or custom
        return [
            'category' => 'manual',
            'post_id' => 0,
            'note' => __('Source not found in database - may be in theme, widget, or plugin', 'screaming-fixes'),
            'location' => 'unknown',
        ];
    }

    /**
     * Convert a URL to a WordPress post ID
     *
     * @param string $url The URL to convert
     * @return int Post ID or 0 if not found
     */
    private function url_to_post_id($url) {
        if (empty($url)) {
            return 0;
        }

        // Try WordPress built-in function first
        $post_id = url_to_postid($url);

        if ($post_id) {
            return $post_id;
        }

        // Try with trailing slash variations
        $url_with_slash = trailingslashit($url);
        $url_without_slash = untrailingslashit($url);

        $post_id = url_to_postid($url_with_slash);
        if ($post_id) {
            return $post_id;
        }

        $post_id = url_to_postid($url_without_slash);
        if ($post_id) {
            return $post_id;
        }

        // Try extracting path and searching by slug
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $path = trim($path, '/');
            $slug = basename($path);

            if ($slug) {
                global $wpdb;

                // Search by post_name (slug)
                $post_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts}
                     WHERE post_name = %s
                     AND post_status IN ('publish', 'draft', 'pending', 'private')
                     AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')
                     LIMIT 1",
                    $slug
                ));

                if ($post_id) {
                    return (int) $post_id;
                }
            }
        }

        return 0;
    }

    /**
     * Get issue count for dashboard display
     *
     * @return int Count of images missing alt text
     */
    public function get_issue_count() {
        $results = $this->get_results();

        if (empty($results) || !isset($results['total_count'])) {
            // Check uploads table
            $upload_data = $this->get_upload_data();
            if ($upload_data && isset($upload_data['total_count'])) {
                return (int) $upload_data['total_count'];
            }
            return 0;
        }

        return (int) $results['total_count'];
    }

    /**
     * Apply approved alt text fixes
     *
     * @param array $fixes Array of fixes to apply
     * @return array Results with success/failure counts and per-image results
     */
    public function apply_fixes($fixes) {
        $results = [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => [],
            'details' => [],
            'image_results' => [], // Per-image results for frontend
        ];

        // Start batch tracking for undo capability
        SF_Batch_Restore::start_batch('image-alt-text', $fixes);

        foreach ($fixes as $fix) {
            // Skip if no alt text or ignored
            if (empty($fix['new_alt']) || $fix['action'] === 'ignore') {
                $results['skipped']++;
                continue;
            }

            $image_url = $fix['image_url'] ?? '';
            $new_alt = $fix['new_alt'] ?? '';
            $post_ids = $fix['post_ids'] ?? [];

            if (empty($image_url) || empty($new_alt)) {
                $results['skipped']++;
                continue;
            }

            $results['total']++;

            // Track per-image result
            $image_result = [
                'image_url' => $image_url,
                'new_alt' => $new_alt,
                'success' => false,
                'error' => '',
                'posts_updated' => 0,
                'posts_failed' => 0,
            ];

            // If specific post IDs provided, fix only those posts
            if (!empty($post_ids)) {
                $image_success_count = 0;
                $image_fail_count = 0;
                $image_errors = [];

                foreach ($post_ids as $post_id) {
                    if (!$post_id) {
                        continue;
                    }

                    $fix_result = $this->update_image_alt_in_post($post_id, $image_url, $new_alt);

                    if (is_wp_error($fix_result)) {
                        $image_fail_count++;
                        $results['failed']++;
                        $error_msg = $fix_result->get_error_message();
                        $image_errors[] = $error_msg;
                        $results['errors'][] = [
                            'post_id' => $post_id,
                            'image_url' => $image_url,
                            'error' => $error_msg,
                        ];
                    } else {
                        $image_success_count++;
                        $results['success']++;
                    }
                }

                $image_result['posts_updated'] = $image_success_count;
                $image_result['posts_failed'] = $image_fail_count;
                $image_result['success'] = ($image_success_count > 0);
                if (!empty($image_errors)) {
                    $image_result['error'] = implode('; ', array_unique($image_errors));
                }
            } else {
                // Bulk update across all posts
                $fix_result = $this->bulk_update_image_alt($image_url, $new_alt);
                $results['success'] += $fix_result['success'];
                $results['failed'] += $fix_result['failed'];
                $results['errors'] = array_merge($results['errors'], $fix_result['errors']);

                $image_result['posts_updated'] = $fix_result['success'];
                $image_result['posts_failed'] = $fix_result['failed'];
                $image_result['success'] = ($fix_result['success'] > 0);
                if (!empty($fix_result['errors'])) {
                    $error_messages = array_column($fix_result['errors'], 'error');
                    $image_result['error'] = implode('; ', array_unique($error_messages));
                }
            }

            $results['details'][] = [
                'image_url' => $image_url,
                'new_alt' => $new_alt,
            ];

            $results['image_results'][] = $image_result;
        }

        // Update stored results to reflect fixes applied (only successful ones)
        $this->update_results_after_fixes($results['image_results']);

        // Complete batch tracking
        SF_Batch_Restore::complete_batch($results);

        return $results;
    }

    /**
     * Update image alt text in a specific post
     *
     * @param int $post_id Post ID
     * @param string $image_url Image URL to find
     * @param string $new_alt New alt text
     * @return bool|WP_Error Success or error
     */
    private function update_image_alt_in_post($post_id, $image_url, $new_alt) {
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error('post_not_found', __('Post not found.', 'screaming-fixes'));
        }

        $content = $post->post_content;
        $original_content = $content;

        // Escape the URL for use in regex
        $escaped_url = preg_quote($image_url, '/');

        // Pattern to match img tags with this src
        // Handles both with and without existing alt attribute
        $pattern = '/(<img\s[^>]*src=["\']' . $escaped_url . '["\'][^>]*)(\/?>)/i';

        $content = preg_replace_callback($pattern, function($matches) use ($new_alt) {
            $img_tag = $matches[1];
            $closing = $matches[2];

            // Check if alt attribute already exists
            if (preg_match('/\salt=["\'][^"\']*["\']/', $img_tag)) {
                // Replace existing alt
                $img_tag = preg_replace('/\salt=["\'][^"\']*["\']/', ' alt="' . esc_attr($new_alt) . '"', $img_tag);
            } else {
                // Add alt attribute before closing
                $img_tag .= ' alt="' . esc_attr($new_alt) . '"';
            }

            return $img_tag . $closing;
        }, $content);

        // Also handle images where src might come after other attributes
        $pattern2 = '/(<img\s[^>]*)(\ssrc=["\']' . $escaped_url . '["\'])([^>]*)(\/?>)/i';

        $content = preg_replace_callback($pattern2, function($matches) use ($new_alt) {
            $before_src = $matches[1];
            $src_attr = $matches[2];
            $after_src = $matches[3];
            $closing = $matches[4];

            $full_tag = $before_src . $src_attr . $after_src;

            // Check if alt attribute already exists
            if (preg_match('/\salt=["\'][^"\']*["\']/', $full_tag)) {
                // Replace existing alt
                $full_tag = preg_replace('/\salt=["\'][^"\']*["\']/', ' alt="' . esc_attr($new_alt) . '"', $full_tag);
            } else {
                // Add alt attribute
                $full_tag .= ' alt="' . esc_attr($new_alt) . '"';
            }

            return $full_tag . $closing;
        }, $content);

        // Only update if content changed
        if ($content === $original_content) {
            // Check if the image already has the desired alt text (idempotent success)
            $alt_check_pattern = '/<img\s[^>]*src=["\']' . $escaped_url . '["\'][^>]*\salt=["\']' . preg_quote(esc_attr($new_alt), '/') . '["\'][^>]*\/?>/i';
            $alt_check_pattern2 = '/<img\s[^>]*\salt=["\']' . preg_quote(esc_attr($new_alt), '/') . '["\'][^>]*src=["\']' . $escaped_url . '["\'][^>]*\/?>/i';
            if (preg_match($alt_check_pattern, $original_content) || preg_match($alt_check_pattern2, $original_content)) {
                return true; // Already has the correct alt text
            }
            return new WP_Error('no_changes', __('Image not found in post content. Click "Found In" to check if it\'s inside a widget or page builder element.', 'screaming-fixes'));
        }

        // Update the post
        $updated = wp_update_post([
            'ID' => $post_id,
            'post_content' => $content,
        ], true);

        if (is_wp_error($updated)) {
            return $updated;
        }

        // Log the change
        $logger = new SF_Change_Logger();
        $logger->log_change($post_id, 'alt_text', $original_content, $content, [
            'module' => 'image-alt-text',
            'image_url' => $image_url,
            'new_alt' => $new_alt,
        ]);

        return true;
    }

    /**
     * Bulk update image alt text across all posts
     *
     * @param string $image_url Image URL to find
     * @param string $new_alt New alt text
     * @return array Results with success/failed counts
     */
    private function bulk_update_image_alt($image_url, $new_alt) {
        global $wpdb;

        $results = [
            'success' => 0,
            'failed' => 0,
            'errors' => [],
        ];

        // Find posts containing this image
        $posts = $wpdb->get_results($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts}
             WHERE post_content LIKE %s
             AND post_status IN ('publish', 'draft', 'pending', 'private')
             AND post_type NOT IN ('revision', 'nav_menu_item')",
            '%' . $wpdb->esc_like($image_url) . '%'
        ));

        foreach ($posts as $post) {
            $fix_result = $this->update_image_alt_in_post($post->ID, $image_url, $new_alt);

            if (is_wp_error($fix_result)) {
                $results['failed']++;
                $results['errors'][] = [
                    'post_id' => $post->ID,
                    'error' => $fix_result->get_error_message(),
                ];
            } else {
                $results['success']++;
            }
        }

        // Also update the attachment if this image is in the media library
        $attachment_id = attachment_url_to_postid($image_url);
        if ($attachment_id) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $new_alt);
        }

        return $results;
    }

    /**
     * Get AI suggestion for image alt text
     *
     * @param string $image_url The image URL
     * @param array $context Additional context (filename, source pages, etc.)
     * @return string|WP_Error Suggested alt text or error
     */
    public function get_ai_suggestion($image_url, $context = []) {
        $api_key = get_option('sf_claude_api_key');

        if (empty($api_key)) {
            return new WP_Error('no_api_key', __('Claude API key not configured. Add it in Settings.', 'screaming-fixes'));
        }

        $filename = $context['filename'] ?? basename(wp_parse_url($image_url, PHP_URL_PATH));
        $source_titles = $context['source_titles'] ?? [];

        $prompt = sprintf(
            "Generate a concise, descriptive alt text for an image.\n\n" .
            "Image filename: %s\n" .
            "Image URL: %s\n" .
            "%s\n\n" .
            "Guidelines:\n" .
            "- Be descriptive but concise (ideally 5-15 words)\n" .
            "- Describe what the image shows, not what it is\n" .
            "- Don't start with 'Image of' or 'Picture of'\n" .
            "- Consider the context from page titles if provided\n" .
            "- If the filename suggests the content, use that as a guide\n\n" .
            "Respond with ONLY the suggested alt text, nothing else.",
            $filename,
            $image_url,
            !empty($source_titles) ? "Pages containing this image:\n- " . implode("\n- ", array_slice($source_titles, 0, 5)) : ""
        );

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode([
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 100,
                'messages' => [
                    ['role' => 'user', 'content' => $prompt],
                ],
            ]),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return new WP_Error('api_error', sprintf(__('API returned status %d', 'screaming-fixes'), $status_code));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['content'][0]['text'])) {
            $suggestion = trim($body['content'][0]['text']);
            // Remove quotes if the AI wrapped the response in them
            $suggestion = trim($suggestion, '"\'');
            return $suggestion;
        }

        return new WP_Error('api_error', __('Unexpected API response', 'screaming-fixes'));
    }

    /**
     * AJAX: Process uploaded CSV
     */
    public function ajax_process_csv() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

        if (empty($upload_id)) {
            wp_send_json_error(['message' => __('No upload ID provided.', 'screaming-fixes')]);
        }

        // Get the uploaded file path from the uploads option
        $uploads = get_option('sf_pending_uploads', []);

        if (!isset($uploads[$upload_id])) {
            wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
        }

        $file_path = $uploads[$upload_id]['path'];

        if (!file_exists($file_path)) {
            wp_send_json_error(['message' => __('File not found.', 'screaming-fixes')]);
        }

        // Check if this is a bulk update CSV (has New_Alt_Text column)
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (!is_wp_error($parsed) && !empty($parsed['headers'])) {
            $bulk_check = $this->is_bulk_update_csv($parsed['headers']);

            if (is_array($bulk_check) && isset($bulk_check['error'])) {
                if ($bulk_check['error'] === 'no_url_column') {
                    wp_send_json_error(['message' => __('Bulk update CSV detected but missing a Source URL column. Please include a column named "source", "from", or "page url".', 'screaming-fixes')]);
                    return;
                }
                if ($bulk_check['error'] === 'no_image_column') {
                    wp_send_json_error(['message' => __('Bulk update CSV detected but missing an image URL column. Please include a column named "destination", "image url", or "url".', 'screaming-fixes')]);
                    return;
                }
            }

            if (is_array($bulk_check) && isset($bulk_check['is_bulk']) && $bulk_check['is_bulk'] === true) {
                $results = $this->process_bulk_csv($file_path);

                if (is_wp_error($results)) {
                    wp_send_json_error(['message' => $results->get_error_message()]);
                }

                unset($uploads[$upload_id]);
                update_option('sf_pending_uploads', $uploads);

                wp_send_json_success([
                    'message' => sprintf(
                        __('%d images ready to update', 'screaming-fixes'),
                        $results['ready_count']
                    ),
                    'data' => $results,
                    'is_bulk_update' => true,
                ]);
                return;
            }
        }

        $results = $this->process_csv($file_path);

        if (is_wp_error($results)) {
            wp_send_json_error(['message' => $results->get_error_message()]);
        }

        // Clean up the temp file reference
        unset($uploads[$upload_id]);
        update_option('sf_pending_uploads', $uploads);

        wp_send_json_success([
            'message' => sprintf(
                __('Found %d images missing alt text: %d fixable, %d manual, %d skipped.', 'screaming-fixes'),
                $results['total_count'],
                $results['fixable_count'],
                $results['manual_count'],
                $results['skipped_count']
            ),
            'data' => $results,
        ]);
    }

    /**
     * AJAX: Get stored data
     */
    public function ajax_get_data() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            wp_send_json_success(['data' => null]);
        }

        wp_send_json_success(['data' => $results]);
    }

    /**
     * AJAX: Clear all image alt text data
     */
    public function ajax_clear_data() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();

        // Delete from database
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => 'image-alt-text'
        ]);

        // Clear transient
        delete_transient('sf_image-alt-text_results');

        wp_send_json_success([
            'message' => __('Data cleared successfully. You can now upload a new CSV file.', 'screaming-fixes')
        ]);
    }

    /**
     * AJAX: Apply fixes
     */
    public function ajax_apply_fixes() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $fixes = isset($_POST['fixes']) ? $_POST['fixes'] : [];

        if (empty($fixes)) {
            wp_send_json_error(['message' => __('No fixes provided.', 'screaming-fixes')]);
        }

        // Sanitize fixes array
        // Note: wp_unslash() is required because WordPress adds magic quotes to $_POST data
        $sanitized_fixes = [];
        foreach ($fixes as $fix) {
            $image_url = isset($fix['image_url']) ? esc_url_raw(wp_unslash($fix['image_url'])) : '';

            // Sanitize post_ids array
            $post_ids = [];
            if (isset($fix['post_ids']) && is_array($fix['post_ids'])) {
                $post_ids = array_map('absint', $fix['post_ids']);
                $post_ids = array_filter($post_ids);
            }

            $sanitized_fixes[] = [
                'image_url' => $image_url,
                'action' => isset($fix['action']) ? sanitize_text_field(wp_unslash($fix['action'])) : 'set_alt',
                'new_alt' => isset($fix['new_alt']) ? sanitize_text_field(wp_unslash($fix['new_alt'])) : '',
                'post_ids' => $post_ids,
            ];
        }

        $results = $this->apply_fixes($sanitized_fixes);

        // Log to activity log for dashboard
        if ($results['success'] > 0) {
            SF_Activity_Log::log('image-alt-text', $results['success']);
        }

        wp_send_json_success([
            'message' => sprintf(
                __('Updated alt text on %d images. %d failed.', 'screaming-fixes'),
                $results['success'],
                $results['failed']
            ),
            'results' => $results,
            'image_results' => $results['image_results'], // Per-image results for frontend filtering
        ]);
    }

    /**
     * AJAX: Get AI suggestion for a single image
     */
    public function ajax_get_ai_suggestion() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $image_url = isset($_POST['image_url']) ? esc_url_raw($_POST['image_url']) : '';
        $filename = isset($_POST['filename']) ? sanitize_text_field($_POST['filename']) : '';
        $source_titles = isset($_POST['source_titles']) ? array_map('sanitize_text_field', (array) $_POST['source_titles']) : [];

        if (empty($image_url)) {
            wp_send_json_error(['message' => __('No image URL provided.', 'screaming-fixes')]);
        }

        $suggestion = $this->get_ai_suggestion($image_url, [
            'filename' => $filename,
            'source_titles' => $source_titles,
        ]);

        if (is_wp_error($suggestion)) {
            wp_send_json_error(['message' => $suggestion->get_error_message()]);
        }

        wp_send_json_success([
            'suggestion' => $suggestion,
            'image_url' => $image_url,
        ]);
    }

    /**
     * AJAX: Get AI suggestions for all images
     */
    public function ajax_ai_suggest_all() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results) || empty($results['images'])) {
            wp_send_json_error(['message' => __('No images found.', 'screaming-fixes')]);
        }

        $suggestions = [];
        $success_count = 0;
        $failed_count = 0;

        foreach ($results['images'] as $index => $image) {
            // Skip if already has suggested alt text
            if (!empty($image['suggested_alt']) || !empty($image['new_alt'])) {
                continue;
            }

            // Get source titles for context
            $source_titles = [];
            foreach ($image['sources'] as $source) {
                if (!empty($source['post_title'])) {
                    $source_titles[] = $source['post_title'];
                }
            }

            $suggestion = $this->get_ai_suggestion($image['image_url'], [
                'filename' => $image['filename'] ?? '',
                'source_titles' => $source_titles,
            ]);

            if (!is_wp_error($suggestion)) {
                $suggestions[$image['image_url']] = $suggestion;
                $success_count++;
            } else {
                $failed_count++;
            }

            // Small delay to avoid rate limiting
            usleep(100000); // 100ms
        }

        wp_send_json_success([
            'message' => sprintf(
                __('Generated %d suggestions. %d failed.', 'screaming-fixes'),
                $success_count,
                $failed_count
            ),
            'suggestions' => $suggestions,
        ]);
    }

    /**
     * AJAX: Export results as CSV
     */
    public function ajax_export() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results) || empty($results['images'])) {
            wp_send_json_error(['message' => __('No data to export.', 'screaming-fixes')]);
        }

        $export_data = [];
        foreach ($results['images'] as $image) {
            $source_urls = array_column($image['sources'], 'source_url');

            $export_data[] = [
                'Image URL' => $image['image_url'],
                'Filename' => $image['filename'] ?? '',
                'Found In Pages' => $image['source_count'] ?? count($image['sources']),
                'Source URLs' => implode('; ', $source_urls),
                'Suggested Alt Text' => $image['suggested_alt'] ?? '',
                'New Alt Text' => $image['new_alt'] ?? '',
            ];
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'image-alt-text-export-' . date('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX handler to export results (fixed + can't fix items)
     */
    public function ajax_export_results() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        $fixed_images = $results['fixed_images'] ?? [];
        $manual_images = $results['manual_images'] ?? [];

        if (empty($fixed_images) && empty($manual_images)) {
            wp_send_json_error(['message' => __('No results to export.', 'screaming-fixes')]);
        }

        $export_data = [];

        // Add fixed images
        foreach ($fixed_images as $image) {
            $sources = $image['fixable_sources'] ?? $image['sources'] ?? [];
            $source_info = $this->format_found_in_for_export($sources);

            $export_data[] = [
                'Image' => $image['image_url'] ?? '',
                'Found In' => $source_info,
                'Updated Alt Text' => $image['applied_alt'] ?? '',
                'Status' => __('Fixed', 'screaming-fixes'),
            ];
        }

        // Add can't fix images (manual)
        foreach ($manual_images as $image) {
            $sources = $image['manual_sources'] ?? $image['sources'] ?? [];
            $source_info = $this->format_found_in_for_export($sources);

            // Determine the reason it can't be fixed
            $reason = __('Manual Fix Required', 'screaming-fixes');
            if (!empty($sources[0]['location'])) {
                $location = $sources[0]['location'];
                switch ($location) {
                    case 'navigation':
                        $reason = __("Can't Fix - Navigation", 'screaming-fixes');
                        break;
                    case 'footer':
                        $reason = __("Can't Fix - Footer", 'screaming-fixes');
                        break;
                    case 'sidebar':
                        $reason = __("Can't Fix - Sidebar/Widget", 'screaming-fixes');
                        break;
                    case 'header':
                        $reason = __("Can't Fix - Header", 'screaming-fixes');
                        break;
                    default:
                        $reason = __("Can't Fix - Theme/Widget", 'screaming-fixes');
                }
            }

            $export_data[] = [
                'Image' => $image['image_url'] ?? '',
                'Found In' => $source_info,
                'Updated Alt Text' => '-',
                'Status' => $reason,
            ];
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'alt-text-results-' . gmdate('Y-m-d') . '.csv',
        ]);
    }

    /**
     * Format sources for the "Found In" column in exports
     *
     * @param array $sources Array of source data
     * @return string Formatted string like "Page Title (URL)"
     */
    private function format_found_in_for_export($sources) {
        if (empty($sources)) {
            return '-';
        }

        $formatted = [];
        foreach ($sources as $source) {
            $title = $source['post_title'] ?? '';
            $url = $source['source_url'] ?? '';

            if ($title && $url) {
                $formatted[] = $title . ' (' . $url . ')';
            } elseif ($url) {
                $formatted[] = $url;
            }
        }

        return implode('; ', $formatted);
    }

    /**
     * Save upload data to database
     *
     * @param array $data Data to save
     */
    private function save_upload_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        // Delete any existing data for this session/module
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Insert new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get upload data from database
     *
     * @return array|null Data or null if not found
     */
    private function get_upload_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            return json_decode($row->data, true);
        }

        return null;
    }

    /**
     * Get or create session ID
     *
     * @return string Session ID
     */
    private function get_session_id() {
        if (!session_id()) {
            // Use user ID as session for logged-in users
            return 'user_' . get_current_user_id();
        }
        return session_id();
    }

    /**
     * Update results after fixes are applied
     *
     * @param array $image_results Per-image results with success/failure status
     */
    private function update_results_after_fixes($image_results) {
        $results = $this->get_results();

        // Also check database if transient is empty
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results) || empty($results['images'])) {
            return;
        }

        // Initialize fixed_images array if not exists
        if (!isset($results['fixed_images'])) {
            $results['fixed_images'] = [];
        }

        // Create a map of SUCCESSFULLY fixed images by URL (only if success === true)
        $fixed_map = [];
        foreach ($image_results as $result) {
            // Only add to fixed_map if the fix was successful
            if (!empty($result['success']) && !empty($result['image_url'])) {
                $fixed_map[$result['image_url']] = $result['new_alt'] ?? '';
            }
        }

        // Move only successfully fixed images from images array to fixed_images array
        $remaining_images = [];
        $fixable_images = [];
        $manual_images = [];
        $skipped_images = [];

        foreach ($results['images'] as $image) {
            $image_url = $image['image_url'] ?? '';

            if (isset($fixed_map[$image_url])) {
                // This image was SUCCESSFULLY fixed - add to fixed_images with the new alt text
                $image['applied_alt'] = $fixed_map[$image_url];
                $image['fixed_at'] = current_time('mysql');
                $results['fixed_images'][] = $image;
            } else {
                // Not fixed - keep in appropriate category
                $remaining_images[] = $image;
                $category = $image['overall_category'] ?? 'fixable';
                switch ($category) {
                    case 'fixable':
                        $fixable_images[] = $image;
                        break;
                    case 'manual':
                        $manual_images[] = $image;
                        break;
                    case 'skip':
                        $skipped_images[] = $image;
                        break;
                }
            }
        }

        // Update results arrays
        $results['images'] = $remaining_images;
        $results['fixable_images'] = $fixable_images;
        $results['manual_images'] = $manual_images;
        $results['skipped_images'] = $skipped_images;

        // Update counts
        $results['total_count'] = count($remaining_images);
        $results['fixable_count'] = count($fixable_images);
        $results['manual_count'] = count($manual_images);
        $results['skipped_count'] = count($skipped_images);
        $results['fixed_images_count'] = count($results['fixed_images']);

        // Recalculate total sources
        $total_sources = 0;
        foreach ($remaining_images as $image) {
            $total_sources += $image['source_count'] ?? count($image['sources'] ?? []);
        }
        $results['total_sources'] = $total_sources;

        // Save updated results to both transient and database
        $this->save_results($results);
        $this->save_upload_data($results);
    }

    /**
     * Get module ID
     *
     * @return string Module ID
     */
    public function get_module_id() {
        return $this->module_id;
    }

    /**
     * Normalize a column name for matching
     *
     * @param string $name Column name
     * @return string Normalized name
     */
    private function normalize_column_name($name) {
        $normalized = strtolower(trim($name));
        $normalized = str_replace([' ', '-', '_'], '', $normalized);
        return $normalized;
    }

    /**
     * Normalize a URL for matching
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private function normalize_url($url) {
        $url = trim($url);
        $url = rtrim($url, '/');
        $url = preg_replace('/^http:\/\//i', 'https://', $url);
        $url = strtolower($url);
        return $url;
    }

    /**
     * Check if CSV is a bulk update CSV (has New_Alt_Text column)
     *
     * @param array $headers CSV headers
     * @return array|false Array with column info or false if not bulk
     */
    public function is_bulk_update_csv($headers) {
        $new_alt_header = null;
        $url_header = null;
        $image_header = null;

        // New alt text column patterns
        $alt_patterns = ['newalttext', 'newalt', 'newimagealt', 'newimagealtttext'];

        // Source URL column patterns (page where image appears)
        $url_patterns = ['source', 'sourceurl', 'pageurl', 'from'];

        // Image URL column patterns
        $image_patterns = ['destination', 'imageurl', 'src', 'url'];

        foreach ($headers as $header) {
            $normalized = $this->normalize_column_name($header);
            $header_lower = strtolower(trim($header));

            if (in_array($normalized, $alt_patterns)) {
                $new_alt_header = $header_lower;
            }

            if (in_array($normalized, $url_patterns)) {
                $url_header = $header_lower;
            }

            if (in_array($normalized, $image_patterns)) {
                $image_header = $header_lower;
            }
        }

        // Not a bulk CSV if no new alt column
        if ($new_alt_header === null) {
            return false;
        }

        // Must have source URL column
        if ($url_header === null) {
            return ['error' => 'no_url_column'];
        }

        // Must have image URL column
        if ($image_header === null) {
            return ['error' => 'no_image_column'];
        }

        return [
            'is_bulk' => true,
            'new_alt_header' => $new_alt_header,
            'url_header' => $url_header,
            'image_header' => $image_header,
        ];
    }

    /**
     * Process a bulk update CSV file
     *
     * @param string $file_path Path to CSV file
     * @return array|WP_Error Processing results
     */
    public function process_bulk_csv($file_path) {
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        $bulk_check = $this->is_bulk_update_csv($parsed['headers']);

        if ($bulk_check === false || isset($bulk_check['error'])) {
            return new WP_Error('not_bulk_csv', __('CSV must contain a New_Alt_Text column for bulk updates.', 'screaming-fixes'));
        }

        $new_alt_header = $bulk_check['new_alt_header'];
        $url_header = $bulk_check['url_header'];
        $image_header = $bulk_check['image_header'];

        // Build WordPress URL lookup map
        $wp_urls = $this->build_wp_url_map();

        $ready_updates = [];
        $not_matched = [];
        $skipped_empty = [];

        foreach ($parsed['rows'] as $row) {
            $source_url = isset($row[$url_header]) ? trim($row[$url_header]) : '';
            $image_url = isset($row[$image_header]) ? trim($row[$image_header]) : '';
            $new_alt = isset($row[$new_alt_header]) ? trim($row[$new_alt_header]) : '';

            if (empty($source_url) || empty($image_url)) {
                continue;
            }

            // Skip rows with empty new alt text
            if (empty($new_alt)) {
                $skipped_empty[] = [
                    'source_url' => $source_url,
                    'image_url' => $image_url,
                    'status' => 'Skipped - No new alt text',
                ];
                continue;
            }

            // Match source URL to WordPress post
            $normalized_url = $this->normalize_url($source_url);
            $post_id = null;

            if (isset($wp_urls[$normalized_url])) {
                $post_id = $wp_urls[$normalized_url]['post_id'];
            }

            $image_filename = basename(wp_parse_url($image_url, PHP_URL_PATH) ?: $image_url);

            // Get current alt text from the post content
            $current_alt = '';
            if ($post_id) {
                $post = get_post($post_id);
                if ($post) {
                    $escaped = preg_quote($image_url, '/');
                    if (preg_match('/<img\s[^>]*src=["\']' . $escaped . '["\'][^>]*alt=["\']([^"\']*)["\'][^>]*\/?>/i', $post->post_content, $m)) {
                        $current_alt = $m[1];
                    } elseif (preg_match('/<img\s[^>]*alt=["\']([^"\']*)["\'][^>]*src=["\']' . $escaped . '["\'][^>]*\/?>/i', $post->post_content, $m)) {
                        $current_alt = $m[1];
                    }
                }
            }

            if ($post_id) {
                $ready_updates[] = [
                    'source_url' => $source_url,
                    'image_url' => $image_url,
                    'post_id' => $post_id,
                    'current_alt' => $current_alt,
                    'new_alt' => $new_alt,
                    'image_filename' => $image_filename,
                ];
            } else {
                $not_matched[] = [
                    'source_url' => $source_url,
                    'image_url' => $image_url,
                    'new_alt' => $new_alt,
                    'image_filename' => $image_filename,
                    'status' => 'Skipped - URL not found',
                ];
            }
        }

        $results = [
            'is_bulk_update' => true,
            'ready_updates' => $ready_updates,
            'not_matched' => $not_matched,
            'skipped_empty' => $skipped_empty,
            'ready_count' => count($ready_updates),
            'not_matched_count' => count($not_matched),
            'skipped_empty_count' => count($skipped_empty),
            'total_rows' => $total_rows,
        ];

        $this->save_upload_data($results);

        return $results;
    }

    /**
     * Build a map of WordPress URLs to post IDs
     *
     * @return array URL map
     */
    private function build_wp_url_map() {
        global $wpdb;

        $map = [];

        $posts = $wpdb->get_results(
            "SELECT ID, post_name, post_type FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ('post', 'page')
             LIMIT 50000",
            ARRAY_A
        );

        foreach ($posts as $post) {
            $permalink = get_permalink($post['ID']);
            if ($permalink) {
                $normalized = $this->normalize_url($permalink);
                $map[$normalized] = [
                    'post_id' => (int) $post['ID'],
                    'post_type' => $post['post_type'],
                ];
            }
        }

        return $map;
    }

    /**
     * Apply bulk updates in a batch
     *
     * @param array $updates Array of updates to apply
     * @param int $offset Starting offset
     * @param int $batch_size Batch size
     * @return array Results
     */
    public function apply_bulk_alt_updates($updates, $offset = 0, $batch_size = 50) {
        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => [],
            'details' => [],
            'complete' => false,
            'next_offset' => $offset + $batch_size,
        ];

        $batch = array_slice($updates, $offset, $batch_size);

        if (empty($batch)) {
            $results['complete'] = true;
            return $results;
        }

        foreach ($batch as $update) {
            $results['processed']++;

            $post_id = isset($update['post_id']) ? (int) $update['post_id'] : 0;
            $image_url = isset($update['image_url']) ? trim($update['image_url']) : '';
            $new_alt = isset($update['new_alt']) ? trim($update['new_alt']) : '';

            if (!$post_id || empty($image_url) || $new_alt === '') {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $update['source_url'] ?? '',
                    'image_url' => $image_url,
                    'error' => __('Invalid post ID, image URL, or empty alt text', 'screaming-fixes'),
                ];
                continue;
            }

            $update_result = $this->update_image_alt_in_post($post_id, $image_url, $new_alt);

            if (is_wp_error($update_result)) {
                $results['failed']++;
                $results['errors'][] = [
                    'source_url' => $update['source_url'] ?? '',
                    'image_url' => $image_url,
                    'image_filename' => $update['image_filename'] ?? basename($image_url),
                    'error' => $update_result->get_error_message(),
                ];
            } else {
                $results['success']++;
                $results['details'][] = [
                    'source_url' => $update['source_url'] ?? '',
                    'image_url' => $image_url,
                    'image_filename' => $update['image_filename'] ?? basename($image_url),
                    'post_id' => $post_id,
                    'current_alt' => $update['current_alt'] ?? '',
                    'new_alt' => $new_alt,
                ];
            }
        }

        if ($offset + $batch_size >= count($updates)) {
            $results['complete'] = true;
        }

        return $results;
    }

    /**
     * AJAX: Apply bulk updates (handles batching)
     */
    public function ajax_apply_bulk_updates() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $offset = isset($_POST['offset']) ? (int) $_POST['offset'] : 0;
        $batch_size = isset($_POST['batch_size']) ? (int) $_POST['batch_size'] : 50;

        // Clear accumulated transient at the start
        if ($offset === 0) {
            delete_transient('sf_image_alt_bulk_accumulated_' . get_current_user_id());
        }

        // Get stored bulk data
        $bulk_data = $this->get_upload_data();

        if (empty($bulk_data) || empty($bulk_data['is_bulk_update']) || empty($bulk_data['ready_updates'])) {
            wp_send_json_error(['message' => __('No bulk update data found.', 'screaming-fixes')]);
        }

        $updates = $bulk_data['ready_updates'];
        $results = $this->apply_bulk_alt_updates($updates, $offset, $batch_size);

        if ($results['complete']) {
            $accumulated = get_transient('sf_image_alt_bulk_accumulated_' . get_current_user_id());
            if (!$accumulated) {
                $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
            }

            $accumulated['success'] += $results['success'];
            $accumulated['failed'] += $results['failed'];
            $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
            $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

            $final_results = [
                'is_bulk_update' => true,
                'bulk_complete' => true,
                'fixed_images' => $accumulated['details'],
                'failed_updates' => $accumulated['errors'],
                'success_count' => $accumulated['success'],
                'failed_count' => $accumulated['failed'],
                'not_matched' => $bulk_data['not_matched'] ?? [],
                'skipped_empty' => $bulk_data['skipped_empty'] ?? [],
            ];

            $this->save_upload_data($final_results);

            delete_transient('sf_image_alt_bulk_accumulated_' . get_current_user_id());

            if ($accumulated['success'] > 0) {
                SF_Activity_Log::log('image-alt-text', $accumulated['success']);
            }

            $results['total_success'] = $accumulated['success'];
            $results['total_failed'] = $accumulated['failed'];
            $results['all_details'] = $accumulated['details'];
            $results['all_errors'] = $accumulated['errors'];
        } else {
            $accumulated = get_transient('sf_image_alt_bulk_accumulated_' . get_current_user_id());
            if (!$accumulated) {
                $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
            }

            $accumulated['success'] += $results['success'];
            $accumulated['failed'] += $results['failed'];
            $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
            $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

            set_transient('sf_image_alt_bulk_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
        }

        $results['total_count'] = count($updates);

        wp_send_json_success($results);
    }

    /**
     * AJAX: Download preview CSV
     */
    public function ajax_download_bulk_preview() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_upload_data();

        if (empty($bulk_data) || empty($bulk_data['is_bulk_update'])) {
            wp_die(__('No bulk data found.', 'screaming-fixes'));
        }

        $lines = [];
        $lines[] = 'Image,Source URL,Current Alt Text,New Alt Text,Status';

        foreach ($bulk_data['ready_updates'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['image_filename'] ?? basename($row['image_url']),
                $row['source_url'],
                $row['current_alt'] ?? '',
                $row['new_alt'],
                'Ready',
            ]);
        }

        foreach ($bulk_data['not_matched'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['image_filename'] ?? basename($row['image_url']),
                $row['source_url'],
                '',
                $row['new_alt'],
                'Skipped - URL not found',
            ]);
        }

        foreach ($bulk_data['skipped_empty'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                basename($row['image_url'] ?? ''),
                $row['source_url'] ?? '',
                '',
                '',
                'Skipped - No new alt text',
            ]);
        }

        $csv_content = implode("\n", $lines);
        $filename = 'image-alt-text-preview-' . gmdate('Y-m-d-His') . '.csv';

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * AJAX: Download results CSV
     */
    public function ajax_download_bulk_results() {
        check_ajax_referer('sf_image_alt_text_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_upload_data();

        if (empty($bulk_data)) {
            wp_die(__('No results data found.', 'screaming-fixes'));
        }

        $lines = [];
        $lines[] = 'Image,Source URL,New Alt Text,Status';

        foreach ($bulk_data['fixed_images'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['image_filename'] ?? basename($row['image_url']),
                $row['source_url'] ?? '',
                $row['new_alt'],
                'Updated',
            ]);
        }

        foreach ($bulk_data['failed_updates'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['image_filename'] ?? basename($row['image_url']),
                $row['source_url'] ?? '',
                '',
                'Failed - ' . ($row['error'] ?? 'Unknown error'),
            ]);
        }

        foreach ($bulk_data['not_matched'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['image_filename'] ?? basename($row['image_url']),
                $row['source_url'] ?? '',
                $row['new_alt'] ?? '',
                'Skipped - URL not found',
            ]);
        }

        $csv_content = implode("\n", $lines);
        $filename = 'image-alt-text-results-' . gmdate('Y-m-d-His') . '.csv';

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * Escape a row for CSV output
     *
     * @param array $fields Fields to escape
     * @return string Escaped CSV row
     */
    private function csv_escape_row($fields) {
        $escaped = [];
        foreach ($fields as $field) {
            if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
                $field = '"' . str_replace('"', '""', $field) . '"';
            }
            $escaped[] = $field;
        }
        return implode(',', $escaped);
    }
}
