<?php
/**
 * Image Alt Text Tab Content
 *
 * Main view for the Image Alt Text module with categorized sections:
 * - Fixable: Images in post/page content that can be auto-fixed
 * - Manual: Images in navigation, footer, widgets that need manual fixing
 * - Skipped: Dynamic pages (pagination, archives) that will be fixed when source is fixed
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Helper function to render the sources column for images
 */
if (!function_exists('sf_render_image_sources_column')):
function sf_render_image_sources_column($sources, $image_url) {
    $source_count = count($sources);
    $display_limit = 10;
    ob_start();

    if ($source_count === 1):
        // Single source - show directly without toggle
        $source = $sources[0];
        ?>
        <div class="sf-single-source">
            <div class="sf-source-count-label"><?php esc_html_e('Found on 1 page', 'screaming-fixes'); ?></div>
            <?php if (!empty($source['post_id'])): ?>
                <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" rel="noopener" class="sf-source-title">
                    <?php echo esc_html($source['post_title']); ?>
                </a>
            <?php else: ?>
                <span class="sf-source-not-found">
                    <span class="dashicons dashicons-warning"></span>
                    <?php esc_html_e('Post not found', 'screaming-fixes'); ?>
                </span>
            <?php endif; ?>
            <div class="sf-source-url">
                <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" rel="noopener" class="sf-source-url-link">
                    <?php echo esc_html(wp_parse_url($source['source_url'], PHP_URL_PATH) ?: $source['source_url']); ?>
                </a>
            </div>
        </div>
        <?php
    else:
        // Multiple sources - show toggle button with expandable list
        $sources_json = wp_json_encode(array_map(function($s) {
            return [
                'url' => $s['source_url'] ?? '',
                'path' => wp_parse_url($s['source_url'] ?? '', PHP_URL_PATH) ?: ($s['source_url'] ?? ''),
                'post_id' => $s['post_id'] ?? 0,
                'post_title' => $s['post_title'] ?? '',
                'edit_url' => $s['edit_url'] ?? '',
            ];
        }, $sources));
        ?>
        <div class="sf-sources-wrapper">
            <button type="button" class="sf-sources-toggle"
                    data-image-url="<?php echo esc_attr($image_url); ?>"
                    data-sources="<?php echo esc_attr($sources_json); ?>"
                    data-total="<?php echo esc_attr($source_count); ?>">
                <?php printf(
                    esc_html__('Found on %d pages', 'screaming-fixes'),
                    $source_count
                ); ?>
                <span class="dashicons dashicons-arrow-down-alt2"></span>
            </button>
            <div class="sf-sources-list" style="display: none;">
                <ul class="sf-sources-ul">
                    <?php
                    $displayed = 0;
                    foreach ($sources as $source):
                        if ($displayed >= $display_limit) break;
                        $displayed++;
                        $path = wp_parse_url($source['source_url'], PHP_URL_PATH) ?: $source['source_url'];
                    ?>
                        <li class="sf-source-li">
                            <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" rel="noopener" class="sf-source-path-link">
                                <?php echo esc_html($path); ?>
                            </a>
                            <?php if (!empty($source['post_id'])): ?>
                                <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" rel="noopener" class="sf-edit-link" title="<?php esc_attr_e('Edit post', 'screaming-fixes'); ?>">
                                    <span class="dashicons dashicons-edit"></span>
                                </a>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>
        <?php
    endif;

    return ob_get_clean();
}
endif;

// Get module instance and data
$module = SF_Module_Loader::get_module('image-alt-text');
$results = $module ? $module->get_results() : null;

// Check for upload data if no transient results
if (empty($results) || empty($results['images'])) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
    $session_id = 'user_' . get_current_user_id();

    $row = $wpdb->get_row($wpdb->prepare(
        "SELECT data FROM {$table_name}
         WHERE session_id = %s AND module = %s AND expires_at > NOW()
         ORDER BY created_at DESC LIMIT 1",
        $session_id,
        'image-alt-text'
    ));

    if ($row) {
        $stored_data = json_decode($row->data, true);

        // Check if this is raw CSV data (has 'rows' key) vs processed results (has 'images' key)
        if (isset($stored_data['rows']) && !isset($stored_data['images'])) {
            // Raw CSV data - need to process it through the module
            if ($module) {
                // Get the pending upload path and process
                $pending_uploads = get_option('sf_pending_uploads', []);

                // Find the most recent upload for this module
                foreach (array_reverse($pending_uploads) as $upload_id => $upload_info) {
                    if ($upload_info['module'] === 'image-alt-text' && file_exists($upload_info['path'])) {
                        $results = $module->process_csv($upload_info['path']);
                        if (!is_wp_error($results)) {
                            // Update stored data with processed results
                            $wpdb->update(
                                $table_name,
                                ['data' => wp_json_encode($results)],
                                ['session_id' => $session_id, 'module' => 'image-alt-text'],
                                ['%s'],
                                ['%s', '%s']
                            );
                        }
                        break;
                    }
                }
            }
        } else {
            // Already processed data
            $results = $stored_data;
        }
    }
}

// Check if this is a bulk update result
$is_bulk_update = !empty($results['is_bulk_update']);
$bulk_complete = !empty($results['bulk_complete']);

// Bulk update specific data
$bulk_ready_updates = $is_bulk_update ? ($results['ready_updates'] ?? []) : [];
$bulk_not_matched = $is_bulk_update ? ($results['not_matched'] ?? []) : [];
$bulk_skipped_empty = $is_bulk_update ? ($results['skipped_empty'] ?? []) : [];
$bulk_ready_count = $is_bulk_update ? ($results['ready_count'] ?? count($bulk_ready_updates)) : 0;
$bulk_not_matched_count = $is_bulk_update ? ($results['not_matched_count'] ?? count($bulk_not_matched)) : 0;
$bulk_skipped_empty_count = $is_bulk_update ? ($results['skipped_empty_count'] ?? count($bulk_skipped_empty)) : 0;

// Bulk complete results
$bulk_fixed_images = $bulk_complete ? ($results['fixed_images'] ?? []) : [];
$bulk_failed_updates = $bulk_complete ? ($results['failed_updates'] ?? []) : [];
$bulk_success_count = $bulk_complete ? ($results['success_count'] ?? 0) : 0;
$bulk_failed_count = $bulk_complete ? ($results['failed_count'] ?? 0) : 0;

// Check if we have valid bulk data
$has_valid_bulk_data = false;
if ($is_bulk_update) {
    if (!$bulk_complete && !empty($results['ready_updates']) && count($results['ready_updates']) > 0) {
        $has_valid_bulk_data = true;
    }
    $fixed_count_check = isset($results['fixed_images']) ? count($results['fixed_images']) : 0;
    $failed_count_check = isset($results['failed_updates']) ? count($results['failed_updates']) : 0;
    if ($bulk_complete && ($fixed_count_check > 0 || $failed_count_check > 0)) {
        $has_valid_bulk_data = true;
    }

    if (!$has_valid_bulk_data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();
        $wpdb->delete($table_name, ['session_id' => $session_id, 'module' => 'image-alt-text']);
        delete_transient('sf_image-alt-text_results');
        $results = null;
        $is_bulk_update = false;
        $bulk_complete = false;
    }
}

// Check if we have any data (images OR fixed_images - data persists after fixes)
$has_results = !empty($results) && is_array($results);
$has_images = $has_results && !empty($results['images']);
$has_fixed_images = $has_results && !empty($results['fixed_images']) && !$is_bulk_update;
$has_data = $has_images || $has_fixed_images || $has_valid_bulk_data;

// Get categorized images
$fixable_images = $has_results ? ($results['fixable_images'] ?? []) : [];
$manual_images = $has_results ? ($results['manual_images'] ?? []) : [];
$skipped_images = $has_results ? ($results['skipped_images'] ?? []) : [];
$fixed_images = $has_results && !$is_bulk_update ? ($results['fixed_images'] ?? []) : [];

// Get counts
$total_count = $has_results ? ($results['total_count'] ?? 0) : 0;
$fixable_count = $has_results ? ($results['fixable_count'] ?? count($fixable_images)) : 0;
$manual_count = $has_results ? ($results['manual_count'] ?? count($manual_images)) : 0;
$skipped_count = $has_results ? ($results['skipped_count'] ?? count($skipped_images)) : 0;
$fixed_images_count = count($fixed_images);
?>

<div class="sf-image-alt-text-module">
    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#128444;</span>
            <h2><?php esc_html_e('Image Alt Text', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Generate AI-powered alt text for missing images', 'screaming-fixes'); ?></span>
        </div>
        <?php if ($has_data): ?>
            <div class="sf-module-stats">
                <span class="sf-stat sf-stat-total">
                    <strong><?php echo esc_html($total_count); ?></strong>
                    <?php esc_html_e('images', 'screaming-fixes'); ?>
                </span>
            </div>
        <?php endif; ?>
    </div>

    <!-- Instructions Section (Collapsible) - At top like Redirect Chains -->
    <?php include SF_PLUGIN_DIR . 'modules/image-alt-text/views/instructions.php'; ?>

    <?php if ($is_bulk_update && !$bulk_complete && $has_valid_bulk_data): ?>
        <!-- ==========================================
             BULK UPDATE CONFIRMATION UI
             ========================================== -->
        <div class="sf-bulk-confirmation" id="sf-bulk-confirmation">
            <div class="sf-bulk-summary">
                <div class="sf-bulk-icon">&#128203;</div>
                <h3><?php esc_html_e('Bulk Alt Text Update', 'screaming-fixes'); ?></h3>

                <div class="sf-bulk-stats">
                    <div class="sf-bulk-stat sf-bulk-stat-ready">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_ready_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('images ready to update', 'screaming-fixes'); ?></span>
                    </div>

                    <?php if ($bulk_not_matched_count > 0): ?>
                    <div class="sf-bulk-stat sf-bulk-stat-warning">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_not_matched_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('URLs not matched', 'screaming-fixes'); ?></span>
                    </div>
                    <?php endif; ?>

                    <?php if ($bulk_skipped_empty_count > 0): ?>
                    <div class="sf-bulk-stat sf-bulk-stat-skipped">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_skipped_empty_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('rows skipped - no new alt text provided', 'screaming-fixes'); ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <?php if ($bulk_skipped_empty_count > 0): ?>
                <div class="sf-bulk-notice">
                    <span class="dashicons dashicons-info"></span>
                    <?php esc_html_e("Rows with empty alt text values will be skipped.", 'screaming-fixes'); ?>
                </div>
                <?php endif; ?>

                <div class="sf-bulk-preview-tip" style="margin-bottom: 14px; padding: 10px 14px; background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; font-size: 13px; line-height: 1.5; color: #334155;">
                    <span class="dashicons dashicons-info-outline" style="color: #22c55e; font-size: 16px; width: 16px; height: 16px; margin-right: 4px; vertical-align: text-bottom;"></span>
                    <?php printf(
                        esc_html__('Use %1$sDownload Preview CSV%2$s to review the full list of Ready (%3$d) and Can\'t Update (%4$d) images before confirming.', 'screaming-fixes'),
                        '<strong>', '</strong>',
                        $bulk_ready_count,
                        $bulk_not_matched_count
                    ); ?>
                </div>

                <div class="sf-bulk-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-download-preview" id="sf-bulk-download-preview">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
                    </button>
                    <div class="sf-bulk-buttons">
                        <button type="button" class="sf-button sf-button-secondary sf-bulk-clear-btn" id="sf-bulk-clear">
                            <span class="dashicons dashicons-trash"></span>
                            <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                        </button>
                        <button type="button" class="sf-button sf-button-primary sf-bulk-confirm" id="sf-bulk-confirm" data-count="<?php echo esc_attr($bulk_ready_count); ?>">
                            <?php esc_html_e('Confirm Update', 'screaming-fixes'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Preview Table -->
            <div class="sf-bulk-preview">
                <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, $bulk_ready_count)); ?> of <?php echo esc_html($bulk_ready_count); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Current Alt', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('New Alt', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $preview_items = array_slice($bulk_ready_updates, 0, 10);
                            foreach ($preview_items as $item):
                                $path = wp_parse_url($item['source_url'], PHP_URL_PATH) ?: $item['source_url'];
                                $current_truncated = strlen($item['current_alt']) > 50 ? substr($item['current_alt'], 0, 50) . '...' : $item['current_alt'];
                                $new_truncated = strlen($item['new_alt']) > 50 ? substr($item['new_alt'], 0, 50) . '...' : $item['new_alt'];
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-ready">
                                <td class="sf-col-image" title="<?php echo esc_attr($item['image_url']); ?>">
                                    <?php echo esc_html($item['image_filename']); ?>
                                </td>
                                <td class="sf-col-url">
                                    <a href="<?php echo esc_url($item['source_url']); ?>" target="_blank" title="<?php echo esc_attr($item['source_url']); ?>">
                                        <?php echo esc_html($path); ?>
                                    </a>
                                </td>
                                <td class="sf-col-current" title="<?php echo esc_attr($item['current_alt']); ?>">
                                    <?php echo esc_html($current_truncated ?: '(empty)'); ?>
                                </td>
                                <td class="sf-col-new" title="<?php echo esc_attr($item['new_alt']); ?>">
                                    <?php echo esc_html($new_truncated); ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <?php if ($bulk_not_matched_count > 0): ?>
                <h4 class="sf-mt-20"><?php esc_html_e('Can\'t Update Image Alt Text / Unmatched URLs', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html($bulk_not_matched_count); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Source URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $unmatched_preview = array_slice($bulk_not_matched, 0, 5);
                            foreach ($unmatched_preview as $item):
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-skipped">
                                <td class="sf-col-image"><?php echo esc_html($item['image_filename'] ?? basename($item['image_url'])); ?></td>
                                <td class="sf-col-url"><?php echo esc_html($item['source_url']); ?></td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-skipped"><?php esc_html_e('URL not found', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if ($bulk_not_matched_count > 5): ?>
                            <tr class="sf-bulk-row-more">
                                <td colspan="3"><?php printf(esc_html__('...and %d more unmatched URLs', 'screaming-fixes'), $bulk_not_matched_count - 5); ?></td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="sf-bulk-unmatched-info" style="margin-top: 12px; padding: 12px 16px; background: #f0fdfa; border: 1px solid #99f6e4; border-radius: 6px; font-size: 13px; line-height: 1.5; color: #334155;">
                    <strong><?php esc_html_e('Why can\'t these images be updated?', 'screaming-fixes'); ?></strong>
                    <ul style="margin: 6px 0 0 18px; padding: 0;">
                        <li><?php esc_html_e('The image URL was not found in the WordPress Media Library — it may be hosted externally or have been deleted.', 'screaming-fixes'); ?></li>
                        <li><?php esc_html_e('The source URL where the image appears was not found on this WordPress site (e.g., it belongs to a different domain or the page has been deleted).', 'screaming-fixes'); ?></li>
                    </ul>
                    <p style="margin: 8px 0 0;"><?php esc_html_e('To update alt text for these images, upload them to your Media Library or edit the alt text directly in the Media Library or page editor.', 'screaming-fixes'); ?></p>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Bulk Progress Modal (hidden by default) -->
        <div class="sf-bulk-progress-modal sf-modal" id="sf-bulk-progress-modal" style="display: none;">
            <div class="sf-modal-content sf-bulk-progress-content">
                <div class="sf-bulk-progress-header">
                    <span class="sf-bulk-progress-icon sf-spinning">&#8634;</span>
                    <h3 class="sf-bulk-progress-title"><?php esc_html_e('Updating Alt Text...', 'screaming-fixes'); ?></h3>
                    <p class="sf-bulk-progress-subtitle"><?php esc_html_e('Please do not close this window.', 'screaming-fixes'); ?></p>
                </div>
                <div class="sf-bulk-progress-bar-wrapper">
                    <div class="sf-bulk-progress-bar">
                        <div class="sf-progress-fill" style="width: 0%;"></div>
                    </div>
                    <div class="sf-bulk-progress-stats">
                        <strong><span class="sf-bulk-progress-current">0</span></strong> of <span class="sf-bulk-progress-total">0</span> complete (<span class="sf-bulk-progress-percent">0</span>%)
                    </div>
                </div>
                <div class="sf-bulk-current-url-wrapper">
                    <div class="sf-bulk-current-url-label"><?php esc_html_e('Currently processing:', 'screaming-fixes'); ?></div>
                    <div class="sf-bulk-current-url"><?php esc_html_e('Starting...', 'screaming-fixes'); ?></div>
                </div>
            </div>
        </div>

    <?php elseif ($bulk_complete && $has_valid_bulk_data): ?>
        <!-- ==========================================
             BULK UPDATE COMPLETE RESULTS
             ========================================== -->
        <div class="sf-bulk-complete" id="sf-bulk-complete">
            <div class="sf-bulk-complete-summary <?php echo $bulk_failed_count > 0 ? 'sf-has-failures' : ''; ?>">
                <div class="sf-bulk-complete-icon">
                    <?php echo $bulk_failed_count > 0 ? '&#9888;' : '&#10004;'; ?>
                </div>
                <h3>
                    <?php
                    if ($bulk_failed_count > 0) {
                        printf(
                            esc_html__('%d images updated successfully. %d failed - see details below.', 'screaming-fixes'),
                            $bulk_success_count,
                            $bulk_failed_count
                        );
                    } else {
                        printf(
                            esc_html__('%d images updated successfully.', 'screaming-fixes'),
                            $bulk_success_count
                        );
                    }
                    ?>
                    <?php if ($bulk_not_matched_count > 0): ?>
                        <span class="sf-bulk-skipped-note">
                            <?php printf(esc_html__('%d URLs skipped.', 'screaming-fixes'), $bulk_not_matched_count + $bulk_skipped_empty_count); ?>
                        </span>
                    <?php endif; ?>
                </h3>

                <div class="sf-bulk-complete-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-download-results" id="sf-bulk-download-results">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-primary sf-bulk-new-upload" id="sf-bulk-new-upload">
                        <?php esc_html_e('Upload New CSV', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-complete-clear" id="sf-bulk-complete-clear">
                        <span class="dashicons dashicons-trash"></span>
                        <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <!-- Results Table -->
            <?php if (!empty($bulk_fixed_images)): ?>
            <div class="sf-bulk-results-table">
                <h4><?php esc_html_e('Updated Images', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, count($bulk_fixed_images))); ?> of <?php echo esc_html(count($bulk_fixed_images)); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('New Alt Text', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $results_preview = array_slice($bulk_fixed_images, 0, 10);
                            foreach ($results_preview as $item):
                                $path = wp_parse_url($item['source_url'] ?? '', PHP_URL_PATH) ?: ($item['source_url'] ?? '');
                                $new_truncated = strlen($item['new_alt']) > 75 ? substr($item['new_alt'], 0, 75) . '...' : $item['new_alt'];
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-success">
                                <td class="sf-col-image" title="<?php echo esc_attr($item['image_url'] ?? ''); ?>">
                                    <?php echo esc_html($item['image_filename'] ?? basename($item['image_url'] ?? '')); ?>
                                </td>
                                <td class="sf-col-url">
                                    <a href="<?php echo esc_url($item['source_url'] ?? ''); ?>" target="_blank">
                                        <?php echo esc_html($path); ?>
                                    </a>
                                </td>
                                <td class="sf-col-new" title="<?php echo esc_attr($item['new_alt']); ?>"><?php echo esc_html($new_truncated); ?></td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-success"><?php esc_html_e('Updated', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Failed Updates -->
            <?php if (!empty($bulk_failed_updates)): ?>
            <div class="sf-bulk-failed-table">
                <h4 class="sf-mt-20"><?php esc_html_e('Failed Updates', 'screaming-fixes'); ?></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Error', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($bulk_failed_updates as $item): ?>
                            <tr class="sf-bulk-row sf-bulk-row-failed">
                                <td class="sf-col-image"><?php echo esc_html($item['image_filename'] ?? basename($item['image_url'] ?? '')); ?></td>
                                <td class="sf-col-error"><?php echo esc_html($item['error']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
        </div>

    <?php elseif ($has_data && !$is_bulk_update): ?>
        <!-- Stats Breakdown - Matching Broken Links Style -->
        <div class="sf-stats-breakdown">
            <div class="sf-stat-card sf-stat-fixable">
                <span class="sf-stat-icon">&#10003;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($fixable_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Fixable', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-stat-card sf-stat-manual">
                <span class="sf-stat-icon">&#9888;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($manual_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Manual Fix', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-stat-card sf-stat-skipped">
                <span class="sf-stat-icon">&#8856;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($skipped_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                </div>
            </div>
        </div>

        <!-- Filter Row - Status Filters -->
        <div class="sf-issue-stats-row">
            <span class="sf-filter-label"><?php esc_html_e('Filter By:', 'screaming-fixes'); ?></span>
            <div class="sf-issue-stats">
                <div class="sf-issue-stat sf-issue-all active" data-filter="all">
                    <span class="sf-issue-icon">&#128203;</span>
                    <span class="sf-issue-count sf-count-all"><?php echo esc_html($fixable_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-pending" data-filter="pending">
                    <span class="sf-issue-icon">&#9898;</span>
                    <span class="sf-issue-count sf-count-pending"><?php echo esc_html($fixable_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Pending', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-fixed sf-filter-disabled" data-filter="fixed">
                    <span class="sf-issue-icon">&#9989;</span>
                    <span class="sf-issue-count sf-count-fixed">0</span>
                    <span class="sf-issue-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-failed sf-filter-disabled" data-filter="failed">
                    <span class="sf-issue-icon">&#10060;</span>
                    <span class="sf-issue-count sf-count-failed">0</span>
                    <span class="sf-issue-label"><?php esc_html_e("Can't Fix", 'screaming-fixes'); ?></span>
                </div>
            </div>
            <button type="button" class="sf-button sf-button-link sf-clear-data-btn" title="<?php esc_attr_e('Clear data and upload new CSV', 'screaming-fixes'); ?>">
                <span class="dashicons dashicons-trash"></span>
                <?php esc_html_e('Clear Data', 'screaming-fixes'); ?>
            </button>
            <?php if (!empty($fixed_images) || !empty($manual_images)): ?>
            <button type="button" class="sf-button sf-button-secondary sf-export-results-btn" style="display: none;" title="<?php esc_attr_e('Export fixed and can\'t fix items', 'screaming-fixes'); ?>">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Export Results', 'screaming-fixes'); ?>
            </button>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <?php if (!$has_data): ?>
    <!-- Upload Section - Prominent when no data -->
    <div class="sf-upload-section">
        <h3>
            <span class="dashicons dashicons-upload"></span>
            <?php esc_html_e('Upload Images CSV', 'screaming-fixes'); ?>
        </h3>
        <?php
        $args = [
            'module' => 'image-alt-text',
            'title' => __('Drop your Screaming Frog Images CSV file', 'screaming-fixes'),
            'description' => __('Drag & drop CSV here or click to browse', 'screaming-fixes'),
            'accepted_types' => '.csv',
            'max_rows' => 5000,
        ];
        include SF_PLUGIN_DIR . 'admin/views/partials/upload-box.php';
        ?>
    </div>
    <?php endif; ?>

    <?php if ($has_data && !$is_bulk_update): ?>

        <?php if (!empty($fixable_images)): ?>
        <!-- Section 1: Fixable Images -->
        <div class="sf-results-section sf-section-fixable">
            <div class="sf-results-header sf-results-header-collapsible">
                <button type="button" class="sf-section-toggle sf-fixable-toggle" aria-expanded="true">
                    <span class="sf-section-badge sf-badge-fixable">&#10003;</span>
                    <?php printf(
                        esc_html__('%d Fixable Images', 'screaming-fixes'),
                        $fixable_count
                    ); ?>
                    <span class="sf-section-hint"><?php esc_html_e('Images in post/page content - can be auto-fixed', 'screaming-fixes'); ?></span>
                    <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
                </button>
                <div class="sf-results-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-export-btn">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <div class="sf-fixable-content">

            <!-- Fixable Results Table -->
            <div class="sf-table-wrapper sf-paginated-table" data-section="fixable" data-per-page="100" data-total="<?php echo esc_attr($fixable_count); ?>">
                <table class="sf-results-table" id="sf-image-alt-text-table">
                    <thead>
                        <tr>
                            <th class="sf-col-checkbox">
                                <input type="checkbox" id="sf-select-all" title="<?php esc_attr_e('Select all', 'screaming-fixes'); ?>">
                            </th>
                            <th class="sf-col-image"><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                            <th class="sf-col-sources"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                            <th class="sf-col-alt-text"><?php esc_html_e('Alt Text', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fixable_images as $index => $image):
                            $image_url = $image['image_url'] ?? '';
                            $filename = $image['filename'] ?? basename($image_url);
                            $sources = $image['fixable_sources'] ?? $image['sources'] ?? [];
                            $source_count = count($sources);

                            // Collect post IDs for fixable sources
                            $post_ids = array_filter(array_column($sources, 'post_id'));

                            // Collect source titles for AI context
                            $source_titles = array_filter(array_column($sources, 'post_title'));
                        ?>
                            <tr class="sf-image-row"
                                data-image-url="<?php echo esc_attr($image_url); ?>"
                                data-filename="<?php echo esc_attr($filename); ?>"
                                data-index="<?php echo esc_attr($index); ?>"
                                data-category="fixable"
                                data-post-ids="<?php echo esc_attr(implode(',', $post_ids)); ?>"
                                data-source-titles="<?php echo esc_attr(implode('|', $source_titles)); ?>">
                                <td class="sf-col-checkbox">
                                    <input type="checkbox" class="sf-row-select" value="<?php echo esc_attr($index); ?>">
                                </td>
                                <td class="sf-col-image">
                                    <div class="sf-image-cell">
                                        <div class="sf-image-preview">
                                            <img src="<?php echo esc_url($image_url); ?>" alt="" loading="lazy" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                            <div class="sf-image-placeholder" style="display: none;">
                                                <span class="dashicons dashicons-format-image"></span>
                                            </div>
                                        </div>
                                        <div class="sf-image-info">
                                            <span class="sf-image-filename" title="<?php echo esc_attr($image_url); ?>">
                                                <?php echo esc_html($filename); ?>
                                            </span>
                                            <a href="<?php echo esc_url($image_url); ?>" target="_blank" rel="noopener" class="sf-image-url">
                                                <?php esc_html_e('View full size', 'screaming-fixes'); ?>
                                                <span class="dashicons dashicons-external"></span>
                                            </a>
                                        </div>
                                    </div>
                                </td>
                                <td class="sf-col-sources">
                                    <?php echo sf_render_image_sources_column($sources, $image_url); ?>
                                </td>
                                <td class="sf-col-alt-text">
                                    <div class="sf-alt-text-controls">
                                        <div class="sf-alt-text-input-wrapper">
                                            <input type="text" class="sf-alt-text-input" placeholder="<?php esc_attr_e('Enter alt text...', 'screaming-fixes'); ?>" data-image-url="<?php echo esc_attr($image_url); ?>">
                                        </div>
                                        <div class="sf-alt-text-actions">
                                            <button type="button" class="sf-ai-btn sf-ai-suggest-btn" title="<?php esc_attr_e('Get AI suggestion', 'screaming-fixes'); ?>" data-image-url="<?php echo esc_attr($image_url); ?>">
                                                <span class="dashicons dashicons-welcome-learn-more"></span>
                                                <?php esc_html_e('AI Suggest', 'screaming-fixes'); ?>
                                            </button>
                                            <button type="button" class="sf-ignore-btn sf-ignore-image" title="<?php esc_attr_e('Ignore this image', 'screaming-fixes'); ?>">
                                                <span class="dashicons dashicons-hidden"></span>
                                                <?php esc_html_e('Ignore', 'screaming-fixes'); ?>
                                            </button>
                                            <button type="button" class="sf-save-btn sf-save-alt-btn" disabled title="<?php esc_attr_e('Save alt text', 'screaming-fixes'); ?>" data-image-url="<?php echo esc_attr($image_url); ?>">
                                                <span class="dashicons dashicons-saved"></span>
                                                <?php esc_html_e('Save', 'screaming-fixes'); ?>
                                            </button>
                                        </div>
                                        <div class="sf-fix-status">
                                            <span class="sf-status-icon"></span>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($fixable_count > 100): ?>
            <div class="sf-pagination" data-section="fixable">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $fixable_count); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($fixable_count); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" disabled>
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-btn sf-page-next">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
            </div>
            <?php endif; ?>
            </div><!-- /.sf-fixable-content -->

            <!-- Batch Actions - Outside collapsible content so it stays visible -->
            <div class="sf-batch-actions">
                <div class="sf-selected-count">
                    <span class="sf-count">0</span> <?php esc_html_e('selected', 'screaming-fixes'); ?>
                </div>
                <div class="sf-batch-buttons">
                    <button type="button" class="sf-button sf-button-primary sf-apply-fixes" disabled>
                        <span class="dashicons dashicons-yes"></span>
                        <?php esc_html_e('Apply Alt Text', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($manual_images)):
            // Calculate total manual sources for display
            $total_manual_sources = $has_data ? ($results['manual_sources'] ?? 0) : 0;
            if ($total_manual_sources === 0) {
                foreach ($manual_images as $image) {
                    $total_manual_sources += count($image['manual_sources'] ?? $image['sources'] ?? []);
                }
            }
        ?>
        <!-- Section 2: Manual Fix Required (Collapsed by default) -->
        <div class="sf-results-section sf-section-manual">
            <button type="button" class="sf-section-toggle sf-manual-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-manual">&#9888;</span>
                <?php printf(
                    esc_html__('%d Manual Fix Required', 'screaming-fixes'),
                    $manual_count
                ); ?>
                <?php if ($total_manual_sources > $manual_count): ?>
                    <span class="sf-section-hint"><?php printf(
                        esc_html__('Found on %d pages - fix in navigation, footer, widgets', 'screaming-fixes'),
                        $total_manual_sources
                    ); ?></span>
                <?php else: ?>
                    <span class="sf-section-hint"><?php esc_html_e('Images in navigation, footer, widgets - fix manually', 'screaming-fixes'); ?></span>
                <?php endif; ?>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-manual-content" style="display: none;">
            <!-- Manual Results Table -->
            <div class="sf-table-wrapper sf-paginated-table" data-section="manual" data-per-page="100" data-total="<?php echo esc_attr($manual_count); ?>">
                <table class="sf-results-table sf-manual-table">
                    <thead>
                        <tr>
                            <th class="sf-col-image"><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                            <th class="sf-col-sources"><?php esc_html_e('Found On', 'screaming-fixes'); ?></th>
                            <th class="sf-col-howto"><?php esc_html_e('How to Fix', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($manual_images as $image):
                            $image_url = $image['image_url'] ?? '';
                            $filename = $image['filename'] ?? basename($image_url);
                            $sources = $image['manual_sources'] ?? $image['sources'] ?? [];

                            // Get the primary fix note (from first source)
                            $primary_note = '';
                            $primary_location = 'unknown';
                            if (!empty($sources[0])) {
                                $primary_note = $sources[0]['fix_note'] ?? '';
                                $primary_location = $sources[0]['location'] ?? 'unknown';
                            }

                            $location_icons = [
                                'navigation' => '&#9776;',
                                'footer' => '&#9604;',
                                'sidebar' => '&#9612;',
                                'header' => '&#9600;',
                                'homepage' => '&#8962;',
                                'unknown' => '&#63;',
                            ];
                            $icon = $location_icons[$primary_location] ?? '&#63;';
                        ?>
                            <tr class="sf-image-row sf-manual-row"
                                data-image-url="<?php echo esc_attr($image_url); ?>"
                                data-category="manual">
                                <td class="sf-col-image">
                                    <div class="sf-image-cell">
                                        <div class="sf-image-preview sf-image-preview-small">
                                            <img src="<?php echo esc_url($image_url); ?>" alt="" loading="lazy" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                            <div class="sf-image-placeholder" style="display: none;">
                                                <span class="dashicons dashicons-format-image"></span>
                                            </div>
                                        </div>
                                        <div class="sf-image-info">
                                            <span class="sf-image-filename" title="<?php echo esc_attr($image_url); ?>">
                                                <?php echo esc_html($filename); ?>
                                            </span>
                                            <a href="<?php echo esc_url($image_url); ?>" target="_blank" rel="noopener" class="sf-image-url">
                                                <?php esc_html_e('View full size', 'screaming-fixes'); ?>
                                                <span class="dashicons dashicons-external"></span>
                                            </a>
                                        </div>
                                    </div>
                                </td>
                                <td class="sf-col-sources">
                                    <?php echo sf_render_image_sources_column($sources, $image_url); ?>
                                </td>
                                <td class="sf-col-howto">
                                    <div class="sf-fix-howto">
                                        <span class="sf-location-icon"><?php echo $icon; ?></span>
                                        <span class="sf-fix-note"><?php echo esc_html($primary_note); ?></span>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php if ($manual_count > 100): ?>
            <div class="sf-pagination" data-section="manual">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $manual_count); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($manual_count); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" disabled>
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-btn sf-page-next">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
            </div>
            <?php endif; ?>
            </div><!-- /.sf-manual-content -->
        </div>
        <?php endif; ?>

        <?php if (!empty($skipped_images)): ?>
        <!-- Section 3: Skipped (Collapsed by default) -->
        <div class="sf-results-section sf-section-skipped">
            <button type="button" class="sf-section-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-skipped">&#8856;</span>
                <?php printf(
                    esc_html__('%d Skipped (Dynamic Pages)', 'screaming-fixes'),
                    $skipped_count
                ); ?>
                <span class="sf-section-hint"><?php esc_html_e('Will be fixed when source content is fixed', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-skipped-content" style="display: none;">
                <div class="sf-skipped-note">
                    <span class="dashicons dashicons-info"></span>
                    <?php esc_html_e('These images appear on dynamic pages (pagination, archives, etc.) that pull content from posts. They will be automatically fixed when you fix the images in the source posts.', 'screaming-fixes'); ?>
                </div>
                <div class="sf-skipped-links-list sf-paginated-table" data-section="skipped" data-per-page="100" data-total="<?php echo esc_attr($skipped_count); ?>">
                    <?php foreach ($skipped_images as $image):
                        $image_url = $image['image_url'] ?? '';
                        $filename = $image['filename'] ?? basename($image_url);
                        $sources = $image['skip_sources'] ?? $image['sources'] ?? [];
                    ?>
                        <div class="sf-skipped-link-item">
                            <div class="sf-skipped-image-preview">
                                <img src="<?php echo esc_url($image_url); ?>" alt="" loading="lazy" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="sf-image-placeholder" style="display: none;">
                                    <span class="dashicons dashicons-format-image"></span>
                                </div>
                            </div>
                            <div class="sf-skipped-link-info">
                                <div class="sf-skipped-link-url">
                                    <span class="sf-image-filename" title="<?php echo esc_attr($image_url); ?>"><?php echo esc_html($filename); ?></span>
                                    <a href="<?php echo esc_url($image_url); ?>" target="_blank" rel="noopener" class="sf-image-url">
                                        <?php esc_html_e('View full size', 'screaming-fixes'); ?>
                                        <span class="dashicons dashicons-external"></span>
                                    </a>
                                </div>
                                <div class="sf-skipped-sources">
                                    <?php foreach ($sources as $source):
                                        // Get short location label
                                        $location = $source['location'] ?? '';
                                        $note = $source['fix_note'] ?? '';
                                        $label = '';
                                        switch ($location) {
                                            case 'pagination': $label = __('Pagination', 'screaming-fixes'); break;
                                            case 'category': $label = __('Category archive', 'screaming-fixes'); break;
                                            case 'author': $label = __('Author archive', 'screaming-fixes'); break;
                                            case 'tag': $label = __('Tag archive', 'screaming-fixes'); break;
                                            default: $label = $note ?: $location ?: __('Dynamic page', 'screaming-fixes');
                                        }
                                    ?>
                                        <span class="sf-skipped-source" title="<?php echo esc_attr($source['source_url'] ?? ''); ?>">
                                            <?php echo esc_html($label); ?>
                                        </span>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <?php if ($skipped_count > 100): ?>
                <div class="sf-pagination" data-section="skipped">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $skipped_count); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($skipped_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-btn sf-page-prev" disabled>
                            <span class="dashicons dashicons-arrow-left-alt2"></span>
                        </button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-btn sf-page-next">
                            <span class="dashicons dashicons-arrow-right-alt2"></span>
                        </button>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($fixed_images)): ?>
        <!-- Section 4: Fixed Images (Collapsed by default) -->
        <div class="sf-results-section sf-section-fixed">
            <div class="sf-results-header sf-results-header-collapsible sf-fixed-header">
                <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="false">
                    <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
                    <?php printf(
                        esc_html(_n('Fixed Images (%d)', 'Fixed Images (%d)', $fixed_images_count, 'screaming-fixes')),
                        $fixed_images_count
                    ); ?>
                    <span class="sf-section-hint"><?php esc_html_e('Alt text has been applied to these images', 'screaming-fixes'); ?></span>
                    <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
                </button>
                <div class="sf-results-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-export-fixed-btn">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <div class="sf-fixed-content" style="display: none;">
                <div class="sf-fixed-note">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <?php esc_html_e('These images have had alt text applied. You can undo recent changes from the Settings page.', 'screaming-fixes'); ?>
                </div>

                <!-- Fixed Images Table -->
                <div class="sf-table-wrapper sf-paginated-table" data-section="fixed" data-per-page="100" data-total="<?php echo esc_attr($fixed_images_count); ?>">
                    <table class="sf-results-table sf-fixed-table">
                        <thead>
                            <tr>
                                <th class="sf-col-image"><?php esc_html_e('Image', 'screaming-fixes'); ?></th>
                                <th class="sf-col-sources"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                                <th class="sf-col-new-alt"><?php esc_html_e('New Alt Text', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($fixed_images as $image):
                                $image_url = $image['image_url'] ?? '';
                                $filename = $image['filename'] ?? basename($image_url);
                                $applied_alt = $image['applied_alt'] ?? '';
                                $fixed_at = $image['fixed_at'] ?? '';
                                $sources = $image['fixable_sources'] ?? $image['sources'] ?? [];
                            ?>
                                <tr class="sf-image-row sf-fixed-row"
                                    data-image-url="<?php echo esc_attr($image_url); ?>"
                                    data-applied-alt="<?php echo esc_attr($applied_alt); ?>">
                                    <td class="sf-col-image">
                                        <div class="sf-image-cell">
                                            <div class="sf-image-preview">
                                                <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($applied_alt); ?>" loading="lazy" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                                <div class="sf-image-placeholder" style="display: none;">
                                                    <span class="dashicons dashicons-format-image"></span>
                                                </div>
                                            </div>
                                            <div class="sf-image-info">
                                                <span class="sf-image-filename" title="<?php echo esc_attr($image_url); ?>">
                                                    <?php echo esc_html($filename); ?>
                                                </span>
                                                <a href="<?php echo esc_url($image_url); ?>" target="_blank" rel="noopener" class="sf-image-url">
                                                    <?php esc_html_e('View full size', 'screaming-fixes'); ?>
                                                    <span class="dashicons dashicons-external"></span>
                                                </a>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="sf-col-sources">
                                        <?php echo sf_render_image_sources_column($sources, $image_url); ?>
                                    </td>
                                    <td class="sf-col-new-alt">
                                        <div class="sf-applied-alt-wrapper">
                                            <span class="sf-applied-alt-text"><?php echo esc_html($applied_alt); ?></span>
                                            <?php if (!empty($fixed_at)): ?>
                                                <span class="sf-fixed-time"><?php echo esc_html(human_time_diff(strtotime($fixed_at), current_time('timestamp')) . ' ' . __('ago', 'screaming-fixes')); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($fixed_images_count > 100): ?>
                <div class="sf-pagination" data-section="fixed">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $fixed_images_count); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($fixed_images_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-btn sf-page-prev" disabled>
                            <span class="dashicons dashicons-arrow-left-alt2"></span>
                        </button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-btn sf-page-next">
                            <span class="dashicons dashicons-arrow-right-alt2"></span>
                        </button>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Progress Modal -->
        <div class="sf-modal sf-progress-modal" style="display: none;">
            <div class="sf-modal-content">
                <div class="sf-progress-header">
                    <span class="sf-progress-icon dashicons dashicons-update sf-spinning"></span>
                    <h3 class="sf-progress-title"><?php esc_html_e('Applying Alt Text...', 'screaming-fixes'); ?></h3>
                </div>
                <div class="sf-progress-bar">
                    <div class="sf-progress-fill" style="width: 0%;"></div>
                </div>
                <div class="sf-progress-status">
                    <span class="sf-progress-current">0</span> / <span class="sf-progress-total">0</span>
                    <?php esc_html_e('images updated', 'screaming-fixes'); ?>
                </div>
            </div>
        </div>

        <!-- Sources Modal (for "Show all X sources") -->
        <div class="sf-modal sf-sources-modal" style="display: none;">
            <div class="sf-modal-content sf-sources-modal-content">
                <div class="sf-modal-header">
                    <h3 class="sf-modal-title">
                        <span class="sf-sources-count">0</span>
                        <?php esc_html_e('Pages Containing This Image', 'screaming-fixes'); ?>
                    </h3>
                    <button type="button" class="sf-modal-close" aria-label="<?php esc_attr_e('Close', 'screaming-fixes'); ?>">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                <div class="sf-modal-body">
                    <div class="sf-modal-image-preview"></div>
                    <div class="sf-sources-full-list">
                        <ul class="sf-sources-full-ul"></ul>
                    </div>
                </div>
                <div class="sf-modal-footer">
                    <button type="button" class="sf-button sf-button-secondary sf-copy-sources">
                        <span class="dashicons dashicons-clipboard"></span>
                        <?php esc_html_e('Copy All URLs', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>
        </div>

    <?php else: ?>
        <!-- Empty State -->
        <div class="sf-empty-state">
            <div class="sf-empty-icon">&#128444;</div>
            <h3><?php esc_html_e('No image alt text data yet', 'screaming-fixes'); ?></h3>
            <p><?php esc_html_e('Upload a Screaming Frog Images CSV to find and fix images missing alt text.', 'screaming-fixes'); ?></p>
        </div>
    <?php endif; ?>
</div>
