/**
 * Screaming Fixes - Internal Link Builder Module
 *
 * Handles Discovery Mode and Bulk Upload Mode for adding internal links
 */

(function($) {
    'use strict';

    window.ScreamingFixes = window.ScreamingFixes || {};

    /**
     * Internal Link Builder Handler
     */
    ScreamingFixes.InternalLinkBuilder = {
        // State
        processing: false,
        discoveryData: null,
        bulkData: null,

        // Fun loading messages for scanning opportunities
        scanLoadingMessages: [
            "🔍 Finding internal link options...",
            "🔗 Researching the best link options...",
            "⚖️ Dividing link equity...",
            "🎯 Considering relevance...",
            "🧮 Subtracting and then adding again...",
            "📊 Analyzing content connections...",
            "🕸️ Weaving the link web...",
            "🔬 Examining keyword matches...",
            "📝 Scanning page content...",
            "🎪 Juggling anchor texts...",
            "🏗️ Building link pathways...",
            "🧭 Mapping site structure...",
            "✨ Almost there...",
            "☕ This is taking a moment...",
            "🚀 Finalizing opportunities..."
        ],

        // Fun loading messages for bulk CSV processing
        bulkCSVLoadingMessages: [
            "📄 Evaluating bulk internal link fixes...",
            "🎯 Targeting link equity...",
            "🔥 Firing up the updates...",
            "🤔 Contemplating references...",
            "🏗️ Building authority...",
            "🌐 Contemplating domain authority...",
            "🔗 Matching source URLs...",
            "📊 Analyzing anchor text data...",
            "🔍 Validating target pages...",
            "⚡ Processing link opportunities...",
            "🧮 Calculating link relevance...",
            "🎪 Juggling keyword matches...",
            "✨ Almost there...",
            "☕ This is taking a moment...",
            "🚀 Finalizing the analysis..."
        ],

        /**
         * Initialize the module
         */
        init: function() {
            this.bindEvents();
            this.loadExistingData();
        },

        /**
         * Bind all event handlers
         */
        bindEvents: function() {
            var self = this;

            // Discovery Mode - URL input and extract keywords
            $(document).on('click', '#sf-ilb-extract-btn', function(e) {
                e.preventDefault();
                self.extractKeywords();
            });

            // Discovery Mode - Enter key in URL input
            $(document).on('keypress', '#sf-ilb-priority-url', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    self.extractKeywords();
                }
            });

            // Discovery Mode - Keyword selection
            $(document).on('change', '.sf-ilb-keyword-checkbox', function() {
                self.updateKeywordCount();
            });

            // Discovery Mode - Custom keyword
            $(document).on('click', '#sf-ilb-add-keyword-btn', function(e) {
                e.preventDefault();
                self.addCustomKeyword();
            });

            $(document).on('keypress', '#sf-ilb-new-keyword', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    self.addCustomKeyword();
                }
            });

            // Discovery Mode - Scan for opportunities
            $(document).on('click', '#sf-ilb-scan-btn', function(e) {
                e.preventDefault();
                self.scanOpportunities();
            });

            // Discovery Mode - Back button
            $(document).on('click', '#sf-ilb-back-btn', function(e) {
                e.preventDefault();
                self.goBackToInput();
            });

            // Discovery Mode - Select all opportunities
            $(document).on('change', '#sf-ilb-select-all', function() {
                var checked = $(this).prop('checked');
                $('.sf-ilb-opportunity-row:visible .sf-ilb-opp-select').prop('checked', checked);
                self.updateSelectedCount();
            });

            // Discovery Mode - Individual opportunity selection
            $(document).on('change', '.sf-ilb-opp-select', function() {
                self.updateSelectedCount();
            });

            // Discovery Mode - Add selected links
            $(document).on('click', '#sf-ilb-add-selected', function(e) {
                e.preventDefault();
                self.addSelectedLinks();
            });

            // Bulk Upload Mode - CSV upload event
            $(document).on('sf:csv-uploaded', function(e, data, box) {
                if (box.data('module') === 'internal-link-builder') {
                    self.processBulkCSV(data);
                }
            });

            // Bulk Upload Mode - Apply bulk links
            $(document).on('click', '.sf-bulk-confirm', function(e) {
                e.preventDefault();
                self.applyBulkLinks();
            });

            // Common - Clear data
            $(document).on('click', '#sf-ilb-clear, .sf-bulk-clear-btn, .sf-ilb-bulk-clear', function(e) {
                e.preventDefault();
                self.clearData();
            });

            // Common - Download preview CSV
            $(document).on('click', '#sf-ilb-download-preview, .sf-bulk-download-preview', function(e) {
                e.preventDefault();
                self.downloadPreview();
            });

            // Common - Download results CSV
            $(document).on('click', '.sf-bulk-download-results, .sf-ilb-download-results', function(e) {
                e.preventDefault();
                self.downloadResults();
            });

            // Common - New upload/scan button
            $(document).on('click', '.sf-bulk-new-upload, .sf-ilb-clear-results', function(e) {
                e.preventDefault();
                self.startNew();
            });

            // Instructions toggle
            $(document).on('click', '.sf-internal-link-builder-module .sf-instructions-header', function() {
                var $box = $(this).closest('.sf-instructions-box');
                var $content = $box.find('.sf-instructions-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Section toggles (Manual section in opportunities)
            $(document).on('click', '.sf-internal-link-builder-module .sf-ilb-manual-section .sf-section-toggle', function() {
                var $content = $(this).closest('.sf-ilb-manual-section').find('.sf-ilb-manual-content');
                var icon = $(this).find('.sf-toggle-icon');
                var expanded = $(this).attr('aria-expanded') === 'true';

                $content.slideToggle(200);
                $(this).attr('aria-expanded', !expanded);
                icon.toggleClass('dashicons-arrow-down-alt2 dashicons-arrow-up-alt2');
            });

            // Keyword breakdown toggle
            $(document).on('click', '.sf-ilb-breakdown-toggle', function() {
                var $content = $(this).siblings('.sf-ilb-breakdown-content');
                var expanded = $(this).attr('aria-expanded') === 'true';

                $content.slideToggle(200);
                $(this).attr('aria-expanded', !expanded);
            });

            // Keyword badge click - switch selected keyword and update context
            $(document).on('click', '.sf-ilb-keyword-badge', function(e) {
                e.preventDefault();
                self.selectKeywordBadge($(this));
            });

            // Modal close
            $(document).on('click', '.sf-internal-link-builder-module .sf-modal-close, .sf-internal-link-builder-module .sf-modal', function(e) {
                if (e.target === this) {
                    $(this).closest('.sf-modal').hide();
                }
            });
        },

        /**
         * Load existing data on page load
         */
        loadExistingData: function() {
            var self = this;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_get_data',
                    nonce: sfILBData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.discoveryData = response.data.discovery;
                        self.bulkData = response.data.bulk;
                    }
                }
            });
        },

        // =====================================================================
        // DISCOVERY MODE
        // =====================================================================

        /**
         * Extract keywords from priority URL
         */
        extractKeywords: function() {
            var self = this;
            var url = $('#sf-ilb-priority-url').val().trim();

            if (!url) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning('Please enter a URL.');
                }
                return;
            }

            var $btn = $('#sf-ilb-extract-btn');
            $btn.prop('disabled', true).addClass('sf-loading');

            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.info(sfILBData.i18n.extracting);
            }

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_extract_keywords',
                    nonce: sfILBData.nonce,
                    url: url
                },
                success: function(response) {
                    if (response.success) {
                        self.showKeywordSelection(response.data);
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfILBData.i18n.error);
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfILBData.i18n.error);
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false).removeClass('sf-loading');
                }
            });
        },

        /**
         * Show keyword selection UI
         */
        showKeywordSelection: function(data) {
            // Store data for later use
            this.discoveryData = data;

            // Update target URL display
            $('.sf-ilb-target-url-display').html(
                '<strong>Target:</strong> ' + this.escapeHtml(data.post_title) +
                ' <a href="' + this.escapeAttr(data.url) + '" target="_blank">&#8599;</a>'
            );

            // Build phrase list HTML (without source labels)
            var html = '';
            data.keywords.forEach(function(kw, index) {
                var checked = kw.selected ? 'checked' : '';
                html += '<label class="sf-ilb-keyword-item">';
                // Use escapeAttr for the value attribute to handle quotes properly
                html += '<input type="checkbox" class="sf-ilb-keyword-checkbox" value="' + this.escapeAttr(kw.keyword) + '" ' + checked + '>';
                html += '<span class="sf-ilb-keyword-text">' + this.escapeHtml(kw.keyword) + '</span>';
                html += '</label>';
            }.bind(this));

            $('#sf-ilb-keywords-list').html(html);

            // Hide input section, show keywords section
            $('#sf-ilb-input').hide();
            $('#sf-ilb-keywords').show();

            this.updateKeywordCount();
        },

        /**
         * Go back to URL input
         */
        goBackToInput: function() {
            $('#sf-ilb-keywords').hide();
            $('#sf-ilb-input').show();
        },

        /**
         * Update keyword selection count
         */
        updateKeywordCount: function() {
            var count = $('.sf-ilb-keyword-checkbox:checked').length;
            $('#sf-ilb-selected-count').text(count);

            // Enable/disable scan button
            var $scanBtn = $('#sf-ilb-scan-btn');
            $scanBtn.prop('disabled', count === 0 || count > 10);

            if (count > 10) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfILBData.i18n.max_keywords);
                }
            }
        },

        /**
         * Add custom keyword
         */
        addCustomKeyword: function() {
            var keyword = $('#sf-ilb-new-keyword').val().trim();

            if (!keyword) {
                return;
            }

            // Check if already exists
            var exists = false;
            $('.sf-ilb-keyword-checkbox').each(function() {
                if ($(this).val().toLowerCase() === keyword.toLowerCase()) {
                    exists = true;
                    return false;
                }
            });

            if (exists) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfILBData.i18n.keyword_exists);
                }
                return;
            }

            // Use escapeAttr for the value attribute to handle quotes properly
            var html = '<label class="sf-ilb-keyword-item sf-ilb-custom">';
            html += '<input type="checkbox" class="sf-ilb-keyword-checkbox" value="' + this.escapeAttr(keyword) + '" checked>';
            html += '<span class="sf-ilb-keyword-text">' + this.escapeHtml(keyword) + '</span>';
            html += '</label>';

            $('#sf-ilb-keywords-list').append(html);
            $('#sf-ilb-new-keyword').val('');
            this.updateKeywordCount();
        },

        /**
         * Scan for link opportunities
         */
        scanOpportunities: function() {
            var self = this;

            if (!this.discoveryData) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.error('No target URL set. Please start over.');
                }
                return;
            }

            var targetUrl = this.discoveryData.url;
            var targetPostId = this.discoveryData.post_id;
            var keywords = [];

            $('.sf-ilb-keyword-checkbox:checked').each(function() {
                keywords.push($(this).val());
            });

            if (keywords.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfILBData.i18n.no_keywords);
                }
                return;
            }

            // Get exclude headings filter state (default to true if not found)
            var excludeHeadings = $('#sf-ilb-exclude-headings').prop('checked');
            if (typeof excludeHeadings === 'undefined') {
                excludeHeadings = true;
            }

            var $btn = $('#sf-ilb-scan-btn');
            $btn.prop('disabled', true).addClass('sf-loading').text('Scanning...');

            // Show loading overlay with rotating messages
            var loadingState = self.showScanLoadingContainer();

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_scan_opportunities',
                    nonce: sfILBData.nonce,
                    target_url: targetUrl,
                    target_post_id: targetPostId,
                    keywords: keywords,
                    exclude_headings: excludeHeadings ? 1 : 0
                },
                success: function(response) {
                    if (response.success) {
                        // Hide loading overlay (will complete progress bar first)
                        self.hideScanLoadingContainer(loadingState);
                        // Reload page to show server-rendered results after brief delay
                        setTimeout(function() {
                            location.reload();
                        }, 800);
                    } else {
                        // Hide loading overlay immediately on error
                        self.hideScanLoadingContainer(loadingState);
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfILBData.i18n.error);
                        }
                        $btn.prop('disabled', false).removeClass('sf-loading').text('Find Link Opportunities');
                    }
                },
                error: function() {
                    // Hide loading overlay immediately on error
                    self.hideScanLoadingContainer(loadingState);
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfILBData.i18n.error);
                    }
                    $btn.prop('disabled', false).removeClass('sf-loading').text('Find Link Opportunities');
                }
            });
        },

        /**
         * Update selected opportunities count
         */
        updateSelectedCount: function() {
            var count = $('.sf-ilb-opp-select:checked').length;
            $('#sf-ilb-selected-count-action').text(count);

            // Update select all checkbox state
            var visibleCheckboxes = $('.sf-ilb-opportunity-row:visible .sf-ilb-opp-select');
            var checkedCheckboxes = visibleCheckboxes.filter(':checked');

            if (visibleCheckboxes.length > 0 && checkedCheckboxes.length === visibleCheckboxes.length) {
                $('#sf-ilb-select-all').prop('checked', true).prop('indeterminate', false);
            } else if (checkedCheckboxes.length > 0) {
                $('#sf-ilb-select-all').prop('checked', false).prop('indeterminate', true);
            } else {
                $('#sf-ilb-select-all').prop('checked', false).prop('indeterminate', false);
            }

            // Enable/disable add button
            $('#sf-ilb-add-selected').prop('disabled', count === 0);
        },

        /**
         * Add selected links
         */
        addSelectedLinks: function() {
            var self = this;
            var links = [];
            var targetUrl = $('.sf-ilb-opportunities-section').data('target-url');

            $('.sf-ilb-opp-select:checked').each(function() {
                var $row = $(this).closest('.sf-ilb-opportunity-row');
                links.push({
                    source_post_id: $(this).data('post-id'),
                    source_url: $(this).data('source-url'),
                    keyword: $(this).data('keyword')
                });
            });

            if (links.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfILBData.i18n.no_links_selected);
                }
                return;
            }

            // Show progress modal
            var $modal = $('.sf-bulk-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-bulk-progress-current');
            var $total = $modal.find('.sf-bulk-progress-total');
            var $percent = $modal.find('.sf-bulk-progress-percent');

            $total.text(links.length);
            $current.text(0);
            $percent.text('0');
            $fill.css('width', '0%');
            $modal.show();

            self.processing = true;
            self.processLinkBatch(links, targetUrl, 0);
        },

        /**
         * Process link additions in batches
         */
        processLinkBatch: function(links, targetUrl, offset) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_add_links',
                    nonce: sfILBData.nonce,
                    links: links,
                    target_url: targetUrl,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = data.processed || 0;
                        var currentOffset = offset + processed;
                        var total = data.total_count || links.length;
                        var percent = Math.min(100, Math.round((currentOffset / total) * 100));

                        // Update progress
                        var $modal = $('.sf-bulk-progress-modal');
                        $modal.find('.sf-bulk-progress-current').text(currentOffset);
                        $modal.find('.sf-bulk-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        if (data.complete) {
                            self.processing = false;
                            setTimeout(function() {
                                location.reload();
                            }, 500);
                        } else {
                            self.processLinkBatch(links, targetUrl, currentOffset);
                        }
                    } else {
                        self.processing = false;
                        $('.sf-bulk-progress-modal').hide();
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfILBData.i18n.error);
                        }
                    }
                },
                error: function() {
                    self.processing = false;
                    $('.sf-bulk-progress-modal').hide();
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfILBData.i18n.error);
                    }
                }
            });
        },

        // =====================================================================
        // BULK UPLOAD MODE
        // =====================================================================

        /**
         * Process uploaded bulk CSV
         */
        processBulkCSV: function(uploadData) {
            var self = this;

            // Show loading overlay with rotating messages
            var loadingState = self.showBulkCSVLoadingContainer();

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_process_csv',
                    nonce: sfILBData.nonce,
                    upload_id: uploadData.upload_id
                },
                success: function(response) {
                    if (response.success) {
                        // Hide loading overlay (will complete progress bar first)
                        self.hideBulkCSVLoadingContainer(loadingState);
                        // Reload to show confirmation after brief delay
                        setTimeout(function() {
                            location.reload();
                        }, 800);
                    } else {
                        // Show error state in loading overlay, then remove
                        self.hideBulkCSVLoadingContainer(loadingState, true, response.data.message || sfILBData.i18n.error);
                    }
                },
                error: function() {
                    // Show error state in loading overlay, then remove
                    self.hideBulkCSVLoadingContainer(loadingState, true, sfILBData.i18n.error || 'An error occurred while processing the CSV.');
                }
            });
        },

        /**
         * Apply bulk links from CSV
         */
        applyBulkLinks: function() {
            var self = this;

            if (self.processing) {
                return;
            }

            self.processing = true;

            // Get count from button data or stats
            var $btn = $('.sf-bulk-confirm');
            var total = parseInt($btn.data('count'), 10) ||
                       parseInt($('.sf-bulk-stat-matched .sf-bulk-stat-value').text(), 10) || 0;

            if (total === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning('No links to apply.');
                }
                self.processing = false;
                return;
            }

            // Show progress modal
            var $modal = $('.sf-bulk-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-bulk-progress-current');
            var $total = $modal.find('.sf-bulk-progress-total');
            var $percent = $modal.find('.sf-bulk-progress-percent');

            $total.text(total);
            $current.text(0);
            $percent.text('0');
            $fill.css('width', '0%');
            $modal.show();

            $btn.prop('disabled', true);

            self.processBulkBatch(0, total);
        },

        /**
         * Process bulk links in batches
         */
        processBulkBatch: function(offset, total) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_apply_bulk_links',
                    nonce: sfILBData.nonce,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = data.processed || 0;
                        var currentOffset = offset + processed;
                        var percent = Math.min(100, Math.round((currentOffset / total) * 100));

                        // Update progress
                        var $modal = $('.sf-bulk-progress-modal');
                        $modal.find('.sf-bulk-progress-current').text(currentOffset);
                        $modal.find('.sf-bulk-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        if (data.complete) {
                            self.processing = false;
                            setTimeout(function() {
                                location.reload();
                            }, 500);
                        } else {
                            self.processBulkBatch(currentOffset, total);
                        }
                    } else {
                        self.processing = false;
                        $('.sf-bulk-progress-modal').hide();
                        $('.sf-bulk-confirm').prop('disabled', false);
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfILBData.i18n.error);
                        }
                    }
                },
                error: function() {
                    self.processing = false;
                    $('.sf-bulk-progress-modal').hide();
                    $('.sf-bulk-confirm').prop('disabled', false);
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfILBData.i18n.error);
                    }
                }
            });
        },

        // =====================================================================
        // COMMON
        // =====================================================================

        /**
         * Clear all data
         */
        clearData: function() {
            if (!confirm('Are you sure you want to clear all data? This will allow you to start fresh.')) {
                return;
            }

            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_clear_data',
                    nonce: sfILBData.nonce
                },
                success: function(response) {
                    location.reload();
                },
                error: function() {
                    location.reload();
                }
            });
        },

        /**
         * Download preview CSV
         */
        downloadPreview: function() {
            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_download_preview',
                    nonce: sfILBData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var blob = new Blob([response.data.csv], { type: 'text/csv' });
                        var url = window.URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = response.data.filename;
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                        window.URL.revokeObjectURL(url);
                    }
                }
            });
        },

        /**
         * Download results CSV
         */
        downloadResults: function() {
            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_download_results',
                    nonce: sfILBData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var blob = new Blob([response.data.csv], { type: 'text/csv' });
                        var url = window.URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = response.data.filename;
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                        window.URL.revokeObjectURL(url);
                    }
                }
            });
        },

        /**
         * Start new scan/upload
         */
        startNew: function() {
            $.ajax({
                url: sfILBData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_ilb_clear_data',
                    nonce: sfILBData.nonce
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Select a keyword badge and update the context display
         */
        selectKeywordBadge: function($badge) {
            var $row = $badge.closest('.sf-ilb-opportunity-row');
            var $badges = $row.find('.sf-ilb-keyword-badge');

            // Don't do anything if this is the only badge
            if ($badges.length <= 1) {
                return;
            }

            // Update selected state on badges
            $badges.removeClass('sf-ilb-keyword-selected');
            $badge.addClass('sf-ilb-keyword-selected');

            // Get the keyword and context from the badge data
            var keyword = $badge.data('keyword');
            var context = $badge.data('context');

            // Update the checkbox data-keyword attribute
            $row.find('.sf-ilb-opp-select').attr('data-keyword', keyword);

            // Update the context cell with highlighted keyword
            var $contextCell = $row.find('.sf-ilb-context-cell');
            var highlightedContext = this.highlightKeyword(context, keyword);
            $contextCell.html(highlightedContext);
        },

        /**
         * Highlight keyword in context string
         */
        highlightKeyword: function(context, keyword) {
            var escapedContext = this.escapeHtml(context);
            var escapedKeyword = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            var pattern = new RegExp('(' + escapedKeyword + ')', 'gi');
            return escapedContext.replace(pattern, '<mark class="sf-keyword-highlight">$1</mark>');
        },

        /**
         * Show animated loading container for scanning opportunities
         */
        showScanLoadingContainer: function() {
            var self = this;

            // Remove any existing loading container
            $('.sf-ilb-scan-loading-overlay').remove();

            // Create loading overlay
            var $overlay = $('<div class="sf-ilb-scan-loading-overlay">' +
                '<div class="sf-ilb-scan-loading-container">' +
                    '<div class="sf-ilb-scan-loading-spinner"></div>' +
                    '<div class="sf-ilb-scan-loading-progress">' +
                        '<div class="sf-ilb-scan-progress-bar"><div class="sf-ilb-scan-progress-fill"></div></div>' +
                    '</div>' +
                    '<div class="sf-ilb-scan-loading-message">' + self.scanLoadingMessages[0] + '</div>' +
                    '<div class="sf-ilb-scan-loading-subtext">Please wait while we find internal linking opportunities...</div>' +
                '</div>' +
            '</div>');

            // Insert into the module
            var $module = $('.sf-internal-link-builder-module');
            if ($module.length) {
                $module.append($overlay);
            } else {
                $('body').append($overlay);
            }

            // Animate progress bar
            var $progressFill = $overlay.find('.sf-ilb-scan-progress-fill');
            $progressFill.css('width', '0%');

            // Simulate progress (0-90% over ~15 seconds, slower as it goes)
            var progress = 0;
            var progressInterval = setInterval(function() {
                if (progress < 90) {
                    // Progress slows down as it gets higher
                    var increment = Math.max(0.5, (90 - progress) / 30);
                    progress = Math.min(90, progress + increment);
                    $progressFill.css('width', progress + '%');
                }
            }, 200);

            // Start message rotation (every 2.5 seconds)
            var messageIndex = 0;
            var messageInterval = setInterval(function() {
                messageIndex = (messageIndex + 1) % self.scanLoadingMessages.length;
                var $message = $overlay.find('.sf-ilb-scan-loading-message');
                $message.addClass('sf-fade-out');
                setTimeout(function() {
                    $message.text(self.scanLoadingMessages[messageIndex]).removeClass('sf-fade-out');
                }, 300);
            }, 2500);

            return {
                overlay: $overlay,
                messageInterval: messageInterval,
                progressInterval: progressInterval
            };
        },

        /**
         * Hide scan loading container
         */
        hideScanLoadingContainer: function(loadingState) {
            if (loadingState) {
                if (loadingState.messageInterval) {
                    clearInterval(loadingState.messageInterval);
                }
                if (loadingState.progressInterval) {
                    clearInterval(loadingState.progressInterval);
                }
                if (loadingState.overlay) {
                    // Complete the progress bar before hiding
                    loadingState.overlay.find('.sf-ilb-scan-progress-fill').css('width', '100%');
                    loadingState.overlay.find('.sf-ilb-scan-loading-message').text('✅ Scan complete!');

                    setTimeout(function() {
                        loadingState.overlay.fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, 500);
                }
            }
        },

        /**
         * Show animated loading container for bulk CSV processing
         */
        showBulkCSVLoadingContainer: function() {
            var self = this;

            // Remove any existing loading container
            $('.sf-ilb-bulk-loading-overlay').remove();

            // Create loading overlay
            var $overlay = $('<div class="sf-ilb-bulk-loading-overlay">' +
                '<div class="sf-ilb-bulk-loading-container">' +
                    '<div class="sf-ilb-bulk-loading-spinner"></div>' +
                    '<div class="sf-ilb-bulk-loading-progress">' +
                        '<div class="sf-ilb-bulk-progress-bar"><div class="sf-ilb-bulk-progress-fill"></div></div>' +
                    '</div>' +
                    '<div class="sf-ilb-bulk-loading-message">' + self.bulkCSVLoadingMessages[0] + '</div>' +
                    '<div class="sf-ilb-bulk-loading-subtext">Please wait while we process your CSV file...</div>' +
                '</div>' +
            '</div>');

            // Insert into the module
            var $module = $('.sf-internal-link-builder-module');
            if ($module.length) {
                $module.append($overlay);
            } else {
                $('body').append($overlay);
            }

            // Animate progress bar
            var $progressFill = $overlay.find('.sf-ilb-bulk-progress-fill');
            $progressFill.css('width', '0%');

            // Simulate progress (0-90% over ~15 seconds, slower as it goes)
            var progress = 0;
            var progressInterval = setInterval(function() {
                if (progress < 90) {
                    // Progress slows down as it gets higher
                    var increment = Math.max(0.5, (90 - progress) / 30);
                    progress = Math.min(90, progress + increment);
                    $progressFill.css('width', progress + '%');
                }
            }, 200);

            // Start message rotation (every 2.5 seconds)
            var messageIndex = 0;
            var messageInterval = setInterval(function() {
                messageIndex = (messageIndex + 1) % self.bulkCSVLoadingMessages.length;
                var $message = $overlay.find('.sf-ilb-bulk-loading-message');
                $message.addClass('sf-fade-out');
                setTimeout(function() {
                    $message.text(self.bulkCSVLoadingMessages[messageIndex]).removeClass('sf-fade-out');
                }, 300);
            }, 2500);

            return {
                overlay: $overlay,
                messageInterval: messageInterval,
                progressInterval: progressInterval
            };
        },

        /**
         * Hide bulk CSV loading container
         */
        hideBulkCSVLoadingContainer: function(loadingState, isError, errorMessage) {
            if (loadingState) {
                if (loadingState.messageInterval) {
                    clearInterval(loadingState.messageInterval);
                }
                if (loadingState.progressInterval) {
                    clearInterval(loadingState.progressInterval);
                }
                if (loadingState.overlay) {
                    if (isError) {
                        // Show error state
                        loadingState.overlay.find('.sf-ilb-bulk-progress-fill').css({
                            'width': '100%',
                            'background': '#ef4444'
                        });
                        loadingState.overlay.find('.sf-ilb-bulk-loading-message').text('❌ CSV processing failed');
                        loadingState.overlay.find('.sf-ilb-bulk-loading-subtext').text(errorMessage || 'An error occurred.');
                        loadingState.overlay.find('.sf-ilb-bulk-loading-spinner').hide();

                        setTimeout(function() {
                            loadingState.overlay.fadeOut(500, function() {
                                $(this).remove();
                            });
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error(errorMessage || 'CSV processing failed.');
                            }
                        }, 2500);
                    } else {
                        // Complete the progress bar before hiding
                        loadingState.overlay.find('.sf-ilb-bulk-progress-fill').css('width', '100%');
                        loadingState.overlay.find('.sf-ilb-bulk-loading-message').text('✅ CSV processed!');

                        setTimeout(function() {
                            loadingState.overlay.fadeOut(300, function() {
                                $(this).remove();
                            });
                        }, 500);
                    }
                }
            }
        },

        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        /**
         * Escape text for use in HTML attribute values (handles quotes)
         */
        escapeAttr: function(text) {
            return String(text)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;');
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        if ($('.sf-internal-link-builder-module').length > 0) {
            ScreamingFixes.InternalLinkBuilder.init();
        }
    });

})(jQuery);
