<?php
/**
 * Internal Link Builder Module for Screaming Fixes
 *
 * Helps users add internal links to a priority page through Discovery Mode
 * (find opportunities) or Bulk Upload Mode (CSV with pre-defined links).
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Internal_Link_Builder extends SF_Module {

    /**
     * Maximum number of CSV rows allowed for processing.
     * This limit protects against memory exhaustion on limited hosting plans.
     *
     * @var int
     */
    const MAX_CSV_ROWS = 5000;

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'internal-link-builder';

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Internal Link Builder', 'screaming-fixes');
        $this->slug = 'internal-link-builder';
        $this->description = __('Build internal links to boost priority pages.', 'screaming-fixes');

        parent::__construct();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Discovery Mode AJAX handlers
        add_action('wp_ajax_sf_ilb_extract_keywords', [$this, 'ajax_extract_keywords']);
        add_action('wp_ajax_sf_ilb_scan_opportunities', [$this, 'ajax_scan_opportunities']);
        add_action('wp_ajax_sf_ilb_add_links', [$this, 'ajax_add_links']);

        // Bulk Upload Mode AJAX handlers
        add_action('wp_ajax_sf_ilb_process_csv', [$this, 'ajax_process_csv']);
        add_action('wp_ajax_sf_ilb_apply_bulk_links', [$this, 'ajax_apply_bulk_links']);

        // Common AJAX handlers
        add_action('wp_ajax_sf_ilb_download_preview', [$this, 'ajax_download_preview']);
        add_action('wp_ajax_sf_ilb_download_results', [$this, 'ajax_download_results']);
        add_action('wp_ajax_sf_ilb_clear_data', [$this, 'ajax_clear_data']);
        add_action('wp_ajax_sf_ilb_get_data', [$this, 'ajax_get_data']);

        // Asset enqueuing
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'internal-link-builder') {
            return;
        }

        // Use file modification time for cache busting
        $page_title_css = SF_PLUGIN_DIR . 'modules/page-title/assets/page-title.css';
        $css_file = SF_PLUGIN_DIR . 'modules/internal-link-builder/assets/internal-link-builder.css';
        $js_file = SF_PLUGIN_DIR . 'modules/internal-link-builder/assets/internal-link-builder.js';
        $page_title_version = file_exists($page_title_css) ? SF_VERSION . '.' . filemtime($page_title_css) : SF_VERSION;
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        // Enqueue page-title CSS for bulk styling reuse
        wp_enqueue_style(
            'sf-page-title',
            SF_PLUGIN_URL . 'modules/page-title/assets/page-title.css',
            ['screaming-fixes-admin'],
            $page_title_version
        );

        wp_enqueue_style(
            'sf-internal-link-builder',
            SF_PLUGIN_URL . 'modules/internal-link-builder/assets/internal-link-builder.css',
            ['screaming-fixes-admin', 'sf-page-title'],
            $css_version
        );

        wp_enqueue_script(
            'sf-internal-link-builder',
            SF_PLUGIN_URL . 'modules/internal-link-builder/assets/internal-link-builder.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        wp_localize_script('sf-internal-link-builder', 'sfILBData', [
            'nonce' => wp_create_nonce('sf_ilb_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'siteUrl' => home_url(),
            'i18n' => [
                'extracting' => __('Finding phrases...', 'screaming-fixes'),
                'scanning' => __('Finding link opportunities...', 'screaming-fixes'),
                'adding_links' => __('Adding internal links...', 'screaming-fixes'),
                'processing' => __('Processing CSV...', 'screaming-fixes'),
                'error' => __('An error occurred. Please try again.', 'screaming-fixes'),
                'no_keywords' => __('Please select at least one phrase.', 'screaming-fixes'),
                'max_keywords' => __('Maximum 10 phrases allowed.', 'screaming-fixes'),
                'confirm_add' => __('Add links to %d pages?', 'screaming-fixes'),
                'links_added' => __('Links added successfully!', 'screaming-fixes'),
                'no_links_selected' => __('No links selected.', 'screaming-fixes'),
                'keyword_exists' => __('Phrase already exists.', 'screaming-fixes'),
                'large_site_warning' => __('Large site detected (%d posts). Scan may take 30-60 seconds.', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Run the module scan
     *
     * @return array Scan results
     */
    public function run_scan() {
        return $this->get_results();
    }

    /**
     * Render module content
     */
    public function render() {
        include SF_PLUGIN_DIR . 'modules/internal-link-builder/views/tab-content.php';
    }

    /**
     * Render instructions
     */
    public function render_instructions() {
        include SF_PLUGIN_DIR . 'modules/internal-link-builder/views/instructions.php';
    }

    // =========================================================================
    // SHARED UTILITIES
    // =========================================================================

    /**
     * Normalize URL for comparison
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private function normalize_url($url) {
        $url = trim($url);
        $url = rtrim($url, '/');
        $url = preg_replace('/^https?:\/\//i', '', $url);
        $url = preg_replace('/^www\./i', '', $url);
        $url = strtolower($url);
        return $url;
    }

    /**
     * Convert URL to WordPress post ID
     *
     * @param string $url URL to convert
     * @return int Post ID or 0 if not found
     */
    private function url_to_post_id($url) {
        if (empty($url)) {
            return 0;
        }

        // Try WordPress built-in function
        $post_id = url_to_postid($url);
        if ($post_id) {
            return $post_id;
        }

        // Try with trailing slash variations
        $post_id = url_to_postid(trailingslashit($url));
        if ($post_id) {
            return $post_id;
        }

        $post_id = url_to_postid(untrailingslashit($url));
        if ($post_id) {
            return $post_id;
        }

        // Try slug extraction
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $slug = basename(rtrim($path, '/'));
            global $wpdb;
            $post_id = $wpdb->get_var($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts}
                 WHERE post_name = %s
                 AND post_status IN ('publish', 'draft', 'pending', 'private')
                 AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')
                 LIMIT 1",
                $slug
            ));
            if ($post_id) {
                return (int) $post_id;
            }
        }

        return 0;
    }

    /**
     * Build WordPress URL map for quick lookups
     *
     * @param array $post_types Post types to include
     * @return array Map of normalized URL => post data
     */
    private function build_wp_url_map($post_types = ['post', 'page']) {
        global $wpdb;
        $map = [];

        $post_types_sql = "'" . implode("','", array_map('esc_sql', $post_types)) . "'";

        $posts = $wpdb->get_results(
            "SELECT ID, post_name, post_type FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ({$post_types_sql})
             LIMIT 50000",
            ARRAY_A
        );

        foreach ($posts as $post) {
            $permalink = get_permalink($post['ID']);
            if ($permalink) {
                $normalized = $this->normalize_url($permalink);
                $map[$normalized] = [
                    'post_id' => (int) $post['ID'],
                    'post_type' => $post['post_type'],
                ];
            }
        }

        return $map;
    }

    /**
     * Categorize source page as fixable or manual
     *
     * @param int $post_id Post ID
     * @param string $source_url Source URL
     * @return array Category info with 'category' and 'note' keys
     */
    private function categorize_source_page($post_id, $source_url) {
        $path = wp_parse_url($source_url, PHP_URL_PATH) ?: '/';

        // Manual fix patterns (archives, pagination, etc.)
        $manual_patterns = [
            '/\/category\//' => __('Category archive - edit in theme or SEO plugin', 'screaming-fixes'),
            '/\/tag\//' => __('Tag archive - edit in theme or SEO plugin', 'screaming-fixes'),
            '/\/author\//' => __('Author archive - edit in theme or SEO plugin', 'screaming-fixes'),
            '/\/page\/\d+/' => __('Pagination page - controlled by source', 'screaming-fixes'),
            '/\/feed\/?/' => __('RSS feed - not editable', 'screaming-fixes'),
            '/\/\d{4}\/\d{2}\/?' . '$/' => __('Date archive - edit in theme settings', 'screaming-fixes'),
        ];

        foreach ($manual_patterns as $pattern => $note) {
            if (preg_match($pattern, $path)) {
                return ['category' => 'manual', 'note' => $note];
            }
        }

        // Check for homepage
        if ($path === '/' || $path === '' || $path === '/index.php') {
            $front_page_id = get_option('page_on_front');
            if ($front_page_id && $post_id == $front_page_id) {
                return ['category' => 'fixable', 'note' => ''];
            }
            return ['category' => 'manual', 'note' => __('Homepage - edit in theme or widgets', 'screaming-fixes')];
        }

        // Check for blog index
        $blog_page_id = get_option('page_for_posts');
        if ($blog_page_id && $post_id == $blog_page_id) {
            return ['category' => 'manual', 'note' => __('Blog index - edit in theme settings', 'screaming-fixes')];
        }

        // Valid post/page with content
        if ($post_id > 0) {
            $post = get_post($post_id);
            if ($post && in_array($post->post_type, ['post', 'page'])) {
                return ['category' => 'fixable', 'note' => ''];
            }
            // Allow custom post types in bulk mode
            if ($post) {
                return ['category' => 'fixable', 'note' => ''];
            }
        }

        return ['category' => 'manual', 'note' => __('Page not found in WordPress', 'screaming-fixes')];
    }

    /**
     * Add link to content for first occurrence of keyword
     *
     * @param string $content Post content
     * @param string $keyword Anchor text to link
     * @param string $target_url URL to link to
     * @return string Modified content (or unchanged if no match)
     */
    private function add_link_to_content($content, $keyword, $target_url) {
        if (empty($keyword) || empty($target_url)) {
            return $content;
        }

        // Check if keyword is already linked to this target
        if ($this->is_keyword_already_linked($content, $keyword)) {
            return $content;
        }

        // Escape for regex
        $escaped_keyword = preg_quote($keyword, '/');

        // Pattern: Match keyword that's NOT inside an existing <a> tag
        // The negative lookbehind prevents matching inside HTML attributes (after = or ")
        // The negative lookahead prevents matching text that's followed by </a> (inside a link)
        $pattern = '/(?<![="])(\b' . $escaped_keyword . '\b)(?![^<]*<\/a>)/iu';

        // Check if we have a match
        if (!preg_match($pattern, $content)) {
            // Try without word boundaries for phrases
            $pattern = '/(?<![="])(' . $escaped_keyword . ')(?![^<]*<\/a>)/iu';
            if (!preg_match($pattern, $content)) {
                return $content;
            }
        }

        // Replace first occurrence only
        $replacement = '<a href="' . esc_url($target_url) . '">$1</a>';
        $new_content = preg_replace($pattern, $replacement, $content, 1);

        return $new_content;
    }

    /**
     * Check if keyword is already wrapped in a link
     *
     * @param string $html_content HTML content
     * @param string $keyword Keyword to check
     * @return bool True if already linked
     */
    private function is_keyword_already_linked($html_content, $keyword) {
        $escaped_keyword = preg_quote($keyword, '/');

        // Check if keyword appears within any anchor tag
        $pattern = '/<a\s[^>]*>[^<]*' . $escaped_keyword . '[^<]*<\/a>/is';

        return (bool) preg_match($pattern, $html_content);
    }

    /**
     * Normalize column name for flexible matching
     *
     * @param string $name Column name
     * @return string Normalized name
     */
    private function normalize_column_name($name) {
        $normalized = strtolower(trim($name));
        $normalized = str_replace([' ', '-', '_'], '', $normalized);
        return $normalized;
    }

    /**
     * Truncate URL for display
     *
     * @param string $url URL to truncate
     * @param int $max_length Maximum length
     * @return string Truncated URL
     */
    private function truncate_url($url, $max_length = 40) {
        if (strlen($url) <= $max_length) {
            return $url;
        }
        return substr($url, 0, $max_length) . '...';
    }

    /**
     * Highlight keyword in context string
     *
     * @param string $context Context string
     * @param string $keyword Keyword to highlight
     * @return string HTML with highlighted keyword
     */
    private function highlight_keyword_in_context($context, $keyword) {
        $escaped_context = esc_html($context);
        $escaped_keyword = preg_quote(esc_html($keyword), '/');

        $pattern = '/(' . $escaped_keyword . ')/i';
        return preg_replace($pattern, '<mark class="sf-keyword-highlight">$1</mark>', $escaped_context);
    }

    /**
     * Strip shortcode tags and attributes while preserving inner content
     *
     * This prevents matching keywords inside shortcode attributes
     * (e.g., "menu" inside menu_anchor="" from Avada/Fusion Builder)
     *
     * @param string $content Content with shortcodes
     * @return string Content with shortcode tags removed but inner content preserved
     */
    private function strip_shortcode_tags($content) {
        if (empty($content)) {
            return $content;
        }

        // Pattern to match shortcode opening tags with attributes: [shortcode attr="value"]
        // This captures the shortcode name and all attributes, removing them
        $pattern_opening = '/\[[a-zA-Z_][a-zA-Z0-9_-]*(?:\s[^\]]*?)?\]/';

        // Pattern to match shortcode closing tags: [/shortcode]
        $pattern_closing = '/\[\/[a-zA-Z_][a-zA-Z0-9_-]*\]/';

        // Remove opening tags (including self-closing shortcodes with attributes)
        $content = preg_replace($pattern_opening, '', $content);

        // Remove closing tags
        $content = preg_replace($pattern_closing, '', $content);

        return $content;
    }

    /**
     * CSV escape helper
     *
     * @param array $fields Fields to escape
     * @return string CSV row
     */
    private function csv_escape_row($fields) {
        $escaped = [];
        foreach ($fields as $field) {
            $field = (string) $field;
            if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
                $field = '"' . str_replace('"', '""', $field) . '"';
            }
            $escaped[] = $field;
        }
        return implode(',', $escaped);
    }

    // =========================================================================
    // DATA STORAGE
    // =========================================================================

    /**
     * Save discovery mode data
     *
     * @param array $data Data to save
     */
    private function save_discovery_data($data) {
        $user_id = get_current_user_id();
        set_transient('sf_ilb_discovery_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
    }

    /**
     * Get discovery mode data
     *
     * @return array|false Data or false
     */
    private function get_discovery_data() {
        $user_id = get_current_user_id();
        return get_transient('sf_ilb_discovery_' . $user_id);
    }

    /**
     * Save bulk mode data
     *
     * @param array $data Data to save
     */
    private function save_bulk_data($data) {
        $user_id = get_current_user_id();
        set_transient('sf_ilb_bulk_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
    }

    /**
     * Get bulk mode data
     *
     * @return array|false Data or false
     */
    private function get_bulk_data() {
        $user_id = get_current_user_id();
        return get_transient('sf_ilb_bulk_' . $user_id);
    }

    /**
     * Save results data
     *
     * @param array $data Data to save
     */
    private function save_results_data($data) {
        $user_id = get_current_user_id();
        set_transient('sf_ilb_results_' . $user_id, $data, 2 * HOUR_IN_SECONDS);
    }

    /**
     * Get results data
     *
     * @return array Data or empty array
     */
    private function get_results_data() {
        $user_id = get_current_user_id();
        return get_transient('sf_ilb_results_' . $user_id) ?: [];
    }

    /**
     * Clear all module data
     */
    private function clear_all_data() {
        $user_id = get_current_user_id();
        delete_transient('sf_ilb_discovery_' . $user_id);
        delete_transient('sf_ilb_bulk_' . $user_id);
        delete_transient('sf_ilb_results_' . $user_id);
        delete_transient('sf_ilb_accumulated_' . $user_id);
        delete_transient('sf_ilb_bulk_accumulated_' . $user_id);
    }

    // =========================================================================
    // DISCOVERY MODE - KEYWORD EXTRACTION
    // =========================================================================

    /**
     * AJAX: Extract keywords from priority URL
     */
    public function ajax_extract_keywords() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';

        if (empty($url)) {
            wp_send_json_error(['message' => __('Please enter a URL.', 'screaming-fixes')]);
        }

        // Handle relative URLs
        if (!preg_match('/^https?:\/\//i', $url)) {
            $url = home_url($url);
        }

        // Convert URL to post ID
        $post_id = $this->url_to_post_id($url);

        if (!$post_id) {
            wp_send_json_error(['message' => __('URL not found in WordPress.', 'screaming-fixes')]);
        }

        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(['message' => __('Post not found.', 'screaming-fixes')]);
        }

        $keywords = $this->extract_keywords_from_post($post_id, $post);

        wp_send_json_success([
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'url' => get_permalink($post_id),
            'keywords' => $keywords,
        ]);
    }

    /**
     * Extract keywords from post using multiple sources
     *
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     * @return array Keywords with source info
     */
    private function extract_keywords_from_post($post_id, $post) {
        $keywords = [];

        // 1. SEO Plugin Focus Keyword (highest priority)
        $focus_keyword = $this->get_focus_keyword($post_id);
        if ($focus_keyword) {
            // Handle comma-separated focus keywords (Rank Math supports multiple)
            $focus_parts = array_map('trim', explode(',', $focus_keyword));
            foreach ($focus_parts as $kw) {
                if (!empty($kw) && strlen($kw) >= 3) {
                    $keywords[] = [
                        'keyword' => $kw,
                        'source' => 'focus_keyword',
                        'source_label' => __('SEO focus keyword', 'screaming-fixes'),
                        'selected' => true,
                    ];
                }
            }
        }

        // 2. Page Title
        if (!empty($post->post_title)) {
            $title_phrases = $this->extract_meaningful_phrases($post->post_title);
            foreach ($title_phrases as $phrase) {
                if (!$this->keyword_exists($keywords, $phrase)) {
                    $keywords[] = [
                        'keyword' => $phrase,
                        'source' => 'title',
                        'source_label' => __('from page title', 'screaming-fixes'),
                        'selected' => count($keywords) < 4,
                    ];
                }
            }
        }

        // 3. H1 Tag (if different from title)
        $h1 = $this->extract_h1_from_content($post->post_content);
        if ($h1 && strtolower($h1) !== strtolower($post->post_title)) {
            $h1_phrases = $this->extract_meaningful_phrases($h1);
            foreach ($h1_phrases as $phrase) {
                if (!$this->keyword_exists($keywords, $phrase)) {
                    $keywords[] = [
                        'keyword' => $phrase,
                        'source' => 'h1',
                        'source_label' => __('from H1', 'screaming-fixes'),
                        'selected' => count($keywords) < 4,
                    ];
                }
            }
        }

        // 4. URL Slug
        $slug = $post->post_name;
        if ($slug) {
            $slug_phrase = str_replace('-', ' ', $slug);
            if (strlen($slug_phrase) >= 3 && !$this->keyword_exists($keywords, $slug_phrase)) {
                $keywords[] = [
                    'keyword' => $slug_phrase,
                    'source' => 'slug',
                    'source_label' => __('from URL', 'screaming-fixes'),
                    'selected' => count($keywords) < 4,
                ];
            }
        }

        return array_slice($keywords, 0, 15);
    }

    /**
     * Get focus keyword from SEO plugins
     *
     * @param int $post_id Post ID
     * @return string|null Focus keyword or null
     */
    private function get_focus_keyword($post_id) {
        // Rank Math
        $rm_keyword = get_post_meta($post_id, 'rank_math_focus_keyword', true);
        if ($rm_keyword) {
            return $rm_keyword;
        }

        // Yoast
        $yoast_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        if ($yoast_keyword) {
            return $yoast_keyword;
        }

        // AIOSEO
        $aioseo_keyword = get_post_meta($post_id, '_aioseo_keywords', true);
        if ($aioseo_keyword) {
            return $aioseo_keyword;
        }

        return null;
    }

    /**
     * Extract H1 from post content
     *
     * @param string $content Post content
     * @return string|null H1 text or null
     */
    private function extract_h1_from_content($content) {
        if (preg_match('/<h1[^>]*>(.*?)<\/h1>/is', $content, $matches)) {
            return wp_strip_all_tags($matches[1]);
        }
        return null;
    }

    /**
     * Extract meaningful phrases from text
     *
     * @param string $text Text to extract from
     * @return array Phrases
     */
    private function extract_meaningful_phrases($text) {
        $phrases = [];

        // Clean text
        $text = wp_strip_all_tags($text);
        $text = strtolower(trim($text));

        // Common stop words to filter
        $stop_words = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for',
                       'of', 'with', 'by', 'from', 'is', 'are', 'was', 'were', 'be', 'been',
                       'being', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would',
                       'could', 'should', 'may', 'might', 'must', 'shall', 'can', 'need',
                       'this', 'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it',
                       'we', 'they', 'what', 'which', 'who', 'whom', 'how', 'why', 'when',
                       'where', 'your', 'our', 'my', 'his', 'her', 'its', 'their'];

        $words = preg_split('/\s+/', $text);
        $meaningful_words = array_filter($words, function($w) use ($stop_words) {
            return strlen($w) >= 3 && !in_array($w, $stop_words);
        });

        // Full phrase (if 2-5 meaningful words)
        if (count($meaningful_words) >= 2 && count($meaningful_words) <= 5) {
            $phrases[] = $text;
        }

        // 2-3 word combinations
        $word_count = count($words);
        for ($i = 0; $i < $word_count - 1; $i++) {
            // 2-word phrase
            if ($i < $word_count - 1) {
                $two_word = $words[$i] . ' ' . $words[$i + 1];
                if ($this->is_meaningful_phrase($two_word, $stop_words)) {
                    $phrases[] = $two_word;
                }
            }

            // 3-word phrase
            if ($i < $word_count - 2) {
                $three_word = $words[$i] . ' ' . $words[$i + 1] . ' ' . $words[$i + 2];
                if ($this->is_meaningful_phrase($three_word, $stop_words)) {
                    $phrases[] = $three_word;
                }
            }
        }

        return array_unique($phrases);
    }

    /**
     * Check if phrase is meaningful
     *
     * @param string $phrase Phrase to check
     * @param array $stop_words Stop words list
     * @return bool True if meaningful
     */
    private function is_meaningful_phrase($phrase, $stop_words) {
        $words = explode(' ', $phrase);
        $meaningful_count = 0;

        foreach ($words as $word) {
            if (strlen($word) >= 3 && !in_array($word, $stop_words)) {
                $meaningful_count++;
            }
        }

        return $meaningful_count >= 1 && strlen($phrase) >= 5;
    }

    /**
     * Check if keyword already exists in array
     *
     * @param array $keywords Existing keywords
     * @param string $new_keyword New keyword to check
     * @return bool True if exists
     */
    private function keyword_exists($keywords, $new_keyword) {
        $new_lower = strtolower(trim($new_keyword));
        foreach ($keywords as $kw) {
            if (strtolower(trim($kw['keyword'])) === $new_lower) {
                return true;
            }
        }
        return false;
    }

    // =========================================================================
    // DISCOVERY MODE - OPPORTUNITY SCANNING
    // =========================================================================

    /**
     * AJAX: Scan for link opportunities
     */
    public function ajax_scan_opportunities() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $target_url = isset($_POST['target_url']) ? esc_url_raw($_POST['target_url']) : '';
        $target_post_id = isset($_POST['target_post_id']) ? absint($_POST['target_post_id']) : 0;
        $keywords = isset($_POST['keywords']) ? array_map('sanitize_text_field', $_POST['keywords']) : [];
        // Default to true (exclude headings) if not provided
        $exclude_headings = isset($_POST['exclude_headings']) ? (bool) $_POST['exclude_headings'] : true;

        if (empty($target_url) || !$target_post_id) {
            wp_send_json_error(['message' => __('Invalid target URL.', 'screaming-fixes')]);
        }

        if (empty($keywords)) {
            wp_send_json_error(['message' => __('Please select at least one keyword.', 'screaming-fixes')]);
        }

        if (count($keywords) > 10) {
            wp_send_json_error(['message' => __('Maximum 10 keywords allowed.', 'screaming-fixes')]);
        }

        // Check site size
        $post_count = wp_count_posts('post')->publish + wp_count_posts('page')->publish;
        $large_site = $post_count > 2000;

        if ($large_site) {
            set_time_limit(120);
        }

        $start_time = microtime(true);
        $opportunities = $this->find_link_opportunities($target_post_id, $target_url, $keywords, $exclude_headings);
        $scan_time = round(microtime(true) - $start_time, 2);

        // Store results
        $this->save_discovery_data([
            'target_url' => $target_url,
            'target_post_id' => $target_post_id,
            'keywords' => $keywords,
            'opportunities' => $opportunities,
            'scan_time' => $scan_time,
        ]);

        wp_send_json_success([
            'fixable' => $opportunities['fixable'],
            'manual' => $opportunities['manual'],
            'fixable_count' => count($opportunities['fixable']),
            'manual_count' => count($opportunities['manual']),
            'scan_time' => $scan_time,
            'large_site' => $large_site,
            'post_count' => $post_count,
            'limit_reached' => $opportunities['limit_reached'] ?? false,
            'max_opportunities' => self::MAX_OPPORTUNITIES,
        ]);
    }

    /**
     * Maximum number of fixable opportunities to return
     * Limits results to prevent long scans on large sites
     */
    const MAX_OPPORTUNITIES = 30;

    /**
     * Find link opportunities across all content
     *
     * @param int $target_post_id Target post ID
     * @param string $target_url Target URL
     * @param array $keywords Keywords to search for
     * @param bool $exclude_headings Whether to exclude matches found in headings (default true)
     * @return array Opportunities grouped by category, plus keyword stats
     */
    private function find_link_opportunities($target_post_id, $target_url, $keywords, $exclude_headings = true) {
        global $wpdb;

        $opportunities = [
            'fixable' => [],
            'manual' => [],
            'keyword_stats' => [], // Track match counts per keyword
            'limit_reached' => false, // Track if we hit the limit
        ];

        // Initialize keyword stats
        foreach ($keywords as $keyword) {
            $opportunities['keyword_stats'][$keyword] = 0;
        }

        // Build keyword search pattern for SQL
        $keyword_conditions = [];
        $keyword_values = [];

        foreach ($keywords as $keyword) {
            $keyword_conditions[] = "post_content LIKE %s";
            $keyword_values[] = '%' . $wpdb->esc_like($keyword) . '%';
        }

        $keyword_sql = implode(' OR ', $keyword_conditions);

        // Query for posts/pages containing any keyword
        $query = $wpdb->prepare(
            "SELECT ID, post_title, post_content, post_type, post_name
             FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ('post', 'page')
             AND ID != %d
             AND ({$keyword_sql})
             ORDER BY post_date DESC
             LIMIT 500",
            array_merge([$target_post_id], $keyword_values)
        );

        $posts = $wpdb->get_results($query);

        if (empty($posts)) {
            return $opportunities;
        }

        // Get posts that already link to target
        $already_linked = $this->get_posts_linking_to($target_url);

        foreach ($posts as $post) {
            // Stop if we've reached the maximum number of fixable opportunities
            if (count($opportunities['fixable']) >= self::MAX_OPPORTUNITIES) {
                $opportunities['limit_reached'] = true;
                break;
            }

            // Skip if already links to target
            if (in_array($post->ID, $already_linked)) {
                continue;
            }

            // Prepare content for search - optionally strip headings
            $content_for_search = $post->post_content;
            if ($exclude_headings) {
                $content_for_search = preg_replace('/<h[1-6][^>]*>.*?<\/h[1-6]>/is', '', $content_for_search);
            }

            // Find keyword matches
            $matches = $this->find_keyword_matches($content_for_search, $keywords, $target_url);

            if (empty($matches)) {
                continue;
            }

            // Update keyword stats for each match found
            foreach ($matches as $match) {
                $opportunities['keyword_stats'][$match['keyword']]++;
            }

            $primary_match = $matches[0];
            $source_url = get_permalink($post->ID);
            $category = $this->categorize_source_page($post->ID, $source_url);

            $opportunity = [
                'source_post_id' => $post->ID,
                'source_url' => $source_url,
                'source_title' => $post->post_title,
                'edit_url' => get_edit_post_link($post->ID, 'raw'),
                'keyword' => $primary_match['keyword'],
                'context' => $primary_match['context'],
                'all_matches' => $matches,
                'match_count' => count($matches),
            ];

            if ($category['category'] === 'fixable') {
                $opportunities['fixable'][] = $opportunity;
            } else {
                $opportunity['note'] = $category['note'];
                $opportunities['manual'][] = $opportunity;
            }
        }

        return $opportunities;
    }

    /**
     * Find keyword matches in content with context
     *
     * @param string $content Post content
     * @param array $keywords Keywords to find
     * @param string $target_url Target URL (to check existing links)
     * @return array Matches with context
     */
    private function find_keyword_matches($content, $keywords, $target_url) {
        $matches = [];

        // First, strip shortcodes to avoid matching keywords inside shortcode attributes
        // (e.g., menu_anchor="" in Avada/Fusion Builder shortcodes)
        $content_without_shortcodes = $this->strip_shortcode_tags($content);

        // Then strip HTML tags to get plain text
        $text_content = wp_strip_all_tags($content_without_shortcodes);

        foreach ($keywords as $keyword) {
            $keyword_lower = strtolower($keyword);
            $content_lower = strtolower($text_content);

            $pos = strpos($content_lower, $keyword_lower);

            if ($pos === false) {
                continue;
            }

            // Check if already linked
            if ($this->is_keyword_already_linked($content, $keyword)) {
                continue;
            }

            // Extract context (~50 chars around keyword)
            $context_start = max(0, $pos - 25);
            $context_end = min(strlen($text_content), $pos + strlen($keyword) + 25);
            $context = substr($text_content, $context_start, $context_end - $context_start);

            if ($context_start > 0) {
                $context = '...' . $context;
            }
            if ($context_end < strlen($text_content)) {
                $context = $context . '...';
            }

            $matches[] = [
                'keyword' => $keyword,
                'context' => $context,
                'position' => $pos,
            ];
        }

        return $matches;
    }

    /**
     * Get post IDs that already link to target URL
     *
     * @param string $target_url Target URL
     * @return array Post IDs
     */
    private function get_posts_linking_to($target_url) {
        global $wpdb;

        $post_ids = [];
        $url_path = wp_parse_url($target_url, PHP_URL_PATH);

        $url_variations = [
            $target_url,
            rtrim($target_url, '/'),
            trailingslashit($target_url),
        ];

        if ($url_path) {
            $url_variations[] = $url_path;
            $url_variations[] = rtrim($url_path, '/');
        }

        foreach ($url_variations as $url) {
            if (empty($url)) {
                continue;
            }
            $results = $wpdb->get_col($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts}
                 WHERE post_status = 'publish'
                 AND post_type IN ('post', 'page')
                 AND post_content LIKE %s",
                '%href="' . $wpdb->esc_like($url) . '%'
            ));

            $post_ids = array_merge($post_ids, $results);
        }

        return array_unique(array_map('intval', $post_ids));
    }

    // =========================================================================
    // DISCOVERY MODE - LINK ADDITION
    // =========================================================================

    /**
     * AJAX: Add internal links
     */
    public function ajax_add_links() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $links = isset($_POST['links']) ? $_POST['links'] : [];
        $target_url = isset($_POST['target_url']) ? esc_url_raw($_POST['target_url']) : '';
        $offset = isset($_POST['offset']) ? absint($_POST['offset']) : 0;
        $batch_size = isset($_POST['batch_size']) ? absint($_POST['batch_size']) : 50;

        if (empty($links) || empty($target_url)) {
            wp_send_json_error(['message' => __('Invalid request data.', 'screaming-fixes')]);
        }

        // Clear accumulated results at start
        if ($offset === 0) {
            delete_transient('sf_ilb_accumulated_' . get_current_user_id());
        }

        $results = $this->process_link_additions($links, $target_url, $offset, $batch_size);

        // Accumulate results
        $accumulated = get_transient('sf_ilb_accumulated_' . get_current_user_id());
        if (!$accumulated) {
            $accumulated = ['success' => 0, 'failed' => 0, 'skipped' => 0, 'details' => [], 'errors' => []];
        }

        $accumulated['success'] += $results['success'];
        $accumulated['failed'] += $results['failed'];
        $accumulated['skipped'] += $results['skipped'];
        $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
        $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

        if ($results['complete']) {
            $this->save_results_data($accumulated);
            delete_transient('sf_ilb_accumulated_' . get_current_user_id());

            // Log to activity log
            if ($accumulated['success'] > 0) {
                SF_Activity_Log::log('internal-link-builder', $accumulated['success']);
            }

            $results['total_success'] = $accumulated['success'];
            $results['total_failed'] = $accumulated['failed'];
            $results['total_skipped'] = $accumulated['skipped'];
            $results['all_details'] = $accumulated['details'];
        } else {
            set_transient('sf_ilb_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
        }

        $results['total_count'] = count($links);
        wp_send_json_success($results);
    }

    /**
     * Process link additions in batch
     *
     * @param array $links Links to add
     * @param string $target_url Target URL
     * @param int $offset Current offset
     * @param int $batch_size Batch size
     * @return array Results
     */
    private function process_link_additions($links, $target_url, $offset, $batch_size) {
        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'details' => [],
            'errors' => [],
            'complete' => false,
            'next_offset' => $offset + $batch_size,
        ];

        $batch = array_slice($links, $offset, $batch_size);

        if (empty($batch)) {
            $results['complete'] = true;
            return $results;
        }

        // Group by source post to minimize DB operations
        $grouped = [];
        foreach ($batch as $link) {
            $post_id = isset($link['source_post_id']) ? absint($link['source_post_id']) : 0;
            if (!$post_id) {
                continue;
            }

            if (!isset($grouped[$post_id])) {
                $grouped[$post_id] = [];
            }
            $grouped[$post_id][] = $link;
        }

        foreach ($grouped as $post_id => $post_links) {
            $post = get_post($post_id);
            if (!$post) {
                foreach ($post_links as $link) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'source_url' => $link['source_url'] ?? '',
                        'keyword' => $link['keyword'] ?? '',
                        'error' => __('Post not found', 'screaming-fixes'),
                    ];
                }
                continue;
            }

            $content = $post->post_content;
            $original_content = $content;

            foreach ($post_links as $link) {
                $results['processed']++;
                $keyword = isset($link['keyword']) ? $link['keyword'] : '';
                $source_url = isset($link['source_url']) ? $link['source_url'] : get_permalink($post_id);

                if (empty($keyword)) {
                    $results['skipped']++;
                    continue;
                }

                $new_content = $this->add_link_to_content($content, $keyword, $target_url);

                if ($new_content !== $content) {
                    $content = $new_content;
                    $results['success']++;
                    $results['details'][] = [
                        'source_url' => $source_url,
                        'source_title' => $post->post_title,
                        'keyword' => $keyword,
                        'target_url' => $target_url,
                        'status' => 'added',
                    ];
                } else {
                    $results['skipped']++;
                    $results['errors'][] = [
                        'source_url' => $source_url,
                        'keyword' => $keyword,
                        'error' => __('Anchor text not found or already linked', 'screaming-fixes'),
                    ];
                }
            }

            // Save if any changes were made
            if ($content !== $original_content) {
                wp_update_post([
                    'ID' => $post_id,
                    'post_content' => $content,
                ]);
            }
        }

        if ($offset + $batch_size >= count($links)) {
            $results['complete'] = true;
        }

        return $results;
    }

    // =========================================================================
    // BULK UPLOAD MODE
    // =========================================================================

    /**
     * AJAX: Process bulk upload CSV
     */
    public function ajax_process_csv() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

        $uploads = get_option('sf_pending_uploads', []);
        if (!isset($uploads[$upload_id])) {
            wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
        }

        $file_path = $uploads[$upload_id]['path'];

        // Parse CSV
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            wp_send_json_error(['message' => $parsed->get_error_message()]);
        }

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            wp_send_json_error([
                'message' => sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                ),
            ]);
        }

        // Detect columns
        $columns = $this->detect_bulk_columns($parsed['headers']);

        if (!$columns['source'] || !$columns['anchor'] || !$columns['target']) {
            wp_send_json_error([
                'message' => __('CSV must contain Source_URL, Anchor_Text, and Target_URL columns. For Orphan Pages format, use URL (target), Source_URL, and Anchor_Text.', 'screaming-fixes'),
            ]);
        }

        // Process rows
        $results = $this->process_bulk_csv_rows($parsed['rows'], $columns);

        // Store for later use
        $this->save_bulk_data($results);

        wp_send_json_success([
            'is_bulk_upload' => true,
            'ready_count' => $results['ready_count'],
            'manual_count' => $results['manual_count'],
            'skipped_count' => $results['skipped_count'],
            'duplicates_count' => $results['duplicates_count'],
            'total_rows' => $results['total_rows'],
            'ready_links' => array_slice($results['ready_links'], 0, 10),
            'manual_links' => array_slice($results['manual_links'], 0, 5),
            'skipped_links' => array_slice($results['skipped_links'], 0, 5),
        ]);
    }

    /**
     * Detect column mappings from headers
     *
     * Supports two formats:
     * 1. Standard format: Source_URL, Anchor_Text, Target_URL
     * 2. Orphan Pages format: URL (target), Source (ignored), Source_URL, Anchor_Text
     *
     * @param array $headers CSV headers
     * @return array Column mappings
     */
    private function detect_bulk_columns($headers) {
        $columns = [
            'source' => null,
            'anchor' => null,
            'target' => null,
        ];

        // Check if this is Orphan Pages format (has 'url' column as target)
        // The CSV parser normalizes 'source_url' to 'source', so we need to detect this case
        $has_url_target = in_array('url', $headers, true);

        // Source patterns - where the link will be added
        // Note: The CSV parser normalizes 'source_url' to 'source', so we include 'source'
        // when we have a 'url' column (Orphan Pages format)
        // Note: The CSV parser normalizes 'Source_URL' to 'source', so we must include 'source'
        $source_patterns = ['sourceurl', 'source', 'pageurl', 'page', 'fromurl', 'from'];

        // Anchor text patterns
        // Note: CSV parser may normalize 'anchor_text' to 'anchor text' (with space)
        $anchor_patterns = ['anchortext', 'anchor', 'linktext', 'anchor text'];

        // Target patterns - where the link points to
        // Note: 'url' is included for Orphan Pages format from Screaming Frog
        $target_patterns = ['targeturl', 'target', 'destination', 'tourl', 'linkurl', 'url'];

        foreach ($headers as $header) {
            $normalized = $this->normalize_column_name($header);
            $header_lower = strtolower(trim($header));

            // Check source patterns first (more specific)
            if (in_array($normalized, $source_patterns) && !$columns['source']) {
                $columns['source'] = $header_lower;
            }
            // Check anchor patterns
            elseif (in_array($normalized, $anchor_patterns) && !$columns['anchor']) {
                $columns['anchor'] = $header_lower;
            }
            // Check target patterns
            // For 'url', only match if it's exactly 'url' (Orphan Pages format)
            elseif (!$columns['target']) {
                if ($normalized === 'url' && $header_lower === 'url') {
                    $columns['target'] = $header_lower;
                } elseif (in_array($normalized, $target_patterns) && $normalized !== 'url') {
                    $columns['target'] = $header_lower;
                }
            }
        }

        return $columns;
    }

    /**
     * Process bulk CSV rows
     *
     * @param array $rows CSV rows
     * @param array $columns Column mappings
     * @return array Processed results
     */
    private function process_bulk_csv_rows($rows, $columns) {
        $ready_links = [];
        $manual_links = [];
        $skipped_links = [];
        $seen_combinations = [];
        $duplicates = 0;

        // Build URL map for quick lookups
        $url_map = $this->build_wp_url_map();

        foreach ($rows as $row) {
            $source_url = isset($row[$columns['source']]) ? trim($row[$columns['source']]) : '';
            $anchor_text = isset($row[$columns['anchor']]) ? trim($row[$columns['anchor']]) : '';
            $target_url = isset($row[$columns['target']]) ? trim($row[$columns['target']]) : '';

            // Skip empty rows
            if (empty($source_url) || empty($anchor_text) || empty($target_url)) {
                continue;
            }

            // Check for duplicates (use last occurrence)
            $combo_key = md5($source_url . '|' . $anchor_text . '|' . $target_url);
            if (isset($seen_combinations[$combo_key])) {
                $duplicates++;
                $ready_links = array_filter($ready_links, function($item) use ($combo_key) {
                    return $item['combo_key'] !== $combo_key;
                });
                $ready_links = array_values($ready_links);
            }
            $seen_combinations[$combo_key] = true;

            // Resolve source URL to post ID
            $normalized_source = $this->normalize_url($source_url);
            $source_post_id = 0;

            if (isset($url_map[$normalized_source])) {
                $source_post_id = $url_map[$normalized_source]['post_id'];
            } else {
                $source_post_id = $this->url_to_post_id($source_url);
            }

            // Categorize
            if (!$source_post_id) {
                $skipped_links[] = [
                    'source_url' => $source_url,
                    'anchor_text' => $anchor_text,
                    'target_url' => $target_url,
                    'status' => __('Skipped - Source URL not found', 'screaming-fixes'),
                ];
                continue;
            }

            // Check if source is archive/non-editable
            $category = $this->categorize_source_page($source_post_id, $source_url);

            if ($category['category'] === 'manual') {
                $manual_links[] = [
                    'source_url' => $source_url,
                    'source_post_id' => $source_post_id,
                    'anchor_text' => $anchor_text,
                    'target_url' => $target_url,
                    'status' => __('Manual Fix', 'screaming-fixes'),
                    'note' => $category['note'],
                ];
                continue;
            }

            // Check if anchor text exists in source content (in actual text, not shortcode attributes)
            $post = get_post($source_post_id);
            if (!$post) {
                $skipped_links[] = [
                    'source_url' => $source_url,
                    'anchor_text' => $anchor_text,
                    'target_url' => $target_url,
                    'status' => sprintf(__('Skipped - Anchor text "%s" not found on page', 'screaming-fixes'), $anchor_text),
                ];
                continue;
            }

            // Strip shortcode tags to avoid matching in shortcode attributes (e.g., menu_anchor="")
            $searchable_content = $this->strip_shortcode_tags($post->post_content);
            $searchable_text = wp_strip_all_tags($searchable_content);

            if (stripos($searchable_text, $anchor_text) === false) {
                $skipped_links[] = [
                    'source_url' => $source_url,
                    'anchor_text' => $anchor_text,
                    'target_url' => $target_url,
                    'status' => sprintf(__('Skipped - Anchor text "%s" not found on page', 'screaming-fixes'), $anchor_text),
                ];
                continue;
            }

            // Check if link already exists
            if ($this->link_already_exists($post->post_content, $anchor_text, $target_url)) {
                $skipped_links[] = [
                    'source_url' => $source_url,
                    'anchor_text' => $anchor_text,
                    'target_url' => $target_url,
                    'status' => __('Skipped - Link already exists', 'screaming-fixes'),
                ];
                continue;
            }

            // Ready to add
            $ready_links[] = [
                'source_url' => $source_url,
                'source_post_id' => $source_post_id,
                'source_title' => $post->post_title,
                'anchor_text' => $anchor_text,
                'target_url' => $target_url,
                'status' => __('Ready', 'screaming-fixes'),
                'combo_key' => $combo_key,
            ];
        }

        return [
            'ready_links' => $ready_links,
            'manual_links' => $manual_links,
            'skipped_links' => $skipped_links,
            'ready_count' => count($ready_links),
            'manual_count' => count($manual_links),
            'skipped_count' => count($skipped_links),
            'duplicates_count' => $duplicates,
            'total_rows' => count($rows),
        ];
    }

    /**
     * Check if link already exists from anchor to target
     *
     * @param string $content Post content
     * @param string $anchor_text Anchor text
     * @param string $target_url Target URL
     * @return bool True if link exists
     */
    private function link_already_exists($content, $anchor_text, $target_url) {
        $escaped_anchor = preg_quote($anchor_text, '/');

        // Normalize target URL for comparison
        $target_path = wp_parse_url($target_url, PHP_URL_PATH);
        $target_patterns = [$target_url];
        if ($target_path) {
            $target_patterns[] = $target_path;
            $target_patterns[] = rtrim($target_path, '/');
        }

        foreach ($target_patterns as $pattern) {
            if (empty($pattern)) {
                continue;
            }
            $escaped_url = preg_quote($pattern, '/');
            $regex = '/<a\s[^>]*href=["\'][^"\']*' . $escaped_url . '[^"\']*["\'][^>]*>[^<]*' . $escaped_anchor . '[^<]*<\/a>/is';

            if (preg_match($regex, $content)) {
                return true;
            }
        }

        return false;
    }

    /**
     * AJAX: Apply bulk links
     */
    public function ajax_apply_bulk_links() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $offset = isset($_POST['offset']) ? absint($_POST['offset']) : 0;
        $batch_size = isset($_POST['batch_size']) ? absint($_POST['batch_size']) : 50;

        // Clear accumulated results at start
        if ($offset === 0) {
            delete_transient('sf_ilb_bulk_accumulated_' . get_current_user_id());
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data) || empty($bulk_data['ready_links'])) {
            wp_send_json_error(['message' => __('No bulk data found.', 'screaming-fixes')]);
        }

        $links = $bulk_data['ready_links'];
        $results = $this->apply_bulk_links($links, $offset, $batch_size);

        // Accumulate results
        $accumulated = get_transient('sf_ilb_bulk_accumulated_' . get_current_user_id());
        if (!$accumulated) {
            $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
        }

        $accumulated['success'] += $results['success'];
        $accumulated['failed'] += $results['failed'];
        $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
        $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

        if ($results['complete']) {
            $final_results = [
                'is_bulk_upload' => true,
                'bulk_complete' => true,
                'added_links' => $accumulated['details'],
                'failed_links' => $accumulated['errors'],
                'success_count' => $accumulated['success'],
                'failed_count' => $accumulated['failed'],
                'manual_links' => $bulk_data['manual_links'] ?? [],
                'skipped_links' => $bulk_data['skipped_links'] ?? [],
            ];

            $this->save_bulk_data($final_results);
            delete_transient('sf_ilb_bulk_accumulated_' . get_current_user_id());

            // Log to activity log
            if ($accumulated['success'] > 0) {
                SF_Activity_Log::log('internal-link-builder', $accumulated['success']);
            }

            $results['total_success'] = $accumulated['success'];
            $results['total_failed'] = $accumulated['failed'];
            $results['all_details'] = $accumulated['details'];
        } else {
            set_transient('sf_ilb_bulk_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
        }

        $results['total_count'] = count($links);
        wp_send_json_success($results);
    }

    /**
     * Apply bulk links in batch
     *
     * @param array $links Links to add
     * @param int $offset Current offset
     * @param int $batch_size Batch size
     * @return array Results
     */
    private function apply_bulk_links($links, $offset, $batch_size) {
        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'details' => [],
            'errors' => [],
            'complete' => false,
            'next_offset' => $offset + $batch_size,
        ];

        $batch = array_slice($links, $offset, $batch_size);

        if (empty($batch)) {
            $results['complete'] = true;
            return $results;
        }

        // Group by source post
        $grouped = [];
        foreach ($batch as $link) {
            $post_id = $link['source_post_id'];
            if (!isset($grouped[$post_id])) {
                $grouped[$post_id] = [];
            }
            $grouped[$post_id][] = $link;
        }

        foreach ($grouped as $post_id => $post_links) {
            $post = get_post($post_id);
            if (!$post) {
                foreach ($post_links as $link) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'source_url' => $link['source_url'],
                        'anchor_text' => $link['anchor_text'],
                        'target_url' => $link['target_url'],
                        'error' => __('Post not found', 'screaming-fixes'),
                    ];
                }
                continue;
            }

            $content = $post->post_content;
            $original_content = $content;

            foreach ($post_links as $link) {
                $results['processed']++;

                $new_content = $this->add_link_to_content(
                    $content,
                    $link['anchor_text'],
                    $link['target_url']
                );

                if ($new_content !== $content) {
                    $content = $new_content;
                    $results['success']++;
                    $results['details'][] = [
                        'source_url' => $link['source_url'],
                        'source_title' => $post->post_title,
                        'anchor_text' => $link['anchor_text'],
                        'target_url' => $link['target_url'],
                        'status' => 'added',
                    ];
                } else {
                    $results['failed']++;
                    $results['errors'][] = [
                        'source_url' => $link['source_url'],
                        'anchor_text' => $link['anchor_text'],
                        'target_url' => $link['target_url'],
                        'error' => __('Failed to add link', 'screaming-fixes'),
                    ];
                }
            }

            // Save if changes made
            if ($content !== $original_content) {
                wp_update_post([
                    'ID' => $post_id,
                    'post_content' => $content,
                ]);
            }
        }

        if ($offset + $batch_size >= count($links)) {
            $results['complete'] = true;
        }

        return $results;
    }

    // =========================================================================
    // COMMON AJAX HANDLERS
    // =========================================================================

    /**
     * AJAX: Get current data state
     */
    public function ajax_get_data() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $discovery_data = $this->get_discovery_data();
        $bulk_data = $this->get_bulk_data();
        $results_data = $this->get_results_data();

        wp_send_json_success([
            'has_discovery' => !empty($discovery_data['opportunities']),
            'has_bulk' => !empty($bulk_data['ready_links']) || !empty($bulk_data['bulk_complete']),
            'has_results' => !empty($results_data['details']) || !empty($bulk_data['added_links']),
            'discovery' => $discovery_data,
            'bulk' => $bulk_data,
            'results' => $results_data,
        ]);
    }

    /**
     * AJAX: Download preview CSV
     */
    public function ajax_download_preview() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $bulk_data = $this->get_bulk_data();
        $discovery_data = $this->get_discovery_data();

        $lines = ['Source_URL,Anchor_Text,Target_URL,Status'];

        if (!empty($bulk_data['ready_links'])) {
            foreach ($bulk_data['ready_links'] as $link) {
                $lines[] = $this->csv_escape_row([
                    $link['source_url'],
                    $link['anchor_text'],
                    $link['target_url'],
                    'Ready',
                ]);
            }
            foreach ($bulk_data['manual_links'] ?? [] as $link) {
                $lines[] = $this->csv_escape_row([
                    $link['source_url'],
                    $link['anchor_text'],
                    $link['target_url'],
                    'Manual Fix - ' . ($link['note'] ?? ''),
                ]);
            }
            foreach ($bulk_data['skipped_links'] ?? [] as $link) {
                $lines[] = $this->csv_escape_row([
                    $link['source_url'],
                    $link['anchor_text'],
                    $link['target_url'],
                    $link['status'],
                ]);
            }
        } elseif (!empty($discovery_data['opportunities'])) {
            $target_url = $discovery_data['target_url'];
            foreach ($discovery_data['opportunities']['fixable'] ?? [] as $opp) {
                $lines[] = $this->csv_escape_row([
                    $opp['source_url'],
                    $opp['keyword'],
                    $target_url,
                    'Ready',
                ]);
            }
            foreach ($discovery_data['opportunities']['manual'] ?? [] as $opp) {
                $lines[] = $this->csv_escape_row([
                    $opp['source_url'],
                    $opp['keyword'],
                    $target_url,
                    'Manual Fix - ' . ($opp['note'] ?? ''),
                ]);
            }
        }

        wp_send_json_success([
            'csv' => implode("\n", $lines),
            'filename' => 'internal-links-preview-' . date('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Download results CSV
     */
    public function ajax_download_results() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results_data();
        $bulk_data = $this->get_bulk_data();

        $lines = ['Source_URL,Anchor_Text,Target_URL,Status,Notes'];

        // Added links
        $details = $results['details'] ?? $bulk_data['added_links'] ?? [];
        foreach ($details as $link) {
            $lines[] = $this->csv_escape_row([
                $link['source_url'],
                $link['anchor_text'] ?? $link['keyword'] ?? '',
                $link['target_url'],
                'Added',
                '',
            ]);
        }

        // Failed/skipped
        $errors = $results['errors'] ?? $bulk_data['failed_links'] ?? [];
        foreach ($errors as $link) {
            $lines[] = $this->csv_escape_row([
                $link['source_url'],
                $link['anchor_text'] ?? $link['keyword'] ?? '',
                $link['target_url'] ?? '',
                'Failed',
                $link['error'] ?? '',
            ]);
        }

        wp_send_json_success([
            'csv' => implode("\n", $lines),
            'filename' => 'internal-links-results-' . date('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Clear all data
     */
    public function ajax_clear_data() {
        check_ajax_referer('sf_ilb_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $this->clear_all_data();

        wp_send_json_success(['message' => __('Data cleared.', 'screaming-fixes')]);
    }
}
