<?php
/**
 * Internal Link Builder - Tab Content
 *
 * @package Screaming_Fixes
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Helper function to truncate URLs for display
 */
if (!function_exists('sf_ilb_truncate_url')) {
    function sf_ilb_truncate_url($url, $max_length = 50) {
        if (strlen($url) <= $max_length) {
            return $url;
        }

        $parsed = wp_parse_url($url);
        $path = isset($parsed['path']) ? $parsed['path'] : '';

        if (strlen($path) <= $max_length) {
            return $path;
        }

        return substr($path, 0, $max_length - 3) . '...';
    }
}

/**
 * Helper function to highlight keyword in context
 */
if (!function_exists('sf_ilb_highlight_keyword')) {
    function sf_ilb_highlight_keyword($context, $keyword) {
        $pattern = '/(' . preg_quote($keyword, '/') . ')/i';
        return preg_replace($pattern, '<mark class="sf-keyword-highlight">$1</mark>', esc_html($context));
    }
}

// Get stored data
$user_id = get_current_user_id();
$discovery_data = get_transient('sf_ilb_discovery_' . $user_id);
$bulk_data = get_transient('sf_ilb_bulk_' . $user_id);
$results_data = get_transient('sf_ilb_results_' . $user_id);

$has_discovery = !empty($discovery_data['opportunities']);
// Check if we have any bulk data (ready, manual, or skipped links)
$has_bulk = !empty($bulk_data['ready_links']) || !empty($bulk_data['manual_links']) || !empty($bulk_data['skipped_links']) || !empty($bulk_data['bulk_complete']);
$has_results = !empty($results_data['details']) || !empty($bulk_data['added_links']);
$bulk_complete = !empty($bulk_data['bulk_complete']);

// Determine current state
$show_input = !$has_discovery && !$has_bulk && !$has_results;
$show_opportunities = $has_discovery && !$has_results;
$show_bulk_confirmation = $has_bulk && !$bulk_complete;
$show_results = $has_results || $bulk_complete;
?>

<div class="sf-internal-link-builder-module">

    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#128279;</span>
            <h2><?php esc_html_e('Internal Link Builder', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Build internal links to boost priority pages', 'screaming-fixes'); ?></span>
        </div>
    </div>

    <!-- Instructions Section -->
    <?php include SF_PLUGIN_DIR . 'modules/internal-link-builder/views/instructions.php'; ?>

    <!-- STEP 1: URL Input -->
    <div class="sf-ilb-input-section" id="sf-ilb-input" style="<?php echo $show_input ? '' : 'display: none;'; ?>">

        <!-- Build Internal Links Section -->
        <div class="sf-upload-section">
            <h3>
                <span class="dashicons dashicons-search"></span>
                <?php esc_html_e('Build Internal Links to a Page', 'screaming-fixes'); ?>
            </h3>
            <p class="sf-section-description"><?php esc_html_e('Enter a URL you want more internal links pointing to. Perfect for boosting priority pages or fixing orphan pages.', 'screaming-fixes'); ?></p>

            <div class="sf-ilb-url-input-wrapper">
                <input type="url"
                       id="sf-ilb-priority-url"
                       class="sf-ilb-url-input"
                       placeholder="https://mysite.com/best-coffee-shops/"
                       autocomplete="off">
                <button type="button" class="sf-button sf-button-primary" id="sf-ilb-extract-btn">
                    <?php esc_html_e('Find Link Opportunities', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <div class="sf-ilb-divider">
            <span><?php esc_html_e('or', 'screaming-fixes'); ?></span>
        </div>

        <!-- Bulk Upload Section -->
        <div class="sf-upload-section">
            <h3>
                <span class="dashicons dashicons-upload"></span>
                <?php esc_html_e('Bulk Upload Mode', 'screaming-fixes'); ?>
            </h3>
            <?php
            $args = [
                'module' => 'internal-link-builder',
                'title' => __('Drop your Internal Links CSV file', 'screaming-fixes'),
                'description' => __('Drag & drop CSV here or click to browse', 'screaming-fixes'),
                'auto_detect' => false,
                'accepted_types' => __('CSV with Source_URL, Anchor_Text, Target_URL columns', 'screaming-fixes'),
                'max_rows' => 5000,
            ];
            include SF_PLUGIN_DIR . 'admin/views/partials/upload-box.php';
            ?>
        </div>
    </div>

    <!-- STEP 2: Link Text Options -->
    <div class="sf-ilb-keywords-section" id="sf-ilb-keywords" style="display: none;">
        <div class="sf-section-card">
            <div class="sf-section-header">
                <div class="sf-ilb-header-row">
                    <h3><?php esc_html_e('Internal Link Text Options', 'screaming-fixes'); ?></h3>
                    <span class="sf-ilb-selection-count"><span id="sf-ilb-selected-count" class="sf-ilb-count-number">0</span> <?php esc_html_e('of 10 selected', 'screaming-fixes'); ?></span>
                </div>
                <p class="sf-ilb-helper-text"><?php esc_html_e("We'll find pages containing these phrases and suggest internal link opportunities.", 'screaming-fixes'); ?></p>
                <p class="sf-ilb-target-url-display"></p>
            </div>

            <div class="sf-ilb-keywords-list" id="sf-ilb-keywords-list">
                <!-- Populated by JavaScript -->
            </div>

            <div class="sf-ilb-add-keyword-row">
                <input type="text" id="sf-ilb-new-keyword" placeholder="<?php esc_attr_e('Add phrase...', 'screaming-fixes'); ?>">
                <button type="button" class="sf-button sf-button-secondary" id="sf-ilb-add-keyword-btn">
                    <span class="dashicons dashicons-plus-alt2"></span>
                    <?php esc_html_e('Add', 'screaming-fixes'); ?>
                </button>
            </div>

            <div class="sf-ilb-filter-options">
                <label class="sf-ilb-filter-checkbox">
                    <input type="checkbox" id="sf-ilb-exclude-headings" checked>
                    <span class="sf-ilb-filter-label"><?php esc_html_e('Exclude matches in headings (H1-H6)', 'screaming-fixes'); ?></span>
                </label>
            </div>

            <div class="sf-ilb-keywords-footer">
                <div class="sf-ilb-keywords-actions">
                    <button type="button" class="sf-button sf-button-secondary" id="sf-ilb-back-btn">
                        <?php esc_html_e('Back', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-primary" id="sf-ilb-scan-btn">
                        <?php esc_html_e('Find Link Opportunities', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- STEP 3: Opportunities Display (Discovery Mode) -->
    <?php if ($show_opportunities): ?>
    <?php
    // Calculate keyword statistics
    $keyword_stats = $discovery_data['opportunities']['keyword_stats'] ?? [];
    $keywords_with_matches = 0;
    $keywords_no_matches = 0;
    foreach ($keyword_stats as $keyword => $count) {
        if ($count > 0) {
            $keywords_with_matches++;
        } else {
            $keywords_no_matches++;
        }
    }
    ?>
    <div class="sf-ilb-opportunities-section" id="sf-ilb-opportunities" data-target-url="<?php echo esc_attr($discovery_data['target_url']); ?>">
        <div class="sf-section-card">
            <div class="sf-section-header">
                <h3>
                    <?php
                    $total_opps = count($discovery_data['opportunities']['fixable']) + count($discovery_data['opportunities']['manual']);
                    printf(
                        esc_html__('Found %d link opportunities for %s', 'screaming-fixes'),
                        $total_opps,
                        '<code>' . esc_html(wp_parse_url($discovery_data['target_url'], PHP_URL_PATH)) . '</code>'
                    );
                    ?>
                </h3>
            </div>

            <!-- Stats -->
            <div class="sf-bulk-stats">
                <div class="sf-bulk-stat sf-bulk-stat-matched">
                    <div class="sf-bulk-stat-icon">&#10003;</div>
                    <div class="sf-bulk-stat-content">
                        <span class="sf-bulk-stat-value"><?php echo count($discovery_data['opportunities']['fixable']); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('Link opportunities', 'screaming-fixes'); ?></span>
                    </div>
                </div>
                <div class="sf-bulk-stat sf-bulk-stat-pages">
                    <div class="sf-bulk-stat-icon">&#128196;</div>
                    <div class="sf-bulk-stat-content">
                        <span class="sf-bulk-stat-value"><?php echo count($discovery_data['opportunities']['fixable']) + count($discovery_data['opportunities']['manual']); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('Pages matched', 'screaming-fixes'); ?></span>
                    </div>
                </div>
                <?php if ($keywords_no_matches > 0): ?>
                <div class="sf-bulk-stat sf-bulk-stat-nomatch">
                    <div class="sf-bulk-stat-icon">&#10005;</div>
                    <div class="sf-bulk-stat-content">
                        <span class="sf-bulk-stat-value"><?php echo $keywords_no_matches; ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('Keywords with no matches', 'screaming-fixes'); ?></span>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <?php
            // Show notice if limit was reached
            $limit_reached = !empty($discovery_data['opportunities']['limit_reached']);
            if ($limit_reached):
            ?>
            <div class="sf-notice sf-notice-info" style="margin: 15px 0;">
                <p>
                    <span class="dashicons dashicons-info" style="color: #0073aa;"></span>
                    <?php
                    printf(
                        esc_html__('Showing first %d link opportunities. This limit helps ensure fast scans on large sites.', 'screaming-fixes'),
                        30
                    );
                    ?>
                </p>
            </div>
            <?php endif; ?>

            <!-- Keyword Search Results Breakdown (Collapsible) -->
            <?php if (!empty($keyword_stats)): ?>
            <div class="sf-ilb-keyword-breakdown">
                <button type="button" class="sf-ilb-breakdown-toggle" aria-expanded="false">
                    <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
                    <?php esc_html_e('Keyword Search Results', 'screaming-fixes'); ?>
                    <span class="sf-breakdown-summary">
                        (<?php printf(esc_html__('%d matched, %d no results', 'screaming-fixes'), $keywords_with_matches, $keywords_no_matches); ?>)
                    </span>
                </button>
                <div class="sf-ilb-breakdown-content" style="display: none;">
                    <div class="sf-ilb-keyword-results-list">
                        <?php foreach ($keyword_stats as $keyword => $count): ?>
                        <div class="sf-ilb-keyword-result <?php echo $count > 0 ? 'sf-ilb-keyword-has-matches' : 'sf-ilb-keyword-no-matches'; ?>">
                            <span class="sf-ilb-keyword-result-icon"><?php echo $count > 0 ? '&#10003;' : '&#10005;'; ?></span>
                            <span class="sf-ilb-keyword-result-text">"<?php echo esc_html($keyword); ?>"</span>
                            <span class="sf-ilb-keyword-result-count">
                                <?php
                                if ($count > 0) {
                                    printf(esc_html(_n('%d match', '%d matches', $count, 'screaming-fixes')), $count);
                                } else {
                                    esc_html_e('0 matches', 'screaming-fixes');
                                }
                                ?>
                            </span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Fixable Opportunities Table -->
            <?php if (!empty($discovery_data['opportunities']['fixable'])): ?>
            <div class="sf-ilb-table-section">
                <table class="sf-bulk-preview-table sf-ilb-opportunities-table">
                    <thead>
                        <tr>
                            <th class="sf-col-checkbox">
                                <input type="checkbox" id="sf-ilb-select-all" class="sf-ilb-checkbox" checked>
                            </th>
                            <th><?php esc_html_e('Source Page', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Keyword Found', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Context', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($discovery_data['opportunities']['fixable'] as $index => $opp): ?>
                        <?php
                        // Prepare all matches data for JavaScript
                        $all_matches_json = wp_json_encode($opp['all_matches']);
                        $primary_keyword = $opp['all_matches'][0]['keyword'];
                        ?>
                        <tr class="sf-ilb-opportunity-row"
                            data-index="<?php echo $index; ?>"
                            data-all-matches="<?php echo esc_attr($all_matches_json); ?>">
                            <td class="sf-col-checkbox">
                                <input type="checkbox" class="sf-ilb-checkbox sf-ilb-opp-select" checked
                                       data-post-id="<?php echo esc_attr($opp['source_post_id']); ?>"
                                       data-keyword="<?php echo esc_attr($primary_keyword); ?>"
                                       data-source-url="<?php echo esc_attr($opp['source_url']); ?>">
                            </td>
                            <td class="sf-bulk-preview-url">
                                <a href="<?php echo esc_url($opp['source_url']); ?>" target="_blank" title="<?php echo esc_attr($opp['source_url']); ?>">
                                    <?php echo esc_html(sf_ilb_truncate_url($opp['source_url'], 40)); ?>
                                </a>
                            </td>
                            <td class="sf-ilb-keyword-cell">
                                <div class="sf-ilb-keyword-badges">
                                    <?php foreach ($opp['all_matches'] as $match_index => $match): ?>
                                    <button type="button"
                                            class="sf-ilb-keyword-badge <?php echo $match_index === 0 ? 'sf-ilb-keyword-selected' : ''; ?>"
                                            data-keyword="<?php echo esc_attr($match['keyword']); ?>"
                                            data-context="<?php echo esc_attr($match['context']); ?>"
                                            title="<?php esc_attr_e('Click to use this keyword', 'screaming-fixes'); ?>">
                                        <?php echo esc_html($match['keyword']); ?>
                                    </button>
                                    <?php endforeach; ?>
                                </div>
                            </td>
                            <td class="sf-ilb-context-cell">
                                <?php echo sf_ilb_highlight_keyword($opp['all_matches'][0]['context'], $opp['all_matches'][0]['keyword']); ?>
                            </td>
                            <td class="sf-ilb-status-cell">
                                <span class="sf-bulk-status sf-bulk-status-ready"><?php esc_html_e('Ready', 'screaming-fixes'); ?></span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>

            <!-- Manual Fix Section (Collapsible) -->
            <?php if (!empty($discovery_data['opportunities']['manual'])): ?>
            <div class="sf-ilb-manual-section">
                <button type="button" class="sf-section-toggle" aria-expanded="false">
                    <span class="sf-section-badge sf-badge-manual">&#9888;</span>
                    <?php printf(esc_html__('Manual Fix Required (%d)', 'screaming-fixes'), count($discovery_data['opportunities']['manual'])); ?>
                    <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
                </button>
                <div class="sf-ilb-manual-content" style="display: none;">
                    <p class="sf-ilb-manual-explanation">
                        <?php esc_html_e('These pages can\'t be edited automatically because they don\'t have editable post content in WordPress (category archives, tag archives, author archives, etc.).', 'screaming-fixes'); ?>
                    </p>
                    <table class="sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Keyword', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Reason', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($discovery_data['opportunities']['manual'] as $opp): ?>
                            <tr>
                                <td class="sf-bulk-preview-url"><?php echo esc_html(sf_ilb_truncate_url($opp['source_url'], 40)); ?></td>
                                <td><?php echo esc_html($opp['keyword']); ?></td>
                                <td><?php echo esc_html($opp['note']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Actions -->
            <div class="sf-bulk-actions">
                <button type="button" class="sf-bulk-download-preview" id="sf-ilb-download-preview">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Download All Opportunities CSV', 'screaming-fixes'); ?>
                </button>
                <div class="sf-bulk-buttons">
                    <button type="button" class="sf-bulk-clear-btn" id="sf-ilb-clear">
                        <span class="dashicons dashicons-dismiss"></span>
                        <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-bulk-confirm" id="sf-ilb-add-selected">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <?php esc_html_e('Add Selected Links', 'screaming-fixes'); ?>
                        (<span id="sf-ilb-selected-count-action"><?php echo count($discovery_data['opportunities']['fixable']); ?></span>)
                    </button>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Bulk Upload Confirmation -->
    <?php if ($show_bulk_confirmation): ?>
    <div class="sf-bulk-confirmation" id="sf-ilb-bulk-confirmation">
        <div class="sf-bulk-header">
            <div class="sf-bulk-icon">&#128279;</div>
            <div>
                <h3><?php esc_html_e('Bulk Internal Link Upload', 'screaming-fixes'); ?></h3>
            </div>
        </div>

        <div class="sf-bulk-stats">
            <div class="sf-bulk-stat sf-bulk-stat-matched">
                <div class="sf-bulk-stat-icon">&#10003;</div>
                <div class="sf-bulk-stat-content">
                    <span class="sf-bulk-stat-value"><?php echo $bulk_data['ready_count']; ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('links ready to add', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php if ($bulk_data['manual_count'] > 0): ?>
            <div class="sf-bulk-stat sf-bulk-stat-notfound">
                <div class="sf-bulk-stat-icon">&#9888;</div>
                <div class="sf-bulk-stat-content">
                    <span class="sf-bulk-stat-value"><?php echo $bulk_data['manual_count']; ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('manual fix required', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php endif; ?>
            <?php if ($bulk_data['skipped_count'] > 0): ?>
            <div class="sf-bulk-stat sf-bulk-stat-skipped">
                <div class="sf-bulk-stat-icon">&#10005;</div>
                <div class="sf-bulk-stat-content">
                    <span class="sf-bulk-stat-value"><?php echo $bulk_data['skipped_count']; ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('skipped (see details)', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <?php if ($bulk_data['duplicates_count'] > 0): ?>
        <div class="sf-bulk-warning">
            <span class="dashicons dashicons-warning"></span>
            <span class="sf-bulk-warning-text">
                <?php printf(esc_html__('%d duplicate entries detected - using last occurrence for each.', 'screaming-fixes'), $bulk_data['duplicates_count']); ?>
            </span>
        </div>
        <?php endif; ?>

        <?php if ($bulk_data['ready_count'] > 500): ?>
        <div class="sf-bulk-warning">
            <span class="dashicons dashicons-warning"></span>
            <span class="sf-bulk-warning-text">
                <?php esc_html_e('Large file detected. Processing may take a few minutes.', 'screaming-fixes'); ?>
            </span>
        </div>
        <?php endif; ?>

        <!-- Preview Table -->
        <?php if ($bulk_data['ready_count'] > 0): ?>
        <div class="sf-bulk-preview">
            <div class="sf-bulk-preview-header">
                <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?>
                    <span class="sf-preview-count">(<?php printf(esc_html__('showing %d of %d', 'screaming-fixes'), min(10, $bulk_data['ready_count']), $bulk_data['ready_count']); ?>)</span>
                </h4>
            </div>
            <table class="sf-bulk-preview-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Anchor Text', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Target', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_slice($bulk_data['ready_links'], 0, 10) as $link): ?>
                    <tr>
                        <td class="sf-bulk-preview-url" title="<?php echo esc_attr($link['source_url']); ?>">
                            <a href="<?php echo esc_url($link['source_url']); ?>" target="_blank">
                                <?php echo esc_html(sf_ilb_truncate_url($link['source_url'], 35)); ?>
                            </a>
                        </td>
                        <td><?php echo esc_html($link['anchor_text']); ?></td>
                        <td class="sf-bulk-preview-url" title="<?php echo esc_attr($link['target_url']); ?>">
                            <a href="<?php echo esc_url($link['target_url']); ?>" target="_blank">
                                <?php echo esc_html(sf_ilb_truncate_url($link['target_url'], 35)); ?>
                            </a>
                        </td>
                        <td><span class="sf-bulk-status sf-bulk-status-ready"><?php esc_html_e('Ready', 'screaming-fixes'); ?></span></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php else: ?>
        <!-- No ready links - show skipped details -->
        <div class="sf-bulk-no-ready">
            <div class="sf-bulk-no-ready-icon">&#9888;</div>
            <h4><?php esc_html_e('No links ready to add', 'screaming-fixes'); ?></h4>
            <p><?php esc_html_e('All rows from your CSV were either skipped or require manual fixing. Check the details below.', 'screaming-fixes'); ?></p>

            <?php if (!empty($bulk_data['skipped_links'])): ?>
            <div class="sf-bulk-skipped-details">
                <h5><?php esc_html_e('Skipped Links', 'screaming-fixes'); ?></h5>
                <table class="sf-bulk-preview-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Anchor Text', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Reason', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_slice($bulk_data['skipped_links'], 0, 10) as $link): ?>
                        <tr>
                            <td class="sf-bulk-preview-url" title="<?php echo esc_attr($link['source_url']); ?>">
                                <?php echo esc_html(sf_ilb_truncate_url($link['source_url'], 35)); ?>
                            </td>
                            <td><?php echo esc_html($link['anchor_text']); ?></td>
                            <td><span class="sf-bulk-status sf-bulk-status-skipped"><?php echo esc_html($link['status']); ?></span></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <!-- Actions -->
        <div class="sf-bulk-actions">
            <button type="button" class="sf-bulk-download-preview">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
            </button>
            <div class="sf-bulk-buttons">
                <button type="button" class="sf-bulk-clear-btn sf-ilb-bulk-clear">
                    <span class="dashicons dashicons-dismiss"></span>
                    <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-bulk-confirm sf-ilb-bulk-confirm" data-count="<?php echo esc_attr($bulk_data['ready_count']); ?>">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <?php esc_html_e('Confirm', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Progress Modal -->
    <div class="sf-bulk-progress-modal" id="sf-ilb-progress-modal" style="display: none;">
        <div class="sf-bulk-progress-content">
            <div class="sf-bulk-progress-header">
                <div class="sf-bulk-progress-icon">&#8635;</div>
                <h3 class="sf-bulk-progress-title"><?php esc_html_e('Adding internal links...', 'screaming-fixes'); ?></h3>
            </div>
            <div class="sf-bulk-progress-bar-wrapper">
                <div class="sf-bulk-progress-bar">
                    <div class="sf-progress-fill" style="width: 0%;"></div>
                </div>
            </div>
            <div class="sf-bulk-progress-stats">
                <span class="sf-ilb-progress-current">0</span> / <span class="sf-ilb-progress-total">0</span>
                (<span class="sf-ilb-progress-percent">0</span>%)
            </div>
            <div class="sf-bulk-current-url-wrapper">
                <div class="sf-bulk-current-url-label"><?php esc_html_e('Current:', 'screaming-fixes'); ?></div>
                <div class="sf-bulk-current-url sf-ilb-current-item"></div>
            </div>
        </div>
    </div>

    <!-- Results Section -->
    <?php if ($show_results): ?>
    <?php
    $success_count = $results_data['success'] ?? $bulk_data['success_count'] ?? 0;
    $failed_count = ($results_data['failed'] ?? 0) + ($results_data['skipped'] ?? 0) + ($bulk_data['failed_count'] ?? 0);
    $details = $results_data['details'] ?? $bulk_data['added_links'] ?? [];
    ?>
    <div class="sf-bulk-complete" id="sf-ilb-results">
        <div class="sf-bulk-complete-header">
            <div class="sf-bulk-complete-icon"><?php echo $failed_count > 0 ? '&#9888;' : '&#10003;'; ?></div>
            <h3>
                <?php
                if ($failed_count > 0) {
                    printf(esc_html__('%d internal links added successfully. %d skipped.', 'screaming-fixes'), $success_count, $failed_count);
                } else {
                    printf(esc_html__('%d internal links added successfully.', 'screaming-fixes'), $success_count);
                }
                ?>
            </h3>
        </div>

        <div class="sf-bulk-results-summary">
            <div class="sf-bulk-result-stat sf-bulk-result-stat-success">
                <span class="sf-bulk-result-value"><?php echo $success_count; ?></span>
                <span class="sf-bulk-result-label"><?php esc_html_e('Added', 'screaming-fixes'); ?></span>
            </div>
            <?php if ($failed_count > 0): ?>
            <div class="sf-bulk-result-stat sf-bulk-result-stat-failed">
                <span class="sf-bulk-result-value"><?php echo $failed_count; ?></span>
                <span class="sf-bulk-result-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
            </div>
            <?php endif; ?>
        </div>

        <div class="sf-bulk-complete-actions">
            <button type="button" class="sf-bulk-download-results sf-ilb-download-results">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
            </button>
            <button type="button" class="sf-bulk-new-upload sf-ilb-clear-results">
                <span class="dashicons dashicons-upload"></span>
                <?php esc_html_e('Start Over', 'screaming-fixes'); ?>
            </button>
        </div>

        <?php if (!empty($details)): ?>
        <div class="sf-bulk-results-table-wrapper">
            <h4><?php esc_html_e('Links Added', 'screaming-fixes'); ?>
                <span class="sf-preview-count">(<?php printf(esc_html__('showing %d of %d', 'screaming-fixes'), min(10, count($details)), count($details)); ?>)</span>
            </h4>
            <table class="sf-bulk-results-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Anchor Text', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Target', 'screaming-fixes'); ?></th>
                        <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_slice($details, 0, 10) as $link): ?>
                    <tr>
                        <td class="sf-bulk-result-url" title="<?php echo esc_attr($link['source_url']); ?>">
                            <a href="<?php echo esc_url($link['source_url']); ?>" target="_blank">
                                <?php echo esc_html(sf_ilb_truncate_url($link['source_url'], 35)); ?>
                            </a>
                        </td>
                        <td><?php echo esc_html($link['anchor_text'] ?? $link['keyword'] ?? ''); ?></td>
                        <td class="sf-bulk-result-url" title="<?php echo esc_attr($link['target_url']); ?>">
                            <?php echo esc_html(sf_ilb_truncate_url($link['target_url'], 35)); ?>
                        </td>
                        <td><span class="sf-bulk-status sf-bulk-status-success">&#10003; <?php esc_html_e('Added', 'screaming-fixes'); ?></span></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
    <?php endif; ?>

</div>
