/**
 * Screaming Fixes - Meta Description Module
 *
 * Handles UI interactions, AI suggestions, filtering, and applying meta description fixes
 */

(function($) {
    'use strict';

    // Namespace
    window.ScreamingFixes = window.ScreamingFixes || {};

    /**
     * Meta Description Handler
     */
    ScreamingFixes.MetaDescription = {
        // Store current data
        data: null,
        selectedRows: [],
        currentFilter: 'all',
        bulkData: null,
        bulkProcessing: false,

        // Pagination state for each section
        pagination: {
            fixable: { currentPage: 1, perPage: 100, totalItems: 0 },
            manual: { currentPage: 1, perPage: 100, totalItems: 0 },
            skipped: { currentPage: 1, perPage: 100, totalItems: 0 },
            fixed: { currentPage: 1, perPage: 100, totalItems: 0 },
            failed: { currentPage: 1, perPage: 100, totalItems: 0 }
        },

        // Fun loading messages for CSV processing
        csvLoadingMessages: [
            "📄 Parsing CSV file...",
            "🔍 Analyzing meta descriptions...",
            "📊 Evaluating SEO data points...",
            "🔧 Rejiggering inputs...",
            "🎯 Bouncing outputs...",
            "🧮 Counting character lengths...",
            "🏷️ Categorizing issue types...",
            "🔗 Matching URLs to pages...",
            "📝 Reading description data...",
            "⚡ Processing at lightning speed...",
            "🎪 Juggling meta tags...",
            "🔬 Inspecting for duplicates...",
            "📏 Measuring optimal lengths...",
            "🧹 Cleaning up the data...",
            "🎨 Painting the results...",
            "🚀 Almost there...",
            "☕ This is taking a moment...",
            "🎯 Targeting improvements...",
            "✨ Polishing the final details..."
        ],

        /**
         * Initialize the module
         */
        init: function() {
            this.bindEvents();
            this.bindFixedSectionEvents();
            this.loadExistingData();
            this.initCharCounters();
            this.initPagination();
        },

        /**
         * Bind all event handlers
         */
        bindEvents: function() {
            var self = this;

            // Select all checkbox
            $(document).on('change', '#sf-select-all', function() {
                var checked = $(this).prop('checked');
                $('.sf-description-row:visible .sf-row-select').prop('checked', checked);
                self.updateSelectedCount();
            });

            // Individual row selection
            $(document).on('change', '.sf-row-select', function() {
                self.updateSelectedCount();
            });

            // New description input change
            $(document).on('input', '.sf-new-desc-input', function() {
                self.updateCharCounter($(this));
                self.updateApplyButtonState();
                self.updateSaveButtonState($(this));
            });

            // AI suggest single description
            $(document).on('click', '.sf-ai-suggest-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-description-row');
                self.getAISuggestion(row, btn);
            });

            // Save single description
            $(document).on('click', '.sf-save-desc-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-description-row');
                self.saveSingleDescription(row, btn);
            });

            // Apply fixes
            $(document).on('click', '.sf-apply-fixes', function(e) {
                e.preventDefault();
                self.applyFixes();
            });

            // Clickable issue stats - toggle filter on click
            $(document).on('click', '.sf-issue-stat', function() {
                var $stat = $(this);
                if ($stat.hasClass('sf-filter-disabled')) {
                    return;
                }
                var filter = $stat.data('filter');

                // If already active, clear the filter (show all)
                if ($stat.hasClass('active')) {
                    self.filterByIssue('all');
                } else {
                    self.filterByIssue(filter);
                }
            });

            // Dismiss duplicate info notice (session-based)
            $(document).on('click', '.sf-duplicate-info-dismiss', function() {
                $('.sf-duplicate-info-notice').slideUp(200);
                // Store dismissal in session storage
                if (typeof sessionStorage !== 'undefined') {
                    sessionStorage.setItem('sf_duplicate_notice_dismissed', 'true');
                }
            });

            // Section toggles (Fixable, Manual, Skipped, Fixed, Failed)
            $(document).on('click', '.sf-section-toggle', function() {
                var section = $(this).closest('.sf-results-section');
                var content = section.find('.sf-fixable-content, .sf-manual-content, .sf-skipped-content, .sf-fixed-content, .sf-failed-content');
                var expanded = $(this).attr('aria-expanded') === 'true';

                content.slideToggle(200);
                $(this).attr('aria-expanded', !expanded);
            });

            // Instructions toggle
            $(document).on('click', '.sf-instructions-header', function() {
                var $box = $(this).closest('.sf-instructions-box');
                var $content = $box.find('.sf-instructions-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Verify info box toggle (inside Fixed Descriptions section)
            $(document).on('click', '.sf-verify-info-header', function() {
                var $box = $(this).closest('.sf-verify-info-box');
                var $content = $box.find('.sf-verify-info-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Failed info box toggle (inside Unable to Update section)
            $(document).on('click', '.sf-failed-info-header', function() {
                var $box = $(this).closest('.sf-failed-info-box');
                var $content = $box.find('.sf-failed-info-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Retry failed update button
            $(document).on('click', '.sf-retry-btn', function(e) {
                e.preventDefault();
                var $btn = $(this);
                var $row = $btn.closest('.sf-failed-row');
                self.retryFailedUpdate($row, $btn);
            });

            // Pagination: Previous button
            $(document).on('click', '.sf-page-prev:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage - 1);
            });

            // Pagination: Next button
            $(document).on('click', '.sf-page-next:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage + 1);
            });

            // Pagination: Page number buttons
            $(document).on('click', '.sf-page-number:not(.active):not(.sf-page-ellipsis)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                var page = parseInt($(this).data('page'), 10);
                self.goToPage(section, page);
            });

            // Close modal
            $(document).on('click', '.sf-modal-close, .sf-modal', function(e) {
                if (e.target === this) {
                    $('.sf-modal').hide();
                }
            });

            // Listen for CSV upload events
            $(document).on('sf:csv-uploaded', function(e, data, box) {
                if (box.data('module') === 'meta-description') {
                    self.processUploadedCSV(data);
                }
            });

            // Clear data button
            $(document).on('click', '.sf-clear-data-btn', function(e) {
                e.preventDefault();
                self.clearData();
            });

            // Expandable description toggle - click on the description text or expand button
            $(document).on('click', '.sf-description-expandable', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $(this).toggleClass('sf-expanded');
            });

            // Prevent expand button from double-triggering
            $(document).on('click', '.sf-expand-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $(this).closest('.sf-description-expandable').toggleClass('sf-expanded');
            });

            // Bulk Upload Handlers
            // Expand/collapse bulk preview description rows
            $(document).on('click', '.sf-desc-expandable', function(e) {
                // Don't toggle if clicking a link
                if ($(e.target).closest('a').length) return;

                var $row = $(this);
                var rowIndex = $row.data('row-index');
                var $detailRow = $('tr.sf-bulk-detail-row[data-for-row="' + rowIndex + '"]');

                if ($row.hasClass('sf-expanded')) {
                    $row.removeClass('sf-expanded');
                    $detailRow.slideUp(200);
                } else {
                    $row.addClass('sf-expanded');
                    $detailRow.slideDown(200);
                }
            });

            // Confirm bulk update
            $(document).on('click', '.sf-meta-description-module .sf-bulk-confirm', function(e) {
                e.preventDefault();
                self.processBulkUpdate();
            });

            // Clear bulk update (replaces cancel)
            $(document).on('click', '.sf-meta-description-module .sf-bulk-clear-btn', function(e) {
                e.preventDefault();
                self.clearBulkData();
            });

            // Download preview CSV
            $(document).on('click', '.sf-meta-description-module .sf-bulk-download-preview', function(e) {
                e.preventDefault();
                self.downloadBulkPreview();
            });

            // Download results CSV
            $(document).on('click', '.sf-meta-description-module .sf-bulk-download-results', function(e) {
                e.preventDefault();
                self.downloadBulkResults();
            });

            // New upload button (after bulk complete)
            $(document).on('click', '.sf-meta-description-module .sf-bulk-new-upload', function(e) {
                e.preventDefault();
                self.startNewUpload();
            });

            // Clear button on bulk complete section
            $(document).on('click', '.sf-meta-description-module .sf-bulk-complete-clear', function(e) {
                e.preventDefault();
                self.clearBulkCompleteData();
            });

            // Fixed section - Download results CSV (legacy button)
            $(document).on('click', '.sf-meta-description-module .sf-fixed-download-results', function(e) {
                e.preventDefault();
                self.downloadFixedResults();
            });

            // Fixed section - Download CSV button (new primary button)
            $(document).on('click', '.sf-meta-description-module .sf-fixed-download-csv', function(e) {
                e.preventDefault();
                self.downloadFixedResults();
            });

            // Fixed section - Clear button
            $(document).on('click', '.sf-meta-description-module .sf-fixed-clear-btn', function(e) {
                e.preventDefault();
                self.clearData();
            });
        },

        /**
         * Initialize character counters for all textareas
         */
        initCharCounters: function() {
            var self = this;
            $('.sf-new-desc-input').each(function() {
                self.updateCharCounter($(this));
            });
        },

        /**
         * Update character counter for a textarea
         */
        updateCharCounter: function($textarea) {
            var length = $textarea.val().length;
            var $row = $textarea.closest('.sf-description-row');
            var $counter = $row.find('.sf-char-counter');
            var $current = $counter.find('.sf-char-current');
            var $status = $counter.find('.sf-char-status');

            $current.text(length);

            // Remove all status classes
            $counter.removeClass('sf-char-optimal sf-char-warning sf-char-bad');

            if (length === 0) {
                $status.text('');
            } else if (length >= 70 && length <= 155) {
                $counter.addClass('sf-char-optimal');
                $status.text(sfMetaDescriptionData.i18n.charCountOptimal);
            } else if ((length >= 50 && length < 70) || (length > 155 && length <= 165)) {
                $counter.addClass('sf-char-warning');
                $status.text(sfMetaDescriptionData.i18n.charCountWarning);
            } else if (length < 50) {
                $counter.addClass('sf-char-bad');
                $status.text(sfMetaDescriptionData.i18n.charCountTooShort);
            } else {
                // length > 165
                $counter.addClass('sf-char-bad');
                $status.text(sfMetaDescriptionData.i18n.charCountTooLong);
            }
        },

        /**
         * Load existing data if available
         */
        loadExistingData: function() {
            var self = this;

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_get_data',
                    nonce: sfMetaDescriptionData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.data) {
                        self.data = response.data.data;
                    }
                }
            });
        },

        /**
         * Process uploaded CSV
         */
        processUploadedCSV: function(uploadData) {
            var self = this;

            // Show loading container with rotating messages
            var loadingState = self.showCSVLoadingContainer();

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_process_csv',
                    nonce: sfMetaDescriptionData.nonce,
                    upload_id: uploadData.upload_id
                },
                success: function(response) {
                    // Clean up loading state
                    self.hideCSVLoadingContainer(loadingState);

                    if (response.success) {
                        self.data = response.data.data;

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message);
                        }

                        // Reload page to show results
                        location.reload();
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    }
                },
                error: function() {
                    // Clean up loading state
                    self.hideCSVLoadingContainer(loadingState);

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to process CSV');
                    }
                }
            });
        },

        /**
         * Show animated loading container for CSV processing
         */
        showCSVLoadingContainer: function() {
            var self = this;

            // Remove any existing loading container
            $('.sf-csv-loading-overlay').remove();

            // Create loading overlay
            var $overlay = $('<div class="sf-csv-loading-overlay">' +
                '<div class="sf-csv-loading-container">' +
                    '<div class="sf-csv-loading-spinner"></div>' +
                    '<div class="sf-csv-loading-progress">' +
                        '<div class="sf-csv-progress-bar"><div class="sf-csv-progress-fill"></div></div>' +
                    '</div>' +
                    '<div class="sf-csv-loading-message">' + self.csvLoadingMessages[0] + '</div>' +
                    '<div class="sf-csv-loading-subtext">Please wait while we analyze your data...</div>' +
                '</div>' +
            '</div>');

            // Insert into the upload box or module
            var $module = $('.sf-meta-description-module');
            if ($module.length) {
                $module.append($overlay);
            } else {
                $('body').append($overlay);
            }

            // Animate progress bar
            var $progressFill = $overlay.find('.sf-csv-progress-fill');
            $progressFill.css('width', '0%');

            // Simulate progress (0-90% over ~15 seconds, slower as it goes)
            var progress = 0;
            var progressInterval = setInterval(function() {
                if (progress < 90) {
                    // Progress slows down as it gets higher
                    var increment = Math.max(0.5, (90 - progress) / 30);
                    progress = Math.min(90, progress + increment);
                    $progressFill.css('width', progress + '%');
                }
            }, 200);

            // Start message rotation (every 2.5 seconds)
            var messageIndex = 0;
            var messageInterval = setInterval(function() {
                messageIndex = (messageIndex + 1) % self.csvLoadingMessages.length;
                var $message = $overlay.find('.sf-csv-loading-message');
                $message.addClass('sf-fade-out');
                setTimeout(function() {
                    $message.text(self.csvLoadingMessages[messageIndex]).removeClass('sf-fade-out');
                }, 300);
            }, 2500);

            return {
                overlay: $overlay,
                messageInterval: messageInterval,
                progressInterval: progressInterval
            };
        },

        /**
         * Hide CSV loading container
         */
        hideCSVLoadingContainer: function(loadingState) {
            if (loadingState) {
                if (loadingState.messageInterval) {
                    clearInterval(loadingState.messageInterval);
                }
                if (loadingState.progressInterval) {
                    clearInterval(loadingState.progressInterval);
                }
                if (loadingState.overlay) {
                    // Complete the progress bar before hiding
                    loadingState.overlay.find('.sf-csv-progress-fill').css('width', '100%');
                    loadingState.overlay.find('.sf-csv-loading-message').text('✅ Processing complete!');

                    setTimeout(function() {
                        loadingState.overlay.fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, 500);
                }
            }
        },

        /**
         * Filter rows by issue type
         */
        filterByIssue: function(issueType) {
            this.currentFilter = issueType;

            var $fixableRows = $('.sf-meta-description-module .sf-section-fixable .sf-description-row');
            var $manualRows = $('.sf-meta-description-module .sf-section-manual .sf-description-row');

            if (issueType === 'fixed' || issueType === 'failed') {
                // Hide all manual rows for status filters
                $manualRows.addClass('sf-filtered-out');

                // Filter fixable rows by data-status
                $fixableRows.each(function() {
                    var $row = $(this);
                    var rowStatus = $row.attr('data-status') || 'pending';

                    if (issueType === 'fixed') {
                        if (rowStatus === 'success' || rowStatus === 'fixed') {
                            $row.removeClass('sf-filtered-out');
                        } else {
                            $row.addClass('sf-filtered-out');
                        }
                    } else if (issueType === 'failed') {
                        if (rowStatus === 'failed') {
                            $row.removeClass('sf-filtered-out');
                        } else {
                            $row.addClass('sf-filtered-out');
                        }
                    }
                });
            } else if (issueType === 'all') {
                $fixableRows.removeClass('sf-filtered-out');
                $manualRows.removeClass('sf-filtered-out');
            } else {
                // Filter by issue type
                $fixableRows.each(function() {
                    var rowIssue = $(this).data('issue-type');
                    if (rowIssue === issueType) {
                        $(this).removeClass('sf-filtered-out');
                    } else {
                        $(this).addClass('sf-filtered-out');
                    }
                });

                $manualRows.each(function() {
                    var rowIssue = $(this).data('issue-type');
                    if (rowIssue === issueType) {
                        $(this).removeClass('sf-filtered-out');
                    } else {
                        $(this).addClass('sf-filtered-out');
                    }
                });
            }

            // Reset pagination to page 1 when filtering changes
            this.resetPagination('fixable');
            this.resetPagination('manual');

            // Update section visibility based on filtered rows
            this.updateSectionVisibility();

            // Update select all to only affect visible rows
            this.updateSelectedCount();

            // Update issue stat active state
            $('.sf-meta-description-module .sf-issue-stat').removeClass('active');
            $('.sf-meta-description-module .sf-issue-stat[data-filter="' + issueType + '"]').addClass('active');

            // Auto-expand and scroll to Fixed section when Fixed or Can't Fix filter is selected
            if (issueType === 'fixed' || issueType === 'failed') {
                var $fixedSection = $('.sf-meta-description-module .sf-section-fixed');
                var $fixedToggle = $fixedSection.find('.sf-fixed-toggle');
                var $fixedContent = $fixedSection.find('.sf-fixed-content');
                if ($fixedContent.length && $fixedToggle.attr('aria-expanded') !== 'true') {
                    $fixedContent.slideDown(200);
                    $fixedToggle.attr('aria-expanded', 'true');
                    $fixedToggle.find('.sf-toggle-icon').addClass('sf-rotated');
                }
                if ($fixedSection.length) {
                    setTimeout(function() {
                        $('html, body').animate({
                            scrollTop: $fixedSection.offset().top - 50
                        }, 400);
                    }, 250);
                }
            }

            // Show/hide duplicate info notice
            this.updateDuplicateInfoNotice(issueType);
        },

        /**
         * Show or hide the duplicate info notice based on filter
         */
        updateDuplicateInfoNotice: function(issueType) {
            var $notice = $('.sf-duplicate-info-notice');

            // Check if user has dismissed the notice this session
            var isDismissed = false;
            if (typeof sessionStorage !== 'undefined') {
                isDismissed = sessionStorage.getItem('sf_duplicate_notice_dismissed') === 'true';
            }

            // Show notice only when duplicate filter is active and not dismissed
            if (issueType === 'duplicate' && !isDismissed) {
                $notice.slideDown(200);
            } else {
                $notice.slideUp(200);
            }
        },

        /**
         * Update section visibility based on filtered content
         */
        updateSectionVisibility: function() {
            // Check fixable section
            var $fixableSection = $('.sf-section-fixable');
            var visibleFixable = $fixableSection.find('.sf-description-row:not(.sf-filtered-out)').length;
            if (visibleFixable === 0 && this.currentFilter !== 'all') {
                $fixableSection.addClass('sf-section-hidden');
            } else {
                $fixableSection.removeClass('sf-section-hidden');
            }

            // Check manual section - auto-expand if it has visible rows and filter is active
            var $manualSection = $('.sf-section-manual');
            var visibleManual = $manualSection.find('.sf-description-row:not(.sf-filtered-out)').length;
            if (visibleManual === 0 && this.currentFilter !== 'all') {
                $manualSection.addClass('sf-section-hidden');
            } else {
                $manualSection.removeClass('sf-section-hidden');
                // Auto-expand manual section if filter is active and has results
                if (this.currentFilter !== 'all' && visibleManual > 0) {
                    var $toggle = $manualSection.find('.sf-section-toggle');
                    var $content = $manualSection.find('.sf-manual-content');
                    if ($toggle.attr('aria-expanded') !== 'true') {
                        $content.slideDown(200);
                        $toggle.attr('aria-expanded', 'true');
                    }
                }
            }
        },

        /**
         * Update selected count display
         */
        updateSelectedCount: function() {
            var count = $('.sf-description-row:visible .sf-row-select:checked').length;
            $('.sf-selected-count .sf-count').text(count);

            // Update select all checkbox state
            var visibleCheckboxes = $('.sf-description-row:visible .sf-row-select');
            var checkedCheckboxes = visibleCheckboxes.filter(':checked');

            if (visibleCheckboxes.length > 0 && checkedCheckboxes.length === visibleCheckboxes.length) {
                $('#sf-select-all').prop('checked', true).prop('indeterminate', false);
            } else if (checkedCheckboxes.length > 0) {
                $('#sf-select-all').prop('checked', false).prop('indeterminate', true);
            } else {
                $('#sf-select-all').prop('checked', false).prop('indeterminate', false);
            }

            this.updateApplyButtonState();
        },

        /**
         * Update apply button state
         */
        updateApplyButtonState: function() {
            var hasValidFixes = false;

            $('.sf-description-row:visible').each(function() {
                var $row = $(this);
                var $checkbox = $row.find('.sf-row-select');
                var $input = $row.find('.sf-new-desc-input');

                if ($checkbox.prop('checked') && $input.val().trim().length > 0) {
                    hasValidFixes = true;
                    return false; // break
                }
            });

            $('.sf-apply-fixes').prop('disabled', !hasValidFixes);
        },

        /**
         * Update fixed/failed filter counts by scanning row data-status attributes
         */
        updateStatusCounts: function() {
            var $rows = $('.sf-meta-description-module .sf-section-fixable .sf-description-row');
            var counts = {
                fixed: 0,
                failed: 0
            };

            $rows.each(function() {
                var status = $(this).attr('data-status') || 'pending';
                if (status === 'success' || status === 'fixed') {
                    counts.fixed++;
                } else if (status === 'failed') {
                    counts.failed++;
                }
            });

            // Also count from the Fixes Applied section
            var $fixedRows = $('.sf-meta-description-module .sf-section-fixed .sf-fixed-row');
            counts.fixed += $fixedRows.filter('[data-status="success"]').length;
            counts.failed += $fixedRows.filter('[data-status="failed"]').length;

            var $fixedFilter = $('.sf-meta-description-module .sf-issue-fixed');
            var $failedFilter = $('.sf-meta-description-module .sf-issue-failed');

            $fixedFilter.find('.sf-fixed-filter-count').text(counts.fixed);
            $failedFilter.find('.sf-failed-filter-count').text(counts.failed);

            if (counts.fixed > 0) {
                $fixedFilter.removeClass('sf-filter-disabled');
            } else {
                $fixedFilter.addClass('sf-filter-disabled');
            }

            if (counts.failed > 0) {
                $failedFilter.removeClass('sf-filter-disabled');
            } else {
                $failedFilter.addClass('sf-filter-disabled');
            }
        },

        /**
         * Update save button state based on textarea content
         */
        updateSaveButtonState: function($textarea) {
            var $row = $textarea.closest('.sf-description-row');
            var $saveBtn = $row.find('.sf-save-desc-btn');
            var hasContent = $textarea.val().trim().length > 0;

            // Enable/disable save button based on content
            $saveBtn.prop('disabled', !hasContent);

            // If row was previously saved but content changed, remove saved state
            if ($row.hasClass('sf-row-saved')) {
                $row.removeClass('sf-row-saved');
                $saveBtn.removeClass('sf-saved').text('✓ Save');
            }
        },

        /**
         * Save a single description
         */
        saveSingleDescription: function($row, $btn) {
            var self = this;
            var $input = $row.find('.sf-new-desc-input');
            var newDesc = $input.val().trim();

            if (newDesc.length === 0) {
                return;
            }

            var postId = $row.data('post-id');
            var address = $row.data('address');

            // Show saving state
            $btn.addClass('sf-saving');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_apply_fixes',
                    nonce: sfMetaDescriptionData.nonce,
                    fixes: [{
                        post_id: postId,
                        address: address,
                        new_description: newDesc
                    }]
                },
                success: function(response) {
                    if (response.success) {
                        var results = response.data.results || {};
                        var successCount = results.success || 0;

                        if (successCount > 0) {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.success('Meta description saved successfully!');
                            }
                        } else {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error('Failed to save meta description.');
                            }
                        }

                        // Remove row from Fixable (it moves to Fixes Applied)
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            self.updateStatusCounts();
                        });

                        // Refresh the Fixes Applied section
                        self.refreshFixedSection();
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Failed to save description.');
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to save description.');
                    }
                },
                complete: function() {
                    $btn.removeClass('sf-saving');
                    // Only re-enable if there's still content
                    $btn.prop('disabled', $input.val().trim().length === 0);
                }
            });
        },

        /**
         * Refresh the Fixed section via AJAX after a successful save
         */
        refreshFixedSection: function() {
            var self = this;
            var $fixedSection = $('.sf-meta-description-module .sf-section-fixed');

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_get_fixed_section',
                    nonce: sfMetaDescriptionData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.html) {
                        if ($fixedSection.length) {
                            $fixedSection.html(response.data.html);
                        } else {
                            var $newSection = $('<div class="sf-results-section sf-section-fixed" id="sf-fixed-results"></div>');
                            $newSection.html(response.data.html);

                            var $progressModal = $('.sf-meta-description-module .sf-progress-modal');
                            if ($progressModal.length) {
                                $newSection.insertBefore($progressModal);
                            } else {
                                $('.sf-meta-description-module').append($newSection);
                            }
                            $fixedSection = $newSection;
                        }

                        $fixedSection.show();
                        var $content = $fixedSection.find('.sf-fixed-content');
                        $content.show();

                        // Re-bind event handlers for the new content
                        self.bindFixedSectionEvents();

                        // Update main Filter By counts now that fixed section is loaded
                        self.updateStatusCounts();

                        setTimeout(function() {
                            $('html, body').animate({
                                scrollTop: $fixedSection.offset().top - 50
                            }, 500);
                        }, 400);
                    }
                }
            });
        },

        /**
         * Bind event handlers for Fixed section elements
         */
        bindFixedSectionEvents: function() {
            var self = this;

            // Fixed section toggle
            $('.sf-meta-description-module .sf-section-fixed .sf-fixed-toggle').off('click').on('click', function() {
                var $btn = $(this);
                var $content = $btn.closest('.sf-section-fixed').find('.sf-fixed-content');
                var isExpanded = $btn.attr('aria-expanded') === 'true';

                if (isExpanded) {
                    $content.slideUp(300);
                    $btn.attr('aria-expanded', 'false');
                    $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
                } else {
                    $content.slideDown(300);
                    $btn.attr('aria-expanded', 'true');
                    $btn.find('.sf-toggle-icon').addClass('sf-rotated');
                }
            });

            // Status filter buttons
            $('.sf-meta-description-module .sf-md-status-filters .sf-status-filter').off('click').on('click', function() {
                var $btn = $(this);
                var status = $btn.data('status');

                // Update active state
                $('.sf-meta-description-module .sf-md-status-filters .sf-status-filter').removeClass('active');
                $btn.addClass('active');

                // Filter rows
                self.filterFixResults(status);
            });

            // Update status counts
            self.updateFixStatusCounts();
        },

        /**
         * Filter fixed section rows by status
         */
        filterFixResults: function(status) {
            var $section = $('.sf-meta-description-module .sf-section-fixed');
            var $rows = $section.find('.sf-fixed-table tbody .sf-fixed-row');

            if (status === 'all') {
                $rows.removeClass('sf-filtered-out');
            } else {
                $rows.addClass('sf-filtered-out');
                $rows.filter('[data-status="' + status + '"]').removeClass('sf-filtered-out');
            }
        },

        /**
         * Update status counts in filter buttons
         */
        updateFixStatusCounts: function() {
            var $section = $('.sf-meta-description-module .sf-section-fixed');
            var $rows = $section.find('.sf-fixed-table tbody .sf-fixed-row');
            var allCount = $rows.length;
            var successCount = $rows.filter('[data-status="success"]').length;
            var failedCount = $rows.filter('[data-status="failed"]').length;
            var skippedCount = $rows.filter('[data-status="skipped"]').length;

            $section.find('.sf-md-status-count-all').text(allCount);
            $section.find('.sf-md-status-count-success').text(successCount);
            $section.find('.sf-md-status-count-failed').text(failedCount);
            $section.find('.sf-md-status-count-skipped').text(skippedCount);
        },

        /**
         * Get AI suggestion for a single description
         */
        getAISuggestion: function($row, $btn) {
            var self = this;
            var address = $row.data('address');
            var postTitle = $row.data('post-title');
            var $input = $row.find('.sf-new-desc-input');
            var currentDesc = $row.find('.sf-description-text').text() || '';

            // Show loading state
            $btn.addClass('sf-loading');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_get_ai_suggestion',
                    nonce: sfMetaDescriptionData.nonce,
                    address: address,
                    post_title: postTitle,
                    current_description: currentDesc
                },
                success: function(response) {
                    if (response.success) {
                        $input.val(response.data.suggestion);
                        self.updateCharCounter($input);
                        self.updateApplyButtonState();
                        self.updateSaveButtonState($input);

                        // Auto-select the row
                        $row.find('.sf-row-select').prop('checked', true);
                        self.updateSelectedCount();

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(sfMetaDescriptionData.i18n.aiComplete);
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfMetaDescriptionData.i18n.aiFailed);
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfMetaDescriptionData.i18n.aiFailed);
                    }
                },
                complete: function() {
                    $btn.removeClass('sf-loading');
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Apply fixes to selected descriptions
         */
        applyFixes: function() {
            var self = this;
            var fixes = [];

            // Collect fixes from selected rows
            $('.sf-description-row:visible .sf-row-select:checked').each(function() {
                var $row = $(this).closest('.sf-description-row');
                var $input = $row.find('.sf-new-desc-input');
                var newDesc = $input.val().trim();

                if (newDesc.length > 0) {
                    fixes.push({
                        post_id: $row.data('post-id'),
                        address: $row.data('address'),
                        new_description: newDesc
                    });
                }
            });

            if (fixes.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfMetaDescriptionData.i18n.noFixesSelected);
                }
                return;
            }

            // Show progress modal
            var $modal = $('.sf-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-progress-current');
            var $total = $modal.find('.sf-progress-total');

            $total.text(fixes.length);
            $current.text(0);
            $fill.css('width', '0%');
            $modal.show();

            // Disable apply button
            $('.sf-apply-fixes').prop('disabled', true);

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_apply_fixes',
                    nonce: sfMetaDescriptionData.nonce,
                    fixes: fixes
                },
                success: function(response) {
                    // Animate to 100%
                    $fill.css('width', '100%');
                    $current.text(fixes.length);

                    setTimeout(function() {
                        $modal.hide();

                        if (response.success) {
                            var results = response.data.results || {};
                            var successCount = results.success || 0;
                            var failedCount = results.failed || 0;

                            // Show appropriate toast
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                if (failedCount > 0) {
                                    ScreamingFixes.Toast.warning(response.data.message || sfMetaDescriptionData.i18n.fixesApplied);
                                } else {
                                    ScreamingFixes.Toast.success(response.data.message || sfMetaDescriptionData.i18n.fixesApplied);
                                }
                            }

                            // Remove all attempted rows from Fixable section
                            fixes.forEach(function(fix) {
                                var $row = $('.sf-meta-description-module .sf-section-fixable .sf-description-row[data-address="' + fix.address + '"]');
                                if ($row.length) {
                                    $row.fadeOut(300, function() {
                                        $(this).remove();
                                        self.updateStatusCounts();
                                    });
                                }
                            });

                            // Uncheck all selected rows
                            $('.sf-meta-description-module .sf-description-row .sf-row-select:checked').prop('checked', false);
                            self.updateSelectedCount();

                            // Always refresh the Fixes Applied section
                            self.refreshFixedSection();
                        } else {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error(response.data.message || sfMetaDescriptionData.i18n.fixesFailed);
                            }
                        }
                    }, 500);
                },
                error: function() {
                    $modal.hide();
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfMetaDescriptionData.i18n.fixesFailed);
                    }
                },
                complete: function() {
                    $('.sf-apply-fixes').prop('disabled', false);
                }
            });
        },

        /**
         * Clear all meta description data
         */
        clearData: function() {
            var self = this;

            // Confirm before clearing
            if (!confirm(sfMetaDescriptionData.i18n.confirmClear || 'Are you sure you want to clear all meta description data? This will allow you to upload a new CSV file.')) {
                return;
            }

            var $btn = $('.sf-clear-data-btn');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_clear_data',
                    nonce: sfMetaDescriptionData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message || sfMetaDescriptionData.i18n.dataCleared || 'Data cleared successfully.');
                        }
                        // Reload page to show upload form
                        location.reload();
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Failed to clear data.');
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to clear data.');
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Process bulk update - applies new descriptions from uploaded CSV
         */
        processBulkUpdate: function() {
            var self = this;

            if (self.bulkProcessing) {
                return;
            }

            self.bulkProcessing = true;

            // Get bulk data from the page
            var $section = $('.sf-meta-description-module .sf-bulk-confirmation');
            // The ready count is in .sf-bulk-stat-ready .sf-bulk-stat-number
            var matchedUrls = parseInt($section.find('.sf-bulk-stat-ready .sf-bulk-stat-number').text(), 10) || 0;
            // Also check the data-count attribute on the confirm button as fallback
            if (matchedUrls === 0) {
                matchedUrls = parseInt($('.sf-meta-description-module .sf-bulk-confirm').data('count'), 10) || 0;
            }

            if (matchedUrls === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfMetaDescriptionData.i18n.bulkNoMatches || 'No matching URLs found to update.');
                }
                self.bulkProcessing = false;
                return;
            }

            // Show progress modal
            var $modal = $('.sf-meta-description-module .sf-bulk-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-bulk-progress-current');
            var $total = $modal.find('.sf-bulk-progress-total');
            var $percent = $modal.find('.sf-bulk-progress-percent');
            var $currentUrl = $modal.find('.sf-bulk-current-url');

            $total.text(matchedUrls);
            $current.text(0);
            $percent.text('0');
            $fill.css('width', '0%');
            $currentUrl.text('Starting...');
            $modal.show();

            // Disable confirm button
            $('.sf-meta-description-module .sf-bulk-confirm').prop('disabled', true);

            // Start processing in batches directly (don't clear data first)
            self.processBulkBatch(0, matchedUrls);
        },

        /**
         * Process a batch of bulk updates
         */
        processBulkBatch: function(offset, total) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_apply_bulk_updates',
                    nonce: sfMetaDescriptionData.nonce,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = data.processed || 0;
                        var currentOffset = offset + processed;
                        var percent = Math.min(100, Math.round((currentOffset / total) * 100));

                        // Update progress UI
                        var $modal = $('.sf-meta-description-module .sf-bulk-progress-modal');
                        $modal.find('.sf-bulk-progress-current').text(currentOffset);
                        $modal.find('.sf-bulk-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        // Update current URL being processed
                        if (data.current_url) {
                            $modal.find('.sf-bulk-current-url').text(data.current_url);
                        }

                        // Check if we need to continue
                        if (data.complete) {
                            // All done - show results
                            // Use accumulated totals (total_success/total_failed/all_details)
                            self.showBulkResults({
                                success: data.total_success || data.success || 0,
                                failed: data.total_failed || data.failed || 0,
                                details: data.all_details || data.details || []
                            });
                        } else {
                            // Continue with next batch
                            self.processBulkBatch(currentOffset, total);
                        }
                    } else {
                        // Error occurred
                        self.bulkProcessing = false;
                        $('.sf-meta-description-module .sf-bulk-progress-modal').hide();
                        $('.sf-meta-description-module .sf-bulk-confirm').prop('disabled', false);

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfMetaDescriptionData.i18n.bulkFailed || 'Bulk update failed.');
                        }
                    }
                },
                error: function() {
                    self.bulkProcessing = false;
                    $('.sf-meta-description-module .sf-bulk-progress-modal').hide();
                    $('.sf-meta-description-module .sf-bulk-confirm').prop('disabled', false);

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfMetaDescriptionData.i18n.bulkFailed || 'Bulk update failed.');
                    }
                }
            });
        },

        /**
         * Show bulk update results
         */
        showBulkResults: function(results) {
            var self = this;
            self.bulkProcessing = false;

            // Hide progress modal
            $('.sf-meta-description-module .sf-bulk-progress-modal').hide();

            // Reload the page to show the server-rendered results view
            // The bulk complete section already displays success/failure counts
            location.reload();
        },

        /**
         * Clear bulk data and return to upload state
         */
        clearBulkData: function() {
            var self = this;

            // Confirm before clearing
            if (!confirm(sfMetaDescriptionData.i18n.confirmClear || 'Are you sure you want to clear all meta description data? This will allow you to upload a new CSV file.')) {
                return;
            }

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_cancel_bulk',
                    nonce: sfMetaDescriptionData.nonce
                },
                success: function(response) {
                    // Reload page to show upload form
                    location.reload();
                },
                error: function() {
                    // Reload anyway
                    location.reload();
                }
            });
        },

        /**
         * Clear bulk complete results and return to upload state
         * Used on the bulk complete results page - no confirmation needed
         */
        clearBulkCompleteData: function() {
            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_cancel_bulk',
                    nonce: sfMetaDescriptionData.nonce
                },
                success: function(response) {
                    // Reload page to show upload form
                    location.reload();
                },
                error: function() {
                    // Reload anyway
                    location.reload();
                }
            });
        },

        /**
         * Download bulk preview CSV
         */
        downloadBulkPreview: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfMetaDescriptionData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_meta_description_download_preview' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfMetaDescriptionData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Download bulk results CSV
         */
        downloadBulkResults: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfMetaDescriptionData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_meta_description_download_results' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfMetaDescriptionData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Download fixed descriptions results CSV (from Fixed Meta Descriptions section)
         */
        downloadFixedResults: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfMetaDescriptionData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_meta_description_download_fixed_results' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfMetaDescriptionData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Start new upload after bulk complete
         */
        startNewUpload: function() {
            // Clear data and reload
            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_clear_data',
                    nonce: sfMetaDescriptionData.nonce
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Retry a failed meta description update
         */
        retryFailedUpdate: function($row, $btn) {
            var self = this;
            var postId = $row.data('post-id');
            var address = $row.data('address');
            var newDesc = $row.data('attempted-description');

            if (!postId || !newDesc) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.error('Cannot retry: missing post ID or description.');
                }
                return;
            }

            // Show retrying state
            $btn.addClass('sf-retrying');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfMetaDescriptionData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_meta_description_apply_fixes',
                    nonce: sfMetaDescriptionData.nonce,
                    fixes: [{
                        post_id: postId,
                        address: address,
                        new_description: newDesc
                    }]
                },
                success: function(response) {
                    if (response.success && response.data.results && response.data.results.success > 0) {
                        // Success! Show feedback and reload to update the view
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success('Meta description updated successfully!');
                        }
                        // Reload to show updated state
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        // Still failed
                        var errorMsg = 'Update still failed.';
                        if (response.data && response.data.results && response.data.results.errors && response.data.results.errors.length > 0) {
                            errorMsg = response.data.results.errors[0].error || errorMsg;
                        }
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(errorMsg);
                        }
                        $btn.removeClass('sf-retrying');
                        $btn.prop('disabled', false);
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to retry update.');
                    }
                    $btn.removeClass('sf-retrying');
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Initialize pagination for all paginated tables
         */
        initPagination: function() {
            var self = this;

            // Initialize each paginated section
            $('.sf-paginated-table').each(function() {
                var $wrapper = $(this);
                var section = $wrapper.data('section');
                var perPage = parseInt($wrapper.data('per-page'), 10) || 100;
                var total = parseInt($wrapper.data('total'), 10) || 0;

                if (self.pagination[section]) {
                    self.pagination[section].perPage = perPage;
                    self.pagination[section].totalItems = total;
                    self.pagination[section].currentPage = 1;

                    // Initial render - apply pagination to show first page
                    self.applyPagination(section);
                    self.renderPageNumbers(section);
                }
            });
        },

        /**
         * Go to a specific page
         */
        goToPage: function(section, page) {
            var pag = this.pagination[section];
            if (!pag) return;

            var totalPages = Math.ceil(pag.totalItems / pag.perPage);
            page = Math.max(1, Math.min(page, totalPages));

            if (page === pag.currentPage) return;

            pag.currentPage = page;
            this.applyPagination(section);
            this.renderPageNumbers(section);
            this.updatePaginationInfo(section);

            // User stays in same scroll position - no auto-scroll on page change
        },

        /**
         * Apply pagination (show/hide rows based on current page)
         */
        applyPagination: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $wrapper = $('.sf-paginated-table[data-section="' + section + '"]');
            var rowSelector = section === 'fixed' ? 'tbody .sf-fixed-row' : (section === 'failed' ? 'tbody .sf-failed-row' : 'tbody .sf-description-row');
            var $rows = $wrapper.find(rowSelector);

            var startIndex = (pag.currentPage - 1) * pag.perPage;
            var endIndex = startIndex + pag.perPage;

            // Get rows that are not filtered out
            var visibleIndex = 0;
            $rows.each(function(i) {
                var $row = $(this);
                var isFilteredOut = $row.hasClass('sf-filtered-out');

                if (isFilteredOut) {
                    // Keep filtered rows hidden
                    $row.addClass('sf-page-hidden');
                } else {
                    // Apply pagination to non-filtered rows
                    if (visibleIndex >= startIndex && visibleIndex < endIndex) {
                        $row.removeClass('sf-page-hidden');
                    } else {
                        $row.addClass('sf-page-hidden');
                    }
                    visibleIndex++;
                }
            });

            // Update button states
            var $pagination = $('.sf-pagination[data-section="' + section + '"]');
            var totalPages = Math.ceil(this.getVisibleRowCount(section) / pag.perPage);

            $pagination.find('.sf-page-prev').prop('disabled', pag.currentPage <= 1);
            $pagination.find('.sf-page-next').prop('disabled', pag.currentPage >= totalPages);
        },

        /**
         * Get count of visible (non-filtered) rows
         */
        getVisibleRowCount: function(section) {
            var $wrapper = $('.sf-paginated-table[data-section="' + section + '"]');
            var rowSelector = section === 'fixed' ? 'tbody .sf-fixed-row:not(.sf-filtered-out)' : (section === 'failed' ? 'tbody .sf-failed-row:not(.sf-filtered-out)' : 'tbody .sf-description-row:not(.sf-filtered-out)');
            return $wrapper.find(rowSelector).length;
        },

        /**
         * Render page number buttons
         */
        renderPageNumbers: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('.sf-pagination[data-section="' + section + '"]');
            var $container = $pagination.find('.sf-page-numbers');

            var visibleCount = this.getVisibleRowCount(section);
            var totalPages = Math.ceil(visibleCount / pag.perPage);
            var currentPage = pag.currentPage;

            // Don't render if only one page
            if (totalPages <= 1) {
                $container.empty();
                $pagination.find('.sf-page-prev, .sf-page-next').prop('disabled', true);
                return;
            }

            var html = '';
            var maxVisible = 7; // Maximum number of page buttons to show

            if (totalPages <= maxVisible) {
                // Show all pages
                for (var i = 1; i <= totalPages; i++) {
                    html += this.renderPageButton(i, currentPage);
                }
            } else {
                // Show with ellipsis
                // Always show first page
                html += this.renderPageButton(1, currentPage);

                if (currentPage > 3) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }

                // Show pages around current
                var startPage = Math.max(2, currentPage - 1);
                var endPage = Math.min(totalPages - 1, currentPage + 1);

                // Adjust if at edges
                if (currentPage <= 3) {
                    endPage = Math.min(4, totalPages - 1);
                }
                if (currentPage >= totalPages - 2) {
                    startPage = Math.max(2, totalPages - 3);
                }

                for (var j = startPage; j <= endPage; j++) {
                    html += this.renderPageButton(j, currentPage);
                }

                if (currentPage < totalPages - 2) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }

                // Always show last page
                html += this.renderPageButton(totalPages, currentPage);
            }

            $container.html(html);
        },

        /**
         * Render a single page button
         */
        renderPageButton: function(page, currentPage) {
            var activeClass = page === currentPage ? ' active' : '';
            return '<button type="button" class="sf-page-number' + activeClass + '" data-page="' + page + '">' + page + '</button>';
        },

        /**
         * Update pagination info text
         */
        updatePaginationInfo: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('.sf-pagination[data-section="' + section + '"]');
            var visibleCount = this.getVisibleRowCount(section);

            var start = Math.min((pag.currentPage - 1) * pag.perPage + 1, visibleCount);
            var end = Math.min(pag.currentPage * pag.perPage, visibleCount);

            $pagination.find('.sf-page-start').text(start);
            $pagination.find('.sf-page-end').text(end);
            $pagination.find('.sf-page-total').text(visibleCount);
        },

        /**
         * Reset pagination to page 1 for a section
         */
        resetPagination: function(section) {
            if (this.pagination[section]) {
                this.pagination[section].currentPage = 1;
                this.applyPagination(section);
                this.renderPageNumbers(section);
                this.updatePaginationInfo(section);
            }
        },

        /**
         * Reset pagination for all sections
         */
        resetAllPagination: function() {
            var self = this;
            ['fixable', 'manual', 'skipped'].forEach(function(section) {
                self.resetPagination(section);
            });
        },

        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        // Only init if we're on the meta description tab
        if ($('.sf-meta-description-module').length > 0) {
            ScreamingFixes.MetaDescription.init();
        }
    });

})(jQuery);
