<?php
/**
 * Meta Description Module for Screaming Fixes
 *
 * Finds and fixes meta description issues from Screaming Frog exports
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Meta_Description extends SF_Module {

    /**
     * Maximum number of CSV rows allowed for processing.
     * This limit protects against memory exhaustion on limited hosting plans.
     *
     * @var int
     */
    const MAX_CSV_ROWS = 5000;

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'meta-description';

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Meta Description', 'screaming-fixes');
        $this->slug = 'meta-description';
        $this->description = __('Find and fix meta description issues automatically.', 'screaming-fixes');

        parent::__construct();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register AJAX handlers
        add_action('wp_ajax_sf_meta_description_process_csv', [$this, 'ajax_process_csv']);
        add_action('wp_ajax_sf_meta_description_apply_fixes', [$this, 'ajax_apply_fixes']);
        add_action('wp_ajax_sf_meta_description_get_ai_suggestion', [$this, 'ajax_get_ai_suggestion']);
        add_action('wp_ajax_sf_meta_description_get_data', [$this, 'ajax_get_data']);
        add_action('wp_ajax_sf_meta_description_clear_data', [$this, 'ajax_clear_data']);

        // Bulk update AJAX handlers
        add_action('wp_ajax_sf_meta_description_process_bulk_csv', [$this, 'ajax_process_bulk_csv']);
        add_action('wp_ajax_sf_meta_description_apply_bulk_updates', [$this, 'ajax_apply_bulk_updates']);
        add_action('wp_ajax_sf_meta_description_cancel_bulk', [$this, 'ajax_cancel_bulk']);
        add_action('wp_ajax_sf_meta_description_download_preview', [$this, 'ajax_download_preview']);
        add_action('wp_ajax_sf_meta_description_download_results', [$this, 'ajax_download_results']);
        add_action('wp_ajax_sf_meta_description_download_fixed_results', [$this, 'ajax_download_fixed_results']);
        add_action('wp_ajax_sf_meta_description_get_fixed_section', [$this, 'ajax_get_fixed_section']);

        // Enqueue module assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        // Check if we're on the meta description tab
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'meta-description') {
            return;
        }

        // Use file modification time for cache busting
        $css_file = SF_PLUGIN_DIR . 'modules/meta-description/assets/meta-description.css';
        $js_file = SF_PLUGIN_DIR . 'modules/meta-description/assets/meta-description.js';
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        wp_enqueue_style(
            'sf-meta-description',
            SF_PLUGIN_URL . 'modules/meta-description/assets/meta-description.css',
            ['screaming-fixes-admin'],
            $css_version
        );

        wp_enqueue_script(
            'sf-meta-description',
            SF_PLUGIN_URL . 'modules/meta-description/assets/meta-description.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        wp_localize_script('sf-meta-description', 'sfMetaDescriptionData', [
            'nonce' => wp_create_nonce('sf_meta_description_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'i18n' => [
                'processing' => __('Processing CSV...', 'screaming-fixes'),
                'applyingFixes' => __('Applying meta description fixes...', 'screaming-fixes'),
                'fixesApplied' => __('Meta description fixes applied successfully!', 'screaming-fixes'),
                'fixesFailed' => __('Some fixes failed. Check the results.', 'screaming-fixes'),
                'noFixesSelected' => __('No fixes selected.', 'screaming-fixes'),
                'aiSuggesting' => __('Getting AI suggestion...', 'screaming-fixes'),
                'aiComplete' => __('AI suggestion complete.', 'screaming-fixes'),
                'aiFailed' => __('Failed to get AI suggestion.', 'screaming-fixes'),
                'confirmApply' => __('Apply meta description to %d pages?', 'screaming-fixes'),
                'charCountOptimal' => __('Optimal length', 'screaming-fixes'),
                'charCountWarning' => __('Could be truncated', 'screaming-fixes'),
                'charCountTooShort' => __('Too short', 'screaming-fixes'),
                'charCountTooLong' => __('Too long', 'screaming-fixes'),
                'confirmClear' => __('Are you sure you want to clear all meta description data? This will allow you to upload a new CSV file.', 'screaming-fixes'),
                'dataCleared' => __('Data cleared successfully.', 'screaming-fixes'),
                // Bulk upload i18n
                'bulkProcessing' => __('Processing bulk update CSV...', 'screaming-fixes'),
                'bulkReady' => __('%d meta descriptions ready to update', 'screaming-fixes'),
                'bulkNotMatched' => __('%d URLs not matched', 'screaming-fixes'),
                'bulkSkippedEmpty' => __('%d rows skipped - no new meta description provided', 'screaming-fixes'),
                'bulkDuplicates' => __('%d duplicate URLs detected - using last occurrence for each', 'screaming-fixes'),
                'bulkEmptyNotice' => __('Rows with empty meta description values will be skipped.', 'screaming-fixes'),
                'bulkUpdating' => __('Updating meta descriptions...', 'screaming-fixes'),
                'bulkComplete' => __('%d meta descriptions updated successfully.', 'screaming-fixes'),
                'bulkPartialComplete' => __('%d meta descriptions updated successfully. %d failed - see details below.', 'screaming-fixes'),
                'bulkLargeFileWarning' => __('Large file detected (%d URLs). For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'),
                'bulkContinueAnyway' => __('Continue Anyway', 'screaming-fixes'),
                'bulkMultipleColumns' => __('Please upload a CSV with only one update column at a time. This file contains multiple update columns.', 'screaming-fixes'),
                'bulkNoNewMetaColumn' => __('CSV must contain a \'new_meta\' or \'new_description\' column for bulk updates.', 'screaming-fixes'),
                'bulkNoUrlColumn' => __('CSV must contain a URL column (url, address, or page_url).', 'screaming-fixes'),
                'bulkLengthWarning' => __('This is unusually long and may cause display issues.', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Run the module scan
     *
     * @return array Scan results
     */
    public function run_scan() {
        // This module doesn't do active scanning - it processes uploaded CSVs
        return $this->get_results();
    }

    /**
     * Check if this module can handle a CSV based on headers
     *
     * @param array $headers CSV column headers (lowercase)
     * @return bool
     */
    public function can_handle_csv($headers) {
        // Normalize headers to lowercase
        $headers = array_map('strtolower', $headers);

        // Screaming Frog "Meta Description" export
        // Must have: Address, Meta Description columns
        $has_address = in_array('address', $headers);
        $has_meta_desc = in_array('meta description 1', $headers) ||
                         in_array('meta description', $headers);

        // Additional indicators for meta description CSV
        $has_occurrences = in_array('occurrences', $headers);
        $has_length = in_array('meta description 1 length', $headers) ||
                      in_array('length', $headers);

        // Must have address AND meta description column
        // Having occurrences or length is a strong indicator
        return $has_address && $has_meta_desc;
    }

    /**
     * Process uploaded CSV file
     *
     * @param string $file_path Path to uploaded CSV
     * @return array|WP_Error Processed results or error
     */
    public function process_csv($file_path) {
        $parser = new SF_CSV_Parser();

        // Parse the CSV
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Standardize column names
        $parsed = $this->standardize_columns($parsed);

        // Detect which SEO plugin is active (but don't require it for viewing data)
        $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();
        $seo_plugin = $this->determine_seo_plugin($seo_plugins);

        // Process each row
        $all_descriptions = [];
        $duplicate_tracker = []; // Track descriptions for duplicate detection

        foreach ($parsed['rows'] as $row) {
            $address = isset($row['address']) ? trim($row['address']) : '';
            $description = isset($row['meta_description']) ? trim($row['meta_description']) : '';
            $length = isset($row['length']) ? (int) $row['length'] : strlen($description);
            $pixel_width = isset($row['pixel_width']) ? (int) $row['pixel_width'] : 0;
            $occurrences = isset($row['occurrences']) ? (int) $row['occurrences'] : 1;
            $indexability = isset($row['indexability']) ? trim($row['indexability']) : '';
            $indexability_status = isset($row['indexability_status']) ? trim($row['indexability_status']) : '';

            if (empty($address)) {
                continue;
            }

            // Categorize based on indexability and URL resolution
            $categorization = $this->categorize_description($address, $indexability, $indexability_status);

            // Determine issue type
            $issue_type = $this->determine_issue_type($description, $length, $occurrences);

            // Skip rows with no issues
            if ($issue_type === 'none') {
                continue;
            }

            $post_id = $categorization['post_id'];
            $post_title = $post_id ? get_the_title($post_id) : '';
            $edit_url = $post_id ? get_edit_post_link($post_id, 'raw') : '';

            // Track for duplicate detection
            if (!empty($description)) {
                $desc_hash = md5($description);
                if (!isset($duplicate_tracker[$desc_hash])) {
                    $duplicate_tracker[$desc_hash] = [];
                }
                $duplicate_tracker[$desc_hash][] = count($all_descriptions);
            }

            $all_descriptions[] = [
                'address' => $address,
                'post_id' => $post_id,
                'post_title' => $post_title,
                'edit_url' => $edit_url,
                'current_description' => $description,
                'description_length' => $length,
                'pixel_width' => $pixel_width,
                'issue_type' => $issue_type,
                'duplicate_group' => null,
                'category' => $categorization['category'],
                'category_note' => $categorization['note'],
                'new_description' => '',
            ];
        }

        // Mark duplicates
        $all_descriptions = $this->mark_duplicates($all_descriptions, $duplicate_tracker);

        // Categorize into arrays
        $fixable_descriptions = [];
        $manual_descriptions = [];
        $skipped_descriptions = [];
        $missing = [];
        $duplicates = [];
        $too_long = [];
        $too_short = [];

        foreach ($all_descriptions as $desc) {
            // Category-based arrays
            switch ($desc['category']) {
                case 'fixable':
                    $fixable_descriptions[] = $desc;
                    break;
                case 'manual':
                    $manual_descriptions[] = $desc;
                    break;
                case 'skip':
                    $skipped_descriptions[] = $desc;
                    break;
            }

            // Issue type arrays (only for non-skipped)
            if ($desc['category'] !== 'skip') {
                switch ($desc['issue_type']) {
                    case 'missing':
                        $missing[] = $desc;
                        break;
                    case 'duplicate':
                        $duplicates[] = $desc;
                        break;
                    case 'too_long':
                        $too_long[] = $desc;
                        break;
                    case 'too_short':
                        $too_short[] = $desc;
                        break;
                }
            }
        }

        $results = [
            'descriptions' => $all_descriptions,
            'fixable_descriptions' => $fixable_descriptions,
            'manual_descriptions' => $manual_descriptions,
            'skipped_descriptions' => $skipped_descriptions,
            'fixed_descriptions' => [],
            'missing' => $missing,
            'duplicates' => $duplicates,
            'too_long' => $too_long,
            'too_short' => $too_short,
            'total_count' => count($fixable_descriptions) + count($manual_descriptions) + count($skipped_descriptions),
            'fixable_count' => count($fixable_descriptions),
            'manual_count' => count($manual_descriptions),
            'skipped_count' => count($skipped_descriptions),
            'missing_count' => count($missing),
            'duplicate_count' => count($duplicates),
            'too_long_count' => count($too_long),
            'too_short_count' => count($too_short),
            'fixed_count' => 0,
            'processed_at' => current_time('mysql'),
            'seo_plugin' => $seo_plugin,
            'seo_plugin_available' => !empty($seo_plugin),
        ];

        // Save results
        $this->save_results($results);

        // Also save to uploads table for persistence
        $this->save_upload_data($results);

        return $results;
    }

    /**
     * Standardize column names for meta description CSV
     *
     * @param array $parsed Parsed CSV data
     * @return array Standardized data
     */
    private function standardize_columns($parsed) {
        $column_map = [
            'address' => ['address', 'url', 'page url'],
            'meta_description' => ['meta description 1', 'meta description', 'description'],
            'length' => ['meta description 1 length', 'length', 'character count'],
            'pixel_width' => ['meta description 1 pixel width', 'pixel width'],
            'occurrences' => ['occurrences', 'count'],
            'indexability' => ['indexability'],
            'indexability_status' => ['indexability status'],
        ];

        // Create a header map
        $header_map = [];
        foreach ($parsed['headers'] as $index => $header) {
            $normalized = strtolower(trim($header));
            foreach ($column_map as $standard => $variants) {
                if (in_array($normalized, $variants)) {
                    $header_map[$index] = $standard;
                    break;
                }
            }
            if (!isset($header_map[$index])) {
                $header_map[$index] = $normalized;
            }
        }

        // Remap rows
        $remapped_rows = [];
        foreach ($parsed['rows'] as $row) {
            $new_row = [];
            foreach ($row as $key => $value) {
                if (is_numeric($key) && isset($header_map[$key])) {
                    $new_row[$header_map[$key]] = $value;
                } else {
                    // Try to find the standard name
                    $normalized_key = strtolower(trim($key));
                    $found = false;
                    foreach ($column_map as $standard => $variants) {
                        if (in_array($normalized_key, $variants) || $normalized_key === $standard) {
                            $new_row[$standard] = $value;
                            $found = true;
                            break;
                        }
                    }
                    if (!$found) {
                        $new_row[$key] = $value;
                    }
                }
            }
            $remapped_rows[] = $new_row;
        }

        $parsed['rows'] = $remapped_rows;
        return $parsed;
    }

    /**
     * Determine which SEO plugin to use
     *
     * @param array $seo_plugins Active SEO plugins
     * @return string|null Plugin key or null
     */
    private function determine_seo_plugin($seo_plugins) {
        // Priority: Rank Math > Yoast > AIOSEO
        if (isset($seo_plugins['rank-math'])) {
            return 'rank-math';
        }
        if (isset($seo_plugins['yoast'])) {
            return 'yoast';
        }
        if (isset($seo_plugins['aioseo'])) {
            return 'aioseo';
        }
        return null;
    }

    /**
     * Categorize a description based on indexability and URL resolution
     *
     * @param string $address Page URL
     * @param string $indexability Indexability value from CSV
     * @param string $indexability_status Indexability status from CSV
     * @return array Category info
     */
    private function categorize_description($address, $indexability, $indexability_status) {
        // Check indexability first (from CSV)
        if (strtolower($indexability) === 'non-indexable') {
            return ['category' => 'skip', 'note' => __('Non-indexable page', 'screaming-fixes'), 'post_id' => 0];
        }

        if (stripos($indexability_status, 'noindex') !== false) {
            return ['category' => 'skip', 'note' => __('Page has noindex', 'screaming-fixes'), 'post_id' => 0];
        }

        if (stripos($indexability_status, 'canonicalised') !== false) {
            return ['category' => 'skip', 'note' => __('Page is canonicalized', 'screaming-fixes'), 'post_id' => 0];
        }

        // Try to resolve to post ID
        $post_id = url_to_postid($address);

        if (!$post_id) {
            // Try with trailing slash variations
            $post_id = url_to_postid(trailingslashit($address));
            if (!$post_id) {
                $post_id = url_to_postid(untrailingslashit($address));
            }
        }

        if (!$post_id) {
            // Check if it's homepage
            $home_url = trailingslashit(home_url());
            if (trailingslashit($address) === $home_url) {
                return [
                    'category' => 'manual',
                    'note' => __('Homepage - update in SEO plugin settings', 'screaming-fixes'),
                    'post_id' => 0,
                ];
            }

            // URL couldn't be matched - show in manual section so user can still see the data
            // This is common when CSV is from production and testing on local/staging
            return [
                'category' => 'manual',
                'note' => __('URL not found in WordPress - may need manual fix or URL mismatch', 'screaming-fixes'),
                'post_id' => 0,
            ];
        }

        // Check post type - verify it's public and SEO plugin supports it
        $post_type = get_post_type($post_id);
        $post_type_obj = get_post_type_object($post_type);

        // Non-public post types can't have SEO managed
        if (!$post_type_obj || !$post_type_obj->public) {
            return [
                'category' => 'skip',
                'note' => sprintf(__('Post type "%s" is not public', 'screaming-fixes'), $post_type),
                'post_id' => $post_id,
            ];
        }

        // Check if the active SEO plugin supports this post type
        $seo_plugin_check = $this->is_post_type_seo_enabled($post_type);
        if (is_wp_error($seo_plugin_check)) {
            return [
                'category' => 'manual',
                'note' => $seo_plugin_check->get_error_message(),
                'post_id' => $post_id,
            ];
        }

        // Fixable!
        return ['category' => 'fixable', 'note' => '', 'post_id' => $post_id];
    }

    /**
     * Check if an SEO plugin has support enabled for a post type
     *
     * @param string $post_type Post type slug
     * @return true|WP_Error True if supported, WP_Error with reason if not
     */
    private function is_post_type_seo_enabled($post_type) {
        $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();

        if (isset($seo_plugins['rank-math'])) {
            // Check if Rank Math has SEO enabled for this post type
            $general_settings = get_option('rank-math-options-general', []);
            $pt_settings = isset($general_settings['pt_' . $post_type . '_add_meta_box']) ? $general_settings['pt_' . $post_type . '_add_meta_box'] : 'on';

            // Also check the titles & meta settings for this post type
            $titles_settings = get_option('rank-math-options-titles', []);
            $pt_robots = isset($titles_settings['pt_' . $post_type . '_robots']) ? $titles_settings['pt_' . $post_type . '_robots'] : [];

            // If noindex is set globally for this post type, it can't be fixed via meta
            if (is_array($pt_robots) && in_array('noindex', $pt_robots, true)) {
                return new WP_Error('post_type_noindex', sprintf(
                    __('Rank Math sets "%s" as noindex by default', 'screaming-fixes'),
                    $post_type
                ));
            }

            return true;
        } elseif (isset($seo_plugins['yoast'])) {
            // Check if Yoast has the metabox enabled for this post type
            $yoast_options = get_option('wpseo_titles', []);

            // Check if metabox is disabled
            $disabled_key = 'display-metabox-pt-' . $post_type;
            if (isset($yoast_options[$disabled_key]) && $yoast_options[$disabled_key] === false) {
                return new WP_Error('post_type_disabled', sprintf(
                    __('Yoast SEO metabox disabled for "%s"', 'screaming-fixes'),
                    $post_type
                ));
            }

            // Check if this post type is set to noindex by default
            $noindex_key = 'noindex-pt-' . $post_type;
            if (isset($yoast_options[$noindex_key]) && $yoast_options[$noindex_key] === true) {
                return new WP_Error('post_type_noindex', sprintf(
                    __('Yoast sets "%s" as noindex by default', 'screaming-fixes'),
                    $post_type
                ));
            }

            return true;
        } elseif (isset($seo_plugins['aioseo'])) {
            // AIOSEO - basic support, assume enabled if plugin is active
            // More complex checks could be added for AIOSEO's dynamic settings
            return true;
        }

        // No SEO plugin active
        return new WP_Error('no_seo_plugin', __('No supported SEO plugin detected', 'screaming-fixes'));
    }

    /**
     * Determine issue type for a description
     *
     * @param string $description Description text
     * @param int $length Character count
     * @param int $occurrences Occurrence count from CSV
     * @return string Issue type
     */
    private function determine_issue_type($description, $length, $occurrences) {
        if (empty($description) || $length === 0) {
            return 'missing';
        }

        if ($occurrences > 1) {
            return 'duplicate';
        }

        if ($length > 155) {
            return 'too_long';
        }

        if ($length < 70) {
            return 'too_short';
        }

        return 'none';
    }

    /**
     * Mark duplicates in the descriptions array
     *
     * @param array $descriptions All descriptions
     * @param array $duplicate_tracker Hash map of description to indices
     * @return array Updated descriptions
     */
    private function mark_duplicates($descriptions, $duplicate_tracker) {
        foreach ($duplicate_tracker as $hash => $indices) {
            if (count($indices) > 1) {
                foreach ($indices as $index) {
                    $descriptions[$index]['issue_type'] = 'duplicate';
                    $descriptions[$index]['duplicate_group'] = $hash;
                    $descriptions[$index]['duplicate_count'] = count($indices);
                }
            }
        }
        return $descriptions;
    }

    /**
     * Get current meta description for a post
     *
     * @param int $post_id Post ID
     * @return string Current description
     */
    public function get_current_description($post_id) {
        $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();

        if (isset($seo_plugins['rank-math'])) {
            return get_post_meta($post_id, 'rank_math_description', true);
        } elseif (isset($seo_plugins['yoast'])) {
            return get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
        } elseif (isset($seo_plugins['aioseo'])) {
            return get_post_meta($post_id, '_aioseo_description', true);
        }

        return '';
    }

    /**
     * Update meta description for a post
     *
     * @param int $post_id Post ID
     * @param string $new_description New description
     * @return array|WP_Error Result
     */
    public function update_meta_description($post_id, $new_description) {
        $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();

        // Verify the post exists and is a valid post type
        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error('invalid_post', __('Post not found.', 'screaming-fixes'));
        }

        // Check if this is a post type that the SEO plugin can manage
        // Most SEO plugins only work with public post types
        $post_type_obj = get_post_type_object($post->post_type);
        if (!$post_type_obj || !$post_type_obj->public) {
            return new WP_Error('unsupported_post_type', sprintf(
                __('Post type "%s" is not publicly queryable and cannot have SEO meta descriptions managed.', 'screaming-fixes'),
                $post->post_type
            ));
        }

        // Get original value for undo tracking
        $original = $this->get_current_description($post_id);

        // Determine which plugin to use and its meta key (priority order)
        if (isset($seo_plugins['rank-math'])) {
            $meta_key = 'rank_math_description';
            $plugin_used = 'rank-math';

            // Check if Rank Math has SEO enabled for this post type
            $disabled_post_types = get_option('rank_math_pt_disabled_cpt', []);
            if (is_array($disabled_post_types) && in_array($post->post_type, $disabled_post_types, true)) {
                return new WP_Error('post_type_disabled', sprintf(
                    __('Rank Math does not manage SEO for the "%s" post type. Enable it in Rank Math settings.', 'screaming-fixes'),
                    $post->post_type
                ));
            }
        } elseif (isset($seo_plugins['yoast'])) {
            $meta_key = '_yoast_wpseo_metadesc';
            $plugin_used = 'yoast';

            // Check if Yoast has SEO enabled for this post type
            $yoast_options = get_option('wpseo_titles', []);
            $disabled_key = 'display-metabox-pt-' . $post->post_type;
            if (isset($yoast_options[$disabled_key]) && $yoast_options[$disabled_key] === false) {
                return new WP_Error('post_type_disabled', sprintf(
                    __('Yoast SEO does not manage SEO for the "%s" post type. Enable it in Yoast settings.', 'screaming-fixes'),
                    $post->post_type
                ));
            }
        } elseif (isset($seo_plugins['aioseo'])) {
            $meta_key = '_aioseo_description';
            $plugin_used = 'aioseo';

            // AIOSEO stores post type settings differently - check if it's excluded
            // Note: AIOSEO's settings are more complex, basic check for common exclusions
        } else {
            return new WP_Error('no_seo_plugin', __('No supported SEO plugin detected.', 'screaming-fixes'));
        }

        // Perform the update
        $update_result = update_post_meta($post_id, $meta_key, $new_description);

        // Verify the write was successful by reading it back
        // Note: We need to clear any object cache to get the fresh value
        wp_cache_delete($post_id, 'post_meta');
        $saved_value = get_post_meta($post_id, $meta_key, true);

        // Use a normalized comparison to account for WordPress sanitization
        // WordPress may modify whitespace, convert smart quotes, or normalize unicode
        $normalized_saved = $this->normalize_for_comparison($saved_value);
        $normalized_expected = $this->normalize_for_comparison($new_description);

        // Check if the values are substantially the same
        // Also check if the saved value is non-empty when we expected to save something
        $save_succeeded = false;

        if ($normalized_saved === $normalized_expected) {
            // Values match after normalization
            $save_succeeded = true;
        } elseif (!empty($new_description) && !empty($saved_value)) {
            // Both have content - check if they're similar enough (WordPress may have sanitized)
            // Consider success if the saved value starts with the expected content
            // This handles cases where WordPress truncates or modifies the end
            $save_succeeded = (strpos($normalized_saved, substr($normalized_expected, 0, 50)) === 0) ||
                              (strpos($normalized_expected, substr($normalized_saved, 0, 50)) === 0);
        }

        if (!$save_succeeded) {
            // Log for debugging
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF Meta Description: Save verification failed for post ' . $post_id);
                error_log('SF Meta Description: Expected (normalized): ' . $normalized_expected);
                error_log('SF Meta Description: Got (normalized): ' . $normalized_saved);
                error_log('SF Meta Description: update_post_meta returned: ' . var_export($update_result, true));
            }

            return new WP_Error('write_failed', sprintf(
                __('Failed to save meta description. The value could not be written to post ID %d.', 'screaming-fixes'),
                $post_id
            ));
        }

        // Log the change
        $logger = new SF_Change_Logger();
        $logger->log_change($post_id, 'meta_description', $original, $new_description, [
            'module' => 'meta-description',
            'meta_key' => $meta_key,
            'plugin' => $plugin_used,
        ]);

        return [
            'success' => true,
            'original' => $original,
            'new' => $new_description,
            'plugin' => $plugin_used,
        ];
    }

    /**
     * Normalize a string for comparison
     * Handles common WordPress/SEO plugin sanitization differences
     *
     * This comprehensive normalization handles character variations that occur when:
     * - WordPress sanitizes content on save
     * - SEO plugins (Yoast, Rank Math, AIOSEO) process meta values
     * - Copy/paste from different sources (Word, Google Docs, etc.)
     * - Different character encodings or keyboard layouts
     *
     * @param string $str String to normalize
     * @return string Normalized string
     */
    private function normalize_for_comparison($str) {
        if (!is_string($str)) {
            return '';
        }

        // Step 0: Strip slashes (WordPress magic quotes can add escaped quotes like \' and \")
        // This must happen before HTML entity decoding
        $str = stripslashes($str);

        // Step 1: Decode HTML entities first (SEO plugins may encode/decode these)
        // This handles &amp; &quot; &apos; &nbsp; &mdash; &ndash; etc.
        $str = html_entity_decode($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // Step 2: Trim whitespace
        $str = trim($str);

        // Step 3: Normalize line endings
        $str = str_replace(["\r\n", "\r"], "\n", $str);

        // Step 4: Normalize special whitespace characters to regular spaces
        $str = str_replace(
            [
                "\u{00A0}",   // Non-breaking space
                "\u{2002}",   // En space
                "\u{2003}",   // Em space
                "\u{2004}",   // Three-per-em space
                "\u{2005}",   // Four-per-em space
                "\u{2006}",   // Six-per-em space
                "\u{2007}",   // Figure space
                "\u{2008}",   // Punctuation space
                "\u{2009}",   // Thin space
                "\u{200A}",   // Hair space
                "\u{200B}",   // Zero-width space (remove)
                "\u{FEFF}",   // Zero-width no-break space / BOM (remove)
                "\u{202F}",   // Narrow no-break space
                "\u{205F}",   // Medium mathematical space
            ],
            [
                ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', '', '', ' ', ' ',
            ],
            $str
        );

        // Step 5: Collapse multiple spaces to single space
        $str = preg_replace('/\s+/', ' ', $str);

        // Step 6: Normalize quotes and apostrophes
        $str = str_replace(
            [
                // Smart/curly double quotes
                "\u{201C}",   // " Left double quotation mark
                "\u{201D}",   // " Right double quotation mark
                "\u{201E}",   // „ Double low-9 quotation mark (German)
                "\u{201F}",   // ‟ Double high-reversed-9 quotation mark
                "\u{00AB}",   // « Left-pointing double angle quotation (French)
                "\u{00BB}",   // » Right-pointing double angle quotation (French)
                // Smart/curly single quotes and apostrophes
                "\u{2018}",   // ' Left single quotation mark
                "\u{2019}",   // ' Right single quotation mark (apostrophe)
                "\u{201A}",   // ‚ Single low-9 quotation mark
                "\u{201B}",   // ‛ Single high-reversed-9 quotation mark
                "\u{2039}",   // ‹ Single left-pointing angle quotation
                "\u{203A}",   // › Single right-pointing angle quotation
                // Prime marks (feet/inches/minutes/seconds)
                "\u{2032}",   // ′ Prime (feet, minutes)
                "\u{2033}",   // ″ Double prime (inches, seconds)
                "\u{2034}",   // ‴ Triple prime
                "\u{2035}",   // ‵ Reversed prime
                "\u{2036}",   // ‶ Reversed double prime
                "\u{2037}",   // ‷ Reversed triple prime
                // Modifier letters
                "\u{02B9}",   // ʹ Modifier letter prime
                "\u{02BA}",   // ʺ Modifier letter double prime
                "\u{02BB}",   // ʻ Modifier letter turned comma
                "\u{02BC}",   // ʼ Modifier letter apostrophe
                "\u{02BD}",   // ʽ Modifier letter reversed comma
                // Accents sometimes used as quotes/apostrophes
                "\u{00B4}",   // ´ Acute accent
                "\u{0060}",   // ` Grave accent
                "\u{02CA}",   // ˊ Modifier letter acute accent
                "\u{02CB}",   // ˋ Modifier letter grave accent
            ],
            [
                '"', '"', '"', '"', '"', '"',
                "'", "'", "'", "'", "'", "'",
                "'", '"', "'''", "'", '"', "'''",
                "'", '"', "'", "'", "'",
                "'", "'", "'", "'",
            ],
            $str
        );

        // Step 7: Normalize dashes and hyphens
        $str = str_replace(
            [
                "\u{2010}",   // ‐ Hyphen
                "\u{2011}",   // ‑ Non-breaking hyphen
                "\u{2012}",   // ‒ Figure dash
                "\u{2013}",   // – En dash
                "\u{2014}",   // — Em dash
                "\u{2015}",   // ― Horizontal bar
                "\u{2212}",   // − Minus sign
                "\u{FE58}",   // ﹘ Small em dash
                "\u{FE63}",   // ﹣ Small hyphen-minus
                "\u{FF0D}",   // - Fullwidth hyphen-minus
            ],
            '-',
            $str
        );

        // Step 8: Normalize common symbols
        $str = str_replace(
            [
                // Ellipsis
                "\u{2026}",   // … Horizontal ellipsis
                // Multiplication/dimension signs
                "\u{00D7}",   // × Multiplication sign
                "\u{2715}",   // ✕ Multiplication X
                "\u{2716}",   // ✖ Heavy multiplication X
                // Division
                "\u{00F7}",   // ÷ Division sign
                // Bullets and dots
                "\u{2022}",   // • Bullet
                "\u{2023}",   // ‣ Triangular bullet
                "\u{2043}",   // ⁃ Hyphen bullet
                "\u{00B7}",   // · Middle dot
                "\u{2027}",   // ‧ Hyphenation point
                // Arrows (normalize to text)
                "\u{2192}",   // → Rightwards arrow
                "\u{2190}",   // ← Leftwards arrow
                "\u{2194}",   // ↔ Left right arrow
                "\u{21D2}",   // ⇒ Rightwards double arrow
                "\u{21D0}",   // ⇐ Leftwards double arrow
            ],
            [
                '...',
                'x', 'x', 'x',
                '/',
                '-', '-', '-', '.', '.',
                '->', '<-', '<->', '=>', '<=',
            ],
            $str
        );

        // Step 9: Normalize common fractions to decimal or text
        $str = str_replace(
            [
                "\u{00BC}",   // ¼
                "\u{00BD}",   // ½
                "\u{00BE}",   // ¾
                "\u{2153}",   // ⅓
                "\u{2154}",   // ⅔
                "\u{2155}",   // ⅕
                "\u{2156}",   // ⅖
                "\u{2157}",   // ⅗
                "\u{2158}",   // ⅘
                "\u{2159}",   // ⅙
                "\u{215A}",   // ⅚
                "\u{215B}",   // ⅛
                "\u{215C}",   // ⅜
                "\u{215D}",   // ⅝
                "\u{215E}",   // ⅞
            ],
            [
                '1/4', '1/2', '3/4',
                '1/3', '2/3',
                '1/5', '2/5', '3/5', '4/5',
                '1/6', '5/6',
                '1/8', '3/8', '5/8', '7/8',
            ],
            $str
        );

        // Step 10: Normalize superscript/subscript numbers
        $str = str_replace(
            [
                // Superscripts
                "\u{00B9}",   // ¹
                "\u{00B2}",   // ²
                "\u{00B3}",   // ³
                "\u{2070}",   // ⁰
                "\u{2074}",   // ⁴
                "\u{2075}",   // ⁵
                "\u{2076}",   // ⁶
                "\u{2077}",   // ⁷
                "\u{2078}",   // ⁸
                "\u{2079}",   // ⁹
                // Subscripts
                "\u{2080}",   // ₀
                "\u{2081}",   // ₁
                "\u{2082}",   // ₂
                "\u{2083}",   // ₃
                "\u{2084}",   // ₄
                "\u{2085}",   // ₅
                "\u{2086}",   // ₆
                "\u{2087}",   // ₇
                "\u{2088}",   // ₈
                "\u{2089}",   // ₉
            ],
            [
                '1', '2', '3', '0', '4', '5', '6', '7', '8', '9',
                '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
            ],
            $str
        );

        // Step 11: Normalize degree and similar symbols
        $str = str_replace(
            [
                "\u{00B0}",   // ° Degree sign
                "\u{00BA}",   // º Masculine ordinal indicator
                "\u{00AA}",   // ª Feminine ordinal indicator
                "\u{2103}",   // ℃ Degree Celsius
                "\u{2109}",   // ℉ Degree Fahrenheit
            ],
            [
                ' degrees', '', '', ' C', ' F',
            ],
            $str
        );

        // Step 12: Normalize ampersand variations
        $str = str_replace(
            [
                "\u{FF06}",   // ＆ Fullwidth ampersand
                "\u{FE60}",   // ﹠ Small ampersand
            ],
            '&',
            $str
        );

        // Step 13: Normalize unicode characters using PHP normalizer (if available)
        // This handles combining characters and other unicode normalization
        if (function_exists('normalizer_normalize')) {
            $str = normalizer_normalize($str, Normalizer::FORM_C);
        }

        // Convert to lowercase for case-insensitive comparison
        $str = mb_strtolower($str, 'UTF-8');

        return $str;
    }

    /**
     * Apply approved meta description fixes
     *
     * @param array $fixes Array of fixes to apply
     * @return array Results with success/failure counts
     */
    public function apply_fixes($fixes) {
        $results = [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => [],
            'details' => [],
        ];

        // Track successful and failed fixes separately
        $successful_fixes = [];
        $failed_fixes = [];

        // Prepare fixes for batch tracking with original values
        $fixes_with_originals = [];
        foreach ($fixes as $fix) {
            $post_id = isset($fix['post_id']) ? (int) $fix['post_id'] : 0;
            if ($post_id) {
                $fix['original_description'] = $this->get_current_description($post_id);
            }
            $fixes_with_originals[] = $fix;
        }

        // Start batch tracking for undo capability
        SF_Batch_Restore::start_batch('meta-description', $fixes_with_originals);

        foreach ($fixes_with_originals as $fix) {
            $post_id = isset($fix['post_id']) ? (int) $fix['post_id'] : 0;
            $new_description = isset($fix['new_description']) ? trim($fix['new_description']) : '';
            $address = isset($fix['address']) ? $fix['address'] : '';

            // Skip if no post ID or description
            if (!$post_id || empty($new_description)) {
                $results['skipped']++;
                continue;
            }

            $results['total']++;

            $update_result = $this->update_meta_description($post_id, $new_description);

            if (is_wp_error($update_result)) {
                $results['failed']++;
                $failed_fix = [
                    'post_id' => $post_id,
                    'address' => $address,
                    'new_description' => $new_description,
                    'original_description' => $fix['original_description'] ?? '',
                    'error' => $update_result->get_error_message(),
                ];
                $results['errors'][] = $failed_fix;
                $failed_fixes[] = $failed_fix;
            } else {
                $results['success']++;
                $successful_fix = [
                    'post_id' => $post_id,
                    'address' => $address,
                    'original' => $update_result['original'],
                    'new' => $new_description,
                    'plugin' => $update_result['plugin'],
                ];
                $results['details'][] = $successful_fix;
                $successful_fixes[] = array_merge($fix, $successful_fix);
            }
        }

        // Update stored results to reflect fixes applied - pass actual success/failure info
        $this->update_results_after_fixes($successful_fixes, $failed_fixes);

        // Complete batch tracking
        SF_Batch_Restore::complete_batch($results);

        return $results;
    }

    /**
     * Get AI suggestion for meta description
     *
     * @param string $address Page URL
     * @param string $post_title Page title
     * @param string $current_description Current description
     * @return string|WP_Error Suggested description or error
     */
    public function get_ai_suggestion($address, $post_title, $current_description = '') {
        $api_key = get_option('sf_claude_api_key');

        if (empty($api_key)) {
            return new WP_Error('no_api_key', __('Claude API key not configured. Add it in Settings.', 'screaming-fixes'));
        }

        $prompt = sprintf(
            "Write a compelling meta description for a web page.\n\n" .
            "Page URL: %s\n" .
            "Page Title: %s\n" .
            "Current Description: %s\n\n" .
            "Guidelines:\n" .
            "- Length: 120-155 characters (optimal for search results)\n" .
            "- Include a call to action when appropriate\n" .
            "- Be specific and descriptive\n" .
            "- Don't start with the page title\n" .
            "- Make it compelling for users to click\n\n" .
            "Respond with ONLY the meta description text, nothing else.",
            $address,
            $post_title,
            $current_description ?: '(none)'
        );

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode([
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 200,
                'messages' => [
                    ['role' => 'user', 'content' => $prompt],
                ],
            ]),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return new WP_Error('api_error', sprintf(__('API returned status %d', 'screaming-fixes'), $status_code));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['content'][0]['text'])) {
            $suggestion = trim($body['content'][0]['text']);
            // Remove quotes if the AI wrapped the response in them
            $suggestion = trim($suggestion, '"\'');
            return $suggestion;
        }

        return new WP_Error('api_error', __('Unexpected API response', 'screaming-fixes'));
    }

    /**
     * Get issue count for dashboard display
     *
     * @return int Count of meta description issues
     */
    public function get_issue_count() {
        $results = $this->get_results();

        if (empty($results) || !isset($results['total_count'])) {
            $upload_data = $this->get_upload_data();
            if ($upload_data && isset($upload_data['total_count'])) {
                return (int) $upload_data['total_count'];
            }
            return 0;
        }

        return (int) $results['total_count'];
    }

    /**
     * AJAX: Process uploaded CSV
     */
    public function ajax_process_csv() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

        if (empty($upload_id)) {
            wp_send_json_error(['message' => __('No upload ID provided.', 'screaming-fixes')]);
        }

        // Get the uploaded file path from the uploads option
        $uploads = get_option('sf_pending_uploads', []);

        if (!isset($uploads[$upload_id])) {
            wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
        }

        $file_path = $uploads[$upload_id]['path'];

        if (!file_exists($file_path)) {
            wp_send_json_error(['message' => __('File not found.', 'screaming-fixes')]);
        }

        // Check if this is a bulk update CSV (has new_meta column)
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (!is_wp_error($parsed) && !empty($parsed['headers'])) {
            $bulk_check = $this->is_bulk_update_csv($parsed['headers']);

            // Check for bulk update errors first (e.g., multiple update columns, missing URL column)
            if (is_array($bulk_check) && isset($bulk_check['error'])) {
                if ($bulk_check['error'] === 'multiple_update_columns') {
                    wp_send_json_error(['message' => __('Please upload a CSV with only one update column at a time. This file contains multiple update columns.', 'screaming-fixes')]);
                    return;
                }
                if ($bulk_check['error'] === 'no_url_column') {
                    wp_send_json_error(['message' => __('Bulk update CSV detected but missing a URL column. Please include a column named "url", "address", or "page_url".', 'screaming-fixes')]);
                    return;
                }
            }

            // If it's a valid bulk update CSV, process it differently
            if (is_array($bulk_check) && isset($bulk_check['is_bulk']) && $bulk_check['is_bulk'] === true) {
                $results = $this->process_bulk_csv($file_path);

                if (is_wp_error($results)) {
                    wp_send_json_error(['message' => $results->get_error_message()]);
                }

                // Clean up the temp file reference
                unset($uploads[$upload_id]);
                update_option('sf_pending_uploads', $uploads);

                wp_send_json_success([
                    'message' => sprintf(
                        __('%d meta descriptions ready to update', 'screaming-fixes'),
                        $results['ready_count']
                    ),
                    'data' => $results,
                    'is_bulk_update' => true,
                ]);
                return;
            }
        }

        // Standard processing for analysis CSVs
        $results = $this->process_csv($file_path);

        if (is_wp_error($results)) {
            wp_send_json_error(['message' => $results->get_error_message()]);
        }

        // Clean up the temp file reference
        unset($uploads[$upload_id]);
        update_option('sf_pending_uploads', $uploads);

        wp_send_json_success([
            'message' => sprintf(
                __('Found %d meta description issues: %d fixable, %d manual, %d skipped.', 'screaming-fixes'),
                $results['total_count'],
                $results['fixable_count'],
                $results['manual_count'],
                $results['skipped_count']
            ),
            'data' => $results,
        ]);
    }

    /**
     * AJAX: Get stored data
     */
    public function ajax_get_data() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            wp_send_json_success(['data' => null]);
        }

        wp_send_json_success(['data' => $results]);
    }

    /**
     * AJAX: Apply fixes
     */
    public function ajax_apply_fixes() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $fixes = isset($_POST['fixes']) ? $_POST['fixes'] : [];

        if (empty($fixes)) {
            wp_send_json_error(['message' => __('No fixes provided.', 'screaming-fixes')]);
        }

        // Sanitize fixes array
        // Note: wp_unslash() is required because WordPress adds magic quotes to $_POST data
        $sanitized_fixes = [];
        foreach ($fixes as $fix) {
            $sanitized_fixes[] = [
                'post_id' => isset($fix['post_id']) ? absint($fix['post_id']) : 0,
                'address' => isset($fix['address']) ? esc_url_raw(wp_unslash($fix['address'])) : '',
                'new_description' => isset($fix['new_description']) ? sanitize_textarea_field(wp_unslash($fix['new_description'])) : '',
            ];
        }

        $results = $this->apply_fixes($sanitized_fixes);

        // Log to activity log for dashboard
        if ($results['success'] > 0) {
            SF_Activity_Log::log('meta-description', $results['success']);
        }

        wp_send_json_success([
            'message' => sprintf(
                __('Updated %d meta descriptions. %d failed.', 'screaming-fixes'),
                $results['success'],
                $results['failed']
            ),
            'results' => $results,
        ]);
    }

    /**
     * AJAX: Get AI suggestion for a single description
     */
    public function ajax_get_ai_suggestion() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $address = isset($_POST['address']) ? esc_url_raw(wp_unslash($_POST['address'])) : '';
        $post_title = isset($_POST['post_title']) ? sanitize_text_field(wp_unslash($_POST['post_title'])) : '';
        $current_description = isset($_POST['current_description']) ? sanitize_textarea_field(wp_unslash($_POST['current_description'])) : '';

        if (empty($address)) {
            wp_send_json_error(['message' => __('No page URL provided.', 'screaming-fixes')]);
        }

        $suggestion = $this->get_ai_suggestion($address, $post_title, $current_description);

        if (is_wp_error($suggestion)) {
            wp_send_json_error(['message' => $suggestion->get_error_message()]);
        }

        wp_send_json_success([
            'suggestion' => $suggestion,
            'address' => $address,
        ]);
    }

    /**
     * AJAX: Clear all meta description data
     */
    public function ajax_clear_data() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Clear transient results
        $this->clear_results();

        // Clear database records
        $this->clear_upload_data();

        wp_send_json_success([
            'message' => __('Meta description data cleared. You can now upload a new CSV file.', 'screaming-fixes'),
        ]);
    }

    /**
     * Clear upload data from database
     */
    private function clear_upload_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);
    }

    /**
     * Save upload data to database
     *
     * @param array $data Data to save
     */
    private function save_upload_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        // Delete any existing data for this session/module
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Insert new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get upload data from database
     *
     * @return array|null Data or null if not found
     */
    private function get_upload_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            return json_decode($row->data, true);
        }

        return null;
    }

    /**
     * Get or create session ID
     *
     * @return string Session ID
     */
    private function get_session_id() {
        if (!session_id()) {
            return 'user_' . get_current_user_id();
        }
        return session_id();
    }

    /**
     * Update results after fixes are applied
     *
     * @param array $successful_fixes Array of successfully applied fixes
     * @param array $failed_fixes Array of fixes that failed to apply
     */
    private function update_results_after_fixes($successful_fixes, $failed_fixes = []) {
        $results = $this->get_results();

        // Also check database if transient is empty
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results) || empty($results['descriptions'])) {
            return;
        }

        // Initialize arrays if not exists
        if (!isset($results['fixed_descriptions'])) {
            $results['fixed_descriptions'] = [];
        }

        // Create a map of successful fixes by address
        $fixed_map = [];
        foreach ($successful_fixes as $fix) {
            $address = $fix['address'] ?? '';
            if (!empty($address)) {
                $fixed_map[$address] = [
                    'new_description' => $fix['new_description'] ?? $fix['new'] ?? '',
                    'original_description' => $fix['original_description'] ?? $fix['original'] ?? '',
                ];
            }
        }

        // Create a map of failed fixes by address
        $failed_map = [];
        foreach ($failed_fixes as $fix) {
            $address = $fix['address'] ?? '';
            if (!empty($address)) {
                $failed_map[$address] = [
                    'new_description' => $fix['new_description'] ?? '',
                    'original_description' => $fix['original_description'] ?? '',
                    'error' => $fix['error'] ?? __('Unknown error', 'screaming-fixes'),
                    'post_id' => $fix['post_id'] ?? 0,
                ];
            }
        }

        // Move fixed/failed descriptions from descriptions array to fixed_descriptions with status
        $remaining_descriptions = [];
        $fixable_descriptions = [];
        $manual_descriptions = [];
        $skipped_descriptions = [];

        foreach ($results['descriptions'] as $desc) {
            $address = $desc['address'] ?? '';

            if (isset($fixed_map[$address])) {
                // This was successfully fixed
                $desc['applied_description'] = $fixed_map[$address]['new_description'];
                $desc['original_description'] = $fixed_map[$address]['original_description'];
                $desc['fixed_at'] = current_time('mysql');
                $desc['status'] = 'success';
                $desc['status_message'] = '';
                $results['fixed_descriptions'][] = $desc;
            } elseif (isset($failed_map[$address])) {
                // This failed to update - also goes to fixed_descriptions with failed status
                $desc['applied_description'] = $failed_map[$address]['new_description'];
                $desc['original_description'] = $failed_map[$address]['original_description'];
                $desc['fixed_at'] = current_time('mysql');
                $desc['status'] = 'failed';
                $desc['status_message'] = $failed_map[$address]['error'];
                $results['fixed_descriptions'][] = $desc;
            } else {
                // Not processed - keep in appropriate category
                $remaining_descriptions[] = $desc;
                $category = $desc['category'] ?? 'fixable';
                switch ($category) {
                    case 'fixable':
                        $fixable_descriptions[] = $desc;
                        break;
                    case 'manual':
                        $manual_descriptions[] = $desc;
                        break;
                    case 'skip':
                        $skipped_descriptions[] = $desc;
                        break;
                }
            }
        }

        // Update results arrays
        $results['descriptions'] = $remaining_descriptions;
        $results['fixable_descriptions'] = $fixable_descriptions;
        $results['manual_descriptions'] = $manual_descriptions;
        $results['skipped_descriptions'] = $skipped_descriptions;

        // Recalculate issue type arrays from remaining
        $results['missing'] = array_filter($remaining_descriptions, function($d) {
            return ($d['issue_type'] ?? '') === 'missing' && ($d['category'] ?? '') !== 'skip';
        });
        $results['duplicates'] = array_filter($remaining_descriptions, function($d) {
            return ($d['issue_type'] ?? '') === 'duplicate' && ($d['category'] ?? '') !== 'skip';
        });
        $results['too_long'] = array_filter($remaining_descriptions, function($d) {
            return ($d['issue_type'] ?? '') === 'too_long' && ($d['category'] ?? '') !== 'skip';
        });
        $results['too_short'] = array_filter($remaining_descriptions, function($d) {
            return ($d['issue_type'] ?? '') === 'too_short' && ($d['category'] ?? '') !== 'skip';
        });

        // Re-index arrays
        $results['missing'] = array_values($results['missing']);
        $results['duplicates'] = array_values($results['duplicates']);
        $results['too_long'] = array_values($results['too_long']);
        $results['too_short'] = array_values($results['too_short']);

        // Update counts
        $results['total_count'] = count($remaining_descriptions);
        $results['fixable_count'] = count($fixable_descriptions);
        $results['manual_count'] = count($manual_descriptions);
        $results['skipped_count'] = count($skipped_descriptions);
        $results['missing_count'] = count($results['missing']);
        $results['duplicate_count'] = count($results['duplicates']);
        $results['too_long_count'] = count($results['too_long']);
        $results['too_short_count'] = count($results['too_short']);
        $results['fixed_count'] = count($results['fixed_descriptions']);

        // Save updated results to both transient and database
        $this->save_results($results);
        $this->save_upload_data($results);
    }

    /**
     * Get module ID
     *
     * @return string Module ID
     */
    public function get_module_id() {
        return $this->module_id;
    }

    /**
     * ============================================
     * BULK UPLOAD METHODS
     * ============================================
     */

    /**
     * Normalize a column name for flexible matching
     * Lowercases, removes spaces, dashes, and underscores
     *
     * @param string $name Column name
     * @return string Normalized name
     */
    private function normalize_column_name($name) {
        $normalized = strtolower(trim($name));
        $normalized = str_replace([' ', '-', '_'], '', $normalized);
        return $normalized;
    }

    /**
     * Normalize a URL for matching
     * Strips trailing slashes and normalizes protocol
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private function normalize_url($url) {
        $url = trim($url);
        // Remove trailing slash
        $url = rtrim($url, '/');
        // Normalize protocol - convert http to https for comparison
        $url = preg_replace('/^http:\/\//i', 'https://', $url);
        // Convert to lowercase for comparison
        $url = strtolower($url);
        return $url;
    }

    /**
     * Check if CSV is a bulk update CSV (has new_meta column)
     *
     * @param array $headers CSV headers
     * @return array|false Array with column header names or false if not a bulk CSV
     */
    public function is_bulk_update_csv($headers) {
        $new_meta_header = null;
        $url_header = null;
        $update_columns_found = [];

        // Known update column patterns (for detecting multiple update columns)
        $update_column_patterns = ['newtitle', 'newdescription', 'newmetadescription', 'newmeta', 'newalttext'];

        // Meta description column patterns
        $meta_patterns = ['newmeta', 'newmetadescription', 'newdescription'];

        // URL column patterns
        $url_patterns = ['url', 'address', 'pageurl'];

        foreach ($headers as $header) {
            $normalized = $this->normalize_column_name($header);
            // The actual header name (lowercase, as returned by parser)
            $header_lower = strtolower(trim($header));

            // Check for new_meta/new_description column
            if (in_array($normalized, $meta_patterns)) {
                $new_meta_header = $header_lower;
                $update_columns_found[] = 'new_meta';
            }

            // Check for other update columns (for validation)
            if ($normalized === 'newtitle') {
                $update_columns_found[] = 'new_title';
            }
            if ($normalized === 'newalttext') {
                $update_columns_found[] = 'new_alt_text';
            }

            // Check for URL column
            if (in_array($normalized, $url_patterns)) {
                $url_header = $header_lower;
            }
        }

        // Check for multiple update columns error
        if (count($update_columns_found) > 1) {
            return ['error' => 'multiple_update_columns', 'columns' => $update_columns_found];
        }

        // Not a bulk update CSV if no new_meta column
        if ($new_meta_header === null) {
            return false;
        }

        // Must have URL column
        if ($url_header === null) {
            return ['error' => 'no_url_column'];
        }

        return [
            'is_bulk' => true,
            'new_meta_header' => $new_meta_header,
            'url_header' => $url_header,
        ];
    }

    /**
     * Process a bulk update CSV file
     *
     * @param string $file_path Path to CSV file
     * @return array|WP_Error Processing results
     */
    public function process_bulk_csv($file_path) {
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Check if this is a bulk update CSV
        $bulk_check = $this->is_bulk_update_csv($parsed['headers']);

        if ($bulk_check === false) {
            return new WP_Error('not_bulk_csv', __('CSV must contain a \'new_meta\' or \'new_description\' column for bulk updates.', 'screaming-fixes'));
        }

        if (isset($bulk_check['error'])) {
            if ($bulk_check['error'] === 'multiple_update_columns') {
                return new WP_Error('multiple_update_columns', __('Please upload a CSV with only one update column at a time. This file contains multiple update columns.', 'screaming-fixes'));
            }
            if ($bulk_check['error'] === 'no_url_column') {
                return new WP_Error('no_url_column', __('CSV must contain a URL column (url, address, or page_url).', 'screaming-fixes'));
            }
        }

        // Get the header names (these are keys in the row associative arrays)
        $new_meta_header = $bulk_check['new_meta_header'];
        $url_header = $bulk_check['url_header'];

        // Get SEO plugin for later use
        $seo_plugins = SF_Plugin_Detector::get_active_seo_plugins();
        $seo_plugin = $this->determine_seo_plugin($seo_plugins);

        if (empty($seo_plugin)) {
            return new WP_Error('no_seo_plugin', __('No supported SEO plugin detected. Please install Rank Math, Yoast SEO, or All in One SEO.', 'screaming-fixes'));
        }

        // Build WordPress URL lookup map
        $wp_urls = $this->build_wp_url_map();

        // Process rows - handle duplicates by using last occurrence
        $url_to_row = []; // Track URLs to handle duplicates
        $skipped_empty = [];
        $duplicate_urls = [];

        foreach ($parsed['rows'] as $row_index => $row) {
            // Get values by header name (CSV parser returns associative arrays)
            $url = isset($row[$url_header]) ? trim($row[$url_header]) : '';
            $new_meta = isset($row[$new_meta_header]) ? trim($row[$new_meta_header]) : '';

            if (empty($url)) {
                continue; // Skip rows without URL
            }

            // Skip rows with empty new_meta
            if (empty($new_meta)) {
                $skipped_empty[] = [
                    'url' => $url,
                    'current_description' => '',
                    'new_description' => '',
                    'status' => 'Skipped - No new meta description',
                ];
                continue;
            }

            $normalized_url = $this->normalize_url($url);

            // Track duplicates (we'll use last occurrence)
            if (isset($url_to_row[$normalized_url])) {
                $duplicate_urls[$normalized_url] = true;
            }

            // Store/overwrite with this row data
            $url_to_row[$normalized_url] = [
                'url' => $url,
                'new_description' => $new_meta,
            ];
        }

        // Now build the ready/not_matched arrays
        $ready_updates = [];
        $not_matched = [];

        foreach ($url_to_row as $normalized_url => $row_data) {
            $url = $row_data['url'];
            $new_meta = $row_data['new_description'];

            // Try to match URL to WordPress
            $post_id = null;
            $current_description = '';

            if (isset($wp_urls[$normalized_url])) {
                $post_id = $wp_urls[$normalized_url]['post_id'];
                $current_description = $this->get_current_description($post_id);
            }

            // Check for overly long descriptions (over 300 chars)
            $length_warning = strlen($new_meta) > 300;

            if ($post_id) {
                $ready_updates[] = [
                    'url' => $url,
                    'post_id' => $post_id,
                    'current_description' => $current_description,
                    'new_description' => $new_meta,
                    'status' => 'Ready',
                    'length_warning' => $length_warning,
                    'char_count' => strlen($new_meta),
                ];
            } else {
                $not_matched[] = [
                    'url' => $url,
                    'current_description' => '',
                    'new_description' => $new_meta,
                    'status' => 'Skipped - URL not found',
                    'length_warning' => $length_warning,
                    'char_count' => strlen($new_meta),
                ];
            }
        }

        $results = [
            'is_bulk_update' => true,
            'ready_updates' => $ready_updates,
            'not_matched' => $not_matched,
            'skipped_empty' => $skipped_empty,
            'ready_count' => count($ready_updates),
            'not_matched_count' => count($not_matched),
            'skipped_empty_count' => count($skipped_empty),
            'duplicate_count' => count($duplicate_urls),
            'total_rows' => count($parsed['rows']),
            'seo_plugin' => $seo_plugin,
        ];

        // Store bulk update data for later use
        $this->save_bulk_data($results);

        return $results;
    }

    /**
     * Build a map of WordPress URLs to post IDs
     *
     * @return array URL map
     */
    private function build_wp_url_map() {
        global $wpdb;

        $map = [];

        // Get all published posts and pages
        $posts = $wpdb->get_results(
            "SELECT ID, post_name, post_type FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ('post', 'page')
             LIMIT 50000",
            ARRAY_A
        );

        foreach ($posts as $post) {
            $permalink = get_permalink($post['ID']);
            if ($permalink) {
                $normalized = $this->normalize_url($permalink);
                $map[$normalized] = [
                    'post_id' => (int) $post['ID'],
                    'post_type' => $post['post_type'],
                ];
            }
        }

        return $map;
    }

    /**
     * Apply bulk updates in a batch
     *
     * @param array $updates Array of updates to apply
     * @param int $offset Starting offset
     * @param int $batch_size Batch size
     * @return array Results
     */
    public function apply_bulk_updates($updates, $offset = 0, $batch_size = 50) {
        $results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => [],
            'details' => [],
            'complete' => false,
            'next_offset' => $offset + $batch_size,
        ];

        $batch = array_slice($updates, $offset, $batch_size);

        if (empty($batch)) {
            $results['complete'] = true;
            return $results;
        }

        foreach ($batch as $update) {
            $results['processed']++;

            $post_id = isset($update['post_id']) ? (int) $update['post_id'] : 0;
            $new_description = isset($update['new_description']) ? trim($update['new_description']) : '';
            $url = isset($update['url']) ? $update['url'] : '';

            if (!$post_id || empty($new_description)) {
                $results['failed']++;
                $results['errors'][] = [
                    'url' => $url,
                    'error' => __('Invalid post ID or empty description', 'screaming-fixes'),
                ];
                continue;
            }

            $update_result = $this->update_meta_description($post_id, $new_description);

            if (is_wp_error($update_result)) {
                $results['failed']++;
                $results['errors'][] = [
                    'url' => $url,
                    'post_id' => $post_id,
                    'error' => $update_result->get_error_message(),
                ];
            } else {
                $results['success']++;
                $results['details'][] = [
                    'url' => $url,
                    'post_id' => $post_id,
                    'original' => $update_result['original'],
                    'new' => $new_description,
                    'plugin' => $update_result['plugin'],
                ];
            }
        }

        // Check if we've processed all updates
        if ($offset + $batch_size >= count($updates)) {
            $results['complete'] = true;
        }

        return $results;
    }

    /**
     * Save bulk update data
     *
     * @param array $data Bulk update data
     */
    private function save_bulk_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        // Clear any existing data
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Save new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get stored bulk data
     *
     * @return array|null Bulk data or null
     */
    private function get_bulk_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            $data = json_decode($row->data, true);
            if (isset($data['is_bulk_update']) && $data['is_bulk_update']) {
                return $data;
            }
        }

        return null;
    }

    /**
     * AJAX: Process bulk CSV upload
     */
    public function ajax_process_bulk_csv() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

        if (empty($upload_id)) {
            wp_send_json_error(['message' => __('No upload ID provided.', 'screaming-fixes')]);
        }

        $uploads = get_option('sf_pending_uploads', []);

        if (!isset($uploads[$upload_id])) {
            wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
        }

        $file_path = $uploads[$upload_id]['path'];

        if (!file_exists($file_path)) {
            wp_send_json_error(['message' => __('File not found.', 'screaming-fixes')]);
        }

        $results = $this->process_bulk_csv($file_path);

        if (is_wp_error($results)) {
            wp_send_json_error(['message' => $results->get_error_message()]);
        }

        // Clean up upload reference
        unset($uploads[$upload_id]);
        update_option('sf_pending_uploads', $uploads);

        wp_send_json_success([
            'message' => sprintf(
                __('%d meta descriptions ready to update', 'screaming-fixes'),
                $results['ready_count']
            ),
            'data' => $results,
        ]);
    }

    /**
     * AJAX: Apply bulk updates (handles batching)
     */
    public function ajax_apply_bulk_updates() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $offset = isset($_POST['offset']) ? (int) $_POST['offset'] : 0;
        $batch_size = isset($_POST['batch_size']) ? (int) $_POST['batch_size'] : 50;

        // Clear accumulated transient at the start of a new batch run
        if ($offset === 0) {
            delete_transient('sf_meta_bulk_accumulated_' . get_current_user_id());
        }

        // Get stored bulk data
        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data) || empty($bulk_data['ready_updates'])) {
            wp_send_json_error(['message' => __('No bulk update data found.', 'screaming-fixes')]);
        }

        $updates = $bulk_data['ready_updates'];
        $results = $this->apply_bulk_updates($updates, $offset, $batch_size);

        // If complete, store the results for display
        if ($results['complete']) {
            // Get accumulated results from all batches
            $accumulated = get_transient('sf_meta_bulk_accumulated_' . get_current_user_id());
            if (!$accumulated) {
                $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
            }

            $accumulated['success'] += $results['success'];
            $accumulated['failed'] += $results['failed'];
            $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
            $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

            // Store final results
            $final_results = [
                'is_bulk_update' => true,
                'bulk_complete' => true,
                'fixed_descriptions' => $accumulated['details'],
                'failed_updates' => $accumulated['errors'],
                'success_count' => $accumulated['success'],
                'failed_count' => $accumulated['failed'],
                'not_matched' => $bulk_data['not_matched'] ?? [],
                'skipped_empty' => $bulk_data['skipped_empty'] ?? [],
            ];

            $this->save_bulk_data($final_results);

            // Clean up accumulated transient
            delete_transient('sf_meta_bulk_accumulated_' . get_current_user_id());

            // Log to activity log
            if ($accumulated['success'] > 0) {
                SF_Activity_Log::log('meta-description', $accumulated['success']);
            }

            $results['total_success'] = $accumulated['success'];
            $results['total_failed'] = $accumulated['failed'];
            $results['all_details'] = $accumulated['details'];
            $results['all_errors'] = $accumulated['errors'];
        } else {
            // Accumulate results for multi-batch operations
            $accumulated = get_transient('sf_meta_bulk_accumulated_' . get_current_user_id());
            if (!$accumulated) {
                $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
            }

            $accumulated['success'] += $results['success'];
            $accumulated['failed'] += $results['failed'];
            $accumulated['details'] = array_merge($accumulated['details'], $results['details']);
            $accumulated['errors'] = array_merge($accumulated['errors'], $results['errors']);

            set_transient('sf_meta_bulk_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
        }

        $results['total_count'] = count($updates);

        wp_send_json_success($results);
    }

    /**
     * AJAX: Cancel bulk update (clear data)
     */
    public function ajax_cancel_bulk() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Clear bulk data
        $this->clear_upload_data();

        // Clear accumulated transient
        delete_transient('sf_meta_bulk_accumulated_' . get_current_user_id());

        wp_send_json_success(['message' => __('Bulk update cancelled.', 'screaming-fixes')]);
    }

    /**
     * AJAX: Download preview CSV
     */
    public function ajax_download_preview() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data)) {
            wp_die(__('No bulk data found.', 'screaming-fixes'));
        }

        $csv_content = $this->generate_preview_csv($bulk_data);
        $filename = 'meta-description-preview-' . date('Y-m-d-His') . '.csv';

        // Send CSV headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output BOM for Excel compatibility
        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * AJAX: Download results CSV
     */
    public function ajax_download_results() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data)) {
            wp_die(__('No results data found.', 'screaming-fixes'));
        }

        $csv_content = $this->generate_results_csv($bulk_data);
        $filename = 'meta-description-results-' . date('Y-m-d-His') . '.csv';

        // Send CSV headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output BOM for Excel compatibility
        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * AJAX: Get Fixed section HTML for dynamic refresh
     */
    public function ajax_get_fixed_section() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        $fixed_descriptions = $results['fixed_descriptions'] ?? [];
        // Also merge in any legacy failed_descriptions with failed status
        if (!empty($results['failed_descriptions'])) {
            foreach ($results['failed_descriptions'] as $fd) {
                $fd['status'] = 'failed';
                $fd['status_message'] = $fd['error'] ?? __('Unknown error', 'screaming-fixes');
                $fd['applied_description'] = $fd['attempted_description'] ?? '';
                $fd['fixed_at'] = $fd['failed_at'] ?? '';
                $fixed_descriptions[] = $fd;
            }
        }
        $fixed_count = count($fixed_descriptions);

        // Count by status
        $fixed_success_count = 0;
        $fixed_failed_count = 0;
        $fixed_skipped_count = 0;
        foreach ($fixed_descriptions as $fd) {
            $s = $fd['status'] ?? 'success';
            if ($s === 'failed') { $fixed_failed_count++; }
            elseif ($s === 'skipped') { $fixed_skipped_count++; }
            else { $fixed_success_count++; }
        }

        ob_start();
        ?>
        <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="true">
            <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
            <?php printf(
                esc_html(_n('Fixes Applied (%d)', 'Fixes Applied (%d)', $fixed_count, 'screaming-fixes')),
                $fixed_count
            ); ?>
            <span class="sf-section-hint"><?php esc_html_e('Meta description fix results', 'screaming-fixes'); ?></span>
            <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
        </button>

        <div class="sf-fixed-content">
            <div class="sf-fixed-header-row">
                <div class="sf-fixed-note">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <?php esc_html_e('Results of meta description fix operations.', 'screaming-fixes'); ?>
                </div>
                <div class="sf-fixed-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-fixed-download-csv" id="sf-fixed-download-csv">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-secondary sf-fixed-clear-btn" id="sf-fixed-clear">
                        <span class="dashicons dashicons-trash"></span>
                        <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <div class="sf-status-filter-row sf-md-status-filters">
                <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                <div class="sf-status-filters">
                    <button type="button" class="sf-status-filter active" data-status="all">
                        <?php esc_html_e('All', 'screaming-fixes'); ?> (<span class="sf-md-status-count-all"><?php echo esc_html($fixed_count); ?></span>)
                    </button>
                    <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                        <?php esc_html_e('Fixed', 'screaming-fixes'); ?> (<span class="sf-md-status-count-success"><?php echo esc_html($fixed_success_count); ?></span>)
                    </button>
                    <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                        <?php esc_html_e('Failed', 'screaming-fixes'); ?> (<span class="sf-md-status-count-failed"><?php echo esc_html($fixed_failed_count); ?></span>)
                    </button>
                    <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                        <?php esc_html_e('Skipped', 'screaming-fixes'); ?> (<span class="sf-md-status-count-skipped"><?php echo esc_html($fixed_skipped_count); ?></span>)
                    </button>
                </div>
            </div>

            <div class="sf-table-wrapper sf-paginated-table" data-section="fixed" data-per-page="100" data-total="<?php echo esc_attr($fixed_count); ?>">
                <table class="sf-results-table sf-fixed-table">
                    <thead>
                        <tr>
                            <th class="sf-col-status" style="width: 80px;"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            <th class="sf-col-page"><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                            <th class="sf-col-description"><?php esc_html_e('New Description', 'screaming-fixes'); ?></th>
                            <th class="sf-col-chars" style="width: 80px;"><?php esc_html_e('Chars', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fixed_descriptions as $desc):
                            $address = $desc['address'] ?? '';
                            $post_title = $desc['post_title'] ?? '';
                            $post_id = $desc['post_id'] ?? 0;
                            $applied_description = $desc['applied_description'] ?? '';
                            $fixed_at = $desc['fixed_at'] ?? '';
                            $path = wp_parse_url($address, PHP_URL_PATH) ?: $address;
                            $char_count = strlen($applied_description);
                            $char_class = $char_count >= 70 && $char_count <= 155 ? 'good' : ($char_count > 155 ? 'bad' : 'warning');
                            $edit_url = $post_id ? get_edit_post_link($post_id) : '';
                            $item_status = $desc['status'] ?? 'success';
                            $item_status_message = $desc['status_message'] ?? '';
                            $row_class = 'sf-fixed-row';
                            if ($item_status === 'failed') { $row_class .= ' sf-fixed-row-failed'; }
                            elseif ($item_status === 'skipped') { $row_class .= ' sf-fixed-row-skipped'; }
                        ?>
                            <tr class="<?php echo esc_attr($row_class); ?>" data-address="<?php echo esc_attr($address); ?>" data-status="<?php echo esc_attr($item_status); ?>">
                                <td class="sf-col-status">
                                    <?php if ($item_status === 'failed'): ?>
                                        <span class="sf-status-badge sf-status-failed" title="<?php echo esc_attr($item_status_message); ?>">&#10007; <?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                                    <?php elseif ($item_status === 'skipped'): ?>
                                        <span class="sf-status-badge sf-status-skipped" title="<?php echo esc_attr($item_status_message); ?>">&#8212; <?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                                    <?php else: ?>
                                        <span class="sf-status-badge sf-status-success">&#10003; <?php esc_html_e('Updated', 'screaming-fixes'); ?></span>
                                    <?php endif; ?>
                                </td>
                                <td class="sf-col-page">
                                    <div class="sf-page-cell">
                                        <div class="sf-page-title">
                                            <?php if ($post_id && $edit_url): ?>
                                                <a href="<?php echo esc_url($edit_url); ?>" target="_blank" rel="noopener" class="sf-edit-link">
                                                    <?php echo esc_html($post_title ?: __('(No title)', 'screaming-fixes')); ?>
                                                    <span class="dashicons dashicons-edit"></span>
                                                </a>
                                            <?php else: ?>
                                                <span><?php echo esc_html($post_title ?: __('(No title)', 'screaming-fixes')); ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="sf-page-url">
                                            <a href="<?php echo esc_url($address); ?>" target="_blank" rel="noopener">
                                                <?php echo esc_html($path); ?>
                                                <span class="dashicons dashicons-external"></span>
                                            </a>
                                        </div>
                                    </div>
                                </td>
                                <td class="sf-col-description">
                                    <span class="sf-description-text"><?php echo esc_html($applied_description); ?></span>
                                </td>
                                <td class="sf-col-chars">
                                    <span class="sf-char-count sf-char-count-<?php echo esc_attr($char_class); ?>"><?php echo esc_html($char_count); ?></span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php

        $html = ob_get_clean();

        wp_send_json_success(['html' => $html, 'fixed_count' => $fixed_count]);
    }

    /**
     * AJAX: Download fixed descriptions results CSV (from Fixed section)
     */
    public function ajax_download_fixed_results() {
        check_ajax_referer('sf_meta_description_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        // Get the stored results data (try transient first, then database)
        $results = $this->get_results();
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results) || empty($results['fixed_descriptions'])) {
            wp_die(__('No fixed descriptions found.', 'screaming-fixes'));
        }

        // Merge in legacy failed_descriptions if any
        $fixed_descriptions = $results['fixed_descriptions'];
        if (!empty($results['failed_descriptions'])) {
            foreach ($results['failed_descriptions'] as $fd) {
                $fd['status'] = 'failed';
                $fd['status_message'] = $fd['error'] ?? __('Unknown error', 'screaming-fixes');
                $fd['applied_description'] = $fd['attempted_description'] ?? '';
                $fd['fixed_at'] = $fd['failed_at'] ?? '';
                $fixed_descriptions[] = $fd;
            }
        }

        $lines = [];
        $lines[] = 'URL,Original Description,New Description,Character Count,Status,Status Message';

        foreach ($fixed_descriptions as $row) {
            // Get the new description - could be stored as 'applied_description' or 'new'
            $new_desc = $row['applied_description'] ?? $row['new'] ?? $row['new_description'] ?? '';
            // Get the original description
            $original_desc = $row['original_description'] ?? $row['original'] ?? $row['current_description'] ?? '';
            $status = $row['status'] ?? 'success';
            $status_label = $status === 'failed' ? 'Failed' : ($status === 'skipped' ? 'Skipped' : 'Updated');

            $lines[] = $this->csv_escape_row([
                $row['address'] ?? $row['url'] ?? '',
                $original_desc,
                $new_desc,
                strlen($new_desc),
                $status_label,
                $row['status_message'] ?? '',
            ]);
        }

        $csv_content = implode("\n", $lines);
        $filename = 'fixed-meta-descriptions-' . date('Y-m-d-His') . '.csv';

        // Send CSV headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output BOM for Excel compatibility
        echo "\xEF\xBB\xBF";
        echo $csv_content;
        exit;
    }

    /**
     * Generate preview CSV data for download
     *
     * @param array $bulk_data Bulk update data
     * @return string CSV content
     */
    public function generate_preview_csv($bulk_data) {
        $lines = [];

        // Header
        $lines[] = 'URL,Current Description,New Description,Character Count,Status';

        // Ready updates
        foreach ($bulk_data['ready_updates'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['url'],
                $row['current_description'],
                $row['new_description'],
                $row['char_count'] ?? strlen($row['new_description']),
                'Ready',
            ]);
        }

        // Not matched
        foreach ($bulk_data['not_matched'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['url'],
                '',
                $row['new_description'],
                $row['char_count'] ?? strlen($row['new_description']),
                'Skipped - URL not found',
            ]);
        }

        // Skipped empty
        foreach ($bulk_data['skipped_empty'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['url'],
                '',
                '',
                0,
                'Skipped - No new meta description',
            ]);
        }

        return implode("\n", $lines);
    }

    /**
     * Generate results CSV data for download
     *
     * @param array $bulk_data Bulk results data
     * @return string CSV content
     */
    public function generate_results_csv($bulk_data) {
        $lines = [];

        // Header
        $lines[] = 'URL,Original Description,New Description,Character Count,Status';

        // Successful updates
        foreach ($bulk_data['fixed_descriptions'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['url'],
                $row['original'] ?? '',
                $row['new'],
                strlen($row['new']),
                'Updated',
            ]);
        }

        // Failed updates
        foreach ($bulk_data['failed_updates'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['url'],
                '',
                '',
                0,
                'Failed - ' . ($row['error'] ?? 'Unknown error'),
            ]);
        }

        // Not matched (from original upload)
        foreach ($bulk_data['not_matched'] ?? [] as $row) {
            $lines[] = $this->csv_escape_row([
                $row['url'],
                '',
                $row['new_description'] ?? '',
                strlen($row['new_description'] ?? ''),
                'Skipped - URL not found',
            ]);
        }

        return implode("\n", $lines);
    }

    /**
     * Escape a row for CSV output
     *
     * @param array $fields Fields to escape
     * @return string Escaped CSV row
     */
    private function csv_escape_row($fields) {
        $escaped = [];
        foreach ($fields as $field) {
            // Escape quotes and wrap in quotes if contains comma, quote, or newline
            if (strpos($field, ',') !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false) {
                $field = '"' . str_replace('"', '""', $field) . '"';
            }
            $escaped[] = $field;
        }
        return implode(',', $escaped);
    }
}
