/**
 * Screaming Fixes - Page Title Module
 *
 * Handles UI interactions, AI suggestions, filtering, and applying page title fixes
 */

(function($) {
    'use strict';

    // Namespace
    window.ScreamingFixes = window.ScreamingFixes || {};

    /**
     * Page Title Handler
     */
    ScreamingFixes.PageTitle = {
        // Store current data
        data: null,
        selectedRows: [],
        currentFilter: 'all',
        bulkData: null,
        bulkProcessing: false,

        // Pagination state for each section
        pagination: {
            fixable: { currentPage: 1, perPage: 100, totalItems: 0 },
            manual: { currentPage: 1, perPage: 100, totalItems: 0 },
            skipped: { currentPage: 1, perPage: 100, totalItems: 0 },
            fixed: { currentPage: 1, perPage: 100, totalItems: 0 }
        },

        // Fun loading messages for CSV processing
        csvLoadingMessages: [
            "📄 Parsing CSV file...",
            "🔍 Analyzing page titles...",
            "📊 Evaluating title lengths...",
            "🏷️ Reading title tags...",
            "☕ Getting coffee...",
            "🚽 Quick bathroom break...",
            "🧮 Counting characters...",
            "🔗 Matching URLs to pages...",
            "📝 Processing title data...",
            "⚡ Working at lightning speed...",
            "🎪 Juggling title tags...",
            "🔬 Inspecting for duplicates...",
            "📏 Measuring optimal lengths...",
            "🧹 Tidying up the data...",
            "🎨 Painting the results...",
            "🚀 Almost there...",
            "🤔 Thinking really hard...",
            "🎯 Targeting improvements...",
            "✨ Polishing the final details..."
        ],

        /**
         * Initialize the module
         */
        init: function() {
            this.bindEvents();
            this.loadExistingData();
            this.initCharCounters();
            this.initPagination();
        },

        /**
         * Bind all event handlers
         */
        bindEvents: function() {
            var self = this;

            // Select all checkbox
            $(document).on('change', '#sf-select-all-titles', function() {
                var checked = $(this).prop('checked');
                $('.sf-title-row:visible .sf-row-select').prop('checked', checked);
                self.updateSelectedCount();
            });

            // Individual row selection
            $(document).on('change', '.sf-page-title-module .sf-row-select', function() {
                self.updateSelectedCount();
            });

            // New title input change
            $(document).on('input', '.sf-new-title-input', function() {
                self.updateCharCounter($(this));
                self.updateApplyButtonState();
                self.updateSaveButtonState($(this));
            });

            // AI suggest single title
            $(document).on('click', '.sf-page-title-module .sf-ai-suggest-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-title-row');
                self.getAISuggestion(row, btn);
            });

            // Save single title
            $(document).on('click', '.sf-save-title-btn', function(e) {
                e.preventDefault();
                var btn = $(this);
                var row = btn.closest('.sf-title-row');
                self.saveSingleTitle(row, btn);
            });

            // Apply fixes
            $(document).on('click', '.sf-page-title-module .sf-apply-fixes', function(e) {
                e.preventDefault();
                self.applyFixes();
            });

            // Clickable issue stats (filter buttons)
            $(document).on('click', '.sf-page-title-module .sf-issue-stat', function() {
                var filter = $(this).data('filter');
                self.filterByIssue(filter);
            });

            // Duplicate URLs dropdown expand/collapse
            $(document).on('click', '.sf-page-title-module .sf-expand-duplicates', function(e) {
                e.stopPropagation();
                var $btn = $(this);
                var $dropdown = $btn.siblings('.sf-duplicates-dropdown');
                var isActive = $btn.hasClass('active');

                // Close all other dropdowns first
                $('.sf-page-title-module .sf-expand-duplicates').not($btn).removeClass('active');
                $('.sf-page-title-module .sf-duplicates-dropdown').not($dropdown).hide();

                // Toggle this dropdown
                if (isActive) {
                    $btn.removeClass('active');
                    $dropdown.hide();
                } else {
                    $btn.addClass('active');
                    $dropdown.show();
                }
            });

            // Close duplicate dropdowns when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.sf-duplicate-cell').length) {
                    $('.sf-page-title-module .sf-expand-duplicates').removeClass('active');
                    $('.sf-page-title-module .sf-duplicates-dropdown').hide();
                }
            });

            // Export Duplicates CSV button
            $(document).on('click', '.sf-page-title-module .sf-download-duplicates-btn', function(e) {
                e.preventDefault();
                self.downloadDuplicatesCsv();
            });

            // Section toggles (Fixable, Manual, Skipped, Fixed)
            $(document).on('click', '.sf-page-title-module .sf-section-toggle', function() {
                var section = $(this).closest('.sf-results-section');
                var content = section.find('.sf-fixable-content, .sf-manual-content, .sf-skipped-content, .sf-fixed-content');
                var expanded = $(this).attr('aria-expanded') === 'true';

                content.slideToggle(200);
                $(this).attr('aria-expanded', !expanded);
            });

            // Pagination: Previous button
            $(document).on('click', '.sf-page-title-module .sf-page-prev:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage - 1);
            });

            // Pagination: Next button
            $(document).on('click', '.sf-page-title-module .sf-page-next:not(:disabled)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                self.goToPage(section, self.pagination[section].currentPage + 1);
            });

            // Pagination: Page number buttons
            $(document).on('click', '.sf-page-title-module .sf-page-number:not(.active):not(.sf-page-ellipsis)', function(e) {
                e.preventDefault();
                var section = $(this).closest('.sf-pagination').data('section');
                var page = parseInt($(this).data('page'), 10);
                self.goToPage(section, page);
            });

            // Instructions toggle
            $(document).on('click', '.sf-page-title-module .sf-instructions-header', function() {
                var $box = $(this).closest('.sf-instructions-box');
                var $content = $box.find('.sf-instructions-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });

            // Close modal
            $(document).on('click', '.sf-page-title-module .sf-modal-close, .sf-page-title-module .sf-modal', function(e) {
                if (e.target === this) {
                    $(this).closest('.sf-modal').hide();
                }
            });

            // Listen for CSV upload events
            $(document).on('sf:csv-uploaded', function(e, data, box) {
                if (box.data('module') === 'page-title') {
                    self.processUploadedCSV(data);
                }
            });

            // Clear data button
            $(document).on('click', '.sf-page-title-module .sf-clear-data-btn', function(e) {
                e.preventDefault();
                self.clearData();
            });

            // Bulk Upload Handlers
            // Expand/collapse bulk preview title rows
            $(document).on('click', '.sf-desc-expandable', function(e) {
                if ($(e.target).closest('a').length) return;

                var $row = $(this);
                var rowIndex = $row.data('row-index');
                var $detailRow = $('tr.sf-bulk-detail-row[data-for-row="' + rowIndex + '"]');

                if ($row.hasClass('sf-expanded')) {
                    $row.removeClass('sf-expanded');
                    $detailRow.slideUp(200);
                } else {
                    $row.addClass('sf-expanded');
                    $detailRow.slideDown(200);
                }
            });

            // Confirm bulk update
            $(document).on('click', '.sf-bulk-confirm', function(e) {
                e.preventDefault();
                self.processBulkUpdate();
            });

            // Clear bulk update (replaces cancel)
            $(document).on('click', '.sf-bulk-clear-btn', function(e) {
                e.preventDefault();
                self.clearBulkData();
            });

            // Download preview CSV
            $(document).on('click', '.sf-bulk-download-preview', function(e) {
                e.preventDefault();
                self.downloadBulkPreview();
            });

            // Download results CSV
            $(document).on('click', '.sf-bulk-download-results', function(e) {
                e.preventDefault();
                self.downloadBulkResults();
            });

            // New upload button (after bulk complete)
            $(document).on('click', '.sf-bulk-new-upload', function(e) {
                e.preventDefault();
                self.startNewUpload();
            });

            // Fixed section - Download results CSV
            $(document).on('click', '.sf-fixed-download-results', function(e) {
                e.preventDefault();
                self.downloadFixedResults();
            });

            // Fixed section - Clear button
            $(document).on('click', '.sf-fixed-clear-btn', function(e) {
                e.preventDefault();
                self.clearData();
            });

            // Download CSV button (in Actions row)
            $(document).on('click', '.sf-page-title-module .sf-download-csv-btn', function(e) {
                e.preventDefault();
                self.downloadAllCSV();
            });
        },

        /**
         * Initialize character counters for all inputs
         */
        initCharCounters: function() {
            var self = this;
            $('.sf-new-title-input').each(function() {
                self.updateCharCounter($(this));
            });
        },

        /**
         * Update character counter for an input
         * Page title thresholds: 30-60 optimal, 20-29 short warning, 61-70 getting long, <20 or >70 too short/long
         */
        updateCharCounter: function($input) {
            var length = $input.val().length;
            var $row = $input.closest('.sf-title-row');
            var $counter = $row.find('.sf-char-counter');
            var $current = $counter.find('.sf-char-current');
            var $status = $counter.find('.sf-char-status');

            $current.text(length);

            // Remove all status classes
            $counter.removeClass('sf-char-optimal sf-char-warning sf-char-bad');

            if (length === 0) {
                $status.text('');
            } else if (length >= 30 && length <= 60) {
                // Optimal range
                $counter.addClass('sf-char-optimal');
                $status.text(sfPageTitleData.i18n.charCountOptimal);
            } else if ((length >= 20 && length < 30) || (length > 60 && length <= 70)) {
                // Warning range
                $counter.addClass('sf-char-warning');
                $status.text(sfPageTitleData.i18n.charCountWarning);
            } else if (length < 20) {
                // Too short
                $counter.addClass('sf-char-bad');
                $status.text(sfPageTitleData.i18n.charCountTooShort);
            } else {
                // length > 70 - Too long
                $counter.addClass('sf-char-bad');
                $status.text(sfPageTitleData.i18n.charCountTooLong);
            }
        },

        /**
         * Load existing data if available
         */
        loadExistingData: function() {
            var self = this;

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_get_data',
                    nonce: sfPageTitleData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.data) {
                        self.data = response.data.data;
                    }
                }
            });
        },

        /**
         * Process uploaded CSV
         */
        processUploadedCSV: function(uploadData) {
            var self = this;

            // Show loading container with rotating messages
            var loadingState = self.showCSVLoadingContainer();

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_process_csv',
                    nonce: sfPageTitleData.nonce,
                    upload_id: uploadData.upload_id
                },
                success: function(response) {
                    // Clean up loading state
                    self.hideCSVLoadingContainer(loadingState);

                    if (response.success) {
                        self.data = response.data.data;

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message);
                        }

                        // Reload page to show results
                        location.reload();
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message);
                        }
                    }
                },
                error: function() {
                    // Clean up loading state
                    self.hideCSVLoadingContainer(loadingState);

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to process CSV');
                    }
                }
            });
        },

        /**
         * Show animated loading container for CSV processing
         */
        showCSVLoadingContainer: function() {
            var self = this;

            // Remove any existing loading container
            $('.sf-csv-loading-overlay').remove();

            // Create loading overlay
            var $overlay = $('<div class="sf-csv-loading-overlay">' +
                '<div class="sf-csv-loading-container">' +
                    '<div class="sf-csv-loading-spinner"></div>' +
                    '<div class="sf-csv-loading-progress">' +
                        '<div class="sf-csv-progress-bar"><div class="sf-csv-progress-fill"></div></div>' +
                    '</div>' +
                    '<div class="sf-csv-loading-message">' + self.csvLoadingMessages[0] + '</div>' +
                    '<div class="sf-csv-loading-subtext">Please wait while we analyze your data...</div>' +
                '</div>' +
            '</div>');

            // Insert into the module or body
            var $module = $('.sf-page-title-module');
            if ($module.length) {
                $module.append($overlay);
            } else {
                $('body').append($overlay);
            }

            // Animate progress bar
            var $progressFill = $overlay.find('.sf-csv-progress-fill');
            $progressFill.css('width', '0%');

            // Simulate progress (0-90% over ~15 seconds, slower as it goes)
            var progress = 0;
            var progressInterval = setInterval(function() {
                if (progress < 90) {
                    // Progress slows down as it gets higher
                    var increment = Math.max(0.5, (90 - progress) / 30);
                    progress = Math.min(90, progress + increment);
                    $progressFill.css('width', progress + '%');
                }
            }, 200);

            // Start message rotation (every 2.5 seconds)
            var messageIndex = 0;
            var messageInterval = setInterval(function() {
                messageIndex = (messageIndex + 1) % self.csvLoadingMessages.length;
                var $message = $overlay.find('.sf-csv-loading-message');
                $message.addClass('sf-fade-out');
                setTimeout(function() {
                    $message.text(self.csvLoadingMessages[messageIndex]).removeClass('sf-fade-out');
                }, 300);
            }, 2500);

            return {
                overlay: $overlay,
                messageInterval: messageInterval,
                progressInterval: progressInterval
            };
        },

        /**
         * Hide CSV loading container
         */
        hideCSVLoadingContainer: function(loadingState) {
            if (loadingState) {
                if (loadingState.messageInterval) {
                    clearInterval(loadingState.messageInterval);
                }
                if (loadingState.progressInterval) {
                    clearInterval(loadingState.progressInterval);
                }
                if (loadingState.overlay) {
                    // Complete the progress bar before hiding
                    loadingState.overlay.find('.sf-csv-progress-fill').css('width', '100%');
                    loadingState.overlay.find('.sf-csv-loading-message').text('✅ Processing complete!');

                    setTimeout(function() {
                        loadingState.overlay.fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, 500);
                }
            }
        },

        /**
         * Filter rows by issue type or status
         */
        filterByIssue: function(issueType) {
            this.currentFilter = issueType;

            // Get all rows from fixable and manual sections
            var $fixableRows = $('.sf-page-title-module .sf-section-fixable .sf-title-row');
            var $manualRows = $('.sf-page-title-module .sf-section-manual .sf-title-row');

            // Handle status-based filters (fixed, failed) - only apply to fixable section
            if (issueType === 'fixed' || issueType === 'failed') {
                // Hide all manual rows for status filters
                $manualRows.addClass('sf-filtered-out');

                // Filter fixable rows by status
                $fixableRows.each(function() {
                    var $row = $(this);
                    var rowStatus = $row.attr('data-status') || 'pending';

                    if (issueType === 'fixed') {
                        // Show rows with success or fixed status
                        if (rowStatus === 'success' || rowStatus === 'fixed') {
                            $row.removeClass('sf-filtered-out');
                        } else {
                            $row.addClass('sf-filtered-out');
                        }
                    } else if (issueType === 'failed') {
                        // Show rows with failed status
                        if (rowStatus === 'failed') {
                            $row.removeClass('sf-filtered-out');
                        } else {
                            $row.addClass('sf-filtered-out');
                        }
                    }
                });
            } else if (issueType === 'all') {
                // Show all rows
                $fixableRows.removeClass('sf-filtered-out');
                $manualRows.removeClass('sf-filtered-out');
            } else {
                // Filter by issue type (missing, duplicate, too_long, too_short, multiple)
                $fixableRows.each(function() {
                    var rowIssue = $(this).data('issue-type');
                    if (rowIssue === issueType) {
                        $(this).removeClass('sf-filtered-out');
                    } else {
                        $(this).addClass('sf-filtered-out');
                    }
                });

                $manualRows.each(function() {
                    var rowIssue = $(this).data('issue-type');
                    if (rowIssue === issueType) {
                        $(this).removeClass('sf-filtered-out');
                    } else {
                        $(this).addClass('sf-filtered-out');
                    }
                });
            }

            // Reset pagination to page 1 when filtering changes
            this.resetPagination('fixable');
            this.resetPagination('manual');

            // Update section visibility based on filtered rows
            this.updateSectionVisibility();

            // Update select all to only affect visible rows
            this.updateSelectedCount();

            // Update issue stat active state
            $('.sf-page-title-module .sf-issue-stat').removeClass('active');
            $('.sf-page-title-module .sf-issue-stat[data-filter="' + issueType + '"]').addClass('active');

            // Show/hide Export Duplicates CSV button based on filter
            if (issueType === 'duplicate') {
                $('.sf-page-title-module .sf-download-duplicates-btn').show();
            } else {
                $('.sf-page-title-module .sf-download-duplicates-btn').hide();
            }

            // Auto-expand and scroll to Fixed Page Titles section when Fixed or Can't Fix filter is selected
            if (issueType === 'fixed' || issueType === 'failed') {
                var $fixedSection = $('.sf-page-title-module .sf-section-fixed');
                var $fixedToggle = $fixedSection.find('.sf-fixed-toggle');
                var $fixedContent = $fixedSection.find('.sf-fixed-content');
                if ($fixedContent.length && $fixedToggle.attr('aria-expanded') !== 'true') {
                    $fixedContent.slideDown(200);
                    $fixedToggle.attr('aria-expanded', 'true');
                    $fixedToggle.find('.sf-toggle-icon').addClass('sf-rotated');
                }
                if ($fixedSection.length) {
                    setTimeout(function() {
                        $('html, body').animate({
                            scrollTop: $fixedSection.offset().top - 50
                        }, 400);
                    }, 250);
                }
            }
        },


        /**
         * Update row status after fix attempt
         */
        updateRowStatus: function(address, status, reason, suggestion) {
            var $row = $('.sf-page-title-module .sf-title-row[data-address="' + address + '"]');

            if ($row.length === 0) {
                return;
            }

            // Update data attribute
            $row.attr('data-status', status);

            // Update row styling
            $row.removeClass('sf-row-fixed sf-row-failed sf-row-pending');
            if (status === 'success' || status === 'fixed') {
                $row.addClass('sf-row-fixed');
            } else if (status === 'failed') {
                $row.addClass('sf-row-failed');
            }

            // Show status indicator
            var $statusContainer = $row.find('.sf-fix-status');
            $statusContainer.show();

            if (status === 'success' || status === 'fixed') {
                $statusContainer.find('.sf-fix-status-success').show();
                $statusContainer.find('.sf-fix-status-failed').hide();
            } else if (status === 'failed') {
                $statusContainer.find('.sf-fix-status-success').hide();
                var $failedStatus = $statusContainer.find('.sf-fix-status-failed');
                $failedStatus.show();

                if (reason) {
                    $failedStatus.find('.sf-fix-status-reason').text(reason).show();
                }
                if (suggestion) {
                    $failedStatus.find('.sf-fix-status-suggestion').text(suggestion).show();
                }
            }
        },

        /**
         * Update status filter counts in the top Filter By row
         */
        updateStatusCounts: function() {
            var $rows = $('.sf-page-title-module .sf-section-fixable .sf-title-row');
            var counts = {
                fixed: 0,
                failed: 0
            };

            $rows.each(function() {
                var status = $(this).attr('data-status') || 'pending';
                if (status === 'success' || status === 'fixed') {
                    counts.fixed++;
                } else if (status === 'failed') {
                    counts.failed++;
                }
            });

            // Also count from the Fixes Applied section
            var $fixedRows = $('.sf-page-title-module .sf-section-fixed .sf-fixed-row');
            counts.fixed += $fixedRows.filter('[data-status="success"]').length;
            counts.failed += $fixedRows.filter('[data-status="failed"]').length;

            // Update the top Filter By row counts
            var $fixedFilter = $('.sf-page-title-module .sf-issue-fixed');
            var $failedFilter = $('.sf-page-title-module .sf-issue-failed');

            $fixedFilter.find('.sf-fixed-filter-count').text(counts.fixed);
            $failedFilter.find('.sf-failed-filter-count').text(counts.failed);

            // Enable/disable filters based on count
            if (counts.fixed > 0) {
                $fixedFilter.removeClass('sf-filter-disabled');
            } else {
                $fixedFilter.addClass('sf-filter-disabled');
            }

            if (counts.failed > 0) {
                $failedFilter.removeClass('sf-filter-disabled');
            } else {
                $failedFilter.addClass('sf-filter-disabled');
            }
        },

        /**
         * Update the Fixable section header count after rows are removed
         */
        updateFixableSectionCount: function() {
            var $section = $('.sf-page-title-module .sf-section-fixable');
            var $toggle = $section.find('.sf-fixable-toggle');
            var remainingCount = $section.find('.sf-title-row').length;

            // Update the toggle button text
            var toggleText = $toggle.contents().filter(function() {
                return this.nodeType === 3; // Text nodes only
            });

            // Find and update the count in the toggle text
            if (toggleText.length) {
                var currentText = $toggle.text();
                var newText = currentText.replace(/\(\d+\)/, '(' + remainingCount + ')');

                // Rebuild the toggle content
                var badgeHtml = $toggle.find('.sf-section-badge').prop('outerHTML');
                var hintHtml = $toggle.find('.sf-section-hint').prop('outerHTML');
                var iconHtml = $toggle.find('.sf-toggle-icon').prop('outerHTML');

                // Extract the label text between badge and hint
                var labelMatch = newText.match(/Fixable Page Titles \(\d+\)/);
                var labelText = labelMatch ? labelMatch[0] : 'Fixable Page Titles (' + remainingCount + ')';

                $toggle.html(badgeHtml + ' ' + labelText + ' ' + hintHtml + ' ' + iconHtml);
            }

            // Hide the section if no rows remain
            if (remainingCount === 0) {
                $section.slideUp(300);
            } else {
                // Ensure section and batch actions are visible when rows remain
                $section.show();
                $section.find('.sf-batch-actions').show();
            }
        },

        /**
         * Refresh the Fixed Page Titles section via AJAX
         * Called after successfully applying fixes
         */
        refreshFixedSection: function() {
            var self = this;
            var $fixedSection = $('.sf-page-title-module .sf-section-fixed');

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_get_fixed_section',
                    nonce: sfPageTitleData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.html) {
                        // Check if Fixed section exists
                        if ($fixedSection.length) {
                            // Replace the content
                            $fixedSection.html(response.data.html);
                        } else {
                            // Create the Fixed section if it doesn't exist
                            var $newSection = $('<div class="sf-results-section sf-section-fixed" id="sf-fixed-results"></div>');
                            $newSection.html(response.data.html);

                            // Insert before the progress modal or at the end of results
                            var $progressModal = $('.sf-page-title-module .sf-progress-modal');
                            if ($progressModal.length) {
                                $newSection.insertBefore($progressModal);
                            } else {
                                $('.sf-page-title-module').append($newSection);
                            }
                            $fixedSection = $newSection;
                        }

                        // Ensure the section is visible
                        $fixedSection.show();

                        // The toggle should already be expanded (aria-expanded="true" in the HTML)
                        // Make sure the content is visible
                        var $content = $fixedSection.find('.sf-fixed-content');
                        $content.show();

                        // Re-bind event handlers for the new content
                        self.bindFixedSectionEvents();

                        // Update main Filter By counts now that fixed section is loaded
                        self.updateStatusCounts();
                    }
                }
            });
        },

        /**
         * Bind event handlers for Fixed section elements
         * Called after the section is refreshed via AJAX
         */
        bindFixedSectionEvents: function() {
            var self = this;

            // Fixed section toggle
            $('.sf-page-title-module .sf-section-fixed .sf-fixed-toggle').off('click').on('click', function() {
                var $btn = $(this);
                var $content = $btn.closest('.sf-section-fixed').find('.sf-fixed-content');
                var isExpanded = $btn.attr('aria-expanded') === 'true';

                if (isExpanded) {
                    $content.slideUp(300);
                    $btn.attr('aria-expanded', 'false');
                    $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
                } else {
                    $content.slideDown(300);
                    $btn.attr('aria-expanded', 'true');
                    $btn.find('.sf-toggle-icon').addClass('sf-rotated');
                }
            });

            // Download Results button
            $('.sf-page-title-module #sf-fixed-download-results').off('click').on('click', function() {
                self.downloadFixedResults();
            });

            // Clear Fixed button
            $('.sf-page-title-module #sf-fixed-clear').off('click').on('click', function() {
                self.clearFixedResults();
            });

            // Status filter buttons
            $('.sf-page-title-module .sf-pt-status-filters .sf-status-filter').off('click').on('click', function() {
                var $btn = $(this);
                var status = $btn.data('status');

                // Update active state
                $('.sf-page-title-module .sf-pt-status-filters .sf-status-filter').removeClass('active');
                $btn.addClass('active');

                // Filter rows
                self.filterFixResults(status);
            });

            // Update status counts
            self.updateFixStatusCounts();
        },

        /**
         * Filter fixed section rows by status
         */
        filterFixResults: function(status) {
            var $section = $('.sf-page-title-module .sf-section-fixed');
            var $rows = $section.find('.sf-fixed-table tbody .sf-fixed-row');

            if (status === 'all') {
                $rows.removeClass('sf-filtered-out');
            } else {
                $rows.addClass('sf-filtered-out');
                $rows.filter('[data-status="' + status + '"]').removeClass('sf-filtered-out');
            }
        },

        /**
         * Update status counts in filter buttons
         */
        updateFixStatusCounts: function() {
            var $section = $('.sf-page-title-module .sf-section-fixed');
            var $rows = $section.find('.sf-fixed-table tbody .sf-fixed-row');
            var allCount = $rows.length;
            var successCount = $rows.filter('[data-status="success"]').length;
            var failedCount = $rows.filter('[data-status="failed"]').length;
            var skippedCount = $rows.filter('[data-status="skipped"]').length;

            $section.find('.sf-pt-status-count-all').text(allCount);
            $section.find('.sf-pt-status-count-success').text(successCount);
            $section.find('.sf-pt-status-count-failed').text(failedCount);
            $section.find('.sf-pt-status-count-skipped').text(skippedCount);
        },

        /**
         * Update section visibility based on filtered content
         */
        updateSectionVisibility: function() {
            // Check fixable section
            var $fixableSection = $('.sf-page-title-module .sf-section-fixable');
            var visibleFixable = $fixableSection.find('.sf-title-row:not(.sf-filtered-out)').length;
            if (visibleFixable === 0 && this.currentFilter !== 'all') {
                $fixableSection.addClass('sf-section-hidden');
            } else {
                $fixableSection.removeClass('sf-section-hidden');
            }

            // Check manual section - auto-expand if it has visible rows and filter is active
            var $manualSection = $('.sf-page-title-module .sf-section-manual');
            var visibleManual = $manualSection.find('.sf-title-row:not(.sf-filtered-out)').length;
            if (visibleManual === 0 && this.currentFilter !== 'all') {
                $manualSection.addClass('sf-section-hidden');
            } else {
                $manualSection.removeClass('sf-section-hidden');
                // Auto-expand manual section if filter is active and has results
                if (this.currentFilter !== 'all' && visibleManual > 0) {
                    var $toggle = $manualSection.find('.sf-section-toggle');
                    var $content = $manualSection.find('.sf-manual-content');
                    if ($toggle.attr('aria-expanded') !== 'true') {
                        $content.slideDown(200);
                        $toggle.attr('aria-expanded', 'true');
                    }
                }
            }
        },

        /**
         * Update selected count display
         */
        updateSelectedCount: function() {
            var count = $('.sf-page-title-module .sf-title-row:visible .sf-row-select:checked').length;
            $('.sf-page-title-module .sf-selected-count .sf-count').text(count);

            // Update select all checkbox state
            var visibleCheckboxes = $('.sf-page-title-module .sf-title-row:visible .sf-row-select');
            var checkedCheckboxes = visibleCheckboxes.filter(':checked');

            if (visibleCheckboxes.length > 0 && checkedCheckboxes.length === visibleCheckboxes.length) {
                $('#sf-select-all-titles').prop('checked', true).prop('indeterminate', false);
            } else if (checkedCheckboxes.length > 0) {
                $('#sf-select-all-titles').prop('checked', false).prop('indeterminate', true);
            } else {
                $('#sf-select-all-titles').prop('checked', false).prop('indeterminate', false);
            }

            this.updateApplyButtonState();
        },

        /**
         * Update apply button state
         */
        updateApplyButtonState: function() {
            var hasValidFixes = false;

            $('.sf-page-title-module .sf-title-row:visible').each(function() {
                var $row = $(this);
                var $checkbox = $row.find('.sf-row-select');
                var $input = $row.find('.sf-new-title-input');

                if ($checkbox.prop('checked') && $input.val().trim().length > 0) {
                    hasValidFixes = true;
                    return false; // break
                }
            });

            $('.sf-page-title-module .sf-apply-fixes').prop('disabled', !hasValidFixes);
        },

        /**
         * Update save button state based on input content
         */
        updateSaveButtonState: function($input) {
            var $row = $input.closest('.sf-title-row');
            var $saveBtn = $row.find('.sf-save-title-btn');
            var hasContent = $input.val().trim().length > 0;

            // Enable/disable save button based on content
            $saveBtn.prop('disabled', !hasContent);

            // If row was previously saved but content changed, remove saved state
            if ($row.hasClass('sf-row-saved')) {
                $row.removeClass('sf-row-saved');
                $saveBtn.removeClass('sf-saved').text('✓ Save');
            }
        },

        /**
         * Save a single title
         */
        saveSingleTitle: function($row, $btn) {
            var self = this;
            var $input = $row.find('.sf-new-title-input');
            var newTitle = $input.val().trim();

            if (newTitle.length === 0) {
                return;
            }

            var postId = $row.data('post-id');
            var address = $row.data('address');

            // Show saving state
            $btn.addClass('sf-saving');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_apply_fixes',
                    nonce: sfPageTitleData.nonce,
                    fixes: [{
                        post_id: postId,
                        address: address,
                        new_title: newTitle
                    }]
                },
                success: function(response) {
                    if (response.success) {
                        // Process per-URL result
                        var titleResults = response.data.title_results || [];
                        var result = titleResults[0] || {};

                        if (result.status === 'success') {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.success('Page title saved successfully!');
                            }

                            // Remove row from Fixable (it moves to Fixes Applied)
                            $row.fadeOut(300, function() {
                                $(this).remove();
                                self.updateFixableSectionCount();
                                self.updateStatusCounts();
                            });

                            // Refresh the Fixes Applied section
                            self.refreshFixedSection();
                        } else if (result.status === 'failed') {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error(result.reason || 'Failed to save title.');
                            }

                            // Remove row from Fixable (it moves to Fixes Applied with failed status)
                            $row.fadeOut(300, function() {
                                $(this).remove();
                                self.updateFixableSectionCount();
                                self.updateStatusCounts();
                            });

                            // Refresh the Fixes Applied section
                            self.refreshFixedSection();
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Failed to save title.');
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to save title.');
                    }
                },
                complete: function() {
                    $btn.removeClass('sf-saving');
                    // Only re-enable if there's still content
                    $btn.prop('disabled', $input.val().trim().length === 0);
                }
            });
        },

        /**
         * Get AI suggestion for a single title
         */
        getAISuggestion: function($row, $btn) {
            var self = this;
            var address = $row.data('address');
            var postTitle = $row.data('post-title');
            var $input = $row.find('.sf-new-title-input');
            var currentTitle = $row.find('.sf-title-text').text() || '';

            // Show loading state
            $btn.addClass('sf-loading');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_get_ai_suggestion',
                    nonce: sfPageTitleData.nonce,
                    address: address,
                    post_title: postTitle,
                    current_title: currentTitle
                },
                success: function(response) {
                    if (response.success) {
                        $input.val(response.data.suggestion);
                        self.updateCharCounter($input);
                        self.updateApplyButtonState();
                        self.updateSaveButtonState($input);

                        // Auto-select the row
                        $row.find('.sf-row-select').prop('checked', true);
                        self.updateSelectedCount();

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(sfPageTitleData.i18n.aiComplete);
                        }
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfPageTitleData.i18n.aiFailed);
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfPageTitleData.i18n.aiFailed);
                    }
                },
                complete: function() {
                    $btn.removeClass('sf-loading');
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Apply fixes to selected titles
         */
        applyFixes: function() {
            var self = this;
            var fixes = [];

            // Collect fixes from selected rows
            $('.sf-page-title-module .sf-title-row:visible .sf-row-select:checked').each(function() {
                var $row = $(this).closest('.sf-title-row');
                var $input = $row.find('.sf-new-title-input');
                var newTitle = $input.val().trim();

                if (newTitle.length > 0) {
                    fixes.push({
                        post_id: $row.data('post-id'),
                        address: $row.data('address'),
                        new_title: newTitle
                    });
                }
            });

            if (fixes.length === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfPageTitleData.i18n.noFixesSelected);
                }
                return;
            }

            // Show progress modal
            var $modal = $('.sf-page-title-module .sf-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-progress-current');
            var $total = $modal.find('.sf-progress-total');

            $total.text(fixes.length);
            $current.text(0);
            $fill.css('width', '0%');
            $modal.show();

            // Disable apply button
            $('.sf-page-title-module .sf-apply-fixes').prop('disabled', true);

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_apply_fixes',
                    nonce: sfPageTitleData.nonce,
                    fixes: fixes
                },
                success: function(response) {
                    // Animate to 100%
                    $fill.css('width', '100%');
                    $current.text(fixes.length);

                    setTimeout(function() {
                        $modal.hide();

                        if (response.success) {
                            // Process per-URL results
                            var titleResults = response.data.title_results || [];
                            var successCount = 0;
                            var failedCount = 0;
                            var successAddresses = [];

                            var allProcessedAddresses = [];

                            titleResults.forEach(function(result) {
                                var status = result.status;
                                var address = result.address;

                                if (status === 'success') {
                                    successCount++;
                                } else if (status === 'failed') {
                                    failedCount++;
                                }
                                // All processed items (success + failed) move to Fixes Applied
                                allProcessedAddresses.push(address);
                            });

                            // Remove all processed rows from Fixable section (with animation)
                            allProcessedAddresses.forEach(function(address) {
                                var $row = $('.sf-page-title-module .sf-section-fixable .sf-title-row[data-address="' + address + '"]');
                                if ($row.length) {
                                    $row.fadeOut(300, function() {
                                        $(this).remove();
                                        self.updateFixableSectionCount();
                                    });
                                }
                            });

                            // Update status filter counts
                            self.updateStatusCounts();

                            // Uncheck all selected rows
                            $('.sf-page-title-module .sf-title-row .sf-row-select:checked').prop('checked', false);
                            self.updateSelectedCount();

                            // Show appropriate toast message
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                if (failedCount > 0) {
                                    ScreamingFixes.Toast.warning(response.data.message || sfPageTitleData.i18n.fixesApplied);
                                } else {
                                    ScreamingFixes.Toast.success(response.data.message || sfPageTitleData.i18n.fixesApplied);
                                }
                            }

                            // Update the fixed count in the issue filter
                            var $fixedFilter = $('.sf-page-title-module .sf-issue-fixed');
                            var currentFixedCount = parseInt($fixedFilter.find('.sf-fixed-filter-count').text(), 10) || 0;
                            $fixedFilter.find('.sf-fixed-filter-count').text(currentFixedCount + successCount);
                            if (currentFixedCount + successCount > 0) {
                                $fixedFilter.removeClass('sf-filter-disabled');
                            }

                            // Always refresh the Fixes Applied section (shows both success and failed)
                            self.refreshFixedSection();

                            // Ensure batch actions stay visible when there are remaining rows
                            var $fixableSection = $('.sf-page-title-module .sf-section-fixable');
                            var remainingRows = $fixableSection.find('.sf-title-row').length;
                            if (remainingRows > 0) {
                                $fixableSection.find('.sf-batch-actions').show();
                                // Ensure the content section is expanded
                                var $content = $fixableSection.find('.sf-fixable-content');
                                var $toggle = $fixableSection.find('.sf-fixable-toggle');
                                if ($content.is(':hidden')) {
                                    $content.slideDown(200);
                                    $toggle.attr('aria-expanded', 'true');
                                }
                            }
                        } else {
                            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                                ScreamingFixes.Toast.error(response.data.message || sfPageTitleData.i18n.fixesFailed);
                            }
                        }
                    }, 500);
                },
                error: function() {
                    $modal.hide();
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfPageTitleData.i18n.fixesFailed);
                    }
                },
                complete: function() {
                    // Re-enable the apply button (will be disabled again by updateApplyButtonState if no rows selected)
                    $('.sf-page-title-module .sf-apply-fixes').prop('disabled', false);
                    // Ensure batch actions are visible
                    $('.sf-page-title-module .sf-section-fixable .sf-batch-actions').show();
                }
            });
        },

        /**
         * Clear all page title data
         */
        clearData: function() {
            var self = this;

            // Confirm before clearing
            if (!confirm(sfPageTitleData.i18n.confirmClear || 'Are you sure you want to clear all page title data? This will allow you to upload a new CSV file.')) {
                return;
            }

            var $btn = $('.sf-page-title-module .sf-clear-data-btn');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_clear_data',
                    nonce: sfPageTitleData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.success(response.data.message || sfPageTitleData.i18n.dataCleared || 'Data cleared successfully.');
                        }
                        // Reload page to show upload form
                        location.reload();
                    } else {
                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || 'Failed to clear data.');
                        }
                    }
                },
                error: function() {
                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error('Failed to clear data.');
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Process bulk update - applies new titles from uploaded CSV
         */
        processBulkUpdate: function() {
            var self = this;

            if (self.bulkProcessing) {
                return;
            }

            self.bulkProcessing = true;

            // Get bulk data from the page
            var $section = $('.sf-bulk-confirmation');
            // The ready count is in .sf-bulk-stat-ready .sf-bulk-stat-number
            var matchedUrls = parseInt($section.find('.sf-bulk-stat-ready .sf-bulk-stat-number').text(), 10) || 0;
            // Also check the data-count attribute on the confirm button as fallback
            if (matchedUrls === 0) {
                matchedUrls = parseInt($('.sf-bulk-confirm').data('count'), 10) || 0;
            }

            if (matchedUrls === 0) {
                if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                    ScreamingFixes.Toast.warning(sfPageTitleData.i18n.bulkNoMatches || 'No matching URLs found to update.');
                }
                self.bulkProcessing = false;
                return;
            }

            // Show progress modal
            var $modal = $('.sf-bulk-progress-modal');
            var $fill = $modal.find('.sf-progress-fill');
            var $current = $modal.find('.sf-bulk-progress-current');
            var $total = $modal.find('.sf-bulk-progress-total');
            var $percent = $modal.find('.sf-bulk-progress-percent');
            var $currentUrl = $modal.find('.sf-bulk-current-url');

            $total.text(matchedUrls);
            $current.text(0);
            $percent.text('0');
            $fill.css('width', '0%');
            $currentUrl.text('Starting...');
            $modal.show();

            // Disable confirm button
            $('.sf-bulk-confirm').prop('disabled', true);

            // Start processing in batches directly (don't clear data first)
            self.processBulkBatch(0, matchedUrls);
        },

        /**
         * Process a batch of bulk updates
         */
        processBulkBatch: function(offset, total) {
            var self = this;
            var batchSize = 50;

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_apply_bulk_updates',
                    nonce: sfPageTitleData.nonce,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = data.processed || 0;
                        var currentOffset = offset + processed;
                        var percent = Math.min(100, Math.round((currentOffset / total) * 100));

                        // Update progress UI
                        var $modal = $('.sf-bulk-progress-modal');
                        $modal.find('.sf-bulk-progress-current').text(currentOffset);
                        $modal.find('.sf-bulk-progress-percent').text(percent);
                        $modal.find('.sf-progress-fill').css('width', percent + '%');

                        // Update current URL being processed
                        if (data.current_url) {
                            $modal.find('.sf-bulk-current-url').text(data.current_url);
                        }

                        // Check if we need to continue
                        if (data.complete) {
                            // All done - show results
                            // Use accumulated totals (total_success/total_failed/all_details)
                            self.showBulkResults({
                                success: data.total_success || data.success || 0,
                                failed: data.total_failed || data.failed || 0,
                                details: data.all_details || data.details || []
                            });
                        } else {
                            // Continue with next batch
                            self.processBulkBatch(currentOffset, total);
                        }
                    } else {
                        // Error occurred
                        self.bulkProcessing = false;
                        $('.sf-bulk-progress-modal').hide();
                        $('.sf-bulk-confirm').prop('disabled', false);

                        if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                            ScreamingFixes.Toast.error(response.data.message || sfPageTitleData.i18n.bulkFailed || 'Bulk update failed.');
                        }
                    }
                },
                error: function() {
                    self.bulkProcessing = false;
                    $('.sf-bulk-progress-modal').hide();
                    $('.sf-bulk-confirm').prop('disabled', false);

                    if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                        ScreamingFixes.Toast.error(sfPageTitleData.i18n.bulkFailed || 'Bulk update failed.');
                    }
                }
            });
        },

        /**
         * Show bulk update results
         */
        showBulkResults: function(results) {
            var self = this;
            self.bulkProcessing = false;

            // Hide progress modal
            $('.sf-bulk-progress-modal').hide();

            // Reload the page to show the server-rendered results view
            // The bulk complete section already displays success/failure counts
            location.reload();
        },

        /**
         * Clear bulk data and return to upload state
         */
        clearBulkData: function() {
            var self = this;

            // Confirm before clearing
            if (!confirm(sfPageTitleData.i18n.confirmClear || 'Are you sure you want to clear all page title data? This will allow you to upload a new CSV file.')) {
                return;
            }

            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_cancel_bulk',
                    nonce: sfPageTitleData.nonce
                },
                success: function(response) {
                    // Reload page to show upload form
                    location.reload();
                },
                error: function() {
                    // Reload anyway
                    location.reload();
                }
            });
        },

        /**
         * Download bulk preview CSV
         */
        downloadBulkPreview: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfPageTitleData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_page_title_download_preview' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfPageTitleData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Download bulk results CSV
         */
        downloadBulkResults: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfPageTitleData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_page_title_download_results' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfPageTitleData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Download fixed titles results CSV (from Fixed Page Titles section)
         */
        downloadFixedResults: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfPageTitleData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_page_title_download_fixed_results' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfPageTitleData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Download duplicates CSV (all duplicate titles with grouping)
         */
        downloadDuplicatesCsv: function() {
            var form = $('<form>', {
                method: 'POST',
                action: sfPageTitleData.ajaxUrl
            });

            form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_page_title_download_duplicates' }));
            form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfPageTitleData.nonce }));

            form.appendTo('body').submit().remove();
        },

        /**
         * Start new upload after bulk complete
         */
        startNewUpload: function() {
            // Clear data and reload
            $.ajax({
                url: sfPageTitleData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_page_title_clear_data',
                    nonce: sfPageTitleData.nonce
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Initialize pagination for all paginated tables
         */
        initPagination: function() {
            var self = this;

            // Initialize each paginated section
            $('.sf-page-title-module .sf-paginated-table').each(function() {
                var $wrapper = $(this);
                var section = $wrapper.data('section');
                var perPage = parseInt($wrapper.data('per-page'), 10) || 100;
                var total = parseInt($wrapper.data('total'), 10) || 0;

                if (self.pagination[section]) {
                    self.pagination[section].perPage = perPage;
                    self.pagination[section].totalItems = total;
                    self.pagination[section].currentPage = 1;

                    // Initial render - apply pagination to show first page
                    self.applyPagination(section);
                    self.renderPageNumbers(section);
                }
            });
        },

        /**
         * Go to a specific page
         */
        goToPage: function(section, page) {
            var pag = this.pagination[section];
            if (!pag) return;

            var totalPages = Math.ceil(pag.totalItems / pag.perPage);
            page = Math.max(1, Math.min(page, totalPages));

            if (page === pag.currentPage) return;

            pag.currentPage = page;
            this.applyPagination(section);
            this.renderPageNumbers(section);
            this.updatePaginationInfo(section);

            // User stays in same scroll position - no auto-scroll on page change
        },

        /**
         * Apply pagination (show/hide rows based on current page)
         */
        applyPagination: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $wrapper = $('.sf-page-title-module .sf-paginated-table[data-section="' + section + '"]');
            var rowSelector = section === 'fixed' ? 'tbody .sf-fixed-row' : 'tbody .sf-title-row';
            var $rows = $wrapper.find(rowSelector);

            var startIndex = (pag.currentPage - 1) * pag.perPage;
            var endIndex = startIndex + pag.perPage;

            // Get rows that are not filtered out
            var visibleIndex = 0;
            $rows.each(function(i) {
                var $row = $(this);
                var isFilteredOut = $row.hasClass('sf-filtered-out');

                if (isFilteredOut) {
                    // Keep filtered rows hidden
                    $row.addClass('sf-page-hidden');
                } else {
                    // Apply pagination to non-filtered rows
                    if (visibleIndex >= startIndex && visibleIndex < endIndex) {
                        $row.removeClass('sf-page-hidden');
                    } else {
                        $row.addClass('sf-page-hidden');
                    }
                    visibleIndex++;
                }
            });

            // Update button states
            var $pagination = $('.sf-page-title-module .sf-pagination[data-section="' + section + '"]');
            var totalPages = Math.ceil(this.getVisibleRowCount(section) / pag.perPage);

            $pagination.find('.sf-page-prev').prop('disabled', pag.currentPage <= 1);
            $pagination.find('.sf-page-next').prop('disabled', pag.currentPage >= totalPages);
        },

        /**
         * Get count of visible (non-filtered) rows
         */
        getVisibleRowCount: function(section) {
            var $wrapper = $('.sf-page-title-module .sf-paginated-table[data-section="' + section + '"]');
            var rowSelector = section === 'fixed' ? 'tbody .sf-fixed-row:not(.sf-filtered-out)' : 'tbody .sf-title-row:not(.sf-filtered-out)';
            return $wrapper.find(rowSelector).length;
        },

        /**
         * Render page number buttons
         */
        renderPageNumbers: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('.sf-page-title-module .sf-pagination[data-section="' + section + '"]');
            var $container = $pagination.find('.sf-page-numbers');

            var visibleCount = this.getVisibleRowCount(section);
            var totalPages = Math.ceil(visibleCount / pag.perPage);
            var currentPage = pag.currentPage;

            // Don't render if only one page
            if (totalPages <= 1) {
                $container.empty();
                $pagination.find('.sf-page-prev, .sf-page-next').prop('disabled', true);
                return;
            }

            var html = '';
            var maxVisible = 7; // Maximum number of page buttons to show

            if (totalPages <= maxVisible) {
                // Show all pages
                for (var i = 1; i <= totalPages; i++) {
                    html += this.renderPageButton(i, currentPage);
                }
            } else {
                // Show with ellipsis
                // Always show first page
                html += this.renderPageButton(1, currentPage);

                if (currentPage > 3) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }

                // Show pages around current
                var startPage = Math.max(2, currentPage - 1);
                var endPage = Math.min(totalPages - 1, currentPage + 1);

                // Adjust if at edges
                if (currentPage <= 3) {
                    endPage = Math.min(4, totalPages - 1);
                }
                if (currentPage >= totalPages - 2) {
                    startPage = Math.max(2, totalPages - 3);
                }

                for (var j = startPage; j <= endPage; j++) {
                    html += this.renderPageButton(j, currentPage);
                }

                if (currentPage < totalPages - 2) {
                    html += '<span class="sf-page-number sf-page-ellipsis">...</span>';
                }

                // Always show last page
                html += this.renderPageButton(totalPages, currentPage);
            }

            $container.html(html);
        },

        /**
         * Render a single page button
         */
        renderPageButton: function(page, currentPage) {
            var activeClass = page === currentPage ? ' active' : '';
            return '<button type="button" class="sf-page-number' + activeClass + '" data-page="' + page + '">' + page + '</button>';
        },

        /**
         * Update pagination info text
         */
        updatePaginationInfo: function(section) {
            var pag = this.pagination[section];
            if (!pag) return;

            var $pagination = $('.sf-page-title-module .sf-pagination[data-section="' + section + '"]');
            var visibleCount = this.getVisibleRowCount(section);

            var start = Math.min((pag.currentPage - 1) * pag.perPage + 1, visibleCount);
            var end = Math.min(pag.currentPage * pag.perPage, visibleCount);

            $pagination.find('.sf-page-start').text(start);
            $pagination.find('.sf-page-end').text(end);
            $pagination.find('.sf-page-total').text(visibleCount);
        },

        /**
         * Reset pagination to page 1 for a section
         */
        resetPagination: function(section) {
            if (this.pagination[section]) {
                this.pagination[section].currentPage = 1;
                this.applyPagination(section);
                this.renderPageNumbers(section);
                this.updatePaginationInfo(section);
            }
        },

        /**
         * Reset pagination for all sections
         */
        resetAllPagination: function() {
            var self = this;
            ['fixable', 'manual', 'skipped'].forEach(function(section) {
                self.resetPagination(section);
            });
        },

        /**
         * Download all page title data as CSV
         * Includes all rows (fixable + manual) with current status
         */
        downloadAllCSV: function() {
            var self = this;
            var rows = [];

            // CSV headers
            var headers = ['URL', 'Post Title', 'Current Title', 'New Title', 'Issue Type', 'Chars', 'Status', 'Error Reason'];
            rows.push(headers);

            // Collect fixable rows
            $('.sf-page-title-module .sf-section-fixable .sf-title-row').each(function() {
                var $row = $(this);
                var url = $row.data('address') || '';
                var postTitle = $row.data('post-title') || '';
                var currentTitle = $row.find('.sf-title-text').text().trim() || $row.find('.sf-no-title').text().trim() || '';
                var newTitle = $row.find('.sf-new-title-input').val() || '';
                var issueType = $row.data('issue-type') || '';
                var chars = $row.find('.sf-char-count').first().text().replace(/[^\d]/g, '') || '';
                var status = $row.attr('data-status') || 'pending';
                var errorReason = '';

                // Map status to display value
                var displayStatus = 'Pending';
                if (status === 'success' || status === 'fixed') {
                    displayStatus = 'Fixed';
                } else if (status === 'failed') {
                    displayStatus = "Can't Fix";
                    // Get error reason from the status indicator
                    errorReason = $row.find('.sf-fix-status-reason').text().trim() || '';
                }

                // Map issue type to display value
                var displayIssue = self.formatIssueType(issueType);

                rows.push([url, postTitle, currentTitle, newTitle, displayIssue, chars, displayStatus, errorReason]);
            });

            // Collect manual rows
            $('.sf-page-title-module .sf-section-manual .sf-title-row').each(function() {
                var $row = $(this);
                var url = $row.find('.sf-col-page a').attr('href') || '';
                var postTitle = ''; // Manual rows don't have post titles
                var currentTitle = $row.find('.sf-title-text').text().trim() || $row.find('.sf-no-title').text().trim() || '';
                var newTitle = ''; // Manual rows don't have input
                var issueType = $row.data('issue-type') || '';
                var chars = $row.find('.sf-char-count').first().text().replace(/[^\d]/g, '') || '';
                var status = 'manual';
                var displayStatus = 'Manual Fix Required';
                var errorReason = $row.find('.sf-howto-note').text().trim() || '';

                // Map issue type to display value
                var displayIssue = self.formatIssueType(issueType);

                rows.push([url, postTitle, currentTitle, newTitle, displayIssue, chars, displayStatus, errorReason]);
            });

            // Generate CSV content
            var csvContent = rows.map(function(row) {
                return row.map(function(cell) {
                    // Escape quotes and wrap in quotes if needed
                    var escaped = String(cell).replace(/"/g, '""');
                    if (escaped.indexOf(',') !== -1 || escaped.indexOf('"') !== -1 || escaped.indexOf('\n') !== -1) {
                        return '"' + escaped + '"';
                    }
                    return escaped;
                }).join(',');
            }).join('\n');

            // Create and trigger download
            var blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
            var link = document.createElement('a');
            var url = URL.createObjectURL(blob);

            var filename = 'page-titles-' + new Date().toISOString().slice(0, 10) + '.csv';

            link.setAttribute('href', url);
            link.setAttribute('download', filename);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);

            if (window.ScreamingFixes && window.ScreamingFixes.Toast) {
                ScreamingFixes.Toast.success('CSV downloaded successfully!');
            }
        },

        /**
         * Format issue type for display
         */
        formatIssueType: function(issueType) {
            var issueMap = {
                'missing': 'Missing',
                'duplicate': 'Duplicate',
                'too_long': 'Over 60 Chars',
                'too_short': 'Below 30 Chars',
                'multiple': 'Multiple'
            };
            return issueMap[issueType] || issueType;
        },

        /**
         * Escape HTML to prevent XSS
         */
        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        // Only init if we're on the page title tab
        if ($('.sf-page-title-module').length > 0) {
            ScreamingFixes.PageTitle.init();
        }
    });

})(jQuery);
