<?php
/**
 * Page Title Tab Content
 *
 * Main view for the Page Title module with categorized sections:
 * - Fixable: Posts/pages that can be auto-fixed
 * - Manual: Homepage and special pages that need manual fixing
 * - Skipped: Non-indexable pages and unsupported post types
 * - Bulk Update: Confirmation and results view for bulk CSV uploads
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get module instance and data
$module = SF_Module_Loader::get_module('page-title');
$results = $module ? $module->get_results() : null;

// Check for upload data if no transient results
if (empty($results) || empty($results['titles'])) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
    $session_id = 'user_' . get_current_user_id();

    $row = $wpdb->get_row($wpdb->prepare(
        "SELECT data FROM {$table_name}
         WHERE session_id = %s AND module = %s AND expires_at > NOW()
         ORDER BY created_at DESC LIMIT 1",
        $session_id,
        'page-title'
    ));

    if ($row) {
        $stored_data = json_decode($row->data, true);

        if ($stored_data) {
            // Check if this is raw CSV data vs processed results
            if (isset($stored_data['rows']) && !isset($stored_data['titles'])) {
                // Raw CSV data - need to process it through the module
                if ($module) {
                    $pending_uploads = get_option('sf_pending_uploads', []);

                    foreach (array_reverse($pending_uploads) as $upload_id => $upload_info) {
                        if ($upload_info['module'] === 'page-title' && file_exists($upload_info['path'])) {
                            $processed = $module->process_csv($upload_info['path']);
                            if (!is_wp_error($processed)) {
                                $results = $processed;
                                $wpdb->update(
                                    $table_name,
                                    ['data' => wp_json_encode($results)],
                                    ['session_id' => $session_id, 'module' => 'page-title'],
                                    ['%s'],
                                    ['%s', '%s']
                                );
                            }
                            break;
                        }
                    }
                }
            } else {
                $results = $stored_data;
            }
        }
    }
}

// Check for and store error message if $results is a WP_Error
$error_message = null;
if (is_wp_error($results)) {
    $error_message = $results->get_error_message();
    $results = null;
}

// Check if this is a bulk update result
$is_bulk_update = !empty($results['is_bulk_update']);
$bulk_complete = !empty($results['bulk_complete']);

// Bulk update specific data
$bulk_ready_updates = $is_bulk_update ? ($results['ready_updates'] ?? []) : [];
$bulk_not_matched = $is_bulk_update ? ($results['not_matched'] ?? []) : [];
$bulk_skipped_empty = $is_bulk_update ? ($results['skipped_empty'] ?? []) : [];
$bulk_ready_count = $is_bulk_update ? ($results['ready_count'] ?? count($bulk_ready_updates)) : 0;
$bulk_not_matched_count = $is_bulk_update ? ($results['not_matched_count'] ?? count($bulk_not_matched)) : 0;
$bulk_skipped_empty_count = $is_bulk_update ? ($results['skipped_empty_count'] ?? count($bulk_skipped_empty)) : 0;

// Bulk complete results
$bulk_fixed_titles = $bulk_complete ? ($results['fixed_titles'] ?? []) : [];
$bulk_failed_updates = $bulk_complete ? ($results['failed_updates'] ?? []) : [];
$bulk_success_count = $bulk_complete ? ($results['success_count'] ?? 0) : 0;
$bulk_failed_count = $bulk_complete ? ($results['failed_count'] ?? 0) : 0;

// Check if we have any data (titles OR fixed_titles OR valid bulk update data)
$has_results = !empty($results) && is_array($results);
$has_titles = $has_results && !empty($results['titles']);
$has_fixed_titles = $has_results && !empty($results['fixed_titles']) && !$is_bulk_update;

// For bulk updates, check if we have valid displayable data
$has_valid_bulk_data = false;
if ($is_bulk_update) {
    if (!$bulk_complete && !empty($results['ready_updates']) && count($results['ready_updates']) > 0) {
        $has_valid_bulk_data = true;
    }
    // For bulk complete, require actual results (not just empty arrays)
    $fixed_count_check = isset($results['fixed_titles']) ? count($results['fixed_titles']) : 0;
    $failed_count_check = isset($results['failed_updates']) ? count($results['failed_updates']) : 0;
    if ($bulk_complete && ($fixed_count_check > 0 || $failed_count_check > 0)) {
        $has_valid_bulk_data = true;
    }

    // If bulk update mode but no valid data, clear everything and reset
    if (!$has_valid_bulk_data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();
        $wpdb->delete($table_name, ['session_id' => $session_id, 'module' => 'page-title']);
        delete_transient('sf_page-title_results');
        $results = null;
        $is_bulk_update = false;
        $bulk_complete = false;
        // Recalculate after cleanup
        $has_results = false;
        $has_titles = false;
        $has_fixed_titles = false;
    }
}

// Simple check: do we have data to display?
$has_data = $has_titles || $has_fixed_titles || $has_valid_bulk_data;

// Get categorized titles
$fixable_titles = $has_results ? ($results['fixable_titles'] ?? []) : [];
$manual_titles = $has_results ? ($results['manual_titles'] ?? []) : [];
$skipped_titles = $has_results ? ($results['skipped_titles'] ?? []) : [];
$fixed_titles = $has_results ? ($results['fixed_titles'] ?? []) : [];

// Get issue type arrays
$missing = $has_results ? ($results['missing'] ?? []) : [];
$duplicates = $has_results ? ($results['duplicates'] ?? []) : [];
$too_long = $has_results ? ($results['too_long'] ?? []) : [];
$too_short = $has_results ? ($results['too_short'] ?? []) : [];
$multiple = $has_results ? ($results['multiple'] ?? []) : [];

// Check if CSV had Title 2 column (for Multiple filter visibility)
$has_title_2 = $has_results ? ($results['has_title_2'] ?? false) : false;

// Get counts
$total_count = $has_results ? ($results['total_count'] ?? 0) : 0;
$fixable_count = $has_results ? ($results['fixable_count'] ?? count($fixable_titles)) : 0;
$manual_count = $has_results ? ($results['manual_count'] ?? count($manual_titles)) : 0;
$skipped_count = $has_results ? ($results['skipped_count'] ?? count($skipped_titles)) : 0;
$fixed_count = count($fixed_titles);

// Issue type counts
$missing_count = $has_results ? ($results['missing_count'] ?? count($missing)) : 0;
$duplicate_count = $has_results ? ($results['duplicate_count'] ?? count($duplicates)) : 0;
$too_long_count = $has_results ? ($results['too_long_count'] ?? count($too_long)) : 0;
$too_short_count = $has_results ? ($results['too_short_count'] ?? count($too_short)) : 0;
$multiple_count = $has_results ? ($results['multiple_count'] ?? count($multiple)) : 0;

// SEO Plugin info - check in real-time for accurate detection
$plugin_detector = new SF_Plugin_Detector();
$active_seo_plugins = $plugin_detector->get_active_seo_plugins();

// Determine the active SEO plugin (prioritize rank-math, then yoast, then aioseo)
$seo_plugin = '';
if (isset($active_seo_plugins['rank-math'])) {
    $seo_plugin = 'rank-math';
} elseif (isset($active_seo_plugins['yoast'])) {
    $seo_plugin = 'yoast';
} elseif (isset($active_seo_plugins['aioseo'])) {
    $seo_plugin = 'aioseo';
}

$seo_plugin_available = !empty($seo_plugin);
$seo_plugin_names = [
    'rank-math' => 'Rank Math',
    'yoast' => 'Yoast SEO',
    'aioseo' => 'All in One SEO',
];
$seo_plugin_name = $seo_plugin_names[$seo_plugin] ?? '';
?>

<div class="sf-page-title-module">
    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#127991;</span>
            <h2><?php esc_html_e('Page Title', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Fix missing & duplicate page titles', 'screaming-fixes'); ?></span>
        </div>
        <?php if ($has_data): ?>
            <div class="sf-module-stats">
                <span class="sf-stat sf-stat-total">
                    <strong><?php echo esc_html($total_count); ?></strong>
                    <?php esc_html_e('page title issues', 'screaming-fixes'); ?>
                </span>
                <?php if ($seo_plugin_name): ?>
                    <span class="sf-seo-plugin-badge">
                        <?php printf(esc_html__('Using %s', 'screaming-fixes'), esc_html($seo_plugin_name)); ?>
                    </span>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Instructions Section -->
    <?php include SF_PLUGIN_DIR . 'modules/page-title/views/instructions.php'; ?>

    <?php if ($error_message): ?>
        <!-- Error Message -->
        <div class="sf-notice sf-notice-error" style="margin-bottom: 20px;">
            <p><strong><?php esc_html_e('Error:', 'screaming-fixes'); ?></strong> <?php echo esc_html($error_message); ?></p>
        </div>
    <?php endif; ?>

    <?php if ($has_data && !$seo_plugin_available): ?>
        <!-- SEO Plugin Warning -->
        <div class="sf-notice sf-notice-warning" style="margin-bottom: 20px;">
            <p><strong><?php esc_html_e('SEO Plugin Required:', 'screaming-fixes'); ?></strong>
            <?php esc_html_e('To apply page title fixes, please install and activate one of these SEO plugins: Rank Math, Yoast SEO, or All in One SEO. You can still view your page title issues below.', 'screaming-fixes'); ?></p>
        </div>
    <?php endif; ?>

    <?php if ($is_bulk_update && !$bulk_complete && $has_valid_bulk_data): ?>
        <!-- ==========================================
             BULK UPDATE CONFIRMATION UI
             ========================================== -->
        <div class="sf-bulk-confirmation" id="sf-bulk-confirmation">
            <div class="sf-bulk-summary">
                <div class="sf-bulk-icon">&#128203;</div>
                <h3><?php esc_html_e('Bulk Page Title Update', 'screaming-fixes'); ?></h3>

                <div class="sf-bulk-stats">
                    <div class="sf-bulk-stat sf-bulk-stat-ready">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_ready_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('page titles ready to update', 'screaming-fixes'); ?></span>
                    </div>

                    <?php if ($bulk_not_matched_count > 0): ?>
                    <div class="sf-bulk-stat sf-bulk-stat-warning">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_not_matched_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('URLs not matched', 'screaming-fixes'); ?></span>
                    </div>
                    <?php endif; ?>

                    <?php if ($bulk_skipped_empty_count > 0): ?>
                    <div class="sf-bulk-stat sf-bulk-stat-skipped">
                        <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_skipped_empty_count); ?></span>
                        <span class="sf-bulk-stat-label"><?php esc_html_e('rows skipped - no new title provided', 'screaming-fixes'); ?></span>
                    </div>
                    <?php endif; ?>
                </div>

                <?php if ($bulk_skipped_empty_count > 0): ?>
                <div class="sf-bulk-notice">
                    <span class="dashicons dashicons-info"></span>
                    <?php esc_html_e("Rows with empty 'new_title' values will be skipped.", 'screaming-fixes'); ?>
                </div>
                <?php endif; ?>

                <?php if ($bulk_ready_count > 500): ?>
                <div class="sf-bulk-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <?php printf(
                        esc_html__('Large file detected (%d URLs). For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'),
                        $bulk_ready_count
                    ); ?>
                </div>
                <?php endif; ?>

                <div class="sf-bulk-preview-tip" style="margin-bottom: 14px; padding: 10px 14px; background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; font-size: 13px; line-height: 1.5; color: #334155;">
                    <span class="dashicons dashicons-info-outline" style="color: #22c55e; font-size: 16px; width: 16px; height: 16px; margin-right: 4px; vertical-align: text-bottom;"></span>
                    <?php printf(
                        esc_html__('Use %1$sDownload Preview CSV%2$s to review the full list of Ready (%3$d) and Can\'t Update (%4$d) URLs before confirming.', 'screaming-fixes'),
                        '<strong>', '</strong>',
                        $bulk_ready_count,
                        $bulk_not_matched_count
                    ); ?>
                </div>

                <div class="sf-bulk-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-download-preview" id="sf-bulk-download-preview">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
                    </button>
                    <div class="sf-bulk-buttons">
                        <button type="button" class="sf-button sf-button-secondary sf-bulk-clear-btn" id="sf-bulk-clear">
                            <span class="dashicons dashicons-trash"></span>
                            <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                        </button>
                        <button type="button" class="sf-button sf-button-primary sf-bulk-confirm" id="sf-bulk-confirm" data-count="<?php echo esc_attr($bulk_ready_count); ?>">
                            <?php esc_html_e('Confirm Update', 'screaming-fixes'); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Preview Table -->
            <div class="sf-bulk-preview">
                <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, $bulk_ready_count)); ?> of <?php echo esc_html($bulk_ready_count); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Current Title', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('New Title', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            // Show first 10 ready updates
                            $preview_items = array_slice($bulk_ready_updates, 0, 10);
                            foreach ($preview_items as $index => $item):
                                $path = wp_parse_url($item['url'], PHP_URL_PATH) ?: $item['url'];
                                $current_truncated = strlen($item['current_title']) > 75 ? substr($item['current_title'], 0, 75) . '...' : $item['current_title'];
                                $new_truncated = strlen($item['new_title']) > 75 ? substr($item['new_title'], 0, 75) . '...' : $item['new_title'];
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-ready sf-desc-expandable" data-row-index="<?php echo esc_attr($index); ?>">
                                <td class="sf-col-url">
                                    <a href="<?php echo esc_url($item['url']); ?>" target="_blank" title="<?php echo esc_attr($item['url']); ?>">
                                        <?php echo esc_html($path); ?>
                                    </a>
                                </td>
                                <td class="sf-col-current">
                                    <button type="button" class="sf-desc-toggle" title="<?php esc_attr_e('Click to expand', 'screaming-fixes'); ?>">
                                        <span class="sf-desc-text"><?php echo esc_html($current_truncated ?: '(empty)'); ?></span>
                                        <?php if (strlen($item['current_title']) > 75): ?>
                                            <span class="sf-desc-expand-icon dashicons dashicons-arrow-down-alt2"></span>
                                        <?php endif; ?>
                                    </button>
                                </td>
                                <td class="sf-col-new">
                                    <button type="button" class="sf-desc-toggle" title="<?php esc_attr_e('Click to expand', 'screaming-fixes'); ?>">
                                        <span class="sf-desc-text"><?php echo esc_html($new_truncated); ?></span>
                                        <?php if (strlen($item['new_title']) > 75): ?>
                                            <span class="sf-desc-expand-icon dashicons dashicons-arrow-down-alt2"></span>
                                        <?php endif; ?>
                                    </button>
                                </td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-ready"><?php esc_html_e('Ready', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <!-- Expandable detail row -->
                            <tr class="sf-bulk-detail-row" data-for-row="<?php echo esc_attr($index); ?>" style="display: none;">
                                <td colspan="4">
                                    <div class="sf-desc-detail-panel">
                                        <div class="sf-desc-detail-columns">
                                            <div class="sf-desc-detail-col sf-desc-detail-current">
                                                <div class="sf-desc-detail-label"><?php esc_html_e('Current Title', 'screaming-fixes'); ?></div>
                                                <div class="sf-desc-detail-value"><?php echo esc_html($item['current_title'] ?: '(empty)'); ?></div>
                                            </div>
                                            <div class="sf-desc-detail-arrow">&#8594;</div>
                                            <div class="sf-desc-detail-col sf-desc-detail-new">
                                                <div class="sf-desc-detail-label"><?php esc_html_e('New Title', 'screaming-fixes'); ?></div>
                                                <div class="sf-desc-detail-value"><?php echo esc_html($item['new_title']); ?></div>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <?php if ($bulk_not_matched_count > 0): ?>
                <h4 class="sf-mt-20"><?php esc_html_e('Unmatched / Can\'t Update URLs', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html($bulk_not_matched_count); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table sf-bulk-preview-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('New Title', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $unmatched_preview = array_slice($bulk_not_matched, 0, 5);
                            foreach ($unmatched_preview as $item):
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-skipped">
                                <td class="sf-col-url"><?php echo esc_html($item['url']); ?></td>
                                <td class="sf-col-new"><?php echo esc_html($item['new_title']); ?></td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-skipped"><?php esc_html_e('URL not found', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php if ($bulk_not_matched_count > 5): ?>
                            <tr class="sf-bulk-row-more">
                                <td colspan="3"><?php printf(esc_html__('...and %d more unmatched URLs', 'screaming-fixes'), $bulk_not_matched_count - 5); ?></td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="sf-bulk-unmatched-info" style="margin-top: 12px; padding: 12px 16px; background: #f0fdfa; border: 1px solid #99f6e4; border-radius: 6px; font-size: 13px; line-height: 1.5; color: #334155;">
                    <strong><?php esc_html_e('Why are these URLs unmatched?', 'screaming-fixes'); ?></strong>
                    <ul style="margin: 6px 0 0 18px; padding: 0;">
                        <li><?php esc_html_e('The URL belongs to a category, tag, archive, author, or homepage — these are generated dynamically by WordPress and don\'t have editable page titles in the database.', 'screaming-fixes'); ?></li>
                        <li><?php esc_html_e('The URL was not found on this WordPress site (e.g., it belongs to a different domain or the post has been deleted).', 'screaming-fixes'); ?></li>
                    </ul>
                    <p style="margin: 8px 0 0;"><?php esc_html_e('To update titles for dynamic pages, use your SEO plugin (Yoast, Rank Math, or AIOSEO) title templates, or edit the title directly in each post/page editor.', 'screaming-fixes'); ?></p>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Bulk Progress Modal (hidden by default) -->
        <div class="sf-bulk-progress-modal sf-modal" id="sf-bulk-progress-modal" style="display: none;">
            <div class="sf-modal-content sf-bulk-progress-content">
                <div class="sf-bulk-progress-header">
                    <span class="sf-bulk-progress-icon sf-spinning">&#8634;</span>
                    <h3><?php esc_html_e('Updating Page Titles...', 'screaming-fixes'); ?></h3>
                </div>
                <div class="sf-bulk-progress-bar">
                    <div class="sf-bulk-progress-fill" id="sf-bulk-progress-fill" style="width: 0%;"></div>
                </div>
                <div class="sf-bulk-progress-stats">
                    <span class="sf-bulk-progress-current" id="sf-bulk-progress-current">0</span>
                    <span class="sf-bulk-progress-separator">/</span>
                    <span class="sf-bulk-progress-total" id="sf-bulk-progress-total">0</span>
                    <span class="sf-bulk-progress-label"><?php esc_html_e('complete', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-bulk-progress-url" id="sf-bulk-progress-url">
                    <?php esc_html_e('Preparing...', 'screaming-fixes'); ?>
                </div>
            </div>
        </div>

    <?php elseif ($bulk_complete && $has_valid_bulk_data): ?>
        <!-- ==========================================
             BULK UPDATE COMPLETE RESULTS
             ========================================== -->
        <div class="sf-bulk-complete" id="sf-bulk-complete">
            <div class="sf-bulk-complete-summary <?php echo $bulk_failed_count > 0 ? 'sf-has-failures' : ''; ?>">
                <div class="sf-bulk-complete-icon">
                    <?php echo $bulk_failed_count > 0 ? '&#9888;' : '&#10004;'; ?>
                </div>
                <h3>
                    <?php
                    if ($bulk_failed_count > 0) {
                        printf(
                            esc_html__('%d page titles updated successfully. %d failed - see details below.', 'screaming-fixes'),
                            $bulk_success_count,
                            $bulk_failed_count
                        );
                    } else {
                        printf(
                            esc_html__('%d page titles updated successfully.', 'screaming-fixes'),
                            $bulk_success_count
                        );
                    }
                    ?>
                    <?php if ($bulk_not_matched_count > 0): ?>
                        <span class="sf-bulk-skipped-note">
                            <?php printf(esc_html__('%d URLs skipped.', 'screaming-fixes'), $bulk_not_matched_count + $bulk_skipped_empty_count); ?>
                        </span>
                    <?php endif; ?>
                </h3>

                <div class="sf-bulk-complete-actions">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-download-results" id="sf-bulk-download-results">
                        <span class="dashicons dashicons-download"></span>
                        <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-primary sf-bulk-new-upload" id="sf-bulk-new-upload">
                        <?php esc_html_e('Upload New CSV', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>

            <!-- Results Table (first 10) -->
            <?php if (!empty($bulk_fixed_titles)): ?>
            <div class="sf-bulk-results-table">
                <h4><?php esc_html_e('Updated Titles', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, count($bulk_fixed_titles))); ?> of <?php echo esc_html(count($bulk_fixed_titles)); ?>)</span></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Original Title', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('New Title', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $results_preview = array_slice($bulk_fixed_titles, 0, 10);
                            foreach ($results_preview as $item):
                                $path = wp_parse_url($item['url'], PHP_URL_PATH) ?: $item['url'];
                            ?>
                            <tr class="sf-bulk-row sf-bulk-row-success">
                                <td class="sf-col-url">
                                    <a href="<?php echo esc_url($item['url']); ?>" target="_blank">
                                        <?php echo esc_html($path); ?>
                                    </a>
                                </td>
                                <td class="sf-col-original"><?php echo esc_html($item['original'] ?: '(empty)'); ?></td>
                                <td class="sf-col-new"><?php echo esc_html($item['new']); ?></td>
                                <td class="sf-col-status"><span class="sf-status-badge sf-status-success"><?php esc_html_e('Updated', 'screaming-fixes'); ?></span></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <!-- Failed Updates -->
            <?php if (!empty($bulk_failed_updates)): ?>
            <div class="sf-bulk-failed-table">
                <h4 class="sf-mt-20"><?php esc_html_e('Failed Updates', 'screaming-fixes'); ?></h4>
                <div class="sf-table-wrapper">
                    <table class="sf-results-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('URL', 'screaming-fixes'); ?></th>
                                <th><?php esc_html_e('Error', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($bulk_failed_updates as $item): ?>
                            <tr class="sf-bulk-row sf-bulk-row-failed">
                                <td class="sf-col-url"><?php echo esc_html($item['url']); ?></td>
                                <td class="sf-col-error"><?php echo esc_html($item['error']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
        </div>

    <?php elseif ($has_data && !$is_bulk_update): ?>
        <!-- Stats Breakdown - Horizontal layout like Broken Links -->
        <div class="sf-stats-breakdown">
            <div class="sf-stat-card sf-stat-fixable">
                <span class="sf-stat-icon">&#10003;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($fixable_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Fixable', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-stat-card sf-stat-manual">
                <span class="sf-stat-icon">&#9888;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($manual_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Manual Fix', 'screaming-fixes'); ?></span>
                </div>
            </div>
            <div class="sf-stat-card sf-stat-skipped">
                <span class="sf-stat-icon">&#8856;</span>
                <div class="sf-stat-content">
                    <span class="sf-stat-number"><?php echo esc_html($skipped_count); ?></span>
                    <span class="sf-stat-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                </div>
            </div>
        </div>

        <!-- Issue Type Stats with Filter Label and Clear Button -->
        <div class="sf-issue-stats-row">
            <span class="sf-filter-label"><?php esc_html_e('Filter By:', 'screaming-fixes'); ?></span>
            <div class="sf-issue-stats">
                <div class="sf-issue-stat sf-issue-all active" data-filter="all">
                    <span class="sf-issue-icon">&#128203;</span>
                    <span class="sf-issue-count"><?php echo esc_html($fixable_count + $manual_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-missing" data-filter="missing">
                    <span class="sf-issue-icon">&#10060;</span>
                    <span class="sf-issue-count"><?php echo esc_html($missing_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Missing', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-duplicate" data-filter="duplicate">
                    <span class="sf-issue-icon">&#128257;</span>
                    <span class="sf-issue-count"><?php echo esc_html($duplicate_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Duplicate', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-too-long" data-filter="too_long">
                    <span class="sf-issue-icon">&#10145;</span>
                    <span class="sf-issue-count"><?php echo esc_html($too_long_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Over 60 Chars', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-too-short" data-filter="too_short">
                    <span class="sf-issue-icon">&#10548;</span>
                    <span class="sf-issue-count"><?php echo esc_html($too_short_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Below 30 Chars', 'screaming-fixes'); ?></span>
                </div>
                <?php if ($has_title_2 && $multiple_count > 0): ?>
                <div class="sf-issue-stat sf-issue-multiple" data-filter="multiple">
                    <span class="sf-issue-icon">&#9888;</span>
                    <span class="sf-issue-count"><?php echo esc_html($multiple_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Multiple', 'screaming-fixes'); ?></span>
                </div>
                <?php endif; ?>
                <div class="sf-issue-stat sf-issue-fixed<?php echo $fixed_count === 0 ? ' sf-filter-disabled' : ''; ?>" data-filter="fixed">
                    <span class="sf-issue-icon">&#9989;</span>
                    <span class="sf-issue-count sf-fixed-filter-count"><?php echo esc_html($fixed_count); ?></span>
                    <span class="sf-issue-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-issue-stat sf-issue-failed sf-filter-disabled" data-filter="failed">
                    <span class="sf-issue-icon">&#10060;</span>
                    <span class="sf-issue-count sf-failed-filter-count">0</span>
                    <span class="sf-issue-label"><?php esc_html_e("Can't Fix", 'screaming-fixes'); ?></span>
                </div>
            </div>
        </div>

        <!-- Actions Row -->
        <div class="sf-actions-row">
            <span class="sf-actions-label"><?php esc_html_e('Actions:', 'screaming-fixes'); ?></span>
            <div class="sf-actions-buttons">
                <button type="button" class="sf-action-btn sf-download-csv-btn" title="<?php esc_attr_e('Download all data as CSV', 'screaming-fixes'); ?>">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Download CSV', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-action-btn sf-download-duplicates-btn" title="<?php esc_attr_e('Export all duplicates with grouping', 'screaming-fixes'); ?>" style="display: none;">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export Duplicates CSV', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-action-btn sf-clear-data-btn" title="<?php esc_attr_e('Clear data and upload new CSV', 'screaming-fixes'); ?>">
                    <span class="dashicons dashicons-trash"></span>
                    <?php esc_html_e('Clear Data', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
    <?php endif; ?>

    <?php if (!$has_data): ?>
    <!-- Upload Section -->
    <div class="sf-upload-section">
        <h3>
            <span class="dashicons dashicons-upload"></span>
            <?php esc_html_e('Upload Page Title CSV', 'screaming-fixes'); ?>
        </h3>
        <?php
        $args = [
            'module' => 'page-title',
            'title' => __('Drop your Screaming Frog Page Title CSV file', 'screaming-fixes'),
            'description' => __('Drag & drop CSV here or click to browse', 'screaming-fixes'),
            'accepted_types' => '.csv',
            'max_rows' => 5000,
        ];
        include SF_PLUGIN_DIR . 'admin/views/partials/upload-box.php';
        ?>
    </div>
    <?php endif; ?>

    <?php if ($has_data): ?>

        <?php if (!empty($fixable_titles)): ?>
        <!-- Section 1: Fixable Titles -->
        <div class="sf-results-section sf-section-fixable">
            <button type="button" class="sf-section-toggle sf-fixable-toggle" aria-expanded="true">
                <span class="sf-section-badge sf-badge-fixable">&#10003;</span>
                <?php printf(
                    esc_html__('%d Fixable Page Titles', 'screaming-fixes'),
                    $fixable_count
                ); ?>
                <span class="sf-section-hint"><?php esc_html_e('Posts and pages - can be auto-fixed', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
            </button>

            <div class="sf-fixable-content">
            <!-- Fixable Results Table -->
            <div class="sf-table-wrapper sf-paginated-table" data-section="fixable" data-per-page="100" data-total="<?php echo esc_attr($fixable_count); ?>">
                <!-- Pagination Controls (Top) -->
                <?php if ($fixable_count > 100): ?>
                <div class="sf-pagination" data-section="fixable">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                        <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $fixable_count)); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?>
                        <span class="sf-page-total"><?php echo esc_html($fixable_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                    </div>
                </div>
                <?php endif; ?>
                <table class="sf-results-table" id="sf-page-title-table">
                    <thead>
                        <tr>
                            <th class="sf-col-checkbox">
                                <input type="checkbox" id="sf-select-all-titles" class="sf-select-all" title="<?php esc_attr_e('Select all', 'screaming-fixes'); ?>">
                            </th>
                            <th class="sf-col-page"><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                            <th class="sf-col-current"><?php esc_html_e('Current', 'screaming-fixes'); ?></th>
                            <th class="sf-col-issue"><?php esc_html_e('Issue', 'screaming-fixes'); ?></th>
                            <th class="sf-col-new"><?php esc_html_e('New Page Title', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fixable_titles as $index => $title_item):
                            $address = $title_item['address'] ?? '';
                            $post_id = $title_item['post_id'] ?? 0;
                            $post_title = $title_item['post_title'] ?? '';
                            $edit_url = $title_item['edit_url'] ?? '';
                            $current_title = $title_item['current_title'] ?? '';
                            $title_length = $title_item['title_length'] ?? strlen($current_title);
                            $issue_type = $title_item['issue_type'] ?? '';
                            $duplicate_count_val = $title_item['duplicate_count'] ?? 0;
                            $duplicate_urls = $title_item['duplicate_urls'] ?? [];
                            $path = wp_parse_url($address, PHP_URL_PATH) ?: $address;

                            // Issue badge classes and labels
                            $issue_badges = [
                                'missing' => ['class' => 'sf-badge-missing', 'label' => __('Missing', 'screaming-fixes'), 'icon' => '&#10060;'],
                                'duplicate' => ['class' => 'sf-badge-duplicate', 'label' => __('Duplicate', 'screaming-fixes'), 'icon' => '&#128257;'],
                                'too_long' => ['class' => 'sf-badge-too-long', 'label' => __('Over 60', 'screaming-fixes'), 'icon' => '&#10145;'],
                                'too_short' => ['class' => 'sf-badge-too-short', 'label' => __('Below 30', 'screaming-fixes'), 'icon' => '&#10548;'],
                                'multiple' => ['class' => 'sf-badge-multiple', 'label' => __('Multiple', 'screaming-fixes'), 'icon' => '&#9888;'],
                            ];
                            $badge = $issue_badges[$issue_type] ?? ['class' => '', 'label' => '', 'icon' => ''];
                        ?>
                            <tr class="sf-title-row"
                                data-address="<?php echo esc_attr($address); ?>"
                                data-post-id="<?php echo esc_attr($post_id); ?>"
                                data-post-title="<?php echo esc_attr($post_title); ?>"
                                data-index="<?php echo esc_attr($index); ?>"
                                data-issue-type="<?php echo esc_attr($issue_type); ?>"
                                data-category="fixable"
                                data-status="pending">
                                <td class="sf-col-checkbox">
                                    <input type="checkbox" class="sf-row-select" value="<?php echo esc_attr($index); ?>">
                                </td>
                                <td class="sf-col-page">
                                    <div class="sf-page-cell">
                                        <div class="sf-page-title">
                                            <?php if ($post_id && $edit_url): ?>
                                                <a href="<?php echo esc_url($edit_url); ?>" target="_blank" rel="noopener" class="sf-edit-link">
                                                    <?php echo esc_html($post_title ?: __('(No title)', 'screaming-fixes')); ?>
                                                    <span class="dashicons dashicons-edit"></span>
                                                </a>
                                            <?php else: ?>
                                                <span><?php echo esc_html($post_title ?: __('(No title)', 'screaming-fixes')); ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="sf-page-url">
                                            <a href="<?php echo esc_url($address); ?>" target="_blank" rel="noopener">
                                                <?php echo esc_html($path); ?>
                                            </a>
                                        </div>
                                    </div>
                                </td>
                                <td class="sf-col-current">
                                    <div class="sf-current-title">
                                        <?php if (empty($current_title)): ?>
                                            <span class="sf-no-title"><?php esc_html_e('(No title)', 'screaming-fixes'); ?></span>
                                        <?php else: ?>
                                            <span class="sf-title-text"><?php echo esc_html($current_title); ?></span>
                                        <?php endif; ?>
                                        <span class="sf-char-count sf-char-count-<?php echo $title_length > 60 ? 'bad' : ($title_length < 30 ? 'warning' : 'good'); ?>">
                                            <?php echo esc_html($title_length); ?> <?php esc_html_e('chars', 'screaming-fixes'); ?>
                                        </span>
                                    </div>
                                </td>
                                <td class="sf-col-issue">
                                    <?php if ($issue_type === 'duplicate' && $duplicate_count_val > 0 && !empty($duplicate_urls)): ?>
                                        <div class="sf-duplicate-cell">
                                            <button type="button" class="sf-expand-duplicates sf-issue-badge <?php echo esc_attr($badge['class']); ?>">
                                                <span class="sf-issue-badge-icon"><?php echo $badge['icon']; ?></span>
                                                <?php echo esc_html($badge['label']); ?>
                                                <span class="sf-duplicate-count">(<?php echo esc_html($duplicate_count_val); ?>)</span>
                                                <span class="sf-expand-icon">&#9660;</span>
                                            </button>
                                            <div class="sf-duplicates-dropdown" style="display: none;">
                                                <div class="sf-duplicates-header">
                                                    <strong><?php esc_html_e('Pages with Same Title', 'screaming-fixes'); ?></strong>
                                                    <span class="sf-duplicates-count">
                                                        <?php
                                                        $shown_count = min(count($duplicate_urls), 10);
                                                        printf(
                                                            esc_html__('%d of %d pages', 'screaming-fixes'),
                                                            $shown_count,
                                                            count($duplicate_urls)
                                                        );
                                                        ?>
                                                    </span>
                                                </div>
                                                <table class="sf-duplicates-table">
                                                    <thead>
                                                        <tr>
                                                            <th><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                                            <th><?php esc_html_e('URL', 'screaming-fixes'); ?></th>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach (array_slice($duplicate_urls, 0, 10) as $dup_url): ?>
                                                        <tr>
                                                            <td class="sf-dup-page-cell">
                                                                <?php echo esc_html($dup_url['post_title'] ?: __('(Untitled)', 'screaming-fixes')); ?>
                                                            </td>
                                                            <td class="sf-dup-url-cell">
                                                                <a href="<?php echo esc_url($dup_url['address']); ?>" target="_blank" rel="noopener" title="<?php echo esc_attr($dup_url['address']); ?>">
                                                                    <?php echo esc_html(wp_parse_url($dup_url['address'], PHP_URL_PATH) ?: $dup_url['address']); ?>
                                                                    <span class="sf-external-icon">&#8599;</span>
                                                                </a>
                                                            </td>
                                                        </tr>
                                                        <?php endforeach; ?>
                                                    </tbody>
                                                </table>
                                                <?php if (count($duplicate_urls) > 10): ?>
                                                    <div class="sf-duplicates-footer">
                                                        <?php printf(
                                                            esc_html__('+ %d more pages. Use "Export Duplicates CSV" for full list.', 'screaming-fixes'),
                                                            count($duplicate_urls) - 10
                                                        ); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php else: ?>
                                        <span class="sf-issue-badge <?php echo esc_attr($badge['class']); ?>">
                                            <span class="sf-issue-badge-icon"><?php echo $badge['icon']; ?></span>
                                            <?php echo esc_html($badge['label']); ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td class="sf-col-new">
                                    <div class="sf-new-title-controls">
                                        <div class="sf-title-input-wrapper">
                                            <input type="text" class="sf-new-title-input"
                                                placeholder="<?php esc_attr_e('Enter new page title...', 'screaming-fixes'); ?>"
                                                data-address="<?php echo esc_attr($address); ?>"
                                                maxlength="300">
                                        </div>
                                        <div class="sf-title-actions-row">
                                            <div class="sf-char-counter">
                                                <span class="sf-char-current">0</span>/<span class="sf-char-optimal">60</span>
                                                <span class="sf-char-status"></span>
                                            </div>
                                            <div class="sf-title-buttons">
                                                <button type="button" class="sf-ai-btn sf-ai-suggest-btn" title="<?php esc_attr_e('Get AI suggestion', 'screaming-fixes'); ?>" data-address="<?php echo esc_attr($address); ?>">
                                                    &#129302; <?php esc_html_e('AI Suggest', 'screaming-fixes'); ?>
                                                </button>
                                                <button type="button" class="sf-save-btn sf-save-title-btn" title="<?php esc_attr_e('Save this title', 'screaming-fixes'); ?>" data-address="<?php echo esc_attr($address); ?>" disabled>
                                                    &#10003; <?php esc_html_e('Save', 'screaming-fixes'); ?>
                                                </button>
                                            </div>
                                        </div>
                                        <!-- Status indicator (shown after fix attempt) -->
                                        <div class="sf-fix-status" style="display: none;">
                                            <div class="sf-fix-status-success" style="display: none;">
                                                <span class="sf-fix-status-icon">&#9989;</span>
                                                <span class="sf-fix-status-text"><?php esc_html_e('Fixed successfully', 'screaming-fixes'); ?></span>
                                            </div>
                                            <div class="sf-fix-status-failed" style="display: none;">
                                                <span class="sf-fix-status-icon">&#10060;</span>
                                                <span class="sf-fix-status-text"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                                                <span class="sf-fix-status-reason"></span>
                                                <span class="sf-fix-status-suggestion"></span>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <!-- Pagination Controls (Bottom) -->
                <?php if ($fixable_count > 100): ?>
                <div class="sf-pagination sf-pagination-bottom" data-section="fixable">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                        <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $fixable_count)); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?>
                        <span class="sf-page-total"><?php echo esc_html($fixable_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            </div><!-- /.sf-fixable-content -->

            <!-- Batch Actions - Outside collapsible content so it stays visible -->
            <div class="sf-batch-actions">
                <div class="sf-selected-count">
                    <span class="sf-count">0</span> <?php esc_html_e('selected', 'screaming-fixes'); ?>
                </div>
                <div class="sf-batch-buttons">
                    <?php if ($seo_plugin_available): ?>
                        <button type="button" class="sf-button sf-button-primary sf-apply-fixes" disabled>
                            <span class="dashicons dashicons-yes"></span>
                            <?php esc_html_e('Apply Page Titles', 'screaming-fixes'); ?>
                        </button>
                    <?php else: ?>
                        <button type="button" class="sf-button sf-button-primary" disabled title="<?php esc_attr_e('Install an SEO plugin to apply fixes', 'screaming-fixes'); ?>">
                            <span class="dashicons dashicons-yes"></span>
                            <?php esc_html_e('Apply Page Titles', 'screaming-fixes'); ?>
                        </button>
                        <span class="sf-plugin-required-hint"><?php esc_html_e('SEO plugin required', 'screaming-fixes'); ?></span>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($manual_titles)): ?>
        <!-- Section 2: Manual Fix Required -->
        <div class="sf-results-section sf-section-manual">
            <button type="button" class="sf-section-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-manual">&#9888;</span>
                <?php printf(
                    esc_html__('Manual Fix Required (%d)', 'screaming-fixes'),
                    $manual_count
                ); ?>
                <span class="sf-section-hint"><?php esc_html_e('Homepage, archives, and special pages - fix in SEO plugin settings', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-manual-content" style="display: none;">
                <div class="sf-table-wrapper sf-paginated-table" data-section="manual" data-per-page="100" data-total="<?php echo esc_attr($manual_count); ?>">
                    <!-- Pagination Controls (Top) -->
                    <?php if ($manual_count > 100): ?>
                    <div class="sf-pagination" data-section="manual">
                        <div class="sf-pagination-info">
                            <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                            <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $manual_count)); ?></span>
                            <?php esc_html_e('of', 'screaming-fixes'); ?>
                            <span class="sf-page-total"><?php echo esc_html($manual_count); ?></span>
                        </div>
                        <div class="sf-pagination-controls">
                            <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                            <div class="sf-page-numbers"></div>
                            <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                        </div>
                    </div>
                    <?php endif; ?>
                    <table class="sf-results-table sf-manual-table">
                        <thead>
                            <tr>
                                <th class="sf-col-page"><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                <th class="sf-col-current"><?php esc_html_e('Current', 'screaming-fixes'); ?></th>
                                <th class="sf-col-issue"><?php esc_html_e('Issue', 'screaming-fixes'); ?></th>
                                <th class="sf-col-howto"><?php esc_html_e('How to Fix', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($manual_titles as $title_item):
                                $address = $title_item['address'] ?? '';
                                $current_title = $title_item['current_title'] ?? '';
                                $title_length = $title_item['title_length'] ?? strlen($current_title);
                                $issue_type = $title_item['issue_type'] ?? '';
                                $category_note = $title_item['category_note'] ?? '';
                                $path = wp_parse_url($address, PHP_URL_PATH) ?: $address;

                                $issue_badges = [
                                    'missing' => ['class' => 'sf-badge-missing', 'label' => __('Missing', 'screaming-fixes')],
                                    'duplicate' => ['class' => 'sf-badge-duplicate', 'label' => __('Duplicate', 'screaming-fixes')],
                                    'too_long' => ['class' => 'sf-badge-too-long', 'label' => __('Over 60', 'screaming-fixes')],
                                    'too_short' => ['class' => 'sf-badge-too-short', 'label' => __('Below 30', 'screaming-fixes')],
                                    'multiple' => ['class' => 'sf-badge-multiple', 'label' => __('Multiple', 'screaming-fixes')],
                                ];
                                $badge = $issue_badges[$issue_type] ?? ['class' => '', 'label' => ''];
                            ?>
                                <tr class="sf-title-row sf-manual-row" data-issue-type="<?php echo esc_attr($issue_type); ?>">
                                    <td class="sf-col-page">
                                        <a href="<?php echo esc_url($address); ?>" target="_blank" rel="noopener">
                                            <?php echo esc_html($path); ?>
                                        </a>
                                    </td>
                                    <td class="sf-col-current">
                                        <div class="sf-current-title">
                                            <?php if (empty($current_title)): ?>
                                                <span class="sf-no-title"><?php esc_html_e('(No title)', 'screaming-fixes'); ?></span>
                                            <?php else: ?>
                                                <span class="sf-title-text"><?php echo esc_html($current_title); ?></span>
                                            <?php endif; ?>
                                            <span class="sf-char-count sf-char-count-<?php echo $title_length > 60 ? 'bad' : ($title_length < 30 ? 'warning' : 'good'); ?>">
                                                <?php echo esc_html($title_length); ?> <?php esc_html_e('chars', 'screaming-fixes'); ?>
                                            </span>
                                        </div>
                                    </td>
                                    <td class="sf-col-issue">
                                        <span class="sf-issue-badge <?php echo esc_attr($badge['class']); ?>">
                                            <?php echo esc_html($badge['label']); ?>
                                        </span>
                                    </td>
                                    <td class="sf-col-howto">
                                        <span class="sf-howto-note"><?php echo esc_html($category_note); ?></span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <!-- Pagination Controls (Bottom) -->
                    <?php if ($manual_count > 100): ?>
                    <div class="sf-pagination sf-pagination-bottom" data-section="manual">
                        <div class="sf-pagination-info">
                            <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                            <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $manual_count)); ?></span>
                            <?php esc_html_e('of', 'screaming-fixes'); ?>
                            <span class="sf-page-total"><?php echo esc_html($manual_count); ?></span>
                        </div>
                        <div class="sf-pagination-controls">
                            <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                            <div class="sf-page-numbers"></div>
                            <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($skipped_titles)): ?>
        <!-- Section 3: Skipped -->
        <div class="sf-results-section sf-section-skipped">
            <button type="button" class="sf-section-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-skipped">&#8856;</span>
                <?php printf(
                    esc_html__('Skipped (%d)', 'screaming-fixes'),
                    $skipped_count
                ); ?>
                <span class="sf-section-hint"><?php esc_html_e('Non-indexable pages, canonicalized pages', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-skipped-content" style="display: none;">
                <div class="sf-table-wrapper sf-paginated-table" data-section="skipped" data-per-page="100" data-total="<?php echo esc_attr($skipped_count); ?>">
                    <!-- Pagination Controls (Top) -->
                    <?php if ($skipped_count > 100): ?>
                    <div class="sf-pagination" data-section="skipped">
                        <div class="sf-pagination-info">
                            <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                            <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $skipped_count)); ?></span>
                            <?php esc_html_e('of', 'screaming-fixes'); ?>
                            <span class="sf-page-total"><?php echo esc_html($skipped_count); ?></span>
                        </div>
                        <div class="sf-pagination-controls">
                            <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                            <div class="sf-page-numbers"></div>
                            <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                        </div>
                    </div>
                    <?php endif; ?>
                    <table class="sf-results-table sf-skipped-table">
                        <thead>
                            <tr>
                                <th class="sf-col-page"><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                <th class="sf-col-reason"><?php esc_html_e('Reason Skipped', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($skipped_titles as $title_item):
                                $address = $title_item['address'] ?? '';
                                $category_note = $title_item['category_note'] ?? '';
                                $path = wp_parse_url($address, PHP_URL_PATH) ?: $address;
                            ?>
                                <tr class="sf-title-row sf-skipped-row">
                                    <td class="sf-col-page">
                                        <a href="<?php echo esc_url($address); ?>" target="_blank" rel="noopener">
                                            <?php echo esc_html($path); ?>
                                        </a>
                                    </td>
                                    <td class="sf-col-reason">
                                        <?php echo esc_html($category_note); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <!-- Pagination Controls (Bottom) -->
                    <?php if ($skipped_count > 100): ?>
                    <div class="sf-pagination sf-pagination-bottom" data-section="skipped">
                        <div class="sf-pagination-info">
                            <?php esc_html_e('Showing', 'screaming-fixes'); ?>
                            <span class="sf-page-start">1</span>-<span class="sf-page-end"><?php echo esc_html(min(100, $skipped_count)); ?></span>
                            <?php esc_html_e('of', 'screaming-fixes'); ?>
                            <span class="sf-page-total"><?php echo esc_html($skipped_count); ?></span>
                        </div>
                        <div class="sf-pagination-controls">
                            <button type="button" class="sf-page-prev" disabled>&laquo; <?php esc_html_e('Prev', 'screaming-fixes'); ?></button>
                            <div class="sf-page-numbers"></div>
                            <button type="button" class="sf-page-next"><?php esc_html_e('Next', 'screaming-fixes'); ?> &raquo;</button>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($fixed_titles) && !$bulk_complete):
            // Count by status
            $fixed_success_count = 0;
            $fixed_failed_count = 0;
            $fixed_skipped_count = 0;
            foreach ($fixed_titles as $ft) {
                $s = $ft['status'] ?? 'success';
                if ($s === 'failed') { $fixed_failed_count++; }
                elseif ($s === 'skipped') { $fixed_skipped_count++; }
                else { $fixed_success_count++; }
            }
        ?>
        <!-- Section 4: Fixes Applied (not shown for bulk complete - has its own display) -->
        <div class="sf-results-section sf-section-fixed" id="sf-fixed-results">
            <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="false">
                <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
                <?php printf(
                    esc_html(_n('Fixes Applied (%d)', 'Fixes Applied (%d)', $fixed_count, 'screaming-fixes')),
                    $fixed_count
                ); ?>
                <span class="sf-section-hint"><?php esc_html_e('Page title fix results', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon"></span>
            </button>

            <div class="sf-fixed-content" style="display: none;">
                <div class="sf-fixed-header-row">
                    <div class="sf-fixed-note">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <?php esc_html_e('Results of page title fix operations.', 'screaming-fixes'); ?>
                    </div>
                    <div class="sf-fixed-actions">
                        <button type="button" class="sf-button sf-button-secondary sf-fixed-download-results" id="sf-fixed-download-results">
                            <span class="dashicons dashicons-download"></span>
                            <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                        </button>
                        <button type="button" class="sf-button sf-button-secondary sf-fixed-clear-btn" id="sf-fixed-clear">
                            <span class="dashicons dashicons-trash"></span>
                            <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                        </button>
                    </div>
                </div>

                <div class="sf-status-filter-row sf-pt-status-filters">
                    <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                    <div class="sf-status-filters">
                        <button type="button" class="sf-status-filter active" data-status="all">
                            <?php esc_html_e('All', 'screaming-fixes'); ?> (<span class="sf-pt-status-count-all"><?php echo esc_html($fixed_count); ?></span>)
                        </button>
                        <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                            <?php esc_html_e('Fixed', 'screaming-fixes'); ?> (<span class="sf-pt-status-count-success"><?php echo esc_html($fixed_success_count); ?></span>)
                        </button>
                        <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                            <?php esc_html_e('Failed', 'screaming-fixes'); ?> (<span class="sf-pt-status-count-failed"><?php echo esc_html($fixed_failed_count); ?></span>)
                        </button>
                        <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                            <?php esc_html_e('Skipped', 'screaming-fixes'); ?> (<span class="sf-pt-status-count-skipped"><?php echo esc_html($fixed_skipped_count); ?></span>)
                        </button>
                    </div>
                </div>

                <div class="sf-table-wrapper sf-paginated-table" data-section="fixed" data-per-page="100" data-total="<?php echo esc_attr($fixed_count); ?>">
                    <table class="sf-results-table sf-fixed-table">
                        <thead>
                            <tr>
                                <th class="sf-col-status" style="width: 80px;"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                                <th class="sf-col-page"><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                <th class="sf-col-title"><?php esc_html_e('New Title', 'screaming-fixes'); ?></th>
                                <th class="sf-col-chars" style="width: 80px;"><?php esc_html_e('Chars', 'screaming-fixes'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($fixed_titles as $title_item):
                                $address = $title_item['address'] ?? '';
                                $post_title = $title_item['post_title'] ?? '';
                                $applied_title = $title_item['applied_title'] ?? '';
                                $fixed_at = $title_item['fixed_at'] ?? '';
                                $post_id = $title_item['post_id'] ?? 0;
                                $path = wp_parse_url($address, PHP_URL_PATH) ?: $address;
                                $edit_url = $post_id ? get_edit_post_link($post_id, 'raw') : '';
                                $char_count = strlen($applied_title);
                                $char_class = $char_count >= 30 && $char_count <= 60 ? 'good' : ($char_count > 60 ? 'bad' : 'warning');
                                $item_status = $title_item['status'] ?? 'success';
                                $item_status_message = $title_item['status_message'] ?? '';
                                $row_class = 'sf-fixed-row';
                                if ($item_status === 'failed') { $row_class .= ' sf-fixed-row-failed'; }
                                elseif ($item_status === 'skipped') { $row_class .= ' sf-fixed-row-skipped'; }
                            ?>
                                <tr class="<?php echo esc_attr($row_class); ?>" data-address="<?php echo esc_attr($address); ?>" data-status="<?php echo esc_attr($item_status); ?>">
                                    <td class="sf-col-status">
                                        <?php if ($item_status === 'failed'): ?>
                                            <span class="sf-status-badge sf-status-failed" title="<?php echo esc_attr($item_status_message); ?>">&#10007; <?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                                        <?php elseif ($item_status === 'skipped'): ?>
                                            <span class="sf-status-badge sf-status-skipped" title="<?php echo esc_attr($item_status_message); ?>">&#8212; <?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                                        <?php else: ?>
                                            <span class="sf-status-badge sf-status-success">&#10003; <?php esc_html_e('Updated', 'screaming-fixes'); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="sf-col-page">
                                        <div class="sf-page-cell">
                                            <div class="sf-page-title">
                                                <?php if ($post_id && $edit_url): ?>
                                                    <a href="<?php echo esc_url($edit_url); ?>" target="_blank" rel="noopener" class="sf-edit-link">
                                                        <?php echo esc_html($post_title ?: __('(No title)', 'screaming-fixes')); ?>
                                                        <span class="dashicons dashicons-edit"></span>
                                                    </a>
                                                <?php else: ?>
                                                    <span><?php echo esc_html($post_title ?: __('(No title)', 'screaming-fixes')); ?></span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="sf-page-url">
                                                <a href="<?php echo esc_url($address); ?>" target="_blank" rel="noopener">
                                                    <?php echo esc_html($path); ?>
                                                </a>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="sf-col-title">
                                        <span class="sf-title-text"><?php echo esc_html(wp_unslash($applied_title)); ?></span>
                                    </td>
                                    <td class="sf-col-chars">
                                        <span class="sf-char-count sf-char-count-<?php echo esc_attr($char_class); ?>"><?php echo esc_html($char_count); ?></span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <?php if ($fixed_count > 100): ?>
                <div class="sf-pagination" data-section="fixed">
                    <div class="sf-pagination-info">
                        <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $fixed_count); ?></span>
                        <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($fixed_count); ?></span>
                    </div>
                    <div class="sf-pagination-controls">
                        <button type="button" class="sf-page-btn sf-page-prev" disabled>
                            <span class="dashicons dashicons-arrow-left-alt2"></span>
                        </button>
                        <div class="sf-page-numbers"></div>
                        <button type="button" class="sf-page-btn sf-page-next">
                            <span class="dashicons dashicons-arrow-right-alt2"></span>
                        </button>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Progress Modal -->
        <div class="sf-modal sf-progress-modal" style="display: none;">
            <div class="sf-modal-content">
                <div class="sf-progress-header">
                    <span class="sf-progress-icon dashicons dashicons-update sf-spinning"></span>
                    <h3 class="sf-progress-title"><?php esc_html_e('Applying Page Titles...', 'screaming-fixes'); ?></h3>
                </div>
                <div class="sf-progress-bar">
                    <div class="sf-progress-fill" style="width: 0%;"></div>
                </div>
                <div class="sf-progress-status">
                    <span class="sf-progress-current">0</span> / <span class="sf-progress-total">0</span>
                </div>
            </div>
        </div>

    <?php else: ?>
        <!-- Empty State -->
        <div class="sf-empty-state">
            <div class="sf-empty-icon">&#127991;</div>
            <h3><?php esc_html_e('No page title data yet', 'screaming-fixes'); ?></h3>
            <p><?php esc_html_e('Upload a Screaming Frog Page Title CSV to find and fix page title issues.', 'screaming-fixes'); ?></p>
        </div>
    <?php endif; ?>
</div>
