/**
 * Redirect Chains Module JavaScript
 */
(function($) {
    'use strict';

    var RedirectChains = {
        data: null,
        selectedRedirects: [],

        // Client-side pagination for Fixes Applied section (scan tables use server-side)
        clientPagination: {
            'fixes-applied': { currentPage: 1, perPage: 100, totalItems: 0 }
        },

        init: function() {
            this.bindEvents();
            this.initInstructions();
            this.initClientPagination();
        },

        bindEvents: function() {
            var self = this;

            // Select all checkbox (table header)
            $('#sf-select-all-table').on('change', function() {
                var isChecked = $(this).is(':checked');
                // Only select visible (non-filtered) checkboxes
                $('.sf-redirect-row:not(.sf-filtered-out) .sf-redirect-checkbox').prop('checked', isChecked);
                self.updateSelectedCount();
            });

            // Fixable section toggle (collapsible)
            $(document).on('click', '.sf-fixable-toggle', function() {
                var $btn = $(this);
                var $content = $btn.closest('.sf-fixable-section').find('.sf-fixable-content');
                var isExpanded = $btn.attr('aria-expanded') === 'true';

                if (isExpanded) {
                    $content.slideUp(300);
                    $btn.attr('aria-expanded', 'false');
                    $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
                } else {
                    $content.slideDown(300);
                    $btn.attr('aria-expanded', 'true');
                    $btn.find('.sf-toggle-icon').addClass('sf-rotated');
                }
            });

            // Manual section toggle (collapsible)
            $(document).on('click', '.sf-manual-toggle', function() {
                var $btn = $(this);
                var $content = $btn.closest('.sf-manual-section').find('.sf-manual-content');
                var isExpanded = $btn.attr('aria-expanded') === 'true';

                if (isExpanded) {
                    $content.slideUp(300);
                    $btn.attr('aria-expanded', 'false');
                    $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
                } else {
                    $content.slideDown(300);
                    $btn.attr('aria-expanded', 'true');
                    $btn.find('.sf-toggle-icon').addClass('sf-rotated');
                }
            });

            // Filter by type buttons (internal/external)
            $(document).on('click', '.sf-filter-btn', function() {
                var $btn = $(this);
                var filterType = $btn.data('filter');
                var isActive = $btn.hasClass('active');

                if (isActive) {
                    // Clear filter
                    self.clearFilter();
                } else {
                    // Apply filter
                    self.filterByType(filterType);
                }
            });

            // Filter by issue type buttons (All, Chains, Loops, Temp)
            $(document).on('click', '.sf-filter-stat', function() {
                var $btn = $(this);
                var filter = $btn.data('filter');

                // Check if disabled
                if ($btn.hasClass('sf-filter-disabled')) {
                    return;
                }

                self.filterByIssueType(filter);
            });

            // Pagination buttons
            $(document).on('click', '.sf-page-prev:not(:disabled)', function() {
                var $pagination = $(this).closest('.sf-pagination');
                var section = $pagination.data('section');
                var currentPage = parseInt($pagination.find('.sf-page-num.active').data('page')) || 1;
                self.goToPage(section, currentPage - 1);
            });

            $(document).on('click', '.sf-page-next:not(:disabled)', function() {
                var $pagination = $(this).closest('.sf-pagination');
                var section = $pagination.data('section');
                var currentPage = parseInt($pagination.find('.sf-page-num.active').data('page')) || 1;
                self.goToPage(section, currentPage + 1);
            });

            $(document).on('click', '.sf-page-num:not(.active)', function() {
                var $pagination = $(this).closest('.sf-pagination');
                var section = $pagination.data('section');
                var page = parseInt($(this).data('page'));
                self.goToPage(section, page);
            });

            // Individual checkbox
            $(document).on('change', '.sf-redirect-checkbox', function() {
                self.updateSelectedCount();
            });

            // Expand/collapse sources
            $(document).on('click', '.sf-expand-sources', function() {
                var $btn = $(this);
                var $row = $btn.closest('tr');
                var $sourcesRow = $row.next('.sf-sources-row');
                var isExpanded = $btn.attr('data-expanded') === 'true';

                if (isExpanded) {
                    $sourcesRow.slideUp(200);
                    $btn.attr('data-expanded', 'false');
                    $btn.find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                } else {
                    $sourcesRow.slideDown(200);
                    $btn.attr('data-expanded', 'true');
                    $btn.find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                }
            });

            // Click on Failed badge in Fixes Applied section - show error details
            $(document).on('click', '.sf-fixes-applied-table .sf-status-badge.sf-status-failed', function(e) {
                e.preventDefault();
                self.handleFixedFailedBadgeClick($(this));
            });

            // Fix single redirect (regular chains)
            $(document).on('click', '.sf-fix-single', function() {
                var $row = $(this).closest('.sf-redirect-row');
                self.fixSingleRedirect($row);
            });

            // Fix temp redirect (convert 302 to 301)
            $(document).on('click', '.sf-fix-temp', function() {
                var $btn = $(this);
                var $row = $btn.closest('.sf-redirect-row');
                self.fixTempRedirect($row, $btn);
            });

            // Can't Fix external temp redirect - show info message
            $(document).on('click', '.sf-cant-fix-external', function() {
                var $btn = $(this);
                var $row = $btn.closest('.sf-redirect-row');
                var $info = $row.next('.sf-external-temp-info');

                if ($info.length && $info.is(':visible')) {
                    $info.slideUp(200);
                } else if ($info.length) {
                    $info.slideDown(200);
                } else {
                    // Create the info row
                    var infoHtml = '<tr class="sf-external-temp-info">' +
                        '<td colspan="7">' +
                        '<div class="sf-external-info-box">' +
                        '<div class="sf-external-info-icon">&#128161;</div>' +
                        '<div class="sf-external-info-content">' +
                        '<strong>This redirect lives on an external website</strong>' +
                        '<p>This temporary redirect (302/303) is controlled by the external site (e.g. YouTube, social media, podcast hosts, affiliate platforms). ' +
                        'It can\'t be changed to a 301 with this tool because the redirect rule doesn\'t exist on your website.</p>' +
                        '<p class="sf-external-info-reassurance">This is standard behavior for most external services and is <strong>not an SEO concern</strong>. ' +
                        'Search engines understand these redirects and they won\'t negatively impact your site\'s rankings.</p>' +
                        '</div>' +
                        '</div>' +
                        '</td>' +
                        '</tr>';
                    $row.after(infoHtml);
                    $row.next('.sf-external-temp-info').hide().slideDown(200);
                }
            });

            // Fix redirect loop (delete rule)
            $(document).on('click', '.sf-fix-loop', function() {
                var $btn = $(this);
                var $row = $btn.closest('.sf-redirect-row');
                self.fixRedirectLoop($row, $btn);
            });

            // Apply selected fixes
            $('#sf-apply-redirect-fixes').on('click', function() {
                self.applySelectedFixes();
            });

            // Export
            $('#sf-export-redirects').on('click', function() {
                self.exportResults();
            });

            // Clear data
            $('#sf-clear-redirects').on('click', function() {
                self.clearData();
            });

            // View Fixed button - scroll to fixed section
            $('#sf-view-fixed-redirects').on('click', function() {
                self.scrollToFixedSection();
            });

            // Export fixed redirects
            $(document).on('click', '#sf-export-fixed-redirects', function() {
                self.exportFixedResults();
            });

            // Fixes Applied status filter buttons
            $(document).on('click', '.sf-rc-status-filters .sf-status-filter', function() {
                var status = $(this).data('status');
                $(this).closest('.sf-status-filters').find('.sf-status-filter').removeClass('active');
                $(this).addClass('active');
                self.filterFixResults(status);
            });

            // Client-side pagination for Fixes Applied
            $('#sf-fixed-redirects-section').on('click', '.sf-page-prev', function() {
                var section = $(this).closest('.sf-pagination').data('section');
                if (self.clientPagination[section] && self.clientPagination[section].currentPage > 1) {
                    self.goToClientPage(section, self.clientPagination[section].currentPage - 1);
                }
            });
            $('#sf-fixed-redirects-section').on('click', '.sf-page-next', function() {
                var section = $(this).closest('.sf-pagination').data('section');
                if (self.clientPagination[section]) {
                    var totalPages = Math.ceil(self.getClientVisibleRowCount(section) / self.clientPagination[section].perPage);
                    if (self.clientPagination[section].currentPage < totalPages) {
                        self.goToClientPage(section, self.clientPagination[section].currentPage + 1);
                    }
                }
            });
            $('#sf-fixed-redirects-section').on('click', '.sf-page-number', function() {
                var section = $(this).closest('.sf-pagination').data('section');
                var page = parseInt($(this).data('page'), 10);
                if (section && page) {
                    self.goToClientPage(section, page);
                }
            });

            // Fixed section toggle
            $(document).on('click', '.sf-fixed-toggle', function() {
                var $btn = $(this);
                var $content = $btn.closest('.sf-fixed-section').find('.sf-fixed-content');
                var isExpanded = $btn.attr('aria-expanded') === 'true';

                if (isExpanded) {
                    $content.slideUp(300);
                    $btn.attr('aria-expanded', 'false');
                    $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
                } else {
                    $content.slideDown(300);
                    $btn.attr('aria-expanded', 'true');
                    $btn.find('.sf-toggle-icon').addClass('sf-rotated');
                }
            });

            // Copy URL
            $(document).on('click', '.sf-copy-url', function() {
                var url = $(this).data('url');
                self.copyToClipboard(url);

                var $btn = $(this);
                $btn.addClass('sf-copied');
                setTimeout(function() {
                    $btn.removeClass('sf-copied');
                }, 1500);
            });

            // ========================================
            // BULK UPLOAD HANDLERS
            // ========================================

            // Bulk confirm button
            $(document).on('click', '#sf-bulk-confirm', function() {
                self.handleBulkConfirm();
            });

            // Bulk clear button
            $(document).on('click', '#sf-bulk-clear, #sf-bulk-new-upload', function() {
                self.handleBulkClear();
            });

            // Bulk download preview CSV
            $(document).on('click', '#sf-bulk-download-preview', function() {
                self.handleBulkDownloadPreview();
            });

            // Bulk download results CSV
            $(document).on('click', '#sf-bulk-download-results', function() {
                self.handleBulkDownloadResults();
            });
        },

        initInstructions: function() {
            $('.sf-instructions-header').on('click', function() {
                var $box = $(this).closest('.sf-instructions-box');
                var $content = $box.find('.sf-instructions-content');
                var isCollapsed = $box.attr('data-collapsed') === 'true';

                if (isCollapsed) {
                    $content.slideDown(300);
                    $box.attr('data-collapsed', 'false');
                    $(this).attr('aria-expanded', 'true');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                } else {
                    $content.slideUp(300);
                    $box.attr('data-collapsed', 'true');
                    $(this).attr('aria-expanded', 'false');
                    $(this).find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                }
            });
        },

        updateSelectedCount: function() {
            // Count only visible (non-filtered) checked checkboxes
            var count = $('.sf-redirect-row:not(.sf-filtered-out) .sf-redirect-checkbox:checked').length;
            $('#sf-selected-redirect-count').text(count);
            $('#sf-apply-redirect-fixes').prop('disabled', count === 0);

            // Update table header select all checkbox state
            var visibleCheckboxes = $('.sf-redirect-row:not(.sf-filtered-out) .sf-redirect-checkbox');
            var totalVisible = visibleCheckboxes.length;
            var checkedVisible = visibleCheckboxes.filter(':checked').length;
            var $selectAllTable = $('#sf-select-all-table');

            if (checkedVisible === 0) {
                $selectAllTable.prop('checked', false).prop('indeterminate', false);
            } else if (checkedVisible === totalVisible) {
                $selectAllTable.prop('checked', true).prop('indeterminate', false);
            } else {
                $selectAllTable.prop('checked', false).prop('indeterminate', true);
            }
        },

        filterByType: function(filterType) {
            var $rows = $('.sf-redirect-row');

            $rows.each(function() {
                var $row = $(this);
                var isInternal = $row.find('.sf-type-badge').hasClass('sf-type-internal');
                var rowType = isInternal ? 'internal' : 'external';
                var shouldHide = rowType !== filterType;

                $row.toggleClass('sf-filtered-out', shouldHide);
                $row.next('.sf-sources-row').toggleClass('sf-filtered-out', shouldHide);
            });

            // Update button states
            $('.sf-filter-btn').removeClass('active');
            $('.sf-filter-btn[data-filter="' + filterType + '"]').addClass('active');

            // Uncheck filtered-out rows and update count
            $('.sf-redirect-row.sf-filtered-out .sf-redirect-checkbox').prop('checked', false);
            this.updateSelectedCount();
        },

        clearFilter: function() {
            $('.sf-redirect-row, .sf-sources-row').removeClass('sf-filtered-out');
            $('.sf-filter-btn').removeClass('active');
            this.updateSelectedCount();
        },

        filterByIssueType: function(filterType) {
            var $rows = $('.sf-fixable-section .sf-redirect-row');

            if (filterType === 'all') {
                // Show all rows
                $rows.removeClass('sf-filtered-out');
                $rows.each(function() {
                    $(this).next('.sf-sources-row').removeClass('sf-filtered-out');
                });
            } else {
                // Filter by specific issue type
                $rows.each(function() {
                    var $row = $(this);
                    var isLoop = $row.data('is-loop') === true || $row.data('is-loop') === 'true';
                    var hasTemp = $row.data('has-temp') === true || $row.data('has-temp') === 'true';
                    var isChain = !isLoop && !hasTemp;

                    var show = false;
                    if (filterType === 'chains' && isChain) show = true;
                    if (filterType === 'loops' && isLoop) show = true;
                    if (filterType === 'temp' && hasTemp && !isLoop) show = true;

                    if (show) {
                        $row.removeClass('sf-filtered-out');
                        $row.next('.sf-sources-row').removeClass('sf-filtered-out');
                    } else {
                        $row.addClass('sf-filtered-out');
                        $row.next('.sf-sources-row').addClass('sf-filtered-out');
                    }
                });
            }

            // Update active state
            $('.sf-filter-stat').removeClass('active');
            $('.sf-filter-stat[data-filter="' + filterType + '"]').addClass('active');

            // Update description
            this.updateFilterDescription(filterType);

            // Uncheck filtered-out rows and update count
            $('.sf-redirect-row.sf-filtered-out .sf-redirect-checkbox').prop('checked', false);
            this.updateSelectedCount();

            // Update visible count display if needed
            this.updateVisibleCount();
        },

        updateFilterDescription: function(filterType) {
            var $descContainer = $('#sf-filter-description');
            if (!$descContainer.length) {
                return;
            }

            // Hide all descriptions
            $descContainer.find('.sf-filter-desc-content').hide();

            // Show the matching description
            var $activeDesc = $descContainer.find('.sf-filter-desc-content[data-filter="' + filterType + '"]');
            if ($activeDesc.length) {
                $activeDesc.fadeIn(200);
            }

            // Update container styling based on filter type
            $descContainer.removeClass('sf-filter-desc-chains sf-filter-desc-loops sf-filter-desc-temp');
            if (filterType !== 'all') {
                $descContainer.addClass('sf-filter-desc-' + filterType);
            }
        },

        updateVisibleCount: function() {
            var visibleCount = $('.sf-fixable-section .sf-redirect-row:not(.sf-filtered-out)').length;
            // Could update a display element here if needed
        },

        updateFilterCounts: function() {
            var $rows = $('.sf-fixable-section .sf-redirect-row:not(.sf-row-fixed)');

            var chainsCount = 0;
            var loopsCount = 0;
            var tempCount = 0;

            $rows.each(function() {
                var $row = $(this);
                var isLoop = $row.data('is-loop') === true || $row.data('is-loop') === 'true';
                var hasTemp = $row.data('has-temp') === true || $row.data('has-temp') === 'true';

                if (isLoop) {
                    loopsCount++;
                } else if (hasTemp) {
                    tempCount++;
                } else {
                    chainsCount++;
                }
            });

            var totalCount = chainsCount + loopsCount + tempCount;

            // Update filter button counts
            $('.sf-filter-stat[data-filter="all"] .sf-filter-count').text(totalCount);
            $('.sf-filter-stat[data-filter="chains"] .sf-filter-count').text(chainsCount);
            $('.sf-filter-stat[data-filter="loops"] .sf-filter-count').text(loopsCount);
            $('.sf-filter-stat[data-filter="temp"] .sf-filter-count').text(tempCount);

            // Update disabled state
            $('.sf-filter-stat[data-filter="chains"]').toggleClass('sf-filter-disabled', chainsCount === 0);
            $('.sf-filter-stat[data-filter="loops"]').toggleClass('sf-filter-disabled', loopsCount === 0);
            $('.sf-filter-stat[data-filter="temp"]').toggleClass('sf-filter-disabled', tempCount === 0);
        },

        fixTempRedirect: function($row, $btn) {
            var self = this;
            var source = $btn.data('source');

            if (!source) {
                self.showNotice('error', 'No source URL found.');
                return;
            }

            // Confirm the action
            if (!confirm('Convert this temporary redirect (302) to a permanent redirect (301)?')) {
                return;
            }

            var originalText = $btn.text();
            $btn.prop('disabled', true).text('Converting...');

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_fix_temp_redirect',
                    nonce: sfRedirectChainsData.nonce,
                    source: source
                },
                success: function(response) {
                    if (response.success) {
                        self.showNotice('success', response.data.message);

                        // Add result to Fixes Applied section
                        self.addFixResultRow({
                            address: source,
                            final_address: $row.data('final') || source,
                            status: 'success',
                            status_message: '',
                            source_count: 0,
                            fixed_at: new Date().toISOString()
                        });
                        self.updateFixStatusCounts();

                        // Mark row as fixed and remove it
                        $row.addClass('sf-row-fixed');
                        $row.next('.sf-sources-row').fadeOut(300, function() {
                            $(this).remove();
                        });
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            self.updateStats();
                            self.updateFilterCounts();
                        });
                    } else {
                        self.showNotice('error', response.data.message || 'Failed to convert redirect.');

                        // Add failed result to Fixes Applied section
                        self.addFixResultRow({
                            address: source,
                            final_address: $row.data('final') || source,
                            status: 'failed',
                            status_message: response.data.message || 'Failed to convert redirect.',
                            source_count: 0,
                            fixed_at: new Date().toISOString()
                        });
                        self.updateFixStatusCounts();

                        $btn.prop('disabled', false).text(originalText);
                        $row.addClass('sf-row-error');
                    }
                },
                error: function(xhr) {
                    var errorMsg = 'Failed to convert redirect.';
                    if (xhr.status === 0) {
                        errorMsg = 'Connection lost. Please check your internet connection.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please check PHP error logs.';
                    }
                    self.showNotice('error', errorMsg);
                    $btn.prop('disabled', false).text(originalText);
                    $row.addClass('sf-row-error');
                }
            });
        },

        fixRedirectLoop: function($row, $btn) {
            var self = this;
            var source = $btn.data('source');

            if (!source) {
                self.showNotice('error', 'No source URL found.');
                return;
            }

            // Confirm the action - this is a destructive action
            if (!confirm('Delete the redirect rule causing this loop?\n\nThis will remove the redirect from your redirect plugin. The URL will no longer redirect.')) {
                return;
            }

            var originalText = $btn.text();
            $btn.prop('disabled', true).text('Deleting...');

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_fix_redirect_loop',
                    nonce: sfRedirectChainsData.nonce,
                    source: source
                },
                success: function(response) {
                    if (response.success) {
                        self.showNotice('success', response.data.message);

                        // Add result to Fixes Applied section
                        self.addFixResultRow({
                            address: source,
                            final_address: $row.data('final') || source,
                            status: 'success',
                            status_message: '',
                            source_count: 0,
                            fixed_at: new Date().toISOString()
                        });
                        self.updateFixStatusCounts();

                        // Mark row as fixed and remove it
                        $row.addClass('sf-row-fixed');
                        $row.next('.sf-sources-row').fadeOut(300, function() {
                            $(this).remove();
                        });
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            self.updateStats();
                            self.updateFilterCounts();
                        });
                    } else {
                        self.showNotice('error', response.data.message || 'Failed to delete redirect rule.');

                        // Add failed result to Fixes Applied section
                        self.addFixResultRow({
                            address: source,
                            final_address: $row.data('final') || source,
                            status: 'failed',
                            status_message: response.data.message || 'Failed to delete redirect rule.',
                            source_count: 0,
                            fixed_at: new Date().toISOString()
                        });
                        self.updateFixStatusCounts();

                        $btn.prop('disabled', false).text(originalText);
                        $row.addClass('sf-row-error');
                    }
                },
                error: function(xhr) {
                    var errorMsg = 'Failed to delete redirect rule.';
                    if (xhr.status === 0) {
                        errorMsg = 'Connection lost. Please check your internet connection.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please check PHP error logs.';
                    }
                    self.showNotice('error', errorMsg);
                    $btn.prop('disabled', false).text(originalText);
                    $row.addClass('sf-row-error');
                }
            });
        },

        goToPage: function(section, page) {
            // Build URL with updated page parameter
            var url = new URL(window.location.href);
            url.searchParams.set('paged_' + section, page);
            window.location.href = url.toString();
        },

        fixSingleRedirect: function($row) {
            var self = this;
            var address = $row.data('address');
            var finalAddress = $row.data('final');

            // Collect post IDs from fixable sources
            var postIds = [];
            $row.next('.sf-sources-row').find('tr[data-post-id]').each(function() {
                var postId = $(this).data('post-id');
                if (postId) {
                    postIds.push(postId);
                }
            });

            if (postIds.length === 0) {
                self.showNotice('error', sfRedirectChainsData.i18n.noFixesSelected || 'No fixable pages found.');
                return;
            }

            // Confirm
            if (!confirm('Fix this redirect in ' + postIds.length + ' page(s)?')) {
                return;
            }

            var $btn = $row.find('.sf-fix-single');
            $btn.prop('disabled', true).text(sfRedirectChainsData.i18n.applyingFixes || 'Fixing...');

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_apply_fixes',
                    nonce: sfRedirectChainsData.nonce,
                    fixes: [{
                        address: address,
                        final_address: finalAddress,
                        post_ids: postIds
                    }]
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var status = data.status || 'success';

                        // Add fix result rows to Fixes Applied section
                        if (data.fix_results && data.fix_results.length > 0) {
                            data.fix_results.forEach(function(result) {
                                self.addFixResultRow(result);
                            });
                            self.updateFixStatusCounts();
                        }

                        if (status === 'success') {
                            // Full success - remove the row
                            self.showNotice('success', data.message);
                            $row.addClass('sf-row-fixed');
                            $row.next('.sf-sources-row').fadeOut(300, function() {
                                $(this).remove();
                            });
                            $row.fadeOut(300, function() {
                                $(this).remove();
                                self.updateStats();
                                self.updateFilterCounts();
                            });
                        } else if (status === 'partial') {
                            // Partial success - show warning with details
                            self.showNotice('warning', data.message);
                            if (data.errors && data.errors.length > 0) {
                                self.showErrorDetails($row, data.errors);
                            }
                            $btn.prop('disabled', false).text('Retry');
                            $row.addClass('sf-row-partial');
                        } else if (status === 'error') {
                            // Complete failure
                            self.showNotice('error', data.message);
                            if (data.errors && data.errors.length > 0) {
                                self.showErrorDetails($row, data.errors);
                            }
                            $btn.prop('disabled', false).text('Fix');
                            $row.addClass('sf-row-error');
                        } else {
                            // No changes made
                            self.showNotice('info', data.message);
                            $btn.prop('disabled', false).text('Fix');
                        }
                    } else {
                        self.showNotice('error', response.data.message || 'An error occurred.');
                        $btn.prop('disabled', false).text('Fix');
                        $row.addClass('sf-row-error');
                    }
                },
                error: function(xhr) {
                    var errorMsg = sfRedirectChainsData.i18n.fixesFailed || 'Failed to apply fix.';
                    if (xhr.status === 0) {
                        errorMsg = 'Connection lost. Please check your internet connection.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please check PHP error logs.';
                    }
                    self.showNotice('error', errorMsg);
                    $btn.prop('disabled', false).text('Fix');
                    $row.addClass('sf-row-error');
                }
            });
        },

        showErrorDetails: function($row, errors) {
            // Store error details on the row for display when Failed badge is clicked
            // Don't display inline to avoid taking up vertical space in the Fix column
            var errorMessages = [];
            errors.forEach(function(error) {
                var postInfo = error.post_id ? ' (Post ID: ' + error.post_id + ')' : '';
                errorMessages.push(error.message + postInfo);
            });
            $row.data('error-messages', errorMessages);
        },

        handleFixedFailedBadgeClick: function($badge) {
            var $row = $badge.closest('.sf-fixed-row');
            var $detailRow = $row.next('.sf-fixed-error-detail-row');

            // If detail row already exists, toggle it
            if ($detailRow.length) {
                $detailRow.slideToggle(200);
                $badge.toggleClass('sf-badge-expanded');
                return;
            }

            // Get error message from title attribute (status_message)
            var statusMessage = $badge.attr('title') || '';
            var reason = statusMessage || 'This fix could not be applied.';
            var suggestion = '';

            // Contextual troubleshooting suggestions
            var lowerReason = reason.toLowerCase();
            if (lowerReason.indexOf('not found in') !== -1 || lowerReason.indexOf('url not found') !== -1) {
                suggestion = 'The URL may have already been fixed, or it exists in a format the plugin couldn\'t match (e.g., encoded characters, relative URL, or inside a shortcode/page builder).';
            } else if (lowerReason.indexOf('post') !== -1 && lowerReason.indexOf('not found') !== -1) {
                suggestion = 'The source post may have been deleted or moved to trash. Check the trash or verify the post still exists.';
            } else if (lowerReason.indexOf('permission') !== -1) {
                suggestion = 'Your user role may not have permission to edit this content. Contact a site administrator.';
            } else if (lowerReason.indexOf('could not update') !== -1 || lowerReason.indexOf('database') !== -1) {
                suggestion = 'There may be a database issue. Try again, or check your WordPress error log for details.';
            } else if (lowerReason.indexOf('redirect') !== -1 && lowerReason.indexOf('plugin') !== -1) {
                suggestion = 'The redirect rule may be managed by a plugin that doesn\'t support programmatic changes. Check your redirect plugin settings directly.';
            } else {
                suggestion = 'Try fixing this link manually by editing the post directly. If the issue persists, the link may be in a location the plugin cannot modify (e.g., a page builder, shortcode, or serialized data).';
            }

            var detailHtml = '<tr class="sf-fixed-error-detail-row">' +
                '<td colspan="5">' +
                '<div class="sf-fixed-error-detail">' +
                '<div class="sf-error-info-icon">&#9888;&#65039;</div>' +
                '<div class="sf-error-info-content">' +
                '<strong class="sf-error-info-reason">' + this.escapeHtml(reason) + '</strong>' +
                '<span class="sf-error-info-suggestion">' + this.escapeHtml(suggestion) + '</span>' +
                '</div>' +
                '</div>' +
                '</td>' +
                '</tr>';

            $row.after(detailHtml);
            $row.next('.sf-fixed-error-detail-row').hide().slideDown(200);
            $badge.addClass('sf-badge-expanded');
        },

        applySelectedFixes: function() {
            var self = this;
            var fixes = [];
            var $checkedRows = $('.sf-redirect-checkbox:checked').closest('.sf-redirect-row');

            $('.sf-redirect-checkbox:checked').each(function() {
                var $row = $(this).closest('.sf-redirect-row');
                var address = $row.data('address');
                var finalAddress = $row.data('final');

                var postIds = [];
                $row.next('.sf-sources-row').find('tr[data-post-id]').each(function() {
                    var postId = $(this).data('post-id');
                    if (postId) {
                        postIds.push(postId);
                    }
                });

                if (postIds.length > 0) {
                    fixes.push({
                        address: address,
                        final_address: finalAddress,
                        post_ids: postIds,
                        $row: $row
                    });
                }
            });

            if (fixes.length === 0) {
                self.showNotice('error', sfRedirectChainsData.i18n.noFixesSelected || 'No fixes selected.');
                return;
            }

            var confirmMsg = sfRedirectChainsData.i18n.confirmApply || 'Apply fixes to %d redirect issues?';
            confirmMsg = confirmMsg.replace('%d', fixes.length);

            if (!confirm(confirmMsg)) {
                return;
            }

            var $btn = $('#sf-apply-redirect-fixes');
            var originalText = $btn.html();
            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> ' + (sfRedirectChainsData.i18n.applyingFixes || 'Applying...'));

            // Prepare data without jQuery objects
            var fixesData = fixes.map(function(fix) {
                return {
                    address: fix.address,
                    final_address: fix.final_address,
                    post_ids: fix.post_ids
                };
            });

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_apply_fixes',
                    nonce: sfRedirectChainsData.nonce,
                    fixes: fixesData
                },
                success: function(response) {
                    $btn.prop('disabled', false).html(originalText);

                    if (response.success) {
                        var data = response.data;
                        var status = data.status || 'success';
                        var summary = data.summary || {};

                        // Add fix result rows to Fixes Applied section
                        if (data.fix_results && data.fix_results.length > 0) {
                            data.fix_results.forEach(function(result) {
                                self.addFixResultRow(result);
                            });
                            self.updateFixStatusCounts();
                        }

                        if (status === 'success') {
                            // All succeeded - remove all checked rows
                            self.showNotice('success', data.message);
                            $checkedRows.each(function() {
                                var $row = $(this);
                                $row.addClass('sf-row-fixed');
                                $row.next('.sf-sources-row').fadeOut(200, function() {
                                    $(this).remove();
                                });
                                $row.fadeOut(200, function() {
                                    $(this).remove();
                                });
                            });
                            setTimeout(function() {
                                self.updateStats();
                                self.updateSelectedCount();
                                self.updateFilterCounts();
                            }, 300);
                        } else if (status === 'partial' || status === 'error') {
                            self.showNotice(status === 'partial' ? 'warning' : 'error', data.message);

                            // If partial success, remove successfully fixed rows; keep failed rows with error class
                            if (status === 'partial' && summary.success > 0) {
                                if (data.errors) {
                                    var failedAddresses = data.errors.map(function(e) { return e.address; });
                                    $checkedRows.each(function() {
                                        var $row = $(this);
                                        var address = $row.data('address');
                                        if (failedAddresses.indexOf(address) === -1) {
                                            $row.addClass('sf-row-fixed');
                                            $row.next('.sf-sources-row').fadeOut(200, function() {
                                                $(this).remove();
                                            });
                                            $row.fadeOut(200, function() {
                                                $(this).remove();
                                            });
                                        } else {
                                            $row.addClass('sf-row-error');
                                            $row.find('.sf-redirect-checkbox').prop('checked', false);
                                        }
                                    });
                                }
                                setTimeout(function() {
                                    self.updateStats();
                                    self.updateSelectedCount();
                                    self.updateFilterCounts();
                                }, 300);
                            } else {
                                // All failed - mark all with error
                                $checkedRows.addClass('sf-row-error');
                            }
                        } else {
                            self.showNotice('info', data.message);
                        }
                    } else {
                        self.showNotice('error', response.data.message || 'An error occurred.');
                        $checkedRows.addClass('sf-row-error');
                    }
                },
                error: function(xhr) {
                    var errorMsg = sfRedirectChainsData.i18n.fixesFailed || 'Failed to apply fixes.';
                    if (xhr.status === 0) {
                        errorMsg = 'Connection lost. Please check your internet connection.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please check PHP error logs.';
                    }
                    self.showNotice('error', errorMsg);
                    $btn.prop('disabled', false).html(originalText);
                    $checkedRows.addClass('sf-row-error');
                }
            });
        },

        truncateUrl: function(url, maxLength) {
            if (!url || url.length <= maxLength) {
                return url;
            }
            return url.substring(0, maxLength - 3) + '...';
        },

        escapeHtml: function(str) {
            if (!str) return '';
            var div = document.createElement('div');
            div.textContent = str;
            return div.innerHTML;
        },

        exportResults: function() {
            var self = this;
            var $btn = $('#sf-export-redirects');
            var originalHtml = $btn.html();

            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> ' + (sfRedirectChainsData.i18n.exporting || 'Exporting...'));

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_export',
                    nonce: sfRedirectChainsData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.downloadCSV(response.data.csv, response.data.filename);
                        self.showNotice('success', sfRedirectChainsData.i18n.exportComplete || 'Export complete.');
                    } else {
                        self.showNotice('error', response.data.message);
                    }
                    $btn.prop('disabled', false).html(originalHtml);
                },
                error: function() {
                    self.showNotice('error', 'Export failed.');
                    $btn.prop('disabled', false).html(originalHtml);
                }
            });
        },

        clearData: function() {
            var self = this;

            // Confirm before clearing
            if (!confirm(sfRedirectChainsData.i18n.confirmClear || 'Are you sure you want to clear all redirect data? This will allow you to upload a new CSV file.')) {
                return;
            }

            var $btn = $('#sf-clear-redirects');
            $btn.prop('disabled', true);

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_clear_data',
                    nonce: sfRedirectChainsData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.showNotice('success', response.data.message || 'Data cleared successfully.');
                        // Reload page to show upload form
                        location.reload();
                    } else {
                        self.showNotice('error', response.data.message || 'Failed to clear data.');
                    }
                },
                error: function() {
                    self.showNotice('error', 'Failed to clear data.');
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        downloadCSV: function(content, filename) {
            var blob = new Blob([content], { type: 'text/csv;charset=utf-8;' });
            var link = document.createElement('a');

            if (navigator.msSaveBlob) {
                navigator.msSaveBlob(blob, filename);
            } else {
                link.href = URL.createObjectURL(blob);
                link.download = filename;
                link.style.display = 'none';
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
            }
        },

        copyToClipboard: function(text) {
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(text);
            } else {
                var $temp = $('<textarea>');
                $('body').append($temp);
                $temp.val(text).select();
                document.execCommand('copy');
                $temp.remove();
            }
        },

        scrollToFixedSection: function() {
            var $section = $('#sf-fixed-redirects-section');
            if ($section.length) {
                // Make sure the section is expanded
                var $toggle = $section.find('.sf-fixed-toggle');
                var $content = $section.find('.sf-fixed-content');

                if ($toggle.attr('aria-expanded') !== 'true') {
                    $content.slideDown(300);
                    $toggle.attr('aria-expanded', 'true');
                    $toggle.find('.sf-toggle-icon').addClass('sf-rotated');
                }

                // Scroll to section with offset for header
                $('html, body').animate({
                    scrollTop: $section.offset().top - 100
                }, 500);

                // Briefly highlight the section
                $section.addClass('sf-section-highlight');
                setTimeout(function() {
                    $section.removeClass('sf-section-highlight');
                }, 2000);
            }
        },

        exportFixedResults: function() {
            var self = this;
            var $btn = $('#sf-export-fixed-redirects');
            var originalHtml = $btn.html();

            $btn.prop('disabled', true).html('<span class="sf-spinner"></span> ' + (sfRedirectChainsData.i18n.exporting || 'Exporting...'));

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_export_fixed',
                    nonce: sfRedirectChainsData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.downloadCSV(response.data.csv, response.data.filename);
                        self.showNotice('success', sfRedirectChainsData.i18n.exportComplete || 'Export complete.');
                    } else {
                        self.showNotice('error', response.data.message || 'Export failed.');
                    }
                    $btn.prop('disabled', false).html(originalHtml);
                },
                error: function() {
                    self.showNotice('error', 'Export failed.');
                    $btn.prop('disabled', false).html(originalHtml);
                }
            });
        },

        updateStats: function() {
            var fixableCount = $('.sf-fixable-section .sf-redirect-row').length;
            var manualCount = $('.sf-manual-section .sf-redirect-row').length;
            var totalCount = fixableCount + manualCount;
            var fixedCount = $('.sf-fixed-section .sf-fixed-row').length;

            // Update stat cards
            $('.sf-stat-card:first .sf-stat-number').text(totalCount);
            $('.sf-stat-fixable .sf-stat-number').text(fixableCount);
            $('.sf-stat-manual .sf-stat-number').text(manualCount);

            // Update header fixed count
            var $headerFixedStat = $('.sf-module-stats .sf-stat-success');
            if (fixedCount > 0) {
                if ($headerFixedStat.length) {
                    $headerFixedStat.find('strong').text(fixedCount);
                } else {
                    var fixedStatHtml = '<span class="sf-stat sf-stat-success">' +
                        '<strong>' + fixedCount + '</strong> ' +
                        (sfRedirectChainsData.i18n.fixed || 'fixed') +
                        '</span>';
                    $('.sf-module-stats').append(fixedStatHtml);
                }
            }

            // Update or show Fixed button in action bar
            var $fixedBtn = $('#sf-view-fixed-redirects');
            if (fixedCount > 0) {
                if ($fixedBtn.length) {
                    // Update count in existing button
                    $fixedBtn.html('<span class="dashicons dashicons-yes-alt"></span> ' +
                        (sfRedirectChainsData.i18n.fixedLabel || 'Fixed') + ' (' + fixedCount + ')');
                } else {
                    // Create the button if it doesn't exist
                    var fixedBtnHtml = '<button type="button" class="sf-button sf-button-secondary sf-fixed-btn" id="sf-view-fixed-redirects" title="View fixed redirects">' +
                        '<span class="dashicons dashicons-yes-alt"></span> ' +
                        (sfRedirectChainsData.i18n.fixedLabel || 'Fixed') + ' (' + fixedCount + ')' +
                        '</button>';
                    $('#sf-export-redirects').after(fixedBtnHtml);

                    // Bind click handler to new button
                    var self = this;
                    $('#sf-view-fixed-redirects').on('click', function() {
                        self.scrollToFixedSection();
                    });
                }
            }

            // If no more fixable, hide the section
            if (fixableCount === 0) {
                $('.sf-fixable-section').fadeOut(300);
            }

            // If nothing left, show empty state
            if (totalCount === 0) {
                $('.sf-stats-summary, .sf-action-bar').fadeOut(300);
                var emptyHtml = '<div class="sf-empty-state sf-success-state">' +
                    '<div class="sf-empty-icon">&#10004;</div>' +
                    '<h3>All redirect issues fixed!</h3>' +
                    '<p>Upload another CSV to check for more redirect issues.</p>' +
                    '</div>';
                $('.sf-results-section').not('.sf-fixed-section').remove();

                // Keep the fixed section visible if it exists
                if (fixedCount === 0) {
                    $('.sf-fixed-section').remove();
                }

                if ($('.sf-empty-state').length === 0) {
                    $('.sf-module-content').append(emptyHtml);
                }
            }
        },

        /**
         * Refresh the Fixed Redirects section via AJAX
         */
        refreshFixedSection: function() {
            var self = this;
            var $fixedSection = $('#sf-fixed-redirects-section');

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_get_fixed_section',
                    nonce: sfRedirectChainsData.nonce
                },
                success: function(response) {
                    if (response.success && response.data.html) {
                        if ($fixedSection.length) {
                            // Update existing section
                            $fixedSection.html(response.data.html);
                        } else {
                            // Create section if it doesn't exist
                            var $newSection = $('<div class="sf-results-section sf-fixed-section" id="sf-fixed-redirects-section"></div>');
                            $newSection.html(response.data.html);
                            // Insert after the last results section or at end of module content
                            var $lastSection = $('.sf-results-section').last();
                            if ($lastSection.length) {
                                $lastSection.after($newSection);
                            } else {
                                $('.sf-module-content').append($newSection);
                            }
                            $fixedSection = $newSection;
                        }

                        // Show the section
                        $fixedSection.show();

                        // Ensure content is visible (expanded)
                        var $content = $fixedSection.find('.sf-fixed-content');
                        $content.show();

                        // Re-bind event handlers
                        self.bindFixedSectionEvents();

                        // Scroll to Fixed section
                        setTimeout(function() {
                            $('html, body').animate({
                                scrollTop: $fixedSection.offset().top - 50
                            }, 500);
                        }, 400);
                    }
                }
            });
        },

        /**
         * Bind event handlers for Fixed section after AJAX refresh
         */
        bindFixedSectionEvents: function() {
            var self = this;

            // Toggle handler for collapsible section
            $('.sf-fixed-section .sf-fixed-toggle').off('click').on('click', function() {
                var $btn = $(this);
                var $content = $btn.closest('.sf-fixed-section').find('.sf-fixed-content');
                var isExpanded = $btn.attr('aria-expanded') === 'true';

                if (isExpanded) {
                    $content.slideUp(300);
                    $btn.attr('aria-expanded', 'false');
                    $btn.find('.sf-toggle-icon').removeClass('sf-rotated');
                } else {
                    $content.slideDown(300);
                    $btn.attr('aria-expanded', 'true');
                    $btn.find('.sf-toggle-icon').addClass('sf-rotated');
                }
            });

            // Export CSV handler
            $('#sf-export-fixed-redirects').off('click').on('click', function() {
                self.exportFixedResults();
            });

            // Expand sources toggle in fixed section
            $('.sf-fixed-section .sf-expand-sources').off('click').on('click', function() {
                var $btn = $(this);
                var $row = $btn.closest('tr');
                var $sourcesRow = $row.next('.sf-sources-row');
                var isExpanded = $btn.data('expanded');

                if (isExpanded) {
                    $sourcesRow.slideUp(200);
                    $btn.data('expanded', false);
                    $btn.find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                } else {
                    $sourcesRow.slideDown(200);
                    $btn.data('expanded', true);
                    $btn.find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                }
            });
        },

        /**
         * Add a fix result row to the Fixes Applied section
         */
        addFixResultRow: function(result) {
            var $section = $('#sf-fixed-redirects-section');
            var $tbody = $section.find('.sf-fixes-applied-table tbody');

            // Show the section if hidden
            if (!$section.is(':visible')) {
                $section.show();
                $section.find('.sf-fixed-content').show();
                $section.find('.sf-fixed-toggle').attr('aria-expanded', 'true');
                $section.find('.sf-toggle-icon').addClass('sf-rotated');
            }

            // If section doesn't exist, create it
            if ($section.length === 0) {
                var sectionHtml = '<div class="sf-results-section sf-fixed-section" id="sf-fixed-redirects-section">' +
                    '<div class="sf-results-header sf-results-header-collapsible sf-results-header-fixed">' +
                    '<button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="true">' +
                    '<span class="sf-section-badge sf-badge-fixed">&#10004;</span>' +
                    '<span class="sf-fixes-applied-title">Fixes Applied (<span class="sf-fixes-applied-count">0</span>)</span>' +
                    '<span class="sf-section-hint">View fix results and export to CSV</span>' +
                    '<span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>' +
                    '</button>' +
                    '<div class="sf-results-actions">' +
                    '<button type="button" class="sf-button sf-button-secondary" id="sf-export-fixed-redirects">' +
                    '<span class="dashicons dashicons-download"></span> Export CSV</button>' +
                    '</div></div>' +
                    '<div class="sf-fixed-content">' +
                    '<div class="sf-status-filter-row sf-rc-status-filters">' +
                    '<span class="sf-filter-label">Status:</span>' +
                    '<div class="sf-status-filters">' +
                    '<button type="button" class="sf-status-filter active" data-status="all"><span class="sf-status-icon">&#128203;</span><span class="sf-status-count sf-rc-status-count-all">0</span><span class="sf-status-label">All</span></button>' +
                    '<button type="button" class="sf-status-filter sf-status-success" data-status="success"><span class="sf-status-icon">&#9989;</span><span class="sf-status-count sf-rc-status-count-success">0</span><span class="sf-status-label">Fixed</span></button>' +
                    '<button type="button" class="sf-status-filter sf-status-failed" data-status="failed"><span class="sf-status-icon">&#10060;</span><span class="sf-status-count sf-rc-status-count-failed">0</span><span class="sf-status-label">Failed</span></button>' +
                    '<button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped"><span class="sf-status-icon">&#9203;</span><span class="sf-status-count sf-rc-status-count-skipped">0</span><span class="sf-status-label">Skipped</span></button>' +
                    '</div></div>' +
                    '<div class="sf-results-table-wrapper">' +
                    '<table class="sf-results-table sf-fixed-table sf-fixes-applied-table"><thead><tr>' +
                    '<th class="sf-col-was">Original URL</th>' +
                    '<th class="sf-col-now">Updated To</th>' +
                    '<th class="sf-col-status">Status</th>' +
                    '<th class="sf-col-count">Found In</th>' +
                    '<th class="sf-col-when">Date Applied</th>' +
                    '</tr></thead><tbody id="sf-fixes-applied-tbody"></tbody></table>' +
                    '</div></div></div>';

                var $lastSection = $('.sf-results-section').last();
                if ($lastSection.length) {
                    $lastSection.after(sectionHtml);
                } else {
                    $('.sf-module-content').append(sectionHtml);
                }

                $section = $('#sf-fixed-redirects-section');
                $tbody = $section.find('.sf-fixes-applied-table tbody');
            }

            // Build status display
            var statusIcon, statusLabel, rowClass;
            if (result.status === 'success') {
                statusIcon = '&#9989;';
                statusLabel = 'Fixed';
                rowClass = 'sf-fixed-row-success';
            } else if (result.status === 'failed') {
                statusIcon = '&#10060;';
                statusLabel = 'Failed';
                rowClass = 'sf-fixed-row-failed';
            } else {
                statusIcon = '&#9203;';
                statusLabel = 'Skipped';
                rowClass = 'sf-fixed-row-skipped';
            }

            var statusTooltip = '';
            if (result.status_message && result.status !== 'success') {
                statusTooltip = ' title="' + this.escapeHtml(result.status_message) + '"';
            }

            var sourceCount = result.source_count || 0;
            var sourceLabel = sourceCount === 1 ? 'page' : 'pages';
            var sources = result.sources || [];

            // Build sources sub-table HTML
            var sourcesRowHtml = '';
            if (sources.length > 0) {
                var sourcesTableRows = '';
                for (var i = 0; i < sources.length; i++) {
                    var src = sources[i];
                    var pageCell = '';
                    if (src.post_title) {
                        pageCell = '<a href="' + this.escapeHtml(src.source_url || '') + '" target="_blank" class="sf-source-title-link">' + this.escapeHtml(src.post_title) + '</a>';
                    } else if (src.source_url) {
                        pageCell = '<a href="' + this.escapeHtml(src.source_url) + '" target="_blank" class="sf-source-url-link">' + this.escapeHtml(src.source_url) + '</a>';
                    } else {
                        pageCell = '<span>-</span>';
                    }
                    var editCell = src.edit_url ? '<a href="' + this.escapeHtml(src.edit_url) + '" target="_blank" class="sf-edit-link"><span class="dashicons dashicons-edit"></span></a>' : '';
                    sourcesTableRows += '<tr data-post-id="' + (src.post_id || '') + '">' +
                        '<td>' + pageCell + '</td>' +
                        '<td><span class="sf-location-badge">' + this.escapeHtml((src.location || 'content').charAt(0).toUpperCase() + (src.location || 'content').slice(1)) + '</span></td>' +
                        '<td>' + editCell + '</td>' +
                        '</tr>';
                }
                sourcesRowHtml = '<tr class="sf-sources-row" style="display: none;"><td colspan="5"><div class="sf-sources-list">' +
                    '<table class="sf-sources-table"><thead><tr><th>Page</th><th>Location</th><th>Edit</th></tr></thead>' +
                    '<tbody>' + sourcesTableRows + '</tbody></table></div></td></tr>';
            }

            var rowHtml = '<tr class="sf-fixed-row ' + rowClass + '" data-status="' + result.status + '">' +
                '<td class="sf-col-was"><div class="sf-url-cell">' +
                '<span class="sf-url-text sf-url-strikethrough" title="' + this.escapeHtml(result.address) + '">' + this.escapeHtml(this.truncateUrl(result.address, 50)) + '</span>' +
                '<a href="' + this.escapeHtml(result.address) + '" target="_blank" class="sf-url-link"><span class="dashicons dashicons-external"></span></a>' +
                '</div></td>' +
                '<td class="sf-col-now"><div class="sf-url-cell">' +
                '<span class="sf-url-text sf-final-url" title="' + this.escapeHtml(result.final_address) + '">' + this.escapeHtml(this.truncateUrl(result.final_address, 50)) + '</span>' +
                '<a href="' + this.escapeHtml(result.final_address) + '" target="_blank" class="sf-url-link"><span class="dashicons dashicons-external"></span></a>' +
                '</div></td>' +
                '<td class="sf-col-status"><span class="sf-status-badge sf-status-' + result.status + '"' + statusTooltip + '><span class="sf-status-icon">' + statusIcon + '</span> ' + statusLabel + '</span></td>' +
                '<td class="sf-col-count"><button type="button" class="sf-expand-sources" data-expanded="false" data-redirect-url="' + this.escapeHtml(result.address) + '">' +
                '<span class="sf-source-count">' + sourceCount + '</span> <span class="sf-source-label">' + sourceLabel + '</span> <span class="dashicons dashicons-arrow-down-alt2"></span></button></td>' +
                '<td class="sf-col-when"><span class="sf-fixed-time">Just now</span></td>' +
                '</tr>';

            $tbody.prepend(rowHtml + sourcesRowHtml);

            // Update client pagination
            this.clientPagination['fixes-applied'].totalItems++;
            this.resetClientPagination('fixes-applied');
        },

        /**
         * Update status filter counts in the Fixes Applied section
         */
        updateFixStatusCounts: function() {
            var $section = $('#sf-fixed-redirects-section');
            var $rows = $section.find('.sf-fixes-applied-table tbody .sf-fixed-row');

            var allCount = $rows.length;
            var successCount = $rows.filter('[data-status="success"]').length;
            var failedCount = $rows.filter('[data-status="failed"]').length;
            var skippedCount = $rows.filter('[data-status="skipped"]').length;

            $section.find('.sf-rc-status-count-all').text(allCount);
            $section.find('.sf-rc-status-count-success').text(successCount);
            $section.find('.sf-rc-status-count-failed').text(failedCount);
            $section.find('.sf-rc-status-count-skipped').text(skippedCount);

            // Update header count
            $section.find('.sf-fixes-applied-count').text(allCount);
        },

        // ========================================
        // CLIENT-SIDE PAGINATION (Fixes Applied)
        // ========================================

        initClientPagination: function() {
            var self = this;
            var $section = $('#sf-fixed-redirects-section');
            if (!$section.length) return;

            var $table = $section.find('.sf-paginated-table[data-section="fixes-applied"]');
            if (!$table.length) return;

            var rowCount = $table.find('.sf-fixed-row').length;
            self.clientPagination['fixes-applied'].totalItems = rowCount;

            if (rowCount > self.clientPagination['fixes-applied'].perPage) {
                self.applyClientPagination('fixes-applied');
                self.renderClientPageNumbers('fixes-applied');
                self.updateClientPaginationInfo('fixes-applied');
                $section.find('.sf-pagination[data-section="fixes-applied"]').show();
            } else {
                $section.find('.sf-pagination[data-section="fixes-applied"]').hide();
            }
        },

        goToClientPage: function(section, page) {
            this.clientPagination[section].currentPage = page;
            this.applyClientPagination(section);
            this.renderClientPageNumbers(section);
            this.updateClientPaginationInfo(section);

            var $section = $('#sf-fixed-redirects-section');
            $section[0].scrollIntoView({ behavior: 'smooth', block: 'start' });
        },

        applyClientPagination: function(section) {
            var pagination = this.clientPagination[section];
            var $container = $('#sf-fixed-redirects-section').find('.sf-paginated-table[data-section="' + section + '"]');
            var $rows = $container.find('.sf-fixed-row').not('.sf-filtered-out');

            // Hide all data rows first
            $container.find('.sf-fixed-row').addClass('sf-page-hidden');
            $container.find('.sf-sources-row').hide();

            // Show rows for current page
            var start = (pagination.currentPage - 1) * pagination.perPage;
            var end = start + pagination.perPage;

            $rows.each(function(index) {
                if (index >= start && index < end) {
                    $(this).removeClass('sf-page-hidden');
                }
            });
        },

        getClientVisibleRowCount: function(section) {
            var $container = $('#sf-fixed-redirects-section').find('.sf-paginated-table[data-section="' + section + '"]');
            return $container.find('.sf-fixed-row').not('.sf-filtered-out').length;
        },

        renderClientPageNumbers: function(section) {
            var pagination = this.clientPagination[section];
            var visibleCount = this.getClientVisibleRowCount(section);
            var totalPages = Math.ceil(visibleCount / pagination.perPage);
            var currentPage = pagination.currentPage;
            var $container = $('#sf-fixed-redirects-section').find('.sf-pagination[data-section="' + section + '"]');
            var $numbers = $container.find('.sf-page-numbers');

            $numbers.empty();

            if (totalPages <= 1) {
                $container.find('.sf-page-prev, .sf-page-next').prop('disabled', true);
                return;
            }

            var maxVisible = 7;
            var startPage = Math.max(1, currentPage - 3);
            var endPage = Math.min(totalPages, startPage + maxVisible - 1);
            if (endPage - startPage < maxVisible - 1) {
                startPage = Math.max(1, endPage - maxVisible + 1);
            }

            if (startPage > 1) {
                $numbers.append(this.renderClientPageButton(1, currentPage));
                if (startPage > 2) {
                    $numbers.append('<span class="sf-page-ellipsis">...</span>');
                }
            }

            for (var i = startPage; i <= endPage; i++) {
                $numbers.append(this.renderClientPageButton(i, currentPage));
            }

            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    $numbers.append('<span class="sf-page-ellipsis">...</span>');
                }
                $numbers.append(this.renderClientPageButton(totalPages, currentPage));
            }

            $container.find('.sf-page-prev').prop('disabled', currentPage <= 1);
            $container.find('.sf-page-next').prop('disabled', currentPage >= totalPages);
        },

        renderClientPageButton: function(page, currentPage) {
            var activeClass = page === currentPage ? ' active' : '';
            return '<button type="button" class="sf-page-btn sf-page-number' + activeClass + '" data-page="' + page + '">' + page + '</button>';
        },

        updateClientPaginationInfo: function(section) {
            var pagination = this.clientPagination[section];
            var visibleCount = this.getClientVisibleRowCount(section);
            var $container = $('#sf-fixed-redirects-section').find('.sf-pagination[data-section="' + section + '"]');

            if (visibleCount === 0) {
                $container.hide();
                return;
            }

            var start = (pagination.currentPage - 1) * pagination.perPage + 1;
            var end = Math.min(pagination.currentPage * pagination.perPage, visibleCount);

            $container.find('.sf-page-start').text(start);
            $container.find('.sf-page-end').text(end);
            $container.find('.sf-page-total').text(visibleCount);

            if (visibleCount > pagination.perPage) {
                $container.show();
            } else {
                $container.hide();
            }
        },

        resetClientPagination: function(section) {
            this.clientPagination[section].currentPage = 1;
            this.clientPagination[section].totalItems = this.getClientVisibleRowCount(section);
            this.applyClientPagination(section);
            this.renderClientPageNumbers(section);
            this.updateClientPaginationInfo(section);
        },

        /**
         * Filter fix results table by status
         */
        filterFixResults: function(status) {
            var $section = $('#sf-fixed-redirects-section');
            var $rows = $section.find('.sf-fixes-applied-table tbody .sf-fixed-row');

            // Hide all sources rows when filtering
            $section.find('.sf-sources-row').hide();
            // Reset expand buttons
            $section.find('.sf-expand-sources').attr('data-expanded', 'false')
                .find('.dashicons').removeClass('sf-rotated');

            if (status === 'all') {
                $rows.removeClass('sf-filtered-out');
            } else {
                $rows.addClass('sf-filtered-out');
                $rows.filter('[data-status="' + status + '"]').removeClass('sf-filtered-out');
            }

            this.resetClientPagination('fixes-applied');
        },

        // ========================================
        // BULK UPLOAD METHODS
        // ========================================

        /**
         * Handle bulk confirm button click
         */
        handleBulkConfirm: function() {
            var self = this;
            var $btn = $('#sf-bulk-confirm');
            var totalCount = parseInt($btn.data('count'), 10) || 0;

            if (totalCount === 0) {
                self.showNotice('warning', sfRedirectChainsData.i18n.noFixesSelected || 'No fixes to apply');
                return;
            }

            // Show large file warning if needed
            if (totalCount > 500) {
                var warningMsg = sfRedirectChainsData.i18n.bulkLargeFileWarning
                    ? sfRedirectChainsData.i18n.bulkLargeFileWarning.replace('%d', totalCount)
                    : 'Large file detected (' + totalCount + ' fixes). For best performance, we recommend splitting into batches of 500 or fewer.';

                var continueAnyway = confirm(warningMsg + '\n\n' + (sfRedirectChainsData.i18n.bulkContinueAnyway || 'Continue Anyway?'));
                if (!continueAnyway) {
                    return;
                }
            }

            // Show progress modal
            self.showBulkProgressModal(totalCount);

            // Start batch processing
            self.processBulkFixBatch(0, 50, totalCount);
        },

        /**
         * Show the bulk progress modal
         */
        showBulkProgressModal: function(total) {
            var $modal = $('#sf-bulk-progress-modal');
            $('#sf-bulk-progress-fill').css('width', '0%');
            $('#sf-bulk-progress-current').text('0');
            $('#sf-bulk-progress-total').text(total);
            $('#sf-bulk-progress-url').text(sfRedirectChainsData.i18n.bulkProcessing || 'Processing...');
            $modal.fadeIn(200);
        },

        /**
         * Update the bulk progress modal
         */
        updateBulkProgress: function(current, total) {
            var percent = Math.round((current / total) * 100);
            $('#sf-bulk-progress-fill').css('width', percent + '%');
            $('#sf-bulk-progress-current').text(current);
        },

        /**
         * Hide the bulk progress modal
         */
        hideBulkProgressModal: function() {
            $('#sf-bulk-progress-modal').fadeOut(200);
        },

        /**
         * Process a batch of bulk fixes
         */
        processBulkFixBatch: function(offset, batchSize, totalCount) {
            var self = this;

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_apply_bulk_fixes',
                    nonce: sfRedirectChainsData.nonce,
                    offset: offset,
                    batch_size: batchSize
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        var processed = Math.min(offset + data.processed, totalCount);

                        // Update progress
                        self.updateBulkProgress(processed, totalCount);
                        $('#sf-bulk-progress-url').text(sfRedirectChainsData.i18n.bulkUpdating || 'Applying redirect fixes...');

                        if (data.complete) {
                            // All done! Hide modal and reload
                            self.hideBulkProgressModal();

                            var successCount = data.total_success || 0;
                            var failedCount = data.total_failed || 0;

                            if (failedCount > 0) {
                                var partialMsg = sfRedirectChainsData.i18n.bulkPartialComplete || '%d fixed successfully. %d failed - see details below.';
                                self.showNotice('warning', partialMsg.replace('%d', successCount).replace('%d', failedCount));
                            } else {
                                var successMsg = sfRedirectChainsData.i18n.bulkComplete || '%d redirects fixed successfully.';
                                self.showNotice('success', successMsg.replace('%d', successCount));
                            }

                            // Reload to show results
                            window.location.reload();
                        } else {
                            // Process next batch
                            self.processBulkFixBatch(data.next_offset, batchSize, totalCount);
                        }
                    } else {
                        self.hideBulkProgressModal();
                        self.showNotice('error', response.data.message || 'Bulk fix failed');
                    }
                },
                error: function() {
                    self.hideBulkProgressModal();
                    self.showNotice('error', sfRedirectChainsData.i18n.fixesFailed || 'Bulk fix failed');
                }
            });
        },

        /**
         * Handle bulk clear button click
         */
        handleBulkClear: function() {
            var self = this;

            $.ajax({
                url: sfRedirectChainsData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'sf_redirect_chains_cancel_bulk',
                    nonce: sfRedirectChainsData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.showNotice('success', response.data.message || 'Cleared');
                        window.location.reload();
                    } else {
                        self.showNotice('error', response.data.message || 'Failed to clear');
                    }
                },
                error: function() {
                    self.showNotice('error', 'Failed to clear');
                }
            });
        },

        /**
         * Handle bulk download preview CSV
         */
        handleBulkDownloadPreview: function() {
            // Create a form and submit it to trigger the download
            var $form = $('<form>', {
                method: 'POST',
                action: sfRedirectChainsData.ajaxUrl
            });

            $form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_redirect_chains_download_preview' }));
            $form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfRedirectChainsData.nonce }));

            $form.appendTo('body').submit().remove();
        },

        /**
         * Handle bulk download results CSV
         */
        handleBulkDownloadResults: function() {
            var $form = $('<form>', {
                method: 'POST',
                action: sfRedirectChainsData.ajaxUrl
            });

            $form.append($('<input>', { type: 'hidden', name: 'action', value: 'sf_redirect_chains_download_results' }));
            $form.append($('<input>', { type: 'hidden', name: 'nonce', value: sfRedirectChainsData.nonce }));

            $form.appendTo('body').submit().remove();
        },

        showNotice: function(type, message) {
            var $notice = $('<div class="sf-notice sf-notice-' + type + '" role="alert">' +
                '<p>' + message + '</p>' +
                '<button type="button" class="sf-notice-dismiss"><span class="dashicons dashicons-dismiss"></span></button>' +
                '</div>');

            // Remove existing notices
            $('.sf-module-content > .sf-notice').remove();

            // Add new notice at top of module content
            $('.sf-module-content').prepend($notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);

            // Manual dismiss
            $notice.find('.sf-notice-dismiss').on('click', function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            });
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        if ($('#sf-redirect-chains').length) {
            RedirectChains.init();
        }
    });

})(jQuery);
