<?php
/**
 * Redirect Chains Module for Screaming Fixes
 *
 * Finds and fixes redirect chains from Screaming Frog exports
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Redirect_Chains extends SF_Module {

    /**
     * Module ID
     * @var string
     */
    protected $module_id = 'redirect-chains';

    /**
     * Maximum number of rows allowed in CSV upload
     */
    const MAX_CSV_ROWS = 5000;

    /**
     * Constructor
     */
    public function __construct() {
        $this->name = __('Redirect Chains', 'screaming-fixes');
        $this->slug = 'redirect-chains';
        $this->description = __('Find and fix redirect chains, loops & temp redirects.', 'screaming-fixes');

        parent::__construct();
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register AJAX handlers
        add_action('wp_ajax_sf_redirect_chains_process_csv', [$this, 'ajax_process_csv']);
        add_action('wp_ajax_sf_redirect_chains_apply_fixes', [$this, 'ajax_apply_fixes']);
        add_action('wp_ajax_sf_redirect_chains_export', [$this, 'ajax_export']);
        add_action('wp_ajax_sf_redirect_chains_export_fixed', [$this, 'ajax_export_fixed']);
        add_action('wp_ajax_sf_redirect_chains_get_data', [$this, 'ajax_get_data']);
        add_action('wp_ajax_sf_redirect_chains_clear_data', [$this, 'ajax_clear_data']);
        add_action('wp_ajax_sf_redirect_chains_get_fixed_section', [$this, 'ajax_get_fixed_section']);
        add_action('wp_ajax_sf_redirect_chains_fix_temp_redirect', [$this, 'ajax_fix_temp_redirect']);
        add_action('wp_ajax_sf_redirect_chains_fix_redirect_loop', [$this, 'ajax_fix_redirect_loop']);

        // Bulk upload AJAX handlers
        add_action('wp_ajax_sf_redirect_chains_apply_bulk_fixes', [$this, 'ajax_apply_bulk_fixes']);
        add_action('wp_ajax_sf_redirect_chains_cancel_bulk', [$this, 'ajax_cancel_bulk']);
        add_action('wp_ajax_sf_redirect_chains_download_preview', [$this, 'ajax_download_preview']);
        add_action('wp_ajax_sf_redirect_chains_download_results', [$this, 'ajax_download_results']);

        // Enqueue module assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue module-specific assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'screaming-fixes') === false) {
            return;
        }

        // Check if we're on the redirect chains tab
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        if ($current_tab !== 'redirect-chains') {
            return;
        }

        // Use file modification time for cache busting
        $css_file = SF_PLUGIN_DIR . 'modules/redirect-chains/assets/redirect-chains.css';
        $js_file = SF_PLUGIN_DIR . 'modules/redirect-chains/assets/redirect-chains.js';
        $css_version = file_exists($css_file) ? SF_VERSION . '.' . filemtime($css_file) : SF_VERSION;
        $js_version = file_exists($js_file) ? SF_VERSION . '.' . filemtime($js_file) : SF_VERSION;

        wp_enqueue_style(
            'sf-redirect-chains',
            SF_PLUGIN_URL . 'modules/redirect-chains/assets/redirect-chains.css',
            ['screaming-fixes-admin'],
            $css_version
        );

        wp_enqueue_script(
            'sf-redirect-chains',
            SF_PLUGIN_URL . 'modules/redirect-chains/assets/redirect-chains.js',
            ['jquery', 'screaming-fixes-admin'],
            $js_version,
            true
        );

        wp_localize_script('sf-redirect-chains', 'sfRedirectChainsData', [
            'nonce' => wp_create_nonce('sf_redirect_chains_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'i18n' => [
                'processing' => __('Processing CSV...', 'screaming-fixes'),
                'scanningPosts' => __('Scanning posts...', 'screaming-fixes'),
                'applyingFixes' => __('Applying redirect fixes...', 'screaming-fixes'),
                'fixesApplied' => __('Redirect fixes applied successfully!', 'screaming-fixes'),
                'fixesFailed' => __('Some fixes failed. Check the results.', 'screaming-fixes'),
                'noFixesSelected' => __('No fixes selected.', 'screaming-fixes'),
                'confirmApply' => __('Apply fixes to %d redirect issues?', 'screaming-fixes'),
                'exporting' => __('Exporting...', 'screaming-fixes'),
                'exportComplete' => __('Export complete.', 'screaming-fixes'),
                'confirmClear' => __('Are you sure you want to clear all redirect data? This will allow you to upload a new CSV file.', 'screaming-fixes'),
                'dataCleared' => __('Data cleared successfully.', 'screaming-fixes'),
                'fixed' => __('fixed', 'screaming-fixes'),
                'fixedLabel' => __('Fixed', 'screaming-fixes'),
                // Bulk upload i18n
                'bulkProcessing' => __('Processing bulk fix CSV...', 'screaming-fixes'),
                'bulkReady' => __('%d redirect fixes ready to apply', 'screaming-fixes'),
                'bulkUpdating' => __('Applying redirect fixes...', 'screaming-fixes'),
                'bulkComplete' => __('%d redirects fixed successfully.', 'screaming-fixes'),
                'bulkPartialComplete' => __('%d fixed successfully. %d failed - see details below.', 'screaming-fixes'),
                'bulkLargeFileWarning' => __('Large file detected (%d fixes). For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'),
                'bulkContinueAnyway' => __('Continue Anyway', 'screaming-fixes'),
                'bulkNoActionColumn' => __('This appears to be a bulk fix CSV but is missing the Action column.', 'screaming-fixes'),
            ],
        ]);
    }

    /**
     * Run the module scan
     *
     * @return array Scan results
     */
    public function run_scan() {
        // This module doesn't do active scanning - it processes uploaded CSVs
        return $this->get_results();
    }

    /**
     * Check if this module can handle a CSV based on headers
     *
     * @param array $headers CSV column headers (lowercase)
     * @return bool
     */
    public function can_handle_csv($headers) {
        // Normalize headers to lowercase
        $headers = array_map('strtolower', array_map('trim', $headers));

        // Screaming Frog "Redirect Chains" export
        // Must have: Final Address, Number of Redirects, Chain Type
        $has_final_address = in_array('final address', $headers);
        $has_num_redirects = in_array('number of redirects', $headers);
        $has_chain_type = in_array('chain type', $headers);

        return $has_final_address && $has_num_redirects && $has_chain_type;
    }

    /**
     * Maximum sources to store per redirect URL (for display)
     * Additional sources are tracked but not loaded initially
     */
    const MAX_SOURCES_PER_REDIRECT = 50;

    /**
     * Process uploaded CSV file
     *
     * @param string $file_path Path to uploaded CSV
     * @return array|WP_Error Processed results or error
     */
    public function process_csv($file_path) {
        // Increase memory limit for large CSVs
        @ini_set('memory_limit', '512M');

        $parser = new SF_CSV_Parser();

        // Parse the CSV - only extract needed columns for memory efficiency
        $parsed = $parser->parse($file_path, [
            'extract_columns' => [
                'address', 'final address', 'source', 'number of redirects',
                'chain type', 'anchor text', 'link position', 'status code',
                'loop', 'temp redirect in chain', 'redirect type 1', 'status code 1',
            ],
        ]);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Check row count and enforce limit
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    /* translators: 1: number of rows in uploaded file, 2: maximum allowed rows */
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches of %2$s rows or fewer and upload each batch separately. Tip: In Screaming Frog, you can filter by redirect type (301, 302) or export only internal redirects to create smaller files.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Standardize column names
        $parsed = $this->standardize_redirect_columns($parsed);

        // Get site domain for internal/external classification
        $site_domain = wp_parse_url(home_url(), PHP_URL_HOST);

        // Group by redirect URL FIRST to reduce memory usage
        // Instead of storing all rows, group immediately
        $grouped = [];
        $row_count = 0;

        foreach ($parsed['rows'] as $row) {
            $address = $row['address'] ?? '';
            $final_address = $row['final_address'] ?? '';
            $source_url = $row['source'] ?? '';

            // Skip if no address or final address
            if (empty($address) || empty($final_address)) {
                continue;
            }

            // Skip if address equals final address (no actual redirect)
            if ($address === $final_address) {
                continue;
            }

            $row_count++;

            // Create group key
            $key = md5($address);

            if (!isset($grouped[$key])) {
                $num_redirects = isset($row['number_of_redirects']) ? intval($row['number_of_redirects']) : 1;
                $chain_type = $row['chain_type'] ?? '';
                $status_code = isset($row['status_code_1']) ? intval($row['status_code_1']) : 301;
                $redirect_type_1 = $row['redirect_type_1'] ?? '';
                $is_loop = isset($row['loop']) && strtolower(trim($row['loop'])) === 'true';
                $has_temp_redirect = isset($row['temp_redirect']) && strtolower(trim($row['temp_redirect'])) === 'true';

                // Classify redirect type
                $redirect_type = $this->classify_redirect($address, $final_address, $site_domain);

                $grouped[$key] = [
                    'address' => $address,
                    'final_address' => $final_address,
                    'num_redirects' => $num_redirects,
                    'chain_type' => $chain_type,
                    'status_code' => $status_code,
                    'redirect_type_1' => $redirect_type_1,
                    'is_loop' => $is_loop,
                    'has_temp_redirect' => $has_temp_redirect,
                    'redirect_type' => $redirect_type,
                    'sources' => [],
                    'source_count' => 0,
                    'fixable_count' => 0,
                    'manual_count' => 0,
                    'skip_count' => 0,
                    'selected' => false,
                ];
            }

            // Categorize based on source URL and link position
            $link_position = $row['link_position'] ?? '';
            $categorization = $this->categorize_redirect($source_url, $link_position);

            // Track source count
            $grouped[$key]['source_count']++;

            // Count by category
            switch ($categorization['category']) {
                case 'fixable':
                    $grouped[$key]['fixable_count']++;
                    break;
                case 'manual':
                    $grouped[$key]['manual_count']++;
                    break;
                case 'skip':
                    $grouped[$key]['skip_count']++;
                    break;
            }

            // Only store limited sources for display (to save memory)
            // Full source data is available via AJAX when needed
            $stored_sources = count($grouped[$key]['sources']);
            if ($stored_sources < self::MAX_SOURCES_PER_REDIRECT) {
                $post_id = $categorization['post_id'];
                $anchor_text = $row['anchor_text'] ?? '';

                $source_key = md5($source_url . '_' . $link_position);
                if (!isset($grouped[$key]['sources'][$source_key])) {
                    $grouped[$key]['sources'][$source_key] = [
                        'post_id' => $post_id,
                        'post_title' => $post_id ? get_the_title($post_id) : '',
                        'source_url' => $source_url,
                        'edit_url' => $post_id ? get_edit_post_link($post_id, 'raw') : '',
                        'anchor_text' => $anchor_text,
                        'link_position' => $link_position,
                        'fix_category' => $categorization['category'],
                        'fix_note' => $categorization['note'],
                        'location' => $categorization['location'],
                    ];
                }
            }

            // Periodic cleanup for memory
            if ($row_count % 2000 === 0) {
                gc_collect_cycles();
            }
        }

        // Free memory from parsed data
        unset($parsed);
        gc_collect_cycles();

        // Now categorize grouped redirects
        $fixable_redirects = [];
        $manual_redirects = [];
        $skipped_redirects = [];

        foreach ($grouped as $key => $data) {
            // Convert sources from associative to indexed array
            $data['sources'] = array_values($data['sources']);

            // Categorize sources
            $fixable_sources = [];
            $manual_sources = [];
            $skip_sources = [];

            foreach ($data['sources'] as $source) {
                switch ($source['fix_category']) {
                    case 'fixable':
                        $fixable_sources[] = $source;
                        break;
                    case 'manual':
                        $manual_sources[] = $source;
                        break;
                    case 'skip':
                        $skip_sources[] = $source;
                        break;
                }
            }

            // Store categorized sources (limited for display)
            $data['fixable_sources'] = $fixable_sources;
            $data['manual_sources'] = $manual_sources;
            $data['skip_sources'] = $skip_sources;

            // Note: fixable_count/manual_count/skip_count are already set from full CSV scan

            // Determine overall category based on counts (not just stored sources)
            if ($data['fixable_count'] > 0) {
                $data['overall_category'] = 'fixable';
                $fixable_redirects[] = $data;
            } elseif ($data['manual_count'] > 0) {
                $data['overall_category'] = 'manual';
                $manual_redirects[] = $data;
            } else {
                $data['overall_category'] = 'skip';
                $skipped_redirects[] = $data;
            }
        }

        // Free grouped memory
        unset($grouped);

        // Calculate totals from actual counts
        $total_fixable = 0;
        $total_manual = 0;
        $total_skipped = 0;

        foreach ($fixable_redirects as $redirect) {
            $total_fixable += $redirect['fixable_count'];
        }
        foreach ($manual_redirects as $redirect) {
            $total_manual += $redirect['manual_count'];
        }
        foreach ($skipped_redirects as $redirect) {
            $total_skipped += $redirect['skip_count'];
        }

        $results = [
            'redirects' => array_merge($fixable_redirects, $manual_redirects, $skipped_redirects),
            'fixable_redirects' => $fixable_redirects,
            'manual_redirects' => $manual_redirects,
            'skipped_redirects' => $skipped_redirects,
            'total_count' => count($fixable_redirects) + count($manual_redirects) + count($skipped_redirects),
            'total_sources' => $total_fixable + $total_manual + $total_skipped,
            'fixable_count' => count($fixable_redirects),
            'manual_count' => count($manual_redirects),
            'skipped_count' => count($skipped_redirects),
            'fixable_sources' => $total_fixable,
            'manual_sources' => $total_manual,
            'skipped_sources' => $total_skipped,
            'csv_row_count' => $row_count,
            'processed_at' => current_time('mysql'),
        ];

        // Save results
        $this->save_results($results);

        // Also save to uploads table for persistence
        $this->save_upload_data($results);

        return $results;
    }

    /**
     * Standardize column names for redirect chains CSV
     *
     * @param array $parsed Parsed CSV data
     * @return array Standardized data
     */
    private function standardize_redirect_columns($parsed) {
        $column_map = [
            'address' => ['address', 'url', 'redirect url'],
            'final_address' => ['final address', 'final url', 'destination'],
            'source' => ['source', 'from', 'page url', 'source url'],
            'number_of_redirects' => ['number of redirects', 'redirects', 'hops'],
            'chain_type' => ['chain type', 'type'],
            'anchor_text' => ['anchor text', 'anchor', 'link text'],
            'link_position' => ['link position', 'position', 'location'],
            'status_code_1' => ['status code 1', 'status code', 'status'],
            'redirect_type_1' => ['redirect type 1', 'redirect type'],
            'loop' => ['loop'],
            'temp_redirect' => ['temp redirect in chain', 'temp redirect'],
        ];

        // Create a header map
        $header_map = [];
        foreach ($parsed['headers'] as $index => $header) {
            $normalized = strtolower(trim($header));
            foreach ($column_map as $standard => $variants) {
                if (in_array($normalized, $variants)) {
                    $header_map[$index] = $standard;
                    break;
                }
            }
            if (!isset($header_map[$index])) {
                $header_map[$index] = $normalized;
            }
        }

        // Remap rows
        $remapped_rows = [];
        foreach ($parsed['rows'] as $row) {
            $new_row = [];
            foreach ($row as $key => $value) {
                if (is_numeric($key) && isset($header_map[$key])) {
                    $new_row[$header_map[$key]] = $value;
                } else {
                    // Try to find the standard name
                    $normalized_key = strtolower(trim($key));
                    $found = false;
                    foreach ($column_map as $standard => $variants) {
                        if (in_array($normalized_key, $variants) || $normalized_key === $standard) {
                            $new_row[$standard] = $value;
                            $found = true;
                            break;
                        }
                    }
                    if (!$found) {
                        $new_row[$key] = $value;
                    }
                }
            }
            $remapped_rows[] = $new_row;
        }

        $parsed['rows'] = $remapped_rows;
        return $parsed;
    }

    /**
     * Classify redirect type (internal, external, social, etc.)
     *
     * @param string $address Original URL
     * @param string $final_address Final destination URL
     * @param string $site_domain Site's domain
     * @return array Classification info
     */
    private function classify_redirect($address, $final_address, $site_domain) {
        $address_domain = wp_parse_url($address, PHP_URL_HOST);
        $final_domain = wp_parse_url($final_address, PHP_URL_HOST);

        // Check for internal redirects (your domain to your domain)
        $is_address_internal = $this->is_same_domain($address_domain, $site_domain);
        $is_final_internal = $this->is_same_domain($final_domain, $site_domain);

        if ($is_address_internal && $is_final_internal) {
            return [
                'type' => 'internal',
                'label' => __('Internal', 'screaming-fixes'),
                'class' => 'sf-type-internal',
                'priority' => 1, // High priority to fix
            ];
        }

        // Social media redirects
        $social_domains = [
            'twitter.com' => 'x.com',
            't.co' => 'x.com',
            'facebook.com' => 'facebook.com',
            'fb.com' => 'facebook.com',
            'instagram.com' => 'instagram.com',
            'linkedin.com' => 'linkedin.com',
            'youtube.com' => 'youtube.com',
            'youtu.be' => 'youtube.com',
            'pinterest.com' => 'pinterest.com',
            'pin.it' => 'pinterest.com',
        ];

        foreach ($social_domains as $old => $new) {
            if ($this->is_same_domain($address_domain, $old)) {
                return [
                    'type' => 'social',
                    'label' => __('Social', 'screaming-fixes'),
                    'class' => 'sf-type-social',
                    'priority' => 3, // Low priority - often fine to leave
                ];
            }
        }

        // Check for HTTPS upgrades
        $address_scheme = wp_parse_url($address, PHP_URL_SCHEME);
        $final_scheme = wp_parse_url($final_address, PHP_URL_SCHEME);

        if ($address_scheme === 'http' && $final_scheme === 'https' &&
            $this->is_same_domain($address_domain, $final_domain)) {
            return [
                'type' => 'https_upgrade',
                'label' => __('HTTPS', 'screaming-fixes'),
                'class' => 'sf-type-https',
                'priority' => 2, // Medium priority
            ];
        }

        // Check for WWW normalization
        if ($this->is_www_redirect($address_domain, $final_domain)) {
            return [
                'type' => 'www',
                'label' => __('WWW', 'screaming-fixes'),
                'class' => 'sf-type-www',
                'priority' => 2, // Medium priority
            ];
        }

        // External to external
        return [
            'type' => 'external',
            'label' => __('External', 'screaming-fixes'),
            'class' => 'sf-type-external',
            'priority' => 2, // Medium priority
        ];
    }

    /**
     * Check if two domains are the same (ignoring www)
     *
     * @param string $domain1 First domain
     * @param string $domain2 Second domain
     * @return bool
     */
    private function is_same_domain($domain1, $domain2) {
        $domain1 = preg_replace('/^www\./', '', strtolower($domain1));
        $domain2 = preg_replace('/^www\./', '', strtolower($domain2));
        return $domain1 === $domain2;
    }

    /**
     * Check if redirect is a WWW normalization
     *
     * @param string $from_domain Source domain
     * @param string $to_domain Destination domain
     * @return bool
     */
    private function is_www_redirect($from_domain, $to_domain) {
        $from_www = strpos(strtolower($from_domain), 'www.') === 0;
        $to_www = strpos(strtolower($to_domain), 'www.') === 0;

        if ($from_www !== $to_www) {
            $from_clean = preg_replace('/^www\./', '', strtolower($from_domain));
            $to_clean = preg_replace('/^www\./', '', strtolower($to_domain));
            return $from_clean === $to_clean;
        }

        return false;
    }

    /**
     * Categorize a redirect based on its source URL and link position
     *
     * @param string $source_url The source URL where the redirect link was found
     * @param string $link_position The Link Position from Screaming Frog CSV
     * @return array Category info
     */
    private function categorize_redirect($source_url, $link_position = '') {
        $position = strtolower(trim($link_position));
        $path = wp_parse_url($source_url, PHP_URL_PATH);
        $path = $path ?: '/';

        // First check Link Position from Screaming Frog
        if (!empty($position)) {
            // Navigation links - manual fix
            if (in_array($position, ['navigation', 'nav', 'menu'])) {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Navigation menu - fix in Appearance → Menus', 'screaming-fixes'),
                    'location' => 'navigation',
                ];
            }

            // Footer links - manual fix
            if ($position === 'footer') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Footer - fix in Appearance → Widgets or theme settings', 'screaming-fixes'),
                    'location' => 'footer',
                ];
            }

            // Sidebar links - manual fix
            if ($position === 'sidebar') {
                return [
                    'category' => 'manual',
                    'post_id' => 0,
                    'note' => __('Sidebar - fix in Appearance → Widgets', 'screaming-fixes'),
                    'location' => 'sidebar',
                ];
            }

            // Content links - try to find the post
            if ($position === 'content') {
                $post_id = $this->url_to_post_id($source_url);
                if ($post_id > 0) {
                    return [
                        'category' => 'fixable',
                        'post_id' => $post_id,
                        'note' => '',
                        'location' => 'content',
                    ];
                }
            }
        }

        // Check URL patterns for dynamic/unfixable pages

        // Homepage
        if ($path === '/' || $path === '' || $path === '/index.php') {
            return [
                'category' => 'manual',
                'post_id' => 0,
                'note' => __('Homepage - fix manually in theme or widgets', 'screaming-fixes'),
                'location' => 'homepage',
            ];
        }

        // Pagination
        if (preg_match('/\/page\/\d+\/?$/', $path)) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Pagination - will be fixed when source content is fixed', 'screaming-fixes'),
                'location' => 'pagination',
            ];
        }

        // Archives
        if (strpos($path, '/author/') !== false ||
            strpos($path, '/category/') !== false ||
            strpos($path, '/tag/') !== false) {
            return [
                'category' => 'skip',
                'post_id' => 0,
                'note' => __('Archive page - dynamic page', 'screaming-fixes'),
                'location' => 'archive',
            ];
        }

        // Try to get post ID
        $post_id = $this->url_to_post_id($source_url);

        if ($post_id > 0) {
            return [
                'category' => 'fixable',
                'post_id' => $post_id,
                'note' => '',
                'location' => 'content',
            ];
        }

        // No post ID found
        return [
            'category' => 'manual',
            'post_id' => 0,
            'note' => __('Source not found - may be in theme, widget, or plugin', 'screaming-fixes'),
            'location' => 'unknown',
        ];
    }

    /**
     * Convert a URL to a WordPress post ID
     *
     * @param string $url The URL to convert
     * @return int Post ID or 0 if not found
     */
    private function url_to_post_id($url) {
        if (empty($url)) {
            return 0;
        }

        // Try WordPress built-in function first
        $post_id = url_to_postid($url);

        if ($post_id) {
            return $post_id;
        }

        // Try with trailing slash variations
        $url_with_slash = trailingslashit($url);
        $url_without_slash = untrailingslashit($url);

        $post_id = url_to_postid($url_with_slash);
        if ($post_id) {
            return $post_id;
        }

        $post_id = url_to_postid($url_without_slash);
        if ($post_id) {
            return $post_id;
        }

        // Try extracting path and searching by slug
        $path = wp_parse_url($url, PHP_URL_PATH);
        if ($path) {
            $path = trim($path, '/');
            $slug = basename($path);

            if ($slug) {
                global $wpdb;

                $post_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts}
                     WHERE post_name = %s
                     AND post_status IN ('publish', 'draft', 'pending', 'private')
                     AND post_type NOT IN ('revision', 'nav_menu_item', 'attachment')
                     LIMIT 1",
                    $slug
                ));

                if ($post_id) {
                    return (int) $post_id;
                }
            }
        }

        return 0;
    }

    /**
     * Get issue count for dashboard display
     *
     * @return int Count of redirect chains
     */
    public function get_issue_count() {
        $results = $this->get_results();

        if (empty($results) || !isset($results['total_count'])) {
            $upload_data = $this->get_upload_data();
            if ($upload_data && isset($upload_data['total_count'])) {
                return (int) $upload_data['total_count'];
            }
            return 0;
        }

        return (int) $results['total_count'];
    }

    /**
     * Apply approved fixes
     *
     * @param array $fixes Array of fixes to apply
     * @return array Results with success/failure counts
     */
    public function apply_fixes($fixes) {
        $fixer = new SF_Link_Fixer();

        // Start batch tracking for undo capability
        SF_Batch_Restore::start_batch('redirect-chains', $fixes);

        $results = [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => [],
            'details' => [],
        ];

        foreach ($fixes as $fix) {
            $address = $fix['address'] ?? '';
            $final_address = $fix['final_address'] ?? '';
            $post_ids = $fix['post_ids'] ?? [];

            if (empty($address) || empty($final_address)) {
                $results['skipped']++;
                continue;
            }

            $results['total']++;

            // If specific post IDs provided, fix only those posts
            if (!empty($post_ids)) {
                foreach ($post_ids as $post_id) {
                    if (!$post_id) {
                        continue;
                    }

                    $fix_result = $fixer->replace_url_in_post(
                        $post_id,
                        $address,
                        $final_address,
                        'replace',
                        'redirect-chains'
                    );

                    if (is_wp_error($fix_result)) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'post_id' => $post_id,
                            'address' => $address,
                            'error' => $fix_result->get_error_message(),
                        ];
                    } else {
                        $results['success']++;
                    }
                }
            } else {
                // Bulk replace
                $fix_result = $fixer->bulk_replace_url($address, $final_address, 'replace', 'redirect-chains');

                $results['success'] += $fix_result['success'];
                $results['failed'] += $fix_result['failed'];
                $results['errors'] = array_merge($results['errors'], $fix_result['errors']);
            }

            $results['details'][] = [
                'address' => $address,
                'final_address' => $final_address,
            ];
        }

        // Update stored results to reflect fixes applied
        $this->update_results_after_fixes($fixes, $results);

        // Complete batch tracking
        SF_Batch_Restore::complete_batch($results);

        return $results;
    }

    /**
     * AJAX: Process uploaded CSV
     */
    public function ajax_process_csv() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $upload_id = isset($_POST['upload_id']) ? sanitize_text_field($_POST['upload_id']) : '';

        if (empty($upload_id)) {
            wp_send_json_error(['message' => __('No upload ID provided.', 'screaming-fixes')]);
        }

        $uploads = get_option('sf_pending_uploads', []);

        if (!isset($uploads[$upload_id])) {
            wp_send_json_error(['message' => __('Upload not found.', 'screaming-fixes')]);
        }

        $file_path = $uploads[$upload_id]['path'];

        if (!file_exists($file_path)) {
            wp_send_json_error(['message' => __('File not found.', 'screaming-fixes')]);
        }

        // Check if this is a bulk fix CSV
        $parser = new SF_CSV_Parser();
        $parsed = $parser->parse($file_path, [
            'extract_columns' => null, // Get all headers for detection
            'max_rows' => 1, // Just need headers
        ]);

        if (!is_wp_error($parsed)) {
            $bulk_check = $this->is_bulk_fix_csv($parsed['headers']);

            if ($bulk_check !== false) {
                // This is a bulk fix CSV - process it differently
                $results = $this->process_bulk_csv($file_path);

                if (is_wp_error($results)) {
                    wp_send_json_error(['message' => $results->get_error_message()]);
                }

                // Clean up temp file reference
                unset($uploads[$upload_id]);
                update_option('sf_pending_uploads', $uploads);

                wp_send_json_success([
                    'message' => sprintf(
                        __('Bulk fix CSV processed: %d fixes ready, %d invalid, %d skipped.', 'screaming-fixes'),
                        $results['ready_count'],
                        $results['invalid_count'],
                        $results['skipped_count']
                    ),
                    'data' => $results,
                    'is_bulk_update' => true,
                ]);
                return;
            }
        }

        // Regular CSV processing
        $results = $this->process_csv($file_path);

        if (is_wp_error($results)) {
            wp_send_json_error(['message' => $results->get_error_message()]);
        }

        // Clean up temp file reference
        unset($uploads[$upload_id]);
        update_option('sf_pending_uploads', $uploads);

        wp_send_json_success([
            'message' => sprintf(
                __('Found %d redirect issues: %d fixable, %d manual, %d skipped.', 'screaming-fixes'),
                $results['total_count'],
                $results['fixable_count'],
                $results['manual_count'],
                $results['skipped_count']
            ),
            'data' => $results,
        ]);
    }

    /**
     * AJAX: Get stored data
     */
    public function ajax_get_data() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            wp_send_json_success(['data' => null]);
        }

        wp_send_json_success(['data' => $results]);
    }

    /**
     * AJAX: Clear all redirect chains data
     */
    public function ajax_clear_data() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();

        // Delete from database
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => 'redirect-chains'
        ]);

        // Clear transient
        delete_transient('sf_redirect-chains_results');

        wp_send_json_success([
            'message' => __('Data cleared successfully. You can now upload a new CSV file.', 'screaming-fixes')
        ]);
    }

    /**
     * AJAX: Apply fixes
     */
    public function ajax_apply_fixes() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $fixes = isset($_POST['fixes']) ? $_POST['fixes'] : [];

        if (empty($fixes)) {
            wp_send_json_error(['message' => __('No fixes provided.', 'screaming-fixes')]);
        }

        // Sanitize fixes array
        $sanitized_fixes = [];
        foreach ($fixes as $fix) {
            $post_ids = [];
            if (isset($fix['post_ids']) && is_array($fix['post_ids'])) {
                $post_ids = array_map('absint', $fix['post_ids']);
                $post_ids = array_filter($post_ids);
            }

            $sanitized_fixes[] = [
                'address' => isset($fix['address']) ? esc_url_raw($fix['address']) : '',
                'final_address' => isset($fix['final_address']) ? esc_url_raw($fix['final_address']) : '',
                'post_ids' => $post_ids,
            ];
        }

        // Look up sources for each fix from stored results before applying
        $stored_results = $this->get_results();
        if (empty($stored_results) || empty($stored_results['redirects'])) {
            $stored_results = $this->get_upload_data();
        }
        $sources_by_address = [];
        foreach (array_merge($stored_results['redirects'] ?? [], $stored_results['fixable_redirects'] ?? []) as $redirect) {
            $addr = $redirect['address'] ?? '';
            if (!empty($addr)) {
                $sources_by_address[$addr] = $redirect['fixable_sources'] ?? $redirect['sources'] ?? [];
            }
        }

        $results = $this->apply_fixes($sanitized_fixes);

        // Log to activity log for dashboard
        if ($results['success'] > 0) {
            SF_Activity_Log::log('redirect-chains', $results['success']);
        }

        // Build detailed response
        $response = [
            'results' => $results,
            'summary' => [
                'total' => $results['total'],
                'success' => $results['success'],
                'failed' => $results['failed'],
                'skipped' => $results['skipped'],
            ],
        ];

        // Build appropriate message based on results
        if ($results['failed'] === 0 && $results['success'] > 0) {
            $response['message'] = sprintf(
                __('Successfully fixed %d redirect(s).', 'screaming-fixes'),
                $results['success']
            );
            $response['status'] = 'success';
        } elseif ($results['success'] === 0 && $results['failed'] > 0) {
            $response['message'] = sprintf(
                __('Failed to fix %d redirect(s). See details below.', 'screaming-fixes'),
                $results['failed']
            );
            $response['status'] = 'error';
        } elseif ($results['success'] > 0 && $results['failed'] > 0) {
            $response['message'] = sprintf(
                __('Partially successful: %d fixed, %d failed.', 'screaming-fixes'),
                $results['success'],
                $results['failed']
            );
            $response['status'] = 'partial';
        } else {
            $response['message'] = __('No changes were made.', 'screaming-fixes');
            $response['status'] = 'none';
        }

        // Include error details for failed fixes
        if (!empty($results['errors'])) {
            $response['errors'] = array_map(function($error) {
                $error_msg = $error['error'] ?? __('Unknown error', 'screaming-fixes');

                // Make error messages more user-friendly
                if (strpos($error_msg, 'not found') !== false || strpos($error_msg, 'No occurrences') !== false) {
                    $error_msg = __('URL not found in this content. It may have already been fixed or the URL is on an external site.', 'screaming-fixes');
                } elseif (strpos($error_msg, 'permission') !== false) {
                    $error_msg = __('Permission denied. You may not have access to edit this content.', 'screaming-fixes');
                } elseif (strpos($error_msg, 'post does not exist') !== false || strpos($error_msg, 'Invalid post') !== false) {
                    $error_msg = __('This page no longer exists in the database.', 'screaming-fixes');
                }

                return [
                    'post_id' => $error['post_id'] ?? null,
                    'address' => $error['address'] ?? '',
                    'message' => $error_msg,
                ];
            }, $results['errors']);
        }

        // Build per-fix result items for the Fixes Applied section
        $fix_results = [];
        $error_map = [];
        foreach ($results['errors'] ?? [] as $error) {
            $error_map[$error['address'] ?? ''] = $error['error'] ?? __('Unknown error', 'screaming-fixes');
        }

        foreach ($sanitized_fixes as $fix) {
            $address = $fix['address'] ?? '';
            $final_address = $fix['final_address'] ?? '';
            $source_count = count($fix['post_ids'] ?? []);

            if (isset($error_map[$address])) {
                $error_msg = $error_map[$address];
                if (strpos($error_msg, 'not found') !== false || strpos($error_msg, 'No occurrences') !== false) {
                    $error_msg = __('URL not found in this content. It may have already been fixed or the URL is on an external site.', 'screaming-fixes');
                } elseif (strpos($error_msg, 'permission') !== false) {
                    $error_msg = __('Permission denied. You may not have access to edit this content.', 'screaming-fixes');
                } elseif (strpos($error_msg, 'post does not exist') !== false || strpos($error_msg, 'Invalid post') !== false) {
                    $error_msg = __('This page no longer exists in the database.', 'screaming-fixes');
                }

                $fix_results[] = [
                    'address' => $address,
                    'final_address' => $final_address,
                    'status' => 'failed',
                    'status_message' => $error_msg,
                    'source_count' => $source_count,
                    'sources' => $sources_by_address[$address] ?? [],
                    'fixed_at' => current_time('mysql'),
                ];
            } else {
                $fix_results[] = [
                    'address' => $address,
                    'final_address' => $final_address,
                    'status' => 'success',
                    'status_message' => '',
                    'source_count' => $source_count,
                    'sources' => $sources_by_address[$address] ?? [],
                    'fixed_at' => current_time('mysql'),
                ];
            }
        }
        $response['fix_results'] = $fix_results;

        wp_send_json_success($response);
    }

    /**
     * AJAX: Export results as CSV
     */
    public function ajax_export() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        if (empty($results) || empty($results['redirects'])) {
            $results = $this->get_upload_data();
        }

        if (empty($results) || empty($results['redirects'])) {
            wp_send_json_error(['message' => __('No data to export.', 'screaming-fixes')]);
        }

        $export_data = [];
        foreach ($results['redirects'] as $redirect) {
            $source_urls = array_column($redirect['sources'], 'source_url');

            $export_data[] = [
                'Redirect URL' => $redirect['address'],
                'Final Destination' => $redirect['final_address'],
                'Hops' => $redirect['num_redirects'],
                'Type' => $redirect['redirect_type']['label'] ?? '',
                'Found In Pages' => $redirect['source_count'],
                'Source URLs' => implode('; ', $source_urls),
            ];
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'redirect-chains-export-' . date('Y-m-d') . '.csv',
        ]);
    }

    /**
     * AJAX: Export fixed redirects as CSV
     */
    public function ajax_export_fixed() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $results = $this->get_results();

        // Also check upload data if transient is empty
        if (empty($results) || empty($results['fixed_redirects'])) {
            $results = $this->get_upload_data();
        }

        if (empty($results) || empty($results['fixed_redirects'])) {
            wp_send_json_error(['message' => __('No fixed redirects to export.', 'screaming-fixes')]);
        }

        $export_data = [];
        foreach ($results['fixed_redirects'] as $redirect) {
            $sources = $redirect['fixable_sources'] ?? $redirect['sources'] ?? [];
            $address = $redirect['address'] ?? '';
            $final_address = $redirect['final_address'] ?? '';
            $fixed_at = $redirect['fixed_at'] ?? '';
            $status = $redirect['status'] ?? 'success';
            $status_label = $status === 'success' ? 'Fixed' : ($status === 'failed' ? 'Failed' : 'Skipped');
            $status_message = $redirect['status_message'] ?? '';

            // Add one row per Found In URL
            if (!empty($sources) && is_array($sources)) {
                foreach ($sources as $source) {
                    $export_data[] = [
                        'Was (Original URL)' => $address,
                        'Now (Final URL)' => $final_address,
                        'Status' => $status_label,
                        'Status Message' => $status_message,
                        'Found In URL' => $source['source_url'] ?? '',
                        'Anchor Text' => $source['anchor_text'] ?? '',
                        'Location' => ucfirst($source['location'] ?? 'content'),
                        'When' => $fixed_at,
                    ];
                }
            } else {
                // If no sources data, still add one row for the redirect
                $export_data[] = [
                    'Was (Original URL)' => $address,
                    'Now (Final URL)' => $final_address,
                    'Status' => $status_label,
                    'Status Message' => $status_message,
                    'Found In URL' => '',
                    'Anchor Text' => '',
                    'Location' => '',
                    'When' => $fixed_at,
                ];
            }
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'redirect-chains-fixed-' . date('Y-m-d-His') . '.csv',
        ]);
    }

    /**
     * AJAX handler to get the Fixed Redirects section HTML
     * Used for immediate refresh after applying fixes
     */
    public function ajax_get_fixed_section() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        // Get stored data
        $results = $this->get_results();
        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        $fixed_redirects = $results['fixed_redirects'] ?? [];
        $fixed_count = count($fixed_redirects);

        // Count by status
        $success_count = 0;
        $failed_count = 0;
        $skipped_count = 0;
        foreach ($fixed_redirects as $r) {
            $status = $r['status'] ?? 'success';
            if ($status === 'success') $success_count++;
            elseif ($status === 'failed') $failed_count++;
            else $skipped_count++;
        }

        // Generate the HTML
        ob_start();
        ?>
        <div class="sf-results-header sf-results-header-collapsible sf-results-header-fixed">
            <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="true">
                <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
                <span class="sf-fixes-applied-title">
                    <?php esc_html_e('Fixes Applied', 'screaming-fixes'); ?> (<span class="sf-fixes-applied-count"><?php echo esc_html($fixed_count); ?></span>)
                </span>
                <span class="sf-section-hint"><?php esc_html_e('View fix results and export to CSV', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
            </button>
            <div class="sf-results-actions">
                <button type="button" class="sf-button sf-button-secondary" id="sf-export-fixed-redirects">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <div class="sf-fixed-content">
            <!-- Status Filter Buttons -->
            <div class="sf-status-filter-row sf-rc-status-filters">
                <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                <div class="sf-status-filters">
                    <button type="button" class="sf-status-filter active" data-status="all">
                        <span class="sf-status-icon">&#128203;</span>
                        <span class="sf-status-count sf-rc-status-count-all"><?php echo esc_html($fixed_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                        <span class="sf-status-icon">&#9989;</span>
                        <span class="sf-status-count sf-rc-status-count-success"><?php echo esc_html($success_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                        <span class="sf-status-icon">&#10060;</span>
                        <span class="sf-status-count sf-rc-status-count-failed"><?php echo esc_html($failed_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                        <span class="sf-status-icon">&#9203;</span>
                        <span class="sf-status-count sf-rc-status-count-skipped"><?php echo esc_html($skipped_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                    </button>
                </div>
            </div>

            <div class="sf-results-table-wrapper">
                <table class="sf-results-table sf-fixed-table sf-fixes-applied-table">
                    <thead>
                        <tr>
                            <th class="sf-col-was"><?php esc_html_e('Original URL', 'screaming-fixes'); ?></th>
                            <th class="sf-col-now"><?php esc_html_e('Updated To', 'screaming-fixes'); ?></th>
                            <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            <th class="sf-col-count"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                            <th class="sf-col-when"><?php esc_html_e('Date Applied', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fixed_redirects as $redirect):
                            $address = $redirect['address'] ?? '';
                            $final_address = $redirect['final_address'] ?? '';
                            $fixed_at = $redirect['fixed_at'] ?? '';
                            $sources = $redirect['fixable_sources'] ?? $redirect['sources'] ?? [];
                            $source_count = $redirect['fixable_count'] ?? (is_array($sources) ? count($sources) : ($redirect['source_count'] ?? 0));
                            $status = $redirect['status'] ?? 'success';
                            $status_message = $redirect['status_message'] ?? '';

                            if ($status === 'success') {
                                $status_icon = '&#9989;';
                                $status_label = __('Fixed', 'screaming-fixes');
                                $row_class = 'sf-fixed-row-success';
                            } elseif ($status === 'failed') {
                                $status_icon = '&#10060;';
                                $status_label = __('Failed', 'screaming-fixes');
                                $row_class = 'sf-fixed-row-failed';
                            } else {
                                $status_icon = '&#9203;';
                                $status_label = __('Skipped', 'screaming-fixes');
                                $row_class = 'sf-fixed-row-skipped';
                            }
                        ?>
                        <tr class="sf-fixed-row <?php echo esc_attr($row_class); ?>" data-status="<?php echo esc_attr($status); ?>">
                            <td class="sf-col-was">
                                <div class="sf-url-cell">
                                    <span class="sf-url-text sf-url-strikethrough" title="<?php echo esc_attr($address); ?>">
                                        <?php echo esc_html($this->truncate_url($address)); ?>
                                    </span>
                                    <a href="<?php echo esc_url($address); ?>" target="_blank" class="sf-url-link" title="<?php esc_attr_e('Open URL', 'screaming-fixes'); ?>">
                                        <span class="dashicons dashicons-external"></span>
                                    </a>
                                </div>
                            </td>
                            <td class="sf-col-now">
                                <div class="sf-url-cell">
                                    <span class="sf-url-text sf-final-url" title="<?php echo esc_attr($final_address); ?>">
                                        <?php echo esc_html($this->truncate_url($final_address)); ?>
                                    </span>
                                    <a href="<?php echo esc_url($final_address); ?>" target="_blank" class="sf-url-link" title="<?php esc_attr_e('Open URL', 'screaming-fixes'); ?>">
                                        <span class="dashicons dashicons-external"></span>
                                    </a>
                                </div>
                            </td>
                            <td class="sf-col-status">
                                <span class="sf-status-badge sf-status-<?php echo esc_attr($status); ?>"<?php if (!empty($status_message) && $status !== 'success') echo ' title="' . esc_attr($status_message) . '"'; ?>>
                                    <span class="sf-status-icon"><?php echo $status_icon; ?></span>
                                    <?php echo esc_html($status_label); ?>
                                </span>
                            </td>
                            <td class="sf-col-count">
                                <button type="button" class="sf-expand-sources" data-expanded="false" data-redirect-url="<?php echo esc_attr($address); ?>">
                                    <span class="sf-source-count"><?php echo esc_html($source_count); ?></span>
                                    <span class="sf-source-label"><?php echo esc_html(_n('page', 'pages', $source_count, 'screaming-fixes')); ?></span>
                                    <span class="dashicons dashicons-arrow-down-alt2"></span>
                                </button>
                            </td>
                            <td class="sf-col-when">
                                <?php if (!empty($fixed_at)): ?>
                                    <span class="sf-fixed-time" title="<?php echo esc_attr($fixed_at); ?>">
                                        <?php echo esc_html(human_time_diff(strtotime($fixed_at), current_time('timestamp')) . ' ' . __('ago', 'screaming-fixes')); ?>
                                    </span>
                                <?php else: ?>
                                    <span class="sf-fixed-time">&mdash;</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr class="sf-sources-row" style="display: none;">
                            <td colspan="5">
                                <div class="sf-sources-list">
                                    <table class="sf-sources-table">
                                        <thead>
                                            <tr>
                                                <th><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                                <th><?php esc_html_e('Location', 'screaming-fixes'); ?></th>
                                                <th><?php esc_html_e('Edit', 'screaming-fixes'); ?></th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (is_array($sources)) : ?>
                                            <?php foreach ($sources as $source) : ?>
                                            <tr data-post-id="<?php echo esc_attr($source['post_id'] ?? ''); ?>">
                                                <td>
                                                    <?php if (!empty($source['post_title'])) : ?>
                                                        <a href="<?php echo esc_url($source['source_url'] ?? ''); ?>" target="_blank" class="sf-source-title-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                            <?php echo esc_html($source['post_title']); ?>
                                                        </a>
                                                    <?php elseif (!empty($source['source_url'])) : ?>
                                                        <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" class="sf-source-url-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                            <?php echo esc_html($source['source_url']); ?>
                                                        </a>
                                                    <?php else : ?>
                                                        <span>-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="sf-location-badge"><?php echo esc_html(ucfirst($source['location'] ?? 'content')); ?></span>
                                                </td>
                                                <td>
                                                    <?php if (!empty($source['edit_url'])) : ?>
                                                    <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" class="sf-edit-link" title="<?php esc_attr_e('Edit in WordPress', 'screaming-fixes'); ?>">
                                                        <span class="dashicons dashicons-edit"></span>
                                                    </a>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php
        $html = ob_get_clean();

        wp_send_json_success([
            'html' => $html,
            'fixed_count' => $fixed_count,
        ]);
    }

    /**
     * Helper to truncate URLs for display
     *
     * @param string $url URL to truncate
     * @param int $max_length Maximum length
     * @return string Truncated URL
     */
    private function truncate_url($url, $max_length = 50) {
        if (strlen($url) <= $max_length) {
            return $url;
        }
        return substr($url, 0, $max_length - 3) . '...';
    }

    /**
     * Save upload data to database
     *
     * @param array $data Data to save
     */
    private function save_upload_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        // Delete any existing data for this session/module
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Insert new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get upload data from database
     *
     * @return array|null Data or null if not found
     */
    private function get_upload_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = $this->get_session_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            return json_decode($row->data, true);
        }

        return null;
    }

    /**
     * Get or create session ID
     *
     * @return string Session ID
     */
    private function get_session_id() {
        if (!session_id()) {
            return 'user_' . get_current_user_id();
        }
        return session_id();
    }

    /**
     * Update results after fixes are applied
     *
     * @param array $fixes Applied fixes
     * @param array $fix_results Results from apply_fixes() with success/failed/errors
     */
    private function update_results_after_fixes($fixes, $fix_results = []) {
        $results = $this->get_results();

        // Also check upload data if transient is empty
        if (empty($results) || empty($results['redirects'])) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            return;
        }

        // Initialize fixed_redirects array if not exists
        if (!isset($results['fixed_redirects'])) {
            $results['fixed_redirects'] = [];
        }

        // Build set of failed addresses from fix results errors
        $failed_addresses = [];
        foreach ($fix_results['errors'] ?? [] as $error) {
            $address = $error['address'] ?? '';
            if (!empty($address)) {
                $error_msg = $error['error'] ?? __('Unknown error', 'screaming-fixes');
                if (strpos($error_msg, 'not found') !== false || strpos($error_msg, 'No occurrences') !== false) {
                    $error_msg = __('URL not found in this content. It may have already been fixed or the URL is on an external site.', 'screaming-fixes');
                } elseif (strpos($error_msg, 'permission') !== false) {
                    $error_msg = __('Permission denied. You may not have access to edit this content.', 'screaming-fixes');
                } elseif (strpos($error_msg, 'post does not exist') !== false || strpos($error_msg, 'Invalid post') !== false) {
                    $error_msg = __('This page no longer exists in the database.', 'screaming-fixes');
                }
                $failed_addresses[$address] = $error_msg;
            }
        }

        // Create a map of all attempted addresses with their fix details
        $fixed_addresses = [];
        foreach ($fixes as $fix) {
            $address = $fix['address'] ?? '';
            if (!empty($address)) {
                $fixed_addresses[$address] = [
                    'final_address' => $fix['final_address'] ?? '',
                    'fixed_at' => current_time('mysql'),
                ];
            }
        }

        if (empty($fixed_addresses)) {
            return;
        }

        // Move attempted redirects from redirects to fixed_redirects (both success and failed)
        $remaining_redirects = [];
        $added_addresses = [];
        foreach ($results['redirects'] ?? [] as $redirect) {
            $address = $redirect['address'] ?? '';
            if (isset($fixed_addresses[$address])) {
                // Only add each address once to fixed_redirects (avoid duplicates)
                if (isset($added_addresses[$address])) {
                    continue;
                }
                $added_addresses[$address] = true;

                // Add to fixed_redirects with fix details and status
                $fixed_redirect = $redirect;
                $fixed_redirect['fixed_at'] = $fixed_addresses[$address]['fixed_at'];
                $fixed_redirect['fix_action'] = 'replace';

                if (isset($failed_addresses[$address])) {
                    $fixed_redirect['status'] = 'failed';
                    $fixed_redirect['status_message'] = $failed_addresses[$address];
                } else {
                    $fixed_redirect['status'] = 'success';
                    $fixed_redirect['status_message'] = '';
                }

                $results['fixed_redirects'][] = $fixed_redirect;
            } else {
                $remaining_redirects[] = $redirect;
            }
        }

        // Also check fixable_redirects array
        $remaining_fixable = [];
        foreach ($results['fixable_redirects'] ?? [] as $redirect) {
            $address = $redirect['address'] ?? '';
            if (isset($fixed_addresses[$address])) {
                // Add to fixed_redirects with fix details (if not already added)
                if (!isset($added_addresses[$address])) {
                    $added_addresses[$address] = true;
                    $fixed_redirect = $redirect;
                    $fixed_redirect['fixed_at'] = $fixed_addresses[$address]['fixed_at'];
                    $fixed_redirect['fix_action'] = 'replace';

                    if (isset($failed_addresses[$address])) {
                        $fixed_redirect['status'] = 'failed';
                        $fixed_redirect['status_message'] = $failed_addresses[$address];
                    } else {
                        $fixed_redirect['status'] = 'success';
                        $fixed_redirect['status_message'] = '';
                    }

                    $results['fixed_redirects'][] = $fixed_redirect;
                }
            } else {
                $remaining_fixable[] = $redirect;
            }
        }

        $results['redirects'] = array_values($remaining_redirects);
        $results['fixable_redirects'] = array_values($remaining_fixable);

        // Recalculate counts
        $results['total_count'] = count($results['redirects']);
        $results['fixable_count'] = count($results['fixable_redirects']);
        $results['fixed_count'] = count($results['fixed_redirects']);

        // Recalculate source counts
        $total_sources = 0;
        $fixable_sources = 0;
        foreach ($results['redirects'] ?? [] as $redirect) {
            $total_sources += $redirect['source_count'] ?? count($redirect['sources'] ?? []);
        }
        foreach ($results['fixable_redirects'] ?? [] as $redirect) {
            $fixable_sources += $redirect['fixable_count'] ?? count($redirect['fixable_sources'] ?? $redirect['sources'] ?? []);
        }
        $results['total_sources'] = $total_sources;
        $results['fixable_sources'] = $fixable_sources;

        // Update timestamp
        $results['last_updated'] = current_time('mysql');

        // Save updated results to both transient and database
        $this->save_results($results);
        $this->save_upload_data($results);
    }

    /**
     * Get module ID
     *
     * @return string Module ID
     */
    public function get_module_id() {
        return $this->module_id;
    }

    /**
     * AJAX: Fix a temporary redirect by converting 302 to 301
     */
    public function ajax_fix_temp_redirect() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $source = isset($_POST['source']) ? esc_url_raw($_POST['source']) : '';

        if (empty($source)) {
            wp_send_json_error(['message' => __('No source URL provided.', 'screaming-fixes')]);
        }

        // Use SF_Redirect_Manager to modify the redirect
        $redirect_manager = new SF_Redirect_Manager();

        // Check if we have a redirect plugin available
        if (!$redirect_manager->has_capability()) {
            wp_send_json_error([
                'message' => __('No redirect plugin detected. Install Rank Math, Redirection, or Yoast Premium to fix temporary redirects.', 'screaming-fixes'),
            ]);
        }

        // Convert 302 to 301
        $result = $redirect_manager->modify_redirect($source, ['type' => 301]);

        if (!$result['success']) {
            wp_send_json_error([
                'message' => $result['message'],
                'plugin' => $result['plugin'],
            ]);
        }

        // Log to change log
        $logger = new SF_Change_Logger();
        $logger->log_change(0, 'redirect_temp', $source, $source, [
            'module' => 'redirect-chains',
            'source' => $source,
            'plugin' => $result['plugin'],
            'action' => 'temp_to_permanent',
        ]);

        // Log to activity log
        SF_Activity_Log::log('redirect-chains', 1);

        // Update stored results to mark this as fixed
        $this->mark_redirect_as_fixed($source, 'temp_to_permanent');

        wp_send_json_success([
            'message' => sprintf(
                __('Temporary redirect converted to permanent (301) via %s.', 'screaming-fixes'),
                $result['plugin']
            ),
            'plugin' => $result['plugin'],
        ]);
    }

    /**
     * AJAX: Fix a redirect loop by deleting the redirect rule
     */
    public function ajax_fix_redirect_loop() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $source = isset($_POST['source']) ? esc_url_raw($_POST['source']) : '';

        if (empty($source)) {
            wp_send_json_error(['message' => __('No source URL provided.', 'screaming-fixes')]);
        }

        // Use SF_Redirect_Manager to delete the redirect
        $redirect_manager = new SF_Redirect_Manager();

        // Check if we have a redirect plugin available
        $plugin = $redirect_manager->get_redirect_plugin();
        if (!$plugin) {
            wp_send_json_error([
                'message' => __('No redirect plugin detected. Install Rank Math, Redirection, or Yoast Premium to manage redirects.', 'screaming-fixes'),
            ]);
        }

        // Delete the redirect rule that's causing the loop
        $result = $redirect_manager->delete_redirect($source, $plugin);

        if (is_wp_error($result)) {
            wp_send_json_error([
                'message' => $result->get_error_message(),
                'plugin' => $plugin,
            ]);
        }

        // Log to change log
        $logger = new SF_Change_Logger();
        $logger->log_change(0, 'redirect_loop', $source, '', [
            'module' => 'redirect-chains',
            'source' => $source,
            'plugin' => $plugin,
            'action' => 'loop_deleted',
        ]);

        // Log to activity log
        SF_Activity_Log::log('redirect-chains', 1);

        // Update stored results to mark this as fixed
        $this->mark_redirect_as_fixed($source, 'loop_deleted');

        wp_send_json_success([
            'message' => sprintf(
                __('Redirect loop removed. The redirect rule was deleted from %s.', 'screaming-fixes'),
                $plugin
            ),
            'plugin' => $plugin,
        ]);
    }

    /**
     * Mark a redirect as fixed in stored results
     *
     * @param string $source Source URL that was fixed
     * @param string $fix_action Action taken (temp_to_permanent, loop_deleted)
     */
    private function mark_redirect_as_fixed($source, $fix_action) {
        $results = $this->get_results();

        if (empty($results)) {
            $results = $this->get_upload_data();
        }

        if (empty($results)) {
            return;
        }

        // Initialize fixed_redirects array if not exists
        if (!isset($results['fixed_redirects'])) {
            $results['fixed_redirects'] = [];
        }

        // Find and move the redirect to fixed
        $source_normalized = $this->normalize_source_url($source);
        $remaining_redirects = [];
        $remaining_fixable = [];
        $added_to_fixed = false;

        foreach ($results['redirects'] ?? [] as $redirect) {
            $redirect_source = $this->normalize_source_url($redirect['address'] ?? '');
            if ($redirect_source === $source_normalized || $redirect['address'] === $source) {
                // Move to fixed (only add once per address)
                if (!$added_to_fixed) {
                    $redirect['fixed_at'] = current_time('mysql');
                    $redirect['fix_action'] = $fix_action;
                    $results['fixed_redirects'][] = $redirect;
                    $added_to_fixed = true;
                }
            } else {
                $remaining_redirects[] = $redirect;
            }
        }

        foreach ($results['fixable_redirects'] ?? [] as $redirect) {
            $redirect_source = $this->normalize_source_url($redirect['address'] ?? '');
            if ($redirect_source === $source_normalized || $redirect['address'] === $source) {
                // Only add if not already added from redirects array
                if (!$added_to_fixed) {
                    $redirect['fixed_at'] = current_time('mysql');
                    $redirect['fix_action'] = $fix_action;
                    $results['fixed_redirects'][] = $redirect;
                    $added_to_fixed = true;
                }
            } else {
                $remaining_fixable[] = $redirect;
            }
        }

        $results['redirects'] = array_values($remaining_redirects);
        $results['fixable_redirects'] = array_values($remaining_fixable);

        // Recalculate counts
        $results['total_count'] = count($results['redirects']);
        $results['fixable_count'] = count($results['fixable_redirects']);
        $results['fixed_count'] = count($results['fixed_redirects']);

        // Update timestamp
        $results['last_updated'] = current_time('mysql');

        // Save updated results
        $this->save_results($results);
        $this->save_upload_data($results);
    }

    /**
     * Normalize source URL for comparison
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private function normalize_source_url($url) {
        $parsed = wp_parse_url($url);
        $path = $parsed['path'] ?? '/';

        // Remove trailing slash (except for root)
        if ($path !== '/' && substr($path, -1) === '/') {
            $path = rtrim($path, '/');
        }

        // Ensure leading slash
        if (substr($path, 0, 1) !== '/') {
            $path = '/' . $path;
        }

        return $path;
    }

    // =========================================================================
    // BULK CSV UPLOAD METHODS
    // =========================================================================

    /**
     * Check if CSV is a bulk fix CSV (has an Action column)
     *
     * @param array $headers CSV headers
     * @return array|false Array with column info or false if not a bulk CSV
     */
    public function is_bulk_fix_csv($headers) {
        $action_header = null;
        $override_header = null;
        $code_header = null;

        // Action column patterns (required for bulk mode)
        $action_patterns = ['action', 'fix_action', 'sf_action'];

        // Override destination column patterns (optional)
        $override_patterns = ['override_final', 'override_destination', 'sf_redirect_to', 'new_final'];

        // Redirect code column patterns (optional)
        $code_patterns = ['redirect_code', 'new_code', 'sf_redirect_code', 'status_code_override'];

        foreach ($headers as $header) {
            $header_lower = strtolower(trim($header));
            $header_normalized = str_replace([' ', '-', '_'], '', $header_lower);

            // Check for action column
            foreach ($action_patterns as $pattern) {
                $pattern_normalized = str_replace([' ', '-', '_'], '', $pattern);
                if ($header_lower === $pattern || $header_normalized === $pattern_normalized) {
                    $action_header = $header_lower;
                    break;
                }
            }

            // Check for override column
            foreach ($override_patterns as $pattern) {
                $pattern_normalized = str_replace([' ', '-', '_'], '', $pattern);
                if ($header_lower === $pattern || $header_normalized === $pattern_normalized) {
                    $override_header = $header_lower;
                    break;
                }
            }

            // Check for code column
            foreach ($code_patterns as $pattern) {
                $pattern_normalized = str_replace([' ', '-', '_'], '', $pattern);
                if ($header_lower === $pattern || $header_normalized === $pattern_normalized) {
                    $code_header = $header_lower;
                    break;
                }
            }
        }

        // Not a bulk fix CSV if no action column
        if ($action_header === null) {
            return false;
        }

        return [
            'is_bulk' => true,
            'action_header' => $action_header,
            'override_header' => $override_header,
            'code_header' => $code_header,
        ];
    }

    /**
     * Process a bulk fix CSV file
     *
     * @param string $file_path Path to CSV file
     * @return array|WP_Error Processing results
     */
    public function process_bulk_csv($file_path) {
        $parser = new SF_CSV_Parser();

        // Parse with all needed columns
        $parsed = $parser->parse($file_path, [
            'extract_columns' => [
                'address', 'final address', 'source', 'number of redirects',
                'chain type', 'anchor text', 'link position', 'status code',
                'loop', 'temp redirect in chain', 'redirect type 1', 'status code 1',
                'action', 'override_final', 'override_destination', 'sf_redirect_to', 'new_final',
                'redirect_code', 'new_code', 'sf_redirect_code',
            ],
        ]);

        if (is_wp_error($parsed)) {
            return $parsed;
        }

        // Check row count
        $total_rows = count($parsed['rows']);
        if ($total_rows > self::MAX_CSV_ROWS) {
            return new \WP_Error(
                'csv_too_large',
                sprintf(
                    __('Your CSV contains %1$s rows, which exceeds the maximum limit of %2$s rows. Please split your file into smaller batches.', 'screaming-fixes'),
                    number_format_i18n($total_rows),
                    number_format_i18n(self::MAX_CSV_ROWS)
                )
            );
        }

        // Standardize column names
        $parsed = $this->standardize_redirect_columns($parsed);

        // Detect bulk columns
        $bulk_check = $this->is_bulk_fix_csv($parsed['headers']);
        $action_header = $bulk_check['action_header'] ?? 'action';
        $override_header = $bulk_check['override_header'];
        $code_header = $bulk_check['code_header'];

        // Get site domain
        $site_domain = wp_parse_url(home_url(), PHP_URL_HOST);

        $results = [
            'is_bulk_update' => true,
            'bulk_complete' => false,
            'ready_fixes' => [],
            'invalid_fixes' => [],
            'skipped_fixes' => [],
            'warnings' => [],
            'ready_count' => 0,
            'invalid_count' => 0,
            'skipped_count' => 0,
            'total_rows' => $total_rows,
        ];

        foreach ($parsed['rows'] as $row_index => $row) {
            $address = $row['address'] ?? '';
            $final_address = $row['final_address'] ?? '';
            $source_url = $row['source'] ?? '';

            // Get action from row (case-insensitive)
            $action = '';
            foreach ($row as $key => $value) {
                $key_lower = strtolower(trim($key));
                if ($key_lower === $action_header || str_replace([' ', '-', '_'], '', $key_lower) === str_replace([' ', '-', '_'], '', $action_header)) {
                    $action = strtolower(trim($value));
                    break;
                }
            }

            // Get override destination
            $override_final = '';
            if ($override_header) {
                foreach ($row as $key => $value) {
                    $key_lower = strtolower(trim($key));
                    if ($key_lower === $override_header) {
                        $override_final = trim($value);
                        break;
                    }
                }
            }
            // Also check common override column names directly
            if (empty($override_final)) {
                $override_final = $row['override_final'] ?? $row['override_destination'] ?? $row['sf_redirect_to'] ?? $row['new_final'] ?? '';
            }

            // Get redirect code override
            $redirect_code = '';
            if ($code_header) {
                foreach ($row as $key => $value) {
                    $key_lower = strtolower(trim($key));
                    if ($key_lower === $code_header) {
                        $redirect_code = trim($value);
                        break;
                    }
                }
            }
            // Also check common code column names directly
            if (empty($redirect_code)) {
                $redirect_code = $row['redirect_code'] ?? $row['new_code'] ?? $row['sf_redirect_code'] ?? '';
            }

            // Skip if no address
            if (empty($address)) {
                continue;
            }

            // Determine issue type from row data
            $is_loop = isset($row['loop']) && strtolower(trim($row['loop'])) === 'true';
            $has_temp = isset($row['temp_redirect']) && strtolower(trim($row['temp_redirect'])) === 'true';

            // Also check redirect_type_1 for temp detection
            if (!$has_temp && isset($row['redirect_type_1'])) {
                $rt1 = strtolower(trim($row['redirect_type_1']));
                $has_temp = in_array($rt1, ['302', '307', 'temporary redirect']);
            }

            $is_chain = !$is_loop && !$has_temp;
            $issue_type = $is_loop ? 'loop' : ($has_temp ? 'temp' : 'chain');

            // Skip if no action or action is "skip"
            if (empty($action) || $action === 'skip') {
                $results['skipped_fixes'][] = [
                    'address' => $address,
                    'final_address' => $final_address,
                    'issue_type' => $issue_type,
                    'reason' => empty($action) ? __('No action specified', 'screaming-fixes') : __('Marked as skip', 'screaming-fixes'),
                ];
                $results['skipped_count']++;
                continue;
            }

            // Validate action for issue type
            $validation = $this->validate_bulk_action($action, $is_loop, $has_temp, $is_chain, $row);

            if (!$validation['valid']) {
                $results['invalid_fixes'][] = [
                    'address' => $address,
                    'final_address' => $final_address,
                    'action' => $action,
                    'issue_type' => $issue_type,
                    'reason' => $validation['reason'],
                ];
                $results['invalid_count']++;
                continue;
            }

            // Get source URLs and their post IDs for chain fixes
            $sources = [];
            if ($is_chain && $action === 'fix') {
                // Try to find the post ID for the source URL
                $link_position = $row['link_position'] ?? '';
                $categorization = $this->categorize_redirect($source_url, $link_position);

                if ($categorization['post_id'] > 0) {
                    $sources[] = [
                        'source_url' => $source_url,
                        'post_id' => $categorization['post_id'],
                        'post_title' => get_the_title($categorization['post_id']),
                        'fix_category' => $categorization['category'],
                    ];
                }
            }

            // Add to ready queue
            $target_url = !empty($override_final) ? $override_final : $final_address;
            $results['ready_fixes'][] = [
                'address' => $address,
                'final_address' => $final_address,
                'target_url' => $target_url,
                'override_final' => $override_final,
                'redirect_code' => $redirect_code,
                'action' => $action,
                'is_loop' => $is_loop,
                'has_temp_redirect' => $has_temp,
                'is_chain' => $is_chain,
                'issue_type' => $issue_type,
                'sources' => $sources,
                'source_url' => $source_url,
                'status' => 'ready',
            ];
            $results['ready_count']++;
        }

        // Add warning for large files
        if ($results['ready_count'] > 500) {
            $results['warnings'][] = sprintf(
                __('Large file detected (%d fixes). For best performance, we recommend splitting into batches of 500 or fewer.', 'screaming-fixes'),
                $results['ready_count']
            );
        }

        // Save bulk data for confirmation step
        $this->save_bulk_data($results);

        return $results;
    }

    /**
     * Validate action for issue type
     *
     * @param string $action The action to validate
     * @param bool $is_loop Whether this is a loop
     * @param bool $has_temp Whether this has temp redirects
     * @param bool $is_chain Whether this is a chain
     * @param array $row Full row data
     * @return array Validation result with 'valid' and 'reason'
     */
    private function validate_bulk_action($action, $is_loop, $has_temp, $is_chain, $row) {
        // Normalize action
        $action = strtolower(trim($action));

        // Valid actions
        if (!in_array($action, ['fix', 'delete'])) {
            return [
                'valid' => false,
                'reason' => sprintf(__('Invalid action "%s". Use "fix" or "delete".', 'screaming-fixes'), $action),
            ];
        }

        // Chains: only "fix" is valid
        if ($is_chain) {
            if ($action === 'delete') {
                return [
                    'valid' => false,
                    'reason' => __('Cannot use "delete" on chains. Use "fix" to update content links.', 'screaming-fixes'),
                ];
            }
            return ['valid' => true];
        }

        // Loops: only "delete" is valid
        if ($is_loop) {
            if ($action === 'fix') {
                return [
                    'valid' => false,
                    'reason' => __('Cannot use "fix" on loops. Use "delete" to remove the redirect rule.', 'screaming-fixes'),
                ];
            }
            return ['valid' => true];
        }

        // Temp redirects: "fix" or "delete" are valid
        if ($has_temp) {
            return ['valid' => true];
        }

        return ['valid' => false, 'reason' => __('Could not determine issue type.', 'screaming-fixes')];
    }

    /**
     * AJAX: Apply bulk fixes (handles batching)
     */
    public function ajax_apply_bulk_fixes() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 50;

        // Clear accumulated results at the start
        if ($offset === 0) {
            delete_transient('sf_bulk_redirects_accumulated_' . get_current_user_id());
        }

        // Get stored bulk data
        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data) || empty($bulk_data['ready_fixes'])) {
            wp_send_json_error(['message' => __('No bulk fix data found.', 'screaming-fixes')]);
        }

        $fixes = $bulk_data['ready_fixes'];
        $batch = array_slice($fixes, $offset, $batch_size);

        if (empty($batch)) {
            wp_send_json_error(['message' => __('No fixes in this batch.', 'screaming-fixes')]);
        }

        $redirect_manager = new SF_Redirect_Manager();
        $batch_results = [
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => [],
            'details' => [],
        ];

        foreach ($batch as $index => $item) {
            $batch_results['processed']++;

            $result = $this->apply_single_bulk_fix($item, $redirect_manager);

            if ($result['success']) {
                $batch_results['success']++;
                $batch_results['details'][] = [
                    'address' => $item['address'],
                    'action' => $item['action'],
                    'issue_type' => $item['issue_type'],
                    'status' => 'fixed',
                ];

                // Update the status in the original data
                $bulk_data['ready_fixes'][$offset + $index]['status'] = 'fixed';
            } else {
                $batch_results['failed']++;
                $batch_results['errors'][] = [
                    'address' => $item['address'],
                    'error' => $result['message'],
                ];

                // Update the status in the original data
                $bulk_data['ready_fixes'][$offset + $index]['status'] = 'failed';
                $bulk_data['ready_fixes'][$offset + $index]['error'] = $result['message'];
            }
        }

        // Accumulate results across batches
        $accumulated = get_transient('sf_bulk_redirects_accumulated_' . get_current_user_id());
        if (!$accumulated) {
            $accumulated = ['success' => 0, 'failed' => 0, 'details' => [], 'errors' => []];
        }

        $accumulated['success'] += $batch_results['success'];
        $accumulated['failed'] += $batch_results['failed'];
        $accumulated['details'] = array_merge($accumulated['details'], $batch_results['details']);
        $accumulated['errors'] = array_merge($accumulated['errors'], $batch_results['errors']);

        // Check if complete
        $is_complete = ($offset + $batch_size) >= count($fixes);

        if ($is_complete) {
            // Mark as complete
            $bulk_data['bulk_complete'] = true;
            $bulk_data['fixed_count'] = $accumulated['success'];
            $bulk_data['failed_count'] = $accumulated['failed'];
            $bulk_data['fixed_details'] = $accumulated['details'];
            $bulk_data['failed_details'] = $accumulated['errors'];

            // Log to activity log
            if ($accumulated['success'] > 0) {
                SF_Activity_Log::log('redirect-chains', $accumulated['success']);
            }

            // Clean up accumulated transient
            delete_transient('sf_bulk_redirects_accumulated_' . get_current_user_id());
        } else {
            set_transient('sf_bulk_redirects_accumulated_' . get_current_user_id(), $accumulated, HOUR_IN_SECONDS);
        }

        // Save updated bulk data
        $this->save_bulk_data($bulk_data);

        wp_send_json_success([
            'processed' => $batch_results['processed'],
            'success' => $batch_results['success'],
            'failed' => $batch_results['failed'],
            'complete' => $is_complete,
            'next_offset' => $offset + $batch_size,
            'total_success' => $accumulated['success'],
            'total_failed' => $accumulated['failed'],
        ]);
    }

    /**
     * Apply a single bulk fix
     *
     * @param array $item Fix item data
     * @param SF_Redirect_Manager $redirect_manager Redirect manager instance
     * @return array Result with 'success' and 'message'
     */
    private function apply_single_bulk_fix($item, $redirect_manager) {
        $action = $item['action'];
        $source = $item['address'];

        // CHAINS: fix content links
        if ($item['is_chain'] && $action === 'fix') {
            $target_url = !empty($item['override_final']) ? $item['override_final'] : $item['final_address'];

            // Get sources to fix
            $sources = $item['sources'] ?? [];

            // If no sources pre-identified, try to find them
            if (empty($sources)) {
                $source_url = $item['source_url'] ?? '';
                if (!empty($source_url)) {
                    $categorization = $this->categorize_redirect($source_url, '');
                    if ($categorization['post_id'] > 0) {
                        $sources[] = [
                            'post_id' => $categorization['post_id'],
                            'source_url' => $source_url,
                        ];
                    }
                }
            }

            if (empty($sources)) {
                return [
                    'success' => false,
                    'message' => __('No source pages found to fix.', 'screaming-fixes'),
                ];
            }

            // Apply the fix using the existing apply_fixes method
            $fixes = [[
                'address' => $source,
                'final_address' => $target_url,
                'post_ids' => array_column($sources, 'post_id'),
            ]];

            $fix_results = $this->apply_fixes($fixes);

            if ($fix_results['success'] > 0) {
                return ['success' => true, 'message' => __('Link updated in content.', 'screaming-fixes')];
            } else {
                $error_msg = !empty($fix_results['errors']) ? $fix_results['errors'][0]['error'] : __('Failed to update link.', 'screaming-fixes');
                return ['success' => false, 'message' => $error_msg];
            }
        }

        // LOOPS: delete redirect rule
        if ($item['is_loop'] && $action === 'delete') {
            $plugin = $redirect_manager->get_redirect_plugin();

            if (!$plugin) {
                return [
                    'success' => false,
                    'message' => __('No redirect plugin detected.', 'screaming-fixes'),
                ];
            }

            $result = $redirect_manager->delete_redirect($source, $plugin);

            if (is_wp_error($result)) {
                return ['success' => false, 'message' => $result->get_error_message()];
            }

            return ['success' => true, 'message' => sprintf(__('Redirect rule deleted from %s.', 'screaming-fixes'), $plugin)];
        }

        // TEMP REDIRECTS
        if ($item['has_temp_redirect']) {
            if ($action === 'delete') {
                $plugin = $redirect_manager->get_redirect_plugin();

                if (!$plugin) {
                    return [
                        'success' => false,
                        'message' => __('No redirect plugin detected.', 'screaming-fixes'),
                    ];
                }

                $result = $redirect_manager->delete_redirect($source, $plugin);

                if (is_wp_error($result)) {
                    return ['success' => false, 'message' => $result->get_error_message()];
                }

                return ['success' => true, 'message' => sprintf(__('Redirect rule deleted from %s.', 'screaming-fixes'), $plugin)];
            }

            if ($action === 'fix') {
                // Convert to permanent redirect
                $new_code = !empty($item['redirect_code']) ? intval($item['redirect_code']) : 301;

                if (!$redirect_manager->has_capability()) {
                    return [
                        'success' => false,
                        'message' => __('No redirect plugin detected.', 'screaming-fixes'),
                    ];
                }

                $result = $redirect_manager->modify_redirect($source, ['type' => $new_code]);

                if (!$result['success']) {
                    return ['success' => false, 'message' => $result['message']];
                }

                return ['success' => true, 'message' => sprintf(__('Converted to %d redirect.', 'screaming-fixes'), $new_code)];
            }
        }

        return ['success' => false, 'message' => __('Unknown action or issue type.', 'screaming-fixes')];
    }

    /**
     * Save bulk data to database
     *
     * @param array $data Bulk data to save
     */
    private function save_bulk_data($data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();

        // Clear any existing data
        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        // Save new data
        $wpdb->insert($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
            'data' => wp_json_encode($data),
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours')),
        ]);
    }

    /**
     * Get stored bulk data
     *
     * @return array|null Bulk data or null
     */
    private function get_bulk_data() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();

        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT data FROM {$table_name}
             WHERE session_id = %s AND module = %s AND expires_at > NOW()
             ORDER BY created_at DESC LIMIT 1",
            $session_id,
            $this->slug
        ));

        if ($row) {
            $data = json_decode($row->data, true);
            if (isset($data['is_bulk_update']) && $data['is_bulk_update']) {
                return $data;
            }
        }

        return null;
    }

    /**
     * AJAX: Cancel bulk operation
     */
    public function ajax_cancel_bulk() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();

        $wpdb->delete($table_name, [
            'session_id' => $session_id,
            'module' => $this->slug,
        ]);

        delete_transient('sf_bulk_redirects_accumulated_' . get_current_user_id());

        wp_send_json_success([
            'message' => __('Bulk operation cancelled.', 'screaming-fixes'),
        ]);
    }

    /**
     * AJAX: Download bulk preview CSV
     */
    public function ajax_download_preview() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data) || empty($bulk_data['ready_fixes'])) {
            wp_die(__('No bulk data found.', 'screaming-fixes'));
        }

        $export_data = [];
        foreach ($bulk_data['ready_fixes'] as $item) {
            $export_data[] = [
                'Address' => $item['address'],
                'Final Address' => $item['final_address'],
                'Issue Type' => ucfirst($item['issue_type']),
                'Action' => $item['action'],
                'Override Final' => $item['override_final'] ?? '',
                'Redirect Code' => $item['redirect_code'] ?? '',
                'Status' => 'Ready',
            ];
        }

        // Add invalid items
        foreach ($bulk_data['invalid_fixes'] ?? [] as $item) {
            $export_data[] = [
                'Address' => $item['address'],
                'Final Address' => $item['final_address'] ?? '',
                'Issue Type' => ucfirst($item['issue_type'] ?? ''),
                'Action' => $item['action'] ?? '',
                'Override Final' => '',
                'Redirect Code' => '',
                'Status' => 'Invalid: ' . ($item['reason'] ?? ''),
            ];
        }

        // Add skipped items
        foreach ($bulk_data['skipped_fixes'] ?? [] as $item) {
            $export_data[] = [
                'Address' => $item['address'],
                'Final Address' => $item['final_address'] ?? '',
                'Issue Type' => ucfirst($item['issue_type'] ?? ''),
                'Action' => '',
                'Override Final' => '',
                'Redirect Code' => '',
                'Status' => 'Skipped: ' . ($item['reason'] ?? ''),
            ];
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="redirect-chains-bulk-preview-' . date('Y-m-d') . '.csv"');
        header('Content-Length: ' . strlen($csv_content));
        header('Cache-Control: no-cache, no-store, must-revalidate');

        echo $csv_content;
        exit;
    }

    /**
     * AJAX: Download bulk results CSV
     */
    public function ajax_download_results() {
        check_ajax_referer('sf_redirect_chains_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(__('Permission denied.', 'screaming-fixes'));
        }

        $bulk_data = $this->get_bulk_data();

        if (empty($bulk_data)) {
            wp_die(__('No bulk data found.', 'screaming-fixes'));
        }

        $export_data = [];

        // Add all ready_fixes with their final status
        foreach ($bulk_data['ready_fixes'] ?? [] as $item) {
            $status = $item['status'] ?? 'unknown';
            $error = $item['error'] ?? '';

            $export_data[] = [
                'Address' => $item['address'],
                'Final Address' => $item['final_address'],
                'Issue Type' => ucfirst($item['issue_type']),
                'Action' => $item['action'],
                'Status' => ucfirst($status),
                'Error' => $error,
            ];
        }

        // Add invalid items
        foreach ($bulk_data['invalid_fixes'] ?? [] as $item) {
            $export_data[] = [
                'Address' => $item['address'],
                'Final Address' => $item['final_address'] ?? '',
                'Issue Type' => ucfirst($item['issue_type'] ?? ''),
                'Action' => $item['action'] ?? '',
                'Status' => 'Invalid',
                'Error' => $item['reason'] ?? '',
            ];
        }

        // Add skipped items
        foreach ($bulk_data['skipped_fixes'] ?? [] as $item) {
            $export_data[] = [
                'Address' => $item['address'],
                'Final Address' => $item['final_address'] ?? '',
                'Issue Type' => ucfirst($item['issue_type'] ?? ''),
                'Action' => '',
                'Status' => 'Skipped',
                'Error' => $item['reason'] ?? '',
            ];
        }

        $parser = new SF_CSV_Parser();
        $csv_content = $parser->export_to_csv($export_data);

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="redirect-chains-bulk-results-' . date('Y-m-d') . '.csv"');
        header('Content-Length: ' . strlen($csv_content));
        header('Cache-Control: no-cache, no-store, must-revalidate');

        echo $csv_content;
        exit;
    }
}
