<?php
/**
 * Redirect Chains tab content view
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Helper function to truncate URLs for display
 * Define at top of file so it's available before use
 */
if (!function_exists('sf_truncate_url')) {
    function sf_truncate_url($url, $max_length = 50) {
        if (strlen($url) <= $max_length) {
            return $url;
        }

        $parsed = wp_parse_url($url);
        $path = $parsed['path'] ?? '';

        // Show domain + truncated path
        $domain = ($parsed['host'] ?? '');
        $remaining = $max_length - strlen($domain) - 3; // 3 for "..."

        if ($remaining > 10) {
            return $domain . substr($path, 0, $remaining) . '...';
        }

        return substr($url, 0, $max_length - 3) . '...';
    }
}

// Get module instance and data
$module = SF_Module_Loader::get_module('redirect-chains');
$results = $module ? $module->get_results() : null;
$processing_error = null;
$pending_row_count = 0;

// Check for upload data if no transient results
if (empty($results) || empty($results['redirects'])) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
    $session_id = 'user_' . get_current_user_id();

    $row = $wpdb->get_row($wpdb->prepare(
        "SELECT data FROM {$table_name}
         WHERE session_id = %s AND module = %s AND expires_at > NOW()
         ORDER BY created_at DESC LIMIT 1",
        $session_id,
        'redirect-chains'
    ));

    if ($row) {
        $stored_data = json_decode($row->data, true);

        // Check if this is raw CSV data vs processed results
        if (isset($stored_data['rows']) && !isset($stored_data['redirects'])) {
            // Raw CSV data - need to process it
            $pending_row_count = count($stored_data['rows'] ?? []);

            // Check if the file is too large to process reliably
            if ($pending_row_count > 5000) {
                $processing_error = sprintf(
                    __('Your CSV file contains %s rows, which is too large to process in a single request. Please export a smaller subset from Screaming Frog (under 5,000 rows) or filter your export to focus on specific redirect types.', 'screaming-fixes'),
                    number_format($pending_row_count)
                );
            } elseif ($module) {
                $pending_uploads = get_option('sf_pending_uploads', []);

                foreach (array_reverse($pending_uploads) as $upload_id => $upload_info) {
                    if ($upload_info['module'] === 'redirect-chains' && file_exists($upload_info['path'])) {
                        // Increase limits for processing
                        @set_time_limit(300);
                        @ini_set('memory_limit', '512M');

                        $results = $module->process_csv($upload_info['path']);

                        if (is_wp_error($results)) {
                            $processing_error = $results->get_error_message();
                        } elseif (!empty($results)) {
                            $wpdb->update(
                                $table_name,
                                ['data' => wp_json_encode($results)],
                                ['session_id' => $session_id, 'module' => 'redirect-chains'],
                                ['%s'],
                                ['%s', '%s']
                            );
                        } else {
                            $processing_error = __('Processing completed but no redirect issues were found. The CSV may not contain valid redirect chain data.', 'screaming-fixes');
                        }
                        break;
                    }
                }

                // If no pending upload found
                if (empty($results) && empty($processing_error)) {
                    $processing_error = __('The uploaded CSV file could not be found. Please try uploading again.', 'screaming-fixes');
                }
            }
        } else {
            $results = $stored_data;
        }
    }
}

// Check if this is a bulk update result
$is_bulk_update = !empty($results['is_bulk_update']);
$bulk_complete = !empty($results['bulk_complete']);

// Bulk update specific data
$bulk_ready_fixes = $is_bulk_update ? ($results['ready_fixes'] ?? []) : [];
$bulk_invalid_fixes = $is_bulk_update ? ($results['invalid_fixes'] ?? []) : [];
$bulk_skipped_fixes = $is_bulk_update ? ($results['skipped_fixes'] ?? []) : [];
$bulk_warnings = $is_bulk_update ? ($results['warnings'] ?? []) : [];
$bulk_ready_count = $is_bulk_update ? ($results['ready_count'] ?? count($bulk_ready_fixes)) : 0;
$bulk_invalid_count = $is_bulk_update ? ($results['invalid_count'] ?? count($bulk_invalid_fixes)) : 0;
$bulk_skipped_count = $is_bulk_update ? ($results['skipped_count'] ?? count($bulk_skipped_fixes)) : 0;

// Bulk complete results
$bulk_fixed_count = $bulk_complete ? ($results['fixed_count'] ?? 0) : 0;
$bulk_failed_count = $bulk_complete ? ($results['failed_count'] ?? 0) : 0;

// Check if we have valid bulk data to display
$has_valid_bulk_data = false;
if ($is_bulk_update) {
    if (!$bulk_complete && $bulk_ready_count > 0) {
        $has_valid_bulk_data = true;
    }
    if ($bulk_complete && ($bulk_fixed_count > 0 || $bulk_failed_count > 0)) {
        $has_valid_bulk_data = true;
    }

    // If bulk update mode but no valid data, clear everything
    if (!$has_valid_bulk_data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'screaming_fixes_uploads';
        $session_id = 'user_' . get_current_user_id();
        $wpdb->delete($table_name, ['session_id' => $session_id, 'module' => 'redirect-chains']);
        delete_transient('sf_redirect-chains_results');
        $results = null;
        $is_bulk_update = false;
        $bulk_complete = false;
    }
}

// Check if we have any data (redirects OR fixed redirects - data persists after fixes)
$has_results = !empty($results) && !$is_bulk_update;
$has_redirects = $has_results && !empty($results['redirects']);
$has_fixed_redirects = $has_results && !empty($results['fixed_redirects']);
$has_data = $has_redirects || $has_fixed_redirects || $has_valid_bulk_data;

// Get categorized redirects
$fixable_redirects = $has_results ? ($results['fixable_redirects'] ?? []) : [];
$manual_redirects = $has_results ? ($results['manual_redirects'] ?? []) : [];
$fixed_redirects = $has_results ? ($results['fixed_redirects'] ?? []) : [];
$fixed_count = count($fixed_redirects);

// Count fixed redirects by status
$fixed_success_count = 0;
$fixed_failed_count = 0;
$fixed_skipped_count = 0;
foreach ($fixed_redirects as $fr) {
    $fr_status = $fr['status'] ?? 'success';
    if ($fr_status === 'success') $fixed_success_count++;
    elseif ($fr_status === 'failed') $fixed_failed_count++;
    else $fixed_skipped_count++;
}

// Count internal vs external for filters
$internal_count = 0;
$external_count = 0;
foreach ($fixable_redirects as $redirect) {
    if (($redirect['redirect_type']['class'] ?? '') === 'sf-type-internal') {
        $internal_count++;
    } else {
        $external_count++;
    }
}

// Count by issue type for filters (Chains, Loops, Temp Redirects)
$chains_count = 0;
$loops_count = 0;
$temp_count = 0;
foreach ($fixable_redirects as $redirect) {
    if (!empty($redirect['is_loop'])) {
        $loops_count++;
    } elseif (!empty($redirect['has_temp_redirect'])) {
        $temp_count++;
    } else {
        $chains_count++;
    }
}
$total_issue_count = count($fixable_redirects);

// Pagination settings
$per_page = 100;
$current_page_fixable = isset($_GET['paged_fixable']) ? max(1, intval($_GET['paged_fixable'])) : 1;
$current_page_manual = isset($_GET['paged_manual']) ? max(1, intval($_GET['paged_manual'])) : 1;

// Calculate pagination for fixable
$total_fixable_items = count($fixable_redirects);
$total_fixable_pages = ceil($total_fixable_items / $per_page);
$fixable_offset = ($current_page_fixable - 1) * $per_page;
$fixable_to_display = array_slice($fixable_redirects, $fixable_offset, $per_page);

// Calculate pagination for manual
$total_manual_items = count($manual_redirects);
$total_manual_pages = ceil($total_manual_items / $per_page);
$manual_offset = ($current_page_manual - 1) * $per_page;
$manual_to_display = array_slice($manual_redirects, $manual_offset, $per_page);
?>

<div class="sf-redirect-chains-module" id="sf-redirect-chains">

    <!-- Module Header -->
    <div class="sf-module-header">
        <div class="sf-module-title">
            <span class="sf-module-icon">&#128260;</span>
            <h2><?php esc_html_e('Redirect Chains', 'screaming-fixes'); ?></h2>
            <span class="sf-module-tagline"><?php esc_html_e('Fix chains, loops & temp redirects from one CSV', 'screaming-fixes'); ?></span>
        </div>
        <?php if ($has_data): ?>
        <div class="sf-module-stats">
            <span class="sf-stat sf-stat-total">
                <strong><?php echo esc_html(count($fixable_redirects) + count($manual_redirects)); ?></strong>
                <?php esc_html_e('redirect issues', 'screaming-fixes'); ?>
            </span>
            <?php if ($fixed_count > 0): ?>
            <span class="sf-stat sf-stat-success">
                <strong><?php echo esc_html($fixed_count); ?></strong>
                <?php esc_html_e('fixed', 'screaming-fixes'); ?>
            </span>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>

    <!-- Instructions (collapsible) -->
    <?php include __DIR__ . '/instructions.php'; ?>

    <?php if (!empty($processing_error)) : ?>
    <!-- Processing Error Message -->
    <div class="sf-notice sf-notice-error sf-processing-error">
        <div class="sf-notice-icon">&#9888;</div>
        <div class="sf-notice-content">
            <strong><?php esc_html_e('Unable to Process CSV', 'screaming-fixes'); ?></strong>
            <p><?php echo esc_html($processing_error); ?></p>
            <div class="sf-notice-actions">
                <strong><?php esc_html_e('What you can do:', 'screaming-fixes'); ?></strong>
                <ul>
                    <li><?php esc_html_e('In Screaming Frog, use filters to export only specific redirect types (e.g., only internal redirects)', 'screaming-fixes'); ?></li>
                    <li><?php esc_html_e('Export redirects from specific URL paths or sections of your site', 'screaming-fixes'); ?></li>
                    <li><?php esc_html_e('Sort by "Number of Redirects" and export only the worst offenders', 'screaming-fixes'); ?></li>
                </ul>
                <button type="button" class="sf-button sf-button-secondary" id="sf-clear-redirects">
                    <span class="dashicons dashicons-trash"></span>
                    <?php esc_html_e('Clear & Upload New File', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($is_bulk_update && !$bulk_complete && $has_valid_bulk_data): ?>
    <!-- ==========================================
         BULK FIX CONFIRMATION UI
         ========================================== -->
    <div class="sf-bulk-confirmation" id="sf-bulk-confirmation">
        <div class="sf-bulk-summary">
            <div class="sf-bulk-icon">&#128260;</div>
            <h3><?php esc_html_e('Bulk Redirect Fixes', 'screaming-fixes'); ?></h3>

            <div class="sf-bulk-stats">
                <div class="sf-bulk-stat sf-bulk-stat-ready">
                    <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_ready_count); ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('redirect fixes ready to apply', 'screaming-fixes'); ?></span>
                </div>

                <?php if ($bulk_invalid_count > 0): ?>
                <div class="sf-bulk-stat sf-bulk-stat-warning">
                    <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_invalid_count); ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('invalid (action doesn\'t match issue type)', 'screaming-fixes'); ?></span>
                </div>
                <?php endif; ?>

                <?php if ($bulk_skipped_count > 0): ?>
                <div class="sf-bulk-stat sf-bulk-stat-skipped">
                    <span class="sf-bulk-stat-number"><?php echo esc_html($bulk_skipped_count); ?></span>
                    <span class="sf-bulk-stat-label"><?php esc_html_e('skipped (no action specified)', 'screaming-fixes'); ?></span>
                </div>
                <?php endif; ?>
            </div>

            <?php if (!empty($bulk_warnings)): ?>
                <?php foreach ($bulk_warnings as $warning): ?>
                <div class="sf-bulk-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <?php echo esc_html($warning); ?>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>

            <div class="sf-bulk-actions">
                <button type="button" class="sf-button sf-button-secondary sf-bulk-download-preview" id="sf-bulk-download-preview">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Download Preview CSV', 'screaming-fixes'); ?>
                </button>
                <div class="sf-bulk-buttons">
                    <button type="button" class="sf-button sf-button-secondary sf-bulk-clear-btn" id="sf-bulk-clear">
                        <span class="dashicons dashicons-trash"></span>
                        <?php esc_html_e('Clear', 'screaming-fixes'); ?>
                    </button>
                    <button type="button" class="sf-button sf-button-primary sf-bulk-confirm" id="sf-bulk-confirm" data-count="<?php echo esc_attr($bulk_ready_count); ?>">
                        <?php esc_html_e('Confirm & Apply Fixes', 'screaming-fixes'); ?>
                    </button>
                </div>
            </div>
        </div>

        <!-- Preview Table -->
        <div class="sf-bulk-preview">
            <h4><?php esc_html_e('Preview', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, $bulk_ready_count)); ?> of <?php echo esc_html($bulk_ready_count); ?>)</span></h4>
            <div class="sf-table-wrapper">
                <table class="sf-results-table sf-bulk-preview-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Address', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Issue Type', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Action', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Target', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        // Show first 10 ready fixes
                        $preview_items = array_slice($bulk_ready_fixes, 0, 10);
                        foreach ($preview_items as $item):
                            $address_display = strlen($item['address']) > 40 ? substr($item['address'], 0, 37) . '...' : $item['address'];
                            $target_display = '';
                            if ($item['action'] === 'fix' && $item['is_chain']) {
                                $target = !empty($item['override_final']) ? $item['override_final'] : $item['final_address'];
                                $target_display = '→ ' . (strlen($target) > 30 ? substr($target, 0, 27) . '...' : $target);
                            } elseif ($item['action'] === 'fix' && $item['has_temp_redirect']) {
                                $code = !empty($item['redirect_code']) ? $item['redirect_code'] : '301';
                                $target_display = sprintf(__('Convert to %s', 'screaming-fixes'), $code);
                            } elseif ($item['action'] === 'delete') {
                                $target_display = __('Delete rule', 'screaming-fixes');
                            }
                            $issue_class = 'sf-type-' . $item['issue_type'];
                        ?>
                        <tr class="sf-bulk-row sf-bulk-row-ready">
                            <td class="sf-col-address">
                                <span class="sf-url-address" title="<?php echo esc_attr($item['address']); ?>"><?php echo esc_html($address_display); ?></span>
                            </td>
                            <td class="sf-col-type">
                                <span class="sf-type-badge <?php echo esc_attr($issue_class); ?>"><?php echo esc_html(ucfirst($item['issue_type'])); ?></span>
                            </td>
                            <td class="sf-col-action"><?php echo esc_html(ucfirst($item['action'])); ?></td>
                            <td class="sf-col-target"><?php echo esc_html($target_display); ?></td>
                            <td class="sf-col-status"><span class="sf-status-badge sf-status-ready"><?php esc_html_e('Ready', 'screaming-fixes'); ?></span></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($bulk_invalid_count > 0): ?>
            <h4 class="sf-mt-20"><?php esc_html_e('Invalid Rows', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html($bulk_invalid_count); ?>)</span></h4>
            <div class="sf-table-wrapper">
                <table class="sf-results-table sf-bulk-preview-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Address', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Action', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Reason', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $invalid_preview = array_slice($bulk_invalid_fixes, 0, 5);
                        foreach ($invalid_preview as $item):
                        ?>
                        <tr class="sf-bulk-row sf-bulk-row-invalid">
                            <td class="sf-col-address"><?php echo esc_html($item['address']); ?></td>
                            <td class="sf-col-action"><?php echo esc_html($item['action'] ?? ''); ?></td>
                            <td class="sf-col-reason"><?php echo esc_html($item['reason'] ?? ''); ?></td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if ($bulk_invalid_count > 5): ?>
                        <tr class="sf-bulk-row-more">
                            <td colspan="3"><?php printf(esc_html__('...and %d more invalid rows', 'screaming-fixes'), $bulk_invalid_count - 5); ?></td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Bulk Progress Modal (hidden by default) -->
    <div class="sf-bulk-progress-modal sf-modal" id="sf-bulk-progress-modal" style="display: none;">
        <div class="sf-modal-content sf-bulk-progress-content">
            <div class="sf-bulk-progress-header">
                <span class="sf-bulk-progress-icon sf-spinning">&#8634;</span>
                <h3><?php esc_html_e('Applying Redirect Fixes...', 'screaming-fixes'); ?></h3>
            </div>
            <div class="sf-bulk-progress-bar">
                <div class="sf-bulk-progress-fill" id="sf-bulk-progress-fill" style="width: 0%;"></div>
            </div>
            <div class="sf-bulk-progress-stats">
                <span class="sf-bulk-progress-current" id="sf-bulk-progress-current">0</span>
                <span class="sf-bulk-progress-separator">/</span>
                <span class="sf-bulk-progress-total" id="sf-bulk-progress-total">0</span>
                <span class="sf-bulk-progress-label"><?php esc_html_e('complete', 'screaming-fixes'); ?></span>
            </div>
            <div class="sf-bulk-progress-url" id="sf-bulk-progress-url">
                <?php esc_html_e('Preparing...', 'screaming-fixes'); ?>
            </div>
        </div>
    </div>

    <?php elseif ($is_bulk_update && $bulk_complete && $has_valid_bulk_data): ?>
    <!-- ==========================================
         BULK FIX COMPLETE RESULTS
         ========================================== -->
    <div class="sf-bulk-complete" id="sf-bulk-complete">
        <div class="sf-bulk-complete-summary <?php echo $bulk_failed_count > 0 ? 'sf-has-failures' : ''; ?>">
            <div class="sf-bulk-complete-icon">
                <?php echo $bulk_failed_count > 0 ? '&#9888;' : '&#10004;'; ?>
            </div>
            <h3>
                <?php
                if ($bulk_failed_count > 0) {
                    printf(
                        esc_html__('%d redirect fixes applied successfully. %d failed - see details below.', 'screaming-fixes'),
                        $bulk_fixed_count,
                        $bulk_failed_count
                    );
                } else {
                    printf(
                        esc_html__('%d redirect fixes applied successfully.', 'screaming-fixes'),
                        $bulk_fixed_count
                    );
                }
                ?>
                <?php if ($bulk_invalid_count > 0 || $bulk_skipped_count > 0): ?>
                    <span class="sf-bulk-skipped-note">
                        <?php printf(esc_html__('%d rows skipped.', 'screaming-fixes'), $bulk_invalid_count + $bulk_skipped_count); ?>
                    </span>
                <?php endif; ?>
            </h3>

            <div class="sf-bulk-complete-actions">
                <button type="button" class="sf-button sf-button-secondary sf-bulk-download-results" id="sf-bulk-download-results">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Download Full Results CSV', 'screaming-fixes'); ?>
                </button>
                <button type="button" class="sf-button sf-button-primary sf-bulk-clear-btn" id="sf-bulk-new-upload">
                    <?php esc_html_e('Clear & Start New', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <!-- Results Table (first 10) -->
        <?php
        $fixed_items = array_filter($bulk_ready_fixes, function($item) {
            return ($item['status'] ?? '') === 'fixed';
        });
        $failed_items = array_filter($bulk_ready_fixes, function($item) {
            return ($item['status'] ?? '') === 'failed';
        });
        ?>
        <?php if (!empty($fixed_items)): ?>
        <div class="sf-bulk-results-table">
            <h4><?php esc_html_e('Fixed', 'screaming-fixes'); ?> <span class="sf-preview-count">(<?php echo esc_html(min(10, count($fixed_items))); ?> of <?php echo esc_html(count($fixed_items)); ?>)</span></h4>
            <div class="sf-table-wrapper">
                <table class="sf-results-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Address', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Issue Type', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Action', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $results_preview = array_slice($fixed_items, 0, 10);
                        foreach ($results_preview as $item):
                            $address_display = strlen($item['address']) > 50 ? substr($item['address'], 0, 47) . '...' : $item['address'];
                        ?>
                        <tr class="sf-bulk-row sf-bulk-row-success">
                            <td class="sf-col-address" title="<?php echo esc_attr($item['address']); ?>"><?php echo esc_html($address_display); ?></td>
                            <td class="sf-col-type"><?php echo esc_html(ucfirst($item['issue_type'])); ?></td>
                            <td class="sf-col-action"><?php echo esc_html(ucfirst($item['action'])); ?></td>
                            <td class="sf-col-status"><span class="sf-status-badge sf-status-success"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Failed Updates -->
        <?php if (!empty($failed_items)): ?>
        <div class="sf-bulk-failed-table">
            <h4 class="sf-mt-20"><?php esc_html_e('Failed', 'screaming-fixes'); ?></h4>
            <div class="sf-table-wrapper">
                <table class="sf-results-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Address', 'screaming-fixes'); ?></th>
                            <th><?php esc_html_e('Error', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($failed_items as $item): ?>
                        <tr class="sf-bulk-row sf-bulk-row-failed">
                            <td class="sf-col-address"><?php echo esc_html($item['address']); ?></td>
                            <td class="sf-col-error"><?php echo esc_html($item['error'] ?? __('Unknown error', 'screaming-fixes')); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <?php elseif (!$has_data) : ?>

    <!-- No data yet - show upload box prominently -->
    <div class="sf-upload-section">
        <h3>
            <span class="sf-section-icon">&#128228;</span>
            <?php esc_html_e('Upload Redirect Chains CSV', 'screaming-fixes'); ?>
        </h3>

        <?php
        $args = [
            'module' => 'redirect-chains',
            'title' => __('Drop your Redirect Chains CSV', 'screaming-fixes'),
            'description' => __('Export from Screaming Frog: Reports → Redirect Chains', 'screaming-fixes'),
            'max_rows' => 5000,
        ];
        include SF_PLUGIN_DIR . 'admin/views/partials/upload-box.php';
        ?>
    </div>

    <!-- Empty State (only if no processing error) -->
    <?php if (empty($processing_error)) : ?>
    <div class="sf-empty-state">
        <div class="sf-empty-icon">&#128260;</div>
        <h3><?php esc_html_e('No redirect data yet', 'screaming-fixes'); ?></h3>
        <p><?php esc_html_e('Upload a Screaming Frog Redirect Chains CSV to find and fix redirect chains, loops, and temporary redirects on your site.', 'screaming-fixes'); ?></p>
    </div>
    <?php endif; ?>

    <?php else : ?>

    <!-- Has data - show stats and results first -->

    <!-- Stats Summary -->
    <div class="sf-stats-summary">
        <div class="sf-stat-card">
            <span class="sf-stat-number"><?php echo esc_html($results['total_count']); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Redirect Issues', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-stat-card sf-stat-fixable">
            <span class="sf-stat-number"><?php echo esc_html($results['fixable_count']); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Auto-Fixable', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-stat-card sf-stat-manual">
            <span class="sf-stat-number"><?php echo esc_html($results['manual_count']); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Manual Fix', 'screaming-fixes'); ?></span>
        </div>
        <div class="sf-stat-card">
            <span class="sf-stat-number"><?php echo esc_html($results['total_sources']); ?></span>
            <span class="sf-stat-label"><?php esc_html_e('Total Links', 'screaming-fixes'); ?></span>
        </div>
    </div>

    <!-- Action Bar -->
    <div class="sf-action-bar">
        <div class="sf-action-left">
            <!-- Simplified - removed Select All and counter (moved to bottom of section) -->
        </div>
        <div class="sf-action-right">
            <button type="button" class="sf-button sf-button-secondary sf-clear-data-btn" id="sf-clear-redirects" title="<?php esc_attr_e('Clear data and upload new CSV', 'screaming-fixes'); ?>">
                <span class="dashicons dashicons-trash"></span>
                <?php esc_html_e('Clear', 'screaming-fixes'); ?>
            </button>
            <button type="button" class="sf-button sf-button-secondary" id="sf-export-redirects">
                <span class="dashicons dashicons-download"></span>
                <?php esc_html_e('Export', 'screaming-fixes'); ?>
            </button>
            <?php if ($fixed_count > 0): ?>
            <button type="button" class="sf-button sf-button-secondary sf-fixed-btn" id="sf-view-fixed-redirects" title="<?php esc_attr_e('View fixed redirects', 'screaming-fixes'); ?>">
                <span class="dashicons dashicons-yes-alt"></span>
                <?php printf(esc_html__('Fixed (%d)', 'screaming-fixes'), $fixed_count); ?>
            </button>
            <?php endif; ?>
        </div>
    </div>

    <!-- Results Sections -->
    <?php if (!empty($fixable_redirects)) : ?>
    <div class="sf-results-section sf-fixable-section">
        <button type="button" class="sf-section-toggle sf-fixable-toggle" aria-expanded="true">
            <span class="sf-section-badge sf-badge-fixable">&#10003;</span>
            <?php printf(esc_html__('%d Auto-Fixable Redirects', 'screaming-fixes'), $total_fixable_items); ?>
            <span class="sf-section-hint"><?php esc_html_e('Chains, loops & temp redirects that can be automatically fixed', 'screaming-fixes'); ?></span>
            <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
        </button>

        <div class="sf-fixable-content">
            <p class="sf-section-description">
                <?php esc_html_e('Clicking "Fix" or "Fix Selected" will replace the Redirect URL with the Final Destination URL on all affected pages. This eliminates redirect hops and sends visitors directly to the correct page.', 'screaming-fixes'); ?>
            </p>

            <!-- Filter By: Issue Type -->
            <div class="sf-filter-stats-row">
                <span class="sf-filter-label"><?php esc_html_e('Filter By:', 'screaming-fixes'); ?></span>
                <div class="sf-filter-stats">
                    <div class="sf-filter-stat sf-filter-all active" data-filter="all" title="<?php esc_attr_e('Show all redirect issues', 'screaming-fixes'); ?>">
                        <span class="sf-filter-icon">&#128203;</span>
                        <span class="sf-filter-count"><?php echo esc_html($total_issue_count); ?></span>
                        <span class="sf-filter-name"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                    </div>
                    <div class="sf-filter-stat sf-filter-chains<?php echo $chains_count === 0 ? ' sf-filter-disabled' : ''; ?>" data-filter="chains" title="<?php esc_attr_e('Links pointing to URLs that redirect through multiple hops', 'screaming-fixes'); ?>">
                        <span class="sf-filter-icon">&#9939;</span>
                        <span class="sf-filter-count"><?php echo esc_html($chains_count); ?></span>
                        <span class="sf-filter-name"><?php esc_html_e('Chains', 'screaming-fixes'); ?></span>
                    </div>
                    <div class="sf-filter-stat sf-filter-loops<?php echo $loops_count === 0 ? ' sf-filter-disabled' : ''; ?>" data-filter="loops" title="<?php esc_attr_e('Redirect loops that break pages (A → B → A)', 'screaming-fixes'); ?>">
                        <span class="sf-filter-icon">&#128257;</span>
                        <span class="sf-filter-count"><?php echo esc_html($loops_count); ?></span>
                        <span class="sf-filter-name"><?php esc_html_e('Loops', 'screaming-fixes'); ?></span>
                    </div>
                    <div class="sf-filter-stat sf-filter-temp<?php echo $temp_count === 0 ? ' sf-filter-disabled' : ''; ?>" data-filter="temp" title="<?php esc_attr_e('Temporary redirects (302/307) that should be permanent', 'screaming-fixes'); ?>">
                        <span class="sf-filter-icon">&#9203;</span>
                        <span class="sf-filter-count"><?php echo esc_html($temp_count); ?></span>
                        <span class="sf-filter-name"><?php esc_html_e('Temp (302)', 'screaming-fixes'); ?></span>
                    </div>
                </div>
            </div>

            <!-- Issue Type Descriptions -->
            <div class="sf-filter-description" id="sf-filter-description">
                <div class="sf-filter-desc-content" data-filter="all">
                    <span class="sf-filter-desc-icon">&#128161;</span>
                    <span class="sf-filter-desc-text"><?php esc_html_e('Showing all redirect issues. Use the filters above to focus on specific issue types.', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-desc-content sf-filter-desc-chains" data-filter="chains" style="display: none;">
                    <span class="sf-filter-desc-icon">&#9939;</span>
                    <span class="sf-filter-desc-text"><?php esc_html_e('Links pointing to URLs that redirect. Fixing updates the link to point directly to the final destination, eliminating redirect hops.', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-desc-content sf-filter-desc-loops" data-filter="loops" style="display: none;">
                    <span class="sf-filter-desc-icon">&#128257;</span>
                    <span class="sf-filter-desc-text"><?php esc_html_e('Redirect loops occur when URL A → URL B → URL A. These break pages entirely. Fixing deletes the redirect rule causing the loop.', 'screaming-fixes'); ?></span>
                </div>
                <div class="sf-filter-desc-content sf-filter-desc-temp" data-filter="temp" style="display: none;">
                    <span class="sf-filter-desc-icon">&#9203;</span>
                    <span class="sf-filter-desc-text"><?php esc_html_e('Temporary redirects (302/307) don\'t pass full SEO value to the destination. Convert to 301 if the redirect is permanent.', 'screaming-fixes'); ?></span>
                </div>
            </div>

            <!-- Filter By: Location Type -->
            <div class="sf-filter-row">
                <span class="sf-filter-label"><?php esc_html_e('Filter By:', 'screaming-fixes'); ?></span>
                <div class="sf-filter-buttons">
                    <button type="button" class="sf-filter-btn" data-filter="internal">
                        <span class="sf-filter-icon">&#127968;</span>
                        <span class="sf-filter-text"><?php esc_html_e('Internal', 'screaming-fixes'); ?></span>
                        <span class="sf-filter-count"><?php echo esc_html($internal_count); ?></span>
                    </button>
                    <button type="button" class="sf-filter-btn" data-filter="external">
                        <span class="sf-filter-icon">&#127760;</span>
                        <span class="sf-filter-text"><?php esc_html_e('External', 'screaming-fixes'); ?></span>
                        <span class="sf-filter-count"><?php echo esc_html($external_count); ?></span>
                    </button>
                </div>
            </div>

            <div class="sf-results-table-wrapper">
            <table class="sf-results-table">
                <thead>
                    <tr>
                        <th class="sf-col-checkbox">
                            <input type="checkbox" id="sf-select-all-table" title="<?php esc_attr_e('Select All', 'screaming-fixes'); ?>" />
                        </th>
                        <th class="sf-col-type"><?php esc_html_e('Type', 'screaming-fixes'); ?></th>
                        <th class="sf-col-redirect"><?php esc_html_e('Redirect URL', 'screaming-fixes'); ?></th>
                        <th class="sf-col-final"><?php esc_html_e('Final Destination', 'screaming-fixes'); ?></th>
                        <th class="sf-col-hops"><?php esc_html_e('Hops', 'screaming-fixes'); ?></th>
                        <th class="sf-col-count"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                        <th class="sf-col-actions"><?php esc_html_e('Actions', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($fixable_to_display as $index => $redirect) :
                        $actual_index = $fixable_offset + $index;
                        $is_loop = !empty($redirect['is_loop']);
                        $has_temp = !empty($redirect['has_temp_redirect']);
                        $is_chain = !$is_loop && !$has_temp;

                        // Check if this is an external redirect (can't fix temp redirects on external sites)
                        $is_external = ($redirect['redirect_type']['type'] ?? 'external') !== 'internal';
                        $is_external_temp = $has_temp && $is_external;

                        // Determine row class based on issue type
                        $row_class = 'sf-redirect-row';
                        if ($is_loop) {
                            $row_class .= ' sf-row-danger sf-issue-loop';
                        } elseif ($has_temp) {
                            $row_class .= ' sf-row-warning sf-issue-temp';
                            if ($is_external_temp) {
                                $row_class .= ' sf-external-temp';
                            }
                        } else {
                            $row_class .= ' sf-issue-chain';
                        }
                    ?>
                    <tr class="<?php echo esc_attr($row_class); ?>" data-address="<?php echo esc_attr($redirect['address']); ?>" data-final="<?php echo esc_attr($redirect['final_address']); ?>" data-is-loop="<?php echo $is_loop ? 'true' : 'false'; ?>" data-has-temp="<?php echo $has_temp ? 'true' : 'false'; ?>">
                        <td class="sf-col-checkbox">
                            <?php if ($is_external_temp) : ?>
                            <input type="checkbox" class="sf-redirect-checkbox" data-index="<?php echo esc_attr($actual_index); ?>" disabled title="<?php esc_attr_e('External redirects cannot be fixed with this tool', 'screaming-fixes'); ?>" />
                            <?php else : ?>
                            <input type="checkbox" class="sf-redirect-checkbox" data-index="<?php echo esc_attr($actual_index); ?>" />
                            <?php endif; ?>
                        </td>
                        <td class="sf-col-type">
                            <span class="sf-type-badge <?php echo esc_attr($redirect['redirect_type']['class'] ?? 'sf-type-external'); ?>">
                                <?php echo esc_html($redirect['redirect_type']['label'] ?? __('External', 'screaming-fixes')); ?>
                            </span>
                            <?php if ($is_loop) : ?>
                            <span class="sf-issue-badge sf-loop-badge">&#128257; <?php esc_html_e('Loop', 'screaming-fixes'); ?></span>
                            <?php elseif ($has_temp) : ?>
                            <span class="sf-issue-badge sf-temp-badge">&#9203; <?php echo esc_html($redirect['status_code'] ?: '302'); ?></span>
                            <?php endif; ?>
                        </td>
                        <td class="sf-col-redirect">
                            <div class="sf-url-cell">
                                <span class="sf-url-text" title="<?php echo esc_attr($redirect['address']); ?>">
                                    <?php echo esc_html(sf_truncate_url($redirect['address'])); ?>
                                </span>
                                <a href="<?php echo esc_url($redirect['address']); ?>" target="_blank" class="sf-url-link" title="<?php esc_attr_e('Open URL', 'screaming-fixes'); ?>">
                                    <span class="dashicons dashicons-external"></span>
                                </a>
                            </div>
                        </td>
                        <td class="sf-col-final">
                            <div class="sf-url-cell">
                                <span class="sf-url-text sf-final-url" title="<?php echo esc_attr($redirect['final_address']); ?>">
                                    <?php echo esc_html(sf_truncate_url($redirect['final_address'])); ?>
                                </span>
                                <a href="<?php echo esc_url($redirect['final_address']); ?>" target="_blank" class="sf-url-link" title="<?php esc_attr_e('Open URL', 'screaming-fixes'); ?>">
                                    <span class="dashicons dashicons-external"></span>
                                </a>
                            </div>
                        </td>
                        <td class="sf-col-hops">
                            <span class="sf-hops-badge <?php echo $redirect['num_redirects'] > 2 ? 'sf-hops-many' : ''; ?>">
                                <?php echo esc_html($redirect['num_redirects']); ?>
                            </span>
                        </td>
                        <td class="sf-col-count">
                            <button type="button" class="sf-expand-sources" data-expanded="false" data-redirect-url="<?php echo esc_attr($redirect['address']); ?>">
                                <span class="sf-source-count"><?php echo esc_html($redirect['fixable_count']); ?></span>
                                <span class="sf-source-label"><?php echo esc_html(_n('page', 'pages', $redirect['fixable_count'], 'screaming-fixes')); ?></span>
                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                            </button>
                        </td>
                        <td class="sf-col-actions">
                            <?php if ($is_loop) : ?>
                            <button type="button" class="sf-button sf-button-small sf-button-danger sf-fix-loop" title="<?php esc_attr_e('Delete redirect rule causing loop', 'screaming-fixes'); ?>" data-source="<?php echo esc_attr($redirect['address']); ?>">
                                <?php esc_html_e('Delete Rule', 'screaming-fixes'); ?>
                            </button>
                            <?php elseif ($has_temp && !$is_external_temp) : ?>
                            <button type="button" class="sf-button sf-button-small sf-button-warning sf-fix-temp" title="<?php esc_attr_e('Convert 302 to 301', 'screaming-fixes'); ?>" data-source="<?php echo esc_attr($redirect['address']); ?>">
                                <?php esc_html_e('Make 301', 'screaming-fixes'); ?>
                            </button>
                            <?php elseif ($is_external_temp) : ?>
                            <button type="button" class="sf-button sf-button-small sf-button-muted sf-cant-fix-external" title="<?php esc_attr_e('This redirect is on an external website', 'screaming-fixes'); ?>">
                                <?php esc_html_e("Can't Fix", 'screaming-fixes'); ?>
                            </button>
                            <?php else : ?>
                            <button type="button" class="sf-button sf-button-small sf-fix-single" title="<?php esc_attr_e('Fix this redirect', 'screaming-fixes'); ?>">
                                <?php esc_html_e('Fix', 'screaming-fixes'); ?>
                            </button>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr class="sf-sources-row" style="display: none;">
                        <td colspan="7">
                            <div class="sf-sources-list">
                                <?php
                                $displayed_sources = $redirect['fixable_sources'] ?? [];
                                $total_sources = $redirect['fixable_count'];
                                $showing_limited = count($displayed_sources) < $total_sources;
                                ?>
                                <?php if ($showing_limited) : ?>
                                <div class="sf-sources-notice">
                                    <?php
                                    printf(
                                        esc_html__('Showing %d of %d sources', 'screaming-fixes'),
                                        count($displayed_sources),
                                        $total_sources
                                    );
                                    ?>
                                </div>
                                <?php endif; ?>
                                <table class="sf-sources-table">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                            <th><?php esc_html_e('Location', 'screaming-fixes'); ?></th>
                                            <th><?php esc_html_e('Edit', 'screaming-fixes'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($displayed_sources as $source) : ?>
                                        <tr data-post-id="<?php echo esc_attr($source['post_id']); ?>">
                                            <td>
                                                <?php if (!empty($source['post_title'])) : ?>
                                                    <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" class="sf-source-title-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                        <?php echo esc_html($source['post_title']); ?>
                                                    </a>
                                                <?php else : ?>
                                                    <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" class="sf-source-url-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                        <?php echo esc_html($source['source_url']); ?>
                                                    </a>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="sf-location-badge"><?php echo esc_html(ucfirst($source['location'])); ?></span>
                                            </td>
                                            <td>
                                                <?php if (!empty($source['edit_url'])) : ?>
                                                <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" class="sf-edit-link" title="<?php esc_attr_e('Edit in WordPress', 'screaming-fixes'); ?>">
                                                    <span class="dashicons dashicons-edit"></span>
                                                </a>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

            <?php if ($total_fixable_pages > 1) : ?>
            <div class="sf-pagination" data-section="fixable">
                <div class="sf-pagination-info">
                    <?php
                    printf(
                        esc_html__('Showing %d-%d of %d', 'screaming-fixes'),
                        $fixable_offset + 1,
                        min($fixable_offset + $per_page, $total_fixable_items),
                        $total_fixable_items
                    );
                    ?>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" <?php echo $current_page_fixable <= 1 ? 'disabled' : ''; ?>>
                        &larr; <?php esc_html_e('Previous', 'screaming-fixes'); ?>
                    </button>
                    <div class="sf-page-numbers">
                        <?php
                        $max_visible_pages = 7;
                        $start_page = max(1, $current_page_fixable - 3);
                        $end_page = min($total_fixable_pages, $start_page + $max_visible_pages - 1);
                        if ($end_page - $start_page < $max_visible_pages - 1) {
                            $start_page = max(1, $end_page - $max_visible_pages + 1);
                        }

                        if ($start_page > 1) : ?>
                            <button type="button" class="sf-page-btn sf-page-num" data-page="1">1</button>
                            <?php if ($start_page > 2) : ?>
                                <span class="sf-page-ellipsis">...</span>
                            <?php endif; ?>
                        <?php endif;

                        for ($i = $start_page; $i <= $end_page; $i++) : ?>
                            <button type="button" class="sf-page-btn sf-page-num <?php echo $i === $current_page_fixable ? 'active' : ''; ?>" data-page="<?php echo $i; ?>">
                                <?php echo $i; ?>
                            </button>
                        <?php endfor;

                        if ($end_page < $total_fixable_pages) : ?>
                            <?php if ($end_page < $total_fixable_pages - 1) : ?>
                                <span class="sf-page-ellipsis">...</span>
                            <?php endif; ?>
                            <button type="button" class="sf-page-btn sf-page-num" data-page="<?php echo $total_fixable_pages; ?>"><?php echo $total_fixable_pages; ?></button>
                        <?php endif; ?>
                    </div>
                    <button type="button" class="sf-page-btn sf-page-next" <?php echo $current_page_fixable >= $total_fixable_pages ? 'disabled' : ''; ?>>
                        <?php esc_html_e('Next', 'screaming-fixes'); ?> &rarr;
                    </button>
                </div>
            </div>
            <?php endif; ?>

        </div><!-- /.sf-fixable-content -->

        <!-- Batch Actions Bar - Outside collapsible content so it stays visible -->
        <div class="sf-batch-actions">
            <div class="sf-selected-count">
                <span class="sf-count" id="sf-selected-redirect-count">0</span>
                <?php esc_html_e('selected', 'screaming-fixes'); ?>
            </div>
            <div class="sf-batch-buttons">
                <button type="button" class="sf-button sf-button-primary sf-apply-fixes" id="sf-apply-redirect-fixes" disabled>
                    <span class="dashicons dashicons-yes"></span>
                    <?php esc_html_e('Fix Selected', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if (!empty($manual_redirects)) : ?>
    <div class="sf-results-section sf-manual-section">
        <button type="button" class="sf-section-toggle sf-manual-toggle" aria-expanded="true">
            <span class="sf-section-badge sf-badge-manual">&#9997;</span>
            <?php printf(esc_html__('%d Manual Review Required', 'screaming-fixes'), $total_manual_items); ?>
            <span class="sf-section-hint"><?php esc_html_e('Redirects in navigation, widgets, or theme areas', 'screaming-fixes'); ?></span>
            <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon sf-rotated"></span>
        </button>

        <div class="sf-manual-content">
            <p class="sf-section-description">
                <?php esc_html_e('These redirects are in navigation, widgets, or other theme areas that require manual editing.', 'screaming-fixes'); ?>
            </p>

            <?php if ($total_manual_pages > 1) : ?>
            <div class="sf-pagination sf-pagination-top">
            <span class="sf-page-info">
                <?php
                printf(
                    esc_html__('Showing %d-%d of %d redirects', 'screaming-fixes'),
                    $manual_offset + 1,
                    min($manual_offset + $per_page, $total_manual_items),
                    $total_manual_items
                );
                ?>
            </span>
        </div>
        <?php endif; ?>

        <div class="sf-results-table-wrapper">
            <table class="sf-results-table sf-manual-table">
                <thead>
                    <tr>
                        <th class="sf-col-type"><?php esc_html_e('Type', 'screaming-fixes'); ?></th>
                        <th class="sf-col-redirect"><?php esc_html_e('Redirect URL', 'screaming-fixes'); ?></th>
                        <th class="sf-col-final"><?php esc_html_e('Final Destination', 'screaming-fixes'); ?></th>
                        <th class="sf-col-hops"><?php esc_html_e('Hops', 'screaming-fixes'); ?></th>
                        <th class="sf-col-location"><?php esc_html_e('Location', 'screaming-fixes'); ?></th>
                        <th class="sf-col-count"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($manual_to_display as $redirect) : ?>
                    <tr class="sf-redirect-row sf-manual-row">
                        <td class="sf-col-type">
                            <span class="sf-type-badge <?php echo esc_attr($redirect['redirect_type']['class'] ?? 'sf-type-external'); ?>">
                                <?php echo esc_html($redirect['redirect_type']['label'] ?? __('External', 'screaming-fixes')); ?>
                            </span>
                        </td>
                        <td class="sf-col-redirect">
                            <div class="sf-url-cell">
                                <span class="sf-url-text" title="<?php echo esc_attr($redirect['address']); ?>">
                                    <?php echo esc_html(sf_truncate_url($redirect['address'])); ?>
                                </span>
                                <button type="button" class="sf-copy-url" title="<?php esc_attr_e('Copy URL', 'screaming-fixes'); ?>" data-url="<?php echo esc_attr($redirect['address']); ?>">
                                    <span class="dashicons dashicons-clipboard"></span>
                                </button>
                            </div>
                        </td>
                        <td class="sf-col-final">
                            <div class="sf-url-cell">
                                <span class="sf-url-text sf-final-url" title="<?php echo esc_attr($redirect['final_address']); ?>">
                                    <?php echo esc_html(sf_truncate_url($redirect['final_address'])); ?>
                                </span>
                                <button type="button" class="sf-copy-url" title="<?php esc_attr_e('Copy URL', 'screaming-fixes'); ?>" data-url="<?php echo esc_attr($redirect['final_address']); ?>">
                                    <span class="dashicons dashicons-clipboard"></span>
                                </button>
                            </div>
                        </td>
                        <td class="sf-col-hops">
                            <span class="sf-hops-badge <?php echo $redirect['num_redirects'] > 2 ? 'sf-hops-many' : ''; ?>">
                                <?php echo esc_html($redirect['num_redirects']); ?>
                            </span>
                        </td>
                        <td class="sf-col-location">
                            <?php
                            $locations = array_unique(array_column($redirect['manual_sources'] ?? [], 'location'));
                            foreach ($locations as $location) :
                            ?>
                            <span class="sf-location-badge sf-location-<?php echo esc_attr($location); ?>">
                                <?php echo esc_html(ucfirst($location)); ?>
                            </span>
                            <?php endforeach; ?>
                        </td>
                        <td class="sf-col-count">
                            <button type="button" class="sf-expand-sources" data-expanded="false" data-redirect-url="<?php echo esc_attr($redirect['address']); ?>">
                                <span class="sf-source-count"><?php echo esc_html($redirect['manual_count']); ?></span>
                                <span class="sf-source-label"><?php echo esc_html(_n('page', 'pages', $redirect['manual_count'], 'screaming-fixes')); ?></span>
                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                            </button>
                        </td>
                    </tr>
                    <tr class="sf-sources-row" style="display: none;">
                        <td colspan="6">
                            <div class="sf-sources-list">
                                <?php
                                $displayed_sources = $redirect['manual_sources'] ?? [];
                                $total_sources = $redirect['manual_count'];
                                $showing_limited = count($displayed_sources) < $total_sources;
                                ?>
                                <?php if ($showing_limited) : ?>
                                <div class="sf-sources-notice">
                                    <?php
                                    printf(
                                        esc_html__('Showing %d of %d sources', 'screaming-fixes'),
                                        count($displayed_sources),
                                        $total_sources
                                    );
                                    ?>
                                </div>
                                <?php endif; ?>
                                <table class="sf-sources-table">
                                    <thead>
                                        <tr>
                                            <th><?php esc_html_e('Source', 'screaming-fixes'); ?></th>
                                            <th><?php esc_html_e('Location', 'screaming-fixes'); ?></th>
                                            <th><?php esc_html_e('Note', 'screaming-fixes'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($displayed_sources as $source) : ?>
                                        <tr>
                                            <td>
                                                <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" class="sf-source-url-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                    <?php echo esc_html($source['source_url']); ?>
                                                </a>
                                            </td>
                                            <td>
                                                <span class="sf-location-badge"><?php echo esc_html(ucfirst($source['location'])); ?></span>
                                            </td>
                                            <td>
                                                <span class="sf-fix-note"><?php echo esc_html($source['fix_note']); ?></span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <?php if ($total_manual_pages > 1) : ?>
        <div class="sf-pagination">
            <?php if ($current_page_manual > 1) : ?>
                <a href="<?php echo esc_url(add_query_arg('paged_manual', $current_page_manual - 1)); ?>" class="sf-button sf-button-secondary sf-pagination-prev">
                    &larr; <?php esc_html_e('Previous', 'screaming-fixes'); ?>
                </a>
            <?php endif; ?>

            <span class="sf-page-info">
                <?php
                printf(
                    esc_html__('Page %d of %d', 'screaming-fixes'),
                    $current_page_manual,
                    $total_manual_pages
                );
                ?>
            </span>

            <?php if ($current_page_manual < $total_manual_pages) : ?>
                <a href="<?php echo esc_url(add_query_arg('paged_manual', $current_page_manual + 1)); ?>" class="sf-button sf-button-secondary sf-pagination-next">
                    <?php esc_html_e('Next', 'screaming-fixes'); ?> &rarr;
                </a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        </div><!-- /.sf-manual-content -->
    </div>
    <?php endif; ?>

    <!-- Fixes Applied Section (Collapsible) -->
    <div class="sf-results-section sf-fixed-section" id="sf-fixed-redirects-section" <?php echo $fixed_count === 0 ? 'style="display: none;"' : ''; ?>>
        <div class="sf-results-header sf-results-header-collapsible sf-results-header-fixed">
            <button type="button" class="sf-section-toggle sf-fixed-toggle" aria-expanded="<?php echo $fixed_count > 0 ? 'true' : 'false'; ?>">
                <span class="sf-section-badge sf-badge-fixed">&#10004;</span>
                <span class="sf-fixes-applied-title">
                    <?php esc_html_e('Fixes Applied', 'screaming-fixes'); ?> (<span class="sf-fixes-applied-count"><?php echo esc_html($fixed_count); ?></span>)
                </span>
                <span class="sf-section-hint"><?php esc_html_e('View fix results and export to CSV', 'screaming-fixes'); ?></span>
                <span class="dashicons dashicons-arrow-down-alt2 sf-toggle-icon <?php echo $fixed_count > 0 ? 'sf-rotated' : ''; ?>"></span>
            </button>
            <div class="sf-results-actions">
                <button type="button" class="sf-button sf-button-secondary" id="sf-export-fixed-redirects">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export CSV', 'screaming-fixes'); ?>
                </button>
            </div>
        </div>

        <div class="sf-fixed-content" <?php echo $fixed_count === 0 ? 'style="display: none;"' : ''; ?>>
            <!-- Status Filter Buttons -->
            <div class="sf-status-filter-row sf-rc-status-filters">
                <span class="sf-filter-label"><?php esc_html_e('Status:', 'screaming-fixes'); ?></span>
                <div class="sf-status-filters">
                    <button type="button" class="sf-status-filter active" data-status="all">
                        <span class="sf-status-icon">&#128203;</span>
                        <span class="sf-status-count sf-rc-status-count-all"><?php echo esc_html($fixed_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('All', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-success" data-status="success">
                        <span class="sf-status-icon">&#9989;</span>
                        <span class="sf-status-count sf-rc-status-count-success"><?php echo esc_html($fixed_success_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Fixed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-failed" data-status="failed">
                        <span class="sf-status-icon">&#10060;</span>
                        <span class="sf-status-count sf-rc-status-count-failed"><?php echo esc_html($fixed_failed_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Failed', 'screaming-fixes'); ?></span>
                    </button>
                    <button type="button" class="sf-status-filter sf-status-skipped" data-status="skipped">
                        <span class="sf-status-icon">&#9203;</span>
                        <span class="sf-status-count sf-rc-status-count-skipped"><?php echo esc_html($fixed_skipped_count); ?></span>
                        <span class="sf-status-label"><?php esc_html_e('Skipped', 'screaming-fixes'); ?></span>
                    </button>
                </div>
            </div>

            <div class="sf-results-table-wrapper sf-paginated-table" data-section="fixes-applied" data-per-page="100">
                <table class="sf-results-table sf-fixed-table sf-fixes-applied-table">
                    <thead>
                        <tr>
                            <th class="sf-col-was"><?php esc_html_e('Original URL', 'screaming-fixes'); ?></th>
                            <th class="sf-col-now"><?php esc_html_e('Updated To', 'screaming-fixes'); ?></th>
                            <th class="sf-col-status"><?php esc_html_e('Status', 'screaming-fixes'); ?></th>
                            <th class="sf-col-count"><?php esc_html_e('Found In', 'screaming-fixes'); ?></th>
                            <th class="sf-col-when"><?php esc_html_e('Date Applied', 'screaming-fixes'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="sf-fixes-applied-tbody">
                        <?php foreach ($fixed_redirects as $redirect):
                            $address = $redirect['address'] ?? '';
                            $final_address = $redirect['final_address'] ?? '';
                            $fixed_at = $redirect['fixed_at'] ?? '';
                            $sources = $redirect['fixable_sources'] ?? $redirect['sources'] ?? [];
                            $source_count = $redirect['fixable_count'] ?? (is_array($sources) ? count($sources) : ($redirect['source_count'] ?? 0));
                            $status = $redirect['status'] ?? 'success';
                            $status_message = $redirect['status_message'] ?? '';

                            if ($status === 'success') {
                                $status_icon = '&#9989;';
                                $status_label = __('Fixed', 'screaming-fixes');
                                $row_class = 'sf-fixed-row-success';
                            } elseif ($status === 'failed') {
                                $status_icon = '&#10060;';
                                $status_label = __('Failed', 'screaming-fixes');
                                $row_class = 'sf-fixed-row-failed';
                            } else {
                                $status_icon = '&#9203;';
                                $status_label = __('Skipped', 'screaming-fixes');
                                $row_class = 'sf-fixed-row-skipped';
                            }
                        ?>
                        <tr class="sf-fixed-row <?php echo esc_attr($row_class); ?>" data-status="<?php echo esc_attr($status); ?>">
                            <td class="sf-col-was">
                                <div class="sf-url-cell">
                                    <span class="sf-url-text sf-url-strikethrough" title="<?php echo esc_attr($address); ?>">
                                        <?php echo esc_html(sf_truncate_url($address)); ?>
                                    </span>
                                    <a href="<?php echo esc_url($address); ?>" target="_blank" class="sf-url-link" title="<?php esc_attr_e('Open URL', 'screaming-fixes'); ?>">
                                        <span class="dashicons dashicons-external"></span>
                                    </a>
                                </div>
                            </td>
                            <td class="sf-col-now">
                                <div class="sf-url-cell">
                                    <span class="sf-url-text sf-final-url" title="<?php echo esc_attr($final_address); ?>">
                                        <?php echo esc_html(sf_truncate_url($final_address)); ?>
                                    </span>
                                    <a href="<?php echo esc_url($final_address); ?>" target="_blank" class="sf-url-link" title="<?php esc_attr_e('Open URL', 'screaming-fixes'); ?>">
                                        <span class="dashicons dashicons-external"></span>
                                    </a>
                                </div>
                            </td>
                            <td class="sf-col-status">
                                <span class="sf-status-badge sf-status-<?php echo esc_attr($status); ?>"<?php if (!empty($status_message) && $status !== 'success') echo ' title="' . esc_attr($status_message) . '"'; ?>>
                                    <span class="sf-status-icon"><?php echo $status_icon; ?></span>
                                    <?php echo esc_html($status_label); ?>
                                </span>
                            </td>
                            <td class="sf-col-count">
                                <button type="button" class="sf-expand-sources" data-expanded="false" data-redirect-url="<?php echo esc_attr($address); ?>">
                                    <span class="sf-source-count"><?php echo esc_html($source_count); ?></span>
                                    <span class="sf-source-label"><?php echo esc_html(_n('page', 'pages', $source_count, 'screaming-fixes')); ?></span>
                                    <span class="dashicons dashicons-arrow-down-alt2"></span>
                                </button>
                            </td>
                            <td class="sf-col-when">
                                <?php if (!empty($fixed_at)): ?>
                                    <span class="sf-fixed-time" title="<?php echo esc_attr($fixed_at); ?>">
                                        <?php echo esc_html(human_time_diff(strtotime($fixed_at), current_time('timestamp')) . ' ' . __('ago', 'screaming-fixes')); ?>
                                    </span>
                                <?php else: ?>
                                    <span class="sf-fixed-time">&mdash;</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr class="sf-sources-row" style="display: none;">
                            <td colspan="5">
                                <div class="sf-sources-list">
                                    <table class="sf-sources-table">
                                        <thead>
                                            <tr>
                                                <th><?php esc_html_e('Page', 'screaming-fixes'); ?></th>
                                                <th><?php esc_html_e('Location', 'screaming-fixes'); ?></th>
                                                <th><?php esc_html_e('Edit', 'screaming-fixes'); ?></th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php if (is_array($sources)) : ?>
                                            <?php foreach ($sources as $source) : ?>
                                            <tr data-post-id="<?php echo esc_attr($source['post_id'] ?? ''); ?>">
                                                <td>
                                                    <?php if (!empty($source['post_title'])) : ?>
                                                        <a href="<?php echo esc_url($source['source_url'] ?? ''); ?>" target="_blank" class="sf-source-title-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                            <?php echo esc_html($source['post_title']); ?>
                                                        </a>
                                                    <?php elseif (!empty($source['source_url'])) : ?>
                                                        <a href="<?php echo esc_url($source['source_url']); ?>" target="_blank" class="sf-source-url-link" title="<?php esc_attr_e('View page', 'screaming-fixes'); ?>">
                                                            <?php echo esc_html($source['source_url']); ?>
                                                        </a>
                                                    <?php else : ?>
                                                        <span>-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="sf-location-badge"><?php echo esc_html(ucfirst($source['location'] ?? 'content')); ?></span>
                                                </td>
                                                <td>
                                                    <?php if (!empty($source['edit_url'])) : ?>
                                                    <a href="<?php echo esc_url($source['edit_url']); ?>" target="_blank" class="sf-edit-link" title="<?php esc_attr_e('Edit in WordPress', 'screaming-fixes'); ?>">
                                                        <span class="dashicons dashicons-edit"></span>
                                                    </a>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($fixed_count > 100): ?>
            <div class="sf-pagination" data-section="fixes-applied">
                <div class="sf-pagination-info">
                    <?php esc_html_e('Showing', 'screaming-fixes'); ?> <span class="sf-page-start">1</span>&ndash;<span class="sf-page-end"><?php echo min(100, $fixed_count); ?></span>
                    <?php esc_html_e('of', 'screaming-fixes'); ?> <span class="sf-page-total"><?php echo esc_html($fixed_count); ?></span>
                </div>
                <div class="sf-pagination-controls">
                    <button type="button" class="sf-page-btn sf-page-prev" disabled>
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                    </button>
                    <div class="sf-page-numbers"></div>
                    <button type="button" class="sf-page-btn sf-page-next">
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <?php endif; ?>

</div>
