<?php
/**
 * Activity Log for Screaming Fixes
 *
 * Logs batch fix operations for dashboard display.
 * This is separate from SF_Change_Logger which tracks individual changes for undo.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Activity_Log {

    /**
     * Option name for storing activity log
     * @var string
     */
    const OPTION_NAME = 'sf_activity_log';

    /**
     * Maximum number of entries to keep
     * @var int
     */
    const MAX_ENTRIES = 25;

    /**
     * Module configurations with icons and labels
     * @var array
     */
    private static $modules = [
        'broken-links' => [
            'icon' => '&#128279;', // 🔗
            'label' => 'Broken Links',
            'action_label' => 'Fixed %d broken links',
        ],
        'redirect-chains' => [
            'icon' => '&#128260;', // 🔄
            'label' => 'Redirect Chains',
            'action_label' => 'Resolved %d redirect chains',
        ],
        'backlink-reclaim' => [
            'icon' => '&#128281;', // 🔙
            'label' => 'Backlink Reclaim',
            'action_label' => 'Created %d redirects for dead backlinks',
        ],
        'image-alt-text' => [
            'icon' => '&#128444;', // 🖼️
            'label' => 'Image Alt Text',
            'action_label' => 'Added alt text to %d images',
        ],
        'meta-description' => [
            'icon' => '&#128196;', // 📄
            'label' => 'Meta Descriptions',
            'action_label' => 'Updated %d meta descriptions',
        ],
        'page-title' => [
            'icon' => '&#128221;', // 📝
            'label' => 'Page Titles',
            'action_label' => 'Updated %d page titles',
        ],
        'internal-link-builder' => [
            'icon' => '&#128279;', // 🔗
            'label' => 'Internal Links',
            'action_label' => 'Added %d internal links',
        ],
    ];

    /**
     * Log a batch fix operation
     *
     * @param string $module Module identifier (e.g., 'broken-links', 'image-alt-text')
     * @param int $count Number of items fixed
     * @param array $extra Optional extra data (posts_affected, etc.)
     * @return bool Success
     */
    public static function log($module, $count, $extra = []) {
        if ($count <= 0) {
            return false;
        }

        $entries = self::get_entries();

        $entry = [
            'module' => sanitize_key($module),
            'count' => absint($count),
            'timestamp' => current_time('timestamp'),
            'user_id' => get_current_user_id(),
        ];

        // Add any extra data
        if (!empty($extra['posts_affected'])) {
            $entry['posts_affected'] = absint($extra['posts_affected']);
        }

        // Add to beginning of array (newest first)
        array_unshift($entries, $entry);

        // Trim to max entries
        $entries = array_slice($entries, 0, self::MAX_ENTRIES);

        // Save with autoload = false (only needed on plugin dashboard)
        return update_option(self::OPTION_NAME, $entries, false);
    }

    /**
     * Get all log entries
     *
     * @return array Log entries
     */
    public static function get_entries() {
        $entries = get_option(self::OPTION_NAME, []);
        return is_array($entries) ? $entries : [];
    }

    /**
     * Get formatted entries for display
     *
     * @param int $limit Number of entries to return (default 5)
     * @return array Formatted entries
     */
    public static function get_formatted_entries($limit = 5) {
        $entries = self::get_entries();
        $entries = array_slice($entries, 0, $limit);

        $formatted = [];

        foreach ($entries as $entry) {
            $module_config = self::$modules[$entry['module']] ?? [
                'icon' => '&#10003;',
                'label' => ucfirst(str_replace('-', ' ', $entry['module'])),
                'action_label' => 'Fixed %d items',
            ];

            $formatted[] = [
                'icon' => $module_config['icon'],
                'module_label' => $module_config['label'],
                'description' => sprintf($module_config['action_label'], $entry['count']),
                'count' => $entry['count'],
                'timestamp' => $entry['timestamp'],
                'time_ago' => human_time_diff($entry['timestamp'], current_time('timestamp')) . ' ' . __('ago', 'screaming-fixes'),
                'date_formatted' => date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $entry['timestamp']),
                'posts_affected' => $entry['posts_affected'] ?? null,
            ];
        }

        return $formatted;
    }

    /**
     * Get total count of entries
     *
     * @return int Total entries
     */
    public static function get_total_count() {
        return count(self::get_entries());
    }

    /**
     * Get summary statistics
     *
     * @return array Stats
     */
    public static function get_stats() {
        $entries = self::get_entries();

        $stats = [
            'total_operations' => count($entries),
            'total_fixes' => 0,
            'fixes_today' => 0,
            'fixes_this_week' => 0,
            'by_module' => [],
        ];

        $today_start = strtotime('today', current_time('timestamp'));
        $week_start = strtotime('-7 days', current_time('timestamp'));

        foreach ($entries as $entry) {
            $stats['total_fixes'] += $entry['count'];

            // Count by time period
            if ($entry['timestamp'] >= $today_start) {
                $stats['fixes_today'] += $entry['count'];
            }
            if ($entry['timestamp'] >= $week_start) {
                $stats['fixes_this_week'] += $entry['count'];
            }

            // Count by module
            $module = $entry['module'];
            if (!isset($stats['by_module'][$module])) {
                $stats['by_module'][$module] = 0;
            }
            $stats['by_module'][$module] += $entry['count'];
        }

        return $stats;
    }

    /**
     * Clear all log entries
     *
     * @return bool Success
     */
    public static function clear() {
        return delete_option(self::OPTION_NAME);
    }

    /**
     * Get module configuration
     *
     * @param string $module Module identifier
     * @return array Module config
     */
    public static function get_module_config($module) {
        return self::$modules[$module] ?? [
            'icon' => '&#10003;',
            'label' => ucfirst(str_replace('-', ' ', $module)),
            'action_label' => 'Fixed %d items',
        ];
    }
}
