<?php
/**
 * Backlink API Integration
 *
 * Handles communication with Screaming Fixes hosted API for backlink data
 * The hosted API handles DataForSEO credentials server-side
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Backlink_API {

    /**
     * Demo mode - set to true to use dummy data instead of real API calls
     * Useful for development and testing without API credentials
     */
    const DEMO_MODE = false;

    /**
     * Demo empty results - set to true to test the "no dead backlinks found" state
     * Only works when DEMO_MODE is also true
     */
    const DEMO_EMPTY_RESULTS = false;

    /**
     * Production API endpoint - hosted on screamingfixes.com
     */
    const API_ENDPOINT = 'https://screamingfixes.com/.netlify/functions/scan-backlinks';

    /**
     * API credentials (for users with their own DataForSEO key)
     */
    private $api_login;
    private $api_password;

    /**
     * Whether using free tier
     */
    private $is_free_tier = false;

    /**
     * Free tier credentials (deprecated - now handled by hosted API)
     */
    private $free_tier_login = '';
    private $free_tier_password = '';

    /**
     * DataForSEO API base URL (for users with their own key)
     */
    private $api_url = 'https://api.dataforseo.com/v3/';

    /**
     * Debug info from last scan
     */
    private $last_scan_debug = array();

    /**
     * Constructor
     *
     * @param bool $use_free_tier Whether to use free tier credentials
     */
    public function __construct($use_free_tier = false) {
        if ($use_free_tier && !empty($this->free_tier_login)) {
            $this->api_login = $this->free_tier_login;
            $this->api_password = $this->free_tier_password;
            $this->is_free_tier = true;
        } else {
            $this->api_login = get_option('sf_dataforseo_login', '');
            $this->api_password = get_option('sf_dataforseo_password', '');
        }
    }

    /**
     * Check if API credentials are configured
     * Always returns true since the hosted API handles credentials server-side
     *
     * @return bool
     */
    public function has_credentials() {
        // Demo mode for testing
        if (self::DEMO_MODE) {
            return true;
        }
        // The hosted API handles credentials server-side
        return true;
    }

    /**
     * Check if free tier credentials are available
     *
     * @return bool
     */
    public function has_free_tier() {
        return !empty($this->free_tier_login) && !empty($this->free_tier_password);
    }

    /**
     * Check if user has their own API credentials
     *
     * @return bool
     */
    public static function user_has_api_key() {
        return !empty(get_option('sf_dataforseo_login')) && !empty(get_option('sf_dataforseo_password'));
    }

    /**
     * Check if demo mode is enabled
     *
     * @return bool
     */
    public static function is_demo_mode() {
        return self::DEMO_MODE;
    }

    /**
     * Test API connection - tests DataForSEO if user has credentials, otherwise tests hosted API
     *
     * @return array
     */
    public function test_connection() {
        $login = get_option('sf_dataforseo_login');
        $password = get_option('sf_dataforseo_password');

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF DataForSEO Test - Login: ' . (empty($login) ? 'EMPTY' : 'SET (' . strlen($login) . ' chars)'));
            error_log('SF DataForSEO Test - Password: ' . (empty($password) ? 'EMPTY' : 'SET (' . strlen($password) . ' chars)'));
        }

        // If user has credentials, test DataForSEO directly
        if (!empty($login) && !empty($password)) {
            return $this->test_dataforseo_connection($login, $password);
        }

        // Otherwise test the hosted API
        return $this->test_hosted_api_connection();
    }

    /**
     * Test DataForSEO API connection with user credentials
     *
     * @param string $login DataForSEO login
     * @param string $password DataForSEO password
     * @return array
     */
    private function test_dataforseo_connection($login, $password) {
        $auth = base64_encode($login . ':' . $password);

        $response = wp_remote_get('https://api.dataforseo.com/v3/appendix/user_data', array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Basic ' . $auth,
                'Content-Type' => 'application/json',
            ),
        ));

        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF DataForSEO Test - WP Error: ' . $response->get_error_message());
            }
            return array(
                'success' => false,
                'message' => __('Connection failed: ', 'screaming-fixes') . $response->get_error_message(),
            );
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF DataForSEO Test - Response Code: ' . $code);
            error_log('SF DataForSEO Test - Response Body: ' . substr($body, 0, 500));
        }

        if ($code === 401) {
            return array(
                'success' => false,
                'message' => __('Invalid credentials. Please check your DataForSEO login and password.', 'screaming-fixes'),
            );
        }

        if ($code === 200) {
            $data = json_decode($body, true);
            if (isset($data['status_code']) && $data['status_code'] === 20000) {
                $balance = isset($data['tasks'][0]['result'][0]['money']['balance'])
                    ? '$' . number_format($data['tasks'][0]['result'][0]['money']['balance'], 2)
                    : 'N/A';
                return array(
                    'success' => true,
                    'message' => sprintf(__('Connected to DataForSEO! Balance: %s', 'screaming-fixes'), $balance),
                );
            }
        }

        return array(
            'success' => false,
            'message' => sprintf(__('DataForSEO API returned status %d', 'screaming-fixes'), $code),
        );
    }

    /**
     * Test hosted API connection
     *
     * @return array
     */
    private function test_hosted_api_connection() {
        $response = wp_remote_post(self::API_ENDPOINT, array(
            'timeout' => 10,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'domain' => 'test.com',
            )),
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }

        $code = wp_remote_retrieve_response_code($response);

        // 200 = success, 429 = rate limited (but API is working)
        if ($code === 200 || $code === 429) {
            return array(
                'success' => true,
                'message' => __('Hosted API connected', 'screaming-fixes'),
            );
        }

        return array(
            'success' => false,
            'message' => sprintf(__('API returned status %d', 'screaming-fixes'), $code),
        );
    }

    /**
     * Track which API was used for the last scan (for debugging)
     * @var string
     */
    private $last_api_used = '';

    /**
     * Get which API was used for the last scan
     *
     * @return string 'direct', 'hosted', or 'demo'
     */
    public function get_last_api_used() {
        return $this->last_api_used;
    }

    /**
     * Get backlinks pointing to 404 pages for a domain
     *
     * @param string $domain Domain to scan
     * @return array|WP_Error
     */
    public function get_broken_backlinks($domain) {
        // Return demo data if in demo mode
        if (self::DEMO_MODE) {
            $this->last_api_used = 'demo';
            return $this->get_demo_data($domain);
        }

        // Check if user has their own DataForSEO credentials
        $user_login = get_option('sf_dataforseo_login');
        $user_password = get_option('sf_dataforseo_password');

        if (!empty($user_login) && !empty($user_password)) {
            // User has own credentials - call DataForSEO directly (no rate limit)
            $this->last_api_used = 'direct';
            $domain = $this->normalize_domain($domain);
            return $this->call_dataforseo_direct($domain, $user_login, $user_password);
        }

        // No user credentials - use hosted API (rate limited)
        $this->last_api_used = 'hosted';
        $domain = $this->normalize_domain($domain);
        return $this->call_hosted_api($domain);
    }

    /**
     * Check if user has their own DataForSEO credentials
     *
     * @return bool
     */
    public static function has_user_credentials() {
        $login = get_option('sf_dataforseo_login');
        $password = get_option('sf_dataforseo_password');
        return !empty($login) && !empty($password);
    }

    /**
     * Get debug info about current credential state (for troubleshooting)
     *
     * @return array
     */
    public static function get_credential_debug_info() {
        $login = get_option('sf_dataforseo_login');
        $password = get_option('sf_dataforseo_password');

        return array(
            'has_credentials' => self::has_user_credentials(),
            'login_exists' => !empty($login),
            'login_length' => strlen($login ?? ''),
            'login_preview' => $login ? substr($login, 0, 3) . '***' : '(empty)',
            'password_exists' => !empty($password),
            'password_length' => strlen($password ?? ''),
            'demo_mode' => self::DEMO_MODE,
        );
    }

    /**
     * Call DataForSEO API directly with user's credentials
     *
     * @param string $domain Domain to scan
     * @param string $login DataForSEO login
     * @param string $password DataForSEO password
     * @return array|WP_Error
     */
    private function call_dataforseo_direct($domain, $login, $password) {
        $auth = base64_encode($login . ':' . $password);

        // Debug logging - always log for production troubleshooting
        error_log('=== SF BACKLINK API: STARTING SCAN ===');
        error_log('SF Backlink API: Domain: ' . $domain);
        error_log('SF Backlink API: Login: ' . substr($login, 0, 5) . '***');
        error_log('SF Backlink API: Timestamp: ' . date('Y-m-d H:i:s'));

        $post_data = wp_json_encode(array(
            array(
                'target' => $domain,
                'limit' => 1000,
                'mode' => 'as_is',
            )
        ));

        error_log('SF Backlink API: Request payload: ' . $post_data);

        $response = wp_remote_post('https://api.dataforseo.com/v3/backlinks/backlinks/live', array(
            'timeout' => 120,
            'headers' => array(
                'Authorization' => 'Basic ' . $auth,
                'Content-Type' => 'application/json',
            ),
            'body' => $post_data,
        ));

        error_log('SF Backlink API: DataForSEO response received');

        if (is_wp_error($response)) {
            error_log('SF Backlink API: WP_ERROR - ' . $response->get_error_message());
            return new WP_Error('api_error', 'DataForSEO connection failed: ' . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        error_log('SF Backlink API: HTTP Status: ' . $status_code);
        error_log('SF Backlink API: Response length: ' . strlen($body) . ' bytes');

        // Check for HTTP errors first
        if ($status_code === 0) {
            return new WP_Error('connection_error', __('Could not connect to DataForSEO API. Check your internet connection.', 'screaming-fixes'));
        }

        $data = json_decode($body, true);

        if (!$data) {
            return new WP_Error('invalid_response', __('Invalid response from DataForSEO', 'screaming-fixes'));
        }

        if ($status_code === 401) {
            return new WP_Error('unauthorized', __('Invalid DataForSEO credentials. Please check your login and password.', 'screaming-fixes'));
        }

        if (!isset($data['status_code']) || $data['status_code'] !== 20000) {
            $message = isset($data['status_message']) ? $data['status_message'] : __('DataForSEO API error', 'screaming-fixes');
            return new WP_Error('api_error', $message);
        }

        // Detailed logging of API response structure
        error_log('=== SF BACKLINK API: RESPONSE ANALYSIS ===');
        error_log('SF Backlink API: tasks count: ' . (isset($data['tasks']) ? count($data['tasks']) : 0));
        if (isset($data['tasks'][0])) {
            error_log('SF Backlink API: task status_code: ' . ($data['tasks'][0]['status_code'] ?? 'N/A'));
            error_log('SF Backlink API: task status_message: ' . ($data['tasks'][0]['status_message'] ?? 'N/A'));
        }

        // Check for task-level errors (like subscription issues)
        if (isset($data['tasks'][0]['status_code']) && $data['tasks'][0]['status_code'] !== 20000) {
            $task_status = $data['tasks'][0]['status_code'];
            $task_message = $data['tasks'][0]['status_message'] ?? 'Unknown task error';

            error_log('SF Backlink API: TASK ERROR - ' . $task_status . ': ' . $task_message);

            // 40204 = subscription/access denied
            if ($task_status === 40204) {
                return new WP_Error(
                    'subscription_required',
                    __('DataForSEO Backlinks API subscription required. Please activate your Backlinks subscription at app.dataforseo.com', 'screaming-fixes')
                );
            }

            return new WP_Error('task_error', $task_message);
        }

        // Parse the backlinks response and group by target URL
        $items = isset($data['tasks'][0]['result'][0]['items']) ? $data['tasks'][0]['result'][0]['items'] : array();

        if (isset($data['tasks'][0]['result'][0])) {
            $result = $data['tasks'][0]['result'][0];
            error_log('SF Backlink API: total_count in result: ' . ($result['total_count'] ?? 'N/A'));
            error_log('SF Backlink API: items_count in result: ' . ($result['items_count'] ?? 'N/A'));
        }
        error_log('SF Backlink API: Parsed items count: ' . count($items));

        if (empty($items)) {
            // Return empty array - this is a valid result, not an error
            error_log('SF Backlink API: NO BACKLINKS RETURNED - this may mean:');
            error_log('  - Domain has no external backlinks indexed by DataForSEO');
            error_log('  - Domain name format might need adjustment (www vs non-www)');
            error_log('  - API credentials may have limited access');
            return array();
        }

        // Log first few items for debugging
        error_log('SF Backlink API: First 3 backlink targets:');
        foreach (array_slice($items, 0, 3) as $i => $item) {
            error_log('  [' . $i . '] url_to: ' . ($item['url_to'] ?? 'N/A') . ' | from: ' . ($item['domain_from'] ?? 'N/A'));
        }

        // Group by target URL
        $grouped = array();
        foreach ($items as $link) {
            $target_url = isset($link['url_to']) ? $link['url_to'] : '';
            if (empty($target_url)) {
                continue;
            }

            $target_path = wp_parse_url($target_url, PHP_URL_PATH);
            if (empty($target_path)) {
                $target_path = '/';
            }

            if (!isset($grouped[$target_path])) {
                $grouped[$target_path] = array(
                    'dead_page' => $target_url,
                    'dead_page_path' => $target_path,
                    'backlink_count' => 0,
                    'referring_domains_list' => array(),
                    'top_referrers' => array(),
                    'all_referrers' => array(),
                    'status_code' => null,
                );
            }

            $grouped[$target_path]['backlink_count']++;

            $domain_from = isset($link['domain_from']) ? $link['domain_from'] : '';
            $url_from = isset($link['url_from']) ? $link['url_from'] : '';
            $domain_rank = isset($link['domain_from_rank']) ? $link['domain_from_rank'] : 0;

            // Store every individual referring URL for export
            $grouped[$target_path]['all_referrers'][] = array(
                'domain' => $domain_from,
                'url' => $url_from,
                'domain_rank' => $domain_rank,
            );

            if (!empty($domain_from) && !in_array($domain_from, $grouped[$target_path]['referring_domains_list'])) {
                $grouped[$target_path]['referring_domains_list'][] = $domain_from;

                // Add to top referrers (limit to 10 for UI display)
                if (count($grouped[$target_path]['top_referrers']) < 10) {
                    $grouped[$target_path]['top_referrers'][] = array(
                        'domain' => $domain_from,
                        'url' => $url_from,
                        'domain_rank' => $domain_rank,
                        'backlinks' => 1,
                    );
                }
            }
        }

        // Convert to array and add referring_domains count
        $results = array();
        foreach ($grouped as $item) {
            $item['referring_domains'] = count($item['referring_domains_list']);
            unset($item['referring_domains_list']);
            $results[] = $item;
        }

        // Sort by backlink count descending
        usort($results, function($a, $b) {
            return $b['backlink_count'] - $a['backlink_count'];
        });

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF Backlink API: Grouped into ' . count($results) . ' unique target URLs');
        }

        // Check which URLs are actually 404s
        $broken = array();
        $checked = 0;
        $max_checks = 100; // Limit URL checks for performance

        foreach ($results as $page) {
            if ($checked >= $max_checks) {
                break;
            }

            // Ensure all required fields exist
            if (empty($page['dead_page'])) {
                $checked++;
                continue;
            }

            $status = $this->check_url_status($page['dead_page']);

            if (defined('WP_DEBUG') && WP_DEBUG && $checked < 5) {
                error_log('SF Backlink API: URL check - ' . $page['dead_page'] . ' = ' . $status);
            }

            if ($status === 404 || $status === 410) {
                // Normalize top_referrers to ensure all required fields exist
                $normalized_referrers = array();
                if (!empty($page['top_referrers']) && is_array($page['top_referrers'])) {
                    foreach ($page['top_referrers'] as $ref) {
                        $normalized_referrers[] = array(
                            'domain' => isset($ref['domain']) ? $ref['domain'] : 'Unknown',
                            'url' => isset($ref['url']) ? $ref['url'] : '',
                            'domain_rank' => isset($ref['domain_rank']) ? $ref['domain_rank'] : 0,
                            'backlinks' => isset($ref['backlinks']) ? $ref['backlinks'] : 1,
                        );
                    }
                }

                // Ensure the result has all required fields
                $dead_page_path = isset($page['dead_page_path']) ? $page['dead_page_path'] : wp_parse_url($page['dead_page'], PHP_URL_PATH);
                if (empty($dead_page_path)) {
                    $dead_page_path = '/';
                }

                $broken[] = array(
                    'dead_page' => $page['dead_page'],
                    'dead_page_path' => $dead_page_path,
                    'status_code' => $status,
                    'backlink_count' => isset($page['backlink_count']) ? $page['backlink_count'] : 0,
                    'referring_domains' => isset($page['referring_domains']) ? $page['referring_domains'] : 0,
                    'top_referrers' => $normalized_referrers,
                    'all_referrers' => isset($page['all_referrers']) ? $page['all_referrers'] : array(),
                );
            }

            $checked++;
        }

        // Store debug info for troubleshooting (accessible via scan results)
        $this->last_scan_debug = array(
            'total_backlinks_from_api' => count($items),
            'unique_target_urls' => count($results),
            'urls_checked' => $checked,
            'broken_urls_found' => count($broken),
            'max_checks_limit' => $max_checks,
        );

        // Summary logging
        error_log('=== SF BACKLINK API: SCAN COMPLETE ===');
        error_log('SF Backlink API: Total backlinks from API: ' . count($items));
        error_log('SF Backlink API: Unique target URLs: ' . count($results));
        error_log('SF Backlink API: URLs checked for 404: ' . $checked);
        error_log('SF Backlink API: BROKEN (404/410) URLs found: ' . count($broken));
        if (count($broken) > 0) {
            error_log('SF Backlink API: First broken URL: ' . ($broken[0]['dead_page'] ?? 'N/A'));
        }
        error_log('=====================================');

        return $broken;
    }

    /**
     * Get debug info from last scan
     * @return array
     */
    public function get_last_scan_debug() {
        return isset($this->last_scan_debug) ? $this->last_scan_debug : array();
    }

    /**
     * Call hosted API (rate limited, for users without own credentials)
     *
     * @param string $domain Domain to scan
     * @return array|WP_Error
     */
    private function call_hosted_api($domain) {
        // Debug logging for hosted API
        error_log('=== SF HOSTED API: STARTING SCAN ===');
        error_log('SF Hosted API: Domain: ' . $domain);
        error_log('SF Hosted API: Endpoint: ' . self::API_ENDPOINT);
        error_log('SF Hosted API: Timestamp: ' . date('Y-m-d H:i:s'));

        $response = wp_remote_post(self::API_ENDPOINT, array(
            'timeout' => 60,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'domain' => $domain,
            )),
        ));

        if (is_wp_error($response)) {
            error_log('SF Hosted API: WP_ERROR - ' . $response->get_error_message());
            return new WP_Error('api_error', $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // Debug: Log raw response
        error_log('SF Hosted API: HTTP Status: ' . $status_code);
        error_log('SF Hosted API: Response length: ' . strlen($body) . ' bytes');
        error_log('SF Hosted API: Raw response (first 2000 chars): ' . substr($body, 0, 2000));

        $data = json_decode($body, true);

        if (!$data) {
            error_log('SF Hosted API: JSON decode failed');
            return new WP_Error('invalid_response', __('Invalid response from API', 'screaming-fixes'));
        }

        // Debug: Log parsed data structure
        error_log('SF Hosted API: Response keys: ' . implode(', ', array_keys($data)));
        if (isset($data['results'])) {
            error_log('SF Hosted API: Results count: ' . count($data['results']));
            if (!empty($data['results'])) {
                error_log('SF Hosted API: First result keys: ' . implode(', ', array_keys($data['results'][0])));
                error_log('SF Hosted API: First result: ' . wp_json_encode($data['results'][0]));
            }
        }

        // Handle rate limiting
        if ($status_code === 429) {
            $message = isset($data['error']) ? $data['error'] : __('Rate limit exceeded. Please try again later.', 'screaming-fixes');
            error_log('SF Hosted API: Rate limited - ' . $message);
            return new WP_Error('rate_limited', $message);
        }

        // Handle other errors
        if (isset($data['error'])) {
            error_log('SF Hosted API: Error in response - ' . $data['error']);
            return new WP_Error('api_error', $data['error']);
        }

        // Return results on success
        if (isset($data['success']) && $data['success']) {
            $results = isset($data['results']) ? $data['results'] : array();
            error_log('SF Hosted API: Success! Processing ' . count($results) . ' results');
            $normalized = $this->normalize_api_results($results);
            error_log('SF Hosted API: After normalization: ' . count($normalized) . ' results');
            if (!empty($normalized)) {
                error_log('SF Hosted API: First normalized result: ' . wp_json_encode($normalized[0]));
            }

            // Store debug info for troubleshooting
            $this->last_scan_debug = array(
                'api_type' => 'hosted',
                'raw_results_count' => count($results),
                'normalized_results_count' => count($normalized),
                'first_raw_result' => !empty($results) ? $results[0] : null,
                'first_normalized_result' => !empty($normalized) ? $normalized[0] : null,
            );

            error_log('=== SF HOSTED API: SCAN COMPLETE ===');
            return $normalized;
        }

        error_log('SF Hosted API: Unknown response format');
        return new WP_Error('unknown_error', __('Unknown error from API', 'screaming-fixes'));
    }

    /**
     * Normalize domain input
     *
     * @param string $domain
     * @return string
     */
    private function normalize_domain($domain) {
        // Remove protocol
        $domain = preg_replace('#^https?://#', '', $domain);

        // Remove www.
        $domain = preg_replace('/^www\./', '', $domain);

        // Remove trailing slash
        $domain = rtrim($domain, '/');

        // Remove any path
        $domain = explode('/', $domain)[0];

        return strtolower($domain);
    }

    /**
     * Normalize API results to match expected template format
     *
     * @param array $results Raw API results
     * @return array Normalized results
     */
    private function normalize_api_results($results) {
        $normalized = array();

        foreach ($results as $item) {
            // Get the dead page URL - try multiple possible field names
            $dead_page = isset($item['dead_page']) ? $item['dead_page'] : '';
            if (empty($dead_page) && isset($item['target_url'])) {
                $dead_page = $item['target_url'];
            }

            // Extract the path from the dead page URL
            $dead_page_path = '';
            if (!empty($dead_page)) {
                $dead_page_path = wp_parse_url($dead_page, PHP_URL_PATH);
                if (empty($dead_page_path)) {
                    $dead_page_path = '/';
                }
            } elseif (isset($item['dead_page_path'])) {
                $dead_page_path = $item['dead_page_path'];
            }

            // Get referring domains count
            $referring_domains = 0;
            if (isset($item['referring_domains'])) {
                $referring_domains = $item['referring_domains'];
            } elseif (isset($item['referring_domains_count'])) {
                $referring_domains = $item['referring_domains_count'];
            }

            $normalized[] = array(
                'dead_page' => $dead_page,
                'dead_page_path' => $dead_page_path,
                'status_code' => isset($item['status_code']) && $item['status_code'] ? $item['status_code'] : 404,
                'backlink_count' => isset($item['backlink_count']) ? $item['backlink_count'] : 0,
                'referring_domains' => $referring_domains,
                'top_referrers' => $this->normalize_referrers(isset($item['top_referrers']) ? $item['top_referrers'] : array()),
            );
        }

        return $normalized;
    }

    /**
     * Normalize referrer data to match expected template format
     *
     * @param array $referrers Raw referrer data
     * @return array Normalized referrers
     */
    private function normalize_referrers($referrers) {
        $normalized = array();

        foreach ($referrers as $ref) {
            // Get domain - try multiple field names
            $domain = 'Unknown';
            if (isset($ref['domain'])) {
                $domain = $ref['domain'];
            } elseif (isset($ref['domain_from'])) {
                $domain = $ref['domain_from'];
            }

            // Get URL - try multiple field names
            $url = '';
            if (isset($ref['url'])) {
                $url = $ref['url'];
            } elseif (isset($ref['url_from'])) {
                $url = $ref['url_from'];
            }

            // Get domain rank - try multiple field names
            $domain_rank = 0;
            if (isset($ref['domain_rank'])) {
                $domain_rank = $ref['domain_rank'];
            } elseif (isset($ref['domain_from_rank'])) {
                $domain_rank = $ref['domain_from_rank'];
            }

            $normalized[] = array(
                'domain' => $domain,
                'url' => $url,
                'domain_rank' => $domain_rank,
                'backlinks' => isset($ref['backlinks']) ? $ref['backlinks'] : 1,
            );
        }

        return $normalized;
    }

    /**
     * Get backlinks grouped by target page
     *
     * @param string $domain
     * @return array|WP_Error
     */
    private function get_backlinks_by_page($domain) {
        $endpoint = 'backlinks/backlinks/live';

        // Free tier gets fewer results
        $limit = $this->is_free_tier ? 500 : 1000;

        $post_data = [
            [
                'target' => $domain,
                'limit' => $limit,
                'mode' => 'as_is',
                'filters' => [
                    ['dofollow', '=', true],
                ],
            ],
        ];

        $response = $this->make_request($endpoint, $post_data);

        if (is_wp_error($response)) {
            return $response;
        }

        if (!isset($response['tasks'][0]['result'][0]['items'])) {
            return [];
        }

        $items = $response['tasks'][0]['result'][0]['items'];

        // Group by target page
        $pages = [];

        foreach ($items as $item) {
            $target_url = $item['url_to'] ?? '';

            if (empty($target_url)) {
                continue;
            }

            if (!isset($pages[$target_url])) {
                $pages[$target_url] = [
                    'backlink_count' => 0,
                    'referring_domains' => 0,
                    'referrers' => [],
                    'domains_seen' => [],
                ];
            }

            $pages[$target_url]['backlink_count']++;

            $source_domain = $item['domain_from'] ?? '';
            if ($source_domain && !in_array($source_domain, $pages[$target_url]['domains_seen'])) {
                $pages[$target_url]['domains_seen'][] = $source_domain;
                $pages[$target_url]['referring_domains']++;
                $pages[$target_url]['referrers'][] = [
                    'domain' => $source_domain,
                    'url' => $item['url_from'] ?? '',
                    'anchor' => $item['anchor'] ?? '',
                ];
            }
        }

        // Remove internal tracking array
        foreach ($pages as $url => &$data) {
            unset($data['domains_seen']);
        }

        return $pages;
    }

    /**
     * Check URL status code
     *
     * @param string $url
     * @return int Status code or 0 on error
     */
    private function check_url_status($url) {
        $response = wp_remote_head($url, [
            'timeout' => 10,
            'redirection' => 0,
            'sslverify' => false,
        ]);

        if (is_wp_error($response)) {
            // Try GET request if HEAD fails
            $response = wp_remote_get($url, [
                'timeout' => 10,
                'redirection' => 0,
                'sslverify' => false,
            ]);

            if (is_wp_error($response)) {
                return 0;
            }
        }

        return wp_remote_retrieve_response_code($response);
    }

    /**
     * Make API request
     *
     * @param string $endpoint API endpoint
     * @param array $data Request data
     * @param string $method HTTP method
     * @return array|WP_Error
     */
    private function make_request($endpoint, $data = [], $method = 'POST') {
        $url = $this->api_url . $endpoint;

        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($this->api_login . ':' . $this->api_password),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 120,
        ];

        if ($method === 'POST' && !empty($data)) {
            $args['body'] = wp_json_encode($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return new WP_Error(
                'api_request_failed',
                sprintf(
                    __('API request failed: %s', 'screaming-fixes'),
                    $response->get_error_message()
                )
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status_code === 401) {
            return new WP_Error('unauthorized', __('Invalid API credentials. Please check your DataForSEO login and password.', 'screaming-fixes'));
        }

        if ($status_code !== 200) {
            return new WP_Error(
                'api_error',
                sprintf(
                    __('API returned status %d', 'screaming-fixes'),
                    $status_code
                )
            );
        }

        $decoded = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', __('Failed to parse API response.', 'screaming-fixes'));
        }

        return $decoded;
    }

    /**
     * Get suggested redirect destination for a dead page
     *
     * Uses Claude AI with web search if API key available, otherwise falls back to WP_Query
     *
     * @param string $dead_page_url The dead page URL
     * @param string $scanned_domain The domain that was scanned (target for replacement URLs)
     * @param array $referrers Optional array of referring domains/URLs for context
     * @return array Result with 'destination', 'source' (ai|wp_query|homepage), and optional 'explanation'
     */
    public function get_suggested_destination($dead_page_url, $scanned_domain, $referrers = array()) {
        // Extract path for analysis
        $path = wp_parse_url($dead_page_url, PHP_URL_PATH) ?: '/';

        // Normalize the scanned domain
        $scanned_domain = preg_replace('#^https?://#', '', $scanned_domain);
        $scanned_domain = preg_replace('/^www\./', '', $scanned_domain);
        $scanned_domain = rtrim($scanned_domain, '/');

        // Check for cached AI suggestion first (7-day cache)
        $cache_key = 'sf_ai_suggest_' . md5($dead_page_url . $scanned_domain);
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        // Try Claude AI if API key is configured
        $claude_api_key = get_option('sf_claude_api_key');
        if (!empty($claude_api_key)) {
            $ai_result = $this->get_ai_suggested_destination($dead_page_url, $scanned_domain, $referrers, $claude_api_key);
            if ($ai_result && !empty($ai_result['destination'])) {
                // Cache for 7 days
                set_transient($cache_key, $ai_result, 7 * DAY_IN_SECONDS);
                return $ai_result;
            }
        }

        // Fall back to WP_Query-based similar page finder
        $site_url = 'https://' . $scanned_domain;
        $similar = $this->find_similar_page($path, $site_url);

        if ($similar) {
            $result = array(
                'destination' => $similar,
                'source' => 'wp_query',
                'explanation' => __('Found a similar page in your WordPress site based on the URL slug.', 'screaming-fixes'),
            );
            set_transient($cache_key, $result, 7 * DAY_IN_SECONDS);
            return $result;
        }

        // Fall back to homepage
        $result = array(
            'destination' => trailingslashit($site_url),
            'source' => 'homepage',
            'explanation' => __('No matching page found. Defaulting to homepage.', 'screaming-fixes'),
        );
        return $result;
    }

    /**
     * Get AI-suggested destination using Claude API with web search
     *
     * @param string $dead_page_url The dead page URL
     * @param string $scanned_domain The domain to search for replacement URLs
     * @param array $referrers Array of referring domains for context
     * @param string $api_key Claude API key
     * @return array|null Result array or null on failure
     */
    private function get_ai_suggested_destination($dead_page_url, $scanned_domain, $referrers, $api_key) {
        // Build context about referrers
        $referrer_context = '';
        if (!empty($referrers)) {
            $referrer_list = array();
            foreach (array_slice($referrers, 0, 5) as $ref) {
                if (isset($ref['domain'])) {
                    $referrer_list[] = $ref['domain'];
                }
            }
            if (!empty($referrer_list)) {
                $referrer_context = sprintf(
                    "\n\nThis dead page has backlinks from these domains: %s",
                    implode(', ', $referrer_list)
                );
            }
        }

        // Build the prompt
        $prompt = sprintf(
            "I need to find the best replacement URL on %s for a dead page that used to exist at: %s%s\n\n" .
            "Please search %s to find the most relevant existing page that would be a good 301 redirect destination. " .
            "The replacement URL MUST be on the domain %s (not any other domain).\n\n" .
            "Consider:\n" .
            "1. Content relevance - what topic/content did the dead URL likely cover based on its path?\n" .
            "2. User intent - what were people looking for when they clicked links to this dead page?\n" .
            "3. SEO value - which existing page would best preserve the link equity?\n\n" .
            "Respond with ONLY a JSON object in this exact format (no markdown, no explanation outside JSON):\n" .
            "{\n" .
            "  \"destination\": \"https://%s/path/to/best-match\",\n" .
            "  \"confidence\": \"high|medium|low\",\n" .
            "  \"explanation\": \"Brief explanation of why this page is the best match\"\n" .
            "}\n\n" .
            "If you cannot find any relevant page on %s, respond with:\n" .
            "{\n" .
            "  \"destination\": null,\n" .
            "  \"confidence\": \"none\",\n" .
            "  \"explanation\": \"Reason why no match was found\"\n" .
            "}",
            $scanned_domain,
            $dead_page_url,
            $referrer_context,
            $scanned_domain,
            $scanned_domain,
            $scanned_domain,
            $scanned_domain
        );

        // Call Claude API with web search enabled
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', array(
            'timeout' => 60,
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01',
            ),
            'body' => wp_json_encode(array(
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 1024,
                'tools' => array(
                    array(
                        'type' => 'web_search_20250305',
                        'name' => 'web_search',
                        'max_uses' => 5,
                    ),
                ),
                'messages' => array(
                    array(
                        'role' => 'user',
                        'content' => $prompt,
                    ),
                ),
            )),
        ));

        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF AI Suggest: API error - ' . $response->get_error_message());
            }
            return null;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF AI Suggest: API response status ' . $status_code);
        }

        if ($status_code !== 200) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF AI Suggest: API returned non-200 status: ' . $body);
            }
            return null;
        }

        $data = json_decode($body, true);

        if (!isset($data['content']) || !is_array($data['content'])) {
            return null;
        }

        // Extract text response from Claude
        $text_response = '';
        foreach ($data['content'] as $block) {
            if (isset($block['type']) && $block['type'] === 'text') {
                $text_response .= $block['text'];
            }
        }

        if (empty($text_response)) {
            return null;
        }

        // Parse the JSON response
        $json_match = preg_match('/\{[^{}]*"destination"[^{}]*\}/s', $text_response, $matches);
        if (!$json_match) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF AI Suggest: Could not extract JSON from response: ' . $text_response);
            }
            return null;
        }

        $suggestion = json_decode($matches[0], true);
        if (!$suggestion || !isset($suggestion['destination'])) {
            return null;
        }

        // Check if AI found no match
        if ($suggestion['destination'] === null || $suggestion['confidence'] === 'none') {
            return array(
                'destination' => null,
                'source' => 'ai_no_match',
                'explanation' => isset($suggestion['explanation']) ? $suggestion['explanation'] : __('AI could not find a suitable replacement page.', 'screaming-fixes'),
            );
        }

        // Validate the destination URL is on the correct domain
        $dest_host = wp_parse_url($suggestion['destination'], PHP_URL_HOST);
        $dest_host = preg_replace('/^www\./', '', $dest_host);

        if (stripos($dest_host, $scanned_domain) === false && stripos($scanned_domain, $dest_host) === false) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF AI Suggest: Destination domain mismatch. Got: ' . $dest_host . ', Expected: ' . $scanned_domain);
            }
            return null;
        }

        // Verify the suggested URL is actually live before returning
        $suggested_url = $suggestion['destination'];
        $confidence = isset($suggestion['confidence']) ? $suggestion['confidence'] : 'medium';
        $explanation = isset($suggestion['explanation']) ? $suggestion['explanation'] : '';

        $verify = wp_remote_head($suggested_url, array(
            'timeout' => 5,
            'redirection' => 3,
            'sslverify' => false,
            'user-agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
        ));

        // If HEAD fails, try GET as fallback
        if (is_wp_error($verify)) {
            $verify = wp_remote_get($suggested_url, array(
                'timeout' => 5,
                'redirection' => 3,
                'sslverify' => false,
                'user-agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            ));
        }

        if (is_wp_error($verify)) {
            // Can't verify - mark as unverified but still return
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF AI Suggest: Could not verify URL - ' . $verify->get_error_message());
            }
            return array(
                'destination' => $suggested_url,
                'source' => 'ai',
                'confidence' => $confidence,
                'explanation' => $explanation,
                'verified' => 'unverified',
            );
        }

        $status_code = wp_remote_retrieve_response_code($verify);

        if ($status_code >= 400) {
            // URL is dead - DO NOT return it, treat as no match
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('SF AI Suggest: URL returned HTTP ' . $status_code . ' - treating as no match');
            }
            return array(
                'destination' => null,
                'source' => 'ai_no_match',
                'confidence' => null,
                'explanation' => sprintf(
                    __('AI found a potential match (%s) but it returned HTTP %d. No live replacement found.', 'screaming-fixes'),
                    $suggested_url,
                    $status_code
                ),
                'verified' => 'dead',
            );
        }

        // URL is live - return the suggestion
        return array(
            'destination' => $suggested_url,
            'source' => 'ai',
            'confidence' => $confidence,
            'explanation' => $explanation,
            'verified' => 'live',
        );
    }

    /**
     * Get demo data for testing
     *
     * @param string $domain Domain being scanned
     * @return array Simulated backlink data
     */
    private function get_demo_data($domain) {
        // Return empty array to test "no dead backlinks" success state
        if (self::DEMO_EMPTY_RESULTS) {
            return [];
        }

        $domain = $this->normalize_domain($domain);

        return [
            [
                'dead_page' => 'https://' . $domain . '/blog/seo-tips-2023/',
                'dead_page_path' => '/blog/seo-tips-2023/',
                'backlink_count' => 47,
                'referring_domains' => 12,
                'top_referrers' => [
                    ['domain' => 'forbes.com', 'url' => 'https://www.forbes.com/sites/business/2023/05/best-seo-guides/', 'domain_rank' => 94, 'backlinks' => 5],
                    ['domain' => 'entrepreneur.com', 'url' => 'https://www.entrepreneur.com/article/seo-resources-list', 'domain_rank' => 91, 'backlinks' => 4],
                    ['domain' => 'searchenginejournal.com', 'url' => 'https://www.searchenginejournal.com/seo-tools/12345/', 'domain_rank' => 85, 'backlinks' => 3],
                    ['domain' => 'moz.com', 'url' => 'https://moz.com/blog/link-roundup-2023', 'domain_rank' => 91, 'backlinks' => 2],
                    ['domain' => 'ahrefs.com', 'url' => 'https://ahrefs.com/blog/best-seo-tips/', 'domain_rank' => 89, 'backlinks' => 2],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/resources/free-keyword-tool/',
                'dead_page_path' => '/resources/free-keyword-tool/',
                'backlink_count' => 34,
                'referring_domains' => 8,
                'top_referrers' => [
                    ['domain' => 'neilpatel.com', 'url' => 'https://neilpatel.com/blog/best-keyword-tools/', 'domain_rank' => 92, 'backlinks' => 6],
                    ['domain' => 'hubspot.com', 'url' => 'https://blog.hubspot.com/marketing/keyword-research-tools', 'domain_rank' => 93, 'backlinks' => 4],
                    ['domain' => 'semrush.com', 'url' => 'https://www.semrush.com/blog/free-seo-tools/', 'domain_rank' => 90, 'backlinks' => 3],
                    ['domain' => 'backlinko.com', 'url' => 'https://backlinko.com/keyword-research-tools', 'domain_rank' => 78, 'backlinks' => 2],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/case-studies/ecommerce-seo-success/',
                'dead_page_path' => '/case-studies/ecommerce-seo-success/',
                'backlink_count' => 28,
                'referring_domains' => 6,
                'top_referrers' => [
                    ['domain' => 'shopify.com', 'url' => 'https://www.shopify.com/blog/ecommerce-seo-case-studies', 'domain_rank' => 93, 'backlinks' => 5],
                    ['domain' => 'bigcommerce.com', 'url' => 'https://www.bigcommerce.com/articles/ecommerce-seo/', 'domain_rank' => 82, 'backlinks' => 3],
                    ['domain' => 'practicalecommerce.com', 'url' => 'https://www.practicalecommerce.com/seo-success-stories', 'domain_rank' => 65, 'backlinks' => 2],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/guides/local-seo-checklist/',
                'dead_page_path' => '/guides/local-seo-checklist/',
                'backlink_count' => 23,
                'referring_domains' => 7,
                'top_referrers' => [
                    ['domain' => 'brightlocal.com', 'url' => 'https://www.brightlocal.com/learn/local-seo-resources/', 'domain_rank' => 72, 'backlinks' => 4],
                    ['domain' => 'whitespark.ca', 'url' => 'https://whitespark.ca/blog/local-seo-guide/', 'domain_rank' => 58, 'backlinks' => 3],
                    ['domain' => 'localseoguide.com', 'url' => 'https://www.localseoguide.com/local-seo-checklist/', 'domain_rank' => 45, 'backlinks' => 2],
                    ['domain' => 'searchengineland.com', 'url' => 'https://searchengineland.com/local-seo-resources/', 'domain_rank' => 88, 'backlinks' => 2],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/tools/meta-tag-generator/',
                'dead_page_path' => '/tools/meta-tag-generator/',
                'backlink_count' => 19,
                'referring_domains' => 5,
                'top_referrers' => [
                    ['domain' => 'w3schools.com', 'url' => 'https://www.w3schools.com/tags/tag_meta.asp', 'domain_rank' => 92, 'backlinks' => 3],
                    ['domain' => 'css-tricks.com', 'url' => 'https://css-tricks.com/meta-tags-you-should-know/', 'domain_rank' => 84, 'backlinks' => 2],
                    ['domain' => 'htmlgoodies.com', 'url' => 'https://www.htmlgoodies.com/getting-started/meta-tags/', 'domain_rank' => 52, 'backlinks' => 1],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/blog/google-algorithm-update-march/',
                'dead_page_path' => '/blog/google-algorithm-update-march/',
                'backlink_count' => 15,
                'referring_domains' => 4,
                'top_referrers' => [
                    ['domain' => 'searchengineland.com', 'url' => 'https://searchengineland.com/google-algorithm-updates-history/', 'domain_rank' => 88, 'backlinks' => 4],
                    ['domain' => 'seroundtable.com', 'url' => 'https://www.seroundtable.com/category/google-updates/', 'domain_rank' => 75, 'backlinks' => 3],
                    ['domain' => 'mariehaynes.com', 'url' => 'https://www.mariehaynes.com/algorithm-updates/', 'domain_rank' => 48, 'backlinks' => 2],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/services/link-building-outreach/',
                'dead_page_path' => '/services/link-building-outreach/',
                'backlink_count' => 12,
                'referring_domains' => 4,
                'top_referrers' => [
                    ['domain' => 'reddit.com', 'url' => 'https://www.reddit.com/r/SEO/comments/abc123/best_link_building_services/', 'domain_rank' => 91, 'backlinks' => 3],
                    ['domain' => 'pointblankseo.com', 'url' => 'https://pointblankseo.com/link-building-services/', 'domain_rank' => 55, 'backlinks' => 2],
                    ['domain' => 'fatjoe.com', 'url' => 'https://fatjoe.com/blog/outreach-services/', 'domain_rank' => 42, 'backlinks' => 1],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/webinars/technical-seo-audit/',
                'dead_page_path' => '/webinars/technical-seo-audit/',
                'backlink_count' => 9,
                'referring_domains' => 3,
                'top_referrers' => [
                    ['domain' => 'oncrawl.com', 'url' => 'https://www.oncrawl.com/technical-seo/webinar-resources/', 'domain_rank' => 62, 'backlinks' => 2],
                    ['domain' => 'screamingfrog.co.uk', 'url' => 'https://www.screamingfrog.co.uk/seo-spider/resources/', 'domain_rank' => 71, 'backlinks' => 2],
                    ['domain' => 'sitebulb.com', 'url' => 'https://sitebulb.com/resources/guides/', 'domain_rank' => 55, 'backlinks' => 1],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/templates/content-brief-template/',
                'dead_page_path' => '/templates/content-brief-template/',
                'backlink_count' => 7,
                'referring_domains' => 3,
                'top_referrers' => [
                    ['domain' => 'contentmarketinginstitute.com', 'url' => 'https://contentmarketinginstitute.com/templates/content-brief/', 'domain_rank' => 82, 'backlinks' => 2],
                    ['domain' => 'copyblogger.com', 'url' => 'https://copyblogger.com/content-templates/', 'domain_rank' => 78, 'backlinks' => 2],
                    ['domain' => 'clearscope.io', 'url' => 'https://www.clearscope.io/blog/content-brief-template/', 'domain_rank' => 45, 'backlinks' => 1],
                ],
                'status_code' => 404,
            ],
            [
                'dead_page' => 'https://' . $domain . '/blog/voice-search-optimization/',
                'dead_page_path' => '/blog/voice-search-optimization/',
                'backlink_count' => 5,
                'referring_domains' => 2,
                'top_referrers' => [
                    ['domain' => 'thinkwithgoogle.com', 'url' => 'https://www.thinkwithgoogle.com/marketing-strategies/voice-search/', 'domain_rank' => 95, 'backlinks' => 2],
                    ['domain' => 'searchengineland.com', 'url' => 'https://searchengineland.com/voice-search-seo/', 'domain_rank' => 88, 'backlinks' => 1],
                ],
                'status_code' => 410,
            ],
        ];
    }

    /**
     * Get DR class for styling
     *
     * @param int $dr Domain Rating value
     * @return string CSS class suffix
     */
    public static function get_dr_class($dr) {
        if ($dr >= 80) {
            return 'high';
        }
        if ($dr >= 50) {
            return 'medium';
        }
        if ($dr >= 20) {
            return 'low';
        }
        return 'very-low';
    }

    /**
     * Find a similar existing page based on URL path
     *
     * @param string $path Dead page path
     * @param string $site_url Site URL
     * @return string|null Similar page URL or null
     */
    private function find_similar_page($path, $site_url) {
        // Extract slug from path
        $path = trim($path, '/');
        $parts = explode('/', $path);

        if (empty($parts)) {
            return null;
        }

        // Try to find a page with a similar slug
        $slug = end($parts);

        // Remove common prefixes/suffixes
        $slug = preg_replace('/^(old-|archived-|copy-of-)/i', '', $slug);
        $slug = preg_replace('/(-copy|-old|-archived|\d{4})$/i', '', $slug);

        if (strlen($slug) < 3) {
            return null;
        }

        // Search for pages with similar slug
        $args = [
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'name' => $slug,
            'posts_per_page' => 1,
        ];

        $query = new WP_Query($args);

        if ($query->have_posts()) {
            return get_permalink($query->posts[0]);
        }

        // Try a broader search
        $args = [
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            's' => str_replace('-', ' ', $slug),
            'posts_per_page' => 1,
        ];

        $query = new WP_Query($args);

        if ($query->have_posts()) {
            return get_permalink($query->posts[0]);
        }

        return null;
    }
}
