<?php
/**
 * Backlink Evaluator API Integration
 *
 * Handles communication with DataForSEO Backlinks API for the Backlink Evaluator tool.
 * Uses summary/live and backlinks/live endpoints to fetch comprehensive backlink data.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Backlink_Evaluator_API {

    /**
     * Demo mode - set to true to use dummy data instead of real API calls
     */
    const DEMO_MODE = false;

    /**
     * Hosted API endpoint for free tier users
     */
    const HOSTED_ENDPOINT = 'https://screamingfixes.com/.netlify/functions/backlink-evaluator';

    /**
     * DataForSEO API base URL
     */
    const DATAFORSEO_API_URL = 'https://api.dataforseo.com/v3/';

    /**
     * Maximum backlinks to fetch per request
     */
    const BACKLINK_LIMIT = 1000;

    /**
     * Track which API was used for the last request
     * @var string
     */
    private $last_api_used = '';

    /**
     * Debug info from last request
     * @var array
     */
    private $last_request_debug = [];

    /**
     * Check if user has their own DataForSEO credentials
     *
     * @return bool
     */
    public static function has_user_credentials() {
        $login = get_option('sf_dataforseo_login');
        $password = get_option('sf_dataforseo_password');
        return !empty($login) && !empty($password);
    }

    /**
     * Check if demo mode is enabled
     *
     * @return bool
     */
    public static function is_demo_mode() {
        return self::DEMO_MODE;
    }

    /**
     * Get which API was used for the last request
     *
     * @return string 'direct', 'hosted', or 'demo'
     */
    public function get_last_api_used() {
        return $this->last_api_used;
    }

    /**
     * Get debug info from last request
     *
     * @return array
     */
    public function get_last_request_debug() {
        return $this->last_request_debug;
    }

    /**
     * Fetch all backlink data for a domain (summary + backlink list)
     *
     * @param string $domain Domain to scan
     * @return array|WP_Error Combined data with 'summary' and 'backlinks' keys
     */
    public function fetch_all($domain) {
        // Return demo data if in demo mode
        if (self::DEMO_MODE) {
            $this->last_api_used = 'demo';
            return $this->get_demo_data($domain);
        }

        $domain = $this->normalize_domain($domain);

        // Check if user has their own credentials
        if (self::has_user_credentials()) {
            $this->last_api_used = 'direct';
            return $this->fetch_direct($domain);
        }

        // Use hosted API for free tier
        $this->last_api_used = 'hosted';
        return $this->fetch_hosted($domain);
    }

    /**
     * Fetch data directly from DataForSEO (for users with own API key)
     *
     * @param string $domain Normalized domain
     * @return array|WP_Error
     */
    private function fetch_direct($domain) {
        $login = get_option('sf_dataforseo_login');
        $password = get_option('sf_dataforseo_password');
        $auth = base64_encode($login . ':' . $password);

        error_log('=== SF BACKLINK EVALUATOR API: STARTING SCAN ===');
        error_log('SF Backlink Evaluator: Domain: ' . $domain);
        error_log('SF Backlink Evaluator: Timestamp: ' . date('Y-m-d H:i:s'));

        // Step 1: Fetch summary data
        $summary = $this->fetch_summary_direct($domain, $auth);
        if (is_wp_error($summary)) {
            return $summary;
        }

        // Step 2: Fetch backlinks list
        $backlinks = $this->fetch_backlinks_direct($domain, $auth);
        if (is_wp_error($backlinks)) {
            return $backlinks;
        }

        error_log('=== SF BACKLINK EVALUATOR API: SCAN COMPLETE ===');
        error_log('SF Backlink Evaluator: Summary fetched successfully');
        error_log('SF Backlink Evaluator: Backlinks count: ' . count($backlinks));

        return [
            'summary' => $summary,
            'backlinks' => $backlinks,
            'domain' => $domain,
            'fetched_at' => current_time('mysql'),
        ];
    }

    /**
     * Fetch summary data from DataForSEO
     *
     * @param string $domain Normalized domain
     * @param string $auth Base64 encoded credentials
     * @return array|WP_Error
     */
    private function fetch_summary_direct($domain, $auth) {
        $post_data = wp_json_encode([
            [
                'target' => $domain,
                'include_subdomains' => true,
            ]
        ]);

        error_log('SF Backlink Evaluator: Fetching summary from /v3/backlinks/summary/live');

        $response = wp_remote_post(self::DATAFORSEO_API_URL . 'backlinks/summary/live', [
            'timeout' => 60,
            'headers' => [
                'Authorization' => 'Basic ' . $auth,
                'Content-Type' => 'application/json',
            ],
            'body' => $post_data,
        ]);

        if (is_wp_error($response)) {
            error_log('SF Backlink Evaluator: Summary API error - ' . $response->get_error_message());
            return new WP_Error('api_error', __('Failed to connect to DataForSEO: ', 'screaming-fixes') . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status_code === 401) {
            return new WP_Error('unauthorized', __('Invalid DataForSEO credentials. Please check your login and password.', 'screaming-fixes'));
        }

        $data = json_decode($body, true);

        if (!$data || !isset($data['status_code']) || $data['status_code'] !== 20000) {
            $message = isset($data['status_message']) ? $data['status_message'] : __('DataForSEO API error', 'screaming-fixes');
            return new WP_Error('api_error', $message);
        }

        // Check for task-level errors
        if (isset($data['tasks'][0]['status_code']) && $data['tasks'][0]['status_code'] !== 20000) {
            $task_message = $data['tasks'][0]['status_message'] ?? __('Unknown task error', 'screaming-fixes');

            if ($data['tasks'][0]['status_code'] === 40204) {
                return new WP_Error('subscription_required', __('DataForSEO Backlinks API subscription required. Please activate your Backlinks subscription at app.dataforseo.com', 'screaming-fixes'));
            }

            return new WP_Error('task_error', $task_message);
        }

        $result = isset($data['tasks'][0]['result'][0]) ? $data['tasks'][0]['result'][0] : [];

        error_log('SF Backlink Evaluator: Summary data received - backlinks: ' . ($result['backlinks'] ?? 0) . ', referring_domains: ' . ($result['referring_domains'] ?? 0));

        return [
            'backlinks' => $result['backlinks'] ?? 0,
            'referring_domains' => $result['referring_domains'] ?? 0,
            'referring_domains_nofollow' => $result['referring_domains_nofollow'] ?? 0,
            'referring_main_domains' => $result['referring_main_domains'] ?? 0,
            'referring_ips' => $result['referring_ips'] ?? 0,
            'referring_subnets' => $result['referring_subnets'] ?? 0,
            'referring_pages' => $result['referring_pages'] ?? 0,
            'rank' => $result['rank'] ?? 0,
        ];
    }

    /**
     * Fetch backlinks list from DataForSEO
     *
     * @param string $domain Normalized domain
     * @param string $auth Base64 encoded credentials
     * @return array|WP_Error
     */
    private function fetch_backlinks_direct($domain, $auth) {
        $post_data = wp_json_encode([
            [
                'target' => $domain,
                'include_subdomains' => true,
                'limit' => self::BACKLINK_LIMIT,
                'mode' => 'as_is',
                'order_by' => ['rank,desc'],
            ]
        ]);

        error_log('SF Backlink Evaluator: Fetching backlinks from /v3/backlinks/backlinks/live');

        $response = wp_remote_post(self::DATAFORSEO_API_URL . 'backlinks/backlinks/live', [
            'timeout' => 120,
            'headers' => [
                'Authorization' => 'Basic ' . $auth,
                'Content-Type' => 'application/json',
            ],
            'body' => $post_data,
        ]);

        if (is_wp_error($response)) {
            error_log('SF Backlink Evaluator: Backlinks API error - ' . $response->get_error_message());
            return new WP_Error('api_error', __('Failed to fetch backlinks: ', 'screaming-fixes') . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status_code === 401) {
            return new WP_Error('unauthorized', __('Invalid DataForSEO credentials.', 'screaming-fixes'));
        }

        $data = json_decode($body, true);

        if (!$data || !isset($data['status_code']) || $data['status_code'] !== 20000) {
            $message = isset($data['status_message']) ? $data['status_message'] : __('DataForSEO API error', 'screaming-fixes');
            return new WP_Error('api_error', $message);
        }

        // Check for task-level errors
        if (isset($data['tasks'][0]['status_code']) && $data['tasks'][0]['status_code'] !== 20000) {
            $task_message = $data['tasks'][0]['status_message'] ?? __('Unknown task error', 'screaming-fixes');
            return new WP_Error('task_error', $task_message);
        }

        $items = isset($data['tasks'][0]['result'][0]['items']) ? $data['tasks'][0]['result'][0]['items'] : [];

        error_log('SF Backlink Evaluator: Received ' . count($items) . ' backlink items');

        // Normalize the backlink data
        return $this->normalize_backlinks($items);
    }

    /**
     * Fetch data from hosted API (for free tier users)
     *
     * @param string $domain Normalized domain
     * @return array|WP_Error
     */
    private function fetch_hosted($domain) {
        error_log('=== SF BACKLINK EVALUATOR HOSTED API: STARTING ===');
        error_log('SF Backlink Evaluator Hosted: Domain: ' . $domain);

        $response = wp_remote_post(self::HOSTED_ENDPOINT, [
            'timeout' => 120,
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode([
                'domain' => $domain,
            ]),
        ]);

        if (is_wp_error($response)) {
            error_log('SF Backlink Evaluator Hosted: WP_ERROR - ' . $response->get_error_message());
            return new WP_Error('api_error', $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        error_log('SF Backlink Evaluator Hosted: HTTP Status: ' . $status_code);

        if ($status_code === 429) {
            $data = json_decode($body, true);
            $message = isset($data['error']) ? $data['error'] : __('Rate limit exceeded. Please try again later.', 'screaming-fixes');
            return new WP_Error('rate_limited', $message);
        }

        $data = json_decode($body, true);

        if (!$data) {
            return new WP_Error('invalid_response', __('Invalid response from API', 'screaming-fixes'));
        }

        if (isset($data['error'])) {
            return new WP_Error('api_error', $data['error']);
        }

        if (isset($data['success']) && $data['success']) {
            error_log('SF Backlink Evaluator Hosted: Success!');
            return [
                'summary' => $data['summary'] ?? [],
                'backlinks' => $data['backlinks'] ?? [],
                'domain' => $domain,
                'fetched_at' => current_time('mysql'),
            ];
        }

        return new WP_Error('unknown_error', __('Unknown error from API', 'screaming-fixes'));
    }

    /**
     * Normalize backlink items from DataForSEO response
     *
     * @param array $items Raw backlink items
     * @return array Normalized backlinks
     */
    private function normalize_backlinks($items) {
        $normalized = [];

        foreach ($items as $item) {
            $normalized[] = [
                // Core fields (displayed in UI table)
                'domain_from' => $item['domain_from'] ?? '',
                'url_from' => $item['url_from'] ?? '',
                'url_to' => $item['url_to'] ?? '',
                'anchor' => $item['anchor'] ?? '',
                'dofollow' => $item['dofollow'] ?? false,
                'rank' => $item['rank'] ?? 0,
                'first_seen' => $item['first_seen'] ?? '',
                'is_lost' => $item['is_lost'] ?? false,
                'is_new' => $item['is_new'] ?? false,
                'attributes' => $item['attributes'] ?? [],

                // Bonus fields (CSV only)
                'domain_from_rank' => $item['domain_from_rank'] ?? 0,
                'page_from_rank' => $item['page_from_rank'] ?? 0,
                'backlink_spam_score' => $item['backlink_spam_score'] ?? 0,
                'item_type' => $item['item_type'] ?? 'anchor',
                'domain_from_tld' => $item['domain_from_tld'] ?? '',
                'url_from_https' => $item['url_from_https'] ?? false,
                'text_pre' => $item['text_pre'] ?? '',
                'text_post' => $item['text_post'] ?? '',
                'image_alt' => $item['image_alt'] ?? '',
                'semantic_location' => $item['semantic_location'] ?? '',
                'links_count' => $item['links_count'] ?? 1,
                'last_seen' => $item['last_seen'] ?? '',
                'url_to_status_code' => $item['url_to_status_code'] ?? null,
                'url_from_status_code' => $item['url_from_status_code'] ?? null,
            ];
        }

        return $normalized;
    }

    /**
     * Normalize domain input
     *
     * @param string $domain
     * @return string
     */
    private function normalize_domain($domain) {
        // Remove protocol
        $domain = preg_replace('#^https?://#', '', $domain);
        // Remove www.
        $domain = preg_replace('/^www\./', '', $domain);
        // Remove trailing slash
        $domain = rtrim($domain, '/');
        // Remove any path
        $domain = explode('/', $domain)[0];

        return strtolower($domain);
    }

    /**
     * Get demo data for testing
     *
     * @param string $domain Domain being scanned
     * @return array Simulated backlink data
     */
    private function get_demo_data($domain) {
        $domain = $this->normalize_domain($domain);

        $summary = [
            'backlinks' => 1247,
            'referring_domains' => 312,
            'referring_domains_nofollow' => 86,
            'referring_main_domains' => 298,
            'referring_ips' => 287,
            'referring_subnets' => 245,
            'referring_pages' => 892,
            'rank' => 450,
        ];

        $backlinks = [
            [
                'domain_from' => 'forbes.com',
                'url_from' => 'https://www.forbes.com/sites/business/best-seo-tools/',
                'url_to' => 'https://' . $domain . '/tools/keyword-research/',
                'anchor' => 'powerful keyword research tool',
                'dofollow' => true,
                'rank' => 945,
                'first_seen' => '2024-08-15T10:30:00Z',
                'is_lost' => false,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 94,
                'page_from_rank' => 72,
                'backlink_spam_score' => 2,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'Check out this',
                'text_post' => 'for your SEO needs.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-28T08:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'searchenginejournal.com',
                'url_from' => 'https://www.searchenginejournal.com/seo-resources/',
                'url_to' => 'https://' . $domain . '/blog/seo-guide/',
                'anchor' => 'comprehensive SEO guide',
                'dofollow' => true,
                'rank' => 892,
                'first_seen' => '2024-06-20T14:15:00Z',
                'is_lost' => false,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 85,
                'page_from_rank' => 68,
                'backlink_spam_score' => 3,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'We recommend this',
                'text_post' => 'to beginners.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-27T12:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'hubspot.com',
                'url_from' => 'https://blog.hubspot.com/marketing/link-building-strategies',
                'url_to' => 'https://' . $domain . '/',
                'anchor' => $domain,
                'dofollow' => true,
                'rank' => 867,
                'first_seen' => '2024-09-10T09:00:00Z',
                'is_lost' => false,
                'is_new' => true,
                'attributes' => [],
                'domain_from_rank' => 93,
                'page_from_rank' => 75,
                'backlink_spam_score' => 1,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'Tools like',
                'text_post' => 'can help.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-28T06:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'moz.com',
                'url_from' => 'https://moz.com/blog/seo-tools-roundup',
                'url_to' => 'https://' . $domain . '/features/',
                'anchor' => 'screaming fixes features',
                'dofollow' => true,
                'rank' => 834,
                'first_seen' => '2024-04-05T16:45:00Z',
                'is_lost' => true,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 91,
                'page_from_rank' => 65,
                'backlink_spam_score' => 2,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'Explore',
                'text_post' => 'for more info.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2024-12-15T10:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'reddit.com',
                'url_from' => 'https://www.reddit.com/r/SEO/comments/abc123/',
                'url_to' => 'https://' . $domain . '/pricing/',
                'anchor' => 'click here',
                'dofollow' => false,
                'rank' => 756,
                'first_seen' => '2024-11-20T11:30:00Z',
                'is_lost' => false,
                'is_new' => true,
                'attributes' => ['nofollow', 'ugc'],
                'domain_from_rank' => 91,
                'page_from_rank' => 42,
                'backlink_spam_score' => 8,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'I found this at',
                'text_post' => '- worth checking out.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-26T18:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'neilpatel.com',
                'url_from' => 'https://neilpatel.com/blog/seo-checklist/',
                'url_to' => 'https://' . $domain . '/blog/technical-seo/',
                'anchor' => 'technical SEO tips',
                'dofollow' => true,
                'rank' => 723,
                'first_seen' => '2024-07-12T08:00:00Z',
                'is_lost' => false,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 92,
                'page_from_rank' => 70,
                'backlink_spam_score' => 1,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'Read more about',
                'text_post' => 'on their blog.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-25T14:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'ahrefs.com',
                'url_from' => 'https://ahrefs.com/blog/free-seo-tools/',
                'url_to' => 'https://' . $domain . '/',
                'anchor' => 'Screaming Fixes',
                'dofollow' => true,
                'rank' => 698,
                'first_seen' => '2024-05-28T13:20:00Z',
                'is_lost' => false,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 89,
                'page_from_rank' => 73,
                'backlink_spam_score' => 1,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => '',
                'text_post' => 'is a great free tool.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-28T09:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'backlinko.com',
                'url_from' => 'https://backlinko.com/seo-tools',
                'url_to' => 'https://' . $domain . '/tools/',
                'anchor' => 'free SEO toolkit',
                'dofollow' => true,
                'rank' => 654,
                'first_seen' => '2024-03-15T10:00:00Z',
                'is_lost' => true,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 78,
                'page_from_rank' => 62,
                'backlink_spam_score' => 2,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'Try this',
                'text_post' => 'for on-page fixes.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2024-11-10T16:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'semrush.com',
                'url_from' => 'https://www.semrush.com/blog/wordpress-seo-plugins/',
                'url_to' => 'https://' . $domain . '/',
                'anchor' => '',
                'dofollow' => true,
                'rank' => 621,
                'first_seen' => '2024-10-05T07:45:00Z',
                'is_lost' => false,
                'is_new' => false,
                'attributes' => [],
                'domain_from_rank' => 90,
                'page_from_rank' => 67,
                'backlink_spam_score' => 1,
                'item_type' => 'image',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => '',
                'text_post' => '',
                'image_alt' => 'Screaming Fixes Logo',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-27T22:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
            [
                'domain_from' => 'wpbeginner.com',
                'url_from' => 'https://www.wpbeginner.com/plugins/best-seo-plugins/',
                'url_to' => 'https://' . $domain . '/',
                'anchor' => 'Screaming Fixes plugin',
                'dofollow' => true,
                'rank' => 589,
                'first_seen' => '2024-12-01T15:00:00Z',
                'is_lost' => false,
                'is_new' => true,
                'attributes' => [],
                'domain_from_rank' => 84,
                'page_from_rank' => 71,
                'backlink_spam_score' => 2,
                'item_type' => 'anchor',
                'domain_from_tld' => 'com',
                'url_from_https' => true,
                'text_pre' => 'We recommend',
                'text_post' => 'for beginners.',
                'image_alt' => '',
                'semantic_location' => 'main',
                'links_count' => 1,
                'last_seen' => '2025-01-28T04:00:00Z',
                'url_to_status_code' => 200,
                'url_from_status_code' => 200,
            ],
        ];

        return [
            'summary' => $summary,
            'backlinks' => $backlinks,
            'domain' => $domain,
            'fetched_at' => current_time('mysql'),
        ];
    }

    /**
     * Calculate dofollow percentage from summary data
     *
     * @param array $summary Summary data array
     * @return float Dofollow percentage
     */
    public static function calculate_dofollow_percentage($summary) {
        $total = $summary['referring_domains'] ?? 0;
        $nofollow = $summary['referring_domains_nofollow'] ?? 0;

        if ($total === 0) {
            return 0;
        }

        $dofollow = $total - $nofollow;
        return round(($dofollow / $total) * 100, 1);
    }

    /**
     * Get link attribute display info
     *
     * @param bool $dofollow Whether link is dofollow
     * @param array $attributes Link attributes array
     * @return array Display info with 'label', 'class', and 'title'
     */
    public static function get_link_attribute_display($dofollow, $attributes = []) {
        if ($dofollow) {
            return [
                'label' => 'DF',
                'class' => 'sf-attr-dofollow',
                'title' => __('Dofollow - passes link equity', 'screaming-fixes'),
            ];
        }

        // Check for sponsored/ugc attributes
        if (in_array('sponsored', $attributes)) {
            return [
                'label' => 'SP',
                'class' => 'sf-attr-sponsored',
                'title' => __('Sponsored - paid/advertisement link', 'screaming-fixes'),
            ];
        }

        if (in_array('ugc', $attributes)) {
            return [
                'label' => 'UGC',
                'class' => 'sf-attr-ugc',
                'title' => __('User Generated Content', 'screaming-fixes'),
            ];
        }

        return [
            'label' => 'NF',
            'class' => 'sf-attr-nofollow',
            'title' => __('Nofollow - does not pass link equity', 'screaming-fixes'),
        ];
    }

    /**
     * Get status display info for a backlink
     *
     * @param bool $is_lost Whether link is lost
     * @param bool $is_new Whether link is new
     * @return array Display info with 'label', 'class', and 'title'
     */
    public static function get_status_display($is_lost, $is_new) {
        if ($is_lost) {
            return [
                'label' => __('Lost', 'screaming-fixes'),
                'class' => 'sf-status-lost',
                'title' => __('This link has been removed - reclamation opportunity', 'screaming-fixes'),
            ];
        }

        if ($is_new) {
            return [
                'label' => __('New', 'screaming-fixes'),
                'class' => 'sf-status-new',
                'title' => __('Recently discovered link', 'screaming-fixes'),
            ];
        }

        return [
            'label' => __('Live', 'screaming-fixes'),
            'class' => 'sf-status-live',
            'title' => __('Active, healthy link', 'screaming-fixes'),
        ];
    }

    /**
     * Get CSS class for rank value (color gradient)
     *
     * @param int $rank Rank value (0-1000)
     * @return string CSS class
     */
    public static function get_rank_class($rank) {
        if ($rank >= 700) {
            return 'sf-rank-high';
        }
        if ($rank >= 400) {
            return 'sf-rank-medium';
        }
        if ($rank >= 100) {
            return 'sf-rank-low';
        }
        return 'sf-rank-very-low';
    }
}
