<?php
/**
 * Change Logger for Screaming Fixes
 *
 * Logs all changes for audit trail and CSV export.
 * Rolling log capped at 5,000 entries, trimmed via cron.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Change_Logger {

    /**
     * Log table name
     * @var string
     */
    private $table_name;

    /**
     * Maximum number of log entries to retain
     */
    const MAX_ENTRIES = 5000;

    /**
     * Module labels for CSV export
     */
    private static $module_labels = [
        'broken-links' => 'Broken Links',
        'redirect-chains' => 'Redirect Chains',
        'backlink-reclaim' => 'Backlink Reclaim',
        'image-alt-text' => 'Image Alt Text',
        'page-title' => 'Page Title',
        'meta-description' => 'Meta Description',
        'internal-link-builder' => 'Internal Link Builder',
    ];

    /**
     * Fix type labels for CSV export
     */
    private static $fix_type_labels = [
        'broken_link' => 'Link Replaced',
        'redirect_chain' => 'Redirect Chain Fixed',
        'redirect_temp' => 'Temp Redirect Converted to 301',
        'redirect_loop' => 'Redirect Loop Removed',
        'redirect' => 'Redirect Created',
        'alt_text' => 'Alt Text Updated',
        'page_title' => 'Page Title Updated',
        'meta_description' => 'Meta Description Updated',
        'internal_link' => 'Internal Link Added',
    ];

    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'screaming_fixes_log';
    }

    /**
     * Initialize AJAX handlers and cron
     */
    public static function init() {
        add_action('wp_ajax_sf_export_change_log', [__CLASS__, 'ajax_export_change_log']);
        add_action('wp_ajax_sf_clear_change_log', [__CLASS__, 'ajax_clear_change_log']);
        add_action('sf_trim_change_log', [__CLASS__, 'cron_trim']);
    }

    /**
     * Schedule the daily trim cron event
     */
    public static function schedule_cron() {
        if (!wp_next_scheduled('sf_trim_change_log')) {
            wp_schedule_event(time(), 'daily', 'sf_trim_change_log');
        }
    }

    /**
     * Unschedule the cron event
     */
    public static function unschedule_cron() {
        $timestamp = wp_next_scheduled('sf_trim_change_log');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'sf_trim_change_log');
        }
    }

    /**
     * Cron callback to trim log entries
     */
    public static function cron_trim() {
        $logger = new self();
        $logger->trim_to_limit(self::MAX_ENTRIES);
    }

    /**
     * Log a change
     *
     * @param int $post_id Post ID (0 for non-post changes like redirects)
     * @param string $fix_type Type of fix (broken_link, redirect_chain, alt_text, redirect, etc.)
     * @param mixed $original Original value
     * @param mixed $new New value
     * @param array $context Additional context
     * @return int|false Log entry ID or false on failure
     */
    public function log_change($post_id, $fix_type, $original, $new, $context = []) {
        // Check if logging is enabled
        if (!get_option('sf_enable_logging', true)) {
            return false;
        }

        global $wpdb;

        $data = [
            'post_id' => absint($post_id),
            'fix_type' => sanitize_text_field($fix_type),
            'module' => sanitize_text_field($context['module'] ?? 'unknown'),
            'original_value' => is_string($original) ? $original : wp_json_encode($original),
            'new_value' => is_string($new) ? $new : wp_json_encode($new),
            'context' => wp_json_encode($context),
            'user_id' => get_current_user_id(),
            'created_at' => current_time('mysql'),
        ];

        $result = $wpdb->insert(
            $this->table_name,
            $data,
            ['%d', '%s', '%s', '%s', '%s', '%s', '%d', '%s']
        );

        if ($result === false) {
            return false;
        }

        return $wpdb->insert_id;
    }

    /**
     * Log multiple changes in a single batch INSERT
     *
     * @param array $entries Array of entries, each with keys: post_id, fix_type, original, new, context
     * @return int Number of entries inserted
     */
    public function log_changes_batch($entries) {
        if (empty($entries)) {
            return 0;
        }

        // Check if logging is enabled
        if (!get_option('sf_enable_logging', true)) {
            return 0;
        }

        global $wpdb;

        $user_id = get_current_user_id();
        $now = current_time('mysql');

        $values = [];
        $placeholders = [];

        foreach ($entries as $entry) {
            $original = $entry['original'] ?? '';
            $new = $entry['new'] ?? '';

            $values[] = absint($entry['post_id'] ?? 0);
            $values[] = sanitize_text_field($entry['fix_type'] ?? 'unknown');
            $values[] = sanitize_text_field($entry['context']['module'] ?? 'unknown');
            $values[] = is_string($original) ? $original : wp_json_encode($original);
            $values[] = is_string($new) ? $new : wp_json_encode($new);
            $values[] = wp_json_encode($entry['context'] ?? []);
            $values[] = $user_id;
            $values[] = $now;

            $placeholders[] = '(%d, %s, %s, %s, %s, %s, %d, %s)';
        }

        $placeholders_str = implode(', ', $placeholders);

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $result = $wpdb->query($wpdb->prepare(
            "INSERT INTO {$this->table_name} (post_id, fix_type, module, original_value, new_value, context, user_id, created_at) VALUES {$placeholders_str}",
            $values
        ));

        if ($result === false) {
            // Fallback to individual inserts
            $inserted = 0;
            foreach ($entries as $entry) {
                $id = $this->log_change(
                    $entry['post_id'] ?? 0,
                    $entry['fix_type'] ?? 'unknown',
                    $entry['original'] ?? '',
                    $entry['new'] ?? '',
                    $entry['context'] ?? []
                );
                if ($id) {
                    $inserted++;
                }
            }
            return $inserted;
        }

        return count($entries);
    }

    /**
     * Trim log to a maximum number of entries
     * Uses efficient two-step approach: find cutoff ID, then delete below it
     *
     * @param int $max Maximum entries to keep
     * @return int Number of deleted entries
     */
    public function trim_to_limit($max = 5000) {
        global $wpdb;

        $count = $this->get_count();
        if ($count <= $max) {
            return 0;
        }

        // Get the ID at the cutoff position
        $cutoff_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->table_name} ORDER BY id DESC LIMIT 1 OFFSET %d",
            $max - 1
        ));

        if (!$cutoff_id) {
            return 0;
        }

        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE id < %d",
            $cutoff_id
        ));

        return $deleted ?: 0;
    }

    /**
     * Get total number of log entries
     *
     * @return int
     */
    public function get_count() {
        global $wpdb;

        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name}"
        );
    }

    /**
     * Undo a specific change
     *
     * @param int $log_id Log entry ID
     * @return bool|WP_Error Success or error
     */
    public function undo_change($log_id) {
        global $wpdb;

        $log = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE id = %d",
            $log_id
        ));

        if (!$log) {
            return new WP_Error('not_found', __('Log entry not found.', 'screaming-fixes'));
        }

        if ($log->reverted_at) {
            return new WP_Error('already_reverted', __('This change has already been reverted.', 'screaming-fixes'));
        }

        // Perform the undo based on fix type
        $result = $this->perform_undo($log);

        if (is_wp_error($result)) {
            return $result;
        }

        // Mark as reverted
        $wpdb->update(
            $this->table_name,
            [
                'reverted_at' => current_time('mysql'),
            ],
            ['id' => $log_id],
            ['%s'],
            ['%d']
        );

        return true;
    }

    /**
     * Perform the actual undo operation
     *
     * @param object $log Log entry
     * @return bool|WP_Error Success or error
     */
    private function perform_undo($log) {
        switch ($log->fix_type) {
            case 'broken_link':
            case 'redirect_chain':
            case 'alt_text':
                // Restore original post content
                return $this->undo_content_change($log);

            case 'redirect':
                // Delete the created redirect
                return $this->undo_redirect($log);

            default:
                return new WP_Error('unknown_type', __('Unknown fix type.', 'screaming-fixes'));
        }
    }

    /**
     * Undo a content change
     *
     * @param object $log Log entry
     * @return bool|WP_Error Success or error
     */
    private function undo_content_change($log) {
        if (empty($log->post_id)) {
            return new WP_Error('no_post_id', __('No post ID associated with this change.', 'screaming-fixes'));
        }

        $post = get_post($log->post_id);

        if (!$post) {
            return new WP_Error('post_not_found', __('Original post not found.', 'screaming-fixes'));
        }

        // Restore original content
        $result = wp_update_post([
            'ID' => $log->post_id,
            'post_content' => $log->original_value,
        ], true);

        if (is_wp_error($result)) {
            return $result;
        }

        return true;
    }

    /**
     * Undo a redirect creation
     *
     * @param object $log Log entry
     * @return bool|WP_Error Success or error
     */
    private function undo_redirect($log) {
        $context = json_decode($log->context, true);

        if (empty($context['plugin'])) {
            return new WP_Error('no_plugin_info', __('Redirect plugin info not found.', 'screaming-fixes'));
        }

        $redirect_manager = new SF_Redirect_Manager();

        return $redirect_manager->delete_redirect(
            $context['source'] ?? '',
            $context['plugin'],
            $context['redirect_id'] ?? null
        );
    }

    /**
     * Get log entries
     *
     * @param array $args Query arguments
     * @return array Log entries
     */
    public function get_logs($args = []) {
        global $wpdb;

        $defaults = [
            'fix_type' => '',
            'module' => '',
            'post_id' => 0,
            'include_reverted' => false,
            'limit' => 50,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC',
        ];

        $args = wp_parse_args($args, $defaults);

        $where = ['1=1'];
        $values = [];

        if ($args['fix_type']) {
            $where[] = 'fix_type = %s';
            $values[] = $args['fix_type'];
        }

        if ($args['module']) {
            $where[] = 'module = %s';
            $values[] = $args['module'];
        }

        if ($args['post_id']) {
            $where[] = 'post_id = %d';
            $values[] = $args['post_id'];
        }

        if (!$args['include_reverted']) {
            $where[] = 'reverted_at IS NULL';
        }

        $where_clause = implode(' AND ', $where);

        // Sanitize orderby
        $allowed_orderby = ['id', 'created_at', 'fix_type', 'module', 'post_id'];
        $orderby = in_array($args['orderby'], $allowed_orderby) ? $args['orderby'] : 'created_at';
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';

        $query = "SELECT * FROM {$this->table_name} WHERE {$where_clause} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
        $values[] = $args['limit'];
        $values[] = $args['offset'];

        $results = $wpdb->get_results($wpdb->prepare($query, $values));

        // Enhance with post info
        foreach ($results as &$log) {
            if ($log->post_id) {
                $post = get_post($log->post_id);
                $log->post_title = $post ? $post->post_title : __('(Deleted post)', 'screaming-fixes');
                $log->post_url = $post ? get_permalink($log->post_id) : '';
                $log->edit_url = $post ? get_edit_post_link($log->post_id, 'raw') : '';
            } else {
                $log->post_title = '';
                $log->post_url = '';
                $log->edit_url = '';
            }
            $log->context = json_decode($log->context, true);
        }

        return $results;
    }

    /**
     * Get recent activity for dashboard
     *
     * @param int $limit Number of entries
     * @return array Recent activity
     */
    public function get_recent_activity($limit = 10) {
        $logs = $this->get_logs([
            'limit' => $limit,
            'include_reverted' => false,
        ]);

        $activity = [];

        foreach ($logs as $log) {
            $description = $this->get_activity_description($log);

            $activity[] = [
                'id' => $log->id,
                'type' => $log->fix_type,
                'module' => $log->module,
                'description' => $description,
                'post_id' => $log->post_id,
                'post_title' => $log->post_title ?? '',
                'created_at' => $log->created_at,
                'time_ago' => human_time_diff(strtotime($log->created_at), current_time('timestamp')) . ' ' . __('ago', 'screaming-fixes'),
                'can_undo' => empty($log->reverted_at),
            ];
        }

        return $activity;
    }

    /**
     * Get human-readable activity description
     *
     * @param object $log Log entry
     * @return string Description
     */
    private function get_activity_description($log) {
        $context = $log->context;

        switch ($log->fix_type) {
            case 'broken_link':
                $action = $context['action'] ?? 'replace';
                if ($action === 'replace') {
                    return sprintf(
                        __('Replaced broken link in "%s"', 'screaming-fixes'),
                        $log->post_title ?? __('Unknown post', 'screaming-fixes')
                    );
                } elseif ($action === 'remove_link') {
                    return sprintf(
                        __('Removed broken link from "%s"', 'screaming-fixes'),
                        $log->post_title ?? __('Unknown post', 'screaming-fixes')
                    );
                } else {
                    return sprintf(
                        __('Fixed broken link in "%s"', 'screaming-fixes'),
                        $log->post_title ?? __('Unknown post', 'screaming-fixes')
                    );
                }

            case 'redirect_chain':
                return sprintf(
                    __('Fixed redirect chain in "%s"', 'screaming-fixes'),
                    $log->post_title ?? __('Unknown post', 'screaming-fixes')
                );

            case 'redirect_temp':
                return sprintf(
                    __('Converted temp redirect to 301: %s', 'screaming-fixes'),
                    $context['source'] ?? __('unknown URL', 'screaming-fixes')
                );

            case 'redirect_loop':
                return sprintf(
                    __('Removed redirect loop: %s', 'screaming-fixes'),
                    $context['source'] ?? __('unknown URL', 'screaming-fixes')
                );

            case 'alt_text':
                return sprintf(
                    __('Updated alt text in "%s"', 'screaming-fixes'),
                    $log->post_title ?? __('Unknown post', 'screaming-fixes')
                );

            case 'page_title':
                return sprintf(
                    __('Updated page title for "%s"', 'screaming-fixes'),
                    $log->post_title ?? __('Unknown post', 'screaming-fixes')
                );

            case 'meta_description':
                return sprintf(
                    __('Updated meta description for "%s"', 'screaming-fixes'),
                    $log->post_title ?? __('Unknown post', 'screaming-fixes')
                );

            case 'redirect':
                return sprintf(
                    __('Created redirect from %s', 'screaming-fixes'),
                    $context['source'] ?? __('unknown URL', 'screaming-fixes')
                );

            default:
                return __('Made a change', 'screaming-fixes');
        }
    }

    /**
     * Get statistics for dashboard
     *
     * @return array Statistics
     */
    public function get_stats() {
        global $wpdb;

        $stats = [
            'total_fixes' => 0,
            'fixes_by_type' => [],
            'fixes_today' => 0,
            'fixes_this_week' => 0,
        ];

        // Total fixes (not reverted)
        $stats['total_fixes'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE reverted_at IS NULL"
        );

        // Fixes by type
        $by_type = $wpdb->get_results(
            "SELECT fix_type, COUNT(*) as count
             FROM {$this->table_name}
             WHERE reverted_at IS NULL
             GROUP BY fix_type"
        );

        foreach ($by_type as $row) {
            $stats['fixes_by_type'][$row->fix_type] = (int) $row->count;
        }

        // Fixes today
        $stats['fixes_today'] = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name}
                 WHERE reverted_at IS NULL
                 AND DATE(created_at) = %s",
                current_time('Y-m-d')
            )
        );

        // Fixes this week
        $stats['fixes_this_week'] = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name}
                 WHERE reverted_at IS NULL
                 AND created_at >= %s",
                gmdate('Y-m-d H:i:s', strtotime('-7 days'))
            )
        );

        return $stats;
    }

    /**
     * Delete old log entries (time-based)
     *
     * @param int $days Delete entries older than this many days
     * @return int Number of deleted entries
     */
    public function cleanup_old_logs($days = 90) {
        global $wpdb;

        $cutoff = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE created_at < %s",
            $cutoff
        ));

        return $deleted;
    }

    /**
     * Clear all log entries
     *
     * @return int Number of deleted entries
     */
    public function clear_all() {
        global $wpdb;

        return $wpdb->query("TRUNCATE TABLE {$this->table_name}");
    }

    /**
     * Export logs to CSV with full detail columns
     *
     * @param array $args Filter arguments
     * @return string CSV content
     */
    public function export_logs($args = []) {
        $logs = $this->get_logs(array_merge($args, [
            'limit' => self::MAX_ENTRIES,
            'include_reverted' => true,
        ]));

        $csv_parser = new SF_CSV_Parser();

        $data = array_map(function ($log) {
            $context = is_array($log->context) ? $log->context : [];
            $user = $log->user_id ? get_userdata($log->user_id) : null;

            return [
                'Date' => $log->created_at,
                'Module' => self::$module_labels[$log->module] ?? ucwords(str_replace('-', ' ', $log->module)),
                'Action' => self::$fix_type_labels[$log->fix_type] ?? ucwords(str_replace('_', ' ', $log->fix_type)),
                'Status' => $log->reverted_at ? 'Reverted' : ($context['status'] ?? 'Success'),
                'Original Value' => $log->original_value,
                'New Value' => $log->new_value,
                'Post ID' => $log->post_id ?: '',
                'Post Title' => $log->post_title ?? '',
                'Post URL' => $log->post_url ?? '',
                'Source URL' => $context['source_url'] ?? $context['source'] ?? '',
                'Location' => ucfirst($context['location'] ?? ''),
                'User' => $user ? $user->display_name : __('Unknown', 'screaming-fixes'),
            ];
        }, $logs);

        return $csv_parser->export_to_csv($data);
    }

    /**
     * AJAX handler to export change log as CSV
     */
    public static function ajax_export_change_log() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $logger = new self();
        $csv_content = $logger->export_logs();

        wp_send_json_success([
            'csv' => $csv_content,
            'filename' => 'screaming-fixes-change-log-' . date('Y-m-d-His') . '.csv',
        ]);
    }

    /**
     * AJAX handler to clear change log
     */
    public static function ajax_clear_change_log() {
        check_ajax_referer('screaming_fixes_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'screaming-fixes')]);
        }

        $logger = new self();
        $logger->clear_all();

        wp_send_json_success([
            'message' => __('Change log cleared.', 'screaming-fixes'),
        ]);
    }

    /**
     * Check if a change can be undone
     *
     * @param int $log_id Log entry ID
     * @return bool|WP_Error True if can undo, error otherwise
     */
    public function can_undo($log_id) {
        global $wpdb;

        $log = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE id = %d",
            $log_id
        ));

        if (!$log) {
            return new WP_Error('not_found', __('Log entry not found.', 'screaming-fixes'));
        }

        if ($log->reverted_at) {
            return new WP_Error('already_reverted', __('This change has already been reverted.', 'screaming-fixes'));
        }

        // Check if the post still exists for content changes
        if ($log->post_id && in_array($log->fix_type, ['broken_link', 'redirect_chain', 'alt_text'])) {
            $post = get_post($log->post_id);
            if (!$post) {
                return new WP_Error('post_deleted', __('The original post has been deleted.', 'screaming-fixes'));
            }
        }

        return true;
    }
}
