<?php
/**
 * Link Fixer for Screaming Fixes
 *
 * Handles content replacement logic for fixing links and images
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Link_Fixer {

    /**
     * Change logger instance
     * @var SF_Change_Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = new SF_Change_Logger();
    }

    /**
     * Replace a URL in a post's content
     *
     * @param int $post_id Post ID
     * @param string $old_url URL to replace
     * @param string $new_url Replacement URL (empty to remove)
     * @param string $action Action type: 'replace', 'remove_link', 'remove_all'
     * @param string $module Module making the change
     * @return bool|WP_Error Success or error
     */
    public function replace_url_in_post($post_id, $old_url, $new_url, $action = 'replace', $module = 'unknown') {
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error('post_not_found', __('Post not found.', 'screaming-fixes'));
        }

        $content = $post->post_content;
        $original_content = $content;

        switch ($action) {
            case 'replace':
                $content = $this->do_url_replace($content, $old_url, $new_url);
                break;

            case 'remove_link':
                $content = $this->remove_link_keep_text($content, $old_url);
                break;

            case 'remove_all':
                $content = $this->remove_link_and_text($content, $old_url);
                break;

            default:
                return new WP_Error('invalid_action', __('Invalid action type.', 'screaming-fixes'));
        }

        // Check if content actually changed
        if ($content === $original_content) {
            return new WP_Error('no_change', __('URL not found in content.', 'screaming-fixes'));
        }

        // Log the change
        $this->logger->log_change($post_id, 'broken_link', $original_content, $content, [
            'old_url' => $old_url,
            'new_url' => $new_url,
            'action' => $action,
            'module' => $module,
        ]);

        // Update the post
        $result = wp_update_post([
            'ID' => $post_id,
            'post_content' => $content,
        ], true);

        if (is_wp_error($result)) {
            return $result;
        }

        return true;
    }

    /**
     * Simple URL replacement
     *
     * @param string $content Post content
     * @param string $old_url Old URL
     * @param string $new_url New URL
     * @return string Modified content
     */
    private function do_url_replace($content, $old_url, $new_url) {
        // Normalize URLs - remove trailing slash for matching, but preserve path structure
        $old_url_normalized = rtrim($old_url, '/');
        $new_url_normalized = rtrim($new_url, '/');

        // Build regex pattern that matches with or without trailing slash
        // This handles cases where Screaming Frog has trailing slash but content doesn't (or vice versa)
        $old_url_pattern = preg_quote($old_url_normalized, '/') . '\/?';

        // Replace in href attributes
        $content = preg_replace(
            '/(href=["\'])' . $old_url_pattern . '(["\'])/i',
            '$1' . $new_url . '$2',
            $content
        );

        // Replace in src attributes
        $content = preg_replace(
            '/(src=["\'])' . $old_url_pattern . '(["\'])/i',
            '$1' . $new_url . '$2',
            $content
        );

        // Replace plain URLs not in attributes (be careful with this)
        // Only replace if URL is standalone (not part of another URL)
        $content = preg_replace(
            '/(?<!["\'\w])' . $old_url_pattern . '(?!["\'\w])/i',
            $new_url,
            $content
        );

        return $content;
    }

    /**
     * Remove link but keep anchor text
     *
     * @param string $content Post content
     * @param string $url URL to remove
     * @return string Modified content
     */
    private function remove_link_keep_text($content, $url) {
        // Normalize URL - match with or without trailing slash
        $url_normalized = rtrim($url, '/');
        $url_pattern = preg_quote($url_normalized, '/') . '\/?';

        // Match <a> tags with this URL and replace with just the anchor text
        $pattern = '/<a\s+[^>]*href=["\']' . $url_pattern . '["\'][^>]*>(.*?)<\/a>/is';
        return preg_replace($pattern, '$1', $content);
    }

    /**
     * Remove link and anchor text entirely
     *
     * @param string $content Post content
     * @param string $url URL to remove
     * @return string Modified content
     */
    private function remove_link_and_text($content, $url) {
        // Normalize URL - match with or without trailing slash
        $url_normalized = rtrim($url, '/');
        $url_pattern = preg_quote($url_normalized, '/') . '\/?';

        $pattern = '/<a\s+[^>]*href=["\']' . $url_pattern . '["\'][^>]*>.*?<\/a>/is';
        return preg_replace($pattern, '', $content);
    }

    /**
     * Update image alt text in a post
     *
     * @param int $post_id Post ID
     * @param string $image_src Image source URL
     * @param string $new_alt New alt text
     * @param string $module Module making the change
     * @return bool|WP_Error Success or error
     */
    public function update_image_alt($post_id, $image_src, $new_alt, $module = 'image-alt-text') {
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error('post_not_found', __('Post not found.', 'screaming-fixes'));
        }

        $content = $post->post_content;
        $original_content = $content;

        // Find the image tag
        $filename = basename(wp_parse_url($image_src, PHP_URL_PATH));

        // Pattern to match img tags with this src
        $pattern = '/<img\s+([^>]*?)(?:src=["\']([^"\']*' . preg_quote($filename, '/') . ')["\'])([^>]*)>/is';

        $content = preg_replace_callback($pattern, function ($matches) use ($new_alt) {
            $before_src = $matches[1];
            $src = $matches[2];
            $after_src = $matches[3];

            // Remove existing alt attribute
            $before_src = preg_replace('/\s*alt=["\'][^"\']*["\']\s*/i', ' ', $before_src);
            $after_src = preg_replace('/\s*alt=["\'][^"\']*["\']\s*/i', ' ', $after_src);

            // Clean up extra spaces
            $before_src = preg_replace('/\s+/', ' ', trim($before_src));
            $after_src = preg_replace('/\s+/', ' ', trim($after_src));

            // Build new tag with alt
            $new_tag = '<img';
            if ($before_src) {
                $new_tag .= ' ' . $before_src;
            }
            $new_tag .= ' src="' . esc_attr($src) . '"';
            $new_tag .= ' alt="' . esc_attr($new_alt) . '"';
            if ($after_src) {
                $new_tag .= ' ' . $after_src;
            }
            $new_tag .= '>';

            return $new_tag;
        }, $content);

        // Check if content actually changed
        if ($content === $original_content) {
            return new WP_Error('no_change', __('Image not found in content.', 'screaming-fixes'));
        }

        // Log the change
        $this->logger->log_change($post_id, 'alt_text', $original_content, $content, [
            'image_src' => $image_src,
            'new_alt' => $new_alt,
            'module' => $module,
        ]);

        // Update the post
        $result = wp_update_post([
            'ID' => $post_id,
            'post_content' => $content,
        ], true);

        if (is_wp_error($result)) {
            return $result;
        }

        // Also update attachment meta if this is a media library image
        $scanner = new SF_Link_Scanner();
        $attachment_id = $scanner->get_attachment_id_from_url($image_src);

        if ($attachment_id) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $new_alt);
        }

        return true;
    }

    /**
     * Bulk replace URLs in multiple posts
     *
     * @param string $old_url URL to replace
     * @param string $new_url New URL
     * @param string $action Action type
     * @param string $module Module making the change
     * @return array Results with success/failure counts
     */
    public function bulk_replace_url($old_url, $new_url, $action = 'replace', $module = 'unknown') {
        $scanner = new SF_Link_Scanner();
        $posts = $scanner->find_posts_with_url($old_url);

        $results = [
            'total' => count($posts),
            'success' => 0,
            'failed' => 0,
            'errors' => [],
        ];

        foreach ($posts as $post_info) {
            $result = $this->replace_url_in_post(
                $post_info['post_id'],
                $old_url,
                $new_url,
                $action,
                $module
            );

            if (is_wp_error($result)) {
                $results['failed']++;
                $results['errors'][] = [
                    'post_id' => $post_info['post_id'],
                    'post_title' => $post_info['post_title'],
                    'error' => $result->get_error_message(),
                ];
            } else {
                $results['success']++;
            }
        }

        return $results;
    }

    /**
     * Apply multiple fixes at once
     *
     * @param array $fixes Array of fix definitions
     * @param string $module Module making the changes
     * @return array Results summary
     */
    public function apply_fixes($fixes, $module = 'unknown') {
        $results = [
            'total' => count($fixes),
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => [],
        ];

        foreach ($fixes as $fix) {
            // Skip if no fix defined
            if (empty($fix['new_url']) && empty($fix['action'])) {
                $results['skipped']++;
                continue;
            }

            $action = $fix['action'] ?? 'replace';
            $new_url = $fix['new_url'] ?? '';

            // If action is remove, we don't need a new URL
            if (in_array($action, ['remove_link', 'remove_all'])) {
                $new_url = '';
            }

            // Apply to specific post or all posts
            if (!empty($fix['post_id'])) {
                // Single post fix
                $result = $this->replace_url_in_post(
                    $fix['post_id'],
                    $fix['old_url'],
                    $new_url,
                    $action,
                    $module
                );

                if (is_wp_error($result)) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'post_id' => $fix['post_id'],
                        'old_url' => $fix['old_url'],
                        'error' => $result->get_error_message(),
                    ];
                } else {
                    $results['success']++;
                }
            } else {
                // Bulk fix across all posts
                $bulk_result = $this->bulk_replace_url(
                    $fix['old_url'],
                    $new_url,
                    $action,
                    $module
                );

                $results['success'] += $bulk_result['success'];
                $results['failed'] += $bulk_result['failed'];
                $results['errors'] = array_merge($results['errors'], $bulk_result['errors']);
            }
        }

        return $results;
    }

    /**
     * Fix redirect chain by updating to final URL
     *
     * @param int $post_id Post ID
     * @param string $chain_url Current redirect URL in content
     * @param string $final_url Final destination URL
     * @param string $module Module making the change
     * @return bool|WP_Error Success or error
     */
    public function fix_redirect_chain($post_id, $chain_url, $final_url, $module = 'redirect-chains') {
        $post = get_post($post_id);

        if (!$post) {
            return new WP_Error('post_not_found', __('Post not found.', 'screaming-fixes'));
        }

        $content = $post->post_content;
        $original_content = $content;

        // Replace the redirect URL with final URL
        $content = $this->do_url_replace($content, $chain_url, $final_url);

        if ($content === $original_content) {
            return new WP_Error('no_change', __('Redirect URL not found in content.', 'screaming-fixes'));
        }

        // Log the change
        $this->logger->log_change($post_id, 'redirect_chain', $original_content, $content, [
            'chain_url' => $chain_url,
            'final_url' => $final_url,
            'module' => $module,
        ]);

        // Update the post
        $result = wp_update_post([
            'ID' => $post_id,
            'post_content' => $content,
        ], true);

        if (is_wp_error($result)) {
            return $result;
        }

        return true;
    }

    /**
     * Preview what a fix would look like
     *
     * @param int $post_id Post ID
     * @param string $old_url URL to replace
     * @param string $new_url New URL
     * @param string $action Action type
     * @return array Preview data
     */
    public function preview_fix($post_id, $old_url, $new_url, $action = 'replace') {
        $post = get_post($post_id);

        if (!$post) {
            return [
                'error' => __('Post not found.', 'screaming-fixes'),
            ];
        }

        $content = $post->post_content;
        $original_content = $content;

        switch ($action) {
            case 'replace':
                $content = $this->do_url_replace($content, $old_url, $new_url);
                break;
            case 'remove_link':
                $content = $this->remove_link_keep_text($content, $old_url);
                break;
            case 'remove_all':
                $content = $this->remove_link_and_text($content, $old_url);
                break;
        }

        // Find the changed portions
        $changes = $this->get_content_diff($original_content, $content);

        return [
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'has_changes' => $content !== $original_content,
            'changes' => $changes,
        ];
    }

    /**
     * Get a simple diff between two content strings
     *
     * @param string $original Original content
     * @param string $modified Modified content
     * @return array Array of changes
     */
    private function get_content_diff($original, $modified) {
        $changes = [];

        // Find removed content
        $removed = array_diff(
            explode("\n", $original),
            explode("\n", $modified)
        );

        // Find added content
        $added = array_diff(
            explode("\n", $modified),
            explode("\n", $original)
        );

        foreach ($removed as $line) {
            if (trim($line)) {
                $changes[] = [
                    'type' => 'removed',
                    'content' => wp_strip_all_tags(substr($line, 0, 200)),
                ];
            }
        }

        foreach ($added as $line) {
            if (trim($line)) {
                $changes[] = [
                    'type' => 'added',
                    'content' => wp_strip_all_tags(substr($line, 0, 200)),
                ];
            }
        }

        return $changes;
    }
}
