<?php
/**
 * Plugin detector for competing SEO plugins
 * Detects Rank Math, Yoast SEO, All in One SEO, and Redirection plugins
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Plugin_Detector {
    /**
     * Ensure is_plugin_active() function is available
     */
    private static function ensure_plugin_functions() {
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
    }

    /**
     * Check if Rank Math is installed and active
     */
    public static function is_rank_math_active() {
        // Use class_exists for reliable detection (works regardless of plugin.php loading)
        if (class_exists('RankMath') || class_exists('\\RankMath\\Admin\\Admin_Menu')) {
            return true;
        }

        // Fallback to is_plugin_active check
        self::ensure_plugin_functions();
        return is_plugin_active('rank-math/rank-math.php') ||
               is_plugin_active('rank-math-pro/rank-math-pro.php');
    }

    /**
     * Check if Yoast SEO is installed and active
     */
    public static function is_yoast_active() {
        // Use class_exists for reliable detection
        if (class_exists('WPSEO_Options') || defined('WPSEO_VERSION')) {
            return true;
        }

        // Fallback to is_plugin_active check
        self::ensure_plugin_functions();
        return is_plugin_active('wordpress-seo/wp-seo.php') ||
               is_plugin_active('wordpress-seo-premium/wp-seo-premium.php');
    }

    /**
     * Check if Redirection plugin is installed and active
     */
    public static function is_redirection_active() {
        // Use class_exists for reliable detection
        if (class_exists('Red_Item') || defined('REDIRECTION_VERSION')) {
            return true;
        }

        // Fallback to is_plugin_active check
        self::ensure_plugin_functions();
        return is_plugin_active('redirection/redirection.php');
    }

    /**
     * Check if All in One SEO is installed and active
     */
    public static function is_aioseo_active() {
        // Use class_exists for reliable detection
        if (class_exists('AIOSEO\\Plugin\\AIOSEO') || defined('AIOSEO_VERSION')) {
            return true;
        }

        // Fallback to is_plugin_active check
        self::ensure_plugin_functions();
        return is_plugin_active('all-in-one-seo-pack/all_in_one_seo_pack.php') ||
               is_plugin_active('all-in-one-seo-pack-pro/all_in_one_seo_pack.php');
    }

    /**
     * Get active SEO plugins
     */
    public static function get_active_seo_plugins() {
        $plugins = [];

        if (self::is_rank_math_active()) {
            $plugins['rank-math'] = 'Rank Math';
        }

        if (self::is_yoast_active()) {
            $plugins['yoast'] = 'Yoast SEO';
        }

        if (self::is_aioseo_active()) {
            $plugins['aioseo'] = 'All in One SEO';
        }

        if (self::is_redirection_active()) {
            $plugins['redirection'] = 'Redirection';
        }

        return $plugins;
    }

    /**
     * Check if any competing plugins are active
     */
    public static function has_competing_plugins() {
        return !empty(self::get_active_seo_plugins());
    }

    /**
     * Get Yoast data for a post
     */
    public static function get_yoast_data($post_id) {
        if (!self::is_yoast_active()) {
            return null;
        }

        $data = [
            'focus_keyword' => get_post_meta($post_id, '_yoast_wpseo_focuskw', true),
            'meta_description' => get_post_meta($post_id, '_yoast_wpseo_metadesc', true),
            'title' => get_post_meta($post_id, '_yoast_wpseo_title', true),
            'readability_score' => get_post_meta($post_id, '_yoast_wpseo_content_score', true),
        ];

        return $data;
    }

    /**
     * Get Rank Math data for a post
     */
    public static function get_rank_math_data($post_id) {
        if (!self::is_rank_math_active()) {
            return null;
        }

        $data = [
            'focus_keyword' => get_post_meta($post_id, 'rank_math_focus_keyword', true),
            'description' => get_post_meta($post_id, 'rank_math_description', true),
            'title' => get_post_meta($post_id, 'rank_math_title', true),
        ];

        return $data;
    }

    /**
     * Get All in One SEO data for a post
     */
    public static function get_aioseo_data($post_id) {
        if (!self::is_aioseo_active()) {
            return null;
        }

        $data = [
            'description' => get_post_meta($post_id, '_aioseo_description', true),
            'title' => get_post_meta($post_id, '_aioseo_title', true),
        ];

        return $data;
    }

    /**
     * Get redirects from Redirection plugin
     */
    public static function get_redirections() {
        if (!self::is_redirection_active()) {
            return [];
        }

        global $wpdb;
        
        $table = $wpdb->prefix . 'redirection_items';
        
        if (!$wpdb->get_var("SHOW TABLES LIKE '$table'")) {
            return [];
        }

        $results = $wpdb->get_results("
            SELECT url, action_data as redirect_to, regex
            FROM $table
            LIMIT 1000
        ");

        return $results ?: [];
    }

    /**
     * Get list of installed (not necessarily active) plugins
     */
    public static function get_installed_plugins() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        return get_plugins();
    }

    /**
     * Check if a specific plugin is installed
     */
    public static function is_plugin_installed($plugin_slug) {
        $plugins = self::get_installed_plugins();

        foreach ($plugins as $plugin_file => $plugin_data) {
            if (strpos($plugin_file, $plugin_slug) === 0) {
                return true;
            }
        }

        return false;
    }
}
