<?php
/**
 * Redirect Manager for Screaming Fixes
 *
 * Creates redirects via Rank Math, Yoast Premium, or Redirection plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class SF_Redirect_Manager {

    /**
     * Change logger instance
     * @var SF_Change_Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = new SF_Change_Logger();
    }

    /**
     * Create a redirect using the best available plugin
     *
     * @param string $source Source URL path
     * @param string $destination Destination URL
     * @param int $type Redirect type (301, 302)
     * @param string $module Module creating the redirect
     * @return bool|WP_Error Success or error
     */
    public function create_redirect($source, $destination, $type = 301, $module = 'unknown') {
        // Determine which plugin to use
        $plugin = $this->get_redirect_plugin();

        if (!$plugin) {
            return new WP_Error(
                'no_plugin',
                __('No redirect plugin detected. Install Rank Math, Redirection, or Yoast Premium.', 'screaming-fixes')
            );
        }

        // Normalize source URL
        $source = $this->normalize_source_url($source);

        // Check for duplicate
        if ($this->redirect_exists($source)) {
            return new WP_Error(
                'duplicate',
                __('A redirect from this URL already exists.', 'screaming-fixes')
            );
        }

        // Create redirect based on plugin
        switch ($plugin) {
            case 'rank_math':
                $result = $this->create_redirect_rankmath($source, $destination, $type);
                break;

            case 'redirection':
                $result = $this->create_redirect_redirection($source, $destination, $type);
                break;

            case 'yoast_premium':
                $result = $this->create_redirect_yoast($source, $destination, $type);
                break;

            default:
                return new WP_Error('unknown_plugin', __('Unknown redirect plugin.', 'screaming-fixes'));
        }

        if (is_wp_error($result)) {
            return $result;
        }

        // Log the change
        $this->logger->log_change(0, 'redirect', '', $destination, [
            'source' => $source,
            'destination' => $destination,
            'type' => $type,
            'plugin' => $plugin,
            'redirect_id' => $result['redirect_id'] ?? null,
            'module' => $module,
        ]);

        return true;
    }

    /**
     * Get the preferred redirect plugin
     *
     * @return string|null Plugin identifier or null
     */
    public function get_redirect_plugin() {
        $preferred = get_option('sf_preferred_redirect_plugin', 'auto');

        // If user set a preference, check if it's available
        if ($preferred !== 'auto') {
            if ($this->is_plugin_available($preferred)) {
                return $preferred;
            }
        }

        // Auto-detect in order of preference
        if ($this->is_plugin_available('rank_math')) {
            return 'rank_math';
        }

        if ($this->is_plugin_available('redirection')) {
            return 'redirection';
        }

        if ($this->is_plugin_available('yoast_premium')) {
            return 'yoast_premium';
        }

        return null;
    }

    /**
     * Check if a redirect plugin is available
     *
     * @param string $plugin Plugin identifier
     * @return bool
     */
    public function is_plugin_available($plugin) {
        switch ($plugin) {
            case 'rank_math':
                return class_exists('RankMath') || class_exists('\\RankMath\\Redirections\\Redirection');

            case 'redirection':
                return class_exists('Red_Item') || defined('REDIRECTION_VERSION');

            case 'yoast_premium':
                return class_exists('WPSEO_Redirect');

            default:
                return false;
        }
    }

    /**
     * Get all available redirect plugins
     *
     * @return array Available plugins
     */
    public function get_available_plugins() {
        $plugins = [];

        if ($this->is_plugin_available('rank_math')) {
            $plugins['rank_math'] = [
                'name' => 'Rank Math',
                'detected' => true,
                'ready' => true,
            ];
        }

        if ($this->is_plugin_available('redirection')) {
            $plugins['redirection'] = [
                'name' => 'Redirection',
                'detected' => true,
                'ready' => true,
            ];
        }

        if ($this->is_plugin_available('yoast_premium')) {
            $plugins['yoast_premium'] = [
                'name' => 'Yoast Premium',
                'detected' => true,
                'ready' => true,
            ];
        }

        return $plugins;
    }

    /**
     * Create redirect via Rank Math
     *
     * @param string $source Source URL
     * @param string $destination Destination URL
     * @param int $type Redirect type
     * @return array|WP_Error Result with redirect_id or error
     */
    private function create_redirect_rankmath($source, $destination, $type) {
        if (!class_exists('RankMath')) {
            return new WP_Error('not_available', __('Rank Math not available.', 'screaming-fixes'));
        }

        // Rank Math stores redirects in its own database table
        global $wpdb;

        $table = $wpdb->prefix . 'rank_math_redirections';

        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            // Fall back to option-based storage
            return $this->create_redirect_rankmath_option($source, $destination, $type);
        }

        // Insert into Rank Math redirections table
        $result = $wpdb->insert(
            $table,
            [
                'sources' => serialize([
                    [
                        'pattern' => $source,
                        'comparison' => 'exact',
                    ],
                ]),
                'url_to' => $destination,
                'header_code' => $type,
                'hits' => 0,
                'status' => 'active',
                'created' => current_time('mysql'),
                'updated' => current_time('mysql'),
            ],
            ['%s', '%s', '%d', '%d', '%s', '%s', '%s']
        );

        if ($result === false) {
            return new WP_Error('insert_failed', __('Failed to create redirect in Rank Math.', 'screaming-fixes'));
        }

        // Clear Rank Math cache
        if (function_exists('rank_math_clear_cache')) {
            rank_math_clear_cache();
        }

        return ['redirect_id' => $wpdb->insert_id];
    }

    /**
     * Create redirect via Rank Math using options (fallback)
     *
     * @param string $source Source URL
     * @param string $destination Destination URL
     * @param int $type Redirect type
     * @return array|WP_Error Result or error
     */
    private function create_redirect_rankmath_option($source, $destination, $type) {
        $redirects = get_option('rank_math_redirections_cache', []);

        $new_redirect = [
            'id' => count($redirects) + 1,
            'sources' => [
                [
                    'pattern' => $source,
                    'comparison' => 'exact',
                ],
            ],
            'url_to' => $destination,
            'header_code' => $type,
            'status' => 'active',
        ];

        $redirects[] = $new_redirect;

        update_option('rank_math_redirections_cache', $redirects);

        return ['redirect_id' => $new_redirect['id']];
    }

    /**
     * Create redirect via Redirection plugin
     *
     * @param string $source Source URL
     * @param string $destination Destination URL
     * @param int $type Redirect type
     * @return array|WP_Error Result with redirect_id or error
     */
    private function create_redirect_redirection($source, $destination, $type) {
        if (!class_exists('Red_Item')) {
            // Try to load Redirection classes
            if (defined('REDIRECTION_FILE')) {
                require_once dirname(REDIRECTION_FILE) . '/models/redirect.php';
            }

            if (!class_exists('Red_Item')) {
                return new WP_Error('not_available', __('Redirection plugin not available.', 'screaming-fixes'));
            }
        }

        // Get default group
        $group_id = $this->get_redirection_group();

        // Create the redirect
        $redirect = Red_Item::create([
            'url' => $source,
            'action_data' => ['url' => $destination],
            'action_type' => 'url',
            'action_code' => $type,
            'match_type' => 'url',
            'group_id' => $group_id,
            'status' => 'enabled',
        ]);

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('SF Redirect Manager: Red_Item::create() returned: ' . gettype($redirect));
            if (is_wp_error($redirect)) {
                error_log('SF Redirect Manager: WP_Error: ' . $redirect->get_error_message());
            } elseif (is_object($redirect)) {
                error_log('SF Redirect Manager: Created redirect ID: ' . ($redirect->get_id() ?? 'null'));
            }
        }

        if (is_wp_error($redirect)) {
            return $redirect;
        }

        // Handle case where Red_Item::create returns false/null
        if (!$redirect || !is_object($redirect)) {
            // The redirect might have been created anyway - check if it exists now
            if ($this->redirect_exists($source)) {
                // Redirect exists, so creation succeeded despite odd return value
                return ['redirect_id' => 0, 'note' => 'created_but_id_unavailable'];
            }
            return new WP_Error('creation_failed', __('Failed to create redirect. Please try again.', 'screaming-fixes'));
        }

        return ['redirect_id' => $redirect->get_id()];
    }

    /**
     * Get Redirection plugin group ID
     *
     * @return int Group ID
     */
    private function get_redirection_group() {
        global $wpdb;

        $table = $wpdb->prefix . 'redirection_groups';

        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return 1;
        }

        // Get first active group
        $group_id = $wpdb->get_var(
            "SELECT id FROM {$table} WHERE status = 'enabled' ORDER BY id ASC LIMIT 1"
        );

        return $group_id ? (int) $group_id : 1;
    }

    /**
     * Create redirect via Yoast Premium
     *
     * @param string $source Source URL
     * @param string $destination Destination URL
     * @param int $type Redirect type
     * @return array|WP_Error Result or error
     */
    private function create_redirect_yoast($source, $destination, $type) {
        if (!class_exists('WPSEO_Redirect')) {
            return new WP_Error('not_available', __('Yoast Premium Redirects not available.', 'screaming-fixes'));
        }

        // Yoast stores redirects in options
        $redirects = get_option('wpseo-premium-redirects-base', []);

        // Check for duplicate
        foreach ($redirects as $existing) {
            if ($existing['origin'] === $source) {
                return new WP_Error('duplicate', __('Redirect already exists.', 'screaming-fixes'));
            }
        }

        $redirects[] = [
            'origin' => $source,
            'url' => $destination,
            'type' => $type,
            'format' => 'plain',
        ];

        update_option('wpseo-premium-redirects-base', $redirects);

        return ['redirect_id' => count($redirects) - 1];
    }

    /**
     * Delete a redirect
     *
     * @param string $source Source URL
     * @param string $plugin Plugin that created it
     * @param int|null $redirect_id Redirect ID if known
     * @return bool|WP_Error Success or error
     */
    public function delete_redirect($source, $plugin, $redirect_id = null) {
        switch ($plugin) {
            case 'rank_math':
                return $this->delete_redirect_rankmath($source, $redirect_id);

            case 'redirection':
                return $this->delete_redirect_redirection($source, $redirect_id);

            case 'yoast_premium':
                return $this->delete_redirect_yoast($source);

            default:
                return new WP_Error('unknown_plugin', __('Unknown redirect plugin.', 'screaming-fixes'));
        }
    }

    /**
     * Delete Rank Math redirect
     *
     * @param string $source Source URL
     * @param int|null $redirect_id Redirect ID
     * @return bool|WP_Error
     */
    private function delete_redirect_rankmath($source, $redirect_id) {
        global $wpdb;

        $table = $wpdb->prefix . 'rank_math_redirections';

        if ($redirect_id) {
            $result = $wpdb->delete($table, ['id' => $redirect_id], ['%d']);
        } else {
            // Find and delete by source
            $result = $wpdb->query($wpdb->prepare(
                "DELETE FROM {$table} WHERE sources LIKE %s",
                '%' . $wpdb->esc_like($source) . '%'
            ));
        }

        if ($result === false) {
            return new WP_Error('delete_failed', __('Failed to delete redirect.', 'screaming-fixes'));
        }

        return true;
    }

    /**
     * Delete Redirection plugin redirect
     *
     * @param string $source Source URL
     * @param int|null $redirect_id Redirect ID
     * @return bool|WP_Error
     */
    private function delete_redirect_redirection($source, $redirect_id) {
        if (!class_exists('Red_Item')) {
            return new WP_Error('not_available', __('Redirection plugin not available.', 'screaming-fixes'));
        }

        if ($redirect_id) {
            $redirect = Red_Item::get_by_id($redirect_id);
            if ($redirect) {
                $redirect->delete();
                return true;
            }
        }

        // Find by source URL
        global $wpdb;
        $table = $wpdb->prefix . 'redirection_items';

        $redirect_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE url = %s",
            $source
        ));

        if ($redirect_id) {
            $redirect = Red_Item::get_by_id($redirect_id);
            if ($redirect) {
                $redirect->delete();
                return true;
            }
        }

        return new WP_Error('not_found', __('Redirect not found.', 'screaming-fixes'));
    }

    /**
     * Delete Yoast Premium redirect
     *
     * @param string $source Source URL
     * @return bool|WP_Error
     */
    private function delete_redirect_yoast($source) {
        $redirects = get_option('wpseo-premium-redirects-base', []);

        $found = false;
        foreach ($redirects as $key => $redirect) {
            if ($redirect['origin'] === $source) {
                unset($redirects[$key]);
                $found = true;
                break;
            }
        }

        if (!$found) {
            return new WP_Error('not_found', __('Redirect not found.', 'screaming-fixes'));
        }

        update_option('wpseo-premium-redirects-base', array_values($redirects));

        return true;
    }

    /**
     * Modify an existing redirect's type or destination
     *
     * @param string $source Source URL to find the redirect
     * @param array $changes Changes to apply: 'type' => int, 'destination' => string
     * @param string|null $plugin Specific plugin to use (null = auto-detect)
     * @return array Result with 'success', 'message', 'plugin' keys
     */
    public function modify_redirect($source, $changes = [], $plugin = null) {
        // Determine which plugin to use
        if (!$plugin) {
            $plugin = $this->get_redirect_plugin();
        }

        if (!$plugin) {
            return [
                'success' => false,
                'message' => __('No redirect plugin detected.', 'screaming-fixes'),
                'plugin' => null,
            ];
        }

        // Normalize source URL
        $source = $this->normalize_source_url($source);

        // Validate changes
        if (empty($changes)) {
            return [
                'success' => false,
                'message' => __('No changes specified.', 'screaming-fixes'),
                'plugin' => $plugin,
            ];
        }

        // Get the old values for logging
        $old_values = $this->get_redirect_details($source, $plugin);

        // Modify redirect based on plugin
        switch ($plugin) {
            case 'rank_math':
                $result = $this->modify_redirect_rankmath($source, $changes);
                break;

            case 'redirection':
                $result = $this->modify_redirect_redirection($source, $changes);
                break;

            case 'yoast_premium':
                $result = $this->modify_redirect_yoast($source, $changes);
                break;

            default:
                return [
                    'success' => false,
                    'message' => __('Unknown redirect plugin.', 'screaming-fixes'),
                    'plugin' => $plugin,
                ];
        }

        // Log the change if successful
        if ($result['success'] && $old_values) {
            $this->logger->log_change(0, 'redirect_modify', '', '', [
                'source' => $source,
                'old_type' => $old_values['type'] ?? null,
                'new_type' => $changes['type'] ?? null,
                'old_destination' => $old_values['destination'] ?? null,
                'new_destination' => $changes['destination'] ?? null,
                'plugin' => $plugin,
            ]);
        }

        $result['plugin'] = $plugin;
        return $result;
    }

    /**
     * Get redirect details for a source URL
     *
     * @param string $source Source URL
     * @param string $plugin Plugin to check
     * @return array|null Redirect details or null if not found
     */
    private function get_redirect_details($source, $plugin) {
        switch ($plugin) {
            case 'rank_math':
                return $this->get_redirect_details_rankmath($source);

            case 'redirection':
                return $this->get_redirect_details_redirection($source);

            case 'yoast_premium':
                return $this->get_redirect_details_yoast($source);

            default:
                return null;
        }
    }

    /**
     * Get Rank Math redirect details
     *
     * @param string $source Source URL
     * @return array|null
     */
    private function get_redirect_details_rankmath($source) {
        global $wpdb;

        $table = $wpdb->prefix . 'rank_math_redirections';

        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return null;
        }

        $redirect = $wpdb->get_row($wpdb->prepare(
            "SELECT id, url_to, header_code FROM {$table} WHERE sources LIKE %s LIMIT 1",
            '%' . $wpdb->esc_like($source) . '%'
        ));

        if (!$redirect) {
            return null;
        }

        return [
            'id' => $redirect->id,
            'destination' => $redirect->url_to,
            'type' => (int) $redirect->header_code,
        ];
    }

    /**
     * Get Redirection plugin redirect details
     *
     * @param string $source Source URL
     * @return array|null
     */
    private function get_redirect_details_redirection($source) {
        global $wpdb;

        $table = $wpdb->prefix . 'redirection_items';

        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return null;
        }

        $redirect = $wpdb->get_row($wpdb->prepare(
            "SELECT id, action_data, action_code FROM {$table} WHERE url = %s LIMIT 1",
            $source
        ));

        if (!$redirect) {
            return null;
        }

        return [
            'id' => $redirect->id,
            'destination' => $redirect->action_data,
            'type' => (int) $redirect->action_code,
        ];
    }

    /**
     * Get Yoast Premium redirect details
     *
     * @param string $source Source URL
     * @return array|null
     */
    private function get_redirect_details_yoast($source) {
        $redirects = get_option('wpseo-premium-redirects-base', []);

        foreach ($redirects as $index => $redirect) {
            if ($redirect['origin'] === $source) {
                return [
                    'id' => $index,
                    'destination' => $redirect['url'],
                    'type' => (int) $redirect['type'],
                ];
            }
        }

        return null;
    }

    /**
     * Modify Rank Math redirect
     *
     * @param string $source Source URL
     * @param array $changes Changes to apply
     * @return array Result
     */
    private function modify_redirect_rankmath($source, $changes) {
        global $wpdb;

        $table = $wpdb->prefix . 'rank_math_redirections';

        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return [
                'success' => false,
                'message' => __('Rank Math redirections table not found.', 'screaming-fixes'),
            ];
        }

        // Find the redirect
        $redirect_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE sources LIKE %s LIMIT 1",
            '%' . $wpdb->esc_like($source) . '%'
        ));

        if (!$redirect_id) {
            return [
                'success' => false,
                'message' => __('Redirect not found in Rank Math.', 'screaming-fixes'),
            ];
        }

        // Build update data
        $update_data = ['updated' => current_time('mysql')];
        $update_format = ['%s'];

        if (isset($changes['type'])) {
            $update_data['header_code'] = (int) $changes['type'];
            $update_format[] = '%d';
        }

        if (isset($changes['destination'])) {
            $update_data['url_to'] = $changes['destination'];
            $update_format[] = '%s';
        }

        // Perform update
        $result = $wpdb->update(
            $table,
            $update_data,
            ['id' => $redirect_id],
            $update_format,
            ['%d']
        );

        if ($result === false) {
            return [
                'success' => false,
                'message' => __('Failed to update redirect in Rank Math.', 'screaming-fixes'),
            ];
        }

        // Clear Rank Math cache
        if (function_exists('rank_math_clear_cache')) {
            rank_math_clear_cache();
        }

        return [
            'success' => true,
            'message' => __('Redirect updated in Rank Math.', 'screaming-fixes'),
        ];
    }

    /**
     * Modify Redirection plugin redirect
     *
     * @param string $source Source URL
     * @param array $changes Changes to apply
     * @return array Result
     */
    private function modify_redirect_redirection($source, $changes) {
        if (!class_exists('Red_Item')) {
            // Try to load Redirection classes
            if (defined('REDIRECTION_FILE')) {
                require_once dirname(REDIRECTION_FILE) . '/models/redirect/redirect.php';
            }

            if (!class_exists('Red_Item')) {
                return [
                    'success' => false,
                    'message' => __('Redirection plugin not available.', 'screaming-fixes'),
                ];
            }
        }

        // Find the redirect by source URL
        global $wpdb;
        $table = $wpdb->prefix . 'redirection_items';

        $redirect_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE url = %s LIMIT 1",
            $source
        ));

        if (!$redirect_id) {
            return [
                'success' => false,
                'message' => __('Redirect not found in Redirection plugin.', 'screaming-fixes'),
            ];
        }

        // Get the redirect object
        $redirect = Red_Item::get_by_id($redirect_id);

        if (!$redirect) {
            return [
                'success' => false,
                'message' => __('Could not load redirect from Redirection plugin.', 'screaming-fixes'),
            ];
        }

        // Build update details
        $update_details = [];

        if (isset($changes['type'])) {
            $update_details['action_code'] = (int) $changes['type'];
        }

        if (isset($changes['destination'])) {
            $update_details['action_data'] = ['url' => $changes['destination']];
        }

        if (empty($update_details)) {
            return [
                'success' => false,
                'message' => __('No valid changes to apply.', 'screaming-fixes'),
            ];
        }

        // Perform the update
        $result = $redirect->update($update_details);

        if (is_wp_error($result)) {
            return [
                'success' => false,
                'message' => $result->get_error_message(),
            ];
        }

        return [
            'success' => true,
            'message' => __('Redirect updated in Redirection plugin.', 'screaming-fixes'),
        ];
    }

    /**
     * Modify Yoast Premium redirect
     *
     * @param string $source Source URL
     * @param array $changes Changes to apply
     * @return array Result
     */
    private function modify_redirect_yoast($source, $changes) {
        $redirects = get_option('wpseo-premium-redirects-base', []);

        $found_index = null;
        foreach ($redirects as $index => $redirect) {
            if ($redirect['origin'] === $source) {
                $found_index = $index;
                break;
            }
        }

        if ($found_index === null) {
            return [
                'success' => false,
                'message' => __('Redirect not found in Yoast Premium.', 'screaming-fixes'),
            ];
        }

        // Apply changes
        if (isset($changes['type'])) {
            $redirects[$found_index]['type'] = (int) $changes['type'];
        }

        if (isset($changes['destination'])) {
            $redirects[$found_index]['url'] = $changes['destination'];
        }

        // Save updated redirects
        update_option('wpseo-premium-redirects-base', $redirects);

        return [
            'success' => true,
            'message' => __('Redirect updated in Yoast Premium.', 'screaming-fixes'),
        ];
    }

    /**
     * Check if a redirect already exists for a URL
     *
     * @param string $source Source URL
     * @return bool
     */
    public function redirect_exists($source) {
        $plugin = $this->get_redirect_plugin();

        if (!$plugin) {
            return false;
        }

        switch ($plugin) {
            case 'rank_math':
                return $this->redirect_exists_rankmath($source);

            case 'redirection':
                return $this->redirect_exists_redirection($source);

            case 'yoast_premium':
                return $this->redirect_exists_yoast($source);

            default:
                return false;
        }
    }

    /**
     * Check for existing Rank Math redirect
     *
     * @param string $source Source URL
     * @return bool
     */
    private function redirect_exists_rankmath($source) {
        global $wpdb;

        $table = $wpdb->prefix . 'rank_math_redirections';

        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return false;
        }

        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE sources LIKE %s",
            '%' . $wpdb->esc_like($source) . '%'
        ));

        return $exists > 0;
    }

    /**
     * Check for existing Redirection plugin redirect
     *
     * @param string $source Source URL
     * @return bool
     */
    private function redirect_exists_redirection($source) {
        global $wpdb;

        $table = $wpdb->prefix . 'redirection_items';

        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return false;
        }

        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE url = %s",
            $source
        ));

        return $exists > 0;
    }

    /**
     * Check for existing Yoast Premium redirect
     *
     * @param string $source Source URL
     * @return bool
     */
    private function redirect_exists_yoast($source) {
        $redirects = get_option('wpseo-premium-redirects-base', []);

        foreach ($redirects as $redirect) {
            if ($redirect['origin'] === $source) {
                return true;
            }
        }

        return false;
    }

    /**
     * Normalize source URL for redirect
     *
     * @param string $url URL to normalize
     * @return string Normalized URL path
     */
    private function normalize_source_url($url) {
        // Parse the URL
        $parsed = wp_parse_url($url);

        // Get just the path
        $path = $parsed['path'] ?? '/';

        // Ensure leading slash
        if (substr($path, 0, 1) !== '/') {
            $path = '/' . $path;
        }

        // Check if WordPress uses trailing slashes in permalinks
        // Most WordPress sites with pretty permalinks use trailing slashes
        $permalink_structure = get_option('permalink_structure', '');
        $wp_uses_trailing_slash = !empty($permalink_structure) && substr($permalink_structure, -1) === '/';

        if ($path !== '/') {
            if ($wp_uses_trailing_slash) {
                // WordPress uses trailing slashes - ensure path has one
                if (substr($path, -1) !== '/') {
                    $path = $path . '/';
                }
            } else {
                // WordPress doesn't use trailing slashes - remove any
                $path = rtrim($path, '/');
            }
        }

        return $path;
    }

    /**
     * Bulk create redirects
     *
     * @param array $redirects Array of redirect definitions
     * @param string $module Module creating the redirects
     * @return array Results summary
     */
    public function bulk_create_redirects($redirects, $module = 'unknown') {
        $results = [
            'total' => count($redirects),
            'success' => 0,
            'failed' => 0,
            'errors' => [],
        ];

        foreach ($redirects as $redirect) {
            $source = $redirect['source'] ?? '';
            $destination = $redirect['destination'] ?? '';
            $type = $redirect['type'] ?? 301;

            if (empty($source) || empty($destination)) {
                $results['failed']++;
                $results['errors'][] = [
                    'source' => $source,
                    'error' => __('Missing source or destination.', 'screaming-fixes'),
                ];
                continue;
            }

            $result = $this->create_redirect($source, $destination, $type, $module);

            if (is_wp_error($result)) {
                $results['failed']++;
                $results['errors'][] = [
                    'source' => $source,
                    'error' => $result->get_error_message(),
                ];
            } else {
                $results['success']++;
            }
        }

        return $results;
    }

    /**
     * Check if we have the capability to create redirects
     *
     * @return bool
     */
    public function has_capability() {
        $status = $this->get_capability_status();
        return !empty($status['has_capability']);
    }

    /**
     * Get redirect capability status
     *
     * @return array Status information
     */
    public function get_capability_status() {
        $plugins = $this->get_available_plugins();
        $preferred = get_option('sf_preferred_redirect_plugin', 'auto');
        $active = $this->get_redirect_plugin();

        return [
            'has_capability' => !empty($plugins),
            'plugins' => $plugins,
            'preferred' => $preferred,
            'active_plugin' => $active,
            'active_plugin_name' => $active ? ($plugins[$active]['name'] ?? $active) : null,
        ];
    }
}
